"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyTransformer = void 0;
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_mapping_template_1 = require("graphql-mapping-template");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const graphql_dynamodb_transformer_1 = require("graphql-dynamodb-transformer");
const graphql_1 = require("graphql");
const cloudform_types_1 = require("cloudform-types");
const table_1 = require("cloudform-types/types/dynamoDb/table");
class KeyTransformer extends graphql_transformer_core_1.Transformer {
    constructor() {
        super('KeyTransformer', graphql_transformer_core_1.gql `
        directive @key(name: String, fields: [String!]!, queryField: String) repeatable on OBJECT
      `);
        this.before = (ctx) => {
            if (ctx.isProjectUsingDataStore()) {
                const resolverMap = new Map();
                ctx.metadata.set(graphql_transformer_common_1.ResourceConstants.SNIPPETS.SyncResolverKey, resolverMap);
            }
        };
        this.after = (ctx) => {
            if (ctx.isProjectUsingDataStore()) {
                const resolverMap = ctx.metadata.get(graphql_transformer_common_1.ResourceConstants.SNIPPETS.SyncResolverKey);
                if (resolverMap.size) {
                    resolverMap.forEach((syncVTLContent, resource) => {
                        if (syncVTLContent) {
                            resource.Properties.RequestMappingTemplate = '';
                            resource.Properties.RequestMappingTemplate = joinSnippets([
                                graphql_mapping_template_1.print(generateSyncResolverInit()),
                                syncVTLContent,
                                graphql_mapping_template_1.print(setSyncQueryFilterSnippet()),
                                graphql_mapping_template_1.print(setSyncKeyExpressionForHashKey(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression)),
                                graphql_mapping_template_1.print(setSyncKeyExpressionForRangeKey(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression)),
                                graphql_mapping_template_1.print(makeSyncQueryResolver()),
                            ]);
                        }
                    });
                }
            }
        };
        this.object = (definition, directive, ctx) => {
            this.validate(definition, directive, ctx);
            this.updateIndexStructures(definition, directive, ctx);
            this.updateSchema(definition, directive, ctx);
            this.updateResolvers(definition, directive, ctx);
            this.addKeyConditionInputs(definition, directive, ctx);
            this.updateMutationConditionInput(ctx, definition, directive);
        };
        this.updateIndexStructures = (definition, directive, ctx) => {
            if (this.isPrimaryKey(directive)) {
                this.replacePrimaryKey(definition, directive, ctx);
            }
            else {
                this.appendSecondaryIndex(definition, directive, ctx);
            }
        };
        this.updateSchema = (definition, directive, ctx) => {
            this.updateQueryFields(definition, directive, ctx);
            this.updateInputObjects(definition, directive, ctx);
            const isPrimaryKey = this.isPrimaryKey(directive);
            if (isPrimaryKey) {
                this.removeAutoCreatedPrimaryKey(definition, directive, ctx);
            }
        };
        this.updateResolvers = (definition, directive, ctx) => {
            const directiveArgs = graphql_transformer_core_1.getDirectiveArguments(directive);
            const getResolver = ctx.getResource(graphql_transformer_common_1.ResolverResourceIDs.DynamoDBGetResolverResourceID(definition.name.value));
            const listResolver = ctx.getResource(graphql_transformer_common_1.ResolverResourceIDs.DynamoDBListResolverResourceID(definition.name.value));
            const createResolver = ctx.getResource(graphql_transformer_common_1.ResolverResourceIDs.DynamoDBCreateResolverResourceID(definition.name.value));
            const updateResolver = ctx.getResource(graphql_transformer_common_1.ResolverResourceIDs.DynamoDBUpdateResolverResourceID(definition.name.value));
            const deleteResolver = ctx.getResource(graphql_transformer_common_1.ResolverResourceIDs.DynamoDBDeleteResolverResourceID(definition.name.value));
            const syncResolver = ctx.getResource(graphql_transformer_common_1.ResolverResourceIDs.SyncResolverResourceID(definition.name.value));
            if (this.isPrimaryKey(directive)) {
                if (getResolver) {
                    getResolver.Properties.RequestMappingTemplate = joinSnippets([
                        this.setKeySnippet(directive),
                        getResolver.Properties.RequestMappingTemplate,
                    ]);
                }
                if (listResolver) {
                    listResolver.Properties.RequestMappingTemplate = joinSnippets([
                        graphql_mapping_template_1.print(setQuerySnippet(definition, directive, ctx, true)),
                        listResolver.Properties.RequestMappingTemplate,
                    ]);
                }
                if (createResolver) {
                    createResolver.Properties.RequestMappingTemplate = joinSnippets([
                        this.setKeySnippet(directive, true),
                        ensureCompositeKeySnippet(directive),
                        createResolver.Properties.RequestMappingTemplate,
                    ]);
                }
                if (updateResolver) {
                    updateResolver.Properties.RequestMappingTemplate = joinSnippets([
                        this.setKeySnippet(directive, true),
                        ensureCompositeKeySnippet(directive),
                        updateResolver.Properties.RequestMappingTemplate,
                    ]);
                }
                if (deleteResolver) {
                    deleteResolver.Properties.RequestMappingTemplate = joinSnippets([
                        this.setKeySnippet(directive, true),
                        deleteResolver.Properties.RequestMappingTemplate,
                    ]);
                }
                if (syncResolver) {
                    constructSyncResolver(directive, ctx, syncResolver, true);
                }
            }
            else {
                if (createResolver) {
                    createResolver.Properties.RequestMappingTemplate = joinSnippets([
                        this.validateKeyArgumentSnippet(directive, 'create'),
                        ensureCompositeKeySnippet(directive),
                        createResolver.Properties.RequestMappingTemplate,
                    ]);
                }
                if (updateResolver) {
                    updateResolver.Properties.RequestMappingTemplate = joinSnippets([
                        this.validateKeyArgumentSnippet(directive, 'update'),
                        ensureCompositeKeySnippet(directive),
                        updateResolver.Properties.RequestMappingTemplate,
                    ]);
                }
                if (deleteResolver) {
                    deleteResolver.Properties.RequestMappingTemplate = joinSnippets([
                        ensureCompositeKeySnippet(directive),
                        deleteResolver.Properties.RequestMappingTemplate,
                    ]);
                }
                if (syncResolver) {
                    constructSyncResolver(directive, ctx, syncResolver, false);
                }
                if (directiveArgs.queryField) {
                    const queryTypeName = ctx.getQueryTypeName();
                    const queryResolverId = graphql_transformer_common_1.ResolverResourceIDs.ResolverResourceID(queryTypeName, directiveArgs.queryField);
                    const queryResolver = makeQueryResolver(definition, directive, ctx);
                    ctx.mapResourceToStack(definition.name.value, queryResolverId);
                    ctx.setResource(queryResolverId, queryResolver);
                }
            }
        };
        this.removeAutoCreatedPrimaryKey = (definition, directive, ctx) => {
            const schemaHasIdField = definition.fields.find(f => f.name.value === 'id');
            if (!schemaHasIdField) {
                const obj = ctx.getObject(definition.name.value);
                const fields = obj.fields.filter(f => f.name.value !== 'id');
                const newObj = {
                    ...obj,
                    fields,
                };
                ctx.updateObject(newObj);
            }
        };
        this.addKeyConditionInputs = (definition, directive, ctx) => {
            const args = graphql_transformer_core_1.getDirectiveArguments(directive);
            if (args.fields.length > 2) {
                const compositeKeyFieldNames = args.fields.slice(1);
                const compositeKeyFields = [];
                for (const compositeKeyFieldName of compositeKeyFieldNames) {
                    const field = definition.fields.find(field => field.name.value === compositeKeyFieldName);
                    if (!field) {
                        throw new graphql_transformer_core_1.InvalidDirectiveError(`Can't find field: ${compositeKeyFieldName} in ${definition.name.value}, but it was specified in the @key definition.`);
                    }
                    else {
                        compositeKeyFields.push(field);
                    }
                }
                const keyName = graphql_transformer_common_1.toUpper(args.name || 'Primary');
                const keyConditionInput = graphql_transformer_common_1.makeCompositeKeyConditionInputForKey(definition.name.value, keyName, compositeKeyFields);
                if (!ctx.getType(keyConditionInput.name.value)) {
                    ctx.addInput(keyConditionInput);
                }
                const compositeKeyInput = graphql_transformer_common_1.makeCompositeKeyInputForKey(definition.name.value, keyName, compositeKeyFields);
                if (!ctx.getType(compositeKeyInput.name.value)) {
                    ctx.addInput(compositeKeyInput);
                }
            }
            else if (args.fields.length === 2) {
                const finalSortKeyFieldName = args.fields[1];
                const finalSortKeyField = definition.fields.find(f => f.name.value === finalSortKeyFieldName);
                const typeResolver = (baseType) => {
                    const resolvedEnumType = ctx.getType(baseType);
                    return resolvedEnumType ? 'String' : undefined;
                };
                const sortKeyConditionInput = graphql_transformer_common_1.makeScalarKeyConditionForType(finalSortKeyField.type, typeResolver);
                if (!sortKeyConditionInput) {
                    const checkedKeyName = args.name ? args.name : '<unnamed>';
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`Cannot resolve type for field '${finalSortKeyFieldName}' in @key '${checkedKeyName}' on type '${definition.name.value}'.`);
                }
                if (!ctx.getType(sortKeyConditionInput.name.value)) {
                    ctx.addInput(sortKeyConditionInput);
                }
            }
        };
        this.updateQueryFields = (definition, directive, ctx) => {
            this.updateGetField(definition, directive, ctx);
            this.updateListField(definition, directive, ctx);
            this.ensureQueryField(definition, directive, ctx);
        };
        this.updateGetField = (definition, directive, ctx) => {
            let query = ctx.getQuery();
            const getResourceID = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBGetResolverResourceID(definition.name.value);
            const getResolverResource = ctx.getResource(getResourceID);
            if (getResolverResource && this.isPrimaryKey(directive)) {
                let getField = query.fields.find(field => field.name.value === getResolverResource.Properties.FieldName);
                const args = graphql_transformer_core_1.getDirectiveArguments(directive);
                const getArguments = args.fields.map(keyAttributeName => {
                    const keyField = definition.fields.find(field => field.name.value === keyAttributeName);
                    const keyArgument = graphql_transformer_common_1.makeInputValueDefinition(keyAttributeName, graphql_transformer_common_1.makeNonNullType(graphql_transformer_common_1.makeNamedType(graphql_transformer_common_1.getBaseType(keyField.type))));
                    return keyArgument;
                });
                getField = { ...getField, arguments: getArguments };
                query = { ...query, fields: query.fields.map(field => (field.name.value === getField.name.value ? getField : field)) };
                ctx.putType(query);
            }
        };
        this.updateListField = (definition, directive, ctx) => {
            const listResourceID = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBListResolverResourceID(definition.name.value);
            const listResolverResource = ctx.getResource(listResourceID);
            if (listResolverResource && this.isPrimaryKey(directive)) {
                let query = ctx.getQuery();
                let listField = query.fields.find(field => field.name.value === listResolverResource.Properties.FieldName);
                let listArguments = [...listField.arguments];
                const args = graphql_transformer_core_1.getDirectiveArguments(directive);
                if (args.fields.length > 2) {
                    listArguments = addCompositeSortKey(definition, args, listArguments);
                    listArguments = addHashField(definition, args, listArguments);
                }
                else if (args.fields.length === 2) {
                    listArguments = addSimpleSortKey(ctx, definition, args, listArguments);
                    listArguments = addHashField(definition, args, listArguments);
                }
                else {
                    listArguments = addHashField(definition, args, listArguments);
                }
                listArguments.push(graphql_transformer_common_1.makeInputValueDefinition('sortDirection', graphql_transformer_common_1.makeNamedType('ModelSortDirection')));
                listField = { ...listField, arguments: listArguments };
                query = { ...query, fields: query.fields.map(field => (field.name.value === listField.name.value ? listField : field)) };
                ctx.putType(query);
            }
        };
        this.ensureQueryField = (definition, directive, ctx) => {
            const args = graphql_transformer_core_1.getDirectiveArguments(directive);
            if (args.queryField && !this.isPrimaryKey(directive)) {
                let queryType = ctx.getQuery();
                let queryArguments = [];
                if (args.fields.length > 2) {
                    queryArguments = addCompositeSortKey(definition, args, queryArguments);
                    queryArguments = addHashField(definition, args, queryArguments);
                }
                else if (args.fields.length === 2) {
                    queryArguments = addSimpleSortKey(ctx, definition, args, queryArguments);
                    queryArguments = addHashField(definition, args, queryArguments);
                }
                else {
                    queryArguments = addHashField(definition, args, queryArguments);
                }
                queryArguments.push(graphql_transformer_common_1.makeInputValueDefinition('sortDirection', graphql_transformer_common_1.makeNamedType('ModelSortDirection')));
                const queryField = graphql_transformer_common_1.makeConnectionField(args.queryField, definition.name.value, queryArguments);
                queryType = {
                    ...queryType,
                    fields: [...queryType.fields, queryField],
                };
                ctx.putType(queryType);
                if (!this.typeExist('ModelSortDirection', ctx)) {
                    const modelSortDirection = graphql_dynamodb_transformer_1.makeModelSortDirectionEnumObject();
                    ctx.addEnum(modelSortDirection);
                }
                this.generateFilterInputs(ctx, definition);
                this.generateModelXConnectionType(ctx, definition);
            }
        };
        this.updateInputObjects = (definition, directive, ctx) => {
            if (this.isPrimaryKey(directive)) {
                const directiveArgs = graphql_transformer_core_1.getDirectiveArguments(directive);
                const hasIdField = definition.fields.find(f => f.name.value === 'id');
                if (!hasIdField) {
                    const createInput = ctx.getType(graphql_transformer_common_1.ModelResourceIDs.ModelCreateInputObjectName(definition.name.value));
                    if (createInput) {
                        ctx.putType(replaceCreateInput(definition, createInput, directiveArgs.fields));
                    }
                }
                const updateInput = ctx.getType(graphql_transformer_common_1.ModelResourceIDs.ModelUpdateInputObjectName(definition.name.value));
                if (updateInput) {
                    ctx.putType(replaceUpdateInput(definition, updateInput, directiveArgs.fields));
                }
                const deleteInput = ctx.getType(graphql_transformer_common_1.ModelResourceIDs.ModelDeleteInputObjectName(definition.name.value));
                if (deleteInput) {
                    ctx.putType(replaceDeleteInput(definition, deleteInput, directiveArgs.fields));
                }
            }
        };
        this.setKeySnippet = (directive, isMutation = false) => {
            const directiveArgs = graphql_transformer_core_1.getDirectiveArguments(directive);
            const cmds = [graphql_mapping_template_1.set(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey), modelObjectKey(directiveArgs, isMutation))];
            return graphql_mapping_template_1.printBlock(`Set the primary @key`)(graphql_mapping_template_1.compoundExpression(cmds));
        };
        this.validateKeyArgumentSnippet = (directive, keyOperation) => {
            const directiveArgs = graphql_transformer_core_1.getDirectiveArguments(directive);
            if (!this.isPrimaryKey(directive) && directiveArgs.fields.length > 2) {
                const sortKeyFields = directiveArgs.fields.slice(1);
                return graphql_mapping_template_1.printBlock(`Validate ${keyOperation} mutation for @key '${directiveArgs.name}'`)(graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('hasSeenSomeKeyArg'), graphql_mapping_template_1.bool(false)),
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('keyFieldNames'), graphql_mapping_template_1.list(sortKeyFields.map(f => graphql_mapping_template_1.str(f)))),
                    graphql_mapping_template_1.forEach(graphql_mapping_template_1.ref('keyFieldName'), graphql_mapping_template_1.ref('keyFieldNames'), [
                        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`$ctx.args.input.containsKey("$keyFieldName")`), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('hasSeenSomeKeyArg'), graphql_mapping_template_1.bool(true)), true),
                    ]),
                    graphql_mapping_template_1.forEach(graphql_mapping_template_1.ref('keyFieldName'), graphql_mapping_template_1.ref('keyFieldNames'), [
                        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`$hasSeenSomeKeyArg && !$ctx.args.input.containsKey("$keyFieldName")`), graphql_mapping_template_1.raw(`$util.error("When ${keyOperation.replace(/.$/, 'ing')} any part of the composite sort key for @key '${directiveArgs.name}',` + ` you must provide all fields for the key. Missing key: '$keyFieldName'.")`)),
                    ]),
                ]));
            }
            return '';
        };
        this.validate = (definition, directive, ctx) => {
            const directiveArgs = graphql_transformer_core_1.getDirectiveArguments(directive);
            if (!directiveArgs.name) {
                for (const otherDirective of definition.directives.filter(d => d.name.value === 'key')) {
                    const otherArgs = graphql_transformer_core_1.getDirectiveArguments(otherDirective);
                    if (otherDirective !== directive && !otherArgs.name) {
                        throw new graphql_transformer_core_1.InvalidDirectiveError(`You may only supply one primary @key on type '${definition.name.value}'.`);
                    }
                    const hasPrimarySortKey = directiveArgs.fields.length > 1;
                    const primaryHashField = directiveArgs.fields[0];
                    const otherHashField = otherArgs.fields[0];
                    if (otherDirective !== directive &&
                        !hasPrimarySortKey &&
                        primaryHashField === otherHashField) {
                        throw new graphql_transformer_core_1.InvalidDirectiveError(`Invalid @key "${otherArgs.name}". You may not create a @key where the first field in 'fields' ` +
                            `is the same as that of the primary @key unless the primary @key has multiple 'fields'. ` +
                            `You cannot have a local secondary index without a sort key in the primary index.`);
                    }
                }
                if (directiveArgs.queryField) {
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`You cannot pass 'queryField' to the primary @key on type '${definition.name.value}'.`);
                }
            }
            else {
                for (const otherDirective of definition.directives.filter(d => d.name.value === 'key')) {
                    const otherArgs = graphql_transformer_core_1.getDirectiveArguments(otherDirective);
                    if (otherDirective !== directive && otherArgs.name === directiveArgs.name) {
                        throw new graphql_transformer_core_1.InvalidDirectiveError(`You may only supply one @key with the name '${directiveArgs.name}' on type '${definition.name.value}'.`);
                    }
                }
            }
            const fieldMap = new Map();
            for (const field of definition.fields) {
                fieldMap.set(field.name.value, field);
            }
            for (const fieldName of directiveArgs.fields) {
                if (!fieldMap.has(fieldName)) {
                    const checkedKeyName = directiveArgs.name ? directiveArgs.name : '<unnamed>';
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`You cannot specify a nonexistent field '${fieldName}' in @key '${checkedKeyName}' on type '${definition.name.value}'.`);
                }
                else {
                    const existingField = fieldMap.get(fieldName);
                    const ddbKeyType = attributeTypeFromType(existingField.type, ctx);
                    if (this.isPrimaryKey(directive) && !graphql_transformer_common_1.isNonNullType(existingField.type)) {
                        throw new graphql_transformer_core_1.InvalidDirectiveError(`The primary @key on type '${definition.name.value}' must reference non-null fields.`);
                    }
                    else if (ddbKeyType !== 'S' && ddbKeyType !== 'N' && ddbKeyType !== 'B') {
                        throw new graphql_transformer_core_1.InvalidDirectiveError(`A @key on type '${definition.name.value}' cannot reference non-scalar field ${fieldName}.`);
                    }
                }
            }
        };
        this.isPrimaryKey = (directive) => {
            const directiveArgs = graphql_transformer_core_1.getDirectiveArguments(directive);
            return !Boolean(directiveArgs.name);
        };
        this.replacePrimaryKey = (definition, directive, ctx) => {
            const args = graphql_transformer_core_1.getDirectiveArguments(directive);
            const ks = keySchema(args);
            const attrDefs = attributeDefinitions(args, definition, ctx);
            const tableLogicalID = graphql_transformer_common_1.ModelResourceIDs.ModelTableResourceID(definition.name.value);
            const tableResource = ctx.getResource(tableLogicalID);
            if (!tableResource) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`The @key directive may only be added to object definitions annotated with @model.`);
            }
            else {
                const existingAttrDefSet = new Set(tableResource.Properties.AttributeDefinitions.map(ad => ad.AttributeName));
                for (const existingKey of tableResource.Properties.KeySchema) {
                    if (existingAttrDefSet.has(existingKey.AttributeName)) {
                        tableResource.Properties.AttributeDefinitions = tableResource.Properties.AttributeDefinitions.filter(ad => ad.AttributeName !== existingKey.AttributeName);
                        existingAttrDefSet.delete(existingKey.AttributeName);
                    }
                }
                tableResource.Properties.KeySchema = ks;
                for (const attr of attrDefs) {
                    if (!existingAttrDefSet.has(attr.AttributeName)) {
                        tableResource.Properties.AttributeDefinitions.push(attr);
                    }
                }
            }
        };
        this.appendSecondaryIndex = (definition, directive, ctx) => {
            const args = graphql_transformer_core_1.getDirectiveArguments(directive);
            const ks = keySchema(args);
            const attrDefs = attributeDefinitions(args, definition, ctx);
            const tableLogicalID = graphql_transformer_common_1.ModelResourceIDs.ModelTableResourceID(definition.name.value);
            const tableResource = ctx.getResource(tableLogicalID);
            const primaryKeyDirective = getPrimaryKey(definition);
            const primaryPartitionKeyName = primaryKeyDirective ? graphql_transformer_core_1.getDirectiveArguments(primaryKeyDirective).fields[0] : 'id';
            if (!tableResource) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`The @key directive may only be added to object definitions annotated with @model.`);
            }
            else {
                const baseIndexProperties = {
                    IndexName: args.name,
                    KeySchema: ks,
                    Projection: new table_1.Projection({
                        ProjectionType: 'ALL',
                    }),
                };
                if (primaryPartitionKeyName === ks[0].AttributeName) {
                    tableResource.Properties.LocalSecondaryIndexes = append(tableResource.Properties.LocalSecondaryIndexes, new table_1.LocalSecondaryIndex(baseIndexProperties));
                }
                else {
                    tableResource.Properties.GlobalSecondaryIndexes = append(tableResource.Properties.GlobalSecondaryIndexes, new table_1.GlobalSecondaryIndex({
                        ...baseIndexProperties,
                        ProvisionedThroughput: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.ShouldUsePayPerRequestBilling, cloudform_types_1.Refs.NoValue, {
                            ReadCapacityUnits: cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBModelTableReadIOPS),
                            WriteCapacityUnits: cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBModelTableWriteIOPS),
                        }),
                    }));
                }
                const existingAttrDefSet = new Set(tableResource.Properties.AttributeDefinitions.map(ad => ad.AttributeName));
                for (const attr of attrDefs) {
                    if (!existingAttrDefSet.has(attr.AttributeName)) {
                        tableResource.Properties.AttributeDefinitions.push(attr);
                    }
                }
            }
        };
    }
    generateFilterInputs(ctx, def) {
        const scalarFilters = graphql_dynamodb_transformer_1.makeScalarFilterInputs(this.supportsConditions(ctx));
        for (const filter of scalarFilters) {
            if (!this.typeExist(filter.name.value, ctx)) {
                ctx.addInput(filter);
            }
        }
        const enumFilters = graphql_dynamodb_transformer_1.makeEnumFilterInputObjects(def, ctx, this.supportsConditions(ctx));
        for (const filter of enumFilters) {
            if (!this.typeExist(filter.name.value, ctx)) {
                ctx.addInput(filter);
            }
        }
        const tableXQueryFilterInput = graphql_dynamodb_transformer_1.makeModelXFilterInputObject(def, ctx, this.supportsConditions(ctx));
        if (!this.typeExist(tableXQueryFilterInput.name.value, ctx)) {
            ctx.addInput(tableXQueryFilterInput);
        }
        if (this.supportsConditions(ctx)) {
            const attributeTypeEnum = graphql_dynamodb_transformer_1.makeAttributeTypeEnum();
            if (!this.typeExist(attributeTypeEnum.name.value, ctx)) {
                ctx.addType(attributeTypeEnum);
            }
        }
    }
    generateModelXConnectionType(ctx, def) {
        const tableXConnectionName = graphql_transformer_common_1.ModelResourceIDs.ModelConnectionTypeName(def.name.value);
        if (this.typeExist(tableXConnectionName, ctx)) {
            return;
        }
        const connectionType = graphql_transformer_common_1.blankObject(tableXConnectionName);
        ctx.addObject(connectionType);
        ctx.addObjectExtension(graphql_dynamodb_transformer_1.makeModelConnectionType(def.name.value));
    }
    updateMutationConditionInput(ctx, type, directive) {
        const tableXMutationConditionInputName = graphql_transformer_common_1.ModelResourceIDs.ModelConditionInputTypeName(type.name.value);
        if (this.typeExist(tableXMutationConditionInputName, ctx)) {
            const tableXMutationConditionInput = ctx.getType(tableXMutationConditionInputName);
            const fieldNames = new Set();
            const getKeyFieldNames = () => {
                let fields;
                if (graphql_transformer_common_1.getDirectiveArgument(directive, 'name') === undefined) {
                    const fieldsArg = graphql_transformer_common_1.getDirectiveArgument(directive, 'fields');
                    if (fieldsArg && fieldsArg.length && fieldsArg.length > 0) {
                        fields = type.fields.filter(f => fieldsArg.includes(f.name.value));
                    }
                }
                fieldNames.add('id');
                if (fields && fields.length > 0) {
                    fields.forEach(f => fieldNames.add(f.name.value));
                }
                else {
                    fieldNames.add('id');
                }
            };
            getKeyFieldNames();
            if (fieldNames.size > 0) {
                const reducedFields = tableXMutationConditionInput.fields.filter(field => !fieldNames.has(field.name.value));
                const updatedInput = {
                    ...tableXMutationConditionInput,
                    fields: reducedFields,
                };
                ctx.putType(updatedInput);
            }
        }
    }
    typeExist(type, ctx) {
        return Boolean(type in ctx.nodeMap);
    }
    supportsConditions(context) {
        return context.getTransformerVersion() >= graphql_dynamodb_transformer_1.CONDITIONS_MINIMUM_VERSION;
    }
}
exports.KeyTransformer = KeyTransformer;
function keySchema(args) {
    if (args.fields.length > 1) {
        const condensedSortKey = condenseRangeKey(args.fields.slice(1));
        return [
            { AttributeName: args.fields[0], KeyType: 'HASH' },
            { AttributeName: condensedSortKey, KeyType: 'RANGE' },
        ];
    }
    else {
        return [{ AttributeName: args.fields[0], KeyType: 'HASH' }];
    }
}
function attributeTypeFromType(type, ctx) {
    const baseTypeName = graphql_transformer_common_1.getBaseType(type);
    const ofType = ctx.getType(baseTypeName);
    if (ofType && ofType.kind === graphql_1.Kind.ENUM_TYPE_DEFINITION) {
        return 'S';
    }
    return graphql_transformer_common_1.attributeTypeFromScalar(type);
}
function attributeDefinitions(args, def, ctx) {
    const fieldMap = new Map();
    for (const field of def.fields) {
        fieldMap.set(field.name.value, field);
    }
    if (args.fields.length > 2) {
        const hashName = args.fields[0];
        const condensedSortKey = condenseRangeKey(args.fields.slice(1));
        return [
            { AttributeName: hashName, AttributeType: attributeTypeFromType(fieldMap.get(hashName).type, ctx) },
            { AttributeName: condensedSortKey, AttributeType: 'S' },
        ];
    }
    else if (args.fields.length === 2) {
        const hashName = args.fields[0];
        const sortName = args.fields[1];
        return [
            { AttributeName: hashName, AttributeType: attributeTypeFromType(fieldMap.get(hashName).type, ctx) },
            { AttributeName: sortName, AttributeType: attributeTypeFromType(fieldMap.get(sortName).type, ctx) },
        ];
    }
    else {
        const fieldName = args.fields[0];
        return [{ AttributeName: fieldName, AttributeType: attributeTypeFromType(fieldMap.get(fieldName).type, ctx) }];
    }
}
function append(maybeList, item) {
    if (maybeList) {
        return [...maybeList, item];
    }
    return [item];
}
function getPrimaryKey(obj) {
    for (const directive of obj.directives) {
        if (directive.name.value === 'key' && !graphql_transformer_core_1.getDirectiveArguments(directive).name) {
            return directive;
        }
    }
}
function primaryIdFields(definition, keyFields) {
    return keyFields.map(keyFieldName => {
        const keyField = definition.fields.find(field => field.name.value === keyFieldName);
        return graphql_transformer_common_1.makeInputValueDefinition(keyFieldName, graphql_transformer_common_1.makeNonNullType(graphql_transformer_common_1.makeNamedType(graphql_transformer_common_1.getBaseType(keyField.type))));
    });
}
function replaceUpdateInput(definition, input, keyFields) {
    return {
        ...input,
        fields: input.fields.map(f => {
            if (keyFields.find(k => k === f.name.value)) {
                return graphql_transformer_common_1.makeInputValueDefinition(f.name.value, graphql_transformer_common_1.wrapNonNull(graphql_transformer_common_1.withNamedNodeNamed(f.type, graphql_transformer_common_1.getBaseType(f.type))));
            }
            return f;
        }),
    };
}
function replaceCreateInput(definition, input, keyFields) {
    return {
        ...input,
        fields: input.fields.filter(f => f.name.value !== 'id'),
    };
}
function replaceDeleteInput(definition, input, keyFields) {
    const idFields = primaryIdFields(definition, keyFields);
    const existingFields = input.fields.filter(f => !(idFields.find(pf => pf.name.value === f.name.value) || (graphql_transformer_common_1.getBaseType(f.type) === 'ID' && f.name.value === 'id')));
    return {
        ...input,
        fields: [...idFields, ...existingFields],
    };
}
function modelObjectKey(args, isMutation) {
    const argsPrefix = isMutation ? 'ctx.args.input' : 'ctx.args';
    if (args.fields.length > 2) {
        const rangeKeyFields = args.fields.slice(1);
        const condensedSortKey = condenseRangeKey(rangeKeyFields);
        const condensedSortKeyValue = condenseRangeKey(rangeKeyFields.map(keyField => `\${${argsPrefix}.${keyField}}`));
        return graphql_mapping_template_1.obj({
            [args.fields[0]]: graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDB($${argsPrefix}.${args.fields[0]})`),
            [condensedSortKey]: graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDB("${condensedSortKeyValue}")`),
        });
    }
    else if (args.fields.length === 2) {
        return graphql_mapping_template_1.obj({
            [args.fields[0]]: graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDB($${argsPrefix}.${args.fields[0]})`),
            [args.fields[1]]: graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDB($${argsPrefix}.${args.fields[1]})`),
        });
    }
    else if (args.fields.length === 1) {
        return graphql_mapping_template_1.obj({
            [args.fields[0]]: graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDB($${argsPrefix}.${args.fields[0]})`),
        });
    }
    throw new graphql_transformer_core_1.InvalidDirectiveError('@key directives must include at least one field.');
}
function ensureCompositeKeySnippet(dir) {
    const args = graphql_transformer_core_1.getDirectiveArguments(dir);
    const argsPrefix = 'ctx.args.input';
    if (args.fields.length > 2) {
        const rangeKeyFields = args.fields.slice(1);
        const condensedSortKey = condenseRangeKey(rangeKeyFields);
        const dynamoDBFriendlySortKeyName = graphql_transformer_common_1.toCamelCase(rangeKeyFields.map(f => graphql_transformer_common_1.graphqlName(f)));
        const condensedSortKeyValue = condenseRangeKey(rangeKeyFields.map(keyField => `\${${argsPrefix}.${keyField}}`));
        return graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw(`$util.isNull($${graphql_transformer_common_1.ResourceConstants.SNIPPETS.DynamoDBNameOverrideMap})`), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.DynamoDBNameOverrideMap), graphql_mapping_template_1.obj({
                [condensedSortKey]: graphql_mapping_template_1.str(dynamoDBFriendlySortKeyName),
            })), graphql_mapping_template_1.qref(`$${graphql_transformer_common_1.ResourceConstants.SNIPPETS.DynamoDBNameOverrideMap}.put("${condensedSortKey}", "${dynamoDBFriendlySortKeyName}")`)),
            graphql_mapping_template_1.qref(`$ctx.args.input.put("${condensedSortKey}","${condensedSortKeyValue}")`),
        ]));
    }
    return '';
}
function condenseRangeKey(fields) {
    return fields.join(graphql_transformer_common_1.ModelResourceIDs.ModelCompositeKeySeparator());
}
function makeQueryResolver(definition, directive, ctx) {
    const type = definition.name.value;
    const directiveArgs = graphql_transformer_core_1.getDirectiveArguments(directive);
    const index = directiveArgs.name;
    const fieldName = directiveArgs.queryField;
    const queryTypeName = ctx.getQueryTypeName();
    const requestVariable = 'QueryRequest';
    return new cloudform_types_1.AppSync.Resolver({
        ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
        DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(type), 'Name'),
        FieldName: fieldName,
        TypeName: queryTypeName,
        RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression([
            setQuerySnippet(definition, directive, ctx, false),
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref('limit'), graphql_mapping_template_1.ref(`util.defaultIfNull($context.args.limit, ${graphql_transformer_common_1.ResourceConstants.DEFAULT_PAGE_LIMIT})`)),
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(requestVariable), graphql_mapping_template_1.obj({
                version: graphql_mapping_template_1.str(graphql_mapping_template_1.RESOLVER_VERSION_ID),
                operation: graphql_mapping_template_1.str('Query'),
                limit: graphql_mapping_template_1.ref('limit'),
                query: graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression),
                index: graphql_mapping_template_1.str(index),
            })),
            graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.sortDirection)
                    && $ctx.args.sortDirection == "DESC"`), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.scanIndexForward`), graphql_mapping_template_1.bool(false)), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.scanIndexForward`), graphql_mapping_template_1.bool(true))),
            graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('context.args.nextToken'), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.nextToken`), graphql_mapping_template_1.ref('context.args.nextToken')), true),
            graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('context.args.filter'), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.filter`), graphql_mapping_template_1.ref('util.parseJson("$util.transform.toDynamoDBFilterExpression($ctx.args.filter)")')), true),
            graphql_mapping_template_1.raw(`$util.toJson($${requestVariable})`),
        ])),
        ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('ctx.error'), graphql_mapping_template_1.raw('$util.error($ctx.error.message, $ctx.error.type)')),
            graphql_mapping_template_1.raw('$util.toJson($ctx.result)'),
        ])),
    });
}
function setQuerySnippet(definition, directive, ctx, isListResolver) {
    const args = graphql_transformer_core_1.getDirectiveArguments(directive);
    const keys = args.fields;
    const keyTypes = keys.map(k => {
        const field = definition.fields.find(f => f.name.value === k);
        return attributeTypeFromType(field.type, ctx);
    });
    const expressions = [];
    if (keys.length === 1) {
        const sortDirectionValidation = graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.sortDirection)`), graphql_mapping_template_1.raw(`$util.error("sortDirection is not supported for List operations without a Sort key defined.", "InvalidArgumentsError")`));
        expressions.push(sortDirectionValidation);
    }
    else if (isListResolver === true && keys.length >= 1) {
        const sortDirectionValidation = graphql_mapping_template_1.iff(graphql_mapping_template_1.and([graphql_mapping_template_1.raw(`$util.isNull($ctx.args.${keys[0]})`), graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.sortDirection)`)]), graphql_mapping_template_1.raw(`$util.error("When providing argument 'sortDirection' you must also provide argument '${keys[0]}'.", "InvalidArgumentsError")`));
        expressions.push(sortDirectionValidation);
    }
    expressions.push(graphql_mapping_template_1.set(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression), graphql_mapping_template_1.obj({})), graphql_transformer_common_1.applyKeyExpressionForCompositeKey(keys, keyTypes, graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression));
    return graphql_mapping_template_1.block(`Set query expression for @key`, expressions);
}
function addHashField(definition, args, elems) {
    let hashFieldName = args.fields[0];
    const hashField = definition.fields.find(field => field.name.value === hashFieldName);
    const hashKey = graphql_transformer_common_1.makeInputValueDefinition(hashFieldName, graphql_transformer_common_1.makeNamedType(graphql_transformer_common_1.getBaseType(hashField.type)));
    return [hashKey, ...elems];
}
function addSimpleSortKey(ctx, definition, args, elems) {
    let sortKeyName = args.fields[1];
    const sortField = definition.fields.find(field => field.name.value === sortKeyName);
    const baseType = graphql_transformer_common_1.getBaseType(sortField.type);
    const resolvedTypeIfEnum = ctx.getType(baseType) ? 'String' : undefined;
    const resolvedType = resolvedTypeIfEnum ? resolvedTypeIfEnum : baseType;
    const hashKey = graphql_transformer_common_1.makeInputValueDefinition(sortKeyName, graphql_transformer_common_1.makeNamedType(graphql_transformer_common_1.ModelResourceIDs.ModelKeyConditionInputTypeName(resolvedType)));
    return [hashKey, ...elems];
}
function addCompositeSortKey(definition, args, elems) {
    let sortKeyNames = args.fields.slice(1);
    const compositeSortKeyName = graphql_transformer_common_1.toCamelCase(sortKeyNames);
    const hashKey = graphql_transformer_common_1.makeInputValueDefinition(compositeSortKeyName, graphql_transformer_common_1.makeNamedType(graphql_transformer_common_1.ModelResourceIDs.ModelCompositeKeyConditionInputTypeName(definition.name.value, graphql_transformer_common_1.toUpper(args.name || 'Primary'))));
    return [hashKey, ...elems];
}
function joinSnippets(lines) {
    return lines.join('\n');
}
function setSyncQueryMapSnippet(directive, isTable) {
    const args = graphql_transformer_core_1.getDirectiveArguments(directive);
    const keys = args.fields;
    const expressions = [];
    const index = isTable ? 'dbTable' : args.name;
    let key = '';
    if (keys.length === 1) {
        key = `${keys[0]}+null`;
    }
    else if (keys.length > 1) {
        key = keys.join('+');
    }
    expressions.push(graphql_mapping_template_1.raw(`$util.qr($QueryMap.put('${key}' , '${index}'))`), graphql_mapping_template_1.raw(`$util.qr($PkMap.put('${keys[0]}' , '${index}'))`));
    return graphql_mapping_template_1.block(`Set query expression for @key`, expressions);
}
function setSyncQueryFilterSnippet() {
    const expressions = [];
    expressions.push(graphql_mapping_template_1.compoundExpression([
        graphql_mapping_template_1.set(graphql_mapping_template_1.ref('filterArgsMap'), graphql_mapping_template_1.ref('ctx.args.filter.get("and")')),
        graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw(`!$util.isNullOrEmpty($filterArgsMap)`), graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref('json'), graphql_mapping_template_1.raw(`$filterArgsMap`)),
            graphql_mapping_template_1.forEach(graphql_mapping_template_1.ref('item'), graphql_mapping_template_1.ref(`json`), [
                graphql_mapping_template_1.set(graphql_mapping_template_1.ref('ind'), graphql_mapping_template_1.ref('foreach.index')),
                graphql_mapping_template_1.forEach(graphql_mapping_template_1.ref('entry'), graphql_mapping_template_1.ref('item.entrySet()'), [
                    graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`$ind == 0 && !$util.isNullOrEmpty($entry.value.eq) && !$util.isNullOrEmpty($PkMap.get($entry.key))`), graphql_mapping_template_1.compoundExpression([
                        graphql_mapping_template_1.set(graphql_mapping_template_1.ref('pk'), graphql_mapping_template_1.ref('entry.key')),
                        graphql_mapping_template_1.set(graphql_mapping_template_1.ref('scan'), graphql_mapping_template_1.bool(false)),
                        graphql_mapping_template_1.raw('$util.qr($ctx.args.put($pk,$entry.value.eq))'),
                        graphql_mapping_template_1.set(graphql_mapping_template_1.ref('index'), graphql_mapping_template_1.ref('PkMap.get($pk)')),
                    ])),
                    graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw('$ind == 1 && !$util.isNullOrEmpty($pk) && !$util.isNullOrEmpty($QueryMap.get("${pk}+$entry.key"))'), graphql_mapping_template_1.compoundExpression([
                        graphql_mapping_template_1.set(graphql_mapping_template_1.ref('sk'), graphql_mapping_template_1.ref('entry.key')),
                        graphql_mapping_template_1.raw('$util.qr($ctx.args.put($sk,$entry.value))'),
                        graphql_mapping_template_1.set(graphql_mapping_template_1.ref('index'), graphql_mapping_template_1.ref('QueryMap.get("${pk}+$sk")')),
                    ]), graphql_mapping_template_1.iff(graphql_mapping_template_1.raw('$ind > 0'), graphql_mapping_template_1.qref('$filterMap.put($entry.key,$entry.value)'))),
                ]),
            ]),
        ]), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`filterMap`), graphql_mapping_template_1.raw(`$ctx.args.filter`))),
    ]));
    return graphql_mapping_template_1.block(`Set query expression for @key`, expressions);
}
function generateSyncResolverInit() {
    const expressions = [];
    expressions.push(graphql_mapping_template_1.set(graphql_mapping_template_1.ref('index'), graphql_mapping_template_1.str('')), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('scan'), graphql_mapping_template_1.bool(true)), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('filterMap'), graphql_mapping_template_1.obj({})), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('QueryMap'), graphql_mapping_template_1.obj({})), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('PkMap'), graphql_mapping_template_1.obj({})), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('filterArgsMap'), graphql_mapping_template_1.obj({})));
    return graphql_mapping_template_1.block(`Set map initialization for @key`, expressions);
}
function setSyncKeyExpressionForHashKey(queryExprReference) {
    const expressions = [];
    expressions.push(graphql_mapping_template_1.set(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression), graphql_mapping_template_1.obj({})), graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`!$util.isNull($pk)`), graphql_mapping_template_1.compoundExpression([
        graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expression`), graphql_mapping_template_1.str(`#pk = :pk`)),
        graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expressionNames`), graphql_mapping_template_1.obj({ [`#pk`]: graphql_mapping_template_1.str('$pk') })),
        graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expressionValues`), graphql_mapping_template_1.obj({ [`:pk`]: graphql_mapping_template_1.ref('util.parseJson($util.dynamodb.toDynamoDBJson($ctx.args.get($pk)))') })),
    ])));
    return graphql_mapping_template_1.block(`Set Primary Key initialization @key`, expressions);
}
function setSyncKeyExpressionForRangeKey(queryExprReference) {
    return graphql_mapping_template_1.block('Applying Key Condition', [
        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.get($sk)) && !$util.isNull($ctx.args.get($sk).beginsWith)`), graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expression`), graphql_mapping_template_1.raw(`"$${queryExprReference}.expression AND begins_with(#sortKey, :sortKey)"`)),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionNames.put("#sortKey", $sk)`),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionValues.put(":sortKey", $util.parseJson($util.dynamodb.toDynamoDBJson($ctx.args.get($sk).beginsWith)))`),
        ])),
        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.get($sk)) && !$util.isNull($ctx.args.get($sk).between)`), graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expression`), graphql_mapping_template_1.raw(`"$${queryExprReference}.expression AND #sortKey BETWEEN :sortKey0 AND :sortKey1"`)),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionNames.put("#sortKey", $sk)`),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionValues.put(":sortKey", $util.parseJson($util.dynamodb.toDynamoDBJson($ctx.args.get($sk).between[0])))`),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionValues.put(":sortKey", $util.parseJson($util.dynamodb.toDynamoDBJson($ctx.args.get($sk).between[1])))`),
        ])),
        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.get($sk)) && !$util.isNull($ctx.args.get($sk).eq)`), graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expression`), graphql_mapping_template_1.raw(`"$${queryExprReference}.expression AND #sortKey = :sortKey"`)),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionNames.put("#sortKey", $sk)`),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionValues.put(":sortKey", $util.parseJson($util.dynamodb.toDynamoDBJson($ctx.args.get($sk).eq)))`),
        ])),
        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.get($sk)) && !$util.isNull($ctx.args.get($sk).lt)`), graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expression`), graphql_mapping_template_1.raw(`"$${queryExprReference}.expression AND #sortKey < :sortKey"`)),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionNames.put("#sortKey", $sk)`),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionValues.put(":sortKey", $util.parseJson($util.dynamodb.toDynamoDBJson($ctx.args.get($sk).lt)))`),
        ])),
        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.get($sk)) && !$util.isNull($ctx.args.get($sk).le)`), graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expression`), graphql_mapping_template_1.raw(`"$${queryExprReference}.expression AND #sortKey <= :sortKey"`)),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionNames.put("#sortKey", $sk)`),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionValues.put(":sortKey", $util.parseJson($util.dynamodb.toDynamoDBJson($ctx.args.get($sk).le)))`),
        ])),
        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.get($sk)) && !$util.isNull($ctx.args.get($sk).gt)`), graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expression`), graphql_mapping_template_1.raw(`"$${queryExprReference}.expression AND #sortKey > :sortKey"`)),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionNames.put("#sortKey", $sk)`),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionValues.put(":sortKey", $util.parseJson($util.dynamodb.toDynamoDBJson($ctx.args.get($sk).gt)))`),
        ])),
        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.get($sk)) && !$util.isNull($ctx.args.get($sk).ge)`), graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${queryExprReference}.expression`), graphql_mapping_template_1.raw(`"$${queryExprReference}.expression AND #sortKey >= :sortKey"`)),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionNames.put("#sortKey", $sk)`),
            graphql_mapping_template_1.qref(`$${queryExprReference}.expressionValues.put(":sortKey", $util.parseJson($util.dynamodb.toDynamoDBJson($ctx.args.get($sk).ge)))`),
        ])),
    ]);
}
function makeSyncQueryResolver() {
    const requestVariable = 'QueryRequest';
    const expressions = [];
    expressions.push(graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw('!$scan'), graphql_mapping_template_1.compoundExpression([
        graphql_mapping_template_1.set(graphql_mapping_template_1.ref('limit'), graphql_mapping_template_1.ref(`util.defaultIfNull($context.args.limit, ${graphql_transformer_common_1.ResourceConstants.DEFAULT_PAGE_LIMIT})`)),
        graphql_mapping_template_1.set(graphql_mapping_template_1.ref(requestVariable), graphql_mapping_template_1.obj({
            version: graphql_mapping_template_1.str(graphql_mapping_template_1.RESOLVER_VERSION_ID),
            operation: graphql_mapping_template_1.str('Sync'),
            limit: graphql_mapping_template_1.ref('limit'),
            query: graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression),
        })),
        graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.sortDirection)
                    && $ctx.args.sortDirection == "DESC"`), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.scanIndexForward`), graphql_mapping_template_1.bool(false)), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.scanIndexForward`), graphql_mapping_template_1.bool(true))),
        graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('context.args.nextToken'), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.nextToken`), graphql_mapping_template_1.ref('context.args.nextToken')), true),
        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw('!$util.isNullOrEmpty($filterMap)'), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.filter`), graphql_mapping_template_1.ref('util.parseJson($util.transform.toDynamoDBFilterExpression($filterMap))'))),
        graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`$index != "dbTable"`), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.index`), graphql_mapping_template_1.ref('index'))),
        graphql_mapping_template_1.raw(`$util.toJson($${requestVariable})`),
    ]), graphql_mapping_template_1.DynamoDBMappingTemplate.syncItem({
        filter: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw('!$util.isNullOrEmpty($ctx.args.filter)'), graphql_mapping_template_1.ref('util.transform.toDynamoDBFilterExpression($ctx.args.filter)'), graphql_mapping_template_1.nul()),
        limit: graphql_mapping_template_1.ref(`util.defaultIfNull($ctx.args.limit, ${graphql_transformer_common_1.ResourceConstants.DEFAULT_SYNC_QUERY_PAGE_LIMIT})`),
        lastSync: graphql_mapping_template_1.ref('util.toJson($util.defaultIfNull($ctx.args.lastSync, null))'),
        nextToken: graphql_mapping_template_1.ref('util.toJson($util.defaultIfNull($ctx.args.nextToken, null))'),
    })));
    return graphql_mapping_template_1.block(` Set query expression for @key`, expressions);
}
function constructSyncResolver(directive, ctx, syncResolver, isTable) {
    if (ctx.metadata.has(graphql_transformer_common_1.ResourceConstants.SNIPPETS.SyncResolverKey)) {
        const resolverMap = ctx.metadata.get(graphql_transformer_common_1.ResourceConstants.SNIPPETS.SyncResolverKey);
        if (resolverMap.has(syncResolver)) {
            const prevSnippet = resolverMap.get(syncResolver);
            resolverMap.set(syncResolver, joinSnippets([prevSnippet, graphql_mapping_template_1.print(setSyncQueryMapSnippet(directive, isTable))]));
        }
        else {
            resolverMap.set(syncResolver, graphql_mapping_template_1.print(setSyncQueryMapSnippet(directive, isTable)));
        }
    }
}
//# sourceMappingURL=KeyTransformer.js.map