"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const dns_validated_certificate_1 = require("../lib/dns-validated-certificate");
module.exports = {
    'creates CloudFormation Custom Resource'(test) {
        const stack = new core_1.Stack();
        const exampleDotComZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
            zoneName: 'example.com',
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            hostedZone: exampleDotComZone,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
            DomainName: 'test.example.com',
            ServiceToken: {
                'Fn::GetAtt': [
                    'CertificateCertificateRequestorFunction5E845413',
                    'Arn',
                ],
            },
            HostedZoneId: {
                Ref: 'ExampleDotCom4D1B83AA',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Handler: 'index.certificateRequestHandler',
            Runtime: 'nodejs10.x',
            Timeout: 900,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyName: 'CertificateCertificateRequestorFunctionServiceRoleDefaultPolicy3C8845BC',
            Roles: [
                {
                    Ref: 'CertificateCertificateRequestorFunctionServiceRoleC04C13DA',
                },
            ],
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: [
                            'acm:RequestCertificate',
                            'acm:DescribeCertificate',
                            'acm:DeleteCertificate',
                        ],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: 'route53:GetChange',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: 'route53:changeResourceRecordSets',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':route53:::hostedzone/',
                                    { Ref: 'ExampleDotCom4D1B83AA' },
                                ],
                            ],
                        },
                    },
                ],
            },
        }));
        test.done();
    },
    'adds validation error on domain mismatch'(test) {
        const stack = new core_1.Stack();
        const helloDotComZone = new aws_route53_1.PublicHostedZone(stack, 'HelloDotCom', {
            zoneName: 'hello.com',
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'example.com',
            hostedZone: helloDotComZone,
        });
        test.throws(() => {
            assert_1.SynthUtils.synthesize(stack);
        }, /DNS zone hello.com is not authoritative for certificate domain name example.com/);
        test.done();
    },
    'does not try to validate unresolved tokens'(test) {
        const stack = new core_1.Stack();
        const helloDotComZone = new aws_route53_1.PublicHostedZone(stack, 'HelloDotCom', {
            zoneName: core_1.Token.asString('hello.com'),
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'hello.com',
            hostedZone: helloDotComZone,
        });
        test.doesNotThrow(() => {
            assert_1.SynthUtils.synthesize(stack);
        });
        test.done();
    },
    'test root certificate'(test) {
        const stack = new core_1.Stack();
        const exampleDotComZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
            zoneName: 'example.com',
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'example.com',
            hostedZone: exampleDotComZone,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'CertCertificateRequestorFunction98FDF273',
                    'Arn',
                ],
            },
            DomainName: 'example.com',
            HostedZoneId: {
                Ref: 'ExampleDotCom4D1B83AA',
            },
        }));
        test.done();
    },
    'works with imported zone'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { account: '12345678', region: 'us-blue-5' },
        });
        const imported = aws_route53_1.HostedZone.fromLookup(stack, 'ExampleDotCom', {
            domainName: 'mydomain.com',
        });
        // WHEN
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'mydomain.com',
            hostedZone: imported,
            route53Endpoint: 'https://api.route53.xxx.com',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'CertCertificateRequestorFunction98FDF273',
                    'Arn',
                ],
            },
            DomainName: 'mydomain.com',
            HostedZoneId: 'DUMMY',
            Route53Endpoint: 'https://api.route53.xxx.com',
        }));
        test.done();
    },
    'works with imported role'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { account: '12345678', region: 'us-blue-5' },
        });
        const helloDotComZone = new aws_route53_1.PublicHostedZone(stack, 'HelloDotCom', {
            zoneName: 'hello.com',
        });
        const role = iam.Role.fromRoleArn(stack, 'Role', 'arn:aws:iam::account-id:role/role-name');
        // WHEN
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'hello.com',
            hostedZone: helloDotComZone,
            customResourceRole: role,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Role: 'arn:aws:iam::account-id:role/role-name',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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