"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessCluster = exports.AuroraCapacityUnit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const endpoint_1 = require("./endpoint");
const perms_1 = require("./perms");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * (experimental) Aurora capacity units (ACUs).
 *
 * Each ACU is a combination of processing and memory capacity.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.architecture
 * @experimental
 */
var AuroraCapacityUnit;
(function (AuroraCapacityUnit) {
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_1"] = 1] = "ACU_1";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_2"] = 2] = "ACU_2";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_4"] = 4] = "ACU_4";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_8"] = 8] = "ACU_8";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_16"] = 16] = "ACU_16";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_32"] = 32] = "ACU_32";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_64"] = 64] = "ACU_64";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_128"] = 128] = "ACU_128";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_192"] = 192] = "ACU_192";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_256"] = 256] = "ACU_256";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_384"] = 384] = "ACU_384";
})(AuroraCapacityUnit = exports.AuroraCapacityUnit || (exports.AuroraCapacityUnit = {}));
/**
 * New or imported Serverless Cluster
 */
class ServerlessClusterBase extends core_1.Resource {
    /**
     * (experimental) The ARN of the cluster.
     *
     * @experimental
     */
    get clusterArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'cluster',
            sep: ':',
            resourceName: this.clusterIdentifier,
        });
    }
    /**
     * (experimental) Grant the given identity to access to the Data API, including read access to the secret attached to the cluster if present.
     *
     * @param grantee The principal to grant access to.
     * @experimental
     */
    grantDataApiAccess(grantee) {
        var _b;
        if (this.enableDataApi === false) {
            throw new Error('Cannot grant Data API access when the Data API is disabled');
        }
        this.enableDataApi = true;
        const ret = iam.Grant.addToPrincipal({
            grantee,
            actions: perms_1.DATA_API_ACTIONS,
            resourceArns: ['*'],
            scope: this,
        });
        (_b = this.secret) === null || _b === void 0 ? void 0 : _b.grantRead(grantee);
        return ret;
    }
    /**
     * (experimental) Renders the secret attachment target specifications.
     *
     * @experimental
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
/**
 * (experimental) Create an Aurora Serverless Cluster.
 *
 * @experimental
 * @resource AWS::RDS::DBCluster
 */
class ServerlessCluster extends ServerlessClusterBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        this.enableDataApi = props.enableDataApi;
        const { subnetIds } = this.vpc.selectSubnets(this.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_b = props.subnetGroup) !== null && _b !== void 0 ? _b : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            removalPolicy: props.removalPolicy === core_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        if (props.backupRetention) {
            const backupRetentionDays = props.backupRetention.toDays();
            if (backupRetentionDays < 1 || backupRetentionDays > 35) {
                throw new Error(`backup retention period must be between 1 and 35 days. received: ${backupRetentionDays}`);
            }
        }
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            parameterGroup: props.parameterGroup,
        });
        const clusterParameterGroup = (_c = props.parameterGroup) !== null && _c !== void 0 ? _c : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        const securityGroups = (_d = props.securityGroups) !== null && _d !== void 0 ? _d : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: this.vpc,
            }),
        ];
        const clusterIdentifier = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER)
            ? (_e = props.clusterIdentifier) === null || _e === void 0 ? void 0 : _e.toLowerCase() : props.clusterIdentifier;
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            backupRetentionPeriod: (_f = props.backupRetention) === null || _f === void 0 ? void 0 : _f.toDays(),
            databaseName: props.defaultDatabaseName,
            dbClusterIdentifier: clusterIdentifier,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            engine: props.engine.engineType,
            engineVersion: (_g = props.engine.engineVersion) === null || _g === void 0 ? void 0 : _g.fullVersion,
            engineMode: 'serverless',
            enableHttpEndpoint: core_1.Lazy.anyValue({ produce: () => this.enableDataApi }),
            kmsKeyId: (_h = props.storageEncryptionKey) === null || _h === void 0 ? void 0 : _h.keyArn,
            masterUsername: credentials.username,
            masterUserPassword: (_j = credentials.password) === null || _j === void 0 ? void 0 : _j.toString(),
            scalingConfiguration: props.scaling ? this.renderScalingConfiguration(props.scaling) : undefined,
            storageEncrypted: true,
            vpcSecurityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy((_k = props.removalPolicy) !== null && _k !== void 0 ? _k : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
    }
    /**
     * (experimental) Import an existing DatabaseCluster from properties.
     *
     * @experimental
     */
    static fromServerlessClusterAttributes(scope, id, attrs) {
        return new ImportedServerlessCluster(scope, id, attrs);
    }
    /**
     * (experimental) Adds the single user rotation of the master password to this cluster.
     *
     * @experimental
     */
    addRotationSingleUser(options = {}) {
        var _b;
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
            ...options,
            excludeCharacters: (_b = options.excludeCharacters) !== null && _b !== void 0 ? _b : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        });
    }
    /**
     * (experimental) Adds the multi user rotation to this cluster.
     *
     * @experimental
     */
    addRotationMultiUser(id, options) {
        var _b;
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...options,
            excludeCharacters: (_b = options.excludeCharacters) !== null && _b !== void 0 ? _b : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    renderScalingConfiguration(options) {
        var _b;
        const minCapacity = options.minCapacity;
        const maxCapacity = options.maxCapacity;
        if (minCapacity && maxCapacity && minCapacity > maxCapacity) {
            throw new Error('maximum capacity must be greater than or equal to minimum capacity.');
        }
        const secondsToAutoPause = (_b = options.autoPause) === null || _b === void 0 ? void 0 : _b.toSeconds();
        if (secondsToAutoPause && (secondsToAutoPause < 300 || secondsToAutoPause > 86400)) {
            throw new Error('auto pause time must be between 5 minutes and 1 day.');
        }
        return {
            autoPause: (secondsToAutoPause === 0) ? false : true,
            minCapacity: options.minCapacity,
            maxCapacity: options.maxCapacity,
            secondsUntilAutoPause: (secondsToAutoPause === 0) ? undefined : secondsToAutoPause,
        };
    }
}
exports.ServerlessCluster = ServerlessCluster;
_a = JSII_RTTI_SYMBOL_1;
ServerlessCluster[_a] = { fqn: "@aws-cdk/aws-rds.ServerlessCluster", version: "1.98.0" };
/**
 * Represents an imported database cluster.
 */
class ImportedServerlessCluster extends ServerlessClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.enableDataApi = true;
        this.clusterIdentifier = attrs.clusterIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this.secret = attrs.secret;
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
}
//# sourceMappingURL=data:application/json;base64,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