# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': '.'}

packages = \
['pyramid_openapi3']

package_data = \
{'': ['*'], 'pyramid_openapi3': ['static/*']}

install_requires = \
['openapi-core>=0.13.4,<0.14', 'openapi-schema-validator', 'pyramid>=1.10.7']

setup_kwargs = {
    'name': 'pyramid-openapi3',
    'version': '0.14.1',
    'description': 'Pyramid addon for OpenAPI3 validation of requests and responses.',
    'long_description': '## Validate [Pyramid](https://trypyramid.com) views against an [OpenAPI 3.0](https://swagger.io/specification/) document\n\n<p align="center">\n  <img height="200" src="https://github.com/Pylons/pyramid_openapi3/blob/main/header.jpg?raw=true" />\n</p>\n\n<p align="center">\n  <a href="https://circleci.com/gh/Pylons/pyramid_openapi3">\n    <img alt="CircleCI for pyramid_openapi3 (main branch)"\n         src="https://circleci.com/gh/Pylons/pyramid_openapi3.svg?style=shield">\n  </a>\n  <img alt="Test coverage (main branch)"\n       src="https://img.shields.io/badge/tests_coverage-100%25-brightgreen.svg">\n  <img alt="Test coverage (main branch)"\n       src="https://img.shields.io/badge/types_coverage-100%25-brightgreen.svg">\n  <a href="https://pypi.org/project/pyramid_openapi3/">\n    <img alt="latest version of pyramid_openapi3 on PyPI"\n         src="https://img.shields.io/pypi/v/pyramid_openapi3.svg">\n  </a>\n  <a href="https://pypi.org/project/pyramid_openapi3/">\n    <img alt="Supported Python versions"\n         src="https://img.shields.io/pypi/pyversions/pyramid_openapi3.svg">\n  </a>\n  <a href="https://github.com/Pylons/pyramid_openapi3/blob/main/LICENSE">\n    <img alt="License: MIT"\n         src="https://img.shields.io/badge/License-MIT-yellow.svg">\n  </a>\n  <a href="https://github.com/Pylons/pyramid_openapi3/graphs/contributors">\n    <img alt="Built by these great folks!"\n         src="https://img.shields.io/github/contributors/Pylons/pyramid_openapi3.svg">\n  </a>\n</p>\n\n## Peace of Mind\n\nThe reason this package exists is to give you peace of mind when providing a RESTful API. Instead of chasing down preventable bugs and saying sorry to consumers, you can focus on more important things in life.\n\n- Your **API documentation is never out-of-date**, since it is generated out of the API document that you write.\n- The documentation comes with **_try-it-out_ examples** for every endpoint in your API. You don\'t have to provide (and maintain) `curl` commands to showcase how your API works. Users can try it themselves, right in their browsers.\n- Your **API document is always valid**, since your Pyramid app won\'t even start if the document does not comply with the OpenAPI 3.0 specification.\n- Automatic request **payload validation and sanitization**. Your views do not require any code for validation and input sanitation. Your view code only deals with business logic. Tons of tests never need to be written since every request, and its payload, is validated against your API document before it reaches your view code.\n- Your API **responses always match your API document**. Every response from your view is validated against your document and a `500 Internal Server Error` is returned if the response does not exactly match what your document says the output of a certain API endpoint should be. This decreases the effects of [Hyrum\'s Law](https://www.hyrumslaw.com).\n- **A single source of truth**. Because of the checks outlined above, you can be sure that whatever your API document says is in fact what is going on in reality. You have a single source of truth to consult when asking an API related question, such as "Remind me again, which fields are returned by the endpoint `/user/info`?".\n- Based on [Pyramid](https://trypyramid.com), a **mature Python Web framework**. Companies such as Mozilla, Yelp, RollBar and SurveyMonkey [trust Pyramid](https://trypyramid.com/community-powered-by-pyramid.html), and the new [pypi.org](https://github.com/pypa/warehouse) runs on Pyramid, too. Pyramid is thoroughly [tested](https://github.com/Pylons/Pyramid/actions?query=workflow%3A%22Build+and+test%22) and [documented](https://docs.pylonsproject.org/projects/pyramid/en/latest/), providing flexibility, performance, and a large ecosystem of [high-quality add-ons](https://trypyramid.com/extending-pyramid.html).\n\n<p align="center">\n<a href="https://www.youtube.com/watch?v=P0zNxrDO0sE&amp;t=1061" title="Building Robust APIs" rel="nofollow" class="rich-diff-level-one"><img src="https://user-images.githubusercontent.com/311580/97364772-6d246a80-189c-11eb-84f2-a0ad23236003.png" alt="Building Robust APIs" style="max-width:100%;"></a>\n</p>\n\n## Features\n\n- Validates your API document (for example, `openapi.yaml` or `openapi.json`) against the OpenAPI 3.0 specification using the [openapi-spec-validator](https://github.com/p1c2u/openapi-spec-validator).\n- Generates and serves the [Swagger try-it-out documentation](https://swagger.io/tools/swagger-ui/) for your API.\n- Validates incoming requests *and* outgoing responses against your API document using [openapi-core](https://github.com/p1c2u/openapi-core).\n\n## Getting started\n\n1. Declare `pyramid_openapi3` as a dependency in your Pyramid project.\n\n2. Include the following lines:\n\n```python\nconfig.include("pyramid_openapi3")\nconfig.pyramid_openapi3_spec(\'openapi.yaml\', route=\'/api/v1/openapi.yaml\')\nconfig.pyramid_openapi3_add_explorer(route=\'/api/v1/\')\n```\n\n3. Use the `openapi` [view predicate](https://docs.pylonsproject.org/projects/pyramid/en/latest/narr/viewconfig.html#view-configuration-parameters) to enable request/response validation:\n\n```python\n@view_config(route_name="foobar", openapi=True, renderer=\'json\')\ndef myview(request):\n    return request.openapi_validated.parameters\n```\n\nFor requests, `request.openapi_validated` is available with two fields: `parameters` and `body`.\nFor responses, if the payload does not match the API document, an exception is raised.\n\n## Advanced configuration\n\n### Relative File References in Spec\n\nA feature introduced in OpenAPI3 is the ability to use `$ref` links to external files (https://github.com/OAI/OpenAPI-Specification/blob/main/versions/3.0.0.md#referenceObject).\n\nTo use this, you must ensure that you have all of your spec files in a given directory (ensure that you do not have any code in this directory as all the files in it are exposed as static files), then **replace** the `pyramid_openapi3_spec` call that you did in [Getting Started](#getting-started) with the following:\n\n```python\nconfig.pyramid_openapi3_spec_directory(\'path/to/openapi.yaml\', route=\'/api/v1/spec\')\n```\n\nSome notes:\n\n- Do not set the `route` of your `pyramid_openapi3_spec_directory` to the same value as the `route` of `pyramid_openapi3_add_explorer`.\n- The `route` that you set for `pyramid_openapi3_spec_directory` should not contain any file extensions, as this becomes the root for all of the files in your specified `filepath`.\n- You cannot use `pyramid_openapi3_spec_directory` and `pyramid_openapi3_spec` in the same app.\n\n### Endpoints / Request / Response Validation\n\nProvided with `pyramid_openapi3` are a few validation features:\n * incoming request validation (i.e., what a client sends to your app)\n * outgoing response validation (i.e., what your app sends to a client)\n * endpoint validation (i.e., your app registers routes for all defined API endpoints)\n\nThese features are enabled as a default, but you can disable them if you need to:\n\n```python\nconfig.registry.settings["pyramid_openapi3.enable_endpoint_validation"] = False\nconfig.registry.settings["pyramid_openapi3.enable_request_validation"] = False\nconfig.registry.settings["pyramid_openapi3.enable_response_validation"] = False\n```\n\n> **Warning:**\nDisabling request validation will result in `request.openapi_validated` no longer being available to use.\n\n### Register Pyramid\'s Routes\n\nYou can register routes in your pyramid application.\nFirst, write the `x-pyramid-route-name` extension in the PathItem of the OpenAPI schema.\n\n```yaml\npaths:\n  /foo:\n    x-pyramid-route-name: foo_route\n    get:\n      responses:\n        200:\n          description: GET foo\n```\n\nThen put the config directive `pyramid_openapi3_register_routes` in the app_factory of your application.\n\n```python\nconfig.pyramid_openapi3_register_routes()\n```\n\nThis means is equals to\n\n```python\nconfig.add_route("foo_route", pattern="/foo")\n```\n\n## Demo / Examples\n\nThere are three examples provided with this package:\n* A fairly simple [single-file app providing a Hello World API](https://github.com/Pylons/pyramid_openapi3/tree/main/examples/singlefile).\n* A slightly more [built-out app providing a TODO app API](https://github.com/Pylons/pyramid_openapi3/tree/main/examples/todoapp).\n* Another TODO app API, defined using a [YAML spec split into multiple files](https://github.com/Pylons/pyramid_openapi3/tree/main/examples/splitfile).\n\nBoth examples come with tests that exhibit pyramid_openapi\'s error handling and validation capabilities.\n\nA **fully built-out app**, with 100% test coverage, providing a [RealWorld.io](https://realworld.io) API is available at [niteoweb/pyramid-realworld-example-app](https://github.com/niteoweb/pyramid-realworld-example-app). It is a Heroku-deployable Pyramid app that provides an API for a Medium.com-like social app. You are encouraged to use it as a scaffold for your next project.\n\n\n## Design defense\n\nThe authors of pyramid_openapi3 believe that the approach of validating a manually-written API document is superior to the approach of generating the API document from Python code. Here are the reasons:\n\n1.  Both generation and validation against a document are lossy processes. The underlying libraries running the generation/validation will always have something missing. Either a feature from the latest OpenAPI specification, or an implementation bug. Having to fork the underlying library in order to generate the part of your API document that might only be needed for the frontend is unfortunate.\n\n    Validation on the other hand allows one to skip parts of validation that are not supported yet, and not block a team from shipping the document.\n\n2.  The validation approach does sacrifice DRY-ness, and one has to write the API document and then the (view) code in Pyramid. It feels a bit redundant at first. However, this provides a clear separation between the intent and the implementation.\n\n3.  The generation approach has the drawback of having to write Python code even for parts of the API document that the Pyramid backend does not handle, as it might be handled by a different system, or be specific only to documentation or only to the client side of the API. This bloats your Pyramid codebase with code that does not belong there.\n\n\n## Running tests\n\nYou need to have [poetry](https://python-poetry.org/) and Python 3.7 through 3.11 installed on your machine.\n\nAlternatively, if you use [nix](https://nix.dev/tutorials/declarative-and-reproducible-developer-environments), run `nix-shell` to drop into a shell that has everything prepared for development.\n\nThen you can run:\n\n```shell\nmake tests\n```\n\n## Related packages\n\nThese packages tackle the same problem-space:\n\n- [pyramid_oas3](https://github.com/kazuki/pyramid-oas3) seems to do things very similarly to pyramid_openapi3, but the documentation is not in English and we sadly can\'t fully understand what it does by just reading the code.\n- [pyramid_swagger](https://github.com/striglia/pyramid_swagger) does a similar\n  thing, but for Swagger 2.0 documents.\n- [connexion](https://github.com/zalando/connexion) takes the same "write spec first, code second" approach as pyramid_openapi3, but is based on Flask.\n- [bottle-swagger](https://github.com/ampedandwired/bottle-swagger) takes the same "write spec first, code second" approach too, but is based on Bottle.\n- [pyramid_apispec](https://github.com/ergo/pyramid_apispec) uses generation with\n  help of apispec and the marshmallow validation library. See above [why we prefer validation instead of generation](#design-defense).\n\n## Deprecation policy\n\nWe do our best to follow the rules below.\n\n* Support the latest few releases of Python, currently Python 3.7 through 3.11.\n* Support the latest few releases of Pyramid, currently 1.10.7 through 2.0.\n* Support the latest few releases of `openapi-core`, currently 0.13.4 through 0.13.8.\n* See `poetry.lock` for a frozen-in-time known-good-set of all dependencies.\n\n## Use in the wild\n\nA couple of projects that use pyramid_openapi3 in production:\n\n- [WooCart API](https://app.woocart.com/api/v1) - User control panel for WooCart Managed WooCommerce service.\n- [Kafkai API](https://app.kafkai.com/api/v1) - User control panel for Kafkai text generation service.\n- [Open on-chain data API](https://tradingstrategy.ai/api/explorer/) - Decentralised exchange and blockchain trading data open API\n- [Pareto Security Team Dashboard API](https://dash.paretosecurity.app/api/v1) - Team Dashboard for Pareto Security macOS security app\n',
    'author': 'Niteo',
    'author_email': 'info@niteo.co',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/Pylons/pyramid_openapi3',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
