"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobExecutable = exports.JobType = exports.PythonVersion = exports.JobLanguage = exports.GlueVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * AWS Glue version determines the versions of Apache Spark and Python that are available to the job.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/add-job.html.
 *
 * If you need to use a GlueVersion that doesn't exist as a static member, you
 * can instantiate a `GlueVersion` object, e.g: `GlueVersion.of('1.5')`.
 */
class GlueVersion {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom Glue version
     * @param version custom version
     */
    static of(version) {
        return new GlueVersion(version);
    }
}
exports.GlueVersion = GlueVersion;
_a = JSII_RTTI_SYMBOL_1;
GlueVersion[_a] = { fqn: "@aws-cdk/aws-glue-alpha.GlueVersion", version: "2.51.0-alpha.0" };
/**
 * Glue version using Spark 2.2.1 and Python 2.7
 */
GlueVersion.V0_9 = new GlueVersion('0.9');
/**
 * Glue version using Spark 2.4.3, Python 2.7 and Python 3.6
 */
GlueVersion.V1_0 = new GlueVersion('1.0');
/**
 * Glue version using Spark 2.4.3 and Python 3.7
 */
GlueVersion.V2_0 = new GlueVersion('2.0');
/**
 * Glue version using Spark 3.1.1 and Python 3.7
 */
GlueVersion.V3_0 = new GlueVersion('3.0');
/**
 * Runtime language of the Glue job
 */
var JobLanguage;
(function (JobLanguage) {
    /**
     * Scala
     */
    JobLanguage["SCALA"] = "scala";
    /**
     * Python
     */
    JobLanguage["PYTHON"] = "python";
})(JobLanguage = exports.JobLanguage || (exports.JobLanguage = {}));
/**
 * Python version
 */
var PythonVersion;
(function (PythonVersion) {
    /**
     * Python 2 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["TWO"] = "2";
    /**
     * Python 3 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["THREE"] = "3";
    /**
     * Python 3.9 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["THREE_NINE"] = "3.9";
})(PythonVersion = exports.PythonVersion || (exports.PythonVersion = {}));
/**
 * The job type.
 *
 * If you need to use a JobType that doesn't exist as a static member, you
 * can instantiate a `JobType` object, e.g: `JobType.of('other name')`.
 */
class JobType {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom type name
     * @param name type name
     */
    static of(name) {
        return new JobType(name);
    }
}
exports.JobType = JobType;
_b = JSII_RTTI_SYMBOL_1;
JobType[_b] = { fqn: "@aws-cdk/aws-glue-alpha.JobType", version: "2.51.0-alpha.0" };
/**
 * Command for running a Glue ETL job.
 */
JobType.ETL = new JobType('glueetl');
/**
 * Command for running a Glue streaming job.
 */
JobType.STREAMING = new JobType('gluestreaming');
/**
 * Command for running a Glue python shell job.
 */
JobType.PYTHON_SHELL = new JobType('pythonshell');
/**
 * The executable properties related to the Glue job's GlueVersion, JobType and code
 */
class JobExecutable {
    constructor(config) {
        if (JobType.PYTHON_SHELL === config.type) {
            if (config.language !== JobLanguage.PYTHON) {
                throw new Error('Python shell requires the language to be set to Python');
            }
            if ([GlueVersion.V0_9, GlueVersion.V2_0, GlueVersion.V3_0].includes(config.glueVersion)) {
                throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support Python Shell`);
            }
        }
        if (config.extraJarsFirst && [GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support extraJarsFirst`);
        }
        if (config.pythonVersion === PythonVersion.TWO && ![GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support PythonVersion ${config.pythonVersion}`);
        }
        if (JobLanguage.PYTHON !== config.language && config.extraPythonFiles) {
            throw new Error('extraPythonFiles is not supported for languages other than JobLanguage.PYTHON');
        }
        if (config.pythonVersion === PythonVersion.THREE_NINE && config.type !== JobType.PYTHON_SHELL) {
            throw new Error('Specified PythonVersion PythonVersion.THREE_NINE is only supported for JobType Python Shell');
        }
        this.config = config;
    }
    /**
     * Create Scala executable props for Apache Spark ETL job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Scala executable props for Apache Spark Streaming job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Python executable props for Apache Spark ETL job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for Apache Spark Streaming job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for python shell jobs.
     *
     * @param props Python Shell Job props.
     */
    static pythonShell(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonShellExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonShell);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.PYTHON_SHELL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create a custom JobExecutable.
     *
     * @param config custom job executable configuration.
     */
    static of(config) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobExecutableConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.of);
            }
            throw error;
        }
        return new JobExecutable(config);
    }
    /**
     * Called during Job initialization to get JobExecutableConfig.
     */
    bind() {
        return this.config;
    }
}
exports.JobExecutable = JobExecutable;
_c = JSII_RTTI_SYMBOL_1;
JobExecutable[_c] = { fqn: "@aws-cdk/aws-glue-alpha.JobExecutable", version: "2.51.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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