import unittest
import os
import logging
from gitbuilding.buildup import buildup

class InLineDataTestCase(unittest.TestCase):

    def test_variants(self):
        data_strs = ['qty:6',
                     'Qty:6,   ',
                     'qty:6 , ',
                     '  QTY  :  6']
        for data_str in data_strs:
            expected_dict = {'qty': '6'}
            data_dict = buildup.parse_inline_data(data_str)
            self.assertEqual(data_dict, expected_dict)

    def test_difficult_parse(self):
        data_str = """Qty:6 ,  note: 'This Note is annoying to parse _!"?}}'"""
        expected_dict = {'qty': '6', "note": 'This Note is annoying to parse _!"?}}'}
        data_dict = buildup.parse_inline_data(data_str)
        self.assertEqual(data_dict, expected_dict)

    def test_missing(self):
        data_str = "Step"
        # standard
        expected_dict = {'step': True}
        data_dict = buildup.parse_inline_data(data_str)
        self.assertTrue(data_dict, expected_dict)
        # empty_is_true: false
        expected_dict = {'step': None}
        data_dict = buildup.parse_inline_data(data_str, empty_is_true=False)
        self.assertEqual(data_dict, expected_dict)

    def test_bool(self):
        data_strs = ['step:True',
                     'step:TRUE',
                     'step:true']
        expected_dict = {'step': True}
        for data_str in data_strs:
            data_dict = buildup.parse_inline_data(data_str)
            self.assertTrue(data_dict, expected_dict)
        # step is false
        data_strs = ['step:False',
                     'step:FALSE',
                     'step:false']
        expected_dict = {'step': False}
        for data_str in data_strs:
            data_dict = buildup.parse_inline_data(data_str)
            self.assertTrue(data_dict, expected_dict)

    def test_broken(self):
        data_strs = ['foo1',
                     'foo:bar:foo',
                     'bar:foo,foo-foo',
                     'foo: "bar']
        for data_str in data_strs:
            with self.assertLogs(logger='BuildUp', level=logging.WARN):
                buildup.parse_inline_data(data_str)


class BuildUpParserTestCase(unittest.TestCase):

    def test_parse_link(self):
        bup = buildup.BuildUpParser("[name](part.md){Qty: 1}", 'folder')
        self.assertTrue('name' in bup.part_links)
        self.assertEqual(bup.part_links[0].link_rel_to_page, 'part.md')
        self.assertEqual(bup.part_links[0].link_rel_to_root,
                         os.path.join('folder', 'part.md'))
        self.assertEqual(bup.plain_links, [])
        self.assertEqual(bup.step_links, [])

    def test_parse_link_challenge(self):
        in_txt = """[name](part.md){Qty:6 ,  note: 'This Note is annoying to parse _!"?}}'}"""
        bup = buildup.BuildUpParser(in_txt, '')
        self.assertTrue('name' in bup.part_links)
        expected_dict = {'qty': '6', "note": 'This Note is annoying to parse _!"?}}'}
        self.assertEqual(bup.part_links[0].build_up_dict, expected_dict)

    def test_parse_link_challenge2(self):
        in_txt = """Normal text [text inside bracket [link_name](url) more text] and more"""
        bup = buildup.BuildUpParser(in_txt, "")
        self.assertTrue('link_name' in bup.plain_links)

    def test_parse_step(self):
        bup = buildup.BuildUpParser("[text](page.md){step}", '')
        self.assertEqual(bup.step_links[0].link_rel_to_page, 'page.md')
        self.assertEqual(bup.plain_links, [])
        self.assertEqual(bup.part_links, [])

    def test_parse_image(self):
        bup = buildup.BuildUpParser("![Some text](file.jpg)", '')
        self.assertEqual(len(bup.images), 1)
        self.assertEqual(bup.step_links, [])
        self.assertEqual(bup.plain_links, [])
        self.assertEqual(bup.part_links, [])

    def test_parse_in_page_step(self):
        bup = buildup.BuildUpParser("## Step name{pagestep: an_id}", '')
        self.assertEqual(bup.in_page_steps[0]["id"], 'an_id')

    def test_parse_in_page_step_correct_id(self):
        bup = buildup.BuildUpParser("## Step name{pagestep: An id}", '')
        with self.assertLogs(logger='BuildUp', level=logging.WARN):
            self.assertEqual(bup.in_page_steps[0]["id"], 'an-id')

    def test_parse_in_page_step_no_id(self):
        bup = buildup.BuildUpParser("## Step name{pagestep}", '')
        self.assertEqual(bup.in_page_steps[0]["id"], 'step-name')

    def test_parse_in_page_step_clash(self):
        in_txt = "## Step name{pagestep: foo}\n\n## Another step{pagestep: foo}"
        with self.assertLogs(logger='BuildUp', level=logging.WARN):
            bup = buildup.BuildUpParser(in_txt, '')
            self.assertEqual(len(bup.in_page_steps), 2)
