"""This module is for making an simple example GitBuilding project.

It is used when you run `gitbuilding new`"""

import os
import codecs

def output_example_project():
    """
    Outputs the new project to a folder.
    """
    if os.listdir(".") == []:
        newdir = "."
    else:
        ans = input("This directory is not empty. Build to new sub-dir? [y/N]: ")
        if ans in ("y", "Y"):
            newdir = input("Enter subdir name: ")
            if not os.path.split(newdir)[0] == "":
                print("\n\ngitbuilding new only supports creating a single subdirectory"
                      " to the current folder, not nested directories or full paths\n\n")
                return None
            if os.path.exists(newdir):
                print(f"\n\nCannot create directory '{newdir}', as it already exists\n\n")
                return None
            try:
                os.mkdir(newdir)
            except (PermissionError, FileNotFoundError):
                print(f"\n\nFailed to create directory '{newdir}'"
                      "do you have the correct permissions?\n\n")
                return None
        else:
            if ans not in ("n", "N", ""):
                print("Invalid response.")
            return None
    # writing example project

    # CONFIG FILE
    with codecs.open(os.path.join(newdir, "buildconf.yaml"), "w", encoding='utf-8') as file:
        file.write(example_config())

    # OVERVIEW FILE
    with codecs.open(os.path.join(newdir, "landing.md"), "w", encoding='utf-8') as file:
        file.write(example_landing())

    # TESTPAGES
    for i, page in enumerate(["testpage1.md", "testpage2.md"]):
        with codecs.open(os.path.join(newdir, page), "w", encoding='utf-8') as file:
            file.write(testpage(f"Test Page {i+1}"))

    # PARTS LIST
    with codecs.open(os.path.join(newdir, "Parts.yaml"), "w", encoding='utf-8') as file:
        file.write(basicparts())

    # README
    with codecs.open(os.path.join(newdir, "README.md"), "w", encoding='utf-8') as file:
        file.write(example_readme())

    # PARTS LIST
    with codecs.open(os.path.join(newdir, ".gitignore"), "w", encoding='utf-8') as file:
        file.write(example_gitignore())

    # IMAGE DIRECTORY
    os.mkdir(os.path.join(newdir, "images"))

    return None

def example_config():
    """This function returns and example config"""

    return """#This is required if for build HTML unless index.md exists
LandingPage: landing.md

# Recommended information
Authors:
    - My Name

Affiliation: My Affiliation

#License: CERN-OHL-1.2

Email: my-email@my.domain


#Uncomment below to add custom categories
#CustomCategories:
#    printedtool:
#        Reuse: False
#        DisplayName: Printed tools

#Uncomment below to set a custom default category
#DefaultCategory: printedtool

#Uncomment below to set the title for the bill of material on a page
#PageBOMTitle: '##For this step you will need'

#Uncomment below to override the project title
#Title: 'My project'

#Uncomment below to set a custom Website root for static HTML builds
#WebsiteRoot: '/path/'

#Uncomment below to disable fussy warnings
#Fussy: False

"""


def example_landing():
    """This function returns and example landing page"""

    return """# Test Project

This is a great place to start making your documentation!

You should link to a couple of pages:

* [.](testpage1.md){step}
* [.](testpage2.md){step}

And you should definitely let people know about the [bill of materials]({{BOMlink}}) page."""


def testpage(name):
    """This function returns and example instruction page"""

    return ("[M4x10 screws]:Parts.yaml#M4x10PanSteel\n"
            f"# {name}\n\n"
            "{{BOM}}\n\n## Method\n\n"
            "This page should have some instructions for things in your project"
            " instead it just has a link for three [M4x10 screws]{Qty: 3} and"
            " another link to use [two more][M4x10 screws]{Qty: 2}.\n\n""")


def basicparts():
    """This function returns and example part library"""

    return """
M4x10PanSteel:
    Name: M4x10 Pan Head Steel
    Description: >
        This is lots of text
        about some screw?
    Specs:
        Head: Pan
        Length: 10 mm
        Material: Stainless Steel
        Pitch: 0.7
    Suppliers:
        RS:
            PartNo: 528-817
            Link: https://uk.rs-online.com/web/p/machine-screws/0528817/
        McMasterCarr:
            PartNo: 90116A207
            Link: https://www.mcmaster.com/90116A207"""

def example_gitignore():
    """
    Returns the text for an example gitignore file for a BuildUp project
    """
    return """#This file was auto generated by GitBuilding

# Ignore the directory that GitBuilding will build markdown to
_build/
# Ignore the directory that GitBuilding will HTML markdown to
_site/
"""

def example_readme():
    """
    Returns an example README for a GitBuilding project
    """

    return """# This project is documented with GitBuilding

## What is GitBuilding

GitBuilding is an OpenSource project for documenting hardware projects with minimal
effort, so you can stop writing and git building. GitBuilding is a python program that
works on Windows, Linux, and MacOS. More information on the GitBuilding project, or how
to install GitBuilding please see the [GitBuilding website](http://gitbuilding.io)

## How do I edit the documentation?

To edit the documentation you do not need to install anything, but you will need to
install something to build the final version of the documentation (such as a website).
The documentation files can be opened in a plain text editor such as Windows Notepad,
Notepad++, gedit, VS Code, etc. GitBuilding also comes with a browser-based editor that
displays has a live display of the final HTML documentation.

If you have ever used [markdown](https://www.markdownguide.org/basic-syntax/) you will
notice that the files you are editing are markdown files. GitBuilding uses an extended
markdown syntax (that we call BuildUp). This allows you to keep track of parts in the
documentation. More detail on the documentation is available on the
[GitBuilding website](https://gitbuilding.io/syntax/). There is also additional
[syntax for configuration](https://gitbuilding.io/syntax/buildconfsyntax), and for
[part libraries](https://gitbuilding.io/syntax/builduplibrary/)."""