"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Portfolio = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const association_manager_1 = require("./private/association-manager");
const util_1 = require("./private/util");
const validation_1 = require("./private/validation");
const aws_servicecatalog_1 = require("aws-cdk-lib/aws-servicecatalog");
class PortfolioBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedPrincipals = new Set();
    }
    giveAccessToRole(role) {
        this.associatePrincipal(role.roleArn, role.node.addr);
    }
    giveAccessToUser(user) {
        this.associatePrincipal(user.userArn, user.node.addr);
    }
    giveAccessToGroup(group) {
        this.associatePrincipal(group.groupArn, group.node.addr);
    }
    addProduct(product) {
        association_manager_1.AssociationManager.associateProductWithPortfolio(this, product, undefined);
    }
    shareWithAccount(accountId, options = {}) {
        const hashId = this.generateUniqueHash(accountId);
        new aws_servicecatalog_1.CfnPortfolioShare(this, `PortfolioShare${hashId}`, {
            portfolioId: this.portfolioId,
            accountId: accountId,
            shareTagOptions: options.shareTagOptions,
            acceptLanguage: options.messageLanguage,
        });
    }
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, this.portfolioId, tagOptions);
    }
    constrainTagUpdates(product, options = {}) {
        association_manager_1.AssociationManager.constrainTagUpdates(this, product, options);
    }
    notifyOnStackEvents(product, topic, options = {}) {
        association_manager_1.AssociationManager.notifyOnStackEvents(this, product, topic, options);
    }
    constrainCloudFormationParameters(product, options) {
        association_manager_1.AssociationManager.constrainCloudFormationParameters(this, product, options);
    }
    setLaunchRole(product, launchRole, options = {}) {
        association_manager_1.AssociationManager.setLaunchRole(this, product, launchRole, options);
    }
    setLocalLaunchRoleName(product, launchRoleName, options = {}) {
        const launchRole = new iam.Role(this, `LaunchRole${launchRoleName}`, {
            roleName: launchRoleName,
            assumedBy: new iam.ServicePrincipal('servicecatalog.amazonaws.com'),
        });
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
        return launchRole;
    }
    setLocalLaunchRole(product, launchRole, options = {}) {
        validation_1.InputValidator.validateRoleNameSetForLocalLaunchRole(launchRole);
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
    }
    deployWithStackSets(product, options) {
        association_manager_1.AssociationManager.deployWithStackSets(this, product, options);
    }
    /**
     * Associate a principal with the portfolio.
     * If the principal is already associated, it will skip.
     */
    associatePrincipal(principalArn, principalId) {
        if (!this.associatedPrincipals.has(principalArn)) {
            const hashId = this.generateUniqueHash(principalId);
            new aws_servicecatalog_1.CfnPortfolioPrincipalAssociation(this, `PortolioPrincipalAssociation${hashId}`, {
                portfolioId: this.portfolioId,
                principalArn: principalArn,
                principalType: 'IAM',
            });
            this.associatedPrincipals.add(principalArn);
        }
    }
}
/**
 * A Service Catalog portfolio.
 */
class Portfolio extends PortfolioBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicecatalog_alpha_PortfolioProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.validatePortfolioProps(props);
        this.portfolio = new aws_servicecatalog_1.CfnPortfolio(this, 'Resource', {
            displayName: props.displayName,
            providerName: props.providerName,
            description: props.description,
            acceptLanguage: props.messageLanguage,
        });
        this.portfolioId = this.portfolio.ref;
        this.portfolioArn = cdk.Stack.of(this).formatArn({
            service: 'servicecatalog',
            resource: 'portfolio',
            resourceName: this.portfolioId,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    /**
     * Creates a Portfolio construct that represents an external portfolio.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param portfolioArn the Amazon Resource Name of the existing portfolio.
     */
    static fromPortfolioArn(scope, id, portfolioArn) {
        const arn = cdk.Stack.of(scope).splitArn(portfolioArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        const portfolioId = arn.resourceName;
        if (!portfolioId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + portfolioArn);
        }
        class Import extends PortfolioBase {
            constructor() {
                super(...arguments);
                this.portfolioArn = portfolioArn;
                this.portfolioId = portfolioId;
            }
            generateUniqueHash(value) {
                return util_1.hashValues(this.portfolioArn, value);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: portfolioArn,
        });
    }
    generateUniqueHash(value) {
        return util_1.hashValues(cdk.Names.nodeUniqueId(this.portfolio.node), value);
    }
    validatePortfolioProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio display name', 1, 100, props.displayName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio provider name', 1, 50, props.providerName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio description', 0, 2000, props.description);
    }
}
exports.Portfolio = Portfolio;
_a = JSII_RTTI_SYMBOL_1;
Portfolio[_a] = { fqn: "@aws-cdk/aws-servicecatalog-alpha.Portfolio", version: "2.21.1-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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