"""Parse bel xls output files."""

from itertools import product

import dateutil.parser
import xlrd

from pygaps.parsing.bel_common import _META_DICT
from pygaps.parsing.bel_common import _check
from pygaps.parsing.bel_common import _parse_header


def parse(path):
    """
    Parse an xls file generated by BEL software.

    Parameters
    ----------
    path: str
        The location of an xls file generated by a belsorp instrument.

    Returns
    -------
    dict
        A dictionary containing report information.
    """
    meta = {}
    data = {}
    errors = []

    workbook = xlrd.open_workbook(path, encoding_override='latin-1')
    sheet = workbook.sheet_by_name('AdsDes')

    for row, col in product(range(sheet.nrows), range(sheet.ncols)):
        cell_value = str(sheet.cell(row, col).value).strip()

        if cell_value != "No":
            cell_search = cell_value.lower()
            try:
                name = next(
                    k for k, v in _META_DICT.items()
                    if any(cell_search == n for n in v.get('text', []))
                )
            except StopIteration:
                continue

            ref = _META_DICT[name]['xl_ref']
            tp = _META_DICT[name]['type']
            val = sheet.cell(row + ref[0], col + ref[1]).value

            if tp == 'number':
                meta[name] = val
            elif tp == 'date':
                meta[name] = _handle_date(sheet, val)
            elif tp == 'time':
                meta[name] = _handle_time(sheet, val)
            elif tp == 'string':
                meta[name] = _handle_string(val)

        else:  # If "data" section

            header_list = [
                sheet.cell(row, rcol).value.strip()
                for rcol in range(sheet.ncols)
                if sheet.cell(row, rcol).value.strip() != ''
            ]
            head, units = _parse_header(header_list)  # header
            meta.update(units)

            (ads_start, ads_end, des_start, des_end) = _find_datapoints(sheet, row, col)
            data["branch"] = [0] * (ads_end - ads_start) + [1] * (des_end - des_start)
            for i, item in enumerate(head[1:]):
                ads_points = [sheet.cell(r, i).value for r in range(ads_start, ads_end)]
                des_points = [sheet.cell(r, i).value for r in range(des_start, des_end)]
                data[item] = ads_points + des_points

    if errors:
        meta['errors'] = errors

    _check(meta, data, path)

    # Set extra metadata
    meta['apparatus'] = f'BEL {meta["serialnumber"]}'
    meta['date'] = dateutil.parser.parse(meta['date']).isoformat()

    return meta, data


def _handle_date(sheet, val):
    """
    Convert date to string.

    Input is a cell of type 'date'.
    """
    if val:
        from xlrd.xldate import xldate_as_datetime
        return xldate_as_datetime(val, sheet.book.datemode).strftime("%Y-%m-%d")


def _handle_time(sheet, val):
    """
    Convert date to string.

    Input is a cell of type 'date'.
    """
    if val:
        from xlrd.xldate import xldate_as_datetime
        return xldate_as_datetime(val, sheet.book.datemode).strftime("%H-%M-%S")


def _handle_string(val):
    """
    Replace Comments: and any newline found.

    Input is a cell of type 'string'.
    """
    return str(val).replace('Comments: ', '').replace('\r\n', ' ')


def _find_datapoints(sheet, row, col):
    """Return start and stop points for adsorption and desorption."""
    rowc = 1

    # Check for adsorption branch
    if sheet.cell(row + rowc, col).value == 'ADS':
        ads_start_row = row + rowc + 1
        ads_final_row = ads_start_row

    point = sheet.cell(ads_final_row, col).value

    while point != 'DES':
        ads_final_row += 1
        point = sheet.cell(ads_final_row, col).value

    if sheet.cell(ads_final_row, col).value == 'DES':
        des_start_row = ads_final_row + 1
        des_final_row = des_start_row

    if des_final_row < sheet.nrows:

        point = sheet.cell(des_final_row, col).value

        while str(point).strip():
            des_final_row += 1
            if des_final_row < sheet.nrows:
                point = sheet.cell(des_final_row, col).value
            else:
                point = None

    return (ads_start_row, ads_final_row, des_start_row, des_final_row)
