"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DnsValidatedCertificate = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const certificate_base_1 = require("./certificate-base");
/**
 * A certificate managed by AWS Certificate Manager.
 *
 * Will be automatically
 * validated using DNS validation against the specified Route 53 hosted zone.
 *
 * @stability stable
 * @resource AWS::CertificateManager::Certificate
 */
class DnsValidatedCertificate extends certificate_base_1.CertificateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_certificatemanager_DnsValidatedCertificateProps(props);
        this.region = props.region;
        this.domainName = props.domainName;
        this.normalizedZoneName = props.hostedZone.zoneName;
        // Remove trailing `.` from zone name
        if (this.normalizedZoneName.endsWith('.')) {
            this.normalizedZoneName = this.normalizedZoneName.substring(0, this.normalizedZoneName.length - 1);
        }
        // Remove any `/hostedzone/` prefix from the Hosted Zone ID
        this.hostedZoneId = props.hostedZone.hostedZoneId.replace(/^\/hostedzone\//, '');
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'AWS::CertificateManager::Certificate');
        const requestorFunction = new lambda.Function(this, 'CertificateRequestorFunction', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '..', 'lambda-packages', 'dns_validated_certificate_handler', 'lib')),
            handler: 'index.certificateRequestHandler',
            runtime: lambda.Runtime.NODEJS_12_X,
            timeout: cdk.Duration.minutes(15),
            role: props.customResourceRole,
        });
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['acm:RequestCertificate', 'acm:DescribeCertificate', 'acm:DeleteCertificate', 'acm:AddTagsToCertificate'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:GetChange'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:changeResourceRecordSets'],
            resources: [`arn:${cdk.Stack.of(requestorFunction).partition}:route53:::hostedzone/${this.hostedZoneId}`],
        }));
        const certificate = new cdk.CustomResource(this, 'CertificateRequestorResource', {
            serviceToken: requestorFunction.functionArn,
            properties: {
                DomainName: props.domainName,
                SubjectAlternativeNames: cdk.Lazy.list({ produce: () => props.subjectAlternativeNames }, { omitEmpty: true }),
                HostedZoneId: this.hostedZoneId,
                Region: props.region,
                Route53Endpoint: props.route53Endpoint,
                Tags: cdk.Lazy.list({ produce: () => this.tags.renderTags() }),
            },
        });
        this.certificateArn = certificate.getAtt('Arn').toString();
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const errors = [];
        // Ensure the zone name is a parent zone of the certificate domain name
        if (!cdk.Token.isUnresolved(this.normalizedZoneName) &&
            this.domainName !== this.normalizedZoneName &&
            !this.domainName.endsWith('.' + this.normalizedZoneName)) {
            errors.push(`DNS zone ${this.normalizedZoneName} is not authoritative for certificate domain name ${this.domainName}`);
        }
        return errors;
    }
}
exports.DnsValidatedCertificate = DnsValidatedCertificate;
_a = JSII_RTTI_SYMBOL_1;
DnsValidatedCertificate[_a] = { fqn: "@aws-cdk/aws-certificatemanager.DnsValidatedCertificate", version: "1.136.0" };
//# sourceMappingURL=data:application/json;base64,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