"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const certificate_base_1 = require("./certificate-base");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * How to validate a certificate.
 *
 * @stability stable
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        var _c;
        this.props = props;
        this.method = (_c = props.method) !== null && _c !== void 0 ? _c : ValidationMethod.EMAIL;
    }
    /**
     * Validate the certificate with DNS.
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created.
     * @stability stable
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * Validate the certificate with automatically created DNS records in multiple Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created for the domains in the certificate.
     * @stability stable
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * Validate the certificate with Email.
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *   admin@domain.com
     *   administrator@domain.com
     *   hostmaster@domain.com
     *   postmaster@domain.com
     *   webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate.
     * @stability stable
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
_a = JSII_RTTI_SYMBOL_1;
CertificateValidation[_a] = { fqn: "@aws-cdk/aws-certificatemanager.CertificateValidation", version: "1.136.0" };
/**
 * A certificate managed by AWS Certificate Manager.
 *
 * @stability stable
 */
class Certificate extends certificate_base_1.CertificateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_certificatemanager_CertificateProps(props);
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * Import a certificate.
     *
     * @stability stable
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends certificate_base_1.CertificateBase {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
_b = JSII_RTTI_SYMBOL_1;
Certificate[_b] = { fqn: "@aws-cdk/aws-certificatemanager.Certificate", version: "1.136.0" };
/**
 * Method used to assert ownership of the domain.
 *
 * @stability stable
 */
var ValidationMethod;
(function (ValidationMethod) {
    ValidationMethod["EMAIL"] = "EMAIL";
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    var _c, _d, _e;
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = (_d = (_c = validation.props.hostedZones) === null || _c === void 0 ? void 0 : _c[domainName]) !== null && _d !== void 0 ? _d : validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = (_e = validation.props.validationDomains) === null || _e === void 0 ? void 0 : _e[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain !== null && validationDomain !== void 0 ? validationDomain : util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,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