"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const rds = require("../lib");
module.exports = {
    'create a DB proxy from an instance'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc,
        });
        // WHEN
        new rds.DatabaseProxy(stack, 'Proxy', {
            proxyTarget: rds.ProxyTarget.fromInstance(instance),
            secrets: [instance.secret],
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBProxy', {
            Properties: {
                Auth: [
                    {
                        AuthScheme: 'SECRETS',
                        IAMAuth: 'DISABLED',
                        SecretArn: {
                            Ref: 'InstanceSecretAttachment83BEE581',
                        },
                    },
                ],
                DBProxyName: 'Proxy',
                EngineFamily: 'MYSQL',
                RequireTLS: true,
                RoleArn: {
                    'Fn::GetAtt': [
                        'ProxyIAMRole2FE8AB0F',
                        'Arn',
                    ],
                },
                VpcSubnetIds: [
                    {
                        Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                    },
                    {
                        Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                    },
                ],
            },
        }, assert_1.ResourcePart.CompleteDefinition));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBProxyTargetGroup', {
            Properties: {
                DBProxyName: {
                    Ref: 'ProxyCB0DFB71',
                },
                ConnectionPoolConfigurationInfo: {},
                DBInstanceIdentifiers: [
                    {
                        Ref: 'InstanceC1063A87',
                    },
                ],
                TargetGroupName: 'default',
            },
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'create a DB proxy from a cluster'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new rds.DatabaseCluster(stack, 'Database', {
            engine: rds.DatabaseClusterEngine.auroraPostgres({
                version: rds.AuroraPostgresEngineVersion.VER_10_7,
            }),
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // WHEN
        new rds.DatabaseProxy(stack, 'Proxy', {
            proxyTarget: rds.ProxyTarget.fromCluster(cluster),
            secrets: [cluster.secret],
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBProxy', {
            Properties: {
                Auth: [
                    {
                        AuthScheme: 'SECRETS',
                        IAMAuth: 'DISABLED',
                        SecretArn: {
                            Ref: 'DatabaseSecretAttachmentE5D1B020',
                        },
                    },
                ],
                DBProxyName: 'Proxy',
                EngineFamily: 'POSTGRESQL',
                RequireTLS: true,
                RoleArn: {
                    'Fn::GetAtt': [
                        'ProxyIAMRole2FE8AB0F',
                        'Arn',
                    ],
                },
                VpcSubnetIds: [
                    {
                        Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                    },
                    {
                        Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                    },
                ],
            },
        }, assert_1.ResourcePart.CompleteDefinition));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBProxyTargetGroup', {
            Properties: {
                DBProxyName: {
                    Ref: 'ProxyCB0DFB71',
                },
                ConnectionPoolConfigurationInfo: {},
                DBClusterIdentifiers: [
                    {
                        Ref: 'DatabaseB269D8BB',
                    },
                ],
                TargetGroupName: 'default',
            },
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new rds.DatabaseCluster(stack, 'Database', {
            engine: rds.DatabaseClusterEngine.auroraPostgres({
                version: rds.AuroraPostgresEngineVersion.VER_10_7,
            }),
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // WHEN
        test.doesNotThrow(() => {
            new rds.DatabaseProxy(stack, 'Proxy', {
                proxyTarget: rds.ProxyTarget.fromCluster(cluster),
                secrets: [cluster.secret],
                vpc,
            });
        }, /Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers/);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBProxyTargetGroup', {
            DBInstanceIdentifiers: assert_1.ABSENT,
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'One or more secrets are required.'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new rds.DatabaseCluster(stack, 'Database', {
            engine: rds.DatabaseClusterEngine.auroraPostgres({ version: rds.AuroraPostgresEngineVersion.VER_10_7 }),
            masterUser: { username: 'admin' },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // WHEN
        test.throws(() => {
            new rds.DatabaseProxy(stack, 'Proxy', {
                proxyTarget: rds.ProxyTarget.fromCluster(cluster),
                secrets: [],
                vpc,
            });
        }, 'One or more secrets are required.');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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