"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
function testStack() {
    const stack = new cdk.Stack(undefined, undefined, { env: { account: '12345', region: 'us-test-1' } });
    stack.node.setContext('availability-zones:12345:us-test-1', ['us-test-1a', 'us-test-1b']);
    return stack;
}
module.exports = {
    'creating a Cluster also creates 2 DB Instances'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Properties: {
                Engine: 'aurora',
                DBSubnetGroupName: { Ref: 'DatabaseSubnets56F17B9A' },
                MasterUsername: 'admin',
                MasterUserPassword: 'tooshort',
                VpcSecurityGroupIds: [{ 'Fn::GetAtt': ['DatabaseSecurityGroup5C91FDCB', 'GroupId'] }],
            },
            DeletionPolicy: assert_1.ABSENT,
            UpdateReplacePolicy: 'Snapshot',
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.countResources('AWS::RDS::DBInstance', 2));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DeletionPolicy: assert_1.ABSENT,
            UpdateReplacePolicy: assert_1.ABSENT,
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'can create a cluster with a single instance'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora',
            DBSubnetGroupName: { Ref: 'DatabaseSubnets56F17B9A' },
            MasterUsername: 'admin',
            MasterUserPassword: 'tooshort',
            VpcSecurityGroupIds: [{ 'Fn::GetAtt': ['DatabaseSecurityGroup5C91FDCB', 'GroupId'] }],
        }));
        test.done();
    },
    'can create a cluster with imported vpc and security group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = ec2.Vpc.fromLookup(stack, 'VPC', {
            vpcId: 'VPC12345',
        });
        const sg = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'SecurityGroupId12345');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
                securityGroups: [sg],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora',
            DBSubnetGroupName: { Ref: 'DatabaseSubnets56F17B9A' },
            MasterUsername: 'admin',
            MasterUserPassword: 'tooshort',
            VpcSecurityGroupIds: ['SecurityGroupId12345'],
        }));
        test.done();
    },
    'cluster with parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const group = new lib_1.ParameterGroup(stack, 'Params', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            description: 'bye',
            parameters: {
                param: 'value',
            },
        });
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            parameterGroup: group,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            DBClusterParameterGroupName: { Ref: 'ParamsA8366201' },
        }));
        test.done();
    },
    "sets the retention policy of the SubnetGroup to 'Retain' if the Cluster is created with 'Retain'"(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc');
        new lib_1.DatabaseCluster(stack, 'Cluster', {
            masterUser: { username: 'admin' },
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE),
                vpc,
            },
            removalPolicy: cdk.RemovalPolicy.RETAIN,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::RDS::DBSubnetGroup', {
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain',
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'creates a secret when master credentials are not specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            MasterUsername: {
                'Fn::Join': [
                    '',
                    [
                        '{{resolve:secretsmanager:',
                        {
                            Ref: 'DatabaseSecret3B817195',
                        },
                        ':SecretString:username::}}',
                    ],
                ],
            },
            MasterUserPassword: {
                'Fn::Join': [
                    '',
                    [
                        '{{resolve:secretsmanager:',
                        {
                            Ref: 'DatabaseSecret3B817195',
                        },
                        ':SecretString:password::}}',
                    ],
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                ExcludeCharacters: '\"@/\\',
                GenerateStringKey: 'password',
                PasswordLength: 30,
                SecretStringTemplate: '{"username":"admin"}',
            },
        }));
        test.done();
    },
    'create an encrypted cluster with custom KMS key'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            storageEncryptionKey: new kms.Key(stack, 'Key'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            KmsKeyId: {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'cluster with instance parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const parameterGroup = new lib_1.ParameterGroup(stack, 'ParameterGroup', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            parameters: {
                key: 'value',
            },
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                parameterGroup,
                vpc,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DBParameterGroupName: {
                Ref: 'ParameterGroup5E32DECB',
            },
        }));
        test.done();
    },
    'create a cluster using a specific version of MySQL'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.auroraMysql({
                version: lib_1.AuroraMysqlEngineVersion.VER_2_04_4,
            }),
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora-mysql',
            EngineVersion: '5.7.mysql_aurora.2.04.4',
        }));
        test.done();
    },
    'create a cluster using a specific version of Postgresql'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.auroraPostgres({
                version: lib_1.AuroraPostgresEngineVersion.VER_10_7,
            }),
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora-postgresql',
            EngineVersion: '10.7',
        }));
        test.done();
    },
    'cluster exposes different read and write endpoints'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        test.notDeepEqual(stack.resolve(cluster.clusterEndpoint), stack.resolve(cluster.clusterReadEndpoint));
        test.done();
    },
    'imported cluster with imported security group honors allowAllOutbound'(test) {
        // GIVEN
        const stack = testStack();
        const cluster = lib_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                    allowAllOutbound: false,
                })],
        });
        // WHEN
        cluster.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-123456789',
        }));
        test.done();
    },
    'cluster with enabled monitoring'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            monitoringInterval: cdk.Duration.minutes(1),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                'Fn::GetAtt': ['DatabaseMonitoringRole576991DA', 'Arn'],
            },
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'monitoring.rds.amazonaws.com',
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            ManagedPolicyArns: [
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/service-role/AmazonRDSEnhancedMonitoringRole',
                        ],
                    ],
                },
            ],
        }));
        test.done();
    },
    'create a cluster with imported monitoring role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const monitoringRole = new aws_iam_1.Role(stack, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
            ],
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            monitoringInterval: cdk.Duration.minutes(1),
            monitoringRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                'Fn::GetAtt': ['MonitoringRole90457BF9', 'Arn'],
            },
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'throws when trying to add rotation to a cluster without secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        test.throws(() => cluster.addRotationSingleUser(), /without secret/);
        test.done();
    },
    'throws when trying to add single user rotation multiple times'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: { username: 'admin' },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // WHEN
        cluster.addRotationSingleUser();
        // THEN
        test.throws(() => cluster.addRotationSingleUser(), /A single user rotation was already added to this cluster/);
        test.done();
    },
    'create a cluster with s3 import role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const associatedRole = new aws_iam_1.Role(stack, 'AssociatedRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportRole: associatedRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'AssociatedRole824CFCD3',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'AssociatedRole824CFCD3',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'create a cluster with s3 import buckets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const bucket = new s3.Bucket(stack, 'Bucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportBuckets: [bucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ImportRole377BC9C0',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ImportRole377BC9C0',
                        'Arn',
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'Bucket83908E77',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'Bucket83908E77',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'create a cluster with s3 export role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const associatedRole = new aws_iam_1.Role(stack, 'AssociatedRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportRole: associatedRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'AssociatedRole824CFCD3',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'AssociatedRole824CFCD3',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'create a cluster with s3 export buckets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const bucket = new s3.Bucket(stack, 'Bucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportBuckets: [bucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ExportRole9E328562',
                        'Arn',
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                            's3:DeleteObject*',
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'Bucket83908E77',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'Bucket83908E77',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'create a cluster with s3 import and export buckets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const importBucket = new s3.Bucket(stack, 'ImportBucket');
        const exportBucket = new s3.Bucket(stack, 'ExportBucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportBuckets: [importBucket],
            s3ExportBuckets: [exportBucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ImportRole377BC9C0',
                            'Arn',
                        ],
                    },
                },
                {
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ImportRole377BC9C0',
                        'Arn',
                    ],
                },
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ExportRole9E328562',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'create a cluster with s3 import and export buckets and custom parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const parameterGroup = new lib_1.ParameterGroup(stack, 'ParameterGroup', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            parameters: {
                key: 'value',
            },
        });
        const importBucket = new s3.Bucket(stack, 'ImportBucket');
        const exportBucket = new s3.Bucket(stack, 'ExportBucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            parameterGroup,
            s3ImportBuckets: [importBucket],
            s3ExportBuckets: [exportBucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ImportRole377BC9C0',
                            'Arn',
                        ],
                    },
                },
                {
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                key: 'value',
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ImportRole377BC9C0',
                        'Arn',
                    ],
                },
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ExportRole9E328562',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'PostgreSQL cluster with s3 export buckets does not generate custom parameter group and specifies the correct port'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const bucket = new s3.Bucket(stack, 'Bucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.auroraPostgres({
                version: lib_1.AuroraPostgresEngineVersion.VER_11_4,
            }),
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportBuckets: [bucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
            DBClusterParameterGroupName: 'default.aurora-postgresql11',
            Port: 5432,
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup'));
        test.done();
    },
    'MySQL cluster without S3 exports or imports references the correct default ParameterGroup'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::RDS::DBCluster', {
            DBClusterParameterGroupName: 'default.aurora-mysql5.7',
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup'));
        test.done();
    },
    'throws when s3ExportRole and s3ExportBuckets properties are both specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const exportRole = new aws_iam_1.Role(stack, 'ExportRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        const exportBucket = new s3.Bucket(stack, 'ExportBucket');
        // THEN
        test.throws(() => new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportRole: exportRole,
            s3ExportBuckets: [exportBucket],
        }));
        test.done();
    },
    'throws when s3ImportRole and s3ImportBuckets properties are both specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const importRole = new aws_iam_1.Role(stack, 'ImportRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        const importBucket = new s3.Bucket(stack, 'ImportBucket');
        // THEN
        test.throws(() => new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportRole: importRole,
            s3ImportBuckets: [importBucket],
        }));
        test.done();
    },
    'does not throw (but adds a node error) if a (dummy) VPC does not have sufficient subnets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = ec2.Vpc.fromLookup(stack, 'VPC', { isDefault: true });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
                vpcSubnets: {
                    subnetName: 'DefinitelyDoesNotExist',
                },
            },
        });
        // THEN
        const art = assert_1.SynthUtils.synthesize(stack);
        const meta = art.findMetadataByType('aws:cdk:error');
        test.equal(meta[0].data, 'Cluster requires at least 2 subnets, got 0');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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