"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(_) {
        let engine;
        if (this.dbCluster && this.dbInstance) {
            throw new Error('Proxy cannot target both database cluster and database instance.');
        }
        else if (this.dbCluster) {
            engine = this.dbCluster.node.defaultChild.engine;
        }
        else if (this.dbInstance) {
            engine = this.dbInstance.node.defaultChild.engine;
        }
        let engineFamily;
        switch (engine) {
            case 'aurora':
            case 'aurora-mysql':
            case 'mysql':
                engineFamily = 'MYSQL';
                break;
            case 'aurora-postgresql':
            case 'postgres':
                engineFamily = 'POSTGRESQL';
                break;
            default:
                throw new Error(`Unsupported engine type - ${engine}`);
        }
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        this.connections = new ec2.Connections({ securityGroups: props.securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_a = props.idleClientTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            requireTls: (_b = props.requireTLS) !== null && _b !== void 0 ? _b : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: (_c = props.securityGroups) === null || _c === void 0 ? void 0 : _c.map(_ => _.securityGroupId),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
    }
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
}
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _a, _b;
    return {
        connectionBorrowTimeout: (_a = props.borrowTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_b = props.sessionPinningFilters) === null || _b === void 0 ? void 0 : _b.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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