"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ParameterGroup = void 0;
const core_1 = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * A parameter group.
 * Represents both a cluster parameter group,
 * and an instance parameter group.
 *
 * @resource AWS::RDS::DBParameterGroup
 */
class ParameterGroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const family = props.engine.parameterGroupFamily;
        if (!family) {
            throw new Error("ParameterGroup cannot be used with an engine that doesn't specify a version");
        }
        this.family = family;
        this.description = props.description;
        this.parameters = (_a = props.parameters) !== null && _a !== void 0 ? _a : {};
    }
    /**
     * Imports a parameter group
     */
    static fromParameterGroupName(scope, id, parameterGroupName) {
        class Import extends core_1.Resource {
            bindToCluster(_options) {
                return { parameterGroupName };
            }
            bindToInstance(_options) {
                return { parameterGroupName };
            }
            addParameter(_key, _value) {
                return false;
            }
        }
        return new Import(scope, id);
    }
    bindToCluster(_options) {
        if (!this.clusterCfnGroup) {
            const id = this.instanceCfnGroup ? 'ClusterParameterGroup' : 'Resource';
            this.clusterCfnGroup = new rds_generated_1.CfnDBClusterParameterGroup(this, id, {
                description: this.description || `Cluster parameter group for ${this.family}`,
                family: this.family,
                parameters: core_1.Lazy.anyValue({ produce: () => this.parameters }),
            });
        }
        return {
            parameterGroupName: this.clusterCfnGroup.ref,
        };
    }
    bindToInstance(_options) {
        if (!this.instanceCfnGroup) {
            const id = this.clusterCfnGroup ? 'InstanceParameterGroup' : 'Resource';
            this.instanceCfnGroup = new rds_generated_1.CfnDBParameterGroup(this, id, {
                description: this.description || `Parameter group for ${this.family}`,
                family: this.family,
                parameters: core_1.Lazy.anyValue({ produce: () => this.parameters }),
            });
        }
        return {
            parameterGroupName: this.instanceCfnGroup.ref,
        };
    }
    /**
     * Add a parameter to this parameter group
     *
     * @param key The key of the parameter to be added
     * @param value The value of the parameter to be added
     */
    addParameter(key, value) {
        this.parameters[key] = value;
        return true;
    }
}
exports.ParameterGroup = ParameterGroup;
//# sourceMappingURL=data:application/json;base64,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