"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceEngine = exports.SqlServerEngineVersion = exports.OracleEngineVersion = exports.OracleLegacyEngineVersion = exports.PostgresEngineVersion = exports.MysqlEngineVersion = exports.MariaDbEngineVersion = void 0;
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
class InstanceEngineBase {
    constructor(props) {
        var _a;
        this.engineType = props.engineType;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.engineVersion = props.version;
        this.parameterGroupFamily = (_a = props.parameterGroupFamily) !== null && _a !== void 0 ? _a : (this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined);
    }
    bindToInstance(_scope, options) {
        if (options.timezone) {
            throw new Error(`timezone property can be configured only for Microsoft SQL Server, not ${this.engineType}`);
        }
        return {};
    }
}
/**
 * The versions for the MariaDB instance engines
 * (those returned by {@link DatabaseInstanceEngine.mariaDb}).
 */
class MariaDbEngineVersion {
    constructor(mariaDbFullVersion, mariaDbMajorVersion) {
        this.mariaDbFullVersion = mariaDbFullVersion;
        this.mariaDbMajorVersion = mariaDbMajorVersion;
    }
    /**
     * Create a new MariaDbEngineVersion with an arbitrary version.
     *
     * @param mariaDbFullVersion the full version string,
     *   for example "10.5.28"
     * @param mariaDbMajorVersion the major version of the engine,
     *   for example "10.5"
     */
    static of(mariaDbFullVersion, mariaDbMajorVersion) {
        return new MariaDbEngineVersion(mariaDbFullVersion, mariaDbMajorVersion);
    }
}
exports.MariaDbEngineVersion = MariaDbEngineVersion;
/** Version "10.0" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_0 = MariaDbEngineVersion.of('10.0', '10.0');
/** Version "10.0.17". */
MariaDbEngineVersion.VER_10_0_17 = MariaDbEngineVersion.of('10.0.17', '10.0');
/** Version "10.0.24". */
MariaDbEngineVersion.VER_10_0_24 = MariaDbEngineVersion.of('10.0.24', '10.0');
/** Version "10.0.28". */
MariaDbEngineVersion.VER_10_0_28 = MariaDbEngineVersion.of('10.0.28', '10.0');
/** Version "10.0.31". */
MariaDbEngineVersion.VER_10_0_31 = MariaDbEngineVersion.of('10.0.31', '10.0');
/** Version "10.0.32". */
MariaDbEngineVersion.VER_10_0_32 = MariaDbEngineVersion.of('10.0.32', '10.0');
/** Version "10.0.34". */
MariaDbEngineVersion.VER_10_0_34 = MariaDbEngineVersion.of('10.0.34', '10.0');
/** Version "10.0.35". */
MariaDbEngineVersion.VER_10_0_35 = MariaDbEngineVersion.of('10.0.35', '10.0');
/** Version "10.1" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_1 = MariaDbEngineVersion.of('10.1', '10.1');
/** Version "10.1.14". */
MariaDbEngineVersion.VER_10_1_14 = MariaDbEngineVersion.of('10.1.14', '10.1');
/** Version "10.1.19". */
MariaDbEngineVersion.VER_10_1_19 = MariaDbEngineVersion.of('10.1.19', '10.1');
/** Version "10.1.23". */
MariaDbEngineVersion.VER_10_1_23 = MariaDbEngineVersion.of('10.1.23', '10.1');
/** Version "10.1.26". */
MariaDbEngineVersion.VER_10_1_26 = MariaDbEngineVersion.of('10.1.26', '10.1');
/** Version "10.1.31". */
MariaDbEngineVersion.VER_10_1_31 = MariaDbEngineVersion.of('10.1.31', '10.1');
/** Version "10.1.34". */
MariaDbEngineVersion.VER_10_1_34 = MariaDbEngineVersion.of('10.1.34', '10.1');
/** Version "10.2" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_2 = MariaDbEngineVersion.of('10.2', '10.2');
/** Version "10.2.11". */
MariaDbEngineVersion.VER_10_2_11 = MariaDbEngineVersion.of('10.2.11', '10.2');
/** Version "10.2.12". */
MariaDbEngineVersion.VER_10_2_12 = MariaDbEngineVersion.of('10.2.12', '10.2');
/** Version "10.2.15". */
MariaDbEngineVersion.VER_10_2_15 = MariaDbEngineVersion.of('10.2.15', '10.2');
/** Version "10.2.21". */
MariaDbEngineVersion.VER_10_2_21 = MariaDbEngineVersion.of('10.2.21', '10.2');
/** Version "10.3" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_3 = MariaDbEngineVersion.of('10.3', '10.3');
/** Version "10.3.8". */
MariaDbEngineVersion.VER_10_3_8 = MariaDbEngineVersion.of('10.3.8', '10.3');
/** Version "10.3.13". */
MariaDbEngineVersion.VER_10_3_13 = MariaDbEngineVersion.of('10.3.13', '10.3');
/** Version "10.3.20". */
MariaDbEngineVersion.VER_10_3_20 = MariaDbEngineVersion.of('10.3.20', '10.3');
/** Version "10.3.23". */
MariaDbEngineVersion.VER_10_3_23 = MariaDbEngineVersion.of('10.3.23', '10.3');
/** Version "10.4" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_4 = MariaDbEngineVersion.of('10.4', '10.4');
/** Version "10.4.8". */
MariaDbEngineVersion.VER_10_4_8 = MariaDbEngineVersion.of('10.4.8', '10.4');
/** Version "10.4.13". */
MariaDbEngineVersion.VER_10_4_13 = MariaDbEngineVersion.of('10.4.13', '10.4');
class MariaDbInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mariadb',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mariaDbFullVersion,
                    majorVersion: version.mariaDbMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the MySQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.mysql}).
 */
class MysqlEngineVersion {
    constructor(mysqlFullVersion, mysqlMajorVersion) {
        this.mysqlFullVersion = mysqlFullVersion;
        this.mysqlMajorVersion = mysqlMajorVersion;
    }
    /**
     * Create a new MysqlEngineVersion with an arbitrary version.
     *
     * @param mysqlFullVersion the full version string,
     *   for example "8.1.43"
     * @param mysqlMajorVersion the major version of the engine,
     *   for example "8.1"
     */
    static of(mysqlFullVersion, mysqlMajorVersion) {
        return new MysqlEngineVersion(mysqlFullVersion, mysqlMajorVersion);
    }
}
exports.MysqlEngineVersion = MysqlEngineVersion;
/** Version "5.5" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_5 = MysqlEngineVersion.of('5.5', '5.5');
/** Version "5.5.46". */
MysqlEngineVersion.VER_5_5_46 = MysqlEngineVersion.of('5.5.46', '5.5');
/** Version "5.5.53". */
MysqlEngineVersion.VER_5_5_53 = MysqlEngineVersion.of('5.5.53', '5.5');
/** Version "5.5.57". */
MysqlEngineVersion.VER_5_5_57 = MysqlEngineVersion.of('5.5.57', '5.5');
/** Version "5.5.59". */
MysqlEngineVersion.VER_5_5_59 = MysqlEngineVersion.of('5.5.59', '5.5');
/** Version "5.5.61". */
MysqlEngineVersion.VER_5_5_61 = MysqlEngineVersion.of('5.5.61', '5.5');
/** Version "5.6" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_6 = MysqlEngineVersion.of('5.6', '5.6');
/** Version "5.6.34". */
MysqlEngineVersion.VER_5_6_34 = MysqlEngineVersion.of('5.6.34', '5.6');
/** Version "5.6.35". */
MysqlEngineVersion.VER_5_6_35 = MysqlEngineVersion.of('5.6.35', '5.6');
/** Version "5.6.37". */
MysqlEngineVersion.VER_5_6_37 = MysqlEngineVersion.of('5.6.37', '5.6');
/** Version "5.6.39". */
MysqlEngineVersion.VER_5_6_39 = MysqlEngineVersion.of('5.6.39', '5.6');
/** Version "5.6.40". */
MysqlEngineVersion.VER_5_6_40 = MysqlEngineVersion.of('5.6.40', '5.6');
/** Version "5.6.41". */
MysqlEngineVersion.VER_5_6_41 = MysqlEngineVersion.of('5.6.41', '5.6');
/** Version "5.6.43". */
MysqlEngineVersion.VER_5_6_43 = MysqlEngineVersion.of('5.6.43', '5.6');
/** Version "5.6.44". */
MysqlEngineVersion.VER_5_6_44 = MysqlEngineVersion.of('5.6.44', '5.6');
/** Version "5.6.46". */
MysqlEngineVersion.VER_5_6_46 = MysqlEngineVersion.of('5.6.46', '5.6');
/** Version "5.6.48". */
MysqlEngineVersion.VER_5_6_48 = MysqlEngineVersion.of('5.6.48', '5.6');
/** Version "5.7" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_7 = MysqlEngineVersion.of('5.7', '5.7');
/** Version "5.7.16". */
MysqlEngineVersion.VER_5_7_16 = MysqlEngineVersion.of('5.7.16', '5.7');
/** Version "5.7.17". */
MysqlEngineVersion.VER_5_7_17 = MysqlEngineVersion.of('5.7.17', '5.7');
/** Version "5.7.19". */
MysqlEngineVersion.VER_5_7_19 = MysqlEngineVersion.of('5.7.19', '5.7');
/** Version "5.7.21". */
MysqlEngineVersion.VER_5_7_21 = MysqlEngineVersion.of('5.7.21', '5.7');
/** Version "5.7.22". */
MysqlEngineVersion.VER_5_7_22 = MysqlEngineVersion.of('5.7.22', '5.7');
/** Version "5.7.23". */
MysqlEngineVersion.VER_5_7_23 = MysqlEngineVersion.of('5.7.23', '5.7');
/** Version "5.7.24". */
MysqlEngineVersion.VER_5_7_24 = MysqlEngineVersion.of('5.7.24', '5.7');
/** Version "5.7.25". */
MysqlEngineVersion.VER_5_7_25 = MysqlEngineVersion.of('5.7.25', '5.7');
/** Version "5.7.26". */
MysqlEngineVersion.VER_5_7_26 = MysqlEngineVersion.of('5.7.26', '5.7');
/** Version "5.7.28". */
MysqlEngineVersion.VER_5_7_28 = MysqlEngineVersion.of('5.7.28', '5.7');
/** Version "5.7.30". */
MysqlEngineVersion.VER_5_7_30 = MysqlEngineVersion.of('5.7.30', '5.7');
/** Version "8.0" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_8_0 = MysqlEngineVersion.of('8.0', '8.0');
/** Version "8.0.11". */
MysqlEngineVersion.VER_8_0_11 = MysqlEngineVersion.of('8.0.11', '8.0');
/** Version "8.0.13". */
MysqlEngineVersion.VER_8_0_13 = MysqlEngineVersion.of('8.0.13', '8.0');
/** Version "8.0.15". */
MysqlEngineVersion.VER_8_0_15 = MysqlEngineVersion.of('8.0.15', '8.0');
/** Version "8.0.16". */
MysqlEngineVersion.VER_8_0_16 = MysqlEngineVersion.of('8.0.16', '8.0');
/** Version "8.0.17". */
MysqlEngineVersion.VER_8_0_17 = MysqlEngineVersion.of('8.0.17', '8.0');
/** Version "8.0.19". */
MysqlEngineVersion.VER_8_0_19 = MysqlEngineVersion.of('8.0.19', '8.0');
class MySqlInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mysql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mysqlFullVersion,
                    majorVersion: version.mysqlMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the PostgreSQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.postgres}).
 */
class PostgresEngineVersion {
    constructor(postgresFullVersion, postgresMajorVersion) {
        this.postgresFullVersion = postgresFullVersion;
        this.postgresMajorVersion = postgresMajorVersion;
    }
    /**
     * Create a new PostgresEngineVersion with an arbitrary version.
     *
     * @param postgresFullVersion the full version string,
     *   for example "13.11"
     * @param postgresMajorVersion the major version of the engine,
     *   for example "13"
     */
    static of(postgresFullVersion, postgresMajorVersion) {
        return new PostgresEngineVersion(postgresFullVersion, postgresMajorVersion);
    }
}
exports.PostgresEngineVersion = PostgresEngineVersion;
/** Version "9.5" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_9_5 = PostgresEngineVersion.of('9.5', '9.5');
/** Version "9.5.2". */
PostgresEngineVersion.VER_9_5_2 = PostgresEngineVersion.of('9.5.2', '9.5');
/** Version "9.5.4". */
PostgresEngineVersion.VER_9_5_4 = PostgresEngineVersion.of('9.5.4', '9.5');
/** Version "9.5.6". */
PostgresEngineVersion.VER_9_5_6 = PostgresEngineVersion.of('9.5.6', '9.5');
/** Version "9.5.7". */
PostgresEngineVersion.VER_9_5_7 = PostgresEngineVersion.of('9.5.7', '9.5');
/** Version "9.5.9". */
PostgresEngineVersion.VER_9_5_9 = PostgresEngineVersion.of('9.5.9', '9.5');
/** Version "9.5.10". */
PostgresEngineVersion.VER_9_5_10 = PostgresEngineVersion.of('9.5.10', '9.5');
/** Version "9.5.12". */
PostgresEngineVersion.VER_9_5_12 = PostgresEngineVersion.of('9.5.12', '9.5');
/** Version "9.5.13". */
PostgresEngineVersion.VER_9_5_13 = PostgresEngineVersion.of('9.5.13', '9.5');
/** Version "9.5.14". */
PostgresEngineVersion.VER_9_5_14 = PostgresEngineVersion.of('9.5.14', '9.5');
/** Version "9.5.15". */
PostgresEngineVersion.VER_9_5_15 = PostgresEngineVersion.of('9.5.15', '9.5');
/** Version "9.5.16". */
PostgresEngineVersion.VER_9_5_16 = PostgresEngineVersion.of('9.5.16', '9.5');
/** Version "9.5.18". */
PostgresEngineVersion.VER_9_5_18 = PostgresEngineVersion.of('9.5.18', '9.5');
/** Version "9.5.19". */
PostgresEngineVersion.VER_9_5_19 = PostgresEngineVersion.of('9.5.19', '9.5');
/** Version "9.5.20". */
PostgresEngineVersion.VER_9_5_20 = PostgresEngineVersion.of('9.5.20', '9.5');
/** Version "9.5.21". */
PostgresEngineVersion.VER_9_5_21 = PostgresEngineVersion.of('9.5.21', '9.5');
/** Version "9.5.22". */
PostgresEngineVersion.VER_9_5_22 = PostgresEngineVersion.of('9.5.22', '9.5');
/** Version "9.6" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_9_6 = PostgresEngineVersion.of('9.6', '9.6');
/** Version "9.6.1". */
PostgresEngineVersion.VER_9_6_1 = PostgresEngineVersion.of('9.6.1', '9.6');
/** Version "9.6.2". */
PostgresEngineVersion.VER_9_6_2 = PostgresEngineVersion.of('9.6.2', '9.6');
/** Version "9.6.3". */
PostgresEngineVersion.VER_9_6_3 = PostgresEngineVersion.of('9.6.3', '9.6');
/** Version "9.6.5". */
PostgresEngineVersion.VER_9_6_5 = PostgresEngineVersion.of('9.6.5', '9.6');
/** Version "9.6.6". */
PostgresEngineVersion.VER_9_6_6 = PostgresEngineVersion.of('9.6.6', '9.6');
/** Version "9.6.8". */
PostgresEngineVersion.VER_9_6_8 = PostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
PostgresEngineVersion.VER_9_6_9 = PostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.10". */
PostgresEngineVersion.VER_9_6_10 = PostgresEngineVersion.of('9.6.10', '9.6');
/** Version "9.6.11". */
PostgresEngineVersion.VER_9_6_11 = PostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
PostgresEngineVersion.VER_9_6_12 = PostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.14". */
PostgresEngineVersion.VER_9_6_14 = PostgresEngineVersion.of('9.6.14', '9.6');
/** Version "9.6.15". */
PostgresEngineVersion.VER_9_6_15 = PostgresEngineVersion.of('9.6.15', '9.6');
/** Version "9.6.16". */
PostgresEngineVersion.VER_9_6_16 = PostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
PostgresEngineVersion.VER_9_6_17 = PostgresEngineVersion.of('9.6.17', '9.6');
/** Version "9.6.18". */
PostgresEngineVersion.VER_9_6_18 = PostgresEngineVersion.of('9.6.18', '9.6');
/** Version "10" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_10 = PostgresEngineVersion.of('10', '10');
/** Version "10.1". */
PostgresEngineVersion.VER_10_1 = PostgresEngineVersion.of('10.1', '10');
/** Version "10.3". */
PostgresEngineVersion.VER_10_3 = PostgresEngineVersion.of('10.3', '10');
/** Version "10.4". */
PostgresEngineVersion.VER_10_4 = PostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
PostgresEngineVersion.VER_10_5 = PostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
PostgresEngineVersion.VER_10_6 = PostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
PostgresEngineVersion.VER_10_7 = PostgresEngineVersion.of('10.7', '10');
/** Version "10.9". */
PostgresEngineVersion.VER_10_9 = PostgresEngineVersion.of('10.9', '10');
/** Version "10.10". */
PostgresEngineVersion.VER_10_10 = PostgresEngineVersion.of('10.10', '10');
/** Version "10.11". */
PostgresEngineVersion.VER_10_11 = PostgresEngineVersion.of('10.11', '10');
/** Version "10.12". */
PostgresEngineVersion.VER_10_12 = PostgresEngineVersion.of('10.12', '10');
/** Version "10.13". */
PostgresEngineVersion.VER_10_13 = PostgresEngineVersion.of('10.13', '10');
/** Version "11" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_11 = PostgresEngineVersion.of('11', '11');
/** Version "11.1". */
PostgresEngineVersion.VER_11_1 = PostgresEngineVersion.of('11.1', '11');
/** Version "11.2". */
PostgresEngineVersion.VER_11_2 = PostgresEngineVersion.of('11.2', '11');
/** Version "11.4". */
PostgresEngineVersion.VER_11_4 = PostgresEngineVersion.of('11.4', '11');
/** Version "11.5". */
PostgresEngineVersion.VER_11_5 = PostgresEngineVersion.of('11.5', '11');
/** Version "11.6". */
PostgresEngineVersion.VER_11_6 = PostgresEngineVersion.of('11.6', '11');
/** Version "11.7". */
PostgresEngineVersion.VER_11_7 = PostgresEngineVersion.of('11.7', '11');
/** Version "11.8". */
PostgresEngineVersion.VER_11_8 = PostgresEngineVersion.of('11.8', '11');
/** Version "12" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_12 = PostgresEngineVersion.of('12', '12');
/** Version "12.2". */
PostgresEngineVersion.VER_12_2 = PostgresEngineVersion.of('12.2', '12');
/** Version "12.3". */
PostgresEngineVersion.VER_12_3 = PostgresEngineVersion.of('12.3', '12');
/**
 * The instance engine for PostgreSQL.
 */
class PostgresInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'postgres',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.postgresFullVersion,
                    majorVersion: version.postgresMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the legacy Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe}
 * and {@link DatabaseInstanceEngine.oracleSe1}).
 * Note: RDS will stop allowing creating new databases with this version in August 2020.
 */
class OracleLegacyEngineVersion {
    constructor(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        this.oracleLegacyFullVersion = oracleLegacyFullVersion;
        this.oracleLegacyMajorVersion = oracleLegacyMajorVersion;
    }
    static of(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        return new OracleLegacyEngineVersion(oracleLegacyFullVersion, oracleLegacyMajorVersion);
    }
}
exports.OracleLegacyEngineVersion = OracleLegacyEngineVersion;
/** Version "11.2" (only a major version, without a specific minor version). */
OracleLegacyEngineVersion.VER_11_2 = OracleLegacyEngineVersion.of('11.2', '11.2');
/** Version "11.2.0.2.v2". */
OracleLegacyEngineVersion.VER_11_2_0_2_V2 = OracleLegacyEngineVersion.of('11.2.0.2.v2', '11.2');
/** Version "11.2.0.4.v1". */
OracleLegacyEngineVersion.VER_11_2_0_4_V1 = OracleLegacyEngineVersion.of('11.2.0.4.v1', '11.2');
/** Version "11.2.0.4.v3". */
OracleLegacyEngineVersion.VER_11_2_0_4_V3 = OracleLegacyEngineVersion.of('11.2.0.4.v3', '11.2');
/** Version "11.2.0.4.v4". */
OracleLegacyEngineVersion.VER_11_2_0_4_V4 = OracleLegacyEngineVersion.of('11.2.0.4.v4', '11.2');
/** Version "11.2.0.4.v5". */
OracleLegacyEngineVersion.VER_11_2_0_4_V5 = OracleLegacyEngineVersion.of('11.2.0.4.v5', '11.2');
/** Version "11.2.0.4.v6". */
OracleLegacyEngineVersion.VER_11_2_0_4_V6 = OracleLegacyEngineVersion.of('11.2.0.4.v6', '11.2');
/** Version "11.2.0.4.v7". */
OracleLegacyEngineVersion.VER_11_2_0_4_V7 = OracleLegacyEngineVersion.of('11.2.0.4.v7', '11.2');
/** Version "11.2.0.4.v8". */
OracleLegacyEngineVersion.VER_11_2_0_4_V8 = OracleLegacyEngineVersion.of('11.2.0.4.v8', '11.2');
/** Version "11.2.0.4.v9". */
OracleLegacyEngineVersion.VER_11_2_0_4_V9 = OracleLegacyEngineVersion.of('11.2.0.4.v9', '11.2');
/** Version "11.2.0.4.v10". */
OracleLegacyEngineVersion.VER_11_2_0_4_V10 = OracleLegacyEngineVersion.of('11.2.0.4.v10', '11.2');
/** Version "11.2.0.4.v11". */
OracleLegacyEngineVersion.VER_11_2_0_4_V11 = OracleLegacyEngineVersion.of('11.2.0.4.v11', '11.2');
/** Version "11.2.0.4.v12". */
OracleLegacyEngineVersion.VER_11_2_0_4_V12 = OracleLegacyEngineVersion.of('11.2.0.4.v12', '11.2');
/** Version "11.2.0.4.v13". */
OracleLegacyEngineVersion.VER_11_2_0_4_V13 = OracleLegacyEngineVersion.of('11.2.0.4.v13', '11.2');
/** Version "11.2.0.4.v14". */
OracleLegacyEngineVersion.VER_11_2_0_4_V14 = OracleLegacyEngineVersion.of('11.2.0.4.v14', '11.2');
/** Version "11.2.0.4.v15". */
OracleLegacyEngineVersion.VER_11_2_0_4_V15 = OracleLegacyEngineVersion.of('11.2.0.4.v15', '11.2');
/** Version "11.2.0.4.v16". */
OracleLegacyEngineVersion.VER_11_2_0_4_V16 = OracleLegacyEngineVersion.of('11.2.0.4.v16', '11.2');
/** Version "11.2.0.4.v17". */
OracleLegacyEngineVersion.VER_11_2_0_4_V17 = OracleLegacyEngineVersion.of('11.2.0.4.v17', '11.2');
/** Version "11.2.0.4.v18". */
OracleLegacyEngineVersion.VER_11_2_0_4_V18 = OracleLegacyEngineVersion.of('11.2.0.4.v18', '11.2');
/** Version "11.2.0.4.v19". */
OracleLegacyEngineVersion.VER_11_2_0_4_V19 = OracleLegacyEngineVersion.of('11.2.0.4.v19', '11.2');
/** Version "11.2.0.4.v20". */
OracleLegacyEngineVersion.VER_11_2_0_4_V20 = OracleLegacyEngineVersion.of('11.2.0.4.v20', '11.2');
/** Version "11.2.0.4.v21". */
OracleLegacyEngineVersion.VER_11_2_0_4_V21 = OracleLegacyEngineVersion.of('11.2.0.4.v21', '11.2');
/** Version "11.2.0.4.v22". */
OracleLegacyEngineVersion.VER_11_2_0_4_V22 = OracleLegacyEngineVersion.of('11.2.0.4.v22', '11.2');
/** Version "11.2.0.4.v23". */
OracleLegacyEngineVersion.VER_11_2_0_4_V23 = OracleLegacyEngineVersion.of('11.2.0.4.v23', '11.2');
/** Version "11.2.0.4.v24". */
OracleLegacyEngineVersion.VER_11_2_0_4_V24 = OracleLegacyEngineVersion.of('11.2.0.4.v24', '11.2');
/**
 * The versions for the Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe2} and
 * {@link DatabaseInstanceEngine.oracleEe}).
 */
class OracleEngineVersion {
    constructor(oracleFullVersion, oracleMajorVersion) {
        this.oracleFullVersion = oracleFullVersion;
        this.oracleMajorVersion = oracleMajorVersion;
    }
    /**
     * Creates a new OracleEngineVersion with an arbitrary version.
     *
     * @param oracleFullVersion the full version string,
     *   for example "19.0.0.0.ru-2019-10.rur-2019-10.r1"
     * @param oracleMajorVersion the major version of the engine,
     *   for example "19"
     */
    static of(oracleFullVersion, oracleMajorVersion) {
        return new OracleEngineVersion(oracleFullVersion, oracleMajorVersion);
    }
}
exports.OracleEngineVersion = OracleEngineVersion;
/** Version "12.1" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_12_1 = OracleEngineVersion.of('12.1', '12.1');
/** Version "12.1.0.2.v1". */
OracleEngineVersion.VER_12_1_0_2_V1 = OracleEngineVersion.of('12.1.0.2.v1', '12.1');
/** Version "12.1.0.2.v2". */
OracleEngineVersion.VER_12_1_0_2_V2 = OracleEngineVersion.of('12.1.0.2.v2', '12.1');
/** Version "12.1.0.2.v3". */
OracleEngineVersion.VER_12_1_0_2_V3 = OracleEngineVersion.of('12.1.0.2.v3', '12.1');
/** Version "12.1.0.2.v4". */
OracleEngineVersion.VER_12_1_0_2_V4 = OracleEngineVersion.of('12.1.0.2.v4', '12.1');
/** Version "12.1.0.2.v5". */
OracleEngineVersion.VER_12_1_0_2_V5 = OracleEngineVersion.of('12.1.0.2.v5', '12.1');
/** Version "12.1.0.2.v6". */
OracleEngineVersion.VER_12_1_0_2_V6 = OracleEngineVersion.of('12.1.0.2.v6', '12.1');
/** Version "12.1.0.2.v7". */
OracleEngineVersion.VER_12_1_0_2_V7 = OracleEngineVersion.of('12.1.0.2.v7', '12.1');
/** Version "12.1.0.2.v8". */
OracleEngineVersion.VER_12_1_0_2_V8 = OracleEngineVersion.of('12.1.0.2.v8', '12.1');
/** Version "12.1.0.2.v9". */
OracleEngineVersion.VER_12_1_0_2_V9 = OracleEngineVersion.of('12.1.0.2.v9', '12.1');
/** Version "12.1.0.2.v10". */
OracleEngineVersion.VER_12_1_0_2_V10 = OracleEngineVersion.of('12.1.0.2.v10', '12.1');
/** Version "12.1.0.2.v11". */
OracleEngineVersion.VER_12_1_0_2_V11 = OracleEngineVersion.of('12.1.0.2.v11', '12.1');
/** Version "12.1.0.2.v12". */
OracleEngineVersion.VER_12_1_0_2_V12 = OracleEngineVersion.of('12.1.0.2.v12', '12.1');
/** Version "12.1.0.2.v13". */
OracleEngineVersion.VER_12_1_0_2_V13 = OracleEngineVersion.of('12.1.0.2.v13', '12.1');
/** Version "12.1.0.2.v14". */
OracleEngineVersion.VER_12_1_0_2_V14 = OracleEngineVersion.of('12.1.0.2.v14', '12.1');
/** Version "12.1.0.2.v15". */
OracleEngineVersion.VER_12_1_0_2_V15 = OracleEngineVersion.of('12.1.0.2.v15', '12.1');
/** Version "12.1.0.2.v16". */
OracleEngineVersion.VER_12_1_0_2_V16 = OracleEngineVersion.of('12.1.0.2.v16', '12.1');
/** Version "12.1.0.2.v17". */
OracleEngineVersion.VER_12_1_0_2_V17 = OracleEngineVersion.of('12.1.0.2.v17', '12.1');
/** Version "12.1.0.2.v18". */
OracleEngineVersion.VER_12_1_0_2_V18 = OracleEngineVersion.of('12.1.0.2.v18', '12.1');
/** Version "12.1.0.2.v19". */
OracleEngineVersion.VER_12_1_0_2_V19 = OracleEngineVersion.of('12.1.0.2.v19', '12.1');
/** Version "12.1.0.2.v20". */
OracleEngineVersion.VER_12_1_0_2_V20 = OracleEngineVersion.of('12.1.0.2.v20', '12.1');
/** Version "12.2" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_12_2 = OracleEngineVersion.of('12.2', '12.2');
/** Version "12.2.0.1.ru-2018-10.rur-2018-10.r1". */
OracleEngineVersion.VER_12_2_0_1_2018_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2018-10.rur-2018-10.r1', '12.2');
/** Version "12.2.0.1.ru-2019-01.rur-2019-01.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-01.rur-2019-01.r1', '12.2');
/** Version "12.2.0.1.ru-2019-04.rur-2019-04.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-04.rur-2019-04.r1', '12.2');
/** Version "12.2.0.1.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_07_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-07.rur-2019-07.r1', '12.2');
/** Version "12.2.0.1.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-10.rur-2019-10.r1', '12.2');
/** Version "12.2.0.1.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_12_2_0_1_2020_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-01.rur-2020-01.r1', '12.2');
/** Version "12.2.0.1.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_12_2_0_1_2020_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-04.rur-2020-04.r1', '12.2');
/** Version "18" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_18 = OracleEngineVersion.of('18', '18');
/** Version "18.0.0.0.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_18_0_0_0_2019_07_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-07.rur-2019-07.r1', '18');
/** Version "18.0.0.0.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_18_0_0_0_2019_10_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-10.rur-2019-10.r1', '18');
/** Version "18.0.0.0.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_18_0_0_0_2020_01_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-01.rur-2020-01.r1', '18');
/** Version "18.0.0.0.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_18_0_0_0_2020_04_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-04.rur-2020-04.r1', '18');
/** Version "19" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_19 = OracleEngineVersion.of('19', '19');
/** Version "19.0.0.0.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_19_0_0_0_2019_07_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-07.rur-2019-07.r1', '19');
/** Version "19.0.0.0.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_19_0_0_0_2019_10_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-10.rur-2019-10.r1', '19');
/** Version "19.0.0.0.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_19_0_0_0_2020_01_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-01.rur-2020-01.r1', '19');
/** Version "19.0.0.0.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_19_0_0_0_2020_04_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-04.rur-2020-04.r1', '19');
class OracleInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER,
            parameterGroupFamily: props.version ? `${props.engineType}-${props.version.majorVersion}` : undefined,
        });
    }
}
class OracleSeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
class OracleSe1InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se1',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
class OracleSe2InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se2',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
class OracleEeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-ee',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the SQL Server instance engines
 * (those returned by {@link DatabaseInstanceEngine.sqlServerSe},
 * {@link DatabaseInstanceEngine.sqlServerEx}, {@link DatabaseInstanceEngine.sqlServerWeb}
 * and {@link DatabaseInstanceEngine.sqlServerEe}).
 */
class SqlServerEngineVersion {
    constructor(sqlServerFullVersion, sqlServerMajorVersion) {
        this.sqlServerFullVersion = sqlServerFullVersion;
        this.sqlServerMajorVersion = sqlServerMajorVersion;
    }
    /**
     * Create a new SqlServerEngineVersion with an arbitrary version.
     *
     * @param sqlServerFullVersion the full version string,
     *   for example "15.00.3049.1.v1"
     * @param sqlServerMajorVersion the major version of the engine,
     *   for example "15.00"
     */
    static of(sqlServerFullVersion, sqlServerMajorVersion) {
        return new SqlServerEngineVersion(sqlServerFullVersion, sqlServerMajorVersion);
    }
}
exports.SqlServerEngineVersion = SqlServerEngineVersion;
/** Version "11.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_11 = SqlServerEngineVersion.of('11.00', '11.00');
/** Version "11.00.5058.0.v1". */
SqlServerEngineVersion.VER_11_00_5058_0_V1 = SqlServerEngineVersion.of('11.00.5058.0.v1', '11.00');
/** Version "11.00.6020.0.v1". */
SqlServerEngineVersion.VER_11_00_6020_0_V1 = SqlServerEngineVersion.of('11.00.6020.0.v1', '11.00');
/** Version "11.00.6594.0.v1". */
SqlServerEngineVersion.VER_11_00_6594_0_V1 = SqlServerEngineVersion.of('11.00.6594.0.v1', '11.00');
/** Version "11.00.7462.6.v1". */
SqlServerEngineVersion.VER_11_00_7462_6_V1 = SqlServerEngineVersion.of('11.00.7462.6.v1', '11.00');
/** Version "11.00.7493.4.v1". */
SqlServerEngineVersion.VER_11_00_7493_4_V1 = SqlServerEngineVersion.of('11.00.7493.4.v1', '11.00');
/** Version "12.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_12 = SqlServerEngineVersion.of('12.00', '12.00');
/** Version "12.00.5000.0.v1". */
SqlServerEngineVersion.VER_12_00_5000_0_V1 = SqlServerEngineVersion.of('12.00.5000.0.v1', '12.00');
/** Version "12.00.5546.0.v1". */
SqlServerEngineVersion.VER_12_00_5546_0_V1 = SqlServerEngineVersion.of('12.00.5546.0.v1', '12.00');
/** Version "12.00.5571.0.v1". */
SqlServerEngineVersion.VER_12_00_5571_0_V1 = SqlServerEngineVersion.of('12.00.5571.0.v1', '12.00');
/** Version "12.00.6293.0.v1". */
SqlServerEngineVersion.VER_12_00_6293_0_V1 = SqlServerEngineVersion.of('12.00.6293.0.v1', '12.00');
/** Version "12.00.6329.1.v1". */
SqlServerEngineVersion.VER_12_00_6329_1_V1 = SqlServerEngineVersion.of('12.00.6329.1.v1', '12.00');
/** Version "13.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_13 = SqlServerEngineVersion.of('13.00', '13.00');
/** Version "13.00.2164.0.v1". */
SqlServerEngineVersion.VER_13_00_2164_0_V1 = SqlServerEngineVersion.of('13.00.2164.0.v1', '13.00');
/** Version "13.00.4422.0.v1". */
SqlServerEngineVersion.VER_13_00_4422_0_V1 = SqlServerEngineVersion.of('13.00.4422.0.v1', '13.00');
/** Version "13.00.4451.0.v1". */
SqlServerEngineVersion.VER_13_00_4451_0_V1 = SqlServerEngineVersion.of('13.00.4451.0.v1', '13.00');
/** Version "13.00.4466.4.v1". */
SqlServerEngineVersion.VER_13_00_4466_4_V1 = SqlServerEngineVersion.of('13.00.4466.4.v1', '13.00');
/** Version "13.00.4522.0.v1". */
SqlServerEngineVersion.VER_13_00_4522_0_V1 = SqlServerEngineVersion.of('13.00.4522.0.v1', '13.00');
/** Version "13.00.5216.0.v1". */
SqlServerEngineVersion.VER_13_00_5216_0_V1 = SqlServerEngineVersion.of('13.00.5216.0.v1', '13.00');
/** Version "13.00.5292.0.v1". */
SqlServerEngineVersion.VER_13_00_5292_0_V1 = SqlServerEngineVersion.of('13.00.5292.0.v1', '13.00');
/** Version "13.00.5366.0.v1". */
SqlServerEngineVersion.VER_13_00_5366_0_V1 = SqlServerEngineVersion.of('13.00.5366.0.v1', '13.00');
/** Version "13.00.5426.0.v1". */
SqlServerEngineVersion.VER_13_00_5426_0_V1 = SqlServerEngineVersion.of('13.00.5426.0.v1', '13.00');
/** Version "13.00.5598.27.v1". */
SqlServerEngineVersion.VER_13_00_5598_27_V1 = SqlServerEngineVersion.of('13.00.5598.27.v1', '13.00');
/** Version "14.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_14 = SqlServerEngineVersion.of('14.00', '14.00');
/** Version "14.00.1000.169.v1". */
SqlServerEngineVersion.VER_14_00_1000_169_V1 = SqlServerEngineVersion.of('14.00.1000.169.v1', '14.00');
/** Version "14.00.3015.40.v1". */
SqlServerEngineVersion.VER_14_00_3015_40_V1 = SqlServerEngineVersion.of('14.00.3015.40.v1', '14.00');
/** Version "14.00.3035.2.v1". */
SqlServerEngineVersion.VER_14_00_3035_2_V1 = SqlServerEngineVersion.of('14.00.3035.2.v1', '14.00');
/** Version "14.00.3049.1.v1". */
SqlServerEngineVersion.VER_14_00_3049_1_V1 = SqlServerEngineVersion.of('14.00.3049.1.v1', '14.00');
/** Version "14.00.3192.2.v1". */
SqlServerEngineVersion.VER_14_00_3192_2_V1 = SqlServerEngineVersion.of('14.00.3192.2.v1', '14.00');
class SqlServerInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER,
            version: props.version
                ? {
                    fullVersion: props.version.sqlServerFullVersion,
                    majorVersion: props.version.sqlServerMajorVersion,
                }
                : undefined,
            parameterGroupFamily: props.version
                // for some reason, even though SQL Server major versions usually end in '.00',
                // the ParameterGroup family has to end in '.0'
                ? `${props.engineType}-${props.version.sqlServerMajorVersion.endsWith('.00')
                    ? props.version.sqlServerMajorVersion.slice(0, -1)
                    : props.version.sqlServerMajorVersion}`
                : undefined,
        });
    }
    bindToInstance(_scope, _options) {
        return {};
    }
}
class SqlServerSeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-se',
            version,
        });
    }
}
class SqlServerExInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ex',
            version,
        });
    }
}
class SqlServerWebInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-web',
            version,
        });
    }
}
class SqlServerEeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ee',
            version,
        });
    }
}
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
class DatabaseInstanceEngine {
    /** Creates a new MariaDB instance engine. */
    static mariaDb(props) {
        return new MariaDbInstanceEngine(props.version);
    }
    /** Creates a new MySQL instance engine. */
    static mysql(props) {
        return new MySqlInstanceEngine(props.version);
    }
    /** Creates a new PostgreSQL instance engine. */
    static postgres(props) {
        return new PostgresInstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition instance engine. */
    static oracleSe(props) {
        return new OracleSeInstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition 1 instance engine. */
    static oracleSe1(props) {
        return new OracleSe1InstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition 1 instance engine. */
    static oracleSe2(props) {
        return new OracleSe2InstanceEngine(props.version);
    }
    /** Creates a new Oracle Enterprise Edition instance engine. */
    static oracleEe(props) {
        return new OracleEeInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Standard Edition instance engine. */
    static sqlServerSe(props) {
        return new SqlServerSeInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Express Edition instance engine. */
    static sqlServerEx(props) {
        return new SqlServerExInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Web Edition instance engine. */
    static sqlServerWeb(props) {
        return new SqlServerWebInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Enterprise Edition instance engine. */
    static sqlServerEe(props) {
        return new SqlServerEeInstanceEngine(props.version);
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/**
 * The unversioned 'mariadb' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link mariaDb()} method
 */
DatabaseInstanceEngine.MARIADB = new MariaDbInstanceEngine();
/**
 * The unversioned 'mysql' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link mysql()} method
 */
DatabaseInstanceEngine.MYSQL = new MySqlInstanceEngine();
/**
 * The unversioned 'oracle-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleEe()} method
 */
DatabaseInstanceEngine.ORACLE_EE = new OracleEeInstanceEngine();
/**
 * The unversioned 'oracle-se2' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe2()} method
 */
DatabaseInstanceEngine.ORACLE_SE2 = new OracleSe2InstanceEngine();
/**
 * The unversioned 'oracle-se1' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe1()} method
 */
DatabaseInstanceEngine.ORACLE_SE1 = new OracleSe1InstanceEngine();
/**
 * The unversioned 'oracle-se' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe()} method
 */
DatabaseInstanceEngine.ORACLE_SE = new OracleSeInstanceEngine();
/**
 * The unversioned 'postgres' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link postgres()} method
 */
DatabaseInstanceEngine.POSTGRES = new PostgresInstanceEngine();
/**
 * The unversioned 'sqlserver-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerEe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EE = new SqlServerEeInstanceEngine();
/**
 * The unversioned 'sqlserver-se' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerSe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_SE = new SqlServerSeInstanceEngine();
/**
 * The unversioned 'sqlserver-ex' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerEx()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EX = new SqlServerExInstanceEngine();
/**
 * The unversioned 'sqlserver-web' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerWeb()} method
 */
DatabaseInstanceEngine.SQL_SERVER_WEB = new SqlServerWebInstanceEngine();
//# sourceMappingURL=data:application/json;base64,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