"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Add a new db proxy to this cluster.
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromCluster(this),
            ...options,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        /**
         * Identifiers of the replicas
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual replica.
         */
        this.instanceEndpoints = [];
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            this.node.addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'Subnets', {
            dbSubnetGroupDescription: `Subnets for ${id} database`,
            subnetIds,
        });
        if (props.removalPolicy === core_1.RemovalPolicy.RETAIN) {
            subnetGroup.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        const securityGroups = (_a = props.instanceProps.securityGroups) !== null && _a !== void 0 ? _a : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: props.instanceProps.vpc,
            }),
        ];
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        let s3ImportRole = props.s3ImportRole;
        if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
            if (props.s3ImportRole) {
                throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
            }
            s3ImportRole = new aws_iam_1.Role(this, 'S3ImportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
            });
            for (const bucket of props.s3ImportBuckets) {
                bucket.grantRead(s3ImportRole);
            }
        }
        let s3ExportRole = props.s3ExportRole;
        if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
            if (props.s3ExportRole) {
                throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
            }
            s3ExportRole = new aws_iam_1.Role(this, 'S3ExportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
            });
            for (const bucket of props.s3ExportBuckets) {
                bucket.grantReadWrite(s3ExportRole);
            }
        }
        const clusterAssociatedRoles = [];
        if (s3ImportRole || s3ExportRole) {
            if (s3ImportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn });
            }
            if (s3ExportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn });
            }
        }
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            s3ImportRole,
            s3ExportRole,
            parameterGroup: props.parameterGroup,
        });
        const clusterParameterGroup = (_b = props.parameterGroup) !== null && _b !== void 0 ? _b : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engine: props.engine.engineType,
            engineVersion: (_c = props.engine.engineVersion) === null || _c === void 0 ? void 0 : _c.fullVersion,
            dbClusterIdentifier: props.clusterIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            vpcSecurityGroupIds: securityGroups.map(sg => sg.securityGroupId),
            port: (_d = props.port) !== null && _d !== void 0 ? _d : clusterEngineBindConfig.port,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : (props.masterUser.password
                    ? props.masterUser.password.toString()
                    : undefined),
            backupRetentionPeriod: props.backup && props.backup.retention && props.backup.retention.toDays(),
            preferredBackupWindow: props.backup && props.backup.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            // Encryption
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        // if removalPolicy was not specified,
        // leave it as the default, which is Snapshot
        if (props.removalPolicy) {
            cluster.applyRemovalPolicy(props.removalPolicy);
        }
        else {
            // The CFN default makes sense for DeletionPolicy,
            // but doesn't cover UpdateReplacePolicy.
            // Fix that here.
            cluster.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
        }
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const instanceCount = props.instances != null ? props.instances : 2;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        // Get the actual subnet objects so we can depend on internet connectivity.
        const internetConnected = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets).internetConnectivityEstablished;
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new aws_iam_1.Role(this, 'MonitoringRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
                ],
            });
        }
        const instanceType = (_e = props.instanceProps.instanceType) !== null && _e !== void 0 ? _e : ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MEDIUM);
        const instanceParameterGroupConfig = (_f = props.instanceProps.parameterGroup) === null || _f === void 0 ? void 0 : _f.bindToInstance({});
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
                props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                    undefined;
            const publiclyAccessible = props.instanceProps.vpcSubnets && props.instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC;
            const instance = new rds_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                engine: props.engine.engineType,
                engineVersion: (_g = props.engine.engineVersion) === null || _g === void 0 ? void 0 : _g.fullVersion,
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: databaseInstanceType(instanceType),
                publiclyAccessible,
                // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
                dbSubnetGroupName: subnetGroup.ref,
                dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
                monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
                monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            });
            // If removalPolicy isn't explicitly set,
            // it's Snapshot for Cluster.
            // Because of that, in this case,
            // we can safely use the CFN default of Delete for DbInstances with dbClusterIdentifier set.
            if (props.removalPolicy) {
                instance.applyRemovalPolicy(props.removalPolicy);
            }
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnected);
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups, defaultPort });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.instanceEndpoints = attrs.instanceEndpointAddresses.map(a => new endpoint_1.Endpoint(a, attrs.port));
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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