from __future__ import annotations

import abc
import dataclasses
from pathlib import Path

from kraken.cli.buildenv.environment import get_implied_requirements

from .requirements import RequirementSpec, parse_requirements_from_python_script

DEFAULT_PYTHONPATH = ["build-support"]
DEFAULT_INTERPRETER_CONSTRAINT = ">=3.7"


class ProjectInterface(abc.ABC):
    @abc.abstractmethod
    def get_requirements_path(self) -> Path:
        ...

    @abc.abstractmethod
    def get_requirement_spec(self) -> RequirementSpec:
        """Called to read the requirement spec of the project."""

        raise NotImplementedError

    @abc.abstractmethod
    def get_lock_file(self) -> Path:
        """Returns the path to the lock file."""

        raise NotImplementedError


class DefaultProjectImpl(ProjectInterface):
    """The default implementation for looking up project build requirements and reading/writing lockfiles.

    Buildscript requirements are looked up in the `.kraken.py` header if prefixed with the string `# ::requirements`.
    Lockfiles are written to a file named `.kraken.lock`."""

    LOCK_FILE = Path(".kraken.lock")

    @dataclasses.dataclass
    class Files:
        script: Path
        lock: Path

    def __init__(self, project_dir: Path | None = None, kraken_cli_develop: bool = False) -> None:
        self.project_dir = project_dir or Path.cwd()
        self.kraken_cli_develop = kraken_cli_develop
        self._files: DefaultProjectImpl.Files | None = None

    def _get_files(self) -> Files:
        """Determines the files to read/write."""

        from kraken.core.loader import get_loader_implementations

        if self._files is not None:
            return self._files

        # Find a file we can load requirements from.
        script_file: Path | None = None
        for loader in get_loader_implementations():
            script_file = loader.detect_in_project_directory(self.project_dir)
            if script_file:
                break
        if not script_file:
            raise RuntimeError("no Kraken build script found")

        self._files = self.Files(script_file, self.project_dir / self.LOCK_FILE)
        return self._files

    def get_requirements_path(self) -> Path:
        return self._get_files().script

    def get_requirement_spec(self) -> RequirementSpec:
        files = self._get_files()
        with files.script.open() as fp:
            requirements = parse_requirements_from_python_script(fp)
        requirements.add_requirements(get_implied_requirements(self.kraken_cli_develop))
        for path in DEFAULT_PYTHONPATH:
            if path not in requirements.pythonpath:
                requirements.pythonpath.append(path)
        if not requirements.interpreter_constraint:
            requirements.interpreter_constraint = DEFAULT_INTERPRETER_CONSTRAINT
        return requirements

    def get_lock_file(self) -> Path:
        return self._get_files().lock
