import numpy as np
from scipy.sparse.linalg import eigsh


def top_k(x, k):
    """
    Returns the indices of the top d features and
    the unsorted list of scores on the features.
    """

    valid_features = x.shape[0] - np.sum(np.isinf(x))

    if valid_features < k:
        print(
            f"The requested number of features ({k}) could not be screened; \
only {valid_features} features were screened instead"
        )
        k = valid_features

    selected_indices = x.argsort()[-k:][::-1]
    return selected_indices


def orthonormalize(X):
    """
    Orthonomalizes X.
    Code taken from https://github.com/TwoLittle/PC_Screen
    """
    # X is a 2-d array
    # output: Gram-Schmidt orthogonalization of X

    n, p = X.shape
    Y = np.zeros([n, p])
    Y[:, 0] = X[:, 0] / np.sqrt(np.sum(X[:, 0] ** 2))

    for j in range(1, p):

        Yj = Y[:, range(j)]
        xj = X[:, j]
        w = np.dot(xj, Yj)
        xj_p = np.sum(w * Yj, axis=1)
        yj = xj - xj_p
        yj = yj / np.sqrt(np.sum(yj**2))

        Y[:, j] = yj

    return Y


def get_equi_features(X):
    """
    Builds the knockoff variables with the equicorrelated procedure.
    Code taken from https://github.com/TwoLittle/PC_Screen
    """
    # X is 2-d array
    n, p = X.shape
    scale = np.sqrt(np.sum(X**2, axis=0))
    Xstd = X / scale
    sigma = np.dot(Xstd.T, Xstd)
    sigma_inv = np.linalg.inv(sigma)
    lambd_min = eigsh(sigma, k=1, which="SA")[0].squeeze()
    sj = np.min([1.0, 2.0 * lambd_min])
    sj = sj - 0.00001

    mat_s = np.diag([sj] * p)
    A = 2 * mat_s - sj * sj * sigma_inv
    C = np.linalg.cholesky(A).T

    Xn = np.random.randn(n, p)
    XX = np.hstack([Xstd, Xn])
    XXo = orthonormalize(XX)

    U = XXo[:, range(p, 2 * p)]

    Xnew = np.dot(Xstd, np.eye(p) - sigma_inv * sj) + np.dot(U, C)
    return Xnew
