import functools
import logging
from textwrap import dedent

from flask import request, current_app, g
from flask_mailman import Message
from sqlalchemy.orm import exc

from .. import errors
from ..auth import (
    check, make_signer, authenticate_from_session,
)
from ..utils import jsonify
from ...core.model import db, AccessKey, Identity
from ...core.user import hash_password
from ...utils import format_time
from .blueprint import blueprint


logger = logging.getLogger(__name__)


@blueprint.before_request
def auth_middleware():
    g.aws_service = 'cornac'  # For auth.check()
    g.current_action = build_action_from_request()
    g.signer = make_signer(salt=request.remote_addr)

    try:
        authenticate_from_session()
    except errors.AWSError as e:
        current_app.logger.debug("Session not loaded: %s", e.description)
        g.access_key = g.current_identity = g.current_account = None
    else:
        current_app.logger.debug(
            "Authenticated by session token as %s.", g.current_identity.name)


def login_required(f):
    @functools.wraps(f)
    def wrapper(*a, **kw):
        if not g.access_key:
            return jsonify(status=403, message="Login required")
        g.last_acl_result = False
        response = f(*a, **kw)
        if not g.last_acl_result:
            # This prevent information leaking, but not damage.
            logger.error(
                "Permission not checked by %s, refusing to answer.",
                f.__name__)
            raise errors.AccessDenied()
        return response
    return wrapper


@blueprint.route('/login', methods=['POST'])
def login():
    data = request.json or request.form
    try:
        username = data['username']
        password = data['password']
    except KeyError:
        return jsonify(status=400, message="Missing username or password.")

    error = jsonify(status=403, message="Wrong username or password.")
    try:
        user = Identity.query.users().get_by(name=username)
    except exc.NoResultFound:
        current_app.logger.info("Login faild for %s, inexistant.", username)
        # Generated by fair call to c.u.pwgen.
        # Guaranteed to be random.
        fake_salt = 'ddVMQTMorwAZ'
        # Note: checking password costs some times to compute hash. Thus, an
        # attacker could steal user name by testing the time to response.
        # Compute a useless hash to prevent this.
        hash_password(password, salt=fake_salt)
        return error

    if not user.check_password(password):
        current_app.logger.info("Login of %s failed. Invalid password.", user)
        return error

    g.current_identity = user
    g.current_account = user.account

    policies = ['IAMUserAccessKey', 'IAMUserChangePassword']
    # Simply check for Admin universal allow statement to test for admin
    # policies.
    if check(action='*', raise_=False).allowed:
        policies.extend([
            'IAMFullAccess',
            'CornacAccountFullAccess',
        ])

    access_key = AccessKey.factory(identity=user, temporary=True)
    current_app.logger.info(
        "Login of %s succeed. Access key is %s.", user, access_key)
    db.session.commit()

    return make_login_response(access_key, "Authentified.", policies=policies)


@blueprint.route('/login/renew', methods=['POST'])
@login_required
def renew_session():
    # Declare that one is allowed to renew itself, without relying on ACL.
    g.last_acl_result = True
    new_access_key = AccessKey.factory(
        identity=g.access_key.identity, temporary=True)
    db.session.commit()

    return make_login_response(
        new_access_key, message="Session renewed.",
    )


def build_action_from_request():
    # Builds action name from url_rule object. The output looks like AWS
    # privilege identifier. Action is prefixed by cornac: service name and by
    # HTTP verb. e.g. cornac:GetAccounts, cornac:PostLogin, etc.
    return (
        'cornac' +  # Preserve lowercase
        request.url_rule.endpoint[6:]
        .title()
        .replace('.', ':' + request.method.title())
        .replace('_', '')
    )


def make_login_response(access_key, message, **kw):
    signed_token = g.signer.dumps(access_key.session_token)
    current_app.logger.debug("Signing for %s.", request.remote_addr)
    response = jsonify(
        message=message,
        credentials=dict(
            AccessKeyId=access_key.access_key,
            SecretAccessKey=access_key.data['SecretAccessKey'],
            SessionToken=signed_token,
            Expiration=format_time(access_key.edate)
        ),
        **kw,
    )
    response.set_cookie(
        'session_token', signed_token, expires=access_key.edate)
    return response


@blueprint.route('/reset_password/', methods=['POST'])
def send_reset_password_link():
    data = request.json or request.form
    try:
        username = data['username']
    except KeyError:
        return jsonify(400, message="Missing username.")

    try:
        user = Identity.query.users().get_by(name=username)
    except exc.NoResultFound:
        pass
    else:
        apiurl = user.make_reset_password_url()
        db.session.commit()
        formurl = apiurl.replace('cornac/', 'cornac/#/')
        send_reset_password_email(
            current_app, user.name, user.reset_edate, formurl
        )

    # Always return ok to prevent mail address scrapping.
    return jsonify(200, message="Password reset link sent. Check your mails.")


def send_reset_password_email(app, recipient, expiration_date, formurl):
    product_name = "Cornac"
    body = dedent(f"""
    Hi,

    You recently requested to reset your password for your {product_name}
    account. Use the link below to reset it. This password reset is valid
    until {expiration_date.strftime('%d/%m/%Y %H:%M UTC')}.

    Reset your password at {formurl}

    If you did not request a password reset, please ignore this email.

    Thanks,
    The {product_name} Team
    """)
    msg = Message(
        subject=f"{product_name} password reset",
        body=body,
        sender=current_app.config['MAIL_FROM'],
        recipients=[recipient],
    )
    app.mail.send(msg)
    app.logger.info("Password reset email sent for %s:", recipient)


@blueprint.route('/reset_password/<token>', methods=['POST'])
def reset_password(token):
    try:
        user = Identity.query.users().get_password_reset(token)
    except exc.NoResultFound:
        return jsonify(404, message="Invalid or expired token.")

    data = request.json or request.form
    try:
        password = data['password']
    except KeyError:
        return jsonify(400, message="Missing password.")

    user.store_password(password)
    db.session.commit()

    return jsonify(message="Password set.")
