from pathlib import Path
from typing import Type, Dict, Any, Generator, Optional, Union
import pandas as pd
from fastapi import Query
from progress.bar import Bar
from pydantic import Field, BaseModel
from pydantic.dataclasses import dataclass
from pyimporters_plugins.base import KnowledgeParserBase, Term

from pyimporters_csv.text import TXTOptions


@dataclass
class ExcelOptions(TXTOptions):
    """
    Options for the Excel knowledge import
    """
    multivalue_separator: str = Query("|",
                                      description="Additional separator to split multivalued columns if any")
    header: Optional[int] = Query(0,
                                  description="Row number (0-indexed) to use as the column names, leave blank if there is no header")
    identifier_col: str = Query("identifier",
                                description="Column to use as the identifier of the concept, either given as string name or column index")
    prefLabel_col: str = Query("prefLabel",
                               description="Column to use as the prefLabel of the concept, either given as string name or column index")
    altLabel_cols: str = Query("altLabel",
                               description="Column(s) to use as the altLabel of the concept, either given as a (list of) string name(s) or column index(es)")


    multivalue_separator : str = Field("|", description="Additional separator to split multivalued columns if any")
    header : Optional[int] = Field(0, description="Row number (0-indexed) to use as the column names, leave blank if there is no header")
    identifier_col : str = Field("identifier", description="Column to use as the identifier of the concept, either given as string name or column index, if not set use the prefLabel column")
    prefLabel_col : str = Field("prefLabel", description="Column to use as the prefLabel of the concept, either given as string name or column index")
    altLabel_cols : str = Field("altLabel", description="Column(s) to use as the altLabel of the concept, either given as a (list of) string name(s) or column index(es)")


ExcelOptionsModel = ExcelOptions.__pydantic_model__

class ExcelKnowledgeParser(KnowledgeParserBase):
    def parse(self, source: Path, options: Union[BaseModel, Dict[str, Any]], bar: Bar) -> Generator[Term, None, None]:
        options = ExcelOptionsModel(**options) if isinstance(options, dict) else options
        lines = pd.read_excel(source,
                            header=options.header).fillna(value='')
        bar.max = len(lines)
        bar.start()
        prefLabel_col = 0 if (options.prefLabel_col is None or not options.prefLabel_col.strip()) else col_index(options.prefLabel_col)
        identifier_col = prefLabel_col if (options.identifier_col is None or not options.identifier_col.strip()) else col_index(options.identifier_col)
        altLabel_cols =  None if (options.altLabel_cols is None or not options.altLabel_cols.strip()) else options.altLabel_cols
        all_cols = [col for col in lines.columns if col not in [prefLabel_col, identifier_col]] if altLabel_cols else None
        for index, row in lines.iterrows():
            bar.next()
            prefLabel = row[prefLabel_col].strip()
            identifier = row[identifier_col].strip()
            concept : Term = Term(identifier=identifier, prefLabel=prefLabel)
            if altLabel_cols:
                concept.altLabel = []
                alts_cols = [col_index(x.strip()) for x in altLabel_cols.split(',')]
                restrict =  any(col.startswith("-") for col in alts_cols)
                if restrict:
                    list_cols = [col for col in all_cols if f"-{col}" not in alts_cols]
                    alts_cols = list_cols
                for alt_col in alts_cols:
                    altLabel = row[alt_col].strip()
                    if altLabel:
                        if options.multivalue_separator:
                            altLabels = [x.strip() for x in altLabel.split(options.multivalue_separator)]
                            concept.altLabel.extend(altLabels)
                        else:
                            concept.altLabel.add(altLabel)
            yield concept
        bar.finish()

    @classmethod
    def get_schema(cls) -> Type[BaseModel]:
        return ExcelOptions

def col_index(col):
    return int(col) if col.lstrip('+-').isdigit() else col

