import numpy as np
import pandas as pd


class DetectorPositions:
    """
    Hardcoded detector positions from Armin's "DetectorPositions.txt" file.

    From that file:
    ELEVATION: Vertical FOV in radians --
        A4_11 and A6_11 set the zero line
    AZIMUTH: Horizontal FOV in radians --
        mid-point between A4_11 and A6_11 sets zero line
    These are detector center locations as determined by finding
    the mid-point of the half max amplitude of each detector's signal.
    Note that the detector numbers refer to those of the A arrays
        -- the B arrays are numbered in reverse.
    """

    elevation = pd.DataFrame(
        {
            "Detector": {
                0: 1,
                1: 2,
                2: 3,
                3: 4,
                4: 5,
                5: 6,
                6: 7,
                7: 8,
                8: 9,
                9: 10,
                10: 11,
                11: 12,
                12: 13,
                13: 14,
                14: 15,
                15: 16,
                16: 17,
                17: 18,
                18: 19,
                19: 20,
                20: 21,
            },
            "A4": {
                0: -0.0330511,
                1: -0.0298371,
                2: -0.026534699999999998,
                3: -0.0232305,
                4: -0.019925599999999998,
                5: -0.0166077,
                6: -0.013266,
                7: -0.0099561,
                8: -0.006669499999999999,
                9: -0.0033201999999999997,
                10: 0.0,
                11: 0.0033485,
                12: 0.0066495,
                13: 0.009929799999999999,
                14: 0.0132588,
                15: 0.016555599999999997,
                16: 0.0198787,
                17: 0.0232006,
                18: 0.026476799999999998,
                19: 0.029792900000000004,
                20: 0.033023800000000006,
            },
            "A3": {
                0: -0.0332537,
                1: -0.029964099999999997,
                2: -0.026611799999999998,
                3: -0.023257299999999998,
                4: -0.019928799999999997,
                5: -0.0165891,
                6: -0.013252100000000001,
                7: -0.009898200000000001,
                8: -0.0065889,
                9: -0.0032671999999999996,
                10: 7.8e-05,
                11: 0.0034189999999999997,
                12: 0.006775499999999999,
                13: 0.01012,
                14: 0.013439700000000002,
                15: 0.016770900000000002,
                16: 0.020099400000000003,
                17: 0.023453599999999998,
                18: 0.026805700000000002,
                19: 0.030136400000000004,
                20: 0.0334015,
            },
            "A2": {
                0: -0.033493,
                1: -0.030176099999999997,
                2: -0.0267763,
                3: -0.0234253,
                4: -0.0200733,
                5: -0.0167067,
                6: -0.0133506,
                7: -0.009977,
                8: -0.0066560000000000005,
                9: -0.0033097,
                10: 5.8200000000000005e-05,
                11: 0.0033996,
                12: 0.006776699999999999,
                13: 0.01014,
                14: 0.013495400000000001,
                15: 0.0168669,
                16: 0.02026,
                17: 0.0236118,
                18: 0.026998700000000004,
                19: 0.030366900000000002,
                20: 0.0336718,
            },
            "A1": {
                0: -0.0336025,
                1: -0.030256,
                2: -0.026870299999999996,
                3: -0.0234781,
                4: -0.020126400000000003,
                5: -0.0167499,
                6: -0.013354099999999999,
                7: -0.009979799999999999,
                8: -0.0066268,
                9: -0.0032205999999999997,
                10: 0.00012179999999999999,
                11: 0.0034561,
                12: 0.0068709,
                13: 0.010256200000000002,
                14: 0.0136358,
                15: 0.0170352,
                16: 0.0204296,
                17: 0.0238298,
                18: 0.0272238,
                19: 0.0305887,
                20: 0.033917199999999995,
            },
            "A5": {
                0: -0.0340995,
                1: -0.0306916,
                2: -0.0272803,
                3: -0.0239018,
                4: -0.0204772,
                5: -0.017102799999999998,
                6: -0.0136591,
                7: -0.0102781,
                8: -0.0068899,
                9: -0.0035016999999999995,
                10: -0.0001038,
                11: 0.0032943000000000004,
                12: 0.006701,
                13: 0.010090700000000001,
                14: 0.013513299999999999,
                15: 0.0169124,
                16: 0.020325299999999998,
                17: 0.0237272,
                18: 0.0271402,
                19: 0.0305425,
                20: 0.0338494,
            },
            "A6": {
                0: -0.034055699999999994,
                1: -0.030765300000000002,
                2: -0.0274151,
                3: -0.0239946,
                4: -0.020548900000000002,
                5: -0.017063299999999997,
                6: -0.0136619,
                7: -0.0102388,
                8: -0.0068461,
                9: -0.0034241999999999996,
                10: 0.0,
                11: 0.0033878999999999997,
                12: 0.006829899999999999,
                13: 0.010241,
                14: 0.013654099999999999,
                15: 0.0171002,
                16: 0.0205049,
                17: 0.023906900000000002,
                18: 0.027267000000000003,
                19: 0.030654200000000003,
                20: 0.034033999999999995,
            },
            "B1": {
                20: -0.0344496,
                19: -0.031021299999999998,
                18: -0.0276797,
                17: -0.0242102,
                16: -0.020797,
                15: -0.0172824,
                14: -0.013899799999999999,
                13: -0.010383499999999999,
                12: -0.0069949,
                11: -0.0035597000000000003,
                10: -5.8099999999999996e-05,
                9: 0.0033585999999999998,
                8: 0.0068312,
                7: 0.0102604,
                6: 0.013669,
                5: 0.017148300000000002,
                4: 0.0204754,
                3: 0.0240801,
                2: 0.027438499999999998,
                1: 0.0308632,
                0: 0.0343279,
            },
            "B2": {
                20: -0.0339267,
                19: -0.030581099999999996,
                18: -0.027221699999999998,
                17: -0.023817599999999998,
                16: -0.0204386,
                15: -0.017016099999999996,
                14: -0.013606,
                13: -0.010188299999999999,
                12: -0.0068042,
                11: -0.0033608000000000006,
                10: 3.0299999999999998e-05,
                9: 0.0034279000000000002,
                8: 0.0067472999999999995,
                7: 0.0102082,
                6: 0.013654,
                5: 0.0169993,
                4: 0.0203585,
                3: 0.023827,
                2: 0.0272178,
                1: 0.0306832,
                0: 0.0340398,
            },
            "B3": {
                20: -0.0331856,
                19: -0.030197,
                18: -0.026907999999999998,
                17: -0.023352200000000004,
                16: -0.020097200000000003,
                15: -0.0168174,
                14: -0.0134365,
                13: -0.0100571,
                12: -0.0067622,
                11: -0.0033677,
                10: -0.0001217,
                9: 0.0032018999999999997,
                8: 0.0063743,
                7: 0.009662299999999999,
                6: 0.0132764,
                5: 0.016659900000000002,
                4: 0.0198409,
                3: 0.0232454,
                2: 0.0266402,
                1: 0.0300327,
                0: 0.0331806,
            },
        }
    ).set_index("Detector")
    # AZIMUTH
    azimuth = pd.DataFrame(
        {
            "Detector": {
                0: 1,
                1: 2,
                2: 3,
                3: 4,
                4: 5,
                5: 6,
                6: 7,
                7: 8,
                8: 9,
                9: 10,
                10: 11,
                11: 12,
                12: 13,
                13: 14,
                14: 15,
                15: 16,
                16: 17,
                17: 18,
                18: 19,
                19: 20,
                20: 21,
            },
            "A4": {
                0: -0.0346129,
                1: -0.0346581,
                2: -0.0346773,
                3: -0.0346554,
                4: -0.034694300000000004,
                5: -0.034709699999999996,
                6: -0.0347144,
                7: -0.034721499999999995,
                8: -0.0347323,
                9: -0.0347392,
                10: -0.0347608,
                11: -0.034763800000000004,
                12: -0.0347398,
                13: -0.034735300000000004,
                14: -0.0347392,
                15: -0.034727,
                16: -0.0346689,
                17: -0.0346446,
                18: -0.03467,
                19: -0.034666300000000004,
                20: -0.0346811,
            },
            "A3": {
                0: -0.0218115,
                1: -0.0218108,
                2: -0.0218111,
                3: -0.0218234,
                4: -0.021838200000000002,
                5: -0.0218606,
                6: -0.0218652,
                7: -0.021859599999999996,
                8: -0.0218577,
                9: -0.0218644,
                10: -0.021897,
                11: -0.0218825,
                12: -0.0218851,
                13: -0.021881099999999997,
                14: -0.021889099999999998,
                15: -0.0218782,
                16: -0.0218555,
                17: -0.0218389,
                18: -0.0218207,
                19: -0.021790599999999997,
                20: -0.0218459,
            },
            "A2": {
                0: -0.0084075,
                1: -0.0084012,
                2: -0.0084079,
                3: -0.0084637,
                4: -0.0084589,
                5: -0.0084791,
                6: -0.008468700000000001,
                7: -0.0084564,
                8: -0.0084615,
                9: -0.0084737,
                10: -0.0084719,
                11: -0.0084477,
                12: -0.0084338,
                13: -0.0084572,
                14: -0.0084678,
                15: -0.0085116,
                16: -0.0084744,
                17: -0.008443,
                18: -0.0083374,
                19: -0.0082824,
                20: -0.008445000000000001,
            },
            "A1": {
                0: 0.0052574,
                1: 0.0052588,
                2: 0.0052503,
                3: 0.0052413,
                4: 0.0052097,
                5: 0.0052349,
                6: 0.005238399999999999,
                7: 0.0052645,
                8: 0.0052541,
                9: 0.0052619,
                10: 0.0052347999999999995,
                11: 0.0052419,
                12: 0.0052405,
                13: 0.005242899999999999,
                14: 0.0052507,
                15: 0.0052558,
                16: 0.0052417,
                17: 0.005248,
                18: 0.0052738,
                19: 0.005303,
                20: 0.0052775,
            },
            "A5": {
                0: 0.0194176,
                1: 0.0193504,
                2: 0.0193608,
                3: 0.0193233,
                4: 0.019343700000000002,
                5: 0.0193371,
                6: 0.0193759,
                7: 0.0193502,
                8: 0.0193535,
                9: 0.0193416,
                10: 0.0193233,
                11: 0.0193776,
                12: 0.0194239,
                13: 0.019438499999999997,
                14: 0.019409799999999998,
                15: 0.019411900000000003,
                16: 0.019476499999999997,
                17: 0.0194698,
                18: 0.0194697,
                19: 0.0194563,
                20: 0.0193865,
            },
            "A6": {
                0: 0.0346912,
                1: 0.03472430000000001,
                2: 0.03473130000000001,
                3: 0.0347273,
                4: 0.034721499999999995,
                5: 0.0347309,
                6: 0.034756,
                7: 0.034761400000000005,
                8: 0.0347654,
                9: 0.0347706,
                10: 0.0347608,
                11: 0.0347694,
                12: 0.0347734,
                13: 0.03476919999999999,
                14: 0.0347715,
                15: 0.0347403,
                16: 0.034735699999999994,
                17: 0.034736699999999995,
                18: 0.034736199999999995,
                19: 0.034736199999999995,
                20: 0.0347452,
            },
            "B1": {
                20: -0.0311116,
                19: -0.0313861,
                18: -0.0310815,
                17: -0.031399500000000004,
                16: -0.0308452,
                15: -0.0305785,
                14: -0.030702099999999996,
                13: -0.030823700000000002,
                12: -0.0307541,
                11: -0.0309668,
                10: -0.030662799999999997,
                9: -0.030484800000000003,
                8: -0.030647900000000006,
                7: -0.030590199999999998,
                6: -0.0308623,
                5: -0.030697199999999997,
                4: -0.0306303,
                3: -0.0308746,
                2: -0.0312946,
                1: -0.0312732,
                0: -0.0310621,
            },
            "B2": {
                20: 0.0047225,
                19: 0.004520000000000001,
                18: 0.004425899999999999,
                17: 0.0045302,
                16: 0.0046396,
                15: 0.0046931,
                14: 0.0047833,
                13: 0.0047709,
                12: 0.0048442,
                11: 0.0049118,
                10: 0.0048927,
                9: 0.0048866000000000005,
                8: 0.0048263,
                7: 0.0047776,
                6: 0.0046365,
                5: 0.004396199999999999,
                4: 0.004247,
                3: 0.0044155,
                2: 0.0051536,
                1: 0.0049661,
                0: 0.004933,
            },
            "B3": {
                20: 0.0376707,
                19: 0.0375575,
                18: 0.0375384,
                17: 0.037490699999999995,
                16: 0.03752369999999999,
                15: 0.037520099999999994,
                14: 0.0375666,
                13: 0.0375965,
                12: 0.0375365,
                11: 0.0374781,
                10: 0.0373413,
                9: 0.037200699999999996,
                8: 0.0372699,
                7: 0.03742519999999999,
                6: 0.0376603,
                5: 0.0378506,
                4: 0.0378733,
                3: 0.03777369999999999,
                2: 0.037735000000000005,
                1: 0.0377218,
                0: 0.0377606,
            },
        }
    )

    def convert_fov_to_altitude(
        self, spacecraft_radius, scene_radius, scene_altitude, fov
    ):
        return scene_altitude - np.sqrt(
            spacecraft_radius**2 - scene_radius**2
        ) * np.arctan(fov)

    def get_detector_altitudes(
        self, df: pd.DataFrame, channel: str, colname_prefix="alt"
    ):
        """
        Calculate altitudes for each detector (1-21) for each L1b observation.
        Returns DF detectors as index.

        Parameters
        ----------
        df: L1b data with at least cols: ["SC_rad", "Scene_rad", "Scene_alt"]
        channel: A1-A6, B1-B3
        dp: Instance of DetectorPositions (not instantiated here for use in dask apply)
        """
        data_alts = df.apply(
            lambda row: self.convert_fov_to_altitude(
                row["SC_rad"],
                row["Scene_rad"],
                row["Scene_alt"],
                self.elevation[channel],
            ),
            axis=1,
        )
        data_alts = data_alts.rename(
            {x: f"{colname_prefix}_{str(x+1).zfill(2)}" for x in data_alts.columns},
            axis=1,
        )
        return data_alts
