"""HIT L2 constants."""

# Constants for HIT L2 intensity calculation
SECONDS_PER_MIN = 60  # seconds per minute
SECONDS_PER_10_MIN = 600  # seconds per 10 minutes

# Fill values for missing data
FILLVAL_FLOAT32 = -1.00e31

# Number of azimuth angles
N_AZIMUTH = 15

# Valid species for HIT L2 standard and summed intensity products
VALID_SPECIES = [
    "h",
    "he3",
    "he4",
    "he",
    "c",
    "n",
    "o",
    "ne",
    "na",
    "mg",
    "al",
    "si",
    "s",
    "ar",
    "ca",
    "fe",
    "ni",
]

# Valid species for HIT L2 sectored intensity product
VALID_SECTORED_SPECIES = ["h", "he4", "cno", "nemgsi", "fe"]

# Particle energy range mapping for HIT L2 standard intensity product.
# The standard intensity calculation requires the L1B foreground rates to
# be summed by particle type, energy range, and detector penetration range
# (Range 2, Range 3, and Range 4).
# See section 7.1.1 and equation 9 in the algorithm document for more details.

# The rates to sum are in the L2FGRATES, L3FGRATES, and PENFGRATES data
# variables in the L1B standard rates product. These variables represent
# different detector ranges for each particle type and energy range.

# Indices at each detector range are provided for each particle type
# and energy range in the dictionary below and the rates at these indices will be
# summed in L2 processing to produce the standard intensity product.
# R2 = Indices for Range 2 (L2FGRATES)
# R3 = Indices for Range 3 (L3FGRATES)
# R4 = Indices for Range 4 (PENFGRATES)
# energy_units: MeV/n

STANDARD_PARTICLE_ENERGY_RANGE_MAPPING = {
    "h": [
        {"energy_min": 1.8, "energy_max": 2.2, "R2": [1], "R3": [], "R4": []},
        {"energy_min": 2.2, "energy_max": 2.7, "R2": [2], "R3": [], "R4": []},
        {"energy_min": 2.7, "energy_max": 3.2, "R2": [3], "R3": [], "R4": []},
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [4], "R3": [1], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [5], "R3": [2], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [6], "R3": [3], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [7], "R3": [4], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [], "R3": [5], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [], "R3": [6], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [], "R3": [7], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [8], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [9], "R4": [1]},
    ],
    "he3": [
        {"energy_min": 2.2, "energy_max": 2.7, "R2": [10], "R3": [], "R4": []},
        {"energy_min": 2.7, "energy_max": 3.2, "R2": [11], "R3": [], "R4": []},
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [12], "R3": [], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [13], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [14], "R3": [12], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [15], "R3": [13], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [16], "R3": [14], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [], "R3": [15], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [], "R3": [16], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [17], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [18], "R4": []},
    ],
    "he4": [
        {"energy_min": 1.8, "energy_max": 2.2, "R2": [19], "R3": [], "R4": []},
        {"energy_min": 2.2, "energy_max": 2.7, "R2": [20], "R3": [], "R4": []},
        {"energy_min": 2.7, "energy_max": 3.2, "R2": [21], "R3": [], "R4": []},
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [22], "R3": [21], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [23], "R3": [22], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [24], "R3": [23], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [25], "R3": [24], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [], "R3": [25], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [], "R3": [26], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [], "R3": [27], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [28], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [29], "R4": [4]},
    ],
    "he": [
        {"energy_min": 2.2, "energy_max": 2.7, "R2": [10, 20], "R3": [], "R4": []},
        {"energy_min": 2.7, "energy_max": 3.2, "R2": [11, 21], "R3": [], "R4": []},
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [22], "R3": [21], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [23], "R3": [22], "R4": []},
        {
            "energy_min": 4.0,
            "energy_max": 4.5,
            "R2": [14, 24],
            "R3": [12, 23],
            "R4": [],
        },
        {
            "energy_min": 4.5,
            "energy_max": 5.0,
            "R2": [15, 25],
            "R3": [13, 24],
            "R4": [],
        },
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [], "R3": [14, 25], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [], "R3": [15, 26], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [], "R3": [16, 27], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [17, 28], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [29], "R4": [4]},
    ],
    "c": [
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [28], "R3": [], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [29], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [30], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [31], "R3": [32], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [32], "R3": [33], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [33], "R3": [34], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [34], "R3": [35], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [36], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [37], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [38], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [39], "R4": [7]},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [40], "R4": [8]},
    ],
    "n": [
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [37], "R3": [], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [38], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [39], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [40], "R3": [], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [41], "R3": [43], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [42], "R3": [44], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [43], "R3": [45], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [46], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [47], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [48], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [49], "R4": [11]},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [50], "R4": [12]},
    ],
    "o": [
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [46], "R3": [], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [47], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [48], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [49], "R3": [], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [50], "R3": [53], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [51], "R3": [54], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [52], "R3": [55], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [56], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [57], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [58], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [59], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [60], "R4": [15]},
    ],
    "ne": [
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [55], "R3": [], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [56], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [57], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [58], "R3": [], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [59], "R3": [], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [60], "R3": [63], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [61], "R3": [64], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [62], "R3": [65], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [66], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [67], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [68], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [69], "R4": [18]},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [70], "R4": [19]},
    ],
    "na": [
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [], "R3": [73], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [], "R3": [74], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [75], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [76], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [77], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [78], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [79], "R4": []},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [80], "R4": []},
    ],
    "mg": [
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [65], "R3": [], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [66], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [67], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [68], "R3": [], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [69], "R3": [], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [70], "R3": [83], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [71], "R3": [84], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [72], "R3": [85], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [73], "R3": [86], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [87], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [88], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [89], "R4": []},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [90], "R4": [22]},
        {"energy_min": 40.0, "energy_max": 52.0, "R2": [], "R3": [], "R4": [23]},
    ],
    "al": [
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [], "R3": [94], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [], "R3": [95], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [96], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [97], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [98], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [99], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [100], "R4": []},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [101], "R4": []},
        {"energy_min": 40.0, "energy_max": 52.0, "R2": [], "R3": [102], "R4": []},
    ],
    "si": [
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [76], "R3": [], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [77], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [78], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [79], "R3": [], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [80], "R3": [], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [81], "R3": [105], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [82], "R3": [106], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [83], "R3": [107], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [84], "R3": [108], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [109], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [110], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [111], "R4": []},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [112], "R4": [26]},
        {"energy_min": 40.0, "energy_max": 52.0, "R2": [], "R3": [113], "R4": [27]},
    ],
    "s": [
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [87], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [88], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [89], "R3": [], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [90], "R3": [], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [91], "R3": [116], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [92], "R3": [117], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [93], "R3": [118], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [94], "R3": [119], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [120], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [121], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [122], "R4": []},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [123], "R4": []},
        {"energy_min": 40.0, "energy_max": 52.0, "R2": [], "R3": [124], "R4": []},
    ],
    "ar": [
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [97], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [98], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [99], "R3": [], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [100], "R3": [], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [101], "R3": [127], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [102], "R3": [128], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [103], "R3": [129], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [104], "R3": [130], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [105], "R3": [131], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [132], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [133], "R4": []},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [134], "R4": []},
        {"energy_min": 40.0, "energy_max": 52.0, "R2": [], "R3": [135], "R4": []},
    ],
    "ca": [
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [108], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [109], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [110], "R3": [], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [111], "R3": [], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [112], "R3": [], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [113], "R3": [138], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [114], "R3": [139], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [115], "R3": [140], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [116], "R3": [141], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [142], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [143], "R4": []},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [144], "R4": []},
        {"energy_min": 40.0, "energy_max": 52.0, "R2": [], "R3": [145], "R4": []},
    ],
    "fe": [
        {"energy_min": 2.7, "energy_max": 3.2, "R2": [119], "R3": [], "R4": []},
        {"energy_min": 3.2, "energy_max": 3.6, "R2": [120], "R3": [], "R4": []},
        {"energy_min": 3.6, "energy_max": 4.0, "R2": [121], "R3": [], "R4": []},
        {"energy_min": 4.0, "energy_max": 4.5, "R2": [122], "R3": [], "R4": []},
        {"energy_min": 4.5, "energy_max": 5.0, "R2": [123], "R3": [], "R4": []},
        {"energy_min": 5.0, "energy_max": 6.0, "R2": [124], "R3": [], "R4": []},
        {"energy_min": 6.0, "energy_max": 8.0, "R2": [125], "R3": [], "R4": []},
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [126], "R3": [148], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [127], "R3": [149], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [128], "R3": [150], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [129], "R3": [151], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [152], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [153], "R4": []},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [154], "R4": []},
        {"energy_min": 40.0, "energy_max": 52.0, "R2": [], "R3": [155], "R4": [30]},
        {"energy_min": 52.0, "energy_max": 70.0, "R2": [], "R3": [156], "R4": [31]},
    ],
    "ni": [
        {"energy_min": 8.0, "energy_max": 10.0, "R2": [], "R3": [158], "R4": []},
        {"energy_min": 10.0, "energy_max": 12.0, "R2": [], "R3": [159], "R4": []},
        {"energy_min": 12.0, "energy_max": 15.0, "R2": [], "R3": [160], "R4": []},
        {"energy_min": 15.0, "energy_max": 21.0, "R2": [], "R3": [161], "R4": []},
        {"energy_min": 21.0, "energy_max": 27.0, "R2": [], "R3": [162], "R4": []},
        {"energy_min": 27.0, "energy_max": 33.0, "R2": [], "R3": [163], "R4": []},
        {"energy_min": 33.0, "energy_max": 40.0, "R2": [], "R3": [164], "R4": []},
        {"energy_min": 40.0, "energy_max": 52.0, "R2": [], "R3": [165], "R4": []},
        {"energy_min": 52.0, "energy_max": 70.0, "R2": [], "R3": [166], "R4": []},
    ],
}
