"""
Contains constants variables to support CoDICE processing.

Notes
-----
The ``plan_id``, ``plan_step``, and ``view_id`` mentioned in this module are
derived from the packet data.

Some notable acronyms:
SW = SunWard
NSW = Non-SunWard
PUI = PickUp Ion
ESA = ElectroStatic Analyzer
"""

from typing import Any

import numpy as np

from imap_processing.codice.utils import CODICEAPID, CoDICECompression

# Grouping of APIDs used to signify similar L1a processing
APIDS_FOR_SCIENCE_PROCESSING = [
    CODICEAPID.COD_HI_INST_COUNTS_AGGREGATED,
    CODICEAPID.COD_HI_INST_COUNTS_PRIORITIES,
    CODICEAPID.COD_HI_INST_COUNTS_SINGLES,
    CODICEAPID.COD_HI_SECT_SPECIES_COUNTS,
    CODICEAPID.COD_LO_INST_COUNTS_AGGREGATED,
    CODICEAPID.COD_LO_INST_COUNTS_SINGLES,
    CODICEAPID.COD_LO_SW_ANGULAR_COUNTS,
    CODICEAPID.COD_LO_NSW_ANGULAR_COUNTS,
    CODICEAPID.COD_LO_SW_PRIORITY_COUNTS,
    CODICEAPID.COD_LO_NSW_PRIORITY_COUNTS,
    CODICEAPID.COD_LO_SW_SPECIES_COUNTS,
    CODICEAPID.COD_LO_NSW_SPECIES_COUNTS,
]

# Mapping between descriptors and APID
CODICEAPID_MAPPING = {
    "hskp": CODICEAPID.COD_NHK,
    "lo-ialirt": CODICEAPID.COD_LO_IAL,
    "lo-direct-events": CODICEAPID.COD_LO_PHA,
    "lo-sw-priority": CODICEAPID.COD_LO_SW_PRIORITY_COUNTS,
    "lo-sw-species": CODICEAPID.COD_LO_SW_SPECIES_COUNTS,
    "lo-nsw-species": CODICEAPID.COD_LO_NSW_SPECIES_COUNTS,
    "lo-sw-angular": CODICEAPID.COD_LO_SW_ANGULAR_COUNTS,
    "lo-nsw-angular": CODICEAPID.COD_LO_NSW_ANGULAR_COUNTS,
    "lo-nsw-priority": CODICEAPID.COD_LO_NSW_PRIORITY_COUNTS,
    "lo-counters-aggregated": CODICEAPID.COD_LO_INST_COUNTS_AGGREGATED,
    "lo-counters-singles": CODICEAPID.COD_LO_INST_COUNTS_SINGLES,
    "hi-ialirt": CODICEAPID.COD_HI_IAL,
    "hi-direct-events": CODICEAPID.COD_HI_PHA,
    "hi-counters-aggregated": CODICEAPID.COD_HI_INST_COUNTS_AGGREGATED,
    "hi-counters-singles": CODICEAPID.COD_HI_INST_COUNTS_SINGLES,
    "hi-omni": CODICEAPID.COD_HI_OMNI_SPECIES_COUNTS,
    "hi-sectored": CODICEAPID.COD_HI_SECT_SPECIES_COUNTS,
    "hi-priority": CODICEAPID.COD_HI_INST_COUNTS_PRIORITIES,
}

# Numerical constants
SPIN_PERIOD_CONVERSION = 0.00032
K_FACTOR = 5.76  # This is used to convert voltages to energies in L2
HI_ACQUISITION_TIME = 0.59916

# CDF variable names used for lo data products
LO_COUNTERS_SINGLES_VARIABLE_NAMES = ["apd_singles"]
LO_SW_ANGULAR_VARIABLE_NAMES = ["hplus", "heplusplus", "oplus6", "fe_loq"]
LO_NSW_ANGULAR_VARIABLE_NAMES = ["heplusplus"]
LO_SW_PRIORITY_VARIABLE_NAMES = [
    "p0_tcrs",
    "p1_hplus",
    "p2_heplusplus",
    "p3_heavies",
    "p4_dcrs",
]
LO_NSW_PRIORITY_VARIABLE_NAMES = ["p5_heavies", "p6_hplus_heplusplus"]
LO_SW_SPECIES_VARIABLE_NAMES = [
    "hplus",
    "heplusplus",
    "cplus4",
    "cplus5",
    "cplus6",
    "oplus5",
    "oplus6",
    "oplus7",
    "oplus8",
    "ne",
    "mg",
    "si",
    "fe_loq",
    "fe_hiq",
    "heplus",
    "cnoplus",
]
LO_NSW_SPECIES_VARIABLE_NAMES = [
    "hplus",
    "heplusplus",
    "c",
    "o",
    "ne_si_mg",
    "fe",
    "heplus",
    "cnoplus",
]
LO_IALIRT_VARIABLE_NAMES = [
    "heplusplus",
    "cplus5",
    "cplus6",
    "oplus6",
    "oplus7",
    "oplus8",
    "mg",
    "fe_loq",
    "fe_hiq",
]

# CDF variable names used for hi data products
HI_COUNTERS_SINGLES_VARIABLE_NAMES = ["tcr", "ssdo", "stssd"]
HI_OMNI_VARIABLE_NAMES = ["h", "he3", "he4", "c", "o", "ne_mg_si", "fe", "uh", "junk"]
HI_PRIORITY_VARIABLE_NAMES = [
    "Priority0",
    "Priority1",
    "Priority2",
    "Priority3",
    "Priority4",
    "Priority5",
]
HI_SECTORED_VARIABLE_NAMES = ["h", "he3he4", "cno", "fe"]
HI_IALIRT_VARIABLE_NAMES = ["h"]

# CDF variable names used for direct event data products
HI_DE_CDF_FIELDS = [
    "num_events",
    "data_quality",
    "ssd_energy",
    "tof",
    "ssd_id",
    "gain",
    "multi_flag",
    "type",
    "spin_sector",
    "spin_number",
]
HI_DIRECT_EVENTS_VARIABLE_NAMES = [
    f"p{n}_{field}" for n in range(6) for field in HI_DE_CDF_FIELDS
]
LO_DE_CDF_FIELDS = [
    "num_events",
    "data_quality",
    "gain",
    "apd_id",
    "apd_energy",
    "tof",
    "multi_flag",
    "type",
    "spin_sector",
    "energy_step",
]
LO_DIRECT_EVENTS_VARIABLE_NAMES = [
    f"p{n}_{field}" for n in range(8) for field in LO_DE_CDF_FIELDS
]

# Final I-ALiRT data product fields
CODICE_LO_IAL_DATA_FIELDS = [
    "c_over_o_abundance",
    "mg_over_o_abundance",
    "fe_over_o_abundance",
    "c_plus_6_over_c_plus_5_ratio",
    "o_plus_7_over_o_plus_6_ratio",
    "fe_low_over_fe_high_ratio",
]
CODICE_HI_IAL_DATA_FIELDS = ["h"]

# lo- and hi-counters-aggregated data product variables are dynamically
# determined based on the number of active counters
LO_COUNTERS_AGGREGATED_ACTIVE_VARIABLES = {
    "tcr": True,
    "dcr": True,
    "tof_plus_apd": False,
    "tof_only": False,
    "position_plus_apd": False,
    "position_only": False,
    "sta_or_stb_plus_apd": False,
    "sta_or_stb_only": False,
    "reserved1": False,
    "reserved2": False,
    "sp_only": False,
    "apd_only": False,
    "low_tof_cutoff": False,
    "sta": True,
    "stb": True,
    "sp": True,
    "total_position_count": True,
    "invalid_position_count": False,
    "asic1_flag_invalid": False,
    "asic2_flag_invalid": False,
    "asic1_channel_invalid": False,
    "asic2_channel_invalid": False,
    "tec4_timeout_tof_no_pos": False,
    "tec4_timeout_pos_no_tof": False,
    "tec4_timeout_no_pos_tof": False,
    "tec5_timeout_tof_no_pos": False,
    "tec5_timeout_pos_no_tof": False,
    "tec5_timeout_no_pos_tof": False,
}
LO_COUNTERS_AGGREGATED_VARIABLE_NAMES = [
    name
    for name, is_active in LO_COUNTERS_AGGREGATED_ACTIVE_VARIABLES.items()
    if is_active
]
HI_COUNTERS_AGGREGATED_ACTIVE_VARIABLES = {
    "dcr": True,
    "sto": True,
    "spo": True,
    "reserved1": False,
    "mst": True,
    "ssdo": True,
    "stssd": True,
    "reserved4": False,
    "reserved5": False,
    "low_tof_cutoff": True,
    "reserved6": False,
    "reserved7": False,
    "asic1_flag_invalid": True,
    "asic2_flag_invalid": True,
    "asic1_channel_invalid": True,
    "asic2_channel_invalid": True,
}
HI_COUNTERS_AGGREGATED_VARIABLE_NAMES = [
    name
    for name, is_active in HI_COUNTERS_AGGREGATED_ACTIVE_VARIABLES.items()
    if is_active
]

# List of data products that require application of despinning algorithm
REQUIRES_DESPINNING = [
    "imap_codice_l1a_lo-sw-angular",
    "imap_codice_l1a_lo-nsw-angular",
    # TBD if this requires despinning
    # "imap_codice_l1a_lo-sw-priority",
    # "imap_codice_l1a_lo-nsw-priority",
]

# Energy tables for CoDICE-Hi data products. These values represent the edges
# of the bins, and are used in the CoDICE L1a pipeline to compute the centers
# and deltas of the bins, which then get stored in the CDF files for future use.
# These are defined in the "Data Products - Hi" tab of the "*-SCI-LUT-*.xml"
# spreadsheet that largely defines CoDICE processing.
IALIRT_ENERGY_TABLE = {
    "h": [
        0.05,
        0.070710678,
        0.1,
        0.141421356,
        0.2,
        0.282842712,
        0.4,
        0.565685425,
        0.8,
        1.13137085,
        1.6,
        2.2627417,
        3.2,
        4.5254834,
        6.4,
        9.050966799,
    ],
}

OMNI_ENERGY_TABLE = {
    "h": [
        0.0200000000,
        0.0282842712,
        0.0400000000,
        0.0565685425,
        0.0800000000,
        0.1131370850,
        0.1600000000,
        0.2262741700,
        0.3200000000,
        0.4525483400,
        0.6400000000,
        0.9050966799,
        1.2800000000,
        1.8101933598,
        2.5600000000,
        3.6203867197,
    ],
    "he3": [
        0.0200000000,
        0.0282842712,
        0.0400000000,
        0.0565685425,
        0.0800000000,
        0.1131370850,
        0.1600000000,
        0.2262741700,
        0.3200000000,
        0.4525483400,
        0.6400000000,
        0.9050966799,
        1.2800000000,
        1.8101933598,
        2.5600000000,
        3.6203867197,
    ],
    "he4": [
        0.0200000000,
        0.0282842712,
        0.0400000000,
        0.0565685425,
        0.0800000000,
        0.1131370850,
        0.1600000000,
        0.2262741700,
        0.3200000000,
        0.4525483400,
        0.6400000000,
        0.9050966799,
        1.2800000000,
        1.8101933598,
        2.5600000000,
        3.6203867197,
    ],
    "c": [
        0.0200000000,
        0.0282842712,
        0.0400000000,
        0.0565685425,
        0.0800000000,
        0.1131370850,
        0.1600000000,
        0.2262741700,
        0.3200000000,
        0.4525483400,
        0.6400000000,
        0.9050966799,
        1.2800000000,
        1.8101933598,
        2.5600000000,
        3.6203867197,
        5.1200000000,
        7.2407734394,
        10.2400000000,
    ],
    "o": [
        0.0200000000,
        0.0282842712,
        0.0400000000,
        0.0565685425,
        0.0800000000,
        0.1131370850,
        0.1600000000,
        0.2262741700,
        0.3200000000,
        0.4525483400,
        0.6400000000,
        0.9050966799,
        1.2800000000,
        1.8101933598,
        2.5600000000,
        3.6203867197,
        5.1200000000,
        7.2407734394,
        10.2400000000,
    ],
    "ne_mg_si": [
        0.0200000000,
        0.0282842712,
        0.0400000000,
        0.0565685425,
        0.0800000000,
        0.1131370850,
        0.1600000000,
        0.2262741700,
        0.3200000000,
        0.4525483400,
        0.6400000000,
        0.9050966799,
        1.2800000000,
        1.8101933598,
        2.5600000000,
        3.6203867197,
    ],
    "fe": [
        0.0200000000,
        0.0282842712,
        0.0400000000,
        0.0565685425,
        0.0800000000,
        0.1131370850,
        0.1600000000,
        0.2262741700,
        0.3200000000,
        0.4525483400,
        0.6400000000,
        0.9050966799,
        1.2800000000,
        1.8101933598,
        2.5600000000,
        3.6203867197,
        5.1200000000,
        7.2407734394,
        10.2400000000,
    ],
    "uh": [
        0.0200000000,
        0.0282842712,
        0.0400000000,
        0.0565685425,
        0.0800000000,
        0.1131370850,
    ],
    "junk": [0.0200000000, 0.0282842712],
}

# In the future, we get csv file with these column:
# species, min_energy, max_energy, product (descriptor of the product)
SECTORED_ENERGY_TABLE = {
    "h": [0.020, 0.040, 0.080, 0.160, 0.320, 0.640, 1.280, 2.560, 5.120],
    "he3he4": [0.020, 0.040, 0.080, 0.160, 0.320, 0.640, 1.280, 2.560, 5.120],
    "cno": [0.020, 0.040, 0.080, 0.160, 0.320, 0.640, 1.280, 2.560, 5.120],
    "fe": [0.020, 0.040, 0.080, 0.160, 0.320, 0.640, 1.280, 2.560, 5.120],
}

# Various configurations to support processing of individual data products
# Much of these are described in the algorithm document in chapter 10 ("Data
# Level 1A")
DATA_PRODUCT_CONFIGURATIONS: dict[CODICEAPID | int, dict] = {
    CODICEAPID.COD_HI_IAL: {
        "dataset_name": "imap_codice_l1a_hi-ialirt",
        "energy_table": IALIRT_ENERGY_TABLE,
        "dims": {"esa_step": 15, "inst_az": 4},
        "instrument": "hi",
        "num_counters": 1,
        "num_spins": 4,
        "support_variables": [
            "data_quality",
            "spin_period",
            "energy_h",
        ],
        "variable_names": HI_IALIRT_VARIABLE_NAMES,
    },
    CODICEAPID.COD_HI_INST_COUNTS_AGGREGATED: {
        "dataset_name": "imap_codice_l1a_hi-counters-aggregated",
        "dims": {},
        "instrument": "hi",
        "num_counters": len(
            HI_COUNTERS_AGGREGATED_VARIABLE_NAMES
        ),  # The number of counters depends on the number of *active* counters
        "support_variables": ["data_quality", "spin_period"],
        "variable_names": HI_COUNTERS_AGGREGATED_VARIABLE_NAMES,
    },
    CODICEAPID.COD_HI_INST_COUNTS_SINGLES: {
        "dataset_name": "imap_codice_l1a_hi-counters-singles",
        "dims": {
            "ssd_index": 12,
        },
        "instrument": "hi",
        "num_counters": 3,
        "support_variables": ["data_quality", "spin_period"],
        "variable_names": HI_COUNTERS_SINGLES_VARIABLE_NAMES,
    },
    CODICEAPID.COD_HI_INST_COUNTS_PRIORITIES: {
        "dataset_name": "imap_codice_l1a_hi-priority",
        "dims": {},
        "instrument": "hi",
        "num_counters": 6,
        "support_variables": ["data_quality", "spin_period"],
        "variable_names": HI_PRIORITY_VARIABLE_NAMES,
    },
    CODICEAPID.COD_HI_OMNI_SPECIES_COUNTS: {
        "dataset_name": "imap_codice_l1a_hi-omni",
        "energy_table": OMNI_ENERGY_TABLE,
        "dims": {"esa_step": 15, "inst_az": 4},
        "instrument": "hi",
        "num_counters": 8,
        "num_spins": 4,
        "support_variables": [
            "data_quality",
            "spin_period",
            "energy_h",
            "energy_he3",
            "energy_he4",
            "energy_c",
            "energy_o",
            "energy_ne_mg_si",
            "energy_fe",
            "energy_uh",
            "energy_junk",
        ],
        "variable_names": HI_OMNI_VARIABLE_NAMES,
    },
    CODICEAPID.COD_HI_SECT_SPECIES_COUNTS: {
        "dataset_name": "imap_codice_l1a_hi-sectored",
        "energy_table": SECTORED_ENERGY_TABLE,
        "dims": {
            "esa_step": 8,
            "ssd_index": 12,
            "spin_sector_index": 12,
        },
        "instrument": "hi",
        "num_counters": 4,
        "num_spins": 16,
        "support_variables": [
            "data_quality",
            "spin_period",
            "energy_h",
            "energy_he3he4",
            "energy_cno",
            "energy_fe",
        ],
        "variable_names": HI_SECTORED_VARIABLE_NAMES,
    },
    CODICEAPID.COD_LO_IAL: {
        "dataset_name": "imap_codice_l1a_lo-ialirt",
        "dims": {"esa_step": 128, "spin_sector": 1},
        "instrument": "lo",
        "num_counters": 9,
        "support_variables": [
            "energy_table",
            "acquisition_time_per_step",
            "rgfo_half_spin",
            "nso_half_spin",
            "sw_bias_gain_mode",
            "st_bias_gain_mode",
            "data_quality",
            "spin_period",
            "k_factor",
        ],
        "variable_names": LO_IALIRT_VARIABLE_NAMES,
    },
    CODICEAPID.COD_LO_INST_COUNTS_AGGREGATED: {
        "dataset_name": "imap_codice_l1a_lo-counters-aggregated",
        "dims": {"esa_step": 128, "spin_sector_pairs": 6},
        "instrument": "lo",
        "num_counters": len(
            LO_COUNTERS_AGGREGATED_VARIABLE_NAMES
        ),  # The number of counters depends on the number of *active* counters
        "support_variables": [
            "energy_table",
            "acquisition_time_per_step",
            "rgfo_half_spin",
            "nso_half_spin",
            "sw_bias_gain_mode",
            "st_bias_gain_mode",
            "data_quality",
            "spin_period",
            "k_factor",
        ],
        "variable_names": LO_COUNTERS_AGGREGATED_VARIABLE_NAMES,
    },
    CODICEAPID.COD_LO_INST_COUNTS_SINGLES: {
        "dataset_name": "imap_codice_l1a_lo-counters-singles",
        "dims": {"esa_step": 128, "inst_az": 24, "spin_sector_pairs": 6},
        "instrument": "lo",
        "num_counters": 1,
        "support_variables": [
            "energy_table",
            "acquisition_time_per_step",
            "rgfo_half_spin",
            "nso_half_spin",
            "sw_bias_gain_mode",
            "st_bias_gain_mode",
            "data_quality",
            "spin_period",
            "k_factor",
        ],
        "variable_names": LO_COUNTERS_SINGLES_VARIABLE_NAMES,
    },
    CODICEAPID.COD_LO_SW_ANGULAR_COUNTS: {
        "dataset_name": "imap_codice_l1a_lo-sw-angular",
        "dims": {"esa_step": 128, "inst_az": 5, "spin_sector": 12},
        "instrument": "lo",
        "num_counters": 4,
        "support_variables": [
            "energy_table",
            "acquisition_time_per_step",
            "rgfo_half_spin",
            "nso_half_spin",
            "sw_bias_gain_mode",
            "st_bias_gain_mode",
            "data_quality",
            "spin_period",
            "k_factor",
        ],
        "variable_names": LO_SW_ANGULAR_VARIABLE_NAMES,
    },
    CODICEAPID.COD_LO_NSW_ANGULAR_COUNTS: {
        "dataset_name": "imap_codice_l1a_lo-nsw-angular",
        "dims": {"esa_step": 128, "inst_az": 19, "spin_sector": 12},
        "instrument": "lo",
        "num_counters": 1,
        "support_variables": [
            "energy_table",
            "acquisition_time_per_step",
            "rgfo_half_spin",
            "nso_half_spin",
            "sw_bias_gain_mode",
            "st_bias_gain_mode",
            "data_quality",
            "spin_period",
            "k_factor",
        ],
        "variable_names": LO_NSW_ANGULAR_VARIABLE_NAMES,
    },
    CODICEAPID.COD_LO_SW_PRIORITY_COUNTS: {
        "dataset_name": "imap_codice_l1a_lo-sw-priority",
        "dims": {"esa_step": 128, "spin_sector": 12},
        "instrument": "lo",
        "num_counters": 5,
        "support_variables": [
            "energy_table",
            "acquisition_time_per_step",
            "rgfo_half_spin",
            "nso_half_spin",
            "sw_bias_gain_mode",
            "st_bias_gain_mode",
            "data_quality",
            "spin_period",
            "k_factor",
        ],
        "variable_names": LO_SW_PRIORITY_VARIABLE_NAMES,
    },
    CODICEAPID.COD_LO_NSW_PRIORITY_COUNTS: {
        "dataset_name": "imap_codice_l1a_lo-nsw-priority",
        "dims": {"esa_step": 128, "spin_sector": 12},
        "instrument": "lo",
        "num_counters": 2,
        "support_variables": [
            "energy_table",
            "acquisition_time_per_step",
            "rgfo_half_spin",
            "nso_half_spin",
            "sw_bias_gain_mode",
            "st_bias_gain_mode",
            "data_quality",
            "spin_period",
            "k_factor",
        ],
        "variable_names": LO_NSW_PRIORITY_VARIABLE_NAMES,
    },
    CODICEAPID.COD_LO_SW_SPECIES_COUNTS: {
        "dataset_name": "imap_codice_l1a_lo-sw-species",
        "dims": {"esa_step": 128, "spin_sector": 1},
        "instrument": "lo",
        "num_counters": 16,
        "support_variables": [
            "energy_table",
            "acquisition_time_per_step",
            "rgfo_half_spin",
            "nso_half_spin",
            "sw_bias_gain_mode",
            "st_bias_gain_mode",
            "data_quality",
            "spin_period",
            "k_factor",
        ],
        "variable_names": LO_SW_SPECIES_VARIABLE_NAMES,
    },
    CODICEAPID.COD_LO_NSW_SPECIES_COUNTS: {
        "dataset_name": "imap_codice_l1a_lo-nsw-species",
        "dims": {"esa_step": 128, "spin_sector": 1},
        "instrument": "lo",
        "num_counters": 8,
        "support_variables": [
            "energy_table",
            "acquisition_time_per_step",
            "rgfo_half_spin",
            "nso_half_spin",
            "sw_bias_gain_mode",
            "st_bias_gain_mode",
            "data_quality",
            "spin_period",
            "k_factor",
        ],
        "variable_names": LO_NSW_SPECIES_VARIABLE_NAMES,
    },
}

# Various configurations to support L1b processing of individual data products
# Much of these are described in the algorithm document in chapter 11 ("Data
# Level 1B")
L1B_DATA_PRODUCT_CONFIGURATIONS: dict[str, dict] = {
    "hi-counters-aggregated": {
        "num_spin_sectors": 24,
        "num_spins": 16,
    },
    "hi-counters-singles": {
        "num_spin_sectors": 24,
        "num_spins": 16,
    },
    "hi-ialirt": {
        "num_spin_sectors": 24,
        "num_spins": 4,
    },
    "hi-omni": {
        "num_spin_sectors": 24,
        "num_spins": 4,
    },
    "hi-priority": {
        "num_spin_sectors": 24,
        "num_spins": 16,
    },
    "hi-sectored": {
        "num_spin_sectors": 2,
        "num_spins": 16,
    },
    "lo-counters-aggregated": {
        "num_spin_sectors": 2,
    },
    "lo-counters-singles": {
        "num_spin_sectors": 2,
    },
    "lo-nsw-angular": {
        "num_spin_sectors": 1,
    },
    "lo-sw-angular": {
        "num_spin_sectors": 1,
    },
    "lo-nsw-priority": {
        "num_spin_sectors": 1,
    },
    "lo-sw-priority": {
        "num_spin_sectors": 1,
    },
    "lo-nsw-species": {
        "num_spin_sectors": 12,
    },
    "lo-sw-species": {
        "num_spin_sectors": 12,
    },
    "lo-ialirt": {
        "num_spin_sectors": 12,
    },
}

# Various configurations to support processing of direct events data products
# These are described in the algorithm document in chapter 10 ("Data Level 1A")
DE_DATA_PRODUCT_CONFIGURATIONS: dict[Any, dict[str, Any]] = {
    CODICEAPID.COD_HI_PHA: {
        "num_priorities": 6,
        "bit_structure": {
            "ssd_energy": {
                "bit_length": 11,
                "dtype": np.uint16,
                "fillval": np.iinfo(np.uint16).max,
            },
            "tof": {
                "bit_length": 10,
                "dtype": np.uint16,
                "fillval": np.iinfo(np.uint16).max,
            },
            "ssd_id": {
                "bit_length": 4,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "gain": {
                "bit_length": 2,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "multi_flag": {
                "bit_length": 1,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "type": {
                "bit_length": 2,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "spin_sector": {
                "bit_length": 5,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "spin_number": {
                "bit_length": 4,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "Priority": {
                "bit_length": 3,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "Spare": {
                "bit_length": 22,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
        },
        "cdf_fields": HI_DE_CDF_FIELDS,
    },
    CODICEAPID.COD_LO_PHA: {
        "num_priorities": 8,
        "bit_structure": {
            "gain": {
                "bit_length": 1,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "apd_id": {
                "bit_length": 5,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "position": {
                "bit_length": 5,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "apd_energy": {
                "bit_length": 9,
                "dtype": np.uint16,
                "fillval": np.iinfo(np.uint16).max,
            },
            "tof": {
                "bit_length": 10,
                "dtype": np.uint16,
                "fillval": np.iinfo(np.uint16).max,
            },
            "multi_flag": {
                "bit_length": 1,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "type": {
                "bit_length": 2,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "spin_sector": {
                "bit_length": 5,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "energy_step": {
                "bit_length": 7,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "Priority": {
                "bit_length": 3,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
            "Spare": {
                "bit_length": 16,
                "dtype": np.uint8,
                "fillval": np.iinfo(np.uint8).max,
            },
        },
        "cdf_fields": LO_DE_CDF_FIELDS,
    },
}

# Define the packet fields needed to be stored in segmented data and their
# corresponding bit lengths for I-ALiRT data products
IAL_BIT_STRUCTURE = {
    "SHCOARSE": 32,
    "PACKET_VERSION": 16,
    "SPIN_PERIOD": 16,
    "ACQ_START_SECONDS": 32,
    "ACQ_START_SUBSECONDS": 20,
    "SPARE_00": 8,
    "ST_BIAS_GAIN_MODE": 2,
    "SW_BIAS_GAIN_MODE": 2,
    "TABLE_ID": 32,
    "PLAN_ID": 16,
    "PLAN_STEP": 4,
    "VIEW_ID": 4,
    "RGFO_HALF_SPIN": 6,
    "NSO_HALF_SPIN": 6,
    "SPARE_01": 1,
    "SUSPECT": 1,
    "COMPRESSION": 3,
    "BYTE_COUNT": 23,
}

# Define the packet fields needed to be stored in segmented data and their
# corresponding bit lengths for direct event data products
DE_METADATA_FIELDS = {
    "packet_version": 16,
    "spin_period": 16,
    "acq_start_seconds": 32,
    "acq_start_subseconds": 20,
    "spare_1": 2,
    "st_bias_gain_mode": 2,
    "sw_bias_gain_mode": 2,
    "priority": 4,
    "suspect": 1,
    "compressed": 1,
    "num_events": 32,
    "byte_count": 32,
}

# Compression ID lookup tables
# The key is the view_id and the value is the ID for the compression algorithm
# (see utils.CoDICECompression to see how the values correspond)
# These are defined in the "Views" tab of the "*-SCI-LUT-*.xml" spreadsheet that
# largely defines CoDICE processing.
LO_COMPRESSION_ID_LOOKUP = {
    0: CoDICECompression.PACK_24_BIT,
    1: CoDICECompression.LOSSY_B_LOSSLESS,
    2: CoDICECompression.LOSSY_B_LOSSLESS,
    3: CoDICECompression.LOSSY_A_LOSSLESS,
    4: CoDICECompression.LOSSY_A_LOSSLESS,
    5: CoDICECompression.LOSSY_A_LOSSLESS,
    6: CoDICECompression.LOSSY_A_LOSSLESS,
    7: CoDICECompression.LOSSY_A_LOSSLESS,
    8: CoDICECompression.LOSSY_A_LOSSLESS,
}
HI_COMPRESSION_ID_LOOKUP = {
    0: CoDICECompression.LOSSY_A,
    1: CoDICECompression.LOSSY_A,
    2: CoDICECompression.LOSSY_A,
    3: CoDICECompression.LOSSY_B_LOSSLESS,
    4: CoDICECompression.LOSSY_B_LOSSLESS,
    5: CoDICECompression.LOSSY_A_LOSSLESS,
    6: CoDICECompression.LOSSY_A_LOSSLESS,
    7: CoDICECompression.LOSSY_A_LOSSLESS,
    8: CoDICECompression.LOSSY_A_LOSSLESS,
    9: CoDICECompression.LOSSY_A_LOSSLESS,
}

# ESA Sweep table ID lookup table
# The combination of plan_id and plan_step determine the ESA sweep Table to use
# Currently, ESA sweep table 0 is used for every plan_id/plan_step combination,
# but may change in the future. These are defined in the "ESA Sweep" tab of the
# "*-SCI-LUT-*.xml" spreadsheet that largely defines CoDICE processing.
ESA_SWEEP_TABLE_ID_LOOKUP = {
    (0, 0): 0,
    (0, 1): 0,
    (0, 2): 0,
    (0, 3): 0,
    (1, 0): 0,
    (1, 1): 0,
    (1, 2): 0,
    (1, 3): 0,
    (2, 0): 0,
    (2, 1): 0,
    (2, 2): 0,
    (2, 3): 0,
    (3, 0): 0,
    (3, 1): 0,
    (3, 2): 0,
    (3, 3): 0,
    (4, 0): 0,
    (4, 1): 0,
    (4, 2): 0,
    (4, 3): 0,
    (5, 0): 0,
    (5, 1): 0,
    (5, 2): 0,
    (5, 3): 0,
    (6, 0): 0,
    (6, 1): 0,
    (6, 2): 0,
    (6, 3): 0,
    (7, 0): 0,
    (7, 1): 0,
    (7, 2): 0,
    (7, 3): 0,
}

# Lo Stepping table ID lookup table
# The combination of plan_id and plan_step determine the Lo Stepping Table to
# use. Currently, LO Stepping table 0 is used for every plan_id/plan_step
# combination, but may change in the future. These are defined in the "Lo
# Stepping" tab of the "*-SCI-LUT-*.xml" spreadsheet that largely defines CoDICE
# processing. Eg.
#   (plan_id, plan_step) -> id of acquisition time
#   (0, 0) -> 0


LO_STEPPING_TABLE_ID_LOOKUP = {
    (0, 0): 0,
    (0, 1): 0,
    (0, 2): 0,
    (0, 3): 0,
    (1, 0): 0,
    (1, 1): 0,
    (1, 2): 0,
    (1, 3): 0,
    (2, 0): 0,
    (2, 1): 0,
    (2, 2): 0,
    (2, 3): 0,
    (3, 0): 0,
    (3, 1): 0,
    (3, 2): 0,
    (3, 3): 0,
    (4, 0): 0,
    (4, 1): 0,
    (4, 2): 0,
    (4, 3): 0,
    (5, 0): 0,
    (5, 1): 0,
    (5, 2): 0,
    (5, 3): 0,
    (6, 0): 0,
    (6, 1): 0,
    (6, 2): 0,
    (6, 3): 0,
    (7, 0): 0,
    (7, 1): 0,
    (7, 2): 0,
    (7, 3): 0,
}

# Lookup tables for Lossy decompression algorithms "A" and "B"
# These were provided by Greg Dunn via his sohis_cdh_utils.v script and then
# transformed into python dictionaries. The values in these tables are subject
# to change, but the format is expected to stay the same.
LOSSY_A_TABLE = {
    0: 0,
    1: 1,
    2: 2,
    3: 3,
    4: 4,
    5: 5,
    6: 6,
    7: 7,
    8: 8,
    9: 9,
    10: 10,
    11: 11,
    12: 12,
    13: 13,
    14: 14,
    15: 15,
    16: 16,
    17: 17,
    18: 18,
    19: 19,
    20: 20,
    21: 21,
    22: 22,
    23: 23,
    24: 24,
    25: 25,
    26: 26,
    27: 27,
    28: 28,
    29: 29,
    30: 30,
    31: 31,
    32: 32,
    33: 34,
    34: 36,
    35: 38,
    36: 40,
    37: 42,
    38: 44,
    39: 46,
    40: 48,
    41: 50,
    42: 52,
    43: 54,
    44: 56,
    45: 58,
    46: 60,
    47: 62,
    48: 65,
    49: 69,
    50: 73,
    51: 77,
    52: 81,
    53: 85,
    54: 89,
    55: 93,
    56: 97,
    57: 101,
    58: 105,
    59: 109,
    60: 113,
    61: 117,
    62: 121,
    63: 125,
    64: 131,
    65: 139,
    66: 147,
    67: 155,
    68: 163,
    69: 171,
    70: 179,
    71: 187,
    72: 195,
    73: 203,
    74: 211,
    75: 219,
    76: 227,
    77: 235,
    78: 243,
    79: 251,
    80: 263,
    81: 279,
    82: 295,
    83: 311,
    84: 327,
    85: 343,
    86: 359,
    87: 375,
    88: 391,
    89: 407,
    90: 423,
    91: 439,
    92: 455,
    93: 471,
    94: 487,
    95: 503,
    96: 527,
    97: 559,
    98: 591,
    99: 623,
    100: 655,
    101: 687,
    102: 719,
    103: 751,
    104: 783,
    105: 815,
    106: 847,
    107: 879,
    108: 911,
    109: 943,
    110: 975,
    111: 1007,
    112: 1055,
    113: 1119,
    114: 1183,
    115: 1247,
    116: 1311,
    117: 1375,
    118: 1439,
    119: 1503,
    120: 1567,
    121: 1631,
    122: 1695,
    123: 1759,
    124: 1823,
    125: 1887,
    126: 1951,
    127: 2015,
    128: 2111,
    129: 2239,
    130: 2367,
    131: 2495,
    132: 2623,
    133: 2751,
    134: 2879,
    135: 3007,
    136: 3135,
    137: 3263,
    138: 3391,
    139: 3519,
    140: 3647,
    141: 3775,
    142: 3903,
    143: 4031,
    144: 4223,
    145: 4479,
    146: 4735,
    147: 4991,
    148: 5247,
    149: 5503,
    150: 5759,
    151: 6015,
    152: 6271,
    153: 6527,
    154: 6783,
    155: 7039,
    156: 7295,
    157: 7551,
    158: 7807,
    159: 8063,
    160: 8447,
    161: 8959,
    162: 9471,
    163: 9983,
    164: 10495,
    165: 11007,
    166: 11519,
    167: 12031,
    168: 12543,
    169: 13055,
    170: 13567,
    171: 14079,
    172: 14591,
    173: 15103,
    174: 15615,
    175: 16127,
    176: 16895,
    177: 17919,
    178: 18943,
    179: 19967,
    180: 20991,
    181: 22015,
    182: 23039,
    183: 24063,
    184: 25087,
    185: 26111,
    186: 27135,
    187: 28159,
    188: 29183,
    189: 30207,
    190: 31231,
    191: 32255,
    192: 33791,
    193: 35839,
    194: 37887,
    195: 39935,
    196: 41983,
    197: 44031,
    198: 46079,
    199: 48127,
    200: 50175,
    201: 52223,
    202: 54271,
    203: 56319,
    204: 58367,
    205: 60415,
    206: 62463,
    207: 64511,
    208: 67583,
    209: 71679,
    210: 75775,
    211: 79871,
    212: 83967,
    213: 88063,
    214: 92159,
    215: 96255,
    216: 100351,
    217: 104447,
    218: 108543,
    219: 112639,
    220: 116735,
    221: 120831,
    222: 124927,
    223: 129023,
    224: 135167,
    225: 143359,
    226: 151551,
    227: 159743,
    228: 167935,
    229: 176127,
    230: 184319,
    231: 192511,
    232: 200703,
    233: 208895,
    234: 217087,
    235: 225279,
    236: 233471,
    237: 241663,
    238: 249855,
    239: 258047,
    240: 270335,
    241: 286719,
    242: 303103,
    243: 319487,
    244: 335871,
    245: 352255,
    246: 368639,
    247: 385023,
    248: 401407,
    249: 417791,
    250: 434175,
    251: 450559,
    252: 466943,
    253: 483327,
    254: 499711,
    255: 2147737599,
}

LOSSY_B_TABLE = {
    0: 0,
    1: 1,
    2: 2,
    3: 3,
    4: 4,
    5: 5,
    6: 6,
    7: 7,
    8: 8,
    9: 9,
    10: 10,
    11: 11,
    12: 12,
    13: 13,
    14: 14,
    15: 15,
    16: 16,
    17: 17,
    18: 18,
    19: 19,
    20: 20,
    21: 21,
    22: 22,
    23: 23,
    24: 24,
    25: 25,
    26: 26,
    27: 27,
    28: 28,
    29: 29,
    30: 30,
    31: 31,
    32: 32,
    33: 34,
    34: 36,
    35: 38,
    36: 40,
    37: 42,
    38: 44,
    39: 46,
    40: 48,
    41: 50,
    42: 52,
    43: 54,
    44: 56,
    45: 58,
    46: 60,
    47: 62,
    48: 65,
    49: 69,
    50: 73,
    51: 77,
    52: 81,
    53: 85,
    54: 89,
    55: 93,
    56: 97,
    57: 101,
    58: 105,
    59: 109,
    60: 113,
    61: 117,
    62: 121,
    63: 125,
    64: 131,
    65: 139,
    66: 147,
    67: 155,
    68: 163,
    69: 171,
    70: 179,
    71: 187,
    72: 195,
    73: 203,
    74: 211,
    75: 219,
    76: 227,
    77: 235,
    78: 243,
    79: 251,
    80: 263,
    81: 279,
    82: 295,
    83: 311,
    84: 327,
    85: 343,
    86: 359,
    87: 375,
    88: 391,
    89: 407,
    90: 423,
    91: 439,
    92: 455,
    93: 471,
    94: 487,
    95: 503,
    96: 527,
    97: 559,
    98: 591,
    99: 623,
    100: 655,
    101: 687,
    102: 719,
    103: 751,
    104: 783,
    105: 815,
    106: 847,
    107: 879,
    108: 911,
    109: 943,
    110: 975,
    111: 1007,
    112: 1055,
    113: 1119,
    114: 1183,
    115: 1247,
    116: 1311,
    117: 1375,
    118: 1439,
    119: 1503,
    120: 1567,
    121: 1631,
    122: 1695,
    123: 1759,
    124: 1823,
    125: 1887,
    126: 1951,
    127: 2015,
    128: 2111,
    129: 2239,
    130: 2367,
    131: 2495,
    132: 2623,
    133: 2751,
    134: 2879,
    135: 3007,
    136: 3135,
    137: 3263,
    138: 3391,
    139: 3519,
    140: 3647,
    141: 3775,
    142: 3903,
    143: 4031,
    144: 4223,
    145: 4479,
    146: 4735,
    147: 4991,
    148: 5247,
    149: 5503,
    150: 5759,
    151: 6015,
    152: 6271,
    153: 6527,
    154: 6783,
    155: 7039,
    156: 7295,
    157: 7551,
    158: 7807,
    159: 8063,
    160: 8447,
    161: 8959,
    162: 9471,
    163: 9983,
    164: 10495,
    165: 11007,
    166: 11519,
    167: 12031,
    168: 12543,
    169: 13055,
    170: 13567,
    171: 14079,
    172: 14591,
    173: 15103,
    174: 15615,
    175: 16127,
    176: 16895,
    177: 17919,
    178: 18943,
    179: 19967,
    180: 20991,
    181: 22015,
    182: 23039,
    183: 24063,
    184: 25087,
    185: 26111,
    186: 27135,
    187: 28159,
    188: 29183,
    189: 30207,
    190: 31231,
    191: 32255,
    192: 34815,
    193: 38911,
    194: 43007,
    195: 47103,
    196: 51199,
    197: 55295,
    198: 59391,
    199: 63487,
    200: 69631,
    201: 77823,
    202: 86015,
    203: 94207,
    204: 102399,
    205: 110591,
    206: 118783,
    207: 126975,
    208: 139263,
    209: 155647,
    210: 172031,
    211: 188415,
    212: 204799,
    213: 221183,
    214: 237567,
    215: 253951,
    216: 278527,
    217: 311295,
    218: 344063,
    219: 376831,
    220: 409599,
    221: 442367,
    222: 475135,
    223: 507903,
    224: 557055,
    225: 622591,
    226: 688127,
    227: 753663,
    228: 819199,
    229: 884735,
    230: 950271,
    231: 1015807,
    232: 1114111,
    233: 1245183,
    234: 1376255,
    235: 1507327,
    236: 1638399,
    237: 1769471,
    238: 1900543,
    239: 2031615,
    240: 2228223,
    241: 2490367,
    242: 2752511,
    243: 3014655,
    244: 3276799,
    245: 3538943,
    246: 3801087,
    247: 4063231,
    248: 4456447,
    249: 4980735,
    250: 5505023,
    251: 6029311,
    252: 6553599,
    253: 7077887,
    254: 7602175,
    255: 2151415807,
}

# Lookup table for CoDICE-Lo despinning pixel orientations
# See section 9.3.4 of the algorithm document for further information
PIXEL_ORIENTATIONS = {
    0: "A",
    1: "B",
    2: "A",
    3: "B",
    4: "A",
    5: "A",
    6: "B",
    7: "B",
    8: "A",
    9: "A",
    10: "B",
    11: "B",
    12: "A",
    13: "A",
    14: "A",
    15: "B",
    16: "B",
    17: "B",
    18: "A",
    19: "A",
    20: "A",
    21: "B",
    22: "B",
    23: "B",
    24: "A",
    25: "A",
    26: "A",
    27: "A",
    28: "B",
    29: "B",
    30: "B",
    31: "B",
    32: "A",
    33: "A",
    34: "A",
    35: "A",
    36: "B",
    37: "B",
    38: "B",
    39: "B",
    40: "A",
    41: "A",
    42: "A",
    43: "A",
    44: "A",
    45: "B",
    46: "B",
    47: "B",
    48: "B",
    49: "B",
    50: "A",
    51: "A",
    52: "A",
    53: "A",
    54: "A",
    55: "B",
    56: "B",
    57: "B",
    58: "B",
    59: "B",
    60: "A",
    61: "A",
    62: "A",
    63: "A",
    64: "A",
    65: "B",
    66: "B",
    67: "B",
    68: "B",
    69: "B",
    70: "A",
    71: "A",
    72: "A",
    73: "A",
    74: "A",
    75: "B",
    76: "B",
    77: "B",
    78: "B",
    79: "B",
    80: "A",
    81: "A",
    82: "A",
    83: "A",
    84: "A",
    85: "A",
    86: "B",
    87: "B",
    88: "B",
    89: "B",
    90: "B",
    91: "B",
    92: "A",
    93: "A",
    94: "A",
    95: "A",
    96: "A",
    97: "A",
    98: "B",
    99: "B",
    100: "B",
    101: "B",
    102: "B",
    103: "B",
    104: "A",
    105: "A",
    106: "A",
    107: "A",
    108: "A",
    109: "A",
    110: "B",
    111: "B",
    112: "B",
    113: "B",
    114: "B",
    115: "B",
    116: "A",
    117: "A",
    118: "A",
    119: "A",
    120: "A",
    121: "A",
    122: "B",
    123: "B",
    124: "B",
    125: "B",
    126: "B",
    127: "B",
}

# Derived acquisition times that get stored in CDF data variables in L1a
# processing. These are taken from the "Acq Time" column in the "Lo Stepping"
# tab of the "*-SCI-LUT-*.xml" spreadsheet that largely defines CoDICE
# processing.
# TODO: Do away with this lookup table and instead calculate the acquisition
#       times. See GitHub issue #1945.
ACQUISITION_TIMES = {
    0: [
        578.70833333,
        578.70833333,
        578.70833333,
        578.70833333,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
    ],
    1: [
        578.70833333,
        578.70833333,
        578.70833333,
        578.70833333,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
    ],
    2: [
        578.70833333,
        578.70833333,
        578.70833333,
        578.70833333,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
    ],
    3: [
        578.70833333,
        578.70833333,
        578.70833333,
        578.70833333,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        289.35416667,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        192.90277778,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        144.67708333,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        115.74166667,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
        96.45138889,
    ],
}

# These are for product that requires despinning in l1b.
SW_INDEX_TO_POSITION = [1, 2, 3, 23, 24]
NSW_INDEX_TO_POSITION = [
    4,
    5,
    6,
    7,
    8,
    9,
    10,
    11,
    12,
    13,
    14,
    15,
    16,
    17,
    18,
    19,
    20,
    21,
    22,
]

# TODO: Update EFFICIENCY value when better information is available.
# Constant for CoDICE Intensity calculations.
EFFICIENCY = 1

# Lookup table for mapping half-spin (keys) to esa steps (values)
# This is used to determine geometry factors L2
HALF_SPIN_LUT = {
    0: [0],
    1: [1],
    2: [2],
    3: [3],
    4: [4, 5],
    5: [6, 7],
    6: [8, 9],
    7: [10, 11],
    8: [12, 13, 14],
    9: [15, 16, 17],
    10: [18, 19, 20],
    11: [21, 22, 23],
    12: [24, 25, 26, 27],
    13: [28, 29, 30, 31],
    14: [32, 33, 34, 35],
    15: [36, 37, 38, 39],
    16: [40, 41, 42, 43, 44],
    17: [45, 46, 47, 48, 49],
    18: [50, 51, 52, 53, 54],
    19: [55, 56, 57, 58, 59],
    20: [60, 61, 62, 63, 64],
    21: [65, 66, 67, 68, 69],
    22: [70, 71, 72, 73, 74],
    23: [75, 76, 77, 78, 79],
    24: [80, 81, 82, 83, 84, 85],
    25: [86, 87, 88, 89, 90, 91],
    26: [92, 93, 94, 95, 96, 97],
    27: [98, 99, 100, 101, 102, 103],
    28: [104, 105, 106, 107, 108, 109],
    29: [110, 111, 112, 113, 114, 115],
    30: [116, 117, 118, 119, 120, 121],
    31: [122, 123, 124, 125, 126, 127],
}
