"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_s3_1 = require("monocdk/aws-s3");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * S3 Buckets require requests to use SSL
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        const bucketLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        const bucketName = monocdk_1.Stack.of(node).resolve(node.bucketName);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_s3_1.CfnBucketPolicy) {
                if (isMatchingCompliantPolicy(child, bucketLogicalId, bucketName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Bucket Policy requires SSL on the given bucket.
 * @param node The CfnBucketPolicy to check.
 * @param bucketLogicalId The Cfn Logical ID of the bucket.
 * @param bucketName The name of the bucket.
 * @returns Whether the CfnBucketPolicy requires SSL on the given bucket.
 */
function isMatchingCompliantPolicy(node, bucketLogicalId, bucketName) {
    var _a, _b;
    const bucket = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.bucket);
    if (bucket !== bucketLogicalId && bucket !== bucketName) {
        return false;
    }
    const resolvedPolicyDocument = monocdk_1.Stack.of(node).resolve(node.policyDocument);
    for (const statement of resolvedPolicyDocument.Statement) {
        const resolvedStatement = monocdk_1.Stack.of(node).resolve(statement);
        const secureTransport = (_b = (_a = resolvedStatement === null || resolvedStatement === void 0 ? void 0 : resolvedStatement.Condition) === null || _a === void 0 ? void 0 : _a.Bool) === null || _b === void 0 ? void 0 : _b['aws:SecureTransport'];
        if (resolvedStatement.Effect === 'Deny' &&
            checkMatchingAction(resolvedStatement.Action) === true &&
            checkMatchingPrincipal(resolvedStatement.Principal) === true &&
            (secureTransport === 'false' || secureTransport === false) &&
            checkMatchingResources(node, bucketLogicalId, bucketName, resolvedStatement.Resource) === true) {
            return true;
        }
    }
    return false;
}
/**
 * Helper function to check whether the Bucket Policy applies to all actions
 * @param node The CfnBucketPolicy to check
 * @param actions The action in the bucket policy
 * @returns Whether the CfnBucketPolicy applies to all actions
 */
function checkMatchingAction(actions) {
    if (Array.isArray(actions)) {
        for (const action of actions) {
            if (action === '*' || action.toLowerCase() === 's3:*') {
                return true;
            }
        }
    }
    else if (actions === '*' || actions.toLowerCase() === 's3:*') {
        return true;
    }
    return false;
}
/**
 * Helper function to check whether the Bucket Policy applies to all principals
 * @param node The CfnBucketPolicy to check
 * @param principal The principals in the bucket policy
 * @returns Whether the CfnBucketPolicy applies to all principals
 */
function checkMatchingPrincipal(principals) {
    if (principals === '*') {
        return true;
    }
    const awsPrincipal = principals.AWS;
    if (Array.isArray(awsPrincipal)) {
        for (const account of awsPrincipal) {
            if (account === '*') {
                return true;
            }
        }
    }
    else if (awsPrincipal === '*') {
        return true;
    }
    return false;
}
/**
 * Helper function to check whether the Bucket Policy applies to the bucket and all of it's resources
 * @param node The CfnBucketPolicy to check
 * @param bucketLogicalId The Cfn Logical ID of the bucket
 * @param bucketName The name of the bucket
 * @param resources The resources in the bucket policy
 * @returns Whether the Bucket Policy applies to the bucket and all of it's resources
 */
function checkMatchingResources(node, bucketLogicalId, bucketName, resources) {
    if (!Array.isArray(resources)) {
        return false;
    }
    const bucketResourceRegexes = Array();
    const bucketObjectsRegexes = Array();
    bucketResourceRegexes.push(`(${bucketLogicalId}(?![\\w\\-]))`);
    bucketObjectsRegexes.push(`(${bucketLogicalId}(?![\\w\\-]).*\\/\\*)`);
    if (bucketName !== undefined) {
        bucketResourceRegexes.push(`(${bucketName}(?![\\w\\-]))`);
        bucketObjectsRegexes.push(`(${bucketName}(?![\\w\\-]).*\\/\\*)`);
    }
    const fullBucketResourceRegex = new RegExp(bucketResourceRegexes.join('|'));
    const fullBucketObjectsRegex = new RegExp(bucketObjectsRegexes.join('|'));
    let matchedBucketResource = false;
    let matchedObjectsResource = false;
    for (const resource of resources) {
        const resolvedResourceString = JSON.stringify(monocdk_1.Stack.of(node).resolve(resource));
        if (matchedBucketResource === false &&
            fullBucketResourceRegex.test(resolvedResourceString) &&
            !resolvedResourceString.includes('/')) {
            matchedBucketResource = true;
        }
        else if (matchedObjectsResource === false &&
            fullBucketObjectsRegex.test(resolvedResourceString) &&
            resolvedResourceString.indexOf('/') ===
                resolvedResourceString.lastIndexOf('/')) {
            matchedObjectsResource = true;
        }
        if (matchedBucketResource === true && matchedObjectsResource === true) {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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