"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_eks_1 = require("monocdk/aws-eks");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * EKS Clusters publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    var _a, _b;
    if (node instanceof aws_eks_1.CfnCluster) {
        const logging = monocdk_1.Stack.of(node).resolve(node.logging);
        if (logging === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const clusterLogging = monocdk_1.Stack.of(node).resolve(logging.clusterLogging);
        if (clusterLogging === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const enabledTypes = monocdk_1.Stack.of(node).resolve(clusterLogging.enabledTypes);
        if (!Array.isArray(enabledTypes)) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requiredTypes = new Set([
            'api',
            'audit',
            'authenticator',
            'controllerManager',
            'scheduler',
        ]);
        for (const enabled of enabledTypes) {
            requiredTypes.delete(nag_rules_1.NagRules.resolveIfPrimitive(node, enabled.type));
            if (requiredTypes.size === 0) {
                break;
            }
        }
        if (requiredTypes.size !== 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node.cfnResourceType === 'Custom::AWSCDK-EKS-Cluster') {
        // The CDK uses a Custom Resource with AWS SDK calls to create EKS Clusters
        const props = monocdk_1.Stack.of(node).resolve(node._cfnProperties);
        const clusterLogging = monocdk_1.Stack.of(node).resolve((_b = (_a = props === null || props === void 0 ? void 0 : props.Config) === null || _a === void 0 ? void 0 : _a.logging) === null || _b === void 0 ? void 0 : _b.clusterLogging);
        if (!Array.isArray(clusterLogging)) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requiredTypes = new Set([
            'api',
            'audit',
            'authenticator',
            'controllerManager',
            'scheduler',
        ]);
        for (const config of clusterLogging) {
            if ((config === null || config === void 0 ? void 0 : config.enabled) === true) {
                for (const type of config === null || config === void 0 ? void 0 : config.types) {
                    requiredTypes.delete(type);
                    if (requiredTypes.size === 0) {
                        break;
                    }
                }
            }
        }
        if (requiredTypes.size !== 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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