"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagSuppressions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
/**
 * Helper class with methods to add cdk-nag suppressions to cdk resources
 */
class NagSuppressions {
    /**
     * Apply cdk-nag suppressions to a Stack and optionally nested stacks
     * @param stack The Stack to apply the suppression to
     * @param suppressions A list of suppressions to apply to the stack
     * @param applyToNestedStacks Apply the suppressions to children stacks (default:false)
     */
    static addStackSuppressions(stack, suppressions, applyToNestedStacks = false) {
        const stacks = applyToNestedStacks
            ? stack.node.findAll().filter((x) => x instanceof monocdk_1.Stack)
            : [stack];
        stacks.forEach((s) => {
            var _b, _c;
            const newSuppressions = [];
            for (const suppression of suppressions) {
                if (suppression.reason.length >= 10) {
                    newSuppressions.push(suppression);
                }
                else {
                    throw Error(`${s.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
                }
            }
            let currentSuppressions = (_c = (_b = s.templateOptions.metadata) === null || _b === void 0 ? void 0 : _b.cdk_nag) === null || _c === void 0 ? void 0 : _c.rules_to_suppress;
            currentSuppressions = Array.isArray(currentSuppressions)
                ? currentSuppressions
                : [];
            currentSuppressions.push(...newSuppressions);
            const dedupSuppressions = new Set();
            const result = currentSuppressions.filter((x) => !dedupSuppressions.has(JSON.stringify(x))
                ? dedupSuppressions.add(JSON.stringify(x))
                : false);
            if (s.templateOptions.metadata) {
                s.templateOptions.metadata.cdk_nag = {
                    rules_to_suppress: result,
                };
            }
            else {
                s.templateOptions.metadata = {
                    cdk_nag: { rules_to_suppress: result },
                };
            }
        });
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children
     * @param construct The IConstruct to apply the suppression to
     * @param suppressions A list of suppressions to apply to the resource
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false)
     */
    static addResourceSuppressions(construct, suppressions, applyToChildren = false) {
        var _b;
        const newSuppressions = [];
        for (const suppression of suppressions) {
            if (suppression.reason.length >= 10) {
                newSuppressions.push(suppression);
            }
            else {
                throw Error(`${construct.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        const constructs = applyToChildren ? construct.node.findAll() : [construct];
        for (const child of constructs) {
            const possibleL1 = child.node.defaultChild
                ? child.node.defaultChild
                : child;
            if (possibleL1 instanceof monocdk_1.CfnResource) {
                const resource = possibleL1;
                let currentSuppressions = (_b = resource.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
                currentSuppressions = Array.isArray(currentSuppressions)
                    ? currentSuppressions
                    : [];
                currentSuppressions.push(...newSuppressions);
                const dedupSuppressions = new Set();
                const result = currentSuppressions.filter((s) => !dedupSuppressions.has(JSON.stringify(s))
                    ? dedupSuppressions.add(JSON.stringify(s))
                    : false);
                resource.addMetadata('cdk_nag', {
                    rules_to_suppress: result,
                });
            }
        }
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children via its path
     * @param stack The Stack the construct belongs to
     * @param path The path to the construct in the provided stack
     * @param suppressions A list of suppressions to apply to the resource
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false)
     */
    static addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren = false) {
        for (const child of stack.node.findAll()) {
            const fixedPath = path.replace(/^\//, '');
            if (child.node.path === fixedPath ||
                child.node.path + '/Resource' === fixedPath) {
                NagSuppressions.addResourceSuppressions(child, suppressions, applyToChildren);
            }
        }
    }
}
exports.NagSuppressions = NagSuppressions;
_a = JSII_RTTI_SYMBOL_1;
NagSuppressions[_a] = { fqn: "monocdk-nag.NagSuppressions", version: "1.6.1" };
//# sourceMappingURL=data:application/json;base64,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