from collections import Counter
from pathlib import Path
from random import shuffle
from string import punctuation

from click import (
    Choice,
    clear,
    command,
    confirm,
    echo,
    echo_via_pager,
    option,
    prompt,
    version_option,
)
from nltk import download, word_tokenize

from .helpers import (
    display_name,
    get_setting,
    prompt_for_word,
    set_setting,
    settings_file,
    show_directions,
    story_list,
    story_name_from_path,
)


@command()
@version_option(version="1.0.3", prog_name="Angry Libs")
@option(
    "--reset-settings",
    help="Reset your Angry Libs settings and exit.",
    is_flag=True,
    flag_value=True,
)
@option(
    "--nltk-data",
    help='Download/update the "punkt" resource from NLTK and exit.',
    is_flag=True,
    flag_value=True,
)
def main(reset_settings=False, nltk_data=False):
    if reset_settings:
        if confirm(
            "Are you sure you want to reset all Angry Libs settings", prompt_suffix="? "
        ):
            with settings_file().open("w") as file:
                file.write("{}")

            echo("Success!")
        else:
            echo("Aborted")

        return

    if nltk_data:
        download("punkt")
        return

    try:
        get_setting("opened")
    except KeyError:
        show_directions()
        download("punkt")
        echo()
        set_setting("opened", True)

    story_paths = story_list()
    shuffle(story_paths)
    story_paths = story_paths[:5]

    stories = {
        str(index) + ". " + story_name_from_path(path): path
        for index, path in enumerate(story_paths, start=1)
    }

    for story in stories.keys():
        echo(story)

    echo()

    story_choice = prompt(
        "Choose a story", type=Choice([x.split(".")[0] for x in stories.keys()])
    )

    story_path = Path(story_paths[int(story_choice) - 1])

    with story_path.open() as file:
        story_template = file.read()

    blanks_dict = dict()
    blanks_counter = Counter()

    for word in word_tokenize(story_template):
        if "__" in word:
            blank_type = word.split("__")[1]

            if "/" in blank_type:
                blanks_dict[blank_type] = ""
            else:
                blanks_dict[blank_type] = []
                blanks_counter[blank_type] += 1

    keys = list(blanks_dict.keys())
    shuffle(keys)

    clear()

    for blank in keys:
        if "/" in blank:
            blanks_dict[blank] = prompt_for_word(blank.split("/")[0])
        else:
            for _ in range(blanks_counter[blank]):
                blanks_dict[blank].append(prompt_for_word(blank))

    new_text = ""

    for paragraph in story_template.split("\n"):
        previous_word = ""
        for word in word_tokenize(paragraph):
            prefix = ""
            suffix = ""

            if "__" in word:
                blank_type = word.split("__")[1]

                try:
                    suffix = word.split("__")[2]
                except:
                    pass

                try:
                    prefix = word.split("__")[0]
                except:
                    pass

                if "/" in blank_type:
                    word = blanks_dict[blank_type]
                else:
                    word = blanks_dict[blank_type].pop()

            if word in punctuation or word in {"n't", "'s", "n"}:
                new_text = new_text[:-1] + word + " "
            else:
                if previous_word == "." or previous_word == "\n" or previous_word == "":
                    word = word.capitalize()
                new_text += prefix + word + suffix + " "

            previous_word = word

        new_text += "\n"

    new_text = new_text.replace("\\", "\n")

    clear()

    echo_via_pager(new_text)


main()