"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_iam_1 = require("monocdk/aws-iam");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * IAM entities with wildcard permissions have a cdk_nag rule suppression with evidence for those permission
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_iam_1.CfnGroup ||
        node instanceof aws_iam_1.CfnUser ||
        node instanceof aws_iam_1.CfnRole) {
        const inlinePolicies = monocdk_1.Stack.of(node).resolve(node.policies);
        const findings = new Set();
        if (inlinePolicies != undefined) {
            for (const policy of inlinePolicies) {
                const resolvedPolicy = monocdk_1.Stack.of(node).resolve(policy);
                const resolvedPolicyDocument = monocdk_1.Stack.of(node).resolve(resolvedPolicy.policyDocument);
                analyzePolicy(resolvedPolicyDocument).forEach((finding) => findings.add(finding));
            }
        }
        return findings.size ? [...findings] : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_iam_1.CfnPolicy || node instanceof aws_iam_1.CfnManagedPolicy) {
        const policyDocument = monocdk_1.Stack.of(node).resolve(node.policyDocument);
        const findings = analyzePolicy(policyDocument);
        return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
const analyzePolicy = (policy) => {
    var _a;
    const result = new Set();
    for (const statement of (_a = policy.Statement) !== null && _a !== void 0 ? _a : []) {
        // we don't report on denys
        if (statement.Effect === 'Allow') {
            const actions = normalizeArray(statement.Action);
            actions
                .filter((action) => action.includes('*'))
                .map((action) => `Action::${action}`)
                .forEach((action) => result.add(action));
            const resources = normalizeArray(statement.Resource);
            resources
                .filter((resource) => JSON.stringify(resource).includes('*'))
                .map((resource) => `Resource::${resource}`)
                .forEach((resource) => result.add(resource));
        }
    }
    return [...result];
};
const normalizeArray = (arrOrStr) => Array.isArray(arrOrStr) ? arrOrStr : [arrOrStr];
//# sourceMappingURL=data:application/json;base64,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