"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_emr_1 = require("monocdk/aws-emr");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * EMR clusters have local disk encryption enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_emr_1.CfnCluster) {
        const securityConfigurationId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.securityConfiguration);
        if (securityConfigurationId == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_emr_1.CfnSecurityConfiguration) {
                if (isMatchingSecurityConfig(child, securityConfigurationId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the referenced Security Configuration specifies local disk encryption
 * @param node the CfnSecurityConfiguration to check
 * @param securityConfigurationId the Cfn Logical ID of the security configuration
 * returns whether the CfnSecurityConfiguration specifies local disk encryption
 */
function isMatchingSecurityConfig(node, securityConfigurationId) {
    var _a, _b;
    const resolvedSecurityConfigurationLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
    if (resolvedSecurityConfigurationLogicalId === securityConfigurationId ||
        node.name === securityConfigurationId) {
        const securityConfiguration = monocdk_1.Stack.of(node).resolve(node.securityConfiguration);
        const enableAtRestEncryption = securityConfiguration === null || securityConfiguration === void 0 ? void 0 : securityConfiguration.EnableAtRestEncryption;
        const encryptionKeyProviderType = (_b = (_a = securityConfiguration === null || securityConfiguration === void 0 ? void 0 : securityConfiguration.AtRestEncryptionConfiguration) === null || _a === void 0 ? void 0 : _a.LocalDiskEncryptionConfiguration) === null || _b === void 0 ? void 0 : _b.EncryptionKeyProviderType;
        if (enableAtRestEncryption === true &&
            encryptionKeyProviderType !== undefined) {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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