"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const fs_1 = require("fs");
const path_1 = require("path");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("./nag-rules");
const nag_suppression_helper_1 = require("./utils/nag-suppression-helper");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The level of the message that the rule applies.
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel["WARN"] = "Warning";
    NagMessageLevel["ERROR"] = "Error";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule packs.
 */
class NagPack {
    constructor(props) {
        this.reportStacks = new Array();
        this.packName = '';
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
        this.reports =
            props == undefined || props.reports == undefined ? true : props.reports;
    }
    get readPackName() {
        return this.packName;
    }
    get readReportStacks() {
        return this.reportStacks;
    }
    /**
     * Create a rule to be used in the NagPack.
     * @param params The @IApplyRule interface with rule details.
     */
    applyRule(params) {
        if (this.packName === '') {
            throw Error('The NagPack does not have a pack name, therefore the rule could not be applied. Set a packName in the NagPack constructor.');
        }
        const allIgnores = nag_suppression_helper_1.NagSuppressionHelper.getSuppressions(params.node);
        const ruleSuffix = params.ruleSuffixOverride
            ? params.ruleSuffixOverride
            : params.rule.name;
        const ruleId = `${this.packName}-${ruleSuffix}`;
        try {
            const ruleCompliance = params.rule(params.node);
            if (this.reports === true &&
                ruleCompliance === nag_rules_1.NagRuleCompliance.COMPLIANT) {
                this.writeToStackComplianceReport(params, ruleId, ruleCompliance);
            }
            else if (this.isNonCompliant(ruleCompliance)) {
                const findings = this.asFindings(ruleCompliance);
                for (const findingId of findings) {
                    const suppressionReason = this.ignoreRule(allIgnores, ruleId, findingId);
                    if (this.reports === true) {
                        this.writeToStackComplianceReport(params, ruleId, nag_rules_1.NagRuleCompliance.NON_COMPLIANT, suppressionReason);
                    }
                    if (suppressionReason) {
                        if (this.logIgnores === true) {
                            const message = this.createMessage(SUPPRESSION_ID, findingId, `${ruleId} was triggered but suppressed.`, `Provided reason: "${suppressionReason}"`);
                            monocdk_1.Annotations.of(params.node).addInfo(message);
                        }
                    }
                    else {
                        const message = this.createMessage(ruleId, findingId, params.info, params.explanation);
                        if (params.level == NagMessageLevel.ERROR) {
                            monocdk_1.Annotations.of(params.node).addError(message);
                        }
                        else if (params.level == NagMessageLevel.WARN) {
                            monocdk_1.Annotations.of(params.node).addWarning(message);
                        }
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allIgnores, VALIDATION_FAILURE_ID, '');
            if (this.reports === true) {
                this.writeToStackComplianceReport(params, ruleId, 'UNKNOWN', reason);
            }
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, '', `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    monocdk_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, '', information, error.message);
                monocdk_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @param findingId The id of the finding that is being checked.
     * @returns The reason the rule was ignored, or an empty string.
     */
    ignoreRule(ignores, ruleId, findingId) {
        for (let ignore of ignores) {
            if (ignore.id &&
                ignore.reason &&
                JSON.stringify(ignore.reason).length >= 10) {
                if (ruleId === ignore.id) {
                    if (!ignore.appliesTo) {
                        // the rule is not granular so it always applies
                        return ignore.reason;
                    }
                    if (findingId && ignore.appliesTo.includes(findingId)) {
                        // the rule is granular so the findingId must match
                        return `[${findingId}] ${ignore.reason}`;
                    }
                }
            }
            else {
                throw Error(`Improperly formatted cdk_nag rule suppression detected: ${JSON.stringify(ignore)}. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered.
     * @param ruleId The id of the rule.
     * @param findingId The id of the finding.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     */
    createMessage(ruleId, findingId, info, explanation) {
        let message = findingId
            ? `${ruleId}[${findingId}]: ${info}`
            : `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}\n` : `${message}\n`;
    }
    /**
     * Write a line to the rule packs compliance report for the resource's Stack
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    writeToStackComplianceReport(params, ruleId, compliance, explanation = '') {
        var _b;
        const line = this.createComplianceReportLine(params, ruleId, compliance, explanation);
        let outDir = (_b = monocdk_1.App.of(params.node)) === null || _b === void 0 ? void 0 : _b.outdir;
        const fileName = `${this.packName}-${params.node.stack.stackName}-NagReport.csv`;
        const filePath = path_1.join(outDir ? outDir : '', fileName);
        if (!this.reportStacks.includes(fileName)) {
            this.reportStacks.push(fileName);
            fs_1.writeFileSync(filePath, 'Rule ID,Resource ID,Compliance,Exception Reason,Rule Level,Rule Info\n');
        }
        fs_1.appendFileSync(filePath, line);
    }
    /**
     * Helper function to create a line for the compliance report
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    createComplianceReportLine(params, ruleId, compliance, explanation = '') {
        //| Rule ID | Resource ID | Compliance | Exception Reason | Rule Level | Rule Info
        const line = Array();
        line.push(ruleId);
        line.push(params.node.node.path);
        if ((compliance === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            compliance === 'UNKNOWN') &&
            explanation !== '') {
            line.push('Suppressed');
            line.push(explanation);
        }
        else {
            line.push(compliance);
            line.push('N/A');
        }
        line.push(params.level);
        line.push(params.info);
        return line.map((i) => '"' + i.replace(/"/g, '""') + '"').join(',') + '\n';
    }
    isNonCompliant(ruleResult) {
        return (ruleResult === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            Array.isArray(ruleResult));
    }
    asFindings(ruleResult) {
        if (Array.isArray(ruleResult)) {
            return ruleResult;
        }
        else {
            return [''];
        }
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "monocdk-nag.NagPack", version: "1.7.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmFnLXBhY2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvbmFnLXBhY2sudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0VBR0U7QUFDRiwyQkFBbUQ7QUFDbkQsK0JBQTRCO0FBQzVCLHFDQU1pQjtBQUVqQiwyQ0FBZ0Y7QUFDaEYsMkVBQXNFO0FBRXRFLE1BQU0scUJBQXFCLEdBQUcseUJBQXlCLENBQUM7QUFDeEQsTUFBTSxjQUFjLEdBQUcsbUJBQW1CLENBQUM7QUFxRDNDOztHQUVHO0FBQ0gsSUFBWSxlQUdYO0FBSEQsV0FBWSxlQUFlO0lBQ3pCLG1DQUFnQixDQUFBO0lBQ2hCLGtDQUFlLENBQUE7QUFDakIsQ0FBQyxFQUhXLGVBQWUsR0FBZix1QkFBZSxLQUFmLHVCQUFlLFFBRzFCO0FBRUQ7O0dBRUc7QUFDSCxNQUFzQixPQUFPO0lBTzNCLFlBQVksS0FBb0I7UUFIdEIsaUJBQVksR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO1FBQ25DLGFBQVEsR0FBRyxFQUFFLENBQUM7UUFHdEIsSUFBSSxDQUFDLE9BQU87WUFDVixLQUFLLElBQUksU0FBUyxJQUFJLEtBQUssQ0FBQyxPQUFPLElBQUksU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDM0UsSUFBSSxDQUFDLFVBQVU7WUFDYixLQUFLLElBQUksU0FBUyxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksU0FBUztnQkFDakQsQ0FBQyxDQUFDLEtBQUs7Z0JBQ1AsQ0FBQyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUM7UUFDdkIsSUFBSSxDQUFDLE9BQU87WUFDVixLQUFLLElBQUksU0FBUyxJQUFJLEtBQUssQ0FBQyxPQUFPLElBQUksU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUM7SUFDNUUsQ0FBQztJQUVELElBQVcsWUFBWTtRQUNyQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUM7SUFDdkIsQ0FBQztJQUNELElBQVcsZ0JBQWdCO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQztJQUMzQixDQUFDO0lBTUQ7OztPQUdHO0lBQ08sU0FBUyxDQUFDLE1BQWtCO1FBQ3BDLElBQUksSUFBSSxDQUFDLFFBQVEsS0FBSyxFQUFFLEVBQUU7WUFDeEIsTUFBTSxLQUFLLENBQ1QsNEhBQTRILENBQzdILENBQUM7U0FDSDtRQUNELE1BQU0sVUFBVSxHQUFHLDZDQUFvQixDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDckUsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLGtCQUFrQjtZQUMxQyxDQUFDLENBQUMsTUFBTSxDQUFDLGtCQUFrQjtZQUMzQixDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7UUFDckIsTUFBTSxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsUUFBUSxJQUFJLFVBQVUsRUFBRSxDQUFDO1FBQ2hELElBQUk7WUFDRixNQUFNLGNBQWMsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNoRCxJQUNFLElBQUksQ0FBQyxPQUFPLEtBQUssSUFBSTtnQkFDckIsY0FBYyxLQUFLLDZCQUFpQixDQUFDLFNBQVMsRUFDOUM7Z0JBQ0EsSUFBSSxDQUFDLDRCQUE0QixDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUM7YUFDbkU7aUJBQU0sSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxFQUFFO2dCQUM5QyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FBQyxDQUFDO2dCQUNqRCxLQUFLLE1BQU0sU0FBUyxJQUFJLFFBQVEsRUFBRTtvQkFDaEMsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUN2QyxVQUFVLEVBQ1YsTUFBTSxFQUNOLFNBQVMsQ0FDVixDQUFDO29CQUVGLElBQUksSUFBSSxDQUFDLE9BQU8sS0FBSyxJQUFJLEVBQUU7d0JBQ3pCLElBQUksQ0FBQyw0QkFBNEIsQ0FDL0IsTUFBTSxFQUNOLE1BQU0sRUFDTiw2QkFBaUIsQ0FBQyxhQUFhLEVBQy9CLGlCQUFpQixDQUNsQixDQUFDO3FCQUNIO29CQUVELElBQUksaUJBQWlCLEVBQUU7d0JBQ3JCLElBQUksSUFBSSxDQUFDLFVBQVUsS0FBSyxJQUFJLEVBQUU7NEJBQzVCLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxhQUFhLENBQ2hDLGNBQWMsRUFDZCxTQUFTLEVBQ1QsR0FBRyxNQUFNLGdDQUFnQyxFQUN6QyxxQkFBcUIsaUJBQWlCLEdBQUcsQ0FDMUMsQ0FBQzs0QkFDRixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO3lCQUM5QztxQkFDRjt5QkFBTTt3QkFDTCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsYUFBYSxDQUNoQyxNQUFNLEVBQ04sU0FBUyxFQUNULE1BQU0sQ0FBQyxJQUFJLEVBQ1gsTUFBTSxDQUFDLFdBQVcsQ0FDbkIsQ0FBQzt3QkFDRixJQUFJLE1BQU0sQ0FBQyxLQUFLLElBQUksZUFBZSxDQUFDLEtBQUssRUFBRTs0QkFDekMscUJBQVcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQzt5QkFDL0M7NkJBQU0sSUFBSSxNQUFNLENBQUMsS0FBSyxJQUFJLGVBQWUsQ0FBQyxJQUFJLEVBQUU7NEJBQy9DLHFCQUFXLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUM7eUJBQ2pEO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtRQUFDLE9BQU8sS0FBSyxFQUFFO1lBQ2QsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxVQUFVLEVBQUUscUJBQXFCLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDdEUsSUFBSSxJQUFJLENBQUMsT0FBTyxLQUFLLElBQUksRUFBRTtnQkFDekIsSUFBSSxDQUFDLDRCQUE0QixDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsU0FBUyxFQUFFLE1BQU0sQ0FBQyxDQUFDO2FBQ3RFO1lBQ0QsSUFBSSxNQUFNLEVBQUU7Z0JBQ1YsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLElBQUksRUFBRTtvQkFDNUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FDaEMsY0FBYyxFQUNkLEVBQUUsRUFDRixHQUFHLHFCQUFxQixnQ0FBZ0MsRUFDeEQsTUFBTSxDQUNQLENBQUM7b0JBQ0YscUJBQVcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztpQkFDOUM7YUFDRjtpQkFBTTtnQkFDTCxNQUFNLFdBQVcsR0FBRyxJQUFJLE1BQU0sMEpBQTBKLENBQUM7Z0JBQ3pMLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxhQUFhLENBQ2hDLHFCQUFxQixFQUNyQixFQUFFLEVBQ0YsV0FBVyxFQUNWLEtBQWUsQ0FBQyxPQUFPLENBQ3pCLENBQUM7Z0JBQ0YscUJBQVcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQzthQUNqRDtTQUNGO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNPLFVBQVUsQ0FDbEIsT0FBNkIsRUFDN0IsTUFBYyxFQUNkLFNBQWlCO1FBRWpCLEtBQUssSUFBSSxNQUFNLElBQUksT0FBTyxFQUFFO1lBQzFCLElBQ0UsTUFBTSxDQUFDLEVBQUU7Z0JBQ1QsTUFBTSxDQUFDLE1BQU07Z0JBQ2IsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsTUFBTSxJQUFJLEVBQUUsRUFDMUM7Z0JBQ0EsSUFBSSxNQUFNLEtBQUssTUFBTSxDQUFDLEVBQUUsRUFBRTtvQkFDeEIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUU7d0JBQ3JCLGdEQUFnRDt3QkFDaEQsT0FBTyxNQUFNLENBQUMsTUFBTSxDQUFDO3FCQUN0QjtvQkFDRCxJQUFJLFNBQVMsSUFBSSxNQUFNLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsRUFBRTt3QkFDckQsbURBQW1EO3dCQUNuRCxPQUFPLElBQUksU0FBUyxLQUFLLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQztxQkFDMUM7aUJBQ0Y7YUFDRjtpQkFBTTtnQkFDTCxNQUFNLEtBQUssQ0FDVCwyREFBMkQsSUFBSSxDQUFDLFNBQVMsQ0FDdkUsTUFBTSxDQUNQLG9HQUFvRyxDQUN0RyxDQUFDO2FBQ0g7U0FDRjtRQUNELE9BQU8sRUFBRSxDQUFDO0lBQ1osQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDTyxhQUFhLENBQ3JCLE1BQWMsRUFDZCxTQUFpQixFQUNqQixJQUFZLEVBQ1osV0FBbUI7UUFFbkIsSUFBSSxPQUFPLEdBQUcsU0FBUztZQUNyQixDQUFDLENBQUMsR0FBRyxNQUFNLElBQUksU0FBUyxNQUFNLElBQUksRUFBRTtZQUNwQyxDQUFDLENBQUMsR0FBRyxNQUFNLEtBQUssSUFBSSxFQUFFLENBQUM7UUFDekIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHLE9BQU8sSUFBSSxXQUFXLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxPQUFPLElBQUksQ0FBQztJQUN2RSxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ08sNEJBQTRCLENBQ3BDLE1BQWtCLEVBQ2xCLE1BQWMsRUFDZCxVQUdhLEVBQ2IsY0FBc0IsRUFBRTs7UUFFeEIsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLDBCQUEwQixDQUMxQyxNQUFNLEVBQ04sTUFBTSxFQUNOLFVBQVUsRUFDVixXQUFXLENBQ1osQ0FBQztRQUNGLElBQUksTUFBTSxTQUFHLGFBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQywwQ0FBRSxNQUFNLENBQUM7UUFDekMsTUFBTSxRQUFRLEdBQUcsR0FBRyxJQUFJLENBQUMsUUFBUSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsZ0JBQWdCLENBQUM7UUFDakYsTUFBTSxRQUFRLEdBQUcsV0FBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQ3pDLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ2pDLGtCQUFhLENBQ1gsUUFBUSxFQUNSLHdFQUF3RSxDQUN6RSxDQUFDO1NBQ0g7UUFDRCxtQkFBYyxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ08sMEJBQTBCLENBQ2xDLE1BQWtCLEVBQ2xCLE1BQWMsRUFDZCxVQUdhLEVBQ2IsY0FBc0IsRUFBRTtRQUV4QixrRkFBa0Y7UUFDbEYsTUFBTSxJQUFJLEdBQUcsS0FBSyxFQUFVLENBQUM7UUFDN0IsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNsQixJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pDLElBQ0UsQ0FBQyxVQUFVLEtBQUssNkJBQWlCLENBQUMsYUFBYTtZQUM3QyxVQUFVLEtBQUssU0FBUyxDQUFDO1lBQzNCLFdBQVcsS0FBSyxFQUFFLEVBQ2xCO1lBQ0EsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUN4QixJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQ3hCO2FBQU07WUFDTCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDbEI7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN4QixJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2QixPQUFPLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsR0FBRyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsR0FBRyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFDO0lBQzdFLENBQUM7SUFFTyxjQUFjLENBQUMsVUFBeUI7UUFDOUMsT0FBTyxDQUNMLFVBQVUsS0FBSyw2QkFBaUIsQ0FBQyxhQUFhO1lBQzlDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQzFCLENBQUM7SUFDSixDQUFDO0lBRU8sVUFBVSxDQUFDLFVBQXlCO1FBQzFDLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUM3QixPQUFPLFVBQVUsQ0FBQztTQUNuQjthQUFNO1lBQ0wsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQ2I7SUFDSCxDQUFDOztBQXpRSCwwQkEwUUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuKi9cbmltcG9ydCB7IGFwcGVuZEZpbGVTeW5jLCB3cml0ZUZpbGVTeW5jIH0gZnJvbSAnZnMnO1xuaW1wb3J0IHsgam9pbiB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0IHtcbiAgSUFzcGVjdCxcbiAgSUNvbnN0cnVjdCxcbiAgQW5ub3RhdGlvbnMsXG4gIENmblJlc291cmNlLFxuICBBcHAsXG59IGZyb20gJ21vbm9jZGsnO1xuaW1wb3J0IHsgTmFnUGFja1N1cHByZXNzaW9uIH0gZnJvbSAnLi9tb2RlbHMvbmFnLXN1cHByZXNzaW9uJztcbmltcG9ydCB7IE5hZ1J1bGVDb21wbGlhbmNlLCBOYWdSdWxlUmVzdWx0LCBOYWdSdWxlRmluZGluZ3MgfSBmcm9tICcuL25hZy1ydWxlcyc7XG5pbXBvcnQgeyBOYWdTdXBwcmVzc2lvbkhlbHBlciB9IGZyb20gJy4vdXRpbHMvbmFnLXN1cHByZXNzaW9uLWhlbHBlcic7XG5cbmNvbnN0IFZBTElEQVRJT05fRkFJTFVSRV9JRCA9ICdDZGtOYWdWYWxpZGF0aW9uRmFpbHVyZSc7XG5jb25zdCBTVVBQUkVTU0lPTl9JRCA9ICdDZGtOYWdTdXBwcmVzc2lvbic7XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciBjcmVhdGluZyBhIE5hZyBydWxlIHBhY2suXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTmFnUGFja1Byb3BzIHtcbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IHRvIGVuYWJsZSBleHRlbmRlZCBleHBsYW5hdG9yeSBkZXNjcmlwdGlvbnMgb24gd2FybmluZywgZXJyb3IsIGFuZCBsb2dnZWQgaWdub3JlIG1lc3NhZ2VzIChkZWZhdWx0OiBmYWxzZSkuXG4gICAqL1xuICByZWFkb25seSB2ZXJib3NlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gbG9nIHRyaWdnZXJlZCBydWxlcyB0aGF0IGhhdmUgYmVlbiBzdXBwcmVzc2VkIGFzIGluZm9ybWF0aW9uYWwgbWVzc2FnZXMgKGRlZmF1bHQ6IGZhbHNlKS5cbiAgICovXG4gIHJlYWRvbmx5IGxvZ0lnbm9yZXM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0byBnZW5lcmF0ZSBDU1YgY29tcGxpYW5jZSByZXBvcnRzIGZvciBhcHBsaWVkIFN0YWNrcyBpbiB0aGUgQXBwJ3Mgb3V0cHV0IGRpcmVjdG9yeSAoZGVmYXVsdDogdHJ1ZSkuXG4gICAqL1xuICByZWFkb25seSByZXBvcnRzPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBJbnRlcmZhY2UgZm9yIEpTSUkgaW50ZXJvcGVyYWJpbGl0eSBmb3IgcGFzc2luZyBwYXJhbWV0ZXJzIGFuZCB0aGUgUnVsZSBDYWxsYmFjayB0byBAYXBwbHlSdWxlIG1ldGhvZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQXBwbHlSdWxlIHtcbiAgLyoqXG4gICAqIE92ZXJyaWRlIGZvciB0aGUgc3VmZml4IG9mIHRoZSBSdWxlIElEIGZvciB0aGlzIHJ1bGVcbiAgICovXG4gIHJ1bGVTdWZmaXhPdmVycmlkZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFdoeSB0aGUgcnVsZSB3YXMgdHJpZ2dlcmVkLlxuICAgKi9cbiAgaW5mbzogc3RyaW5nO1xuICAvKipcbiAgICogV2h5IHRoZSBydWxlIGV4aXN0cy5cbiAgICovXG4gIGV4cGxhbmF0aW9uOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgYW5ub3RhdGlvbnMgbWVzc2FnZSBsZXZlbCB0byBhcHBseSB0byB0aGUgcnVsZSBpZiB0cmlnZ2VyZWQuXG4gICAqL1xuICBsZXZlbDogTmFnTWVzc2FnZUxldmVsO1xuICAvKipcbiAgICogSWdub3JlcyBsaXN0ZWQgaW4gY2RrLW5hZyBtZXRhZGF0YS5cbiAgICovXG4gIG5vZGU6IENmblJlc291cmNlO1xuICAvKipcbiAgICogVGhlIGNhbGxiYWNrIHRvIHRoZSBydWxlLlxuICAgKiBAcGFyYW0gbm9kZSBUaGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2suXG4gICAqL1xuICBydWxlKG5vZGU6IENmblJlc291cmNlKTogTmFnUnVsZVJlc3VsdDtcbn1cblxuLyoqXG4gKiBUaGUgbGV2ZWwgb2YgdGhlIG1lc3NhZ2UgdGhhdCB0aGUgcnVsZSBhcHBsaWVzLlxuICovXG5leHBvcnQgZW51bSBOYWdNZXNzYWdlTGV2ZWwge1xuICBXQVJOID0gJ1dhcm5pbmcnLFxuICBFUlJPUiA9ICdFcnJvcicsXG59XG5cbi8qKlxuICogQmFzZSBjbGFzcyBmb3IgYWxsIHJ1bGUgcGFja3MuXG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBOYWdQYWNrIGltcGxlbWVudHMgSUFzcGVjdCB7XG4gIHByb3RlY3RlZCB2ZXJib3NlOiBib29sZWFuO1xuICBwcm90ZWN0ZWQgbG9nSWdub3JlczogYm9vbGVhbjtcbiAgcHJvdGVjdGVkIHJlcG9ydHM6IGJvb2xlYW47XG4gIHByb3RlY3RlZCByZXBvcnRTdGFja3MgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuICBwcm90ZWN0ZWQgcGFja05hbWUgPSAnJztcblxuICBjb25zdHJ1Y3Rvcihwcm9wcz86IE5hZ1BhY2tQcm9wcykge1xuICAgIHRoaXMudmVyYm9zZSA9XG4gICAgICBwcm9wcyA9PSB1bmRlZmluZWQgfHwgcHJvcHMudmVyYm9zZSA9PSB1bmRlZmluZWQgPyBmYWxzZSA6IHByb3BzLnZlcmJvc2U7XG4gICAgdGhpcy5sb2dJZ25vcmVzID1cbiAgICAgIHByb3BzID09IHVuZGVmaW5lZCB8fCBwcm9wcy5sb2dJZ25vcmVzID09IHVuZGVmaW5lZFxuICAgICAgICA/IGZhbHNlXG4gICAgICAgIDogcHJvcHMubG9nSWdub3JlcztcbiAgICB0aGlzLnJlcG9ydHMgPVxuICAgICAgcHJvcHMgPT0gdW5kZWZpbmVkIHx8IHByb3BzLnJlcG9ydHMgPT0gdW5kZWZpbmVkID8gdHJ1ZSA6IHByb3BzLnJlcG9ydHM7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHJlYWRQYWNrTmFtZSgpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLnBhY2tOYW1lO1xuICB9XG4gIHB1YmxpYyBnZXQgcmVhZFJlcG9ydFN0YWNrcygpOiBzdHJpbmdbXSB7XG4gICAgcmV0dXJuIHRoaXMucmVwb3J0U3RhY2tzO1xuICB9XG4gIC8qKlxuICAgKiBBbGwgYXNwZWN0cyBjYW4gdmlzaXQgYW4gSUNvbnN0cnVjdC5cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCB2aXNpdChub2RlOiBJQ29uc3RydWN0KTogdm9pZDtcblxuICAvKipcbiAgICogQ3JlYXRlIGEgcnVsZSB0byBiZSB1c2VkIGluIHRoZSBOYWdQYWNrLlxuICAgKiBAcGFyYW0gcGFyYW1zIFRoZSBASUFwcGx5UnVsZSBpbnRlcmZhY2Ugd2l0aCBydWxlIGRldGFpbHMuXG4gICAqL1xuICBwcm90ZWN0ZWQgYXBwbHlSdWxlKHBhcmFtczogSUFwcGx5UnVsZSk6IHZvaWQge1xuICAgIGlmICh0aGlzLnBhY2tOYW1lID09PSAnJykge1xuICAgICAgdGhyb3cgRXJyb3IoXG4gICAgICAgICdUaGUgTmFnUGFjayBkb2VzIG5vdCBoYXZlIGEgcGFjayBuYW1lLCB0aGVyZWZvcmUgdGhlIHJ1bGUgY291bGQgbm90IGJlIGFwcGxpZWQuIFNldCBhIHBhY2tOYW1lIGluIHRoZSBOYWdQYWNrIGNvbnN0cnVjdG9yLidcbiAgICAgICk7XG4gICAgfVxuICAgIGNvbnN0IGFsbElnbm9yZXMgPSBOYWdTdXBwcmVzc2lvbkhlbHBlci5nZXRTdXBwcmVzc2lvbnMocGFyYW1zLm5vZGUpO1xuICAgIGNvbnN0IHJ1bGVTdWZmaXggPSBwYXJhbXMucnVsZVN1ZmZpeE92ZXJyaWRlXG4gICAgICA/IHBhcmFtcy5ydWxlU3VmZml4T3ZlcnJpZGVcbiAgICAgIDogcGFyYW1zLnJ1bGUubmFtZTtcbiAgICBjb25zdCBydWxlSWQgPSBgJHt0aGlzLnBhY2tOYW1lfS0ke3J1bGVTdWZmaXh9YDtcbiAgICB0cnkge1xuICAgICAgY29uc3QgcnVsZUNvbXBsaWFuY2UgPSBwYXJhbXMucnVsZShwYXJhbXMubm9kZSk7XG4gICAgICBpZiAoXG4gICAgICAgIHRoaXMucmVwb3J0cyA9PT0gdHJ1ZSAmJlxuICAgICAgICBydWxlQ29tcGxpYW5jZSA9PT0gTmFnUnVsZUNvbXBsaWFuY2UuQ09NUExJQU5UXG4gICAgICApIHtcbiAgICAgICAgdGhpcy53cml0ZVRvU3RhY2tDb21wbGlhbmNlUmVwb3J0KHBhcmFtcywgcnVsZUlkLCBydWxlQ29tcGxpYW5jZSk7XG4gICAgICB9IGVsc2UgaWYgKHRoaXMuaXNOb25Db21wbGlhbnQocnVsZUNvbXBsaWFuY2UpKSB7XG4gICAgICAgIGNvbnN0IGZpbmRpbmdzID0gdGhpcy5hc0ZpbmRpbmdzKHJ1bGVDb21wbGlhbmNlKTtcbiAgICAgICAgZm9yIChjb25zdCBmaW5kaW5nSWQgb2YgZmluZGluZ3MpIHtcbiAgICAgICAgICBjb25zdCBzdXBwcmVzc2lvblJlYXNvbiA9IHRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgICAgIGFsbElnbm9yZXMsXG4gICAgICAgICAgICBydWxlSWQsXG4gICAgICAgICAgICBmaW5kaW5nSWRcbiAgICAgICAgICApO1xuXG4gICAgICAgICAgaWYgKHRoaXMucmVwb3J0cyA9PT0gdHJ1ZSkge1xuICAgICAgICAgICAgdGhpcy53cml0ZVRvU3RhY2tDb21wbGlhbmNlUmVwb3J0KFxuICAgICAgICAgICAgICBwYXJhbXMsXG4gICAgICAgICAgICAgIHJ1bGVJZCxcbiAgICAgICAgICAgICAgTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVCxcbiAgICAgICAgICAgICAgc3VwcHJlc3Npb25SZWFzb25cbiAgICAgICAgICAgICk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaWYgKHN1cHByZXNzaW9uUmVhc29uKSB7XG4gICAgICAgICAgICBpZiAodGhpcy5sb2dJZ25vcmVzID09PSB0cnVlKSB7XG4gICAgICAgICAgICAgIGNvbnN0IG1lc3NhZ2UgPSB0aGlzLmNyZWF0ZU1lc3NhZ2UoXG4gICAgICAgICAgICAgICAgU1VQUFJFU1NJT05fSUQsXG4gICAgICAgICAgICAgICAgZmluZGluZ0lkLFxuICAgICAgICAgICAgICAgIGAke3J1bGVJZH0gd2FzIHRyaWdnZXJlZCBidXQgc3VwcHJlc3NlZC5gLFxuICAgICAgICAgICAgICAgIGBQcm92aWRlZCByZWFzb246IFwiJHtzdXBwcmVzc2lvblJlYXNvbn1cImBcbiAgICAgICAgICAgICAgKTtcbiAgICAgICAgICAgICAgQW5ub3RhdGlvbnMub2YocGFyYW1zLm5vZGUpLmFkZEluZm8obWVzc2FnZSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGNvbnN0IG1lc3NhZ2UgPSB0aGlzLmNyZWF0ZU1lc3NhZ2UoXG4gICAgICAgICAgICAgIHJ1bGVJZCxcbiAgICAgICAgICAgICAgZmluZGluZ0lkLFxuICAgICAgICAgICAgICBwYXJhbXMuaW5mbyxcbiAgICAgICAgICAgICAgcGFyYW1zLmV4cGxhbmF0aW9uXG4gICAgICAgICAgICApO1xuICAgICAgICAgICAgaWYgKHBhcmFtcy5sZXZlbCA9PSBOYWdNZXNzYWdlTGV2ZWwuRVJST1IpIHtcbiAgICAgICAgICAgICAgQW5ub3RhdGlvbnMub2YocGFyYW1zLm5vZGUpLmFkZEVycm9yKG1lc3NhZ2UpO1xuICAgICAgICAgICAgfSBlbHNlIGlmIChwYXJhbXMubGV2ZWwgPT0gTmFnTWVzc2FnZUxldmVsLldBUk4pIHtcbiAgICAgICAgICAgICAgQW5ub3RhdGlvbnMub2YocGFyYW1zLm5vZGUpLmFkZFdhcm5pbmcobWVzc2FnZSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgIGNvbnN0IHJlYXNvbiA9IHRoaXMuaWdub3JlUnVsZShhbGxJZ25vcmVzLCBWQUxJREFUSU9OX0ZBSUxVUkVfSUQsICcnKTtcbiAgICAgIGlmICh0aGlzLnJlcG9ydHMgPT09IHRydWUpIHtcbiAgICAgICAgdGhpcy53cml0ZVRvU3RhY2tDb21wbGlhbmNlUmVwb3J0KHBhcmFtcywgcnVsZUlkLCAnVU5LTk9XTicsIHJlYXNvbik7XG4gICAgICB9XG4gICAgICBpZiAocmVhc29uKSB7XG4gICAgICAgIGlmICh0aGlzLmxvZ0lnbm9yZXMgPT09IHRydWUpIHtcbiAgICAgICAgICBjb25zdCBtZXNzYWdlID0gdGhpcy5jcmVhdGVNZXNzYWdlKFxuICAgICAgICAgICAgU1VQUFJFU1NJT05fSUQsXG4gICAgICAgICAgICAnJyxcbiAgICAgICAgICAgIGAke1ZBTElEQVRJT05fRkFJTFVSRV9JRH0gd2FzIHRyaWdnZXJlZCBidXQgc3VwcHJlc3NlZC5gLFxuICAgICAgICAgICAgcmVhc29uXG4gICAgICAgICAgKTtcbiAgICAgICAgICBBbm5vdGF0aW9ucy5vZihwYXJhbXMubm9kZSkuYWRkSW5mbyhtZXNzYWdlKTtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgY29uc3QgaW5mb3JtYXRpb24gPSBgJyR7cnVsZUlkfScgdGhyZXcgYW4gZXJyb3IgZHVyaW5nIHZhbGlkYXRpb24uIFRoaXMgaXMgZ2VuZXJhbGx5IGNhdXNlZCBieSBhIHBhcmFtZXRlciByZWZlcmVuY2luZyBhbiBpbnRyaW5zaWMgZnVuY3Rpb24uIEZvciBtb3JlIGRldGFpbHMgZW5hYmxlIHZlcmJvc2UgbG9nZ2luZy4nYDtcbiAgICAgICAgY29uc3QgbWVzc2FnZSA9IHRoaXMuY3JlYXRlTWVzc2FnZShcbiAgICAgICAgICBWQUxJREFUSU9OX0ZBSUxVUkVfSUQsXG4gICAgICAgICAgJycsXG4gICAgICAgICAgaW5mb3JtYXRpb24sXG4gICAgICAgICAgKGVycm9yIGFzIEVycm9yKS5tZXNzYWdlXG4gICAgICAgICk7XG4gICAgICAgIEFubm90YXRpb25zLm9mKHBhcmFtcy5ub2RlKS5hZGRXYXJuaW5nKG1lc3NhZ2UpO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayB3aGV0aGVyIGEgc3BlY2lmaWMgcnVsZSBzaG91bGQgYmUgaWdub3JlZC5cbiAgICogQHBhcmFtIGlnbm9yZXMgVGhlIGlnbm9yZXMgbGlzdGVkIGluIGNkay1uYWcgbWV0YWRhdGEuXG4gICAqIEBwYXJhbSBydWxlSWQgVGhlIGlkIG9mIHRoZSBydWxlIHRvIGlnbm9yZS5cbiAgICogQHBhcmFtIGZpbmRpbmdJZCBUaGUgaWQgb2YgdGhlIGZpbmRpbmcgdGhhdCBpcyBiZWluZyBjaGVja2VkLlxuICAgKiBAcmV0dXJucyBUaGUgcmVhc29uIHRoZSBydWxlIHdhcyBpZ25vcmVkLCBvciBhbiBlbXB0eSBzdHJpbmcuXG4gICAqL1xuICBwcm90ZWN0ZWQgaWdub3JlUnVsZShcbiAgICBpZ25vcmVzOiBOYWdQYWNrU3VwcHJlc3Npb25bXSxcbiAgICBydWxlSWQ6IHN0cmluZyxcbiAgICBmaW5kaW5nSWQ6IHN0cmluZ1xuICApOiBzdHJpbmcge1xuICAgIGZvciAobGV0IGlnbm9yZSBvZiBpZ25vcmVzKSB7XG4gICAgICBpZiAoXG4gICAgICAgIGlnbm9yZS5pZCAmJlxuICAgICAgICBpZ25vcmUucmVhc29uICYmXG4gICAgICAgIEpTT04uc3RyaW5naWZ5KGlnbm9yZS5yZWFzb24pLmxlbmd0aCA+PSAxMFxuICAgICAgKSB7XG4gICAgICAgIGlmIChydWxlSWQgPT09IGlnbm9yZS5pZCkge1xuICAgICAgICAgIGlmICghaWdub3JlLmFwcGxpZXNUbykge1xuICAgICAgICAgICAgLy8gdGhlIHJ1bGUgaXMgbm90IGdyYW51bGFyIHNvIGl0IGFsd2F5cyBhcHBsaWVzXG4gICAgICAgICAgICByZXR1cm4gaWdub3JlLnJlYXNvbjtcbiAgICAgICAgICB9XG4gICAgICAgICAgaWYgKGZpbmRpbmdJZCAmJiBpZ25vcmUuYXBwbGllc1RvLmluY2x1ZGVzKGZpbmRpbmdJZCkpIHtcbiAgICAgICAgICAgIC8vIHRoZSBydWxlIGlzIGdyYW51bGFyIHNvIHRoZSBmaW5kaW5nSWQgbXVzdCBtYXRjaFxuICAgICAgICAgICAgcmV0dXJuIGBbJHtmaW5kaW5nSWR9XSAke2lnbm9yZS5yZWFzb259YDtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRocm93IEVycm9yKFxuICAgICAgICAgIGBJbXByb3Blcmx5IGZvcm1hdHRlZCBjZGtfbmFnIHJ1bGUgc3VwcHJlc3Npb24gZGV0ZWN0ZWQ6ICR7SlNPTi5zdHJpbmdpZnkoXG4gICAgICAgICAgICBpZ25vcmVcbiAgICAgICAgICApfS4gU2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9jZGtsYWJzL2Nkay1uYWcjc3VwcHJlc3NpbmctYS1ydWxlIGZvciBpbmZvcm1hdGlvbiBvbiBzdXBwcmVzc2luZyBhIHJ1bGUuYFxuICAgICAgICApO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gJyc7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG1lc3NhZ2UgdG8gb3V0cHV0IHRvIHRoZSBjb25zb2xlIHdoZW4gYSBydWxlIGlzIHRyaWdnZXJlZC5cbiAgICogQHBhcmFtIHJ1bGVJZCBUaGUgaWQgb2YgdGhlIHJ1bGUuXG4gICAqIEBwYXJhbSBmaW5kaW5nSWQgVGhlIGlkIG9mIHRoZSBmaW5kaW5nLlxuICAgKiBAcGFyYW0gaW5mbyBXaHkgdGhlIHJ1bGUgd2FzIHRyaWdnZXJlZC5cbiAgICogQHBhcmFtIGV4cGxhbmF0aW9uIFdoeSB0aGUgcnVsZSBleGlzdHMuXG4gICAqIEByZXR1cm5zIFRoZSBmb3JtYXR0ZWQgbWVzc2FnZSBzdHJpbmcuXG4gICAqL1xuICBwcm90ZWN0ZWQgY3JlYXRlTWVzc2FnZShcbiAgICBydWxlSWQ6IHN0cmluZyxcbiAgICBmaW5kaW5nSWQ6IHN0cmluZyxcbiAgICBpbmZvOiBzdHJpbmcsXG4gICAgZXhwbGFuYXRpb246IHN0cmluZ1xuICApOiBzdHJpbmcge1xuICAgIGxldCBtZXNzYWdlID0gZmluZGluZ0lkXG4gICAgICA/IGAke3J1bGVJZH1bJHtmaW5kaW5nSWR9XTogJHtpbmZvfWBcbiAgICAgIDogYCR7cnVsZUlkfTogJHtpbmZvfWA7XG4gICAgcmV0dXJuIHRoaXMudmVyYm9zZSA/IGAke21lc3NhZ2V9ICR7ZXhwbGFuYXRpb259XFxuYCA6IGAke21lc3NhZ2V9XFxuYDtcbiAgfVxuXG4gIC8qKlxuICAgKiBXcml0ZSBhIGxpbmUgdG8gdGhlIHJ1bGUgcGFja3MgY29tcGxpYW5jZSByZXBvcnQgZm9yIHRoZSByZXNvdXJjZSdzIFN0YWNrXG4gICAqIEBwYXJhbSBwYXJhbXMgVGhlIEBJQXBwbHlSdWxlIGludGVyZmFjZSB3aXRoIHJ1bGUgZGV0YWlscy5cbiAgICogQHBhcmFtIHJ1bGVJZCBUaGUgaWQgb2YgdGhlIHJ1bGUuXG4gICAqIEBwYXJhbSBjb21wbGlhbmNlIFRoZSBjb21wbGlhbmNlIHN0YXR1cyBvZiB0aGUgcnVsZS5cbiAgICogQHBhcmFtIGV4cGxhbmF0aW9uIFRoZSBleHBsYW5hdGlvbiBmb3Igc3VwcHJlc3NlZCBydWxlcy5cbiAgICovXG4gIHByb3RlY3RlZCB3cml0ZVRvU3RhY2tDb21wbGlhbmNlUmVwb3J0KFxuICAgIHBhcmFtczogSUFwcGx5UnVsZSxcbiAgICBydWxlSWQ6IHN0cmluZyxcbiAgICBjb21wbGlhbmNlOlxuICAgICAgfCBOYWdSdWxlQ29tcGxpYW5jZS5DT01QTElBTlRcbiAgICAgIHwgTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVFxuICAgICAgfCAnVU5LTk9XTicsXG4gICAgZXhwbGFuYXRpb246IHN0cmluZyA9ICcnXG4gICk6IHZvaWQge1xuICAgIGNvbnN0IGxpbmUgPSB0aGlzLmNyZWF0ZUNvbXBsaWFuY2VSZXBvcnRMaW5lKFxuICAgICAgcGFyYW1zLFxuICAgICAgcnVsZUlkLFxuICAgICAgY29tcGxpYW5jZSxcbiAgICAgIGV4cGxhbmF0aW9uXG4gICAgKTtcbiAgICBsZXQgb3V0RGlyID0gQXBwLm9mKHBhcmFtcy5ub2RlKT8ub3V0ZGlyO1xuICAgIGNvbnN0IGZpbGVOYW1lID0gYCR7dGhpcy5wYWNrTmFtZX0tJHtwYXJhbXMubm9kZS5zdGFjay5zdGFja05hbWV9LU5hZ1JlcG9ydC5jc3ZgO1xuICAgIGNvbnN0IGZpbGVQYXRoID0gam9pbihvdXREaXIgPyBvdXREaXIgOiAnJywgZmlsZU5hbWUpO1xuICAgIGlmICghdGhpcy5yZXBvcnRTdGFja3MuaW5jbHVkZXMoZmlsZU5hbWUpKSB7XG4gICAgICB0aGlzLnJlcG9ydFN0YWNrcy5wdXNoKGZpbGVOYW1lKTtcbiAgICAgIHdyaXRlRmlsZVN5bmMoXG4gICAgICAgIGZpbGVQYXRoLFxuICAgICAgICAnUnVsZSBJRCxSZXNvdXJjZSBJRCxDb21wbGlhbmNlLEV4Y2VwdGlvbiBSZWFzb24sUnVsZSBMZXZlbCxSdWxlIEluZm9cXG4nXG4gICAgICApO1xuICAgIH1cbiAgICBhcHBlbmRGaWxlU3luYyhmaWxlUGF0aCwgbGluZSk7XG4gIH1cblxuICAvKipcbiAgICogSGVscGVyIGZ1bmN0aW9uIHRvIGNyZWF0ZSBhIGxpbmUgZm9yIHRoZSBjb21wbGlhbmNlIHJlcG9ydFxuICAgKiBAcGFyYW0gcGFyYW1zIFRoZSBASUFwcGx5UnVsZSBpbnRlcmZhY2Ugd2l0aCBydWxlIGRldGFpbHMuXG4gICAqIEBwYXJhbSBydWxlSWQgVGhlIGlkIG9mIHRoZSBydWxlLlxuICAgKiBAcGFyYW0gY29tcGxpYW5jZSBUaGUgY29tcGxpYW5jZSBzdGF0dXMgb2YgdGhlIHJ1bGUuXG4gICAqIEBwYXJhbSBleHBsYW5hdGlvbiBUaGUgZXhwbGFuYXRpb24gZm9yIHN1cHByZXNzZWQgcnVsZXMuXG4gICAqL1xuICBwcm90ZWN0ZWQgY3JlYXRlQ29tcGxpYW5jZVJlcG9ydExpbmUoXG4gICAgcGFyYW1zOiBJQXBwbHlSdWxlLFxuICAgIHJ1bGVJZDogc3RyaW5nLFxuICAgIGNvbXBsaWFuY2U6XG4gICAgICB8IE5hZ1J1bGVDb21wbGlhbmNlLkNPTVBMSUFOVFxuICAgICAgfCBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UXG4gICAgICB8ICdVTktOT1dOJyxcbiAgICBleHBsYW5hdGlvbjogc3RyaW5nID0gJydcbiAgKTogc3RyaW5nIHtcbiAgICAvL3wgUnVsZSBJRCB8IFJlc291cmNlIElEIHwgQ29tcGxpYW5jZSB8IEV4Y2VwdGlvbiBSZWFzb24gfCBSdWxlIExldmVsIHwgUnVsZSBJbmZvXG4gICAgY29uc3QgbGluZSA9IEFycmF5PHN0cmluZz4oKTtcbiAgICBsaW5lLnB1c2gocnVsZUlkKTtcbiAgICBsaW5lLnB1c2gocGFyYW1zLm5vZGUubm9kZS5wYXRoKTtcbiAgICBpZiAoXG4gICAgICAoY29tcGxpYW5jZSA9PT0gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVCB8fFxuICAgICAgICBjb21wbGlhbmNlID09PSAnVU5LTk9XTicpICYmXG4gICAgICBleHBsYW5hdGlvbiAhPT0gJydcbiAgICApIHtcbiAgICAgIGxpbmUucHVzaCgnU3VwcHJlc3NlZCcpO1xuICAgICAgbGluZS5wdXNoKGV4cGxhbmF0aW9uKTtcbiAgICB9IGVsc2Uge1xuICAgICAgbGluZS5wdXNoKGNvbXBsaWFuY2UpO1xuICAgICAgbGluZS5wdXNoKCdOL0EnKTtcbiAgICB9XG4gICAgbGluZS5wdXNoKHBhcmFtcy5sZXZlbCk7XG4gICAgbGluZS5wdXNoKHBhcmFtcy5pbmZvKTtcbiAgICByZXR1cm4gbGluZS5tYXAoKGkpID0+ICdcIicgKyBpLnJlcGxhY2UoL1wiL2csICdcIlwiJykgKyAnXCInKS5qb2luKCcsJykgKyAnXFxuJztcbiAgfVxuXG4gIHByaXZhdGUgaXNOb25Db21wbGlhbnQocnVsZVJlc3VsdDogTmFnUnVsZVJlc3VsdCkge1xuICAgIHJldHVybiAoXG4gICAgICBydWxlUmVzdWx0ID09PSBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UIHx8XG4gICAgICBBcnJheS5pc0FycmF5KHJ1bGVSZXN1bHQpXG4gICAgKTtcbiAgfVxuXG4gIHByaXZhdGUgYXNGaW5kaW5ncyhydWxlUmVzdWx0OiBOYWdSdWxlUmVzdWx0KTogTmFnUnVsZUZpbmRpbmdzIHtcbiAgICBpZiAoQXJyYXkuaXNBcnJheShydWxlUmVzdWx0KSkge1xuICAgICAgcmV0dXJuIHJ1bGVSZXN1bHQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBbJyddO1xuICAgIH1cbiAgfVxufVxuIl19