"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagSuppressions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
/**
 * Helper class with methods to add cdk-nag suppressions to cdk resources.
 *
 * @stability stable
 */
class NagSuppressions {
    /**
     * Add cdk-nag suppressions to the Stack.
     *
     * @param stack the Stack to apply the suppression to.
     * @param suppressions a list of suppressions to apply to the stack.
     * @stability stable
     */
    static addStackSuppressions(stack, suppressions) {
        var _b;
        const newSuppressions = [];
        for (const suppression of suppressions) {
            if (suppression.reason.length >= 10) {
                newSuppressions.push(suppression);
            }
            else {
                throw Error(`${stack.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        const currentSuppressions = (_b = stack.templateOptions.metadata) === null || _b === void 0 ? void 0 : _b.cdk_nag;
        if (Array.isArray(currentSuppressions === null || currentSuppressions === void 0 ? void 0 : currentSuppressions.rules_to_suppress)) {
            newSuppressions.unshift(...currentSuppressions.rules_to_suppress);
        }
        if (stack.templateOptions.metadata) {
            stack.templateOptions.metadata.cdk_nag = {
                rules_to_suppress: newSuppressions,
            };
        }
        else {
            stack.templateOptions.metadata = {
                cdk_nag: { rules_to_suppress: newSuppressions },
            };
        }
    }
    /**
     * Add cdk-nag suppressions to the construct if it is a CfnResource.
     *
     * @param construct the IConstruct to apply the suppression to.
     * @param suppressions a list of suppressions to apply to the resource.
     * @param applyToChildren apply the suppressions to this construct and all of its children if they exist (default:false).
     * @stability stable
     */
    static addResourceSuppressions(construct, suppressions, applyToChildren = false) {
        const newSuppressions = [];
        for (const suppression of suppressions) {
            if (suppression.reason.length >= 10) {
                newSuppressions.push(suppression);
            }
            else {
                throw Error(`${construct.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        const constructs = applyToChildren ? construct.node.findAll() : [construct];
        for (const child of constructs) {
            const possibleL1 = child.node.defaultChild
                ? child.node.defaultChild
                : child;
            if (possibleL1 instanceof monocdk_1.CfnResource) {
                const resource = possibleL1;
                const currentSuppressions = resource.getMetadata('cdk_nag');
                if (Array.isArray(currentSuppressions === null || currentSuppressions === void 0 ? void 0 : currentSuppressions.rules_to_suppress)) {
                    newSuppressions.unshift(...currentSuppressions.rules_to_suppress);
                }
                resource.addMetadata('cdk_nag', {
                    rules_to_suppress: newSuppressions,
                });
            }
        }
    }
    /**
     * Locate a construct by it's path and add cdk-nag suppressions if it both exists and is a CfnResource.
     *
     * @param stack the Stack the construct belongs to.
     * @param path the path of the construct in the provided stack.
     * @param suppressions a list of suppressions to apply to the resource.
     * @param applyToChildren apply the suppressions to this construct and all of its children if they exist (default:false).
     * @stability stable
     */
    static addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren = false) {
        for (const child of stack.node.findAll()) {
            const fixedPath = path.replace(/^\//, '');
            if (child.node.path === fixedPath ||
                child.node.path + '/Resource' === fixedPath) {
                NagSuppressions.addResourceSuppressions(child, suppressions, applyToChildren);
            }
        }
    }
}
exports.NagSuppressions = NagSuppressions;
_a = JSII_RTTI_SYMBOL_1;
NagSuppressions[_a] = { fqn: "monocdk-nag.NagSuppressions", version: "0.1.26" };
//# sourceMappingURL=data:application/json;base64,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