"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const aws_ec2_1 = require("monocdk/aws-ec2");
const nag_pack_1 = require("../../../nag-pack");
/**
 * VPCs have Flow Logs enabled - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SI-4(17), SI-7(8))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnVPC) {
        const vpcLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnFlowLog) {
                if (isMatchingCompliantFlowLog(child, vpcLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Flow Log is compliant and associated with the given VPC
 * @param node the CfnFlowLog to check
 * returns whether the CfnFlowLog is compliant
 */
function isMatchingCompliantFlowLog(node, vpcLogicalId) {
    const resourceLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.resourceId);
    if (node.resourceType === 'VPC' && resourceLogicalId === vpcLogicalId) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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