"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const monocdk_1 = require("monocdk");
const aws_secretsmanager_1 = require("monocdk/aws-secretsmanager");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Secrets have automatic rotation scheduled - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_secretsmanager_1.CfnSecret) {
        const secretLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_secretsmanager_1.CfnRotationSchedule) {
                if (isMatchingRotationSchedule(child, secretLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Rotation Schedule is associated with the given Secret
 * @param node the CfnRotationSchedule to check
 * @param secretLogicalId the Cfn Logical ID of the Secret
 * returns whether the CfnRotationSchedule is associates with the given Secret
 */
function isMatchingRotationSchedule(node, secretLogicalId) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId) {
        if (monocdk_1.Stack.of(node).resolve(node.hostedRotationLambda) !== undefined) {
            return true;
        }
        const rotationRules = monocdk_1.Stack.of(node).resolve(node.rotationRules);
        if (rotationRules !== undefined) {
            const automaticallyAfterDays = monocdk_1.Stack.of(node).resolve(rotationRules.automaticallyAfterDays);
            if (automaticallyAfterDays !== undefined) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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