"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const aws_redshift_1 = require("monocdk/aws-redshift");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Redshift clusters require TLS/SSL encryption - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_redshift_1.CfnCluster) {
        const clusterParameterGroupName = nag_pack_1.resolveResourceFromInstrinsic(node, node.clusterParameterGroupName);
        if (clusterParameterGroupName === undefined) {
            return false;
        }
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_redshift_1.CfnClusterParameterGroup) {
                if (isMatchingParameterGroup(child, clusterParameterGroupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Parameter Group enforces SSL and is associated with the given Redshift cluster
 * @param node the CfnClusterParameterGroup to check
 * @param parameterGroupName The name of the associated parameter group
 * returns whether the CfnClusterParameterGroup enforces SSL and is associated given Redshift cluster
 */
function isMatchingParameterGroup(node, parameterGroupName) {
    const parameterGroupLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
    if (parameterGroupName !== parameterGroupLogicalId ||
        node.parameters == undefined) {
        return false;
    }
    const parameters = monocdk_1.Stack.of(node).resolve(node.parameters);
    for (const parameter of parameters) {
        const resolvedParameter = monocdk_1.Stack.of(node).resolve(parameter);
        if (resolvedParameter.parameterName.toLowerCase() == 'require_ssl' &&
            resolvedParameter.parameterValue.toLowerCase() == 'true') {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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