"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const aws_redshift_1 = require("monocdk/aws-redshift");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Redshift clusters require TLS/SSL encryption - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_redshift_1.CfnCluster) {
        const clusterParameterGroupName = nag_pack_1.resolveResourceFromInstrinsic(node, node.clusterParameterGroupName);
        if (clusterParameterGroupName === undefined) {
            return false;
        }
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_redshift_1.CfnClusterParameterGroup) {
                if (isMatchingParameterGroup(child, clusterParameterGroupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Parameter Group enforces SSL and is associated with the given Redshift cluster
 * @param node the CfnClusterParameterGroup to check
 * @param parameterGroupName The name of the associated parameter group
 * returns whether the CfnClusterParameterGroup enforces SSL and is associated given Redshift cluster
 */
function isMatchingParameterGroup(node, parameterGroupName) {
    const parameterGroupLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
    if (parameterGroupName !== parameterGroupLogicalId ||
        node.parameters == undefined) {
        return false;
    }
    const parameters = monocdk_1.Stack.of(node).resolve(node.parameters);
    for (const parameter of parameters) {
        const resolvedParameter = monocdk_1.Stack.of(node).resolve(parameter);
        if (resolvedParameter.parameterName.toLowerCase() == 'require_ssl' &&
            resolvedParameter.parameterValue.toLowerCase() == 'true') {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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