"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const aws_iam_1 = require("monocdk/aws-iam");
const nag_pack_1 = require("../../../nag-pack");
/**
 * IAM Groups have at least one IAM User - (Control ID: AC-2(j))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_iam_1.CfnGroup) {
        const groupLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const groupName = monocdk_1.Stack.of(node).resolve(node.groupName);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_iam_1.CfnUser) {
                if (isMatchingUser(child, groupLogicalId, groupName)) {
                    found = true;
                    break;
                }
            }
            else if (child instanceof aws_iam_1.CfnUserToGroupAddition) {
                if (isMatchingGroupAddition(child, groupLogicalId, groupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the IAM User belongs to the IAM Group
 * @param node the CfnUser to check
 * @param groupLogicalId the Cfn Logical ID of the group
 * @param groupName the name of the group
 * returns whether the CfnUser is in the given group
 */
function isMatchingUser(node, groupLogicalId, groupName) {
    const groups = monocdk_1.Stack.of(node).resolve(node.groups);
    if (Array.isArray(groups)) {
        for (const group of groups) {
            const resolvedGroup = JSON.stringify(monocdk_1.Stack.of(node).resolve(group));
            if (new RegExp(`${groupLogicalId}(?![\\w])`).test(resolvedGroup) ||
                (groupName != undefined &&
                    new RegExp(`${groupName}(?![\\w\\-_\\.])`).test(resolvedGroup))) {
                return true;
            }
        }
    }
    return false;
}
/**
 * Helper function to check whether the User to Group Addition mentions the specified Group
 * @param node the CfnUserToGroupAddition to check
 * @param groupLogicalId the Cfn Logical ID of the group
 * @param groupName the name of the group
 * returns whether the CfnUserToGroupAddition references the given group
 */
function isMatchingGroupAddition(node, groupLogicalId, groupName) {
    const resolvedGroup = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.groupName));
    if (new RegExp(`${groupLogicalId}(?![\\w])`).test(resolvedGroup) ||
        (groupName != undefined &&
            new RegExp(`${groupName}(?![\\w\\-_\\.])`).test(resolvedGroup))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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