"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const aws_backup_1 = require("monocdk/aws-backup");
const aws_dynamodb_1 = require("monocdk/aws-dynamodb");
const nag_pack_1 = require("../../../nag-pack");
/**
 * DynamoDB tables are part of AWS Backup plan(s) - (Control IDs: CP-9(b), CP-10, SI-12)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_dynamodb_1.CfnTable) {
        const tableLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const tableName = monocdk_1.Stack.of(node).resolve(node.tableName);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_backup_1.CfnBackupSelection) {
                if (isMatchingSelection(child, tableLogicalId, tableName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Backup Plan Selection contains the given Table
 * @param node the CfnBackupSelection to check
 * @param tableLogicalId the Cfn Logical ID of the table
 * @param tableName the name of the table
 * returns whether the CfnBackupSelection contains the given Table
 */
function isMatchingSelection(node, tableLogicalId, tableName) {
    const backupSelection = monocdk_1.Stack.of(node).resolve(node.backupSelection);
    const resources = monocdk_1.Stack.of(node).resolve(backupSelection.resources);
    if (Array.isArray(resources)) {
        for (const resource of resources) {
            const resolvedResource = JSON.stringify(monocdk_1.Stack.of(node).resolve(resource));
            if (new RegExp(`${tableLogicalId}(?![\\w])`).test(resolvedResource) ||
                (tableName != undefined &&
                    new RegExp(`table\/${tableName}(?![\\w\\-_\\.])`).test(resolvedResource))) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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