"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const nag_pack_1 = require("../../../nag-pack");
/**
 * WAFv2 web ACLs have logging enabled - (Control ID: 164.312(b))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_wafv2_1.CfnWebACL) {
        const webAclLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const webAclName = monocdk_1.Stack.of(node).resolve(node.name);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnLoggingConfiguration) {
                if (isMatchingLoggingConfiguration(child, webAclLogicalId, webAclName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Logging Configuration contains the given Web ACL
 * @param node the CfnLoggingConfiguration to check
 * @param webAclLogicalId the Cfn Logical ID of the Web ACL
 * @param webAclName the name of the Web ACL
 * returns whether the CfnLoggingConfiguration contains the given Web ACL
 */
function isMatchingLoggingConfiguration(node, webAclLogicalId, webAclName) {
    const resourceArn = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    if (new RegExp(`${webAclLogicalId}(?![\\w])`).test(resourceArn) ||
        (webAclName != undefined &&
            new RegExp(`webacl\/${webAclName}(?![\\w\\-_\\.])`).test(resourceArn))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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