"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const aws_ec2_1 = require("monocdk/aws-ec2");
const nag_pack_1 = require("../../../nag-pack");
/**
 * VPCs have Flow Logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnVPC) {
        const vpcLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnFlowLog) {
                if (isMatchingCompliantFlowLog(child, vpcLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Flow Log is compliant and associated with the given VPC
 * @param node the CfnFlowLog to check
 * returns whether the CfnFlowLog is compliant
 */
function isMatchingCompliantFlowLog(node, vpcLogicalId) {
    const resourceLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.resourceId);
    if (node.resourceType === 'VPC' && resourceLogicalId === vpcLogicalId) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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