"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const monocdk_1 = require("monocdk");
const aws_secretsmanager_1 = require("monocdk/aws-secretsmanager");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Secrets have automatic rotation scheduled - (Control ID: 164.308(a)(4)(ii)(B))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_secretsmanager_1.CfnSecret) {
        const secretLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_secretsmanager_1.CfnRotationSchedule) {
                if (isMatchingRotationSchedule(child, secretLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Rotation Schedule is associated with the given Secret
 * @param node the CfnRotationSchedule to check
 * @param secretLogicalId the Cfn Logical ID of the Secret
 * returns whether the CfnRotationSchedule is associates with the given Secret
 */
function isMatchingRotationSchedule(node, secretLogicalId) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId) {
        if (monocdk_1.Stack.of(node).resolve(node.hostedRotationLambda) !== undefined) {
            return true;
        }
        const rotationRules = monocdk_1.Stack.of(node).resolve(node.rotationRules);
        if (rotationRules !== undefined) {
            const automaticallyAfterDays = monocdk_1.Stack.of(node).resolve(rotationRules.automaticallyAfterDays);
            if (automaticallyAfterDays !== undefined) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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