"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const aws_apigateway_1 = require("monocdk/aws-apigateway");
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Rest API stages are associated with AWS WAFv2 web ACLs
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_apigateway_1.CfnStage) {
        const stageLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const stageName = nag_pack_1.resolveResourceFromInstrinsic(node, node.stageName);
        const restApiId = nag_pack_1.resolveResourceFromInstrinsic(node, node.restApiId);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                if (isMatchingWebACLAssociation(child, stageLogicalId, stageName, restApiId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Rest API
 * @param node the CfnWebACLAssociation to check
 * @param stageLogicalId the Cfn Logical ID of the REST API Stage
 * @param stageName the name of the REST API Stage
 * @param restApiId the ID or Cfn Resource ID of the REST API associated with the Stage
 * returns whether the CfnWebACLAssociation is associates with the given Rest API
 */
function isMatchingWebACLAssociation(node, stageLogicalId, stageName, restApiId) {
    const resourceLogicalId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    const regexes = Array();
    regexes.push(`${restApiId}.+${stageLogicalId}(?![\\w])`);
    if (stageName !== undefined) {
        regexes.push(`${restApiId}.+${stageName}(?![\\w])`);
    }
    const regex = new RegExp(regexes.join('|'), 'gm');
    if (regex.test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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