"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const aws_redshift_1 = require("monocdk/aws-redshift");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Redshift clusters have user user activity logging enabled
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_redshift_1.CfnCluster) {
        const clusterParameterGroupName = nag_pack_1.resolveResourceFromInstrinsic(node, node.clusterParameterGroupName);
        if (clusterParameterGroupName === undefined) {
            return false;
        }
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_redshift_1.CfnClusterParameterGroup) {
                const childParameterGroupName = nag_pack_1.resolveResourceFromInstrinsic(node, child.ref);
                if (childParameterGroupName === clusterParameterGroupName) {
                    found = isCompliantClusterParameterGroup(child);
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given cluster parameter group is compliant
 * @param node the CfnClusterParameterGroup to check
 * returns whether the Cluster Parameter group is compliant
 */
function isCompliantClusterParameterGroup(node) {
    const resolvedParameters = monocdk_1.Stack.of(node).resolve(node.parameters);
    if (resolvedParameters == undefined) {
        return false;
    }
    for (const parameter of resolvedParameters) {
        const resolvedParam = monocdk_1.Stack.of(node).resolve(parameter);
        if (resolvedParam.parameterName === 'enable_user_activity_logging' &&
            resolvedParam.parameterValue === 'true') {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiUlMxMS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uL3NyYy9Bd3NTb2x1dGlvbnMvcnVsZXMvZGF0YWJhc2VzL1JTMTEudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQTs7O0VBR0U7QUFDRixxQ0FBNkM7QUFDN0MsdURBQTRFO0FBQzVFLGdEQUFrRTtBQUVsRTs7O0dBR0c7QUFDSCxtQkFBeUIsSUFBaUI7SUFDeEMsSUFBSSxJQUFJLFlBQVkseUJBQVUsRUFBRTtRQUM5QixNQUFNLHlCQUF5QixHQUFHLHdDQUE2QixDQUM3RCxJQUFJLEVBQ0osSUFBSSxDQUFDLHlCQUF5QixDQUMvQixDQUFDO1FBQ0YsSUFBSSx5QkFBeUIsS0FBSyxTQUFTLEVBQUU7WUFDM0MsT0FBTyxLQUFLLENBQUM7U0FDZDtRQUNELElBQUksS0FBSyxHQUFHLEtBQUssQ0FBQztRQUNsQixLQUFLLE1BQU0sS0FBSyxJQUFJLGVBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ2pELElBQUksS0FBSyxZQUFZLHVDQUF3QixFQUFFO2dCQUM3QyxNQUFNLHVCQUF1QixHQUFHLHdDQUE2QixDQUMzRCxJQUFJLEVBQ0osS0FBSyxDQUFDLEdBQUcsQ0FDVixDQUFDO2dCQUNGLElBQUksdUJBQXVCLEtBQUsseUJBQXlCLEVBQUU7b0JBQ3pELEtBQUssR0FBRyxnQ0FBZ0MsQ0FBQyxLQUFLLENBQUMsQ0FBQztvQkFDaEQsTUFBTTtpQkFDUDthQUNGO1NBQ0Y7UUFDRCxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ1YsT0FBTyxLQUFLLENBQUM7U0FDZDtLQUNGO0lBQ0QsT0FBTyxJQUFJLENBQUM7QUFDZCxDQUFDO0FBM0JELDRCQTJCQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLGdDQUFnQyxDQUN2QyxJQUE4QjtJQUU5QixNQUFNLGtCQUFrQixHQUFHLGVBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNuRSxJQUFJLGtCQUFrQixJQUFJLFNBQVMsRUFBRTtRQUNuQyxPQUFPLEtBQUssQ0FBQztLQUNkO0lBQ0QsS0FBSyxNQUFNLFNBQVMsSUFBSSxrQkFBa0IsRUFBRTtRQUMxQyxNQUFNLGFBQWEsR0FBRyxlQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN4RCxJQUNFLGFBQWEsQ0FBQyxhQUFhLEtBQUssOEJBQThCO1lBQzlELGFBQWEsQ0FBQyxjQUFjLEtBQUssTUFBTSxFQUN2QztZQUNBLE9BQU8sSUFBSSxDQUFDO1NBQ2I7S0FDRjtJQUNELE9BQU8sS0FBSyxDQUFDO0FBQ2YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuaW1wb3J0IHsgQ2ZuUmVzb3VyY2UsIFN0YWNrIH0gZnJvbSAnbW9ub2Nkayc7XG5pbXBvcnQgeyBDZm5DbHVzdGVyLCBDZm5DbHVzdGVyUGFyYW1ldGVyR3JvdXAgfSBmcm9tICdtb25vY2RrL2F3cy1yZWRzaGlmdCc7XG5pbXBvcnQgeyByZXNvbHZlUmVzb3VyY2VGcm9tSW5zdHJpbnNpYyB9IGZyb20gJy4uLy4uLy4uL25hZy1wYWNrJztcblxuLyoqXG4gKiBSZWRzaGlmdCBjbHVzdGVycyBoYXZlIHVzZXIgdXNlciBhY3Rpdml0eSBsb2dnaW5nIGVuYWJsZWRcbiAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICovXG5leHBvcnQgZGVmYXVsdCBmdW5jdGlvbiAobm9kZTogQ2ZuUmVzb3VyY2UpOiBib29sZWFuIHtcbiAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5DbHVzdGVyKSB7XG4gICAgY29uc3QgY2x1c3RlclBhcmFtZXRlckdyb3VwTmFtZSA9IHJlc29sdmVSZXNvdXJjZUZyb21JbnN0cmluc2ljKFxuICAgICAgbm9kZSxcbiAgICAgIG5vZGUuY2x1c3RlclBhcmFtZXRlckdyb3VwTmFtZSxcbiAgICApO1xuICAgIGlmIChjbHVzdGVyUGFyYW1ldGVyR3JvdXBOYW1lID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG4gICAgbGV0IGZvdW5kID0gZmFsc2U7XG4gICAgZm9yIChjb25zdCBjaGlsZCBvZiBTdGFjay5vZihub2RlKS5ub2RlLmZpbmRBbGwoKSkge1xuICAgICAgaWYgKGNoaWxkIGluc3RhbmNlb2YgQ2ZuQ2x1c3RlclBhcmFtZXRlckdyb3VwKSB7XG4gICAgICAgIGNvbnN0IGNoaWxkUGFyYW1ldGVyR3JvdXBOYW1lID0gcmVzb2x2ZVJlc291cmNlRnJvbUluc3RyaW5zaWMoXG4gICAgICAgICAgbm9kZSxcbiAgICAgICAgICBjaGlsZC5yZWYsXG4gICAgICAgICk7XG4gICAgICAgIGlmIChjaGlsZFBhcmFtZXRlckdyb3VwTmFtZSA9PT0gY2x1c3RlclBhcmFtZXRlckdyb3VwTmFtZSkge1xuICAgICAgICAgIGZvdW5kID0gaXNDb21wbGlhbnRDbHVzdGVyUGFyYW1ldGVyR3JvdXAoY2hpbGQpO1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIGlmICghZm91bmQpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG4gIH1cbiAgcmV0dXJuIHRydWU7XG59XG5cbi8qKlxuICogSGVscGVyIGZ1bmN0aW9uIHRvIGNoZWNrIHdoZXRoZXIgYSBnaXZlbiBjbHVzdGVyIHBhcmFtZXRlciBncm91cCBpcyBjb21wbGlhbnRcbiAqIEBwYXJhbSBub2RlIHRoZSBDZm5DbHVzdGVyUGFyYW1ldGVyR3JvdXAgdG8gY2hlY2tcbiAqIHJldHVybnMgd2hldGhlciB0aGUgQ2x1c3RlciBQYXJhbWV0ZXIgZ3JvdXAgaXMgY29tcGxpYW50XG4gKi9cbmZ1bmN0aW9uIGlzQ29tcGxpYW50Q2x1c3RlclBhcmFtZXRlckdyb3VwKFxuICBub2RlOiBDZm5DbHVzdGVyUGFyYW1ldGVyR3JvdXAsXG4pOiBib29sZWFuIHtcbiAgY29uc3QgcmVzb2x2ZWRQYXJhbWV0ZXJzID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShub2RlLnBhcmFtZXRlcnMpO1xuICBpZiAocmVzb2x2ZWRQYXJhbWV0ZXJzID09IHVuZGVmaW5lZCkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuICBmb3IgKGNvbnN0IHBhcmFtZXRlciBvZiByZXNvbHZlZFBhcmFtZXRlcnMpIHtcbiAgICBjb25zdCByZXNvbHZlZFBhcmFtID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShwYXJhbWV0ZXIpO1xuICAgIGlmIChcbiAgICAgIHJlc29sdmVkUGFyYW0ucGFyYW1ldGVyTmFtZSA9PT0gJ2VuYWJsZV91c2VyX2FjdGl2aXR5X2xvZ2dpbmcnICYmXG4gICAgICByZXNvbHZlZFBhcmFtLnBhcmFtZXRlclZhbHVlID09PSAndHJ1ZSdcbiAgICApIHtcbiAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cbiAgfVxuICByZXR1cm4gZmFsc2U7XG59XG4iXX0=