"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.applyAdditionalShared = exports.applySharedFunction = exports.sharePackages = exports.getNpmPackageSharedConfig = exports.shareWorkspaceLibraries = void 0;
const path_1 = require("path");
const package_json_1 = require("./package-json");
const typescript_1 = require("./typescript");
const secondary_entry_points_1 = require("./secondary-entry-points");
const workspace_root_1 = require("nx/src/utils/workspace-root");
const logger_1 = require("nx/src/utils/logger");
const typescript_2 = require("nx/src/utils/typescript");
/**
 * Build an object of functions to be used with the ModuleFederationPlugin to
 * share Nx Workspace Libraries between Hosts and Remotes.
 *
 * @param libraries - The Nx Workspace Libraries to share
 * @param tsConfigPath - The path to TS Config File that contains the Path Mappings for the Libraries
 */
function shareWorkspaceLibraries(libraries, tsConfigPath) {
    var _a;
    if (tsConfigPath === void 0) { tsConfigPath = (_a = process.env.NX_TSCONFIG_PATH) !== null && _a !== void 0 ? _a : (0, typescript_2.getRootTsConfigPath)(); }
    if (!libraries) {
        return getEmptySharedLibrariesConfig();
    }
    const tsconfigPathAliases = (0, typescript_1.readTsPathMappings)(tsConfigPath);
    if (!Object.keys(tsconfigPathAliases).length) {
        return getEmptySharedLibrariesConfig();
    }
    const pathMappings = [];
    for (const [key, paths] of Object.entries(tsconfigPathAliases)) {
        const library = libraries.find((lib) => lib.importKey === key);
        if (!library) {
            continue;
        }
        // This is for Angular Projects that use ng-package.json
        // It will do nothing for React Projects
        (0, secondary_entry_points_1.collectWorkspaceLibrarySecondaryEntryPoints)(library, tsconfigPathAliases).forEach(({ name, path }) => pathMappings.push({
            name,
            path,
        }));
        pathMappings.push({
            name: key,
            path: (0, path_1.normalize)((0, path_1.join)(workspace_root_1.workspaceRoot, paths[0])),
        });
    }
    const webpack = require('webpack');
    return {
        getAliases: () => pathMappings.reduce((aliases, library) => (Object.assign(Object.assign({}, aliases), { [library.name]: library.path })), {}),
        getLibraries: (eager) => pathMappings.reduce((libraries, library) => (Object.assign(Object.assign({}, libraries), { [library.name]: { requiredVersion: false, eager } })), {}),
        getReplacementPlugin: () => new webpack.NormalModuleReplacementPlugin(/./, (req) => {
            if (!req.request.startsWith('.')) {
                return;
            }
            const from = req.context;
            const to = (0, path_1.normalize)((0, path_1.join)(req.context, req.request));
            for (const library of pathMappings) {
                const libFolder = (0, path_1.normalize)((0, path_1.dirname)(library.path));
                if (!from.startsWith(libFolder) && to.startsWith(libFolder)) {
                    req.request = library.name;
                }
            }
        }),
    };
}
exports.shareWorkspaceLibraries = shareWorkspaceLibraries;
/**
 * Build the Module Federation Share Config for a specific package and the
 * specified version of that package.
 * @param pkgName - Name of the package to share
 * @param version - Version of the package to require by other apps in the Module Federation setup
 */
function getNpmPackageSharedConfig(pkgName, version) {
    if (!version) {
        logger_1.logger.warn(`Could not find a version for "${pkgName}" in the root "package.json" ` +
            'when collecting shared packages for the Module Federation setup. ' +
            'The package will not be shared.');
        return undefined;
    }
    return { singleton: true, strictVersion: true, requiredVersion: version };
}
exports.getNpmPackageSharedConfig = getNpmPackageSharedConfig;
/**
 * Create a dictionary of packages and their Module Federation Shared Config
 * from an array of package names.
 *
 * Lookup the versions of the packages from the root package.json file in the
 * workspace.
 * @param packages - Array of package names as strings
 */
function sharePackages(packages) {
    const pkgJson = (0, package_json_1.readRootPackageJson)();
    const allPackages = [];
    packages.forEach((pkg) => {
        var _a, _b, _c;
        const pkgVersion = (_b = (_a = pkgJson.dependencies) === null || _a === void 0 ? void 0 : _a[pkg]) !== null && _b !== void 0 ? _b : (_c = pkgJson.devDependencies) === null || _c === void 0 ? void 0 : _c[pkg];
        allPackages.push({ name: pkg, version: pkgVersion });
        (0, secondary_entry_points_1.collectPackageSecondaryEntryPoints)(pkg, pkgVersion, allPackages);
    });
    return allPackages.reduce((shared, pkg) => {
        const config = getNpmPackageSharedConfig(pkg.name, pkg.version);
        if (config) {
            shared[pkg.name] = config;
        }
        return shared;
    }, {});
}
exports.sharePackages = sharePackages;
/**
 * Apply a custom function provided by the user that will modify the Shared Config
 * of the dependencies for the Module Federation build.
 *
 * @param sharedConfig - The original Shared Config to be modified
 * @param sharedFn - The custom function to run
 */
function applySharedFunction(sharedConfig, sharedFn) {
    if (!sharedFn) {
        return;
    }
    for (const [libraryName, library] of Object.entries(sharedConfig)) {
        const mappedDependency = sharedFn(libraryName, library);
        if (mappedDependency === false) {
            delete sharedConfig[libraryName];
            continue;
        }
        else if (!mappedDependency) {
            continue;
        }
        sharedConfig[libraryName] = mappedDependency;
    }
}
exports.applySharedFunction = applySharedFunction;
/**
 * Add additional dependencies to the shared package that may not have been
 * discovered by the project graph.
 *
 * This can be useful for applications that use a Dependency Injection system
 * that expects certain Singleton values to be present in the shared injection
 * hierarchy.
 *
 * @param sharedConfig - The original Shared Config
 * @param additionalShared - The additional dependencies to add
 * @param projectGraph - The Nx project graph
 */
function applyAdditionalShared(sharedConfig, additionalShared, projectGraph) {
    if (!additionalShared) {
        return;
    }
    for (const shared of additionalShared) {
        if (typeof shared === 'string') {
            addStringDependencyToSharedConfig(sharedConfig, shared, projectGraph);
        }
        else if (Array.isArray(shared)) {
            sharedConfig[shared[0]] = shared[1];
        }
        else if (typeof shared === 'object') {
            sharedConfig[shared.libraryName] = shared.sharedConfig;
        }
    }
}
exports.applyAdditionalShared = applyAdditionalShared;
function addStringDependencyToSharedConfig(sharedConfig, dependency, projectGraph) {
    var _a, _b, _c, _d;
    if (projectGraph.nodes[dependency]) {
        sharedConfig[dependency] = { requiredVersion: false };
    }
    else if ((_a = projectGraph.externalNodes) === null || _a === void 0 ? void 0 : _a[`npm:${dependency}`]) {
        const pkgJson = (0, package_json_1.readRootPackageJson)();
        const config = getNpmPackageSharedConfig(dependency, (_c = (_b = pkgJson.dependencies) === null || _b === void 0 ? void 0 : _b[dependency]) !== null && _c !== void 0 ? _c : (_d = pkgJson.devDependencies) === null || _d === void 0 ? void 0 : _d[dependency]);
        if (!config) {
            return;
        }
        sharedConfig[dependency] = config;
    }
    else {
        throw new Error(`The specified dependency "${dependency}" in the additionalShared configuration does not exist in the project graph. ` +
            `Please check your additionalShared configuration and make sure you are including valid workspace projects or npm packages.`);
    }
}
function getEmptySharedLibrariesConfig() {
    const webpack = require('webpack');
    return {
        getAliases: () => ({}),
        getLibraries: () => ({}),
        getReplacementPlugin: () => new webpack.NormalModuleReplacementPlugin(/./, () => { }),
    };
}
//# sourceMappingURL=share.js.map