"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts"],
            },
        });
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2");
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec("npx npm-check-updates --deep --rejectVersion 0.0.0 -u");
            upgradeDepsTask.exec("npx syncpack fix-mismatches");
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec("npx projen");
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        this.nxJson = new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: options.nxConfig?.cacheableOperations || [
                                "build",
                                "test",
                            ],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                namedInputs: options.nxConfig?.namedInputs,
                targetDefaults: options.nxConfig?.targetDefaults,
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: this.nxConfig?.affectedBranch || "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.instantiationOrderSubProjects.map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    // Remove this hack once subProjects is made public in Projen
    get instantiationOrderSubProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects;
    }
    get subProjects() {
        return this.instantiationOrderSubProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.wirePythonDependencies();
        this.synthesizeNonNodePackageJson();
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        const subProjectPackages = [];
        this.subProjects.forEach((subProject) => {
            if (isNodeProject(subProject)) {
                const subNodeProject = subProject;
                subProjectPackages.push(subNodeProject.package);
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => { };
            }
        });
        super.synth();
        // Force workspace install deps if any node subproject package has change, unless the workspace changed
        if (
        // @ts-ignore - `file` is private
        this.package.file.changed !== true &&
            // @ts-ignore - `file` is private
            subProjectPackages.find((pkg) => pkg.file.changed === true)) {
            // @ts-ignore - `installDependencies` is private
            this.package.installDependencies();
        }
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {
                name: subProject.name,
                private: true,
                __pdk__: true,
                scripts: subProject.tasks.all.reduce((p, c) => ({
                    [c.name]: `npx projen ${c.name}`,
                    ...p,
                }), {}),
                version: "0.0.0",
            };
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.disableNoHoistBundled !== true) {
            const noHoistBundled = this.subProjects.flatMap((sub) => {
                if (sub instanceof javascript_1.NodeProject) {
                    return sub.deps.all
                        .filter((dep) => dep.type === projen_1.DependencyType.BUNDLED)
                        .flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.workspacePackages,
                nohoist: noHoist,
            });
        }
    }
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    wirePythonDependencies() {
        // Find any python projects
        const pythonProjects = this.subProjects.filter((project) => project instanceof python_1.PythonProject);
        if (pythonProjects.length === 0) {
            // Nothing to do for no python projects
            return;
        }
        // Move all install commands to install-py so that they are not installed in parallel by the monorepo package manager.
        // eg yarn install will run the install task for all packages in parallel which can lead to conflicts for python.
        pythonProjects.forEach((pythonProject) => {
            const installPyTask = pythonProject.tasks.tryFind("install-py") ??
                pythonProject.addTask("install-py");
            const installTask = pythonProject.tasks.tryFind("install");
            (installTask?.steps || []).forEach((step) => {
                this.copyStepIntoTask(step, installPyTask, pythonProject);
            });
            installTask?.reset();
        });
        // Add an install task to the monorepo to include running the install-py command serially to avoid conflicting writes
        // to a shared virtual env. This is also managed by nx so that installs occur in dependency order.
        const monorepoInstallTask = this.tasks.tryFind("install") ?? this.addTask("install");
        monorepoInstallTask.exec(`npx nx run-many --target install-py --projects ${pythonProjects
            .map((project) => project.name)
            .join(",")} --parallel=1`);
        // Update the nx.json to ensure that install-py follows dependency order
        this.nxJson.addOverride("targetDependencies.install-py", [
            {
                target: "install-py",
                projects: "dependencies",
            },
        ]);
    }
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    copyStepIntoTask(step, task, project) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = project.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject", version: "0.13.6" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibngtbW9ub3JlcG8uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvbngtbW9ub3JlcG8udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTtzQ0FDc0M7QUFDdEMseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3QixtQ0FTZ0I7QUFDaEIsc0RBSStCO0FBQy9CLDhDQUFrRDtBQUNsRCxzREFHK0I7QUFFL0IseURBQW9FO0FBRXBFLE1BQU0sdUJBQXVCLEdBQVcsbUNBQW1DLENBQUM7QUFtRjVFOzs7OztHQUtHO0FBQ0gsTUFBYSxpQkFBa0IsU0FBUSw4QkFBaUI7SUFXdEQsWUFBWSxPQUFpQztRQUMzQyxLQUFLLENBQUM7WUFDSixHQUFHLE9BQU87WUFDVixNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sSUFBSSxLQUFLO1lBQy9CLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTyxJQUFJLEtBQUs7WUFDakMsUUFBUSxFQUFFLE9BQU8sQ0FBQyxRQUFRLElBQUksSUFBSTtZQUNsQyxVQUFVLEVBQUUsSUFBSTtZQUNoQixPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU8sSUFBSSxLQUFLO1lBQ2pDLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSSxJQUFJLEtBQUs7WUFDM0Isb0JBQW9CLEVBQUUsT0FBTyxDQUFDLG9CQUFvQixJQUFJLFVBQVU7WUFDaEUsVUFBVSxFQUFFLEtBQUs7WUFDakIsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhLElBQUk7Z0JBQ3RDLElBQUksRUFBRSxDQUFDLEdBQUcsQ0FBQztnQkFDWCxjQUFjLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQzthQUNwQztZQUNELFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUSxJQUFJO2dCQUM1QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFLEdBQUc7aUJBQ2I7Z0JBQ0QsT0FBTyxFQUFFLENBQUMsU0FBUyxDQUFDO2FBQ3JCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFFBQVEsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDO1FBQ2pDLElBQUksQ0FBQyxlQUFlLEdBQUcsT0FBTyxDQUFDLGVBQWUsQ0FBQztRQUMvQyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsT0FBTyxDQUFDLGVBQWUsRUFBRSxrQkFBa0IsSUFBSSxFQUFFLENBQUM7UUFDM0UsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQyxRQUFRLEVBQUUsb0JBQW9CLElBQUksRUFBRSxDQUFDO1FBRXRFLHlDQUF5QztRQUN6QyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFdkMsc0RBQXNEO1FBQ3RELElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDekIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUV0QixJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLFlBQVksRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDLHVEQUF1RDtRQUM3RyxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLCtCQUErQixDQUFDLENBQUM7UUFFcEUsSUFBSSxPQUFPLENBQUMsbUJBQW1CLEtBQUssS0FBSyxFQUFFO1lBQ3pDLElBQUksQ0FBQyxVQUFVLENBQUMsbUJBQW1CLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFFakQsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDbEMsT0FBTyxDQUFDLDBCQUEwQixFQUFFLFFBQVEsSUFBSSxjQUFjLENBQy9ELENBQUM7WUFDRixlQUFlLENBQUMsSUFBSSxDQUNsQix1REFBdUQsQ0FDeEQsQ0FBQztZQUNGLGVBQWUsQ0FBQyxJQUFJLENBQUMsNkJBQTZCLENBQUMsQ0FBQztZQUNwRCxlQUFlLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLFVBQVUsQ0FBQyxDQUFDO1lBQy9ELGVBQWUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7WUFFbkMsSUFBSSxpQkFBUSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtnQkFDckMsR0FBRyxFQUNELE9BQU8sQ0FBQywwQkFBMEIsRUFBRSxjQUFjLElBQUksaUNBQWM7Z0JBQ3RFLFFBQVEsRUFBRSxJQUFJO2FBQ2YsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxPQUFPLENBQUMsUUFBUSxFQUFFLDBCQUEwQjtZQUMxQyxJQUFJLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFFcEMsSUFBSSxtQkFBVSxDQUFDLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQyxPQUFPLENBQ3ZDLGNBQWMsRUFDZCxRQUFRLEVBQ1IsTUFBTSxFQUNOLGVBQWUsRUFDZixHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxRQUFRLElBQUksRUFBRSxDQUFDLENBQ25DLENBQUM7UUFFRixJQUFJLGlCQUFRLENBQUMsSUFBSSxFQUFFLHVCQUF1QixFQUFFO1lBQzFDLFFBQVEsRUFBRSxJQUFJO1lBQ2QsS0FBSyxFQUFFLEVBQUUsQ0FBQyxZQUFZLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQztTQUN0RSxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksaUJBQVEsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzFDLEdBQUcsRUFBRTtnQkFDSCxPQUFPLEVBQUUsa0NBQWtDO2dCQUMzQyxPQUFPLEVBQUUsQ0FBQyxLQUFLLHVCQUF1QixFQUFFLENBQUM7Z0JBQ3pDLFFBQVEsRUFBRSxVQUFVO2dCQUNwQixrQkFBa0IsRUFBRTtvQkFDbEIsT0FBTyxFQUFFO3dCQUNQLE1BQU0sRUFBRSxPQUFPLENBQUMsUUFBUSxFQUFFLDBCQUEwQjs0QkFDbEQsQ0FBQyxDQUFDLGdCQUFnQjs0QkFDbEIsQ0FBQyxDQUFDLHVDQUF1Qzt3QkFDM0MsT0FBTyxFQUFFOzRCQUNQLGdCQUFnQixFQUFFLEtBQUs7NEJBQ3ZCLG1CQUFtQixFQUFFLE9BQU8sQ0FBQyxRQUFRLEVBQUUsbUJBQW1CLElBQUk7Z0NBQzVELE9BQU87Z0NBQ1AsTUFBTTs2QkFDUDs0QkFDRCxXQUFXLEVBQUUsT0FBTyxDQUFDLFFBQVEsRUFBRSwwQkFBMEI7eUJBQzFEO3FCQUNGO2lCQUNGO2dCQUNELFdBQVcsRUFBRSxPQUFPLENBQUMsUUFBUSxFQUFFLFdBQVc7Z0JBQzFDLGNBQWMsRUFBRSxPQUFPLENBQUMsUUFBUSxFQUFFLGNBQWM7Z0JBQ2hELG9CQUFvQixFQUFFLElBQUksQ0FBQyxvQkFBb0I7Z0JBQy9DLGtCQUFrQixFQUFFO29CQUNsQixLQUFLLEVBQUU7d0JBQ0w7NEJBQ0UsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFLGNBQWM7eUJBQ3pCO3FCQUNGO29CQUNELEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLGtCQUFrQixJQUFJLEVBQUUsQ0FBQztpQkFDN0M7Z0JBQ0QsUUFBUSxFQUFFO29CQUNSLFdBQVcsRUFBRSxJQUFJLENBQUMsUUFBUSxFQUFFLGNBQWMsSUFBSSxVQUFVO2lCQUN6RDthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxxQkFBcUIsQ0FBQyxTQUFrQixFQUFFLFFBQWlCO1FBQ2hFLElBQUksSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUM3QyxJQUFJLENBQUMsb0JBQW9CLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDL0Q7YUFBTTtZQUNMLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDN0Q7SUFDSCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksb0JBQW9CLENBQUMsR0FBRyxZQUFzQjtRQUNuRCxnSEFBZ0g7UUFDaEgsaURBQWlEO1FBQ2pELE1BQU0sbUNBQW1DLEdBQ3ZDLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUNqRCxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUMzQyxDQUFDO1FBQ0osTUFBTSx5QkFBeUIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUNsRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUN6QixHQUFHLG1DQUFtQyxDQUFDLE1BQU0sQ0FDM0MsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLENBQUMseUJBQXlCLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUM3QyxDQUNGLENBQUM7UUFFRixtQ0FBbUM7UUFDbkMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxHQUFHLFlBQVksQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRCw2REFBNkQ7SUFDN0QsSUFBWSw2QkFBNkI7UUFDdkMsYUFBYTtRQUNiLE1BQU0sV0FBVyxHQUFjLElBQUksQ0FBQyxXQUFXLElBQUksRUFBRSxDQUFDO1FBQ3RELE9BQU8sV0FBVyxDQUFDO0lBQ3JCLENBQUM7SUFFRCxJQUFXLFdBQVc7UUFDcEIsT0FBTyxJQUFJLENBQUMsNkJBQTZCLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQ3RELENBQUMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FDN0IsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNILEtBQUs7UUFDSCxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztRQUMzQixJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7UUFDdkIsSUFBSSxDQUFDLHNCQUFzQixFQUFFLENBQUM7UUFDOUIsSUFBSSxDQUFDLDRCQUE0QixFQUFFLENBQUM7UUFFcEMseUdBQXlHO1FBQ3pHLDRGQUE0RjtRQUM1RixNQUFNLGtCQUFrQixHQUFrQixFQUFFLENBQUM7UUFDN0MsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxVQUFVLEVBQUUsRUFBRTtZQUN0QyxJQUFJLGFBQWEsQ0FBQyxVQUFVLENBQUMsRUFBRTtnQkFDN0IsTUFBTSxjQUFjLEdBQWdCLFVBQXlCLENBQUM7Z0JBQzlELGtCQUFrQixDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBQ2hELGdEQUFnRDtnQkFDaEQsY0FBYyxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsR0FBRyxHQUFHLEVBQUUsR0FBRSxDQUFDLENBQUM7YUFDdkQ7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUVkLHVHQUF1RztRQUN2RztRQUNFLGlDQUFpQztRQUNoQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQWlCLENBQUMsT0FBTyxLQUFLLElBQUk7WUFDaEQsaUNBQWlDO1lBQ2pDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUUsR0FBRyxDQUFDLElBQWlCLENBQUMsT0FBTyxLQUFLLElBQUksQ0FBQyxFQUN6RTtZQUNBLGdEQUFnRDtZQUNoRCxJQUFJLENBQUMsT0FBTyxDQUFDLG1CQUFtQixFQUFFLENBQUM7U0FDcEM7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxtQkFBbUI7UUFDekIsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxVQUFlLEVBQUUsRUFBRTtZQUMzQyw0RUFBNEU7WUFDNUUsVUFBVSxDQUFDLFdBQVcsRUFBRSxLQUFLLEVBQUUsQ0FBQztZQUVoQyxJQUNFLGFBQWEsQ0FBQyxVQUFVLENBQUM7Z0JBQ3pCLFVBQVUsQ0FBQyxPQUFPLENBQUMsY0FBYyxLQUFLLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUNqRTtnQkFDQSxNQUFNLElBQUksS0FBSyxDQUNiLEdBQUcsVUFBVSxDQUFDLElBQUksK0RBQStELElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxHQUFHLENBQ2hILENBQUM7YUFDSDtRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ssNEJBQTRCO1FBQ2xDLElBQUksQ0FBQyxXQUFXO2FBQ2IsTUFBTSxDQUFDLENBQUMsVUFBZSxFQUFFLEVBQUUsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxVQUFVLENBQUMsQ0FBQzthQUN2RCxNQUFNLENBQUMsQ0FBQyxVQUFtQixFQUFFLEVBQUUsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUM7YUFDeEUsT0FBTyxDQUFDLENBQUMsVUFBbUIsRUFBRSxFQUFFO1lBQy9CLHVDQUF1QztZQUN2QyxNQUFNLFFBQVEsR0FBUTtnQkFDcEIsSUFBSSxFQUFFLFVBQVUsQ0FBQyxJQUFJO2dCQUNyQixPQUFPLEVBQUUsSUFBSTtnQkFDYixPQUFPLEVBQUUsSUFBSTtnQkFDYixPQUFPLEVBQUUsVUFBVSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUNsQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7b0JBQ1QsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsY0FBYyxDQUFDLENBQUMsSUFBSSxFQUFFO29CQUNoQyxHQUFHLENBQUM7aUJBQ0wsQ0FBQyxFQUNGLEVBQUUsQ0FDSDtnQkFDRCxPQUFPLEVBQUUsT0FBTzthQUNqQixDQUFDO1lBRUYsSUFBSSxpQkFBUSxDQUFDLFVBQVUsRUFBRSxjQUFjLEVBQUU7Z0JBQ3ZDLEdBQUcsRUFBRSxRQUFRO2dCQUNiLFFBQVEsRUFBRSxJQUFJO2FBQ2YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxlQUFlO1FBQ3JCLGlIQUFpSDtRQUNqSCwyQ0FBMkM7UUFDM0MsSUFBSSxDQUFDLG9CQUFvQixFQUFFLENBQUM7UUFFNUIsSUFBSSxPQUFPLEdBQUcsSUFBSSxDQUFDLGVBQWUsRUFBRSxPQUFPLENBQUM7UUFDNUMsZ0lBQWdJO1FBQ2hJLElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRSxxQkFBcUIsS0FBSyxJQUFJLEVBQUU7WUFDeEQsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtnQkFDdEQsSUFBSSxHQUFHLFlBQVksd0JBQVcsRUFBRTtvQkFDOUIsT0FBTyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUc7eUJBQ2hCLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksS0FBSyx1QkFBYyxDQUFDLE9BQU8sQ0FBQzt5QkFDcEQsT0FBTyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQzt3QkFDaEIsR0FBRyxHQUFHLENBQUMsSUFBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLEVBQUU7d0JBQ3pCLEdBQUcsR0FBRyxDQUFDLElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxJQUFJO3FCQUM1QixDQUFDLENBQUM7aUJBQ047Z0JBQ0QsT0FBTyxFQUFFLENBQUM7WUFDWixDQUFDLENBQUMsQ0FBQztZQUVILElBQUksY0FBYyxDQUFDLE1BQU0sRUFBRTtnQkFDekIsT0FBTyxHQUFHLENBQUMsR0FBRyxDQUFDLE9BQU8sSUFBSSxFQUFFLENBQUMsRUFBRSxHQUFHLGNBQWMsQ0FBQyxDQUFDO2FBQ25EO1NBQ0Y7UUFFRCxxQ0FBcUM7UUFDckMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsS0FBSywrQkFBa0IsQ0FBQyxJQUFJLEVBQUU7WUFDM0QsSUFBSSxpQkFBUSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtnQkFDeEMsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsR0FBRyxFQUFFO29CQUNILFFBQVEsRUFBRSxJQUFJLENBQUMsaUJBQWlCO2lCQUNqQzthQUNGLENBQUMsQ0FBQztTQUNKO2FBQU07WUFDTCxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7Z0JBQ2xDLFFBQVEsRUFBRSxJQUFJLENBQUMsaUJBQWlCO2dCQUNoQyxPQUFPLEVBQUUsT0FBTzthQUNqQixDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssc0JBQXNCO1FBQzVCLDJCQUEyQjtRQUMzQixNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FDNUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLE9BQU8sWUFBWSxzQkFBYSxDQUMzQixDQUFDO1FBRXJCLElBQUksY0FBYyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDL0IsdUNBQXVDO1lBQ3ZDLE9BQU87U0FDUjtRQUVELHNIQUFzSDtRQUN0SCxpSEFBaUg7UUFDakgsY0FBYyxDQUFDLE9BQU8sQ0FBQyxDQUFDLGFBQWEsRUFBRSxFQUFFO1lBQ3ZDLE1BQU0sYUFBYSxHQUNqQixhQUFhLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUM7Z0JBQ3pDLGFBQWEsQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDdEMsTUFBTSxXQUFXLEdBQUcsYUFBYSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUM7WUFFM0QsQ0FBQyxXQUFXLEVBQUUsS0FBSyxJQUFJLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFO2dCQUMxQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRSxhQUFhLENBQUMsQ0FBQztZQUM1RCxDQUFDLENBQUMsQ0FBQztZQUVILFdBQVcsRUFBRSxLQUFLLEVBQUUsQ0FBQztRQUN2QixDQUFDLENBQUMsQ0FBQztRQUVILHFIQUFxSDtRQUNySCxrR0FBa0c7UUFDbEcsTUFBTSxtQkFBbUIsR0FDdkIsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUMzRCxtQkFBbUIsQ0FBQyxJQUFJLENBQ3RCLGtEQUFrRCxjQUFjO2FBQzdELEdBQUcsQ0FBQyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQzthQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FDNUIsQ0FBQztRQUVGLHdFQUF3RTtRQUN4RSxJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQywrQkFBK0IsRUFBRTtZQUN2RDtnQkFDRSxNQUFNLEVBQUUsWUFBWTtnQkFDcEIsUUFBUSxFQUFFLGNBQWM7YUFDekI7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ssZ0JBQWdCLENBQUMsSUFBYyxFQUFFLElBQVUsRUFBRSxPQUFnQjtRQUNuRSxJQUFJLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDYixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDMUQ7YUFBTSxJQUFJLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLEVBQUUsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQ3hEO2FBQU0sSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ3JCLE1BQU0sV0FBVyxHQUFHLE9BQU8sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUN0RCxJQUFJLFdBQVcsRUFBRTtnQkFDZixJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQzthQUM3RDtTQUNGO2FBQU0sSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ3ZCLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQzVCO0lBQ0gsQ0FBQzs7QUFuWEgsOENBb1hDOzs7QUFFRDs7Ozs7R0FLRztBQUNILFNBQVMsYUFBYSxDQUFDLE9BQVk7SUFDakMsT0FBTyxPQUFPLFlBQVksd0JBQVcsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDO0FBQzNELENBQUM7QUFFRCxTQUFTLGFBQWE7SUFDcEIsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSw4QkFBOEIsQ0FBQyxDQUFDO0FBQzlELENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCAqIGFzIGZzIGZyb20gXCJmc1wiO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHtcbiAgRGVwZW5kZW5jeVR5cGUsXG4gIElnbm9yZUZpbGUsXG4gIEpzb25GaWxlLFxuICBQcm9qZWN0LFxuICBUYXNrLFxuICBUYXNrU3RlcCxcbiAgVGV4dEZpbGUsXG4gIFlhbWxGaWxlLFxufSBmcm9tIFwicHJvamVuXCI7XG5pbXBvcnQge1xuICBOb2RlUGFja2FnZSxcbiAgTm9kZVBhY2thZ2VNYW5hZ2VyLFxuICBOb2RlUHJvamVjdCxcbn0gZnJvbSBcInByb2plbi9saWIvamF2YXNjcmlwdFwiO1xuaW1wb3J0IHsgUHl0aG9uUHJvamVjdCB9IGZyb20gXCJwcm9qZW4vbGliL3B5dGhvblwiO1xuaW1wb3J0IHtcbiAgVHlwZVNjcmlwdFByb2plY3QsXG4gIFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucyxcbn0gZnJvbSBcInByb2plbi9saWIvdHlwZXNjcmlwdFwiO1xuaW1wb3J0IHsgTnggfSBmcm9tIFwiLi9ueC10eXBlc1wiO1xuaW1wb3J0IHsgREVGQVVMVF9DT05GSUcsIFN5bmNwYWNrQ29uZmlnIH0gZnJvbSBcIi4vc3luY3BhY2stb3B0aW9uc1wiO1xuXG5jb25zdCBOWF9NT05PUkVQT19QTFVHSU5fUEFUSDogc3RyaW5nID0gXCIubngvcGx1Z2lucy9ueC1tb25vcmVwby1wbHVnaW4uanNcIjtcblxuLyoqXG4gKiBXb3Jrc3BhY2UgY29uZmlndXJhdGlvbnMuXG4gKlxuICogQHNlZSBodHRwczovL2NsYXNzaWMueWFybnBrZy5jb20vbGFuZy9lbi9kb2NzL3dvcmtzcGFjZXMvXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgV29ya3NwYWNlQ29uZmlnIHtcbiAgLyoqXG4gICAqIExpc3Qgb2YgcGFja2FnZSBnbG9icyB0byBleGNsdWRlIGZyb20gaG9pc3RpbmcgaW4gdGhlIHdvcmtzcGFjZS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2NsYXNzaWMueWFybnBrZy5jb20vYmxvZy8yMDE4LzAyLzE1L25vaG9pc3QvXG4gICAqL1xuICByZWFkb25seSBub0hvaXN0Pzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIERpc2FibGUgYXV0b21hdGljYWxseSBhcHBseWluZyBgbm9Ib2lzdGAgbG9naWMgZm9yIGFsbCBzdWItcHJvamVjdCBcImJ1bmRsZWREZXBlbmRlbmNpZXNcIi5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGRpc2FibGVOb0hvaXN0QnVuZGxlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgYWRkaXRpb25hbCBwYWNrYWdlIGdsb2JzIHRvIGluY2x1ZGUgaW4gdGhlIHdvcmtzcGFjZS5cbiAgICpcbiAgICogQWxsIHBhY2thZ2VzIHdoaWNoIGFyZSBwYXJlbnRlZCBieSB0aGUgbW9ub3JlcG8gYXJlIGF1dG9tYXRpY2FsbHkgYWRkZWQgdG8gdGhlIHdvcmtzcGFjZSwgYnV0IHlvdSBjYW4gdXNlIHRoaXNcbiAgICogcHJvcGVydHkgdG8gc3BlY2lmeSBhbnkgYWRkaXRpb25hbCBwYXRocyB0byBwYWNrYWdlcyB3aGljaCBtYXkgbm90IGJlIG1hbmFnZWQgYnkgcHJvamVuLlxuICAgKi9cbiAgcmVhZG9ubHkgYWRkaXRpb25hbFBhY2thZ2VzPzogc3RyaW5nW107XG59XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBmb3IgTW9ub3JlcG8gVXBncmFkZSBEZXBzIHRhc2suXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTW9ub3JlcG9VcGdyYWRlRGVwc09wdGlvbnMge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgdGFzayB0byBjcmVhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IHVwZ3JhZGUtZGVwc1xuICAgKi9cbiAgcmVhZG9ubHkgdGFza05hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFN5bmNwYWNrIGNvbmZpZ3VyYXRpb24uXG4gICAqXG4gICAqIE5vIG1lcmdpbmcgaXMgcGVyZm9ybWVkIGFuZCBhcyBzdWNoIGEgY29tcGxldGUgc3luY3BhY2tDb25maWcgaXMgcmVxdWlyZWQgaWYgc3VwcGxpZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IFN5bmNwYWNrQ29uZmlnLkRFRkFVTFRfQ09ORklHXG4gICAqL1xuICByZWFkb25seSBzeW5jcGFja0NvbmZpZz86IFN5bmNwYWNrQ29uZmlnO1xufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gb3B0aW9ucyBmb3IgdGhlIE54TW9ub3JlcG9Qcm9qZWN0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE54TW9ub3JlcG9Qcm9qZWN0T3B0aW9ucyBleHRlbmRzIFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIGZvciBOWC5cbiAgICovXG4gIHJlYWRvbmx5IG54Q29uZmlnPzogTnguV29ya3NwYWNlQ29uZmlnO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIGZvciB3b3Jrc3BhY2UuXG4gICAqL1xuICByZWFkb25seSB3b3Jrc3BhY2VDb25maWc/OiBXb3Jrc3BhY2VDb25maWc7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gaW5jbHVkZSBhbiB1cGdyYWRlLWRlcHMgdGFzayBhdCB0aGUgcm9vdCBvZiB0aGUgbW9ub3JlcG8gd2hpY2ggd2lsbCB1cGdyYWRlIGFsbCBkZXBlbmRlbmNpZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IG1vbm9yZXBvVXBncmFkZURlcHM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBNb25vcmVwbyBVcGdyYWRlIERlcHMgb3B0aW9ucy5cbiAgICpcbiAgICogVGhpcyBpcyBvbmx5IHVzZWQgaWYgbW9ub3JlcG9VcGdyYWRlRGVwcyBpcyB0cnVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IG1vbm9yZXBvVXBncmFkZURlcHNPcHRpb25zPzogTW9ub3JlcG9VcGdyYWRlRGVwc09wdGlvbnM7XG59XG5cbi8qKlxuICogVGhpcyBwcm9qZWN0IHR5cGUgd2lsbCBib290c3RyYXAgYSBOWCBiYXNlZCBtb25vcmVwbyB3aXRoIHN1cHBvcnQgZm9yIHBvbHlnb3RcbiAqIGJ1aWxkcywgYnVpbGQgY2FjaGluZywgZGVwZW5kZW5jeSBncmFwaCB2aXN1YWxpemF0aW9uIGFuZCBtdWNoIG1vcmUuXG4gKlxuICogQHBqaWQgbngtbW9ub3JlcG9cbiAqL1xuZXhwb3J0IGNsYXNzIE54TW9ub3JlcG9Qcm9qZWN0IGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3Qge1xuICAvLyBtdXRhYmxlIGRhdGEgc3RydWN0dXJlc1xuICBwcml2YXRlIHJlYWRvbmx5IGltcGxpY2l0RGVwZW5kZW5jaWVzOiBOeC5JbXBsaWNpdERlcGVuZGVuY2llcztcblxuICAvLyBpbW11dGFibGUgZGF0YSBzdHJ1Y3R1cmVzXG4gIHByaXZhdGUgcmVhZG9ubHkgbnhDb25maWc/OiBOeC5Xb3Jrc3BhY2VDb25maWc7XG4gIHByaXZhdGUgcmVhZG9ubHkgd29ya3NwYWNlQ29uZmlnPzogV29ya3NwYWNlQ29uZmlnO1xuICBwcml2YXRlIHJlYWRvbmx5IHdvcmtzcGFjZVBhY2thZ2VzOiBzdHJpbmdbXTtcblxuICBwcml2YXRlIHJlYWRvbmx5IG54SnNvbjogSnNvbkZpbGU7XG5cbiAgY29uc3RydWN0b3Iob3B0aW9uczogTnhNb25vcmVwb1Byb2plY3RPcHRpb25zKSB7XG4gICAgc3VwZXIoe1xuICAgICAgLi4ub3B0aW9ucyxcbiAgICAgIGdpdGh1Yjogb3B0aW9ucy5naXRodWIgPz8gZmFsc2UsXG4gICAgICBwYWNrYWdlOiBvcHRpb25zLnBhY2thZ2UgPz8gZmFsc2UsXG4gICAgICBwcmV0dGllcjogb3B0aW9ucy5wcmV0dGllciA/PyB0cnVlLFxuICAgICAgcHJvamVucmNUczogdHJ1ZSxcbiAgICAgIHJlbGVhc2U6IG9wdGlvbnMucmVsZWFzZSA/PyBmYWxzZSxcbiAgICAgIGplc3Q6IG9wdGlvbnMuamVzdCA/PyBmYWxzZSxcbiAgICAgIGRlZmF1bHRSZWxlYXNlQnJhbmNoOiBvcHRpb25zLmRlZmF1bHRSZWxlYXNlQnJhbmNoID8/IFwibWFpbmxpbmVcIixcbiAgICAgIHNhbXBsZUNvZGU6IGZhbHNlLCAvLyByb290IHNob3VsZCBuZXZlciBoYXZlIHNhbXBsZSBjb2RlLFxuICAgICAgZXNsaW50T3B0aW9uczogb3B0aW9ucy5lc2xpbnRPcHRpb25zID8/IHtcbiAgICAgICAgZGlyczogW1wiLlwiXSxcbiAgICAgICAgaWdub3JlUGF0dGVybnM6IFtcInBhY2thZ2VzLyoqLyouKlwiXSxcbiAgICAgIH0sXG4gICAgICB0c2NvbmZpZzogb3B0aW9ucy50c2NvbmZpZyA/PyB7XG4gICAgICAgIGNvbXBpbGVyT3B0aW9uczoge1xuICAgICAgICAgIHJvb3REaXI6IFwiLlwiLFxuICAgICAgICB9LFxuICAgICAgICBpbmNsdWRlOiBbXCIqKi8qLnRzXCJdLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMubnhDb25maWcgPSBvcHRpb25zLm54Q29uZmlnO1xuICAgIHRoaXMud29ya3NwYWNlQ29uZmlnID0gb3B0aW9ucy53b3Jrc3BhY2VDb25maWc7XG4gICAgdGhpcy53b3Jrc3BhY2VQYWNrYWdlcyA9IG9wdGlvbnMud29ya3NwYWNlQ29uZmlnPy5hZGRpdGlvbmFsUGFja2FnZXMgPz8gW107XG4gICAgdGhpcy5pbXBsaWNpdERlcGVuZGVuY2llcyA9IHRoaXMubnhDb25maWc/LmltcGxpY2l0RGVwZW5kZW5jaWVzIHx8IHt9O1xuXG4gICAgLy8gTmV2ZXIgcHVibGlzaCBhIG1vbm9yZXBvIHJvb3QgcGFja2FnZS5cbiAgICB0aGlzLnBhY2thZ2UuYWRkRmllbGQoXCJwcml2YXRlXCIsIHRydWUpO1xuXG4gICAgLy8gTm8gbmVlZCB0byBjb21waWxlIG9yIHRlc3QgYSBtb25vcmVwbyByb290IHBhY2thZ2UuXG4gICAgdGhpcy5jb21waWxlVGFzay5yZXNldCgpO1xuICAgIHRoaXMudGVzdFRhc2sucmVzZXQoKTtcblxuICAgIHRoaXMuYWRkRGV2RGVwcyhcIkBucndsL2NsaVwiLCBcIkBucndsL3dvcmtzcGFjZVwiKTtcbiAgICB0aGlzLmFkZERlcHMoXCJhd3MtY2RrLWxpYlwiLCBcImNvbnN0cnVjdHNcIiwgXCJjZGstbmFnXCIpOyAvLyBOZWVkZWQgYXMgdGhpcyBjYW4gYmUgYnVuZGxlZCBpbiBhd3MtcHJvdG90eXBpbmctc2RrXG4gICAgdGhpcy5wYWNrYWdlLmFkZFBhY2thZ2VSZXNvbHV0aW9ucyhcIkB0eXBlcy9iYWJlbF9fdHJhdmVyc2VANy4xOC4yXCIpO1xuXG4gICAgaWYgKG9wdGlvbnMubW9ub3JlcG9VcGdyYWRlRGVwcyAhPT0gZmFsc2UpIHtcbiAgICAgIHRoaXMuYWRkRGV2RGVwcyhcIm5wbS1jaGVjay11cGRhdGVzXCIsIFwic3luY3BhY2tcIik7XG5cbiAgICAgIGNvbnN0IHVwZ3JhZGVEZXBzVGFzayA9IHRoaXMuYWRkVGFzayhcbiAgICAgICAgb3B0aW9ucy5tb25vcmVwb1VwZ3JhZGVEZXBzT3B0aW9ucz8udGFza05hbWUgfHwgXCJ1cGdyYWRlLWRlcHNcIlxuICAgICAgKTtcbiAgICAgIHVwZ3JhZGVEZXBzVGFzay5leGVjKFxuICAgICAgICBcIm5weCBucG0tY2hlY2stdXBkYXRlcyAtLWRlZXAgLS1yZWplY3RWZXJzaW9uIDAuMC4wIC11XCJcbiAgICAgICk7XG4gICAgICB1cGdyYWRlRGVwc1Rhc2suZXhlYyhcIm5weCBzeW5jcGFjayBmaXgtbWlzbWF0Y2hlc1wiKTtcbiAgICAgIHVwZ3JhZGVEZXBzVGFzay5leGVjKGAke3RoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcn0gaW5zdGFsbGApO1xuICAgICAgdXBncmFkZURlcHNUYXNrLmV4ZWMoXCJucHggcHJvamVuXCIpO1xuXG4gICAgICBuZXcgSnNvbkZpbGUodGhpcywgXCIuc3luY3BhY2tyYy5qc29uXCIsIHtcbiAgICAgICAgb2JqOlxuICAgICAgICAgIG9wdGlvbnMubW9ub3JlcG9VcGdyYWRlRGVwc09wdGlvbnM/LnN5bmNwYWNrQ29uZmlnIHx8IERFRkFVTFRfQ09ORklHLFxuICAgICAgICByZWFkb25seTogdHJ1ZSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIG9wdGlvbnMubnhDb25maWc/Lm54Q2xvdWRSZWFkT25seUFjY2Vzc1Rva2VuICYmXG4gICAgICB0aGlzLmFkZERldkRlcHMoXCJAbnJ3bC9ueC1jbG91ZFwiKTtcblxuICAgIG5ldyBJZ25vcmVGaWxlKHRoaXMsIFwiLm54aWdub3JlXCIpLmV4Y2x1ZGUoXG4gICAgICBcInRlc3QtcmVwb3J0c1wiLFxuICAgICAgXCJ0YXJnZXRcIixcbiAgICAgIFwiLmVudlwiLFxuICAgICAgXCIucHl0ZXN0X2NhY2hlXCIsXG4gICAgICAuLi4odGhpcy5ueENvbmZpZz8ubnhJZ25vcmUgfHwgW10pXG4gICAgKTtcblxuICAgIG5ldyBUZXh0RmlsZSh0aGlzLCBOWF9NT05PUkVQT19QTFVHSU5fUEFUSCwge1xuICAgICAgcmVhZG9ubHk6IHRydWUsXG4gICAgICBsaW5lczogZnMucmVhZEZpbGVTeW5jKGdldFBsdWdpblBhdGgoKSkudG9TdHJpbmcoXCJ1dGYtOFwiKS5zcGxpdChcIlxcblwiKSxcbiAgICB9KTtcblxuICAgIHRoaXMubnhKc29uID0gbmV3IEpzb25GaWxlKHRoaXMsIFwibnguanNvblwiLCB7XG4gICAgICBvYmo6IHtcbiAgICAgICAgZXh0ZW5kczogXCJAbnJ3bC93b3Jrc3BhY2UvcHJlc2V0cy9ucG0uanNvblwiLFxuICAgICAgICBwbHVnaW5zOiBbYC4vJHtOWF9NT05PUkVQT19QTFVHSU5fUEFUSH1gXSxcbiAgICAgICAgbnBtU2NvcGU6IFwibW9ub3JlcG9cIixcbiAgICAgICAgdGFza3NSdW5uZXJPcHRpb25zOiB7XG4gICAgICAgICAgZGVmYXVsdDoge1xuICAgICAgICAgICAgcnVubmVyOiBvcHRpb25zLm54Q29uZmlnPy5ueENsb3VkUmVhZE9ubHlBY2Nlc3NUb2tlblxuICAgICAgICAgICAgICA/IFwiQG5yd2wvbngtY2xvdWRcIlxuICAgICAgICAgICAgICA6IFwiQG5yd2wvd29ya3NwYWNlL3Rhc2tzLXJ1bm5lcnMvZGVmYXVsdFwiLFxuICAgICAgICAgICAgb3B0aW9uczoge1xuICAgICAgICAgICAgICB1c2VEYWVtb25Qcm9jZXNzOiBmYWxzZSxcbiAgICAgICAgICAgICAgY2FjaGVhYmxlT3BlcmF0aW9uczogb3B0aW9ucy5ueENvbmZpZz8uY2FjaGVhYmxlT3BlcmF0aW9ucyB8fCBbXG4gICAgICAgICAgICAgICAgXCJidWlsZFwiLFxuICAgICAgICAgICAgICAgIFwidGVzdFwiLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBhY2Nlc3NUb2tlbjogb3B0aW9ucy5ueENvbmZpZz8ubnhDbG91ZFJlYWRPbmx5QWNjZXNzVG9rZW4sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIG5hbWVkSW5wdXRzOiBvcHRpb25zLm54Q29uZmlnPy5uYW1lZElucHV0cyxcbiAgICAgICAgdGFyZ2V0RGVmYXVsdHM6IG9wdGlvbnMubnhDb25maWc/LnRhcmdldERlZmF1bHRzLFxuICAgICAgICBpbXBsaWNpdERlcGVuZGVuY2llczogdGhpcy5pbXBsaWNpdERlcGVuZGVuY2llcyxcbiAgICAgICAgdGFyZ2V0RGVwZW5kZW5jaWVzOiB7XG4gICAgICAgICAgYnVpbGQ6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgdGFyZ2V0OiBcImJ1aWxkXCIsXG4gICAgICAgICAgICAgIHByb2plY3RzOiBcImRlcGVuZGVuY2llc1wiLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIC4uLih0aGlzLm54Q29uZmlnPy50YXJnZXREZXBlbmRlbmNpZXMgfHwge30pLFxuICAgICAgICB9LFxuICAgICAgICBhZmZlY3RlZDoge1xuICAgICAgICAgIGRlZmF1bHRCYXNlOiB0aGlzLm54Q29uZmlnPy5hZmZlY3RlZEJyYW5jaCB8fCBcIm1haW5saW5lXCIsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhbiBpbXBsaWNpdCBkZXBlbmRlbmN5IGJldHdlZW4gdHdvIFByb2plY3Qncy4gVGhpcyBpcyB0eXBpY2FsbHlcbiAgICogdXNlZCBpbiBwb2x5Z290IHJlcG9zIHdoZXJlIGEgVHlwZXNjcmlwdCBwcm9qZWN0IHdhbnRzIGEgYnVpbGQgZGVwZW5kZW5jeVxuICAgKiBvbiBhIFB5dGhvbiBwcm9qZWN0IGFzIGFuIGV4YW1wbGUuXG4gICAqXG4gICAqIEBwYXJhbSBkZXBlbmRlbnQgcHJvamVjdCB5b3Ugd2FudCB0byBoYXZlIHRoZSBkZXBlbmRlbmN5LlxuICAgKiBAcGFyYW0gZGVwZW5kZWUgcHJvamVjdCB5b3Ugd2lzaCB0byBkZXBlbmQgb24uXG4gICAqL1xuICBwdWJsaWMgYWRkSW1wbGljaXREZXBlbmRlbmN5KGRlcGVuZGVudDogUHJvamVjdCwgZGVwZW5kZWU6IFByb2plY3QpIHtcbiAgICBpZiAodGhpcy5pbXBsaWNpdERlcGVuZGVuY2llc1tkZXBlbmRlbnQubmFtZV0pIHtcbiAgICAgIHRoaXMuaW1wbGljaXREZXBlbmRlbmNpZXNbZGVwZW5kZW50Lm5hbWVdLnB1c2goZGVwZW5kZWUubmFtZSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuaW1wbGljaXREZXBlbmRlbmNpZXNbZGVwZW5kZW50Lm5hbWVdID0gW2RlcGVuZGVlLm5hbWVdO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgb25lIG9yIG1vcmUgYWRkaXRpb25hbCBwYWNrYWdlIGdsb2JzIHRvIHRoZSB3b3Jrc3BhY2UuXG4gICAqIEBwYXJhbSBwYWNrYWdlR2xvYnMgcGF0aHMgdG8gdGhlIHBhY2thZ2UgdG8gaW5jbHVkZSBpbiB0aGUgd29ya3NwYWNlIChmb3IgZXhhbXBsZSBwYWNrYWdlcy9teS1wYWNrYWdlKVxuICAgKi9cbiAgcHVibGljIGFkZFdvcmtzcGFjZVBhY2thZ2VzKC4uLnBhY2thZ2VHbG9iczogc3RyaW5nW10pIHtcbiAgICAvLyBBbnkgc3VicHJvamVjdHMgdGhhdCB3ZXJlIGFkZGVkIHNpbmNlIHRoZSBsYXN0IGNhbGwgdG8gdGhpcyBtZXRob2QgbmVlZCB0byBiZSBhZGRlZCBmaXJzdCwgaW4gb3JkZXIgdG8gZW5zdXJlXG4gICAgLy8gd2UgYWRkIHRoZSB3b3Jrc3BhY2UgcGFja2FnZXMgaW4gYSBzYW5lIG9yZGVyLlxuICAgIGNvbnN0IHJlbGF0aXZlU3ViUHJvamVjdFdvcmtzcGFjZVBhY2thZ2VzID1cbiAgICAgIHRoaXMuaW5zdGFudGlhdGlvbk9yZGVyU3ViUHJvamVjdHMubWFwKChwcm9qZWN0KSA9PlxuICAgICAgICBwYXRoLnJlbGF0aXZlKHRoaXMub3V0ZGlyLCBwcm9qZWN0Lm91dGRpcilcbiAgICAgICk7XG4gICAgY29uc3QgZXhpc3RpbmdXb3Jrc3BhY2VQYWNrYWdlcyA9IG5ldyBTZXQodGhpcy53b3Jrc3BhY2VQYWNrYWdlcyk7XG4gICAgdGhpcy53b3Jrc3BhY2VQYWNrYWdlcy5wdXNoKFxuICAgICAgLi4ucmVsYXRpdmVTdWJQcm9qZWN0V29ya3NwYWNlUGFja2FnZXMuZmlsdGVyKFxuICAgICAgICAocGtnKSA9PiAhZXhpc3RpbmdXb3Jrc3BhY2VQYWNrYWdlcy5oYXMocGtnKVxuICAgICAgKVxuICAgICk7XG5cbiAgICAvLyBBZGQgdGhlIGFkZGl0aW9uYWwgcGFja2FnZXMgbmV4dFxuICAgIHRoaXMud29ya3NwYWNlUGFja2FnZXMucHVzaCguLi5wYWNrYWdlR2xvYnMpO1xuICB9XG5cbiAgLy8gUmVtb3ZlIHRoaXMgaGFjayBvbmNlIHN1YlByb2plY3RzIGlzIG1hZGUgcHVibGljIGluIFByb2plblxuICBwcml2YXRlIGdldCBpbnN0YW50aWF0aW9uT3JkZXJTdWJQcm9qZWN0cygpOiBQcm9qZWN0W10ge1xuICAgIC8vIEB0cy1pZ25vcmVcbiAgICBjb25zdCBzdWJQcm9qZWN0czogUHJvamVjdFtdID0gdGhpcy5zdWJwcm9qZWN0cyB8fCBbXTtcbiAgICByZXR1cm4gc3ViUHJvamVjdHM7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHN1YlByb2plY3RzKCk6IFByb2plY3RbXSB7XG4gICAgcmV0dXJuIHRoaXMuaW5zdGFudGlhdGlvbk9yZGVyU3ViUHJvamVjdHMuc29ydCgoYSwgYikgPT5cbiAgICAgIGEubmFtZS5sb2NhbGVDb21wYXJlKGIubmFtZSlcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbmhlcml0RG9jXG4gICAqL1xuICBzeW50aCgpIHtcbiAgICB0aGlzLnZhbGlkYXRlU3ViUHJvamVjdHMoKTtcbiAgICB0aGlzLnVwZGF0ZVdvcmtzcGFjZSgpO1xuICAgIHRoaXMud2lyZVB5dGhvbkRlcGVuZGVuY2llcygpO1xuICAgIHRoaXMuc3ludGhlc2l6ZU5vbk5vZGVQYWNrYWdlSnNvbigpO1xuXG4gICAgLy8gUHJldmVudCBzdWIgTm9kZVByb2plY3QgcGFja2FnZXMgZnJvbSBgcG9zdFN5bnRoZXNpc2Agd2hpY2ggd2lsbCBjYXVzZSBpbmRpdmlkdWFsL2V4dHJhbmVvdXMgaW5zdGFsbHMuXG4gICAgLy8gVGhlIHdvcmtzcGFjZSBwYWNrYWdlIGluc3RhbGwgd2lsbCBoYW5kbGUgYWxsIHRoZSBzdWIgTm9kZVByb2plY3QgcGFja2FnZXMgYXV0b21hdGljYWxseS5cbiAgICBjb25zdCBzdWJQcm9qZWN0UGFja2FnZXM6IE5vZGVQYWNrYWdlW10gPSBbXTtcbiAgICB0aGlzLnN1YlByb2plY3RzLmZvckVhY2goKHN1YlByb2plY3QpID0+IHtcbiAgICAgIGlmIChpc05vZGVQcm9qZWN0KHN1YlByb2plY3QpKSB7XG4gICAgICAgIGNvbnN0IHN1Yk5vZGVQcm9qZWN0OiBOb2RlUHJvamVjdCA9IHN1YlByb2plY3QgYXMgTm9kZVByb2plY3Q7XG4gICAgICAgIHN1YlByb2plY3RQYWNrYWdlcy5wdXNoKHN1Yk5vZGVQcm9qZWN0LnBhY2thZ2UpO1xuICAgICAgICAvLyBAdHMtaWdub3JlIC0gYGluc3RhbGxEZXBlbmRlbmNpZXNgIGlzIHByaXZhdGVcbiAgICAgICAgc3ViTm9kZVByb2plY3QucGFja2FnZS5pbnN0YWxsRGVwZW5kZW5jaWVzID0gKCkgPT4ge307XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICBzdXBlci5zeW50aCgpO1xuXG4gICAgLy8gRm9yY2Ugd29ya3NwYWNlIGluc3RhbGwgZGVwcyBpZiBhbnkgbm9kZSBzdWJwcm9qZWN0IHBhY2thZ2UgaGFzIGNoYW5nZSwgdW5sZXNzIHRoZSB3b3Jrc3BhY2UgY2hhbmdlZFxuICAgIGlmIChcbiAgICAgIC8vIEB0cy1pZ25vcmUgLSBgZmlsZWAgaXMgcHJpdmF0ZVxuICAgICAgKHRoaXMucGFja2FnZS5maWxlIGFzIEpzb25GaWxlKS5jaGFuZ2VkICE9PSB0cnVlICYmXG4gICAgICAvLyBAdHMtaWdub3JlIC0gYGZpbGVgIGlzIHByaXZhdGVcbiAgICAgIHN1YlByb2plY3RQYWNrYWdlcy5maW5kKChwa2cpID0+IChwa2cuZmlsZSBhcyBKc29uRmlsZSkuY2hhbmdlZCA9PT0gdHJ1ZSlcbiAgICApIHtcbiAgICAgIC8vIEB0cy1pZ25vcmUgLSBgaW5zdGFsbERlcGVuZGVuY2llc2AgaXMgcHJpdmF0ZVxuICAgICAgdGhpcy5wYWNrYWdlLmluc3RhbGxEZXBlbmRlbmNpZXMoKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogRW5zdXJlcyBzdWJwcm9qZWN0cyBkb24ndCBoYXZlIGEgZGVmYXVsdCB0YXNrIGFuZCB0aGF0IGFsbCBwYWNrYWdlcyB1c2UgdGhlIHNhbWUgcGFja2FnZSBtYW5hZ2VyLlxuICAgKi9cbiAgcHJpdmF0ZSB2YWxpZGF0ZVN1YlByb2plY3RzKCkge1xuICAgIHRoaXMuc3ViUHJvamVjdHMuZm9yRWFjaCgoc3ViUHJvamVjdDogYW55KSA9PiB7XG4gICAgICAvLyBEaXNhYmxlIGRlZmF1bHQgdGFzayBvbiBzdWJwcm9qZWN0cyBhcyB0aGlzIGlzbid0IHN1cHBvcnRlZCBpbiBhIG1vbm9yZXBvXG4gICAgICBzdWJQcm9qZWN0LmRlZmF1bHRUYXNrPy5yZXNldCgpO1xuXG4gICAgICBpZiAoXG4gICAgICAgIGlzTm9kZVByb2plY3Qoc3ViUHJvamVjdCkgJiZcbiAgICAgICAgc3ViUHJvamVjdC5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyICE9PSB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXJcbiAgICAgICkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgYCR7c3ViUHJvamVjdC5uYW1lfSBwYWNrYWdlTWFuYWdlciBkb2VzIG5vdCBtYXRjaCB0aGUgbW9ub3JlcG8gcGFja2FnZU1hbmFnZXI6ICR7dGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyfS5gXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogRm9yIG5vbi1ub2RlIHByb2plY3RzLCBhIHBhY2thZ2UuanNvbiBpcyByZXF1aXJlZCBpbiBvcmRlciB0byBiZSBkaXNjb3ZlcmVkIGJ5IE5YLlxuICAgKi9cbiAgcHJpdmF0ZSBzeW50aGVzaXplTm9uTm9kZVBhY2thZ2VKc29uKCkge1xuICAgIHRoaXMuc3ViUHJvamVjdHNcbiAgICAgIC5maWx0ZXIoKHN1YlByb2plY3Q6IGFueSkgPT4gIWlzTm9kZVByb2plY3Qoc3ViUHJvamVjdCkpXG4gICAgICAuZmlsdGVyKChzdWJQcm9qZWN0OiBQcm9qZWN0KSA9PiAhc3ViUHJvamVjdC50cnlGaW5kRmlsZShcInBhY2thZ2UuanNvblwiKSlcbiAgICAgIC5mb3JFYWNoKChzdWJQcm9qZWN0OiBQcm9qZWN0KSA9PiB7XG4gICAgICAgIC8vIGdlbmVyYXRlIGEgcGFja2FnZS5qc29uIGlmIG5vdCBmb3VuZFxuICAgICAgICBjb25zdCBtYW5pZmVzdDogYW55ID0ge1xuICAgICAgICAgIG5hbWU6IHN1YlByb2plY3QubmFtZSxcbiAgICAgICAgICBwcml2YXRlOiB0cnVlLFxuICAgICAgICAgIF9fcGRrX186IHRydWUsXG4gICAgICAgICAgc2NyaXB0czogc3ViUHJvamVjdC50YXNrcy5hbGwucmVkdWNlKFxuICAgICAgICAgICAgKHAsIGMpID0+ICh7XG4gICAgICAgICAgICAgIFtjLm5hbWVdOiBgbnB4IHByb2plbiAke2MubmFtZX1gLFxuICAgICAgICAgICAgICAuLi5wLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgICB7fVxuICAgICAgICAgICksXG4gICAgICAgICAgdmVyc2lvbjogXCIwLjAuMFwiLFxuICAgICAgICB9O1xuXG4gICAgICAgIG5ldyBKc29uRmlsZShzdWJQcm9qZWN0LCBcInBhY2thZ2UuanNvblwiLCB7XG4gICAgICAgICAgb2JqOiBtYW5pZmVzdCxcbiAgICAgICAgICByZWFkb25seTogdHJ1ZSxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBzdWJtb2R1bGUgZW50cnkgdG8gdGhlIGFwcHJvcHJpYXRlIHdvcmtzcGFjZSBmaWxlLlxuICAgKi9cbiAgcHJpdmF0ZSB1cGRhdGVXb3Jrc3BhY2UoKSB7XG4gICAgLy8gQSBmaW5hbCBjYWxsIHRvIGFkZFdvcmtzcGFjZVBhY2thZ2VzIHdpbGwgdXBkYXRlIHRoZSBsaXN0IG9mIHdvcmtzcGFjZSBwYWNrYWdlcyB3aXRoIGFueSBzdWJwcm9qZWN0cyB0aGF0IGhhdmVcbiAgICAvLyBub3QgeWV0IGJlZW4gYWRkZWQsIGluIHRoZSBjb3JyZWN0IG9yZGVyXG4gICAgdGhpcy5hZGRXb3Jrc3BhY2VQYWNrYWdlcygpO1xuXG4gICAgbGV0IG5vSG9pc3QgPSB0aGlzLndvcmtzcGFjZUNvbmZpZz8ubm9Ib2lzdDtcbiAgICAvLyBBdXRvbWF0aWNhbGx5IGFkZCBhbGwgc3ViLXByb2plY3QgXCJidW5kbGVkRGVwZW5kZW5jaWVzXCIgdG8gd29ya3NwYWNlIFwiaG9ob2lzdFwiLCBvdGhlcndpc2UgdGhleSBhcmUgbm90IGJ1bmRsZWQgaW4gbnBtIHBhY2thZ2VcbiAgICBpZiAodGhpcy53b3Jrc3BhY2VDb25maWc/LmRpc2FibGVOb0hvaXN0QnVuZGxlZCAhPT0gdHJ1ZSkge1xuICAgICAgY29uc3Qgbm9Ib2lzdEJ1bmRsZWQgPSB0aGlzLnN1YlByb2plY3RzLmZsYXRNYXAoKHN1YikgPT4ge1xuICAgICAgICBpZiAoc3ViIGluc3RhbmNlb2YgTm9kZVByb2plY3QpIHtcbiAgICAgICAgICByZXR1cm4gc3ViLmRlcHMuYWxsXG4gICAgICAgICAgICAuZmlsdGVyKChkZXApID0+IGRlcC50eXBlID09PSBEZXBlbmRlbmN5VHlwZS5CVU5ETEVEKVxuICAgICAgICAgICAgLmZsYXRNYXAoKGRlcCkgPT4gW1xuICAgICAgICAgICAgICBgJHtzdWIubmFtZX0vJHtkZXAubmFtZX1gLFxuICAgICAgICAgICAgICBgJHtzdWIubmFtZX0vJHtkZXAubmFtZX0vKmAsXG4gICAgICAgICAgICBdKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gW107XG4gICAgICB9KTtcblxuICAgICAgaWYgKG5vSG9pc3RCdW5kbGVkLmxlbmd0aCkge1xuICAgICAgICBub0hvaXN0ID0gWy4uLihub0hvaXN0IHx8IFtdKSwgLi4ubm9Ib2lzdEJ1bmRsZWRdO1xuICAgICAgfVxuICAgIH1cblxuICAgIC8vIEFkZCB3b3Jrc3BhY2VzIGZvciBlYWNoIHN1YnByb2plY3RcbiAgICBpZiAodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTSkge1xuICAgICAgbmV3IFlhbWxGaWxlKHRoaXMsIFwicG5wbS13b3Jrc3BhY2UueWFtbFwiLCB7XG4gICAgICAgIHJlYWRvbmx5OiB0cnVlLFxuICAgICAgICBvYmo6IHtcbiAgICAgICAgICBwYWNrYWdlczogdGhpcy53b3Jrc3BhY2VQYWNrYWdlcyxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnBhY2thZ2UuYWRkRmllbGQoXCJ3b3Jrc3BhY2VzXCIsIHtcbiAgICAgICAgcGFja2FnZXM6IHRoaXMud29ya3NwYWNlUGFja2FnZXMsXG4gICAgICAgIG5vaG9pc3Q6IG5vSG9pc3QsXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyB0aGUgaW5zdGFsbCB0YXNrIGZvciBweXRob24gcHJvamVjdHMgc28gdGhhdCB0aGV5IGFyZSBydW4gc2VyaWFsbHkgYW5kIGluIGRlcGVuZGVuY3kgb3JkZXIgc3VjaCB0aGF0IHB5dGhvblxuICAgKiBwcm9qZWN0cyB3aXRoaW4gdGhlIG1vbm9yZXBvIGNhbiBkZWNsYXJlIGRlcGVuZGVuY2llcyBvbiBvbmUgYW5vdGhlci5cbiAgICogQHByaXZhdGVcbiAgICovXG4gIHByaXZhdGUgd2lyZVB5dGhvbkRlcGVuZGVuY2llcygpIHtcbiAgICAvLyBGaW5kIGFueSBweXRob24gcHJvamVjdHNcbiAgICBjb25zdCBweXRob25Qcm9qZWN0cyA9IHRoaXMuc3ViUHJvamVjdHMuZmlsdGVyKFxuICAgICAgKHByb2plY3QpID0+IHByb2plY3QgaW5zdGFuY2VvZiBQeXRob25Qcm9qZWN0XG4gICAgKSBhcyBQeXRob25Qcm9qZWN0W107XG5cbiAgICBpZiAocHl0aG9uUHJvamVjdHMubGVuZ3RoID09PSAwKSB7XG4gICAgICAvLyBOb3RoaW5nIHRvIGRvIGZvciBubyBweXRob24gcHJvamVjdHNcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICAvLyBNb3ZlIGFsbCBpbnN0YWxsIGNvbW1hbmRzIHRvIGluc3RhbGwtcHkgc28gdGhhdCB0aGV5IGFyZSBub3QgaW5zdGFsbGVkIGluIHBhcmFsbGVsIGJ5IHRoZSBtb25vcmVwbyBwYWNrYWdlIG1hbmFnZXIuXG4gICAgLy8gZWcgeWFybiBpbnN0YWxsIHdpbGwgcnVuIHRoZSBpbnN0YWxsIHRhc2sgZm9yIGFsbCBwYWNrYWdlcyBpbiBwYXJhbGxlbCB3aGljaCBjYW4gbGVhZCB0byBjb25mbGljdHMgZm9yIHB5dGhvbi5cbiAgICBweXRob25Qcm9qZWN0cy5mb3JFYWNoKChweXRob25Qcm9qZWN0KSA9PiB7XG4gICAgICBjb25zdCBpbnN0YWxsUHlUYXNrID1cbiAgICAgICAgcHl0aG9uUHJvamVjdC50YXNrcy50cnlGaW5kKFwiaW5zdGFsbC1weVwiKSA/P1xuICAgICAgICBweXRob25Qcm9qZWN0LmFkZFRhc2soXCJpbnN0YWxsLXB5XCIpO1xuICAgICAgY29uc3QgaW5zdGFsbFRhc2sgPSBweXRob25Qcm9qZWN0LnRhc2tzLnRyeUZpbmQoXCJpbnN0YWxsXCIpO1xuXG4gICAgICAoaW5zdGFsbFRhc2s/LnN0ZXBzIHx8IFtdKS5mb3JFYWNoKChzdGVwKSA9PiB7XG4gICAgICAgIHRoaXMuY29weVN0ZXBJbnRvVGFzayhzdGVwLCBpbnN0YWxsUHlUYXNrLCBweXRob25Qcm9qZWN0KTtcbiAgICAgIH0pO1xuXG4gICAgICBpbnN0YWxsVGFzaz8ucmVzZXQoKTtcbiAgICB9KTtcblxuICAgIC8vIEFkZCBhbiBpbnN0YWxsIHRhc2sgdG8gdGhlIG1vbm9yZXBvIHRvIGluY2x1ZGUgcnVubmluZyB0aGUgaW5zdGFsbC1weSBjb21tYW5kIHNlcmlhbGx5IHRvIGF2b2lkIGNvbmZsaWN0aW5nIHdyaXRlc1xuICAgIC8vIHRvIGEgc2hhcmVkIHZpcnR1YWwgZW52LiBUaGlzIGlzIGFsc28gbWFuYWdlZCBieSBueCBzbyB0aGF0IGluc3RhbGxzIG9jY3VyIGluIGRlcGVuZGVuY3kgb3JkZXIuXG4gICAgY29uc3QgbW9ub3JlcG9JbnN0YWxsVGFzayA9XG4gICAgICB0aGlzLnRhc2tzLnRyeUZpbmQoXCJpbnN0YWxsXCIpID8/IHRoaXMuYWRkVGFzayhcImluc3RhbGxcIik7XG4gICAgbW9ub3JlcG9JbnN0YWxsVGFzay5leGVjKFxuICAgICAgYG5weCBueCBydW4tbWFueSAtLXRhcmdldCBpbnN0YWxsLXB5IC0tcHJvamVjdHMgJHtweXRob25Qcm9qZWN0c1xuICAgICAgICAubWFwKChwcm9qZWN0KSA9PiBwcm9qZWN0Lm5hbWUpXG4gICAgICAgIC5qb2luKFwiLFwiKX0gLS1wYXJhbGxlbD0xYFxuICAgICk7XG5cbiAgICAvLyBVcGRhdGUgdGhlIG54Lmpzb24gdG8gZW5zdXJlIHRoYXQgaW5zdGFsbC1weSBmb2xsb3dzIGRlcGVuZGVuY3kgb3JkZXJcbiAgICB0aGlzLm54SnNvbi5hZGRPdmVycmlkZShcInRhcmdldERlcGVuZGVuY2llcy5pbnN0YWxsLXB5XCIsIFtcbiAgICAgIHtcbiAgICAgICAgdGFyZ2V0OiBcImluc3RhbGwtcHlcIixcbiAgICAgICAgcHJvamVjdHM6IFwiZGVwZW5kZW5jaWVzXCIsXG4gICAgICB9LFxuICAgIF0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENvcGllcyB0aGUgZ2l2ZW4gc3RlcCBpbnRvIHRoZSBnaXZlbiB0YXNrLiBTdGVwIGFuZCBUYXNrIG11c3QgYmUgZnJvbSB0aGUgZ2l2ZW4gUHJvamVjdFxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgcHJpdmF0ZSBjb3B5U3RlcEludG9UYXNrKHN0ZXA6IFRhc2tTdGVwLCB0YXNrOiBUYXNrLCBwcm9qZWN0OiBQcm9qZWN0KSB7XG4gICAgaWYgKHN0ZXAuZXhlYykge1xuICAgICAgdGFzay5leGVjKHN0ZXAuZXhlYywgeyBuYW1lOiBzdGVwLm5hbWUsIGN3ZDogc3RlcC5jd2QgfSk7XG4gICAgfSBlbHNlIGlmIChzdGVwLnNheSkge1xuICAgICAgdGFzay5zYXkoc3RlcC5zYXksIHsgbmFtZTogc3RlcC5uYW1lLCBjd2Q6IHN0ZXAuY3dkIH0pO1xuICAgIH0gZWxzZSBpZiAoc3RlcC5zcGF3bikge1xuICAgICAgY29uc3Qgc3RlcFRvU3Bhd24gPSBwcm9qZWN0LnRhc2tzLnRyeUZpbmQoc3RlcC5zcGF3bik7XG4gICAgICBpZiAoc3RlcFRvU3Bhd24pIHtcbiAgICAgICAgdGFzay5zcGF3bihzdGVwVG9TcGF3biwgeyBuYW1lOiBzdGVwLm5hbWUsIGN3ZDogc3RlcC5jd2QgfSk7XG4gICAgICB9XG4gICAgfSBlbHNlIGlmIChzdGVwLmJ1aWx0aW4pIHtcbiAgICAgIHRhc2suYnVpbHRpbihzdGVwLmJ1aWx0aW4pO1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIERldGVybWluZXMgaWYgdGhlIHBhc3NlZCBpbiBwcm9qZWN0IGlzIG9mIHR5cGUgTm9kZVByb2plY3QuXG4gKlxuICogQHBhcmFtIHByb2plY3QgUHJvamVjdCBpbnN0YW5jZS5cbiAqIEByZXR1cm5zIHRydWUgaWYgdGhlIHByb2plY3QgaW5zdGFuY2UgaXMgb2YgdHlwZSBOb2RlUHJvamVjdC5cbiAqL1xuZnVuY3Rpb24gaXNOb2RlUHJvamVjdChwcm9qZWN0OiBhbnkpIHtcbiAgcmV0dXJuIHByb2plY3QgaW5zdGFuY2VvZiBOb2RlUHJvamVjdCB8fCBwcm9qZWN0LnBhY2thZ2U7XG59XG5cbmZ1bmN0aW9uIGdldFBsdWdpblBhdGgoKSB7XG4gIHJldHVybiBwYXRoLmpvaW4oX19kaXJuYW1lLCBcInBsdWdpbi9ueC1tb25vcmVwby1wbHVnaW4uanNcIik7XG59XG4iXX0=