# serpent.py - pure Python implementation of the Serpent algorithm.
# Bjorn Edstrom <be@bjrn.se> 13 december 2007.
#
# This code is a derived from an implementation by Dr Brian Gladman
# (gladman@seven77.demon.co.uk) which is subject to the following license.
# This Python implementation is not subject to any other license.
#
# This is an independent implementation of the encryption algorithm:
#
# Serpent by Ross Anderson, Eli Biham and Lars Knudsen
#
# which is a candidate algorithm in the Advanced Encryption Standard
# programme of the US National Institute of Standards and Technology
#
# Copyright in this implementation is held by Dr B R Gladman but I
# hereby give permission for its free direct or derivative use subject
# to acknowledgment of its origin and compliance with any conditions
# that the originators of the algorithm place on its exploitation.
#
# Dr Brian Gladman (gladman@seven77.demon.co.uk) 14th January 1999
#
# The above copyright notice must not be removed.
#
# Information
# ===========
#
# Anyone thinking of using this code should reconsider. It's slow.
# Try python-mcrypt instead. In case a faster library is not installed
# on the target system, this code can be used as a portable fallback.
import struct

from enum import Enum
from typing import Tuple, Optional, ByteString


def rotr32(x, n):
    return (x >> n) | ((x << (32 - n)) & 0xFFFFFFFF)


def rotl32(x, n):
    return ((x << n) & 0xFFFFFFFF) | (x >> (32 - n))


def _serpent_set_key(l_key, key, key_len):
    key_len *= 8
    if key_len > 256:
        return False
    i = 0
    lk = (key_len + 31) / 32
    while i < lk:
        l_key[i] = key[i]
        i += 1
    if key_len < 256:
        while i < 8:
            l_key[i] = 0
            i += 1
        i = key_len / 32
        lk = 1 << (key_len % 32)
        l_key[i] = (l_key[i] & (lk - 1)) | lk
    for i in range(132):
        lk = l_key[i] ^ l_key[i + 3] ^ l_key[i + 5] ^ l_key[i + 7] ^ 0x9e3779b9 ^ i
        l_key[i + 8] = ((lk << 11) & 0xFFFFFFFF) | (lk >> 21)
    key = l_key
    a = key[4 * 0 + 0x8]
    b = key[4 * 0 + 0x9]
    c = key[4 * 0 + 0xA]
    d = key[4 * 0 + 0xB]
    e = 0
    f = 0
    g = 0
    h = 0
    t1 = 0
    t2 = 0
    t3 = 0
    t4 = 0
    t5 = 0
    t6 = 0
    t7 = 0
    t8 = 0
    t9 = 0
    t10 = 0
    t11 = 0
    t12 = 0
    t13 = 0
    t14 = 0
    t15 = 0
    t16 = 0
    t1 = a ^ c
    t2 = d ^ t1
    t3 = a & t2
    t4 = d ^ t3
    t5 = b & t4
    g = t2 ^ t5
    t7 = a | g
    t8 = b | d
    t11 = a | d
    t9 = t4 & t7
    f = t8 ^ t9
    t12 = b ^ t11
    t13 = g ^ t9
    t15 = t3 ^ t8
    h = t12 ^ t13
    t16 = c & t15
    e = t12 ^ t16
    key[4 * 0 + 0x8] = e
    key[4 * 0 + 0x9] = f
    key[4 * 0 + 0xA] = g
    key[4 * 0 + 0xB] = h
    a = key[4 * 1 + 0x8]
    b = key[4 * 1 + 0x9]
    c = key[4 * 1 + 0xA]
    d = key[4 * 1 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ d
    t3 = c & t1
    t13 = d | t1
    e = t2 ^ t3
    t5 = c ^ t1
    t6 = c ^ e
    t7 = b & t6
    t10 = e | t5
    h = t5 ^ t7
    t9 = d | t7
    t11 = t9 & t10
    t14 = t2 ^ h
    g = a ^ t11
    t15 = g ^ t13
    f = t14 ^ t15
    key[4 * 1 + 0x8] = e
    key[4 * 1 + 0x9] = f
    key[4 * 1 + 0xA] = g
    key[4 * 1 + 0xB] = h
    a = key[4 * 2 + 0x8]
    b = key[4 * 2 + 0x9]
    c = key[4 * 2 + 0xA]
    d = key[4 * 2 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ t1
    t3 = a | t2
    t4 = d | t2
    t5 = c ^ t3
    g = d ^ t5
    t7 = b ^ t4
    t8 = t2 ^ g
    t9 = t5 & t7
    h = t8 ^ t9
    t11 = t5 ^ t7
    f = h ^ t11
    t13 = t8 & t11
    e = t5 ^ t13
    key[4 * 2 + 0x8] = e
    key[4 * 2 + 0x9] = f
    key[4 * 2 + 0xA] = g
    key[4 * 2 + 0xB] = h
    a = key[4 * 3 + 0x8]
    b = key[4 * 3 + 0x9]
    c = key[4 * 3 + 0xA]
    d = key[4 * 3 + 0xB]
    t1 = a ^ d
    t2 = a & d
    t3 = c ^ t1
    t6 = b & t1
    t4 = b ^ t3
    t10 = (~t3) % 0x100000000
    h = t2 ^ t4
    t7 = a ^ t6
    t14 = (~t7) % 0x100000000
    t8 = c | t7
    t11 = t3 ^ t7
    g = t4 ^ t8
    t12 = h & t11
    f = t10 ^ t12
    e = t12 ^ t14
    key[4 * 3 + 0x8] = e
    key[4 * 3 + 0x9] = f
    key[4 * 3 + 0xA] = g
    key[4 * 3 + 0xB] = h
    a = key[4 * 4 + 0x8]
    b = key[4 * 4 + 0x9]
    c = key[4 * 4 + 0xA]
    d = key[4 * 4 + 0xB]
    t1 = (~c) % 0x100000000
    t2 = b ^ c
    t3 = b | t1
    t4 = d ^ t3
    t5 = a & t4
    t7 = a ^ d
    h = t2 ^ t5
    t8 = b ^ t5
    t9 = t2 | t8
    t11 = d & t3
    f = t7 ^ t9
    t12 = t5 ^ f
    t15 = t1 | t4
    t13 = h & t12
    g = t11 ^ t13
    t16 = t12 ^ g
    e = t15 ^ t16
    key[4 * 4 + 0x8] = e
    key[4 * 4 + 0x9] = f
    key[4 * 4 + 0xA] = g
    key[4 * 4 + 0xB] = h
    a = key[4 * 5 + 0x8]
    b = key[4 * 5 + 0x9]
    c = key[4 * 5 + 0xA]
    d = key[4 * 5 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ d
    t3 = b ^ t2
    t4 = t1 | t2
    t5 = c ^ t4
    f = b ^ t5
    t13 = (~t5) % 0x100000000
    t7 = t2 | f
    t8 = d ^ t7
    t9 = t5 & t8
    g = t3 ^ t9
    t11 = t5 ^ t8
    e = g ^ t11
    t14 = t3 & t11
    h = t13 ^ t14
    key[4 * 5 + 0x8] = e
    key[4 * 5 + 0x9] = f
    key[4 * 5 + 0xA] = g
    key[4 * 5 + 0xB] = h
    a = key[4 * 6 + 0x8]
    b = key[4 * 6 + 0x9]
    c = key[4 * 6 + 0xA]
    d = key[4 * 6 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ b
    t3 = a ^ d
    t4 = c ^ t1
    t5 = t2 | t3
    e = t4 ^ t5
    t7 = d & e
    t8 = t2 ^ e
    t10 = t1 | e
    f = t7 ^ t8
    t11 = t2 | t7
    t12 = t3 ^ t10
    t14 = b ^ t7
    g = t11 ^ t12
    t15 = f & t12
    h = t14 ^ t15
    key[4 * 6 + 0x8] = e
    key[4 * 6 + 0x9] = f
    key[4 * 6 + 0xA] = g
    key[4 * 6 + 0xB] = h
    a = key[4 * 7 + 0x8]
    b = key[4 * 7 + 0x9]
    c = key[4 * 7 + 0xA]
    d = key[4 * 7 + 0xB]
    t1 = a ^ d
    t2 = d & t1
    t3 = c ^ t2
    t4 = b | t3
    h = t1 ^ t4
    t6 = (~b) % 0x100000000
    t7 = t1 | t6
    e = t3 ^ t7
    t9 = a & e
    t10 = t1 ^ t6
    t11 = t4 & t10
    g = t9 ^ t11
    t13 = a ^ t3
    t14 = t10 & g
    f = t13 ^ t14
    key[4 * 7 + 0x8] = e
    key[4 * 7 + 0x9] = f
    key[4 * 7 + 0xA] = g
    key[4 * 7 + 0xB] = h
    a = key[4 * 8 + 0x8]
    b = key[4 * 8 + 0x9]
    c = key[4 * 8 + 0xA]
    d = key[4 * 8 + 0xB]
    t1 = a ^ c
    t2 = d ^ t1
    t3 = a & t2
    t4 = d ^ t3
    t5 = b & t4
    g = t2 ^ t5
    t7 = a | g
    t8 = b | d
    t11 = a | d
    t9 = t4 & t7
    f = t8 ^ t9
    t12 = b ^ t11
    t13 = g ^ t9
    t15 = t3 ^ t8
    h = t12 ^ t13
    t16 = c & t15
    e = t12 ^ t16
    key[4 * 8 + 0x8] = e
    key[4 * 8 + 0x9] = f
    key[4 * 8 + 0xA] = g
    key[4 * 8 + 0xB] = h
    a = key[4 * 9 + 0x8]
    b = key[4 * 9 + 0x9]
    c = key[4 * 9 + 0xA]
    d = key[4 * 9 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ d
    t3 = c & t1
    t13 = d | t1
    e = t2 ^ t3
    t5 = c ^ t1
    t6 = c ^ e
    t7 = b & t6
    t10 = e | t5
    h = t5 ^ t7
    t9 = d | t7
    t11 = t9 & t10
    t14 = t2 ^ h
    g = a ^ t11
    t15 = g ^ t13
    f = t14 ^ t15
    key[4 * 9 + 0x8] = e
    key[4 * 9 + 0x9] = f
    key[4 * 9 + 0xA] = g
    key[4 * 9 + 0xB] = h
    a = key[4 * 10 + 0x8]
    b = key[4 * 10 + 0x9]
    c = key[4 * 10 + 0xA]
    d = key[4 * 10 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ t1
    t3 = a | t2
    t4 = d | t2
    t5 = c ^ t3
    g = d ^ t5
    t7 = b ^ t4
    t8 = t2 ^ g
    t9 = t5 & t7
    h = t8 ^ t9
    t11 = t5 ^ t7
    f = h ^ t11
    t13 = t8 & t11
    e = t5 ^ t13
    key[4 * 10 + 0x8] = e
    key[4 * 10 + 0x9] = f
    key[4 * 10 + 0xA] = g
    key[4 * 10 + 0xB] = h
    a = key[4 * 11 + 0x8]
    b = key[4 * 11 + 0x9]
    c = key[4 * 11 + 0xA]
    d = key[4 * 11 + 0xB]
    t1 = a ^ d
    t2 = a & d
    t3 = c ^ t1
    t6 = b & t1
    t4 = b ^ t3
    t10 = (~t3) % 0x100000000
    h = t2 ^ t4
    t7 = a ^ t6
    t14 = (~t7) % 0x100000000
    t8 = c | t7
    t11 = t3 ^ t7
    g = t4 ^ t8
    t12 = h & t11
    f = t10 ^ t12
    e = t12 ^ t14
    key[4 * 11 + 0x8] = e
    key[4 * 11 + 0x9] = f
    key[4 * 11 + 0xA] = g
    key[4 * 11 + 0xB] = h
    a = key[4 * 12 + 0x8]
    b = key[4 * 12 + 0x9]
    c = key[4 * 12 + 0xA]
    d = key[4 * 12 + 0xB]
    t1 = (~c) % 0x100000000
    t2 = b ^ c
    t3 = b | t1
    t4 = d ^ t3
    t5 = a & t4
    t7 = a ^ d
    h = t2 ^ t5
    t8 = b ^ t5
    t9 = t2 | t8
    t11 = d & t3
    f = t7 ^ t9
    t12 = t5 ^ f
    t15 = t1 | t4
    t13 = h & t12
    g = t11 ^ t13
    t16 = t12 ^ g
    e = t15 ^ t16
    key[4 * 12 + 0x8] = e
    key[4 * 12 + 0x9] = f
    key[4 * 12 + 0xA] = g
    key[4 * 12 + 0xB] = h
    a = key[4 * 13 + 0x8]
    b = key[4 * 13 + 0x9]
    c = key[4 * 13 + 0xA]
    d = key[4 * 13 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ d
    t3 = b ^ t2
    t4 = t1 | t2
    t5 = c ^ t4
    f = b ^ t5
    t13 = (~t5) % 0x100000000
    t7 = t2 | f
    t8 = d ^ t7
    t9 = t5 & t8
    g = t3 ^ t9
    t11 = t5 ^ t8
    e = g ^ t11
    t14 = t3 & t11
    h = t13 ^ t14
    key[4 * 13 + 0x8] = e
    key[4 * 13 + 0x9] = f
    key[4 * 13 + 0xA] = g
    key[4 * 13 + 0xB] = h
    a = key[4 * 14 + 0x8]
    b = key[4 * 14 + 0x9]
    c = key[4 * 14 + 0xA]
    d = key[4 * 14 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ b
    t3 = a ^ d
    t4 = c ^ t1
    t5 = t2 | t3
    e = t4 ^ t5
    t7 = d & e
    t8 = t2 ^ e
    t10 = t1 | e
    f = t7 ^ t8
    t11 = t2 | t7
    t12 = t3 ^ t10
    t14 = b ^ t7
    g = t11 ^ t12
    t15 = f & t12
    h = t14 ^ t15
    key[4 * 14 + 0x8] = e
    key[4 * 14 + 0x9] = f
    key[4 * 14 + 0xA] = g
    key[4 * 14 + 0xB] = h
    a = key[4 * 15 + 0x8]
    b = key[4 * 15 + 0x9]
    c = key[4 * 15 + 0xA]
    d = key[4 * 15 + 0xB]
    t1 = a ^ d
    t2 = d & t1
    t3 = c ^ t2
    t4 = b | t3
    h = t1 ^ t4
    t6 = (~b) % 0x100000000
    t7 = t1 | t6
    e = t3 ^ t7
    t9 = a & e
    t10 = t1 ^ t6
    t11 = t4 & t10
    g = t9 ^ t11
    t13 = a ^ t3
    t14 = t10 & g
    f = t13 ^ t14
    key[4 * 15 + 0x8] = e
    key[4 * 15 + 0x9] = f
    key[4 * 15 + 0xA] = g
    key[4 * 15 + 0xB] = h
    a = key[4 * 16 + 0x8]
    b = key[4 * 16 + 0x9]
    c = key[4 * 16 + 0xA]
    d = key[4 * 16 + 0xB]
    t1 = a ^ c
    t2 = d ^ t1
    t3 = a & t2
    t4 = d ^ t3
    t5 = b & t4
    g = t2 ^ t5
    t7 = a | g
    t8 = b | d
    t11 = a | d
    t9 = t4 & t7
    f = t8 ^ t9
    t12 = b ^ t11
    t13 = g ^ t9
    t15 = t3 ^ t8
    h = t12 ^ t13
    t16 = c & t15
    e = t12 ^ t16
    key[4 * 16 + 0x8] = e
    key[4 * 16 + 0x9] = f
    key[4 * 16 + 0xA] = g
    key[4 * 16 + 0xB] = h
    a = key[4 * 17 + 0x8]
    b = key[4 * 17 + 0x9]
    c = key[4 * 17 + 0xA]
    d = key[4 * 17 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ d
    t3 = c & t1
    t13 = d | t1
    e = t2 ^ t3
    t5 = c ^ t1
    t6 = c ^ e
    t7 = b & t6
    t10 = e | t5
    h = t5 ^ t7
    t9 = d | t7
    t11 = t9 & t10
    t14 = t2 ^ h
    g = a ^ t11
    t15 = g ^ t13
    f = t14 ^ t15
    key[4 * 17 + 0x8] = e
    key[4 * 17 + 0x9] = f
    key[4 * 17 + 0xA] = g
    key[4 * 17 + 0xB] = h
    a = key[4 * 18 + 0x8]
    b = key[4 * 18 + 0x9]
    c = key[4 * 18 + 0xA]
    d = key[4 * 18 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ t1
    t3 = a | t2
    t4 = d | t2
    t5 = c ^ t3
    g = d ^ t5
    t7 = b ^ t4
    t8 = t2 ^ g
    t9 = t5 & t7
    h = t8 ^ t9
    t11 = t5 ^ t7
    f = h ^ t11
    t13 = t8 & t11
    e = t5 ^ t13
    key[4 * 18 + 0x8] = e
    key[4 * 18 + 0x9] = f
    key[4 * 18 + 0xA] = g
    key[4 * 18 + 0xB] = h
    a = key[4 * 19 + 0x8]
    b = key[4 * 19 + 0x9]
    c = key[4 * 19 + 0xA]
    d = key[4 * 19 + 0xB]
    t1 = a ^ d
    t2 = a & d
    t3 = c ^ t1
    t6 = b & t1
    t4 = b ^ t3
    t10 = (~t3) % 0x100000000
    h = t2 ^ t4
    t7 = a ^ t6
    t14 = (~t7) % 0x100000000
    t8 = c | t7
    t11 = t3 ^ t7
    g = t4 ^ t8
    t12 = h & t11
    f = t10 ^ t12
    e = t12 ^ t14
    key[4 * 19 + 0x8] = e
    key[4 * 19 + 0x9] = f
    key[4 * 19 + 0xA] = g
    key[4 * 19 + 0xB] = h
    a = key[4 * 20 + 0x8]
    b = key[4 * 20 + 0x9]
    c = key[4 * 20 + 0xA]
    d = key[4 * 20 + 0xB]
    t1 = (~c) % 0x100000000
    t2 = b ^ c
    t3 = b | t1
    t4 = d ^ t3
    t5 = a & t4
    t7 = a ^ d
    h = t2 ^ t5
    t8 = b ^ t5
    t9 = t2 | t8
    t11 = d & t3
    f = t7 ^ t9
    t12 = t5 ^ f
    t15 = t1 | t4
    t13 = h & t12
    g = t11 ^ t13
    t16 = t12 ^ g
    e = t15 ^ t16
    key[4 * 20 + 0x8] = e
    key[4 * 20 + 0x9] = f
    key[4 * 20 + 0xA] = g
    key[4 * 20 + 0xB] = h
    a = key[4 * 21 + 0x8]
    b = key[4 * 21 + 0x9]
    c = key[4 * 21 + 0xA]
    d = key[4 * 21 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ d
    t3 = b ^ t2
    t4 = t1 | t2
    t5 = c ^ t4
    f = b ^ t5
    t13 = (~t5) % 0x100000000
    t7 = t2 | f
    t8 = d ^ t7
    t9 = t5 & t8
    g = t3 ^ t9
    t11 = t5 ^ t8
    e = g ^ t11
    t14 = t3 & t11
    h = t13 ^ t14
    key[4 * 21 + 0x8] = e
    key[4 * 21 + 0x9] = f
    key[4 * 21 + 0xA] = g
    key[4 * 21 + 0xB] = h
    a = key[4 * 22 + 0x8]
    b = key[4 * 22 + 0x9]
    c = key[4 * 22 + 0xA]
    d = key[4 * 22 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ b
    t3 = a ^ d
    t4 = c ^ t1
    t5 = t2 | t3
    e = t4 ^ t5
    t7 = d & e
    t8 = t2 ^ e
    t10 = t1 | e
    f = t7 ^ t8
    t11 = t2 | t7
    t12 = t3 ^ t10
    t14 = b ^ t7
    g = t11 ^ t12
    t15 = f & t12
    h = t14 ^ t15
    key[4 * 22 + 0x8] = e
    key[4 * 22 + 0x9] = f
    key[4 * 22 + 0xA] = g
    key[4 * 22 + 0xB] = h
    a = key[4 * 23 + 0x8]
    b = key[4 * 23 + 0x9]
    c = key[4 * 23 + 0xA]
    d = key[4 * 23 + 0xB]
    t1 = a ^ d
    t2 = d & t1
    t3 = c ^ t2
    t4 = b | t3
    h = t1 ^ t4
    t6 = (~b) % 0x100000000
    t7 = t1 | t6
    e = t3 ^ t7
    t9 = a & e
    t10 = t1 ^ t6
    t11 = t4 & t10
    g = t9 ^ t11
    t13 = a ^ t3
    t14 = t10 & g
    f = t13 ^ t14
    key[4 * 23 + 0x8] = e
    key[4 * 23 + 0x9] = f
    key[4 * 23 + 0xA] = g
    key[4 * 23 + 0xB] = h
    a = key[4 * 24 + 0x8]
    b = key[4 * 24 + 0x9]
    c = key[4 * 24 + 0xA]
    d = key[4 * 24 + 0xB]
    t1 = a ^ c
    t2 = d ^ t1
    t3 = a & t2
    t4 = d ^ t3
    t5 = b & t4
    g = t2 ^ t5
    t7 = a | g
    t8 = b | d
    t11 = a | d
    t9 = t4 & t7
    f = t8 ^ t9
    t12 = b ^ t11
    t13 = g ^ t9
    t15 = t3 ^ t8
    h = t12 ^ t13
    t16 = c & t15
    e = t12 ^ t16
    key[4 * 24 + 0x8] = e
    key[4 * 24 + 0x9] = f
    key[4 * 24 + 0xA] = g
    key[4 * 24 + 0xB] = h
    a = key[4 * 25 + 0x8]
    b = key[4 * 25 + 0x9]
    c = key[4 * 25 + 0xA]
    d = key[4 * 25 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ d
    t3 = c & t1
    t13 = d | t1
    e = t2 ^ t3
    t5 = c ^ t1
    t6 = c ^ e
    t7 = b & t6
    t10 = e | t5
    h = t5 ^ t7
    t9 = d | t7
    t11 = t9 & t10
    t14 = t2 ^ h
    g = a ^ t11
    t15 = g ^ t13
    f = t14 ^ t15
    key[4 * 25 + 0x8] = e
    key[4 * 25 + 0x9] = f
    key[4 * 25 + 0xA] = g
    key[4 * 25 + 0xB] = h
    a = key[4 * 26 + 0x8]
    b = key[4 * 26 + 0x9]
    c = key[4 * 26 + 0xA]
    d = key[4 * 26 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ t1
    t3 = a | t2
    t4 = d | t2
    t5 = c ^ t3
    g = d ^ t5
    t7 = b ^ t4
    t8 = t2 ^ g
    t9 = t5 & t7
    h = t8 ^ t9
    t11 = t5 ^ t7
    f = h ^ t11
    t13 = t8 & t11
    e = t5 ^ t13
    key[4 * 26 + 0x8] = e
    key[4 * 26 + 0x9] = f
    key[4 * 26 + 0xA] = g
    key[4 * 26 + 0xB] = h
    a = key[4 * 27 + 0x8]
    b = key[4 * 27 + 0x9]
    c = key[4 * 27 + 0xA]
    d = key[4 * 27 + 0xB]
    t1 = a ^ d
    t2 = a & d
    t3 = c ^ t1
    t6 = b & t1
    t4 = b ^ t3
    t10 = (~t3) % 0x100000000
    h = t2 ^ t4
    t7 = a ^ t6
    t14 = (~t7) % 0x100000000
    t8 = c | t7
    t11 = t3 ^ t7
    g = t4 ^ t8
    t12 = h & t11
    f = t10 ^ t12
    e = t12 ^ t14
    key[4 * 27 + 0x8] = e
    key[4 * 27 + 0x9] = f
    key[4 * 27 + 0xA] = g
    key[4 * 27 + 0xB] = h
    a = key[4 * 28 + 0x8]
    b = key[4 * 28 + 0x9]
    c = key[4 * 28 + 0xA]
    d = key[4 * 28 + 0xB]
    t1 = (~c) % 0x100000000
    t2 = b ^ c
    t3 = b | t1
    t4 = d ^ t3
    t5 = a & t4
    t7 = a ^ d
    h = t2 ^ t5
    t8 = b ^ t5
    t9 = t2 | t8
    t11 = d & t3
    f = t7 ^ t9
    t12 = t5 ^ f
    t15 = t1 | t4
    t13 = h & t12
    g = t11 ^ t13
    t16 = t12 ^ g
    e = t15 ^ t16
    key[4 * 28 + 0x8] = e
    key[4 * 28 + 0x9] = f
    key[4 * 28 + 0xA] = g
    key[4 * 28 + 0xB] = h
    a = key[4 * 29 + 0x8]
    b = key[4 * 29 + 0x9]
    c = key[4 * 29 + 0xA]
    d = key[4 * 29 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ d
    t3 = b ^ t2
    t4 = t1 | t2
    t5 = c ^ t4
    f = b ^ t5
    t13 = (~t5) % 0x100000000
    t7 = t2 | f
    t8 = d ^ t7
    t9 = t5 & t8
    g = t3 ^ t9
    t11 = t5 ^ t8
    e = g ^ t11
    t14 = t3 & t11
    h = t13 ^ t14
    key[4 * 29 + 0x8] = e
    key[4 * 29 + 0x9] = f
    key[4 * 29 + 0xA] = g
    key[4 * 29 + 0xB] = h
    a = key[4 * 30 + 0x8]
    b = key[4 * 30 + 0x9]
    c = key[4 * 30 + 0xA]
    d = key[4 * 30 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ b
    t3 = a ^ d
    t4 = c ^ t1
    t5 = t2 | t3
    e = t4 ^ t5
    t7 = d & e
    t8 = t2 ^ e
    t10 = t1 | e
    f = t7 ^ t8
    t11 = t2 | t7
    t12 = t3 ^ t10
    t14 = b ^ t7
    g = t11 ^ t12
    t15 = f & t12
    h = t14 ^ t15
    key[4 * 30 + 0x8] = e
    key[4 * 30 + 0x9] = f
    key[4 * 30 + 0xA] = g
    key[4 * 30 + 0xB] = h
    a = key[4 * 31 + 0x8]
    b = key[4 * 31 + 0x9]
    c = key[4 * 31 + 0xA]
    d = key[4 * 31 + 0xB]
    t1 = a ^ d
    t2 = d & t1
    t3 = c ^ t2
    t4 = b | t3
    h = t1 ^ t4
    t6 = (~b) % 0x100000000
    t7 = t1 | t6
    e = t3 ^ t7
    t9 = a & e
    t10 = t1 ^ t6
    t11 = t4 & t10
    g = t9 ^ t11
    t13 = a ^ t3
    t14 = t10 & g
    f = t13 ^ t14
    key[4 * 31 + 0x8] = e
    key[4 * 31 + 0x9] = f
    key[4 * 31 + 0xA] = g
    key[4 * 31 + 0xB] = h
    a = key[4 * 32 + 0x8]
    b = key[4 * 32 + 0x9]
    c = key[4 * 32 + 0xA]
    d = key[4 * 32 + 0xB]
    t1 = a ^ c
    t2 = d ^ t1
    t3 = a & t2
    t4 = d ^ t3
    t5 = b & t4
    g = t2 ^ t5
    t7 = a | g
    t8 = b | d
    t11 = a | d
    t9 = t4 & t7
    f = t8 ^ t9
    t12 = b ^ t11
    t13 = g ^ t9
    t15 = t3 ^ t8
    h = t12 ^ t13
    t16 = c & t15
    e = t12 ^ t16
    key[4 * 32 + 0x8] = e
    key[4 * 32 + 0x9] = f
    key[4 * 32 + 0xA] = g
    key[4 * 32 + 0xB] = h


def _serpent_encrypt(key, in_blk: Tuple[int, int, int, int]):
    a = in_blk[0]
    b = in_blk[1]
    c = in_blk[2]
    d = in_blk[3]
    e = 0
    f = 0
    g = 0
    h = 0
    t1 = 0
    t2 = 0
    t3 = 0
    t4 = 0
    t5 = 0
    t6 = 0
    t7 = 0
    t8 = 0
    t9 = 0
    t10 = 0
    t11 = 0
    t12 = 0
    t13 = 0
    t14 = 0
    t15 = 0
    t16 = 0
    a ^= key[4 * 0 + 0x8]
    b ^= key[4 * 0 + 0x9]
    c ^= key[4 * 0 + 0xA]
    d ^= key[4 * 0 + 0xB]
    t1 = a ^ d
    t2 = a & d
    t3 = c ^ t1
    t6 = b & t1
    t4 = b ^ t3
    t10 = (~t3) % 0x100000000
    h = t2 ^ t4
    t7 = a ^ t6
    t14 = (~t7) % 0x100000000
    t8 = c | t7
    t11 = t3 ^ t7
    g = t4 ^ t8
    t12 = h & t11
    f = t10 ^ t12
    e = t12 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 1 + 0x8]
    f ^= key[4 * 1 + 0x9]
    g ^= key[4 * 1 + 0xA]
    h ^= key[4 * 1 + 0xB]
    t1 = (~e) % 0x100000000
    t2 = f ^ t1
    t3 = e | t2
    t4 = h | t2
    t5 = g ^ t3
    c = h ^ t5
    t7 = f ^ t4
    t8 = t2 ^ c
    t9 = t5 & t7
    d = t8 ^ t9
    t11 = t5 ^ t7
    b = d ^ t11
    t13 = t8 & t11
    a = t5 ^ t13
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 2 + 0x8]
    b ^= key[4 * 2 + 0x9]
    c ^= key[4 * 2 + 0xA]
    d ^= key[4 * 2 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ d
    t3 = c & t1
    t13 = d | t1
    e = t2 ^ t3
    t5 = c ^ t1
    t6 = c ^ e
    t7 = b & t6
    t10 = e | t5
    h = t5 ^ t7
    t9 = d | t7
    t11 = t9 & t10
    t14 = t2 ^ h
    g = a ^ t11
    t15 = g ^ t13
    f = t14 ^ t15
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 3 + 0x8]
    f ^= key[4 * 3 + 0x9]
    g ^= key[4 * 3 + 0xA]
    h ^= key[4 * 3 + 0xB]
    t1 = e ^ g
    t2 = h ^ t1
    t3 = e & t2
    t4 = h ^ t3
    t5 = f & t4
    c = t2 ^ t5
    t7 = e | c
    t8 = f | h
    t11 = e | h
    t9 = t4 & t7
    b = t8 ^ t9
    t12 = f ^ t11
    t13 = c ^ t9
    t15 = t3 ^ t8
    d = t12 ^ t13
    t16 = g & t15
    a = t12 ^ t16
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 4 + 0x8]
    b ^= key[4 * 4 + 0x9]
    c ^= key[4 * 4 + 0xA]
    d ^= key[4 * 4 + 0xB]
    t1 = a ^ d
    t2 = d & t1
    t3 = c ^ t2
    t4 = b | t3
    h = t1 ^ t4
    t6 = (~b) % 0x100000000
    t7 = t1 | t6
    e = t3 ^ t7
    t9 = a & e
    t10 = t1 ^ t6
    t11 = t4 & t10
    g = t9 ^ t11
    t13 = a ^ t3
    t14 = t10 & g
    f = t13 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 5 + 0x8]
    f ^= key[4 * 5 + 0x9]
    g ^= key[4 * 5 + 0xA]
    h ^= key[4 * 5 + 0xB]
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = e ^ h
    t4 = g ^ t1
    t5 = t2 | t3
    a = t4 ^ t5
    t7 = h & a
    t8 = t2 ^ a
    t10 = t1 | a
    b = t7 ^ t8
    t11 = t2 | t7
    t12 = t3 ^ t10
    t14 = f ^ t7
    c = t11 ^ t12
    t15 = b & t12
    d = t14 ^ t15
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 6 + 0x8]
    b ^= key[4 * 6 + 0x9]
    c ^= key[4 * 6 + 0xA]
    d ^= key[4 * 6 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ d
    t3 = b ^ t2
    t4 = t1 | t2
    t5 = c ^ t4
    f = b ^ t5
    t13 = (~t5) % 0x100000000
    t7 = t2 | f
    t8 = d ^ t7
    t9 = t5 & t8
    g = t3 ^ t9
    t11 = t5 ^ t8
    e = g ^ t11
    t14 = t3 & t11
    h = t13 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 7 + 0x8]
    f ^= key[4 * 7 + 0x9]
    g ^= key[4 * 7 + 0xA]
    h ^= key[4 * 7 + 0xB]
    t1 = (~g) % 0x100000000
    t2 = f ^ g
    t3 = f | t1
    t4 = h ^ t3
    t5 = e & t4
    t7 = e ^ h
    d = t2 ^ t5
    t8 = f ^ t5
    t9 = t2 | t8
    t11 = h & t3
    b = t7 ^ t9
    t12 = t5 ^ b
    t15 = t1 | t4
    t13 = d & t12
    c = t11 ^ t13
    t16 = t12 ^ c
    a = t15 ^ t16
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 8 + 0x8]
    b ^= key[4 * 8 + 0x9]
    c ^= key[4 * 8 + 0xA]
    d ^= key[4 * 8 + 0xB]
    t1 = a ^ d
    t2 = a & d
    t3 = c ^ t1
    t6 = b & t1
    t4 = b ^ t3
    t10 = (~t3) % 0x100000000
    h = t2 ^ t4
    t7 = a ^ t6
    t14 = (~t7) % 0x100000000
    t8 = c | t7
    t11 = t3 ^ t7
    g = t4 ^ t8
    t12 = h & t11
    f = t10 ^ t12
    e = t12 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 9 + 0x8]
    f ^= key[4 * 9 + 0x9]
    g ^= key[4 * 9 + 0xA]
    h ^= key[4 * 9 + 0xB]
    t1 = (~e) % 0x100000000
    t2 = f ^ t1
    t3 = e | t2
    t4 = h | t2
    t5 = g ^ t3
    c = h ^ t5
    t7 = f ^ t4
    t8 = t2 ^ c
    t9 = t5 & t7
    d = t8 ^ t9
    t11 = t5 ^ t7
    b = d ^ t11
    t13 = t8 & t11
    a = t5 ^ t13
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 10 + 0x8]
    b ^= key[4 * 10 + 0x9]
    c ^= key[4 * 10 + 0xA]
    d ^= key[4 * 10 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ d
    t3 = c & t1
    t13 = d | t1
    e = t2 ^ t3
    t5 = c ^ t1
    t6 = c ^ e
    t7 = b & t6
    t10 = e | t5
    h = t5 ^ t7
    t9 = d | t7
    t11 = t9 & t10
    t14 = t2 ^ h
    g = a ^ t11
    t15 = g ^ t13
    f = t14 ^ t15
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 11 + 0x8]
    f ^= key[4 * 11 + 0x9]
    g ^= key[4 * 11 + 0xA]
    h ^= key[4 * 11 + 0xB]
    t1 = e ^ g
    t2 = h ^ t1
    t3 = e & t2
    t4 = h ^ t3
    t5 = f & t4
    c = t2 ^ t5
    t7 = e | c
    t8 = f | h
    t11 = e | h
    t9 = t4 & t7
    b = t8 ^ t9
    t12 = f ^ t11
    t13 = c ^ t9
    t15 = t3 ^ t8
    d = t12 ^ t13
    t16 = g & t15
    a = t12 ^ t16
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 12 + 0x8]
    b ^= key[4 * 12 + 0x9]
    c ^= key[4 * 12 + 0xA]
    d ^= key[4 * 12 + 0xB]
    t1 = a ^ d
    t2 = d & t1
    t3 = c ^ t2
    t4 = b | t3
    h = t1 ^ t4
    t6 = (~b) % 0x100000000
    t7 = t1 | t6
    e = t3 ^ t7
    t9 = a & e
    t10 = t1 ^ t6
    t11 = t4 & t10
    g = t9 ^ t11
    t13 = a ^ t3
    t14 = t10 & g
    f = t13 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 13 + 0x8]
    f ^= key[4 * 13 + 0x9]
    g ^= key[4 * 13 + 0xA]
    h ^= key[4 * 13 + 0xB]
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = e ^ h
    t4 = g ^ t1
    t5 = t2 | t3
    a = t4 ^ t5
    t7 = h & a
    t8 = t2 ^ a
    t10 = t1 | a
    b = t7 ^ t8
    t11 = t2 | t7
    t12 = t3 ^ t10
    t14 = f ^ t7
    c = t11 ^ t12
    t15 = b & t12
    d = t14 ^ t15
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 14 + 0x8]
    b ^= key[4 * 14 + 0x9]
    c ^= key[4 * 14 + 0xA]
    d ^= key[4 * 14 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ d
    t3 = b ^ t2
    t4 = t1 | t2
    t5 = c ^ t4
    f = b ^ t5
    t13 = (~t5) % 0x100000000
    t7 = t2 | f
    t8 = d ^ t7
    t9 = t5 & t8
    g = t3 ^ t9
    t11 = t5 ^ t8
    e = g ^ t11
    t14 = t3 & t11
    h = t13 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 15 + 0x8]
    f ^= key[4 * 15 + 0x9]
    g ^= key[4 * 15 + 0xA]
    h ^= key[4 * 15 + 0xB]
    t1 = (~g) % 0x100000000
    t2 = f ^ g
    t3 = f | t1
    t4 = h ^ t3
    t5 = e & t4
    t7 = e ^ h
    d = t2 ^ t5
    t8 = f ^ t5
    t9 = t2 | t8
    t11 = h & t3
    b = t7 ^ t9
    t12 = t5 ^ b
    t15 = t1 | t4
    t13 = d & t12
    c = t11 ^ t13
    t16 = t12 ^ c
    a = t15 ^ t16
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 16 + 0x8]
    b ^= key[4 * 16 + 0x9]
    c ^= key[4 * 16 + 0xA]
    d ^= key[4 * 16 + 0xB]
    t1 = a ^ d
    t2 = a & d
    t3 = c ^ t1
    t6 = b & t1
    t4 = b ^ t3
    t10 = (~t3) % 0x100000000
    h = t2 ^ t4
    t7 = a ^ t6
    t14 = (~t7) % 0x100000000
    t8 = c | t7
    t11 = t3 ^ t7
    g = t4 ^ t8
    t12 = h & t11
    f = t10 ^ t12
    e = t12 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 17 + 0x8]
    f ^= key[4 * 17 + 0x9]
    g ^= key[4 * 17 + 0xA]
    h ^= key[4 * 17 + 0xB]
    t1 = (~e) % 0x100000000
    t2 = f ^ t1
    t3 = e | t2
    t4 = h | t2
    t5 = g ^ t3
    c = h ^ t5
    t7 = f ^ t4
    t8 = t2 ^ c
    t9 = t5 & t7
    d = t8 ^ t9
    t11 = t5 ^ t7
    b = d ^ t11
    t13 = t8 & t11
    a = t5 ^ t13
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 18 + 0x8]
    b ^= key[4 * 18 + 0x9]
    c ^= key[4 * 18 + 0xA]
    d ^= key[4 * 18 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ d
    t3 = c & t1
    t13 = d | t1
    e = t2 ^ t3
    t5 = c ^ t1
    t6 = c ^ e
    t7 = b & t6
    t10 = e | t5
    h = t5 ^ t7
    t9 = d | t7
    t11 = t9 & t10
    t14 = t2 ^ h
    g = a ^ t11
    t15 = g ^ t13
    f = t14 ^ t15
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 19 + 0x8]
    f ^= key[4 * 19 + 0x9]
    g ^= key[4 * 19 + 0xA]
    h ^= key[4 * 19 + 0xB]
    t1 = e ^ g
    t2 = h ^ t1
    t3 = e & t2
    t4 = h ^ t3
    t5 = f & t4
    c = t2 ^ t5
    t7 = e | c
    t8 = f | h
    t11 = e | h
    t9 = t4 & t7
    b = t8 ^ t9
    t12 = f ^ t11
    t13 = c ^ t9
    t15 = t3 ^ t8
    d = t12 ^ t13
    t16 = g & t15
    a = t12 ^ t16
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 20 + 0x8]
    b ^= key[4 * 20 + 0x9]
    c ^= key[4 * 20 + 0xA]
    d ^= key[4 * 20 + 0xB]
    t1 = a ^ d
    t2 = d & t1
    t3 = c ^ t2
    t4 = b | t3
    h = t1 ^ t4
    t6 = (~b) % 0x100000000
    t7 = t1 | t6
    e = t3 ^ t7
    t9 = a & e
    t10 = t1 ^ t6
    t11 = t4 & t10
    g = t9 ^ t11
    t13 = a ^ t3
    t14 = t10 & g
    f = t13 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 21 + 0x8]
    f ^= key[4 * 21 + 0x9]
    g ^= key[4 * 21 + 0xA]
    h ^= key[4 * 21 + 0xB]
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = e ^ h
    t4 = g ^ t1
    t5 = t2 | t3
    a = t4 ^ t5
    t7 = h & a
    t8 = t2 ^ a
    t10 = t1 | a
    b = t7 ^ t8
    t11 = t2 | t7
    t12 = t3 ^ t10
    t14 = f ^ t7
    c = t11 ^ t12
    t15 = b & t12
    d = t14 ^ t15
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 22 + 0x8]
    b ^= key[4 * 22 + 0x9]
    c ^= key[4 * 22 + 0xA]
    d ^= key[4 * 22 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ d
    t3 = b ^ t2
    t4 = t1 | t2
    t5 = c ^ t4
    f = b ^ t5
    t13 = (~t5) % 0x100000000
    t7 = t2 | f
    t8 = d ^ t7
    t9 = t5 & t8
    g = t3 ^ t9
    t11 = t5 ^ t8
    e = g ^ t11
    t14 = t3 & t11
    h = t13 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 23 + 0x8]
    f ^= key[4 * 23 + 0x9]
    g ^= key[4 * 23 + 0xA]
    h ^= key[4 * 23 + 0xB]
    t1 = (~g) % 0x100000000
    t2 = f ^ g
    t3 = f | t1
    t4 = h ^ t3
    t5 = e & t4
    t7 = e ^ h
    d = t2 ^ t5
    t8 = f ^ t5
    t9 = t2 | t8
    t11 = h & t3
    b = t7 ^ t9
    t12 = t5 ^ b
    t15 = t1 | t4
    t13 = d & t12
    c = t11 ^ t13
    t16 = t12 ^ c
    a = t15 ^ t16
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 24 + 0x8]
    b ^= key[4 * 24 + 0x9]
    c ^= key[4 * 24 + 0xA]
    d ^= key[4 * 24 + 0xB]
    t1 = a ^ d
    t2 = a & d
    t3 = c ^ t1
    t6 = b & t1
    t4 = b ^ t3
    t10 = (~t3) % 0x100000000
    h = t2 ^ t4
    t7 = a ^ t6
    t14 = (~t7) % 0x100000000
    t8 = c | t7
    t11 = t3 ^ t7
    g = t4 ^ t8
    t12 = h & t11
    f = t10 ^ t12
    e = t12 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 25 + 0x8]
    f ^= key[4 * 25 + 0x9]
    g ^= key[4 * 25 + 0xA]
    h ^= key[4 * 25 + 0xB]
    t1 = (~e) % 0x100000000
    t2 = f ^ t1
    t3 = e | t2
    t4 = h | t2
    t5 = g ^ t3
    c = h ^ t5
    t7 = f ^ t4
    t8 = t2 ^ c
    t9 = t5 & t7
    d = t8 ^ t9
    t11 = t5 ^ t7
    b = d ^ t11
    t13 = t8 & t11
    a = t5 ^ t13
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 26 + 0x8]
    b ^= key[4 * 26 + 0x9]
    c ^= key[4 * 26 + 0xA]
    d ^= key[4 * 26 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = b ^ d
    t3 = c & t1
    t13 = d | t1
    e = t2 ^ t3
    t5 = c ^ t1
    t6 = c ^ e
    t7 = b & t6
    t10 = e | t5
    h = t5 ^ t7
    t9 = d | t7
    t11 = t9 & t10
    t14 = t2 ^ h
    g = a ^ t11
    t15 = g ^ t13
    f = t14 ^ t15
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 27 + 0x8]
    f ^= key[4 * 27 + 0x9]
    g ^= key[4 * 27 + 0xA]
    h ^= key[4 * 27 + 0xB]
    t1 = e ^ g
    t2 = h ^ t1
    t3 = e & t2
    t4 = h ^ t3
    t5 = f & t4
    c = t2 ^ t5
    t7 = e | c
    t8 = f | h
    t11 = e | h
    t9 = t4 & t7
    b = t8 ^ t9
    t12 = f ^ t11
    t13 = c ^ t9
    t15 = t3 ^ t8
    d = t12 ^ t13
    t16 = g & t15
    a = t12 ^ t16
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 28 + 0x8]
    b ^= key[4 * 28 + 0x9]
    c ^= key[4 * 28 + 0xA]
    d ^= key[4 * 28 + 0xB]
    t1 = a ^ d
    t2 = d & t1
    t3 = c ^ t2
    t4 = b | t3
    h = t1 ^ t4
    t6 = (~b) % 0x100000000
    t7 = t1 | t6
    e = t3 ^ t7
    t9 = a & e
    t10 = t1 ^ t6
    t11 = t4 & t10
    g = t9 ^ t11
    t13 = a ^ t3
    t14 = t10 & g
    f = t13 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 29 + 0x8]
    f ^= key[4 * 29 + 0x9]
    g ^= key[4 * 29 + 0xA]
    h ^= key[4 * 29 + 0xB]
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = e ^ h
    t4 = g ^ t1
    t5 = t2 | t3
    a = t4 ^ t5
    t7 = h & a
    t8 = t2 ^ a
    t10 = t1 | a
    b = t7 ^ t8
    t11 = t2 | t7
    t12 = t3 ^ t10
    t14 = f ^ t7
    c = t11 ^ t12
    t15 = b & t12
    d = t14 ^ t15
    a = rotl32(a, 13)
    c = rotl32(c, 3)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    d = rotl32(d, 7)
    b = rotl32(b, 1)
    a ^= b ^ d
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a = rotl32(a, 5)
    c = rotl32(c, 22)
    a ^= key[4 * 30 + 0x8]
    b ^= key[4 * 30 + 0x9]
    c ^= key[4 * 30 + 0xA]
    d ^= key[4 * 30 + 0xB]
    t1 = (~a) % 0x100000000
    t2 = a ^ d
    t3 = b ^ t2
    t4 = t1 | t2
    t5 = c ^ t4
    f = b ^ t5
    t13 = (~t5) % 0x100000000
    t7 = t2 | f
    t8 = d ^ t7
    t9 = t5 & t8
    g = t3 ^ t9
    t11 = t5 ^ t8
    e = g ^ t11
    t14 = t3 & t11
    h = t13 ^ t14
    e = rotl32(e, 13)
    g = rotl32(g, 3)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    h = rotl32(h, 7)
    f = rotl32(f, 1)
    e ^= f ^ h
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e = rotl32(e, 5)
    g = rotl32(g, 22)
    e ^= key[4 * 31 + 0x8]
    f ^= key[4 * 31 + 0x9]
    g ^= key[4 * 31 + 0xA]
    h ^= key[4 * 31 + 0xB]
    t1 = (~g) % 0x100000000
    t2 = f ^ g
    t3 = f | t1
    t4 = h ^ t3
    t5 = e & t4
    t7 = e ^ h
    d = t2 ^ t5
    t8 = f ^ t5
    t9 = t2 | t8
    t11 = h & t3
    b = t7 ^ t9
    t12 = t5 ^ b
    t15 = t1 | t4
    t13 = d & t12
    c = t11 ^ t13
    t16 = t12 ^ c
    a = t15 ^ t16
    a ^= key[4 * 32 + 0x8]
    b ^= key[4 * 32 + 0x9]
    c ^= key[4 * 32 + 0xA]
    d ^= key[4 * 32 + 0xB]
    return a, b, c, d


def _serpent_decrypt(key, in_blk: Tuple[int, int, int, int]):
    a, b, c, d = in_blk
    e = 0
    f = 0
    g = 0
    h = 0
    t1 = 0
    t2 = 0
    t3 = 0
    t4 = 0
    t5 = 0
    t6 = 0
    t7 = 0
    t8 = 0
    t9 = 0
    t10 = 0
    t11 = 0
    t12 = 0
    t13 = 0
    t14 = 0
    t15 = 0
    t16 = 0
    a ^= key[4 * 32 + 0x8]
    b ^= key[4 * 32 + 0x9]
    c ^= key[4 * 32 + 0xA]
    d ^= key[4 * 32 + 0xB]
    t1 = a & b
    t2 = a | b
    t3 = c | t1
    t4 = d & t2
    h = t3 ^ t4
    t6 = (~d) % 0x100000000
    t7 = b ^ t4
    t8 = h ^ t6
    t11 = c ^ t7
    t9 = t7 | t8
    f = a ^ t9
    t12 = d | f
    e = t11 ^ t12
    t14 = a & h
    t15 = t3 ^ f
    t16 = e ^ t14
    g = t15 ^ t16
    e ^= key[4 * 31 + 0x8]
    f ^= key[4 * 31 + 0x9]
    g ^= key[4 * 31 + 0xA]
    h ^= key[4 * 31 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = g ^ t2
    t4 = g | t1
    t5 = h ^ t4
    t13 = h & t1
    b = t3 ^ t5
    t7 = t3 & t5
    t8 = t2 ^ t7
    t9 = f | t8
    d = t5 ^ t9
    t11 = f | d
    a = t8 ^ t11
    t14 = t3 ^ t11
    c = t13 ^ t14
    a ^= key[4 * 30 + 0x8]
    b ^= key[4 * 30 + 0x9]
    c ^= key[4 * 30 + 0xA]
    d ^= key[4 * 30 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = (~c) % 0x100000000
    t2 = b & t1
    t3 = d ^ t2
    t4 = a & t3
    t5 = b ^ t1
    h = t4 ^ t5
    t7 = b | h
    t8 = a & t7
    f = t3 ^ t8
    t10 = a | d
    t11 = t1 ^ t7
    e = t10 ^ t11
    t13 = a ^ c
    t14 = b & t10
    t15 = t4 | t13
    g = t14 ^ t15
    e ^= key[4 * 29 + 0x8]
    f ^= key[4 * 29 + 0x9]
    g ^= key[4 * 29 + 0xA]
    h ^= key[4 * 29 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = g ^ h
    t2 = g | h
    t3 = f ^ t2
    t4 = e & t3
    b = t1 ^ t4
    t6 = e ^ h
    t7 = f | h
    t8 = t6 & t7
    d = t3 ^ t8
    t10 = (~e) % 0x100000000
    t11 = g ^ d
    t12 = t10 | t11
    a = t3 ^ t12
    t14 = g | t4
    t15 = t7 ^ t14
    t16 = d | t10
    c = t15 ^ t16
    a ^= key[4 * 28 + 0x8]
    b ^= key[4 * 28 + 0x9]
    c ^= key[4 * 28 + 0xA]
    d ^= key[4 * 28 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = b ^ c
    t2 = b | c
    t3 = a ^ c
    t7 = a ^ d
    t4 = t2 ^ t3
    t5 = d | t4
    t9 = t2 ^ t7
    e = t1 ^ t5
    t8 = t1 | t5
    t11 = a & t4
    g = t8 ^ t9
    t12 = e | t9
    f = t11 ^ t12
    t14 = a & g
    t15 = t2 ^ t14
    t16 = e & t15
    h = t4 ^ t16
    e ^= key[4 * 27 + 0x8]
    f ^= key[4 * 27 + 0x9]
    g ^= key[4 * 27 + 0xA]
    h ^= key[4 * 27 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = f ^ h
    t2 = (~t1) % 0x100000000
    t3 = e ^ g
    t4 = g ^ t1
    t7 = e | t2
    t5 = f & t4
    t8 = h ^ t7
    t11 = (~t4) % 0x100000000
    a = t3 ^ t5
    t9 = t3 | t8
    t14 = h & t11
    d = t1 ^ t9
    t12 = a | d
    b = t11 ^ t12
    t15 = t3 ^ t12
    c = t14 ^ t15
    a ^= key[4 * 26 + 0x8]
    b ^= key[4 * 26 + 0x9]
    c ^= key[4 * 26 + 0xA]
    d ^= key[4 * 26 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = a ^ d
    t2 = a & b
    t3 = b ^ c
    t4 = a ^ t3
    t5 = b | d
    t7 = c | t1
    h = t4 ^ t5
    t8 = b ^ t7
    t11 = (~t2) % 0x100000000
    t9 = t4 & t8
    f = t1 ^ t9
    t13 = t9 ^ t11
    t12 = h & f
    g = t12 ^ t13
    t15 = a & d
    t16 = c ^ t13
    e = t15 ^ t16
    e ^= key[4 * 25 + 0x8]
    f ^= key[4 * 25 + 0x9]
    g ^= key[4 * 25 + 0xA]
    h ^= key[4 * 25 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = t1 | t2
    t4 = h ^ t3
    t7 = h & t2
    t5 = g ^ t4
    t8 = t1 ^ t7
    c = t2 ^ t5
    t11 = e & t4
    t9 = c & t8
    t14 = t5 ^ t8
    b = t4 ^ t9
    t12 = t5 | b
    d = t11 ^ t12
    a = d ^ t14
    a ^= key[4 * 24 + 0x8]
    b ^= key[4 * 24 + 0x9]
    c ^= key[4 * 24 + 0xA]
    d ^= key[4 * 24 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = a & b
    t2 = a | b
    t3 = c | t1
    t4 = d & t2
    h = t3 ^ t4
    t6 = (~d) % 0x100000000
    t7 = b ^ t4
    t8 = h ^ t6
    t11 = c ^ t7
    t9 = t7 | t8
    f = a ^ t9
    t12 = d | f
    e = t11 ^ t12
    t14 = a & h
    t15 = t3 ^ f
    t16 = e ^ t14
    g = t15 ^ t16
    e ^= key[4 * 23 + 0x8]
    f ^= key[4 * 23 + 0x9]
    g ^= key[4 * 23 + 0xA]
    h ^= key[4 * 23 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = g ^ t2
    t4 = g | t1
    t5 = h ^ t4
    t13 = h & t1
    b = t3 ^ t5
    t7 = t3 & t5
    t8 = t2 ^ t7
    t9 = f | t8
    d = t5 ^ t9
    t11 = f | d
    a = t8 ^ t11
    t14 = t3 ^ t11
    c = t13 ^ t14
    a ^= key[4 * 22 + 0x8]
    b ^= key[4 * 22 + 0x9]
    c ^= key[4 * 22 + 0xA]
    d ^= key[4 * 22 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = (~c) % 0x100000000
    t2 = b & t1
    t3 = d ^ t2
    t4 = a & t3
    t5 = b ^ t1
    h = t4 ^ t5
    t7 = b | h
    t8 = a & t7
    f = t3 ^ t8
    t10 = a | d
    t11 = t1 ^ t7
    e = t10 ^ t11
    t13 = a ^ c
    t14 = b & t10
    t15 = t4 | t13
    g = t14 ^ t15
    e ^= key[4 * 21 + 0x8]
    f ^= key[4 * 21 + 0x9]
    g ^= key[4 * 21 + 0xA]
    h ^= key[4 * 21 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = g ^ h
    t2 = g | h
    t3 = f ^ t2
    t4 = e & t3
    b = t1 ^ t4
    t6 = e ^ h
    t7 = f | h
    t8 = t6 & t7
    d = t3 ^ t8
    t10 = (~e) % 0x100000000
    t11 = g ^ d
    t12 = t10 | t11
    a = t3 ^ t12
    t14 = g | t4
    t15 = t7 ^ t14
    t16 = d | t10
    c = t15 ^ t16
    a ^= key[4 * 20 + 0x8]
    b ^= key[4 * 20 + 0x9]
    c ^= key[4 * 20 + 0xA]
    d ^= key[4 * 20 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = b ^ c
    t2 = b | c
    t3 = a ^ c
    t7 = a ^ d
    t4 = t2 ^ t3
    t5 = d | t4
    t9 = t2 ^ t7
    e = t1 ^ t5
    t8 = t1 | t5
    t11 = a & t4
    g = t8 ^ t9
    t12 = e | t9
    f = t11 ^ t12
    t14 = a & g
    t15 = t2 ^ t14
    t16 = e & t15
    h = t4 ^ t16
    e ^= key[4 * 19 + 0x8]
    f ^= key[4 * 19 + 0x9]
    g ^= key[4 * 19 + 0xA]
    h ^= key[4 * 19 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = f ^ h
    t2 = (~t1) % 0x100000000
    t3 = e ^ g
    t4 = g ^ t1
    t7 = e | t2
    t5 = f & t4
    t8 = h ^ t7
    t11 = (~t4) % 0x100000000
    a = t3 ^ t5
    t9 = t3 | t8
    t14 = h & t11
    d = t1 ^ t9
    t12 = a | d
    b = t11 ^ t12
    t15 = t3 ^ t12
    c = t14 ^ t15
    a ^= key[4 * 18 + 0x8]
    b ^= key[4 * 18 + 0x9]
    c ^= key[4 * 18 + 0xA]
    d ^= key[4 * 18 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = a ^ d
    t2 = a & b
    t3 = b ^ c
    t4 = a ^ t3
    t5 = b | d
    t7 = c | t1
    h = t4 ^ t5
    t8 = b ^ t7
    t11 = (~t2) % 0x100000000
    t9 = t4 & t8
    f = t1 ^ t9
    t13 = t9 ^ t11
    t12 = h & f
    g = t12 ^ t13
    t15 = a & d
    t16 = c ^ t13
    e = t15 ^ t16
    e ^= key[4 * 17 + 0x8]
    f ^= key[4 * 17 + 0x9]
    g ^= key[4 * 17 + 0xA]
    h ^= key[4 * 17 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = t1 | t2
    t4 = h ^ t3
    t7 = h & t2
    t5 = g ^ t4
    t8 = t1 ^ t7
    c = t2 ^ t5
    t11 = e & t4
    t9 = c & t8
    t14 = t5 ^ t8
    b = t4 ^ t9
    t12 = t5 | b
    d = t11 ^ t12
    a = d ^ t14
    a ^= key[4 * 16 + 0x8]
    b ^= key[4 * 16 + 0x9]
    c ^= key[4 * 16 + 0xA]
    d ^= key[4 * 16 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = a & b
    t2 = a | b
    t3 = c | t1
    t4 = d & t2
    h = t3 ^ t4
    t6 = (~d) % 0x100000000
    t7 = b ^ t4
    t8 = h ^ t6
    t11 = c ^ t7
    t9 = t7 | t8
    f = a ^ t9
    t12 = d | f
    e = t11 ^ t12
    t14 = a & h
    t15 = t3 ^ f
    t16 = e ^ t14
    g = t15 ^ t16
    e ^= key[4 * 15 + 0x8]
    f ^= key[4 * 15 + 0x9]
    g ^= key[4 * 15 + 0xA]
    h ^= key[4 * 15 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = g ^ t2
    t4 = g | t1
    t5 = h ^ t4
    t13 = h & t1
    b = t3 ^ t5
    t7 = t3 & t5
    t8 = t2 ^ t7
    t9 = f | t8
    d = t5 ^ t9
    t11 = f | d
    a = t8 ^ t11
    t14 = t3 ^ t11
    c = t13 ^ t14
    a ^= key[4 * 14 + 0x8]
    b ^= key[4 * 14 + 0x9]
    c ^= key[4 * 14 + 0xA]
    d ^= key[4 * 14 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = (~c) % 0x100000000
    t2 = b & t1
    t3 = d ^ t2
    t4 = a & t3
    t5 = b ^ t1
    h = t4 ^ t5
    t7 = b | h
    t8 = a & t7
    f = t3 ^ t8
    t10 = a | d
    t11 = t1 ^ t7
    e = t10 ^ t11
    t13 = a ^ c
    t14 = b & t10
    t15 = t4 | t13
    g = t14 ^ t15
    e ^= key[4 * 13 + 0x8]
    f ^= key[4 * 13 + 0x9]
    g ^= key[4 * 13 + 0xA]
    h ^= key[4 * 13 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = g ^ h
    t2 = g | h
    t3 = f ^ t2
    t4 = e & t3
    b = t1 ^ t4
    t6 = e ^ h
    t7 = f | h
    t8 = t6 & t7
    d = t3 ^ t8
    t10 = (~e) % 0x100000000
    t11 = g ^ d
    t12 = t10 | t11
    a = t3 ^ t12
    t14 = g | t4
    t15 = t7 ^ t14
    t16 = d | t10
    c = t15 ^ t16
    a ^= key[4 * 12 + 0x8]
    b ^= key[4 * 12 + 0x9]
    c ^= key[4 * 12 + 0xA]
    d ^= key[4 * 12 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = b ^ c
    t2 = b | c
    t3 = a ^ c
    t7 = a ^ d
    t4 = t2 ^ t3
    t5 = d | t4
    t9 = t2 ^ t7
    e = t1 ^ t5
    t8 = t1 | t5
    t11 = a & t4
    g = t8 ^ t9
    t12 = e | t9
    f = t11 ^ t12
    t14 = a & g
    t15 = t2 ^ t14
    t16 = e & t15
    h = t4 ^ t16
    e ^= key[4 * 11 + 0x8]
    f ^= key[4 * 11 + 0x9]
    g ^= key[4 * 11 + 0xA]
    h ^= key[4 * 11 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = f ^ h
    t2 = (~t1) % 0x100000000
    t3 = e ^ g
    t4 = g ^ t1
    t7 = e | t2
    t5 = f & t4
    t8 = h ^ t7
    t11 = (~t4) % 0x100000000
    a = t3 ^ t5
    t9 = t3 | t8
    t14 = h & t11
    d = t1 ^ t9
    t12 = a | d
    b = t11 ^ t12
    t15 = t3 ^ t12
    c = t14 ^ t15
    a ^= key[4 * 10 + 0x8]
    b ^= key[4 * 10 + 0x9]
    c ^= key[4 * 10 + 0xA]
    d ^= key[4 * 10 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = a ^ d
    t2 = a & b
    t3 = b ^ c
    t4 = a ^ t3
    t5 = b | d
    t7 = c | t1
    h = t4 ^ t5
    t8 = b ^ t7
    t11 = (~t2) % 0x100000000
    t9 = t4 & t8
    f = t1 ^ t9
    t13 = t9 ^ t11
    t12 = h & f
    g = t12 ^ t13
    t15 = a & d
    t16 = c ^ t13
    e = t15 ^ t16
    e ^= key[4 * 9 + 0x8]
    f ^= key[4 * 9 + 0x9]
    g ^= key[4 * 9 + 0xA]
    h ^= key[4 * 9 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = t1 | t2
    t4 = h ^ t3
    t7 = h & t2
    t5 = g ^ t4
    t8 = t1 ^ t7
    c = t2 ^ t5
    t11 = e & t4
    t9 = c & t8
    t14 = t5 ^ t8
    b = t4 ^ t9
    t12 = t5 | b
    d = t11 ^ t12
    a = d ^ t14
    a ^= key[4 * 8 + 0x8]
    b ^= key[4 * 8 + 0x9]
    c ^= key[4 * 8 + 0xA]
    d ^= key[4 * 8 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = a & b
    t2 = a | b
    t3 = c | t1
    t4 = d & t2
    h = t3 ^ t4
    t6 = (~d) % 0x100000000
    t7 = b ^ t4
    t8 = h ^ t6
    t11 = c ^ t7
    t9 = t7 | t8
    f = a ^ t9
    t12 = d | f
    e = t11 ^ t12
    t14 = a & h
    t15 = t3 ^ f
    t16 = e ^ t14
    g = t15 ^ t16
    e ^= key[4 * 7 + 0x8]
    f ^= key[4 * 7 + 0x9]
    g ^= key[4 * 7 + 0xA]
    h ^= key[4 * 7 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = g ^ t2
    t4 = g | t1
    t5 = h ^ t4
    t13 = h & t1
    b = t3 ^ t5
    t7 = t3 & t5
    t8 = t2 ^ t7
    t9 = f | t8
    d = t5 ^ t9
    t11 = f | d
    a = t8 ^ t11
    t14 = t3 ^ t11
    c = t13 ^ t14
    a ^= key[4 * 6 + 0x8]
    b ^= key[4 * 6 + 0x9]
    c ^= key[4 * 6 + 0xA]
    d ^= key[4 * 6 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = (~c) % 0x100000000
    t2 = b & t1
    t3 = d ^ t2
    t4 = a & t3
    t5 = b ^ t1
    h = t4 ^ t5
    t7 = b | h
    t8 = a & t7
    f = t3 ^ t8
    t10 = a | d
    t11 = t1 ^ t7
    e = t10 ^ t11
    t13 = a ^ c
    t14 = b & t10
    t15 = t4 | t13
    g = t14 ^ t15
    e ^= key[4 * 5 + 0x8]
    f ^= key[4 * 5 + 0x9]
    g ^= key[4 * 5 + 0xA]
    h ^= key[4 * 5 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = g ^ h
    t2 = g | h
    t3 = f ^ t2
    t4 = e & t3
    b = t1 ^ t4
    t6 = e ^ h
    t7 = f | h
    t8 = t6 & t7
    d = t3 ^ t8
    t10 = (~e) % 0x100000000
    t11 = g ^ d
    t12 = t10 | t11
    a = t3 ^ t12
    t14 = g | t4
    t15 = t7 ^ t14
    t16 = d | t10
    c = t15 ^ t16
    a ^= key[4 * 4 + 0x8]
    b ^= key[4 * 4 + 0x9]
    c ^= key[4 * 4 + 0xA]
    d ^= key[4 * 4 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = b ^ c
    t2 = b | c
    t3 = a ^ c
    t7 = a ^ d
    t4 = t2 ^ t3
    t5 = d | t4
    t9 = t2 ^ t7
    e = t1 ^ t5
    t8 = t1 | t5
    t11 = a & t4
    g = t8 ^ t9
    t12 = e | t9
    f = t11 ^ t12
    t14 = a & g
    t15 = t2 ^ t14
    t16 = e & t15
    h = t4 ^ t16
    e ^= key[4 * 3 + 0x8]
    f ^= key[4 * 3 + 0x9]
    g ^= key[4 * 3 + 0xA]
    h ^= key[4 * 3 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = f ^ h
    t2 = (~t1) % 0x100000000
    t3 = e ^ g
    t4 = g ^ t1
    t7 = e | t2
    t5 = f & t4
    t8 = h ^ t7
    t11 = (~t4) % 0x100000000
    a = t3 ^ t5
    t9 = t3 | t8
    t14 = h & t11
    d = t1 ^ t9
    t12 = a | d
    b = t11 ^ t12
    t15 = t3 ^ t12
    c = t14 ^ t15
    a ^= key[4 * 2 + 0x8]
    b ^= key[4 * 2 + 0x9]
    c ^= key[4 * 2 + 0xA]
    d ^= key[4 * 2 + 0xB]
    c = rotr32(c, 22)
    a = rotr32(a, 5)
    c ^= d ^ ((b << 7) & 0xFFFFFFFF)
    a ^= b ^ d
    d = rotr32(d, 7)
    b = rotr32(b, 1)
    d ^= c ^ ((a << 3) & 0xFFFFFFFF)
    b ^= a ^ c
    c = rotr32(c, 3)
    a = rotr32(a, 13)
    t1 = a ^ d
    t2 = a & b
    t3 = b ^ c
    t4 = a ^ t3
    t5 = b | d
    t7 = c | t1
    h = t4 ^ t5
    t8 = b ^ t7
    t11 = (~t2) % 0x100000000
    t9 = t4 & t8
    f = t1 ^ t9
    t13 = t9 ^ t11
    t12 = h & f
    g = t12 ^ t13
    t15 = a & d
    t16 = c ^ t13
    e = t15 ^ t16
    e ^= key[4 * 1 + 0x8]
    f ^= key[4 * 1 + 0x9]
    g ^= key[4 * 1 + 0xA]
    h ^= key[4 * 1 + 0xB]
    g = rotr32(g, 22)
    e = rotr32(e, 5)
    g ^= h ^ ((f << 7) & 0xFFFFFFFF)
    e ^= f ^ h
    h = rotr32(h, 7)
    f = rotr32(f, 1)
    h ^= g ^ ((e << 3) & 0xFFFFFFFF)
    f ^= e ^ g
    g = rotr32(g, 3)
    e = rotr32(e, 13)
    t1 = (~e) % 0x100000000
    t2 = e ^ f
    t3 = t1 | t2
    t4 = h ^ t3
    t7 = h & t2
    t5 = g ^ t4
    t8 = t1 ^ t7
    c = t2 ^ t5
    t11 = e & t4
    t9 = c & t8
    t14 = t5 ^ t8
    b = t4 ^ t9
    t12 = t5 | b
    d = t11 ^ t12
    a = d ^ t14
    a ^= key[4 * 0 + 0x8]
    b ^= key[4 * 0 + 0x9]
    c ^= key[4 * 0 + 0xA]
    d ^= key[4 * 0 + 0xB]
    return a, b, c, d


class SerpentMode(Enum):
    Encrypt = _serpent_encrypt
    Decrypt = _serpent_decrypt


class Serpent:
    block_size = 0x10
    key_size = 0x20

    def __init__(self, key: Optional[ByteString] = None):
        if key:
            self.set_key(key)

    def set_key(self, key):
        key_len = len(key)
        key_blocks, r = divmod(key_len, 4)
        if r != 0:
            raise ValueError('key length not a multiple of 4')
        if key_len > self.key_size:
            raise ValueError('key length is more than 32')
        self.key_context = [0] * 140
        key_word32 = [0] * 32
        key_word32[:key_blocks] = struct.unpack(F'<{key_blocks}L', key)
        _serpent_set_key(self.key_context, key_word32, key_len)

    def __call__(self, mode: SerpentMode, data: bytearray):
        if len(data) % 16:
            raise ValueError('block size must be a multiple of 16')
        view = memoryview(data)
        blk = self.block_size
        for k in range(0, len(data), blk):
            index = slice(k, k + blk)
            block = struct.unpack('<4L', view[index])
            block = mode(self.key_context, block)
            data[index] = struct.pack("<4L", *block)
