"""Repeat the following paper for `CCPSO2`:
    Li, X. and Yao, X., 2012.
    Cooperatively coevolving particle swarms for large scale optimization.
    IEEE Transactions on Evolutionary Computation, 16(2), pp.210-224.
    https://ieeexplore.ieee.org/document/5910380/

    Luckily our code could repeat the data generated by the original paper *well*.
    Therefore, we argue that the repeatability of `CCPSO2` could be **well-documented**.
"""
import time

import numpy as np

from pypop7.benchmarks.base_functions import sphere, rastrigin, griewank
from pypop7.optimizers.pso.ccpso2 import CCPSO2 as Solver


if __name__ == '__main__':
    ndim_problem = 1000

    start_run = time.time()
    problem = {'fitness_function': sphere,
               'ndim_problem': ndim_problem,
               'lower_boundary': -100 * np.ones((ndim_problem,)),
               'upper_boundary': 100 * np.ones((ndim_problem,))}
    options = {'max_function_evaluations': 5e6,
               'seed_rng': 0,
               'verbose': 1e1}
    solver = Solver(problem, options)
    results = solver.optimize()
    print(results)  # 1.472676192640866e-28 vs 5.18e-13 (from the original paper)
    print('*** Runtime: {:7.5e}'.format(time.time() - start_run))

    start_run = time.time()
    problem = {'fitness_function': rastrigin,
               'ndim_problem': ndim_problem,
               'lower_boundary': -5 * np.ones((ndim_problem,)),
               'upper_boundary': 5 * np.ones((ndim_problem,))}
    options = {'max_function_evaluations': 5e6,
               'seed_rng': 0,
               'verbose': 1e1}
    solver = Solver(problem, options)
    results = solver.optimize()
    print(results)  # 0.2947450894425856 vs 1.99e-01 (from the original paper)
    print('*** Runtime: {:7.5e}'.format(time.time() - start_run))

    start_run = time.time()
    problem = {'fitness_function': griewank,
               'ndim_problem': ndim_problem,
               'lower_boundary': -600 * np.ones((ndim_problem,)),
               'upper_boundary': 600 * np.ones((ndim_problem,))}
    options = {'max_function_evaluations': 5e6,
               'seed_rng': 0,
               'verbose': 1e1}
    solver = Solver(problem, options)
    results = solver.optimize()
    print(results)  # 4.218847493575595e-15 vs 1.18e-03 (from the original paper)
    print('*** Runtime: {:7.5e}'.format(time.time() - start_run))
