/*
 * This file is part of MULTEM.
 * Copyright 2020 Ivan Lobato <Ivanlh20@gmail.com>
 *
 * MULTEM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * MULTEM is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MULTEM. If not, see <http:// www.gnu.org/licenses/>.
 */

#ifndef ATOMIC_DATA_H
#define ATOMIC_DATA_H

#ifdef _MSC_VER
#pragma once
#endif // _MSC_VER

#include <string>
#include "math.cuh"
#include "types.cuh"
#include "atomic_fcns_mt.hpp"

namespace mt
{
	class Atomic_Data{
		public:
			Atomic_Data(ePotential_Type PotPar_i = ePT_none)
			{
				init();

				Load_Atomic_Data_names();
				Load_Atomic_Data();
				Load_eels_edges();

				if(PotPar_i != ePT_none)
				{
					Load_Data(PotPar_i);
				}
			}

			// Load data tables and Parameterization
			void Load_Data(ePotential_Type PotPar_i)
			{
				Load_Atomic_Data_names();
				Load_Atomic_Data();
				Load_eels_edges();

				potential_type = PotPar_i;
				switch(potential_type)
				{
					case ePT_Doyle_0_4:
						Load_feg_Doyle_neutral_0_4();
						break;
					case ePT_Peng_0_4:
						Load_feg_Peng_neutral_0_4();
						break;
					case ePT_Peng_0_12:
						Load_feg_Peng_neutral_0_12();
						break;
					case ePT_Kirkland_0_12:
						Load_feg_Kirkland_neutral_0_12();
						break;
					case ePT_Weickenmeier_0_12:
						Load_feg_Weickenmeier_neutral_0_12();
						break;
					case ePT_Lobato_0_12:
						Load_feg_Lobato_neutral_0_12();
						break;
				}
				Load_feg_Peng_ion_0_4();
			}

			std::string Z_name(const int &Z) const 
			{ 
				return (Z_bound(Z))? data_table[Z-1].name:"";
			}

			Vector<float, e_host> extract_major_edges(const int &Z)
			{ 
				Vector<float, e_host> data;
				data.reserve(5);

				auto &edges = data_table[Z-1].edges;
				for(auto ik=0; ik<5; ik++)
				{
					if(nonZero(edges[ik]))
					{
						data.push_back(edges[ik]);
					}
				}
				data.shrink_to_fit();

				return data;
			}

			Vector<float, e_host> extract_minor_edges(const int &Z)
			{ 
				Vector<float, e_host> data;
				data.reserve(7);

				auto &edges = data_table[Z-1].edges;
				for(auto ik=5; ik<12; ik++)
				{
					if(nonZero(edges[ik]))
					{
						data.push_back(edges[ik]);
					}
				}
				data.shrink_to_fit();

				return data;
			}

			bool Z_bound(const int &Z) const 
			{ 
				return (Z >= 1) && (Z <= c_nAtomsTypes);
			}

			int mass_number(const int &Z) const 
			{ 
				return (Z_bound(Z))? data_table[Z-1].A:0;
			}

			double atomic_mass(const int &Z) const
			{
				return (Z_bound(Z))?data_table[Z-1].m:0;
			}

			double nuclear_radius(const int &Z) const
			{
				return (Z_bound(Z))?data_table[Z-1].rn:0;
			}

			double atomic_radius(const int &Z) const
			{
				return (Z_bound(Z))?data_table[Z-1].ra:0;
			}

			template <class TAtom_Type>
			void To_atom_type_CPU(int Z_i, double Vrl_i, int nR_i, double R_min_i, TAtom_Type &atom_type)
			{
				using value_type = Value_type<decltype(c_atom_type)>;

				int iZ = Z_i - 1;
				c_atom_type.Z = Z_i;
				c_atom_type.A = data_table[iZ].A;
				c_atom_type.m = data_table[iZ].m;
				c_atom_type.rn_e = data_table[iZ].rn*1e-05;
				c_atom_type.rn_c = (1.2e-05)*pow(static_cast<value_type>(data_table[iZ].A), 1.0/3.0);
				c_atom_type.ra_e = data_table[iZ].ra;
				c_atom_type.ra_c = 0.0;

				R_min_i = ::fmax(c_atom_type.rn_c, R_min_i);
				get_atom_coef(Z_i, Vrl_i, nR_i, R_min_i, c_atom_type);

				// ra_c and R_max
				int charge = 0;
				c_atom_cal.Set_Atom_Type(potential_type, charge, &c_atom_type);
				c_atom_type.ra_c = 3.0*c_atom_cal.AtomicRadius_rms(3);

				if(isZero(c_atom_type.ra_c))
				{
					c_atom_type.ra_c = 1.75*c_atom_type.ra_e;
				}

				atom_type.assign(c_atom_type);
			}

		private:
			struct Data_Table
			{
				std::string name;
				int Z;						// atomic number
				double m;					// atomic mass
				int A;						// mass number	
				double rn;					// nuclear radius	
				double ra;					// atomic radius
				Vector<PP_Coef<double, e_host>, e_host> feg;
				Vector<double, e_host> edges;
			};

			// initialization
			void init()
			{
				potential_type = ePT_Lobato_0_12;
				c_feg.resize(c_nAtomsIons);
				c_fxg.resize(c_nAtomsIons);
				c_Pr.resize(c_nAtomsIons);
				c_Vr.resize(c_nAtomsIons);
				c_VR.resize(c_nAtomsIons);
				for(auto j = 0; j < c_feg.size(); j++)
				{
					c_feg[j].resize(6);
					c_fxg[j].resize(6);
					c_Pr[j].resize(6);
					c_Vr[j].resize(6);
					c_VR[j].resize(6);
				}

				data_table.resize(c_nAtomsTypes);
				for(auto i = 0; i < data_table.size(); i++)
				{
					data_table[i].edges.resize(12);

					data_table[i].feg.resize(c_nAtomsIons); // -7:1:7
					for(auto j = 0; j < data_table[i].feg.size(); j++)
					{
						data_table[i].feg[j].resize(6, 0);
					}
				}
			}

			// load names
			void Load_Atomic_Data_names()
			{
				data_table[0].name = "H";
				data_table[1].name = "He";
				data_table[2].name = "Li";
				data_table[3].name = "Be";
				data_table[4].name = "B";
				data_table[5].name = "C";
				data_table[6].name = "N";
				data_table[7].name = "O";
				data_table[8].name = "F";
				data_table[9].name = "Ne";
				data_table[10].name = "Na";
				data_table[11].name = "Mg";
				data_table[12].name = "Al";
				data_table[13].name = "Si";
				data_table[14].name = "P";
				data_table[15].name = "S";
				data_table[16].name = "Cl";
				data_table[17].name = "Ar";
				data_table[18].name = "K";
				data_table[19].name = "Ca";
				data_table[20].name = "Sc";
				data_table[21].name = "Ti";
				data_table[22].name = "V";
				data_table[23].name = "Cr";
				data_table[24].name = "Mn";
				data_table[25].name = "Fe";
				data_table[26].name = "Co";
				data_table[27].name = "Ni";
				data_table[28].name = "Cu";
				data_table[29].name = "Zn";
				data_table[30].name = "Ga";
				data_table[31].name = "Ge";
				data_table[32].name = "As";
				data_table[33].name = "Se";
				data_table[34].name = "Br";
				data_table[35].name = "Kr";
				data_table[36].name = "Rb";
				data_table[37].name = "Sr";
				data_table[38].name = "Y";
				data_table[39].name = "Zr";
				data_table[40].name = "Nb";
				data_table[41].name = "Mo";
				data_table[42].name = "Tc";
				data_table[43].name = "Ru";
				data_table[44].name = "Rh";
				data_table[45].name = "Pd";
				data_table[46].name = "Ag";
				data_table[47].name = "Cd";
				data_table[48].name = "In";
				data_table[49].name = "Sn";
				data_table[50].name = "Sb";
				data_table[51].name = "Te";
				data_table[52].name = "I";
				data_table[53].name = "Xe";
				data_table[54].name = "Cs";
				data_table[55].name = "Ba";
				data_table[56].name = "La";
				data_table[57].name = "Ce";
				data_table[58].name = "Pr";
				data_table[59].name = "Nd";
				data_table[60].name = "Pm";
				data_table[61].name = "Sm";
				data_table[62].name = "Eu";
				data_table[63].name = "Gd";
				data_table[64].name = "Tb";
				data_table[65].name = "Dy";
				data_table[66].name = "Ho";
				data_table[67].name = "Er";
				data_table[68].name = "Tm";
				data_table[69].name = "Yb";
				data_table[70].name = "Lu";
				data_table[71].name = "Hf";
				data_table[72].name = "Ta";
				data_table[73].name = "W";
				data_table[74].name = "Re";
				data_table[75].name = "Os";
				data_table[76].name = "Ir";
				data_table[77].name = "Pt";
				data_table[78].name = "Au";
				data_table[79].name = "Hg";
				data_table[80].name = "Tl";
				data_table[81].name = "Pb";
				data_table[82].name = "Bi";
				data_table[83].name = "Po";
				data_table[84].name = "At";
				data_table[85].name = "Rn";
				data_table[86].name = "Fr";
				data_table[87].name = "Ra";
				data_table[88].name = "Ac";
				data_table[89].name = "Th";
				data_table[90].name = "Pa";
				data_table[91].name = "U";
				data_table[92].name = "Np";
				data_table[93].name = "Pu";
				data_table[94].name = "Am";
				data_table[95].name = "Cm";
				data_table[96].name = "Bk";
				data_table[97].name = "Cf";
			}

			// Load atomic data tables
			void Load_Atomic_Data()
			{
				data_table[0].Z = 1;		data_table[0].A = 1;		data_table[0].m = 1.00797;		data_table[0].rn = 0.8777;		data_table[0].ra = 0.79;
				data_table[1].Z = 2;		data_table[1].A = 4;		data_table[1].m = 4.0026;		data_table[1].rn = 1.6753;		data_table[1].ra = 0.49;
				data_table[2].Z = 3;		data_table[2].A = 7;		data_table[2].m = 6.941;		data_table[2].rn = 2.4173;		data_table[2].ra = 2.05;
				data_table[3].Z = 4;		data_table[3].A = 9;		data_table[3].m = 9.01218;		data_table[3].rn = 2.518;		data_table[3].ra = 1.4;
				data_table[4].Z = 5;		data_table[4].A = 11;		data_table[4].m = 10.811;		data_table[4].rn = 2.406;		data_table[4].ra = 1.17;
				data_table[5].Z = 6;		data_table[5].A = 12;		data_table[5].m = 12.011;		data_table[5].rn = 2.4702;		data_table[5].ra = 0.91;
				data_table[6].Z = 7;		data_table[6].A = 14;		data_table[6].m = 14.0067;		data_table[6].rn = 2.5582;		data_table[6].ra = 0.75;
				data_table[7].Z = 8;		data_table[7].A = 16;		data_table[7].m = 15.9994;		data_table[7].rn = 2.6991;		data_table[7].ra = 0.65;
				data_table[8].Z = 9;		data_table[8].A = 19;		data_table[8].m = 18.9984;		data_table[8].rn = 2.8976;		data_table[8].ra = 0.57;
				data_table[9].Z = 10;		data_table[9].A = 20;		data_table[9].m = 20.1797;		data_table[9].rn = 3.0058;		data_table[9].ra = 0.51;
				data_table[10].Z = 11;		data_table[10].A = 23;		data_table[10].m = 22.98977;	data_table[10].rn = 2.9935;		data_table[10].ra = 2.23;
				data_table[11].Z = 12;		data_table[11].A = 24;		data_table[11].m = 24.305;		data_table[11].rn = 3.057;		data_table[11].ra = 1.72;
				data_table[12].Z = 13;		data_table[12].A = 27;		data_table[12].m = 26.98154;	data_table[12].rn = 3.061;		data_table[12].ra = 1.62;
				data_table[13].Z = 14;		data_table[13].A = 28;		data_table[13].m = 28.0855;		data_table[13].rn = 3.1224;		data_table[13].ra = 1.44;
				data_table[14].Z = 15;		data_table[14].A = 31;		data_table[14].m = 30.97376;	data_table[14].rn = 3.1889;		data_table[14].ra = 1.23;
				data_table[15].Z = 16;		data_table[15].A = 32;		data_table[15].m = 32.066;		data_table[15].rn = 3.2847;		data_table[15].ra = 1.09;
				data_table[16].Z = 17;		data_table[16].A = 35;		data_table[16].m = 35.4527;		data_table[16].rn = 3.3654;		data_table[16].ra = 0.97;
				data_table[17].Z = 18;		data_table[17].A = 40;		data_table[17].m = 39.948;		data_table[17].rn = 3.4276;		data_table[17].ra = 0.88;
				data_table[18].Z = 19;		data_table[18].A = 39;		data_table[18].m = 39.0983;		data_table[18].rn = 3.435;		data_table[18].ra = 2.77;
				data_table[19].Z = 20;		data_table[19].A = 40;		data_table[19].m = 40.078;		data_table[19].rn = 3.4777;		data_table[19].ra = 2.23;
				data_table[20].Z = 21;		data_table[20].A = 45;		data_table[20].m = 44.9559;		data_table[20].rn = 3.546;		data_table[20].ra = 2.09;
				data_table[21].Z = 22;		data_table[21].A = 48;		data_table[21].m = 47.88;		data_table[21].rn = 3.5922;		data_table[21].ra = 2;
				data_table[22].Z = 23;		data_table[22].A = 51;		data_table[22].m = 50.9415;		data_table[22].rn = 3.6002;		data_table[22].ra = 1.92;
				data_table[23].Z = 24;		data_table[23].A = 52;		data_table[23].m = 51.996;		data_table[23].rn = 3.6452;		data_table[23].ra = 1.85;
				data_table[24].Z = 25;		data_table[24].A = 55;		data_table[24].m = 54.938;		data_table[24].rn = 3.7057;		data_table[24].ra = 1.79;
				data_table[25].Z = 26;		data_table[25].A = 56;		data_table[25].m = 55.847;		data_table[25].rn = 3.7384;		data_table[25].ra = 1.72;
				data_table[26].Z = 27;		data_table[26].A = 59;		data_table[26].m = 58.9332;		data_table[26].rn = 3.7875;		data_table[26].ra = 1.67;
				data_table[27].Z = 28;		data_table[27].A = 58;		data_table[27].m = 58.6934;		data_table[27].rn = 3.7757;		data_table[27].ra = 1.62;
				data_table[28].Z = 29;		data_table[28].A = 63;		data_table[28].m = 63.456;		data_table[28].rn = 3.8823;		data_table[28].ra = 1.57;
				data_table[29].Z = 30;		data_table[29].A = 64;		data_table[29].m = 65.39;		data_table[29].rn = 3.9279;		data_table[29].ra = 1.53;
				data_table[30].Z = 31;		data_table[30].A = 69;		data_table[30].m = 69.723;		data_table[30].rn = 3.9973;		data_table[30].ra = 1.81;
				data_table[31].Z = 32;		data_table[31].A = 74;		data_table[31].m = 72.61;		data_table[31].rn = 4.0742;		data_table[31].ra = 1.52;
				data_table[32].Z = 33;		data_table[32].A = 75;		data_table[32].m = 74.9216;		data_table[32].rn = 4.0968;		data_table[32].ra = 1.33;
				data_table[33].Z = 34;		data_table[33].A = 80;		data_table[33].m = 78.96;		data_table[33].rn = 4.14;		data_table[33].ra = 1.22;
				data_table[34].Z = 35;		data_table[34].A = 79;		data_table[34].m = 79.904;		data_table[34].rn = 4.1629;		data_table[34].ra = 1.12;
				data_table[35].Z = 36;		data_table[35].A = 84;		data_table[35].m = 83.8;		data_table[35].rn = 4.1883;		data_table[35].ra = 1.03;
				data_table[36].Z = 37;		data_table[36].A = 85;		data_table[36].m = 85.4678;		data_table[36].rn = 4.2037;		data_table[36].ra = 2.98;
				data_table[37].Z = 38;		data_table[37].A = 88;		data_table[37].m = 87.62;		data_table[37].rn = 4.2246;		data_table[37].ra = 2.45;
				data_table[38].Z = 39;		data_table[38].A = 89;		data_table[38].m = 88.9059;		data_table[38].rn = 4.2438;		data_table[38].ra = 2.27;
				data_table[39].Z = 40;		data_table[39].A = 90;		data_table[39].m = 91.224;		data_table[39].rn = 4.269;		data_table[39].ra = 2.16;
				data_table[40].Z = 41;		data_table[40].A = 93;		data_table[40].m = 92.9064;		data_table[40].rn = 4.324;		data_table[40].ra = 2.08;
				data_table[41].Z = 42;		data_table[41].A = 98;		data_table[41].m = 95.94;		data_table[41].rn = 4.409;		data_table[41].ra = 2.01;
				data_table[42].Z = 43;		data_table[42].A = 98;		data_table[42].m = 98;			data_table[42].rn = 4.413;		data_table[42].ra = 1.95;
				data_table[43].Z = 44;		data_table[43].A = 102;		data_table[43].m = 101.07;		data_table[43].rn = 4.4811;		data_table[43].ra = 1.89;
				data_table[44].Z = 45;		data_table[44].A = 103;		data_table[44].m = 102.9055;	data_table[44].rn = 4.4945;		data_table[44].ra = 1.83;
				data_table[45].Z = 46;		data_table[45].A = 106;		data_table[45].m = 106.42;		data_table[45].rn = 4.5324;		data_table[45].ra = 1.79;
				data_table[46].Z = 47;		data_table[46].A = 107;		data_table[46].m = 107.868;		data_table[46].rn = 4.5464;		data_table[46].ra = 1.75;
				data_table[47].Z = 48;		data_table[47].A = 114;		data_table[47].m = 112.41;		data_table[47].rn = 4.6068;		data_table[47].ra = 1.71;
				data_table[48].Z = 49;		data_table[48].A = 115;		data_table[48].m = 114.82;		data_table[48].rn = 4.6155;		data_table[48].ra = 2;
				data_table[49].Z = 50;		data_table[49].A = 120;		data_table[49].m = 118.71;		data_table[49].rn = 4.6525;		data_table[49].ra = 1.72;
				data_table[50].Z = 51;		data_table[50].A = 121;		data_table[50].m = 121.757;		data_table[50].rn = 4.6802;		data_table[50].ra = 1.53;
				data_table[51].Z = 52;		data_table[51].A = 130;		data_table[51].m = 127.6;		data_table[51].rn = 4.742;		data_table[51].ra = 1.42;
				data_table[52].Z = 53;		data_table[52].A = 127;		data_table[52].m = 126.9045;	data_table[52].rn = 4.75;		data_table[52].ra = 1.32;
				data_table[53].Z = 54;		data_table[53].A = 132;		data_table[53].m = 131.29;		data_table[53].rn = 4.785;		data_table[53].ra = 1.24;
				data_table[54].Z = 55;		data_table[54].A = 133;		data_table[54].m = 132.9054;	data_table[54].rn = 4.804;		data_table[54].ra = 3.34;
				data_table[55].Z = 56;		data_table[55].A = 138;		data_table[55].m = 137.33;		data_table[55].rn = 4.837;		data_table[55].ra = 2.76;
				data_table[56].Z = 57;		data_table[56].A = 139;		data_table[56].m = 138.9055;	data_table[56].rn = 4.8549;		data_table[56].ra = 2.74;
				data_table[57].Z = 58;		data_table[57].A = 140;		data_table[57].m = 140.12;		data_table[57].rn = 4.8773;		data_table[57].ra = 2.7;
				data_table[58].Z = 59;		data_table[58].A = 141;		data_table[58].m = 140.9077;	data_table[58].rn = 4.8919;		data_table[58].ra = 2.67;
				data_table[59].Z = 60;		data_table[59].A = 142;		data_table[59].m = 144.24;		data_table[59].rn = 4.9115;		data_table[59].ra = 2.64;
				data_table[60].Z = 61;		data_table[60].A = 145;		data_table[60].m = 145;			data_table[60].rn = 4.953;		data_table[60].ra = 2.62;
				data_table[61].Z = 62;		data_table[61].A = 152;		data_table[61].m = 150.36;		data_table[61].rn = 5.0823;		data_table[61].ra = 2.59;
				data_table[62].Z = 63;		data_table[62].A = 153;		data_table[62].m = 151.965;		data_table[62].rn = 5.1093;		data_table[62].ra = 2.56;
				data_table[63].Z = 64;		data_table[63].A = 158;		data_table[63].m = 157.25;		data_table[63].rn = 5.1578;		data_table[63].ra = 2.54;
				data_table[64].Z = 65;		data_table[64].A = 159;		data_table[64].m = 158.9253;	data_table[64].rn = 5.06;		data_table[64].ra = 2.51;
				data_table[65].Z = 66;		data_table[65].A = 164;		data_table[65].m = 162.5;		data_table[65].rn = 5.224;		data_table[65].ra = 2.49;
				data_table[66].Z = 67;		data_table[66].A = 165;		data_table[66].m = 164.9303;	data_table[66].rn = 5.2022;		data_table[66].ra = 2.47;
				data_table[67].Z = 68;		data_table[67].A = 166;		data_table[67].m = 167.26;		data_table[67].rn = 5.2527;		data_table[67].ra = 2.45;
				data_table[68].Z = 69;		data_table[68].A = 169;		data_table[68].m = 168.9342;	data_table[68].rn = 5.2256;		data_table[68].ra = 2.42;
				data_table[69].Z = 70;		data_table[69].A = 174;		data_table[69].m = 173.04;		data_table[69].rn = 5.3105;		data_table[69].ra = 2.4;
				data_table[70].Z = 71;		data_table[70].A = 175;		data_table[70].m = 174.967;		data_table[70].rn = 5.37;		data_table[70].ra = 2.25;
				data_table[71].Z = 72;		data_table[71].A = 180;		data_table[71].m = 178.49;		data_table[71].rn = 5.3482;		data_table[71].ra = 2.16;
				data_table[72].Z = 73;		data_table[72].A = 181;		data_table[72].m = 180.9479;	data_table[72].rn = 5.3507;		data_table[72].ra = 2.09;
				data_table[73].Z = 74;		data_table[73].A = 184;		data_table[73].m = 183.85;		data_table[73].rn = 5.3646;		data_table[73].ra = 2.02;
				data_table[74].Z = 75;		data_table[74].A = 187;		data_table[74].m = 186.207;		data_table[74].rn = 5.3697;		data_table[74].ra = 1.97;
				data_table[75].Z = 76;		data_table[75].A = 192;		data_table[75].m = 190.2;		data_table[75].rn = 5.4122;		data_table[75].ra = 1.92;
				data_table[76].Z = 77;		data_table[76].A = 193;		data_table[76].m = 192.22;		data_table[76].rn = 5.402;		data_table[76].ra = 1.87;
				data_table[77].Z = 78;		data_table[77].A = 195;		data_table[77].m = 195.08;		data_table[77].rn = 5.425;		data_table[77].ra = 1.83;
				data_table[78].Z = 79;		data_table[78].A = 197;		data_table[78].m = 196.9665;	data_table[78].rn = 5.4379;		data_table[78].ra = 1.79;
				data_table[79].Z = 80;		data_table[79].A = 202;		data_table[79].m = 200.59;		data_table[79].rn = 5.4637;		data_table[79].ra = 1.76;
				data_table[80].Z = 81;		data_table[80].A = 205;		data_table[80].m = 204.383;		data_table[80].rn = 5.4757;		data_table[80].ra = 2.08;
				data_table[81].Z = 82;		data_table[81].A = 208;		data_table[81].m = 207.2;		data_table[81].rn = 5.501;		data_table[81].ra = 1.81;
				data_table[82].Z = 83;		data_table[82].A = 209;		data_table[82].m = 208.9804;	data_table[82].rn = 5.521;		data_table[82].ra = 1.63;
				data_table[83].Z = 84;		data_table[83].A = 209;		data_table[83].m = 209;			data_table[83].rn = 5.5262;		data_table[83].ra = 1.53;
				data_table[84].Z = 85;		data_table[84].A = 210;		data_table[84].m = 210;			data_table[84].rn = 5.531;		data_table[84].ra = 1.43;
				data_table[85].Z = 86;		data_table[85].A = 222;		data_table[85].m = 222;			data_table[85].rn = 5.6547;		data_table[85].ra = 1.34;
				data_table[86].Z = 87;		data_table[86].A = 223;		data_table[86].m = 223;			data_table[86].rn = 5.6584;		data_table[86].ra = 2.7;
				data_table[87].Z = 88;		data_table[87].A = 226;		data_table[87].m = 226.0254;	data_table[87].rn = 5.684;		data_table[87].ra = 2.23;
				data_table[88].Z = 89;		data_table[88].A = 227;		data_table[88].m = 227;			data_table[88].rn = 5.712;		data_table[88].ra = 1.88;
				data_table[89].Z = 90;		data_table[89].A = 232;		data_table[89].m = 232.0381;	data_table[89].rn = 5.78;		data_table[89].ra = 1.8;
				data_table[90].Z = 91;		data_table[90].A = 231;		data_table[90].m = 231.0359;	data_table[90].rn = 5.766;		data_table[90].ra = 1.61;
				data_table[91].Z = 92;		data_table[91].A = 238;		data_table[91].m = 238.029;		data_table[91].rn = 5.8571;		data_table[91].ra = 1.38;
				data_table[92].Z = 93;		data_table[92].A = 237;		data_table[92].m = 237.0482;	data_table[92].rn = 5.841;		data_table[92].ra = 1.3;
				data_table[93].Z = 94;		data_table[93].A = 244;		data_table[93].m = 244;			data_table[93].rn = 5.8948;		data_table[93].ra = 1.51;
				data_table[94].Z = 95;		data_table[94].A = 243;		data_table[94].m = 243;			data_table[94].rn = 5.9042;		data_table[94].ra = 1.84;
				data_table[95].Z = 96;		data_table[95].A = 247;		data_table[95].m = 247;			data_table[95].rn = 5.963;		data_table[95].ra = 1.84;
				data_table[96].Z = 97;		data_table[96].A = 247;		data_table[96].m = 247;			data_table[96].rn = 5.93;		data_table[96].ra = 1.84;
				data_table[97].Z = 98;		data_table[97].A = 251;		data_table[97].m = 251;			data_table[97].rn = 6.02;		data_table[97].ra = 1.84;
				data_table[98].Z = 99;		data_table[98].A = 252;		data_table[98].m = 252;			data_table[98].rn = 6.034;		data_table[98].ra = 1.84;
				data_table[99].Z = 100;		data_table[99].A = 257;		data_table[99].m = 257;			data_table[99].rn = 6.107;		data_table[99].ra = 1.84;
				data_table[100].Z = 101;	data_table[100].A = 258;	data_table[100].m = 258;		data_table[100].rn = 6.122;		data_table[100].ra = 1.84;
				data_table[101].Z = 102;	data_table[101].A = 259;	data_table[101].m = 259;		data_table[101].rn = 6.137;		data_table[101].ra = 1.84;
				data_table[102].Z = 103;	data_table[102].A = 262;	data_table[102].m = 260;		data_table[102].rn = 6.182;		data_table[102].ra = 1.84;
			}

			// Load eels edges
			void Load_eels_edges()
			{
				data_table[0].edges[0] = 13.00;	data_table[0].edges[1] = 0.00;	data_table[0].edges[2] = 0.00;	data_table[0].edges[3] = 0.00;	data_table[0].edges[4] = 0.00;	data_table[0].edges[5] = 0.00;	data_table[0].edges[6] = 0.00;	data_table[0].edges[7] = 0.00;	data_table[0].edges[8] = 0.00;	data_table[0].edges[9] = 0.00;	data_table[0].edges[10] = 0.00;	data_table[0].edges[11] = 0.00;
				data_table[1].edges[0] = 22.00;	data_table[1].edges[1] = 0.00;	data_table[1].edges[2] = 0.00;	data_table[1].edges[3] = 0.00;	data_table[1].edges[4] = 0.00;	data_table[1].edges[5] = 0.00;	data_table[1].edges[6] = 0.00;	data_table[1].edges[7] = 0.00;	data_table[1].edges[8] = 0.00;	data_table[1].edges[9] = 0.00;	data_table[1].edges[10] = 0.00;	data_table[1].edges[11] = 0.00;
				data_table[2].edges[0] = 55.00;	data_table[2].edges[1] = 0.00;	data_table[2].edges[2] = 0.00;	data_table[2].edges[3] = 0.00;	data_table[2].edges[4] = 0.00;	data_table[2].edges[5] = 0.00;	data_table[2].edges[6] = 0.00;	data_table[2].edges[7] = 0.00;	data_table[2].edges[8] = 0.00;	data_table[2].edges[9] = 0.00;	data_table[2].edges[10] = 0.00;	data_table[2].edges[11] = 0.00;
				data_table[3].edges[0] = 111.00;	data_table[3].edges[1] = 0.00;	data_table[3].edges[2] = 0.00;	data_table[3].edges[3] = 0.00;	data_table[3].edges[4] = 0.00;	data_table[3].edges[5] = 0.00;	data_table[3].edges[6] = 0.00;	data_table[3].edges[7] = 0.00;	data_table[3].edges[8] = 0.00;	data_table[3].edges[9] = 0.00;	data_table[3].edges[10] = 0.00;	data_table[3].edges[11] = 0.00;
				data_table[4].edges[0] = 188.00;	data_table[4].edges[1] = 0.00;	data_table[4].edges[2] = 0.00;	data_table[4].edges[3] = 0.00;	data_table[4].edges[4] = 0.00;	data_table[4].edges[5] = 0.00;	data_table[4].edges[6] = 0.00;	data_table[4].edges[7] = 0.00;	data_table[4].edges[8] = 0.00;	data_table[4].edges[9] = 0.00;	data_table[4].edges[10] = 0.00;	data_table[4].edges[11] = 0.00;
				data_table[5].edges[0] = 284.00;	data_table[5].edges[1] = 0.00;	data_table[5].edges[2] = 0.00;	data_table[5].edges[3] = 0.00;	data_table[5].edges[4] = 0.00;	data_table[5].edges[5] = 0.00;	data_table[5].edges[6] = 0.00;	data_table[5].edges[7] = 0.00;	data_table[5].edges[8] = 0.00;	data_table[5].edges[9] = 0.00;	data_table[5].edges[10] = 0.00;	data_table[5].edges[11] = 0.00;
				data_table[6].edges[0] = 401.00;	data_table[6].edges[1] = 0.00;	data_table[6].edges[2] = 0.00;	data_table[6].edges[3] = 0.00;	data_table[6].edges[4] = 0.00;	data_table[6].edges[5] = 0.00;	data_table[6].edges[6] = 0.00;	data_table[6].edges[7] = 0.00;	data_table[6].edges[8] = 0.00;	data_table[6].edges[9] = 0.00;	data_table[6].edges[10] = 0.00;	data_table[6].edges[11] = 0.00;
				data_table[7].edges[0] = 532.00;	data_table[7].edges[1] = 0.00;	data_table[7].edges[2] = 0.00;	data_table[7].edges[3] = 0.00;	data_table[7].edges[4] = 0.00;	data_table[7].edges[5] = 0.00;	data_table[7].edges[6] = 0.00;	data_table[7].edges[7] = 0.00;	data_table[7].edges[8] = 0.00;	data_table[7].edges[9] = 0.00;	data_table[7].edges[10] = 0.00;	data_table[7].edges[11] = 0.00;
				data_table[8].edges[0] = 685.00;	data_table[8].edges[1] = 0.00;	data_table[8].edges[2] = 0.00;	data_table[8].edges[3] = 0.00;	data_table[8].edges[4] = 0.00;	data_table[8].edges[5] = 0.00;	data_table[8].edges[6] = 0.00;	data_table[8].edges[7] = 0.00;	data_table[8].edges[8] = 0.00;	data_table[8].edges[9] = 0.00;	data_table[8].edges[10] = 0.00;	data_table[8].edges[11] = 0.00;
				data_table[9].edges[0] = 867.00;	data_table[9].edges[1] = 18.00;	data_table[9].edges[2] = 0.00;	data_table[9].edges[3] = 0.00;	data_table[9].edges[4] = 0.00;	data_table[9].edges[5] = 45.00;	data_table[9].edges[6] = 0.00;	data_table[9].edges[7] = 0.00;	data_table[9].edges[8] = 0.00;	data_table[9].edges[9] = 0.00;	data_table[9].edges[10] = 0.00;	data_table[9].edges[11] = 0.00;
				data_table[10].edges[0] = 1072.00;	data_table[10].edges[1] = 31.00;	data_table[10].edges[2] = 0.00;	data_table[10].edges[3] = 0.00;	data_table[10].edges[4] = 0.00;	data_table[10].edges[5] = 63.00;	data_table[10].edges[6] = 0.00;	data_table[10].edges[7] = 0.00;	data_table[10].edges[8] = 0.00;	data_table[10].edges[9] = 0.00;	data_table[10].edges[10] = 0.00;	data_table[10].edges[11] = 0.00;
				data_table[11].edges[0] = 1305.00;	data_table[11].edges[1] = 51.00;	data_table[11].edges[2] = 0.00;	data_table[11].edges[3] = 0.00;	data_table[11].edges[4] = 0.00;	data_table[11].edges[5] = 89.00;	data_table[11].edges[6] = 0.00;	data_table[11].edges[7] = 0.00;	data_table[11].edges[8] = 0.00;	data_table[11].edges[9] = 0.00;	data_table[11].edges[10] = 0.00;	data_table[11].edges[11] = 0.00;
				data_table[12].edges[0] = 1560.00;	data_table[12].edges[1] = 73.00;	data_table[12].edges[2] = 0.00;	data_table[12].edges[3] = 0.00;	data_table[12].edges[4] = 0.00;	data_table[12].edges[5] = 118.00;	data_table[12].edges[6] = 0.00;	data_table[12].edges[7] = 0.00;	data_table[12].edges[8] = 0.00;	data_table[12].edges[9] = 0.00;	data_table[12].edges[10] = 0.00;	data_table[12].edges[11] = 0.00;
				data_table[13].edges[0] = 1839.00;	data_table[13].edges[1] = 99.00;	data_table[13].edges[2] = 0.00;	data_table[13].edges[3] = 0.00;	data_table[13].edges[4] = 0.00;	data_table[13].edges[5] = 149.00;	data_table[13].edges[6] = 0.00;	data_table[13].edges[7] = 0.00;	data_table[13].edges[8] = 0.00;	data_table[13].edges[9] = 0.00;	data_table[13].edges[10] = 0.00;	data_table[13].edges[11] = 0.00;
				data_table[14].edges[0] = 2146.00;	data_table[14].edges[1] = 132.00;	data_table[14].edges[2] = 0.00;	data_table[14].edges[3] = 0.00;	data_table[14].edges[4] = 0.00;	data_table[14].edges[5] = 189.00;	data_table[14].edges[6] = 0.00;	data_table[14].edges[7] = 0.00;	data_table[14].edges[8] = 0.00;	data_table[14].edges[9] = 0.00;	data_table[14].edges[10] = 0.00;	data_table[14].edges[11] = 0.00;
				data_table[15].edges[0] = 2472.00;	data_table[15].edges[1] = 165.00;	data_table[15].edges[2] = 0.00;	data_table[15].edges[3] = 0.00;	data_table[15].edges[4] = 0.00;	data_table[15].edges[5] = 229.00;	data_table[15].edges[6] = 0.00;	data_table[15].edges[7] = 0.00;	data_table[15].edges[8] = 0.00;	data_table[15].edges[9] = 0.00;	data_table[15].edges[10] = 0.00;	data_table[15].edges[11] = 0.00;
				data_table[16].edges[0] = 2822.00;	data_table[16].edges[1] = 200.00;	data_table[16].edges[2] = 0.00;	data_table[16].edges[3] = 0.00;	data_table[16].edges[4] = 0.00;	data_table[16].edges[5] = 270.00;	data_table[16].edges[6] = 0.00;	data_table[16].edges[7] = 0.00;	data_table[16].edges[8] = 0.00;	data_table[16].edges[9] = 0.00;	data_table[16].edges[10] = 0.00;	data_table[16].edges[11] = 0.00;
				data_table[17].edges[0] = 3203.00;	data_table[17].edges[1] = 245.00;	data_table[17].edges[2] = 12.00;	data_table[17].edges[3] = 0.00;	data_table[17].edges[4] = 0.00;	data_table[17].edges[5] = 320.00;	data_table[17].edges[6] = 25.00;	data_table[17].edges[7] = 0.00;	data_table[17].edges[8] = 0.00;	data_table[17].edges[9] = 0.00;	data_table[17].edges[10] = 0.00;	data_table[17].edges[11] = 0.00;
				data_table[18].edges[0] = 3607.00;	data_table[18].edges[1] = 296.00;	data_table[18].edges[2] = 294.00;	data_table[18].edges[3] = 18.00;	data_table[18].edges[4] = 0.00;	data_table[18].edges[5] = 377.00;	data_table[18].edges[6] = 34.00;	data_table[18].edges[7] = 0.00;	data_table[18].edges[8] = 0.00;	data_table[18].edges[9] = 0.00;	data_table[18].edges[10] = 0.00;	data_table[18].edges[11] = 0.00;
				data_table[19].edges[0] = 4038.00;	data_table[19].edges[1] = 350.00;	data_table[19].edges[2] = 346.00;	data_table[19].edges[3] = 25.00;	data_table[19].edges[4] = 0.00;	data_table[19].edges[5] = 438.00;	data_table[19].edges[6] = 44.00;	data_table[19].edges[7] = 0.00;	data_table[19].edges[8] = 0.00;	data_table[19].edges[9] = 0.00;	data_table[19].edges[10] = 0.00;	data_table[19].edges[11] = 0.00;
				data_table[20].edges[0] = 4493.00;	data_table[20].edges[1] = 407.00;	data_table[20].edges[2] = 402.00;	data_table[20].edges[3] = 32.00;	data_table[20].edges[4] = 0.00;	data_table[20].edges[5] = 500.00;	data_table[20].edges[6] = 54.00;	data_table[20].edges[7] = 0.00;	data_table[20].edges[8] = 0.00;	data_table[20].edges[9] = 0.00;	data_table[20].edges[10] = 0.00;	data_table[20].edges[11] = 0.00;
				data_table[21].edges[0] = 4966.00;	data_table[21].edges[1] = 462.00;	data_table[21].edges[2] = 456.00;	data_table[21].edges[3] = 35.00;	data_table[21].edges[4] = 0.00;	data_table[21].edges[5] = 564.00;	data_table[21].edges[6] = 60.00;	data_table[21].edges[7] = 0.00;	data_table[21].edges[8] = 0.00;	data_table[21].edges[9] = 0.00;	data_table[21].edges[10] = 0.00;	data_table[21].edges[11] = 0.00;
				data_table[22].edges[0] = 521.00;	data_table[22].edges[1] = 513.00;	data_table[22].edges[2] = 38.00;	data_table[22].edges[3] = 0.00;	data_table[22].edges[4] = 0.00;	data_table[22].edges[5] = 628.00;	data_table[22].edges[6] = 66.00;	data_table[22].edges[7] = 0.00;	data_table[22].edges[8] = 0.00;	data_table[22].edges[9] = 0.00;	data_table[22].edges[10] = 0.00;	data_table[22].edges[11] = 0.00;
				data_table[23].edges[0] = 584.00;	data_table[23].edges[1] = 575.00;	data_table[23].edges[2] = 42.00;	data_table[23].edges[3] = 0.00;	data_table[23].edges[4] = 0.00;	data_table[23].edges[5] = 695.00;	data_table[23].edges[6] = 74.00;	data_table[23].edges[7] = 0.00;	data_table[23].edges[8] = 0.00;	data_table[23].edges[9] = 0.00;	data_table[23].edges[10] = 0.00;	data_table[23].edges[11] = 0.00;
				data_table[24].edges[0] = 651.00;	data_table[24].edges[1] = 640.00;	data_table[24].edges[2] = 49.00;	data_table[24].edges[3] = 0.00;	data_table[24].edges[4] = 0.00;	data_table[24].edges[5] = 769.00;	data_table[24].edges[6] = 84.00;	data_table[24].edges[7] = 0.00;	data_table[24].edges[8] = 0.00;	data_table[24].edges[9] = 0.00;	data_table[24].edges[10] = 0.00;	data_table[24].edges[11] = 0.00;
				data_table[25].edges[0] = 721.00;	data_table[25].edges[1] = 708.00;	data_table[25].edges[2] = 54.00;	data_table[25].edges[3] = 0.00;	data_table[25].edges[4] = 0.00;	data_table[25].edges[5] = 846.00;	data_table[25].edges[6] = 93.00;	data_table[25].edges[7] = 0.00;	data_table[25].edges[8] = 0.00;	data_table[25].edges[9] = 0.00;	data_table[25].edges[10] = 0.00;	data_table[25].edges[11] = 0.00;
				data_table[26].edges[0] = 794.00;	data_table[26].edges[1] = 779.00;	data_table[26].edges[2] = 60.00;	data_table[26].edges[3] = 0.00;	data_table[26].edges[4] = 0.00;	data_table[26].edges[5] = 926.00;	data_table[26].edges[6] = 101.00;	data_table[26].edges[7] = 0.00;	data_table[26].edges[8] = 0.00;	data_table[26].edges[9] = 0.00;	data_table[26].edges[10] = 0.00;	data_table[26].edges[11] = 0.00;
				data_table[27].edges[0] = 872.00;	data_table[27].edges[1] = 855.00;	data_table[27].edges[2] = 68.00;	data_table[27].edges[3] = 0.00;	data_table[27].edges[4] = 0.00;	data_table[27].edges[5] = 1008.00;	data_table[27].edges[6] = 112.00;	data_table[27].edges[7] = 0.00;	data_table[27].edges[8] = 0.00;	data_table[27].edges[9] = 0.00;	data_table[27].edges[10] = 0.00;	data_table[27].edges[11] = 0.00;
				data_table[28].edges[0] = 951.00;	data_table[28].edges[1] = 931.00;	data_table[28].edges[2] = 74.00;	data_table[28].edges[3] = 0.00;	data_table[28].edges[4] = 0.00;	data_table[28].edges[5] = 1096.00;	data_table[28].edges[6] = 120.00;	data_table[28].edges[7] = 0.00;	data_table[28].edges[8] = 0.00;	data_table[28].edges[9] = 0.00;	data_table[28].edges[10] = 0.00;	data_table[28].edges[11] = 0.00;
				data_table[29].edges[0] = 1043.00;	data_table[29].edges[1] = 1020.00;	data_table[29].edges[2] = 0.00;	data_table[29].edges[3] = 0.00;	data_table[29].edges[4] = 0.00;	data_table[29].edges[5] = 1194.00;	data_table[29].edges[6] = 136.00;	data_table[29].edges[7] = 87.00;	data_table[29].edges[8] = 0.00;	data_table[29].edges[9] = 0.00;	data_table[29].edges[10] = 0.00;	data_table[29].edges[11] = 0.00;
				data_table[30].edges[0] = 1142.00;	data_table[30].edges[1] = 1115.00;	data_table[30].edges[2] = 0.00;	data_table[30].edges[3] = 0.00;	data_table[30].edges[4] = 0.00;	data_table[30].edges[5] = 1298.00;	data_table[30].edges[6] = 158.00;	data_table[30].edges[7] = 103.00;	data_table[30].edges[8] = 0.00;	data_table[30].edges[9] = 0.00;	data_table[30].edges[10] = 0.00;	data_table[30].edges[11] = 0.00;
				data_table[31].edges[0] = 1248.00;	data_table[31].edges[1] = 1217.00;	data_table[31].edges[2] = 29.00;	data_table[31].edges[3] = 0.00;	data_table[31].edges[4] = 0.00;	data_table[31].edges[5] = 1414.00;	data_table[31].edges[6] = 180.00;	data_table[31].edges[7] = 121.00;	data_table[31].edges[8] = 0.00;	data_table[31].edges[9] = 0.00;	data_table[31].edges[10] = 0.00;	data_table[31].edges[11] = 0.00;
				data_table[32].edges[0] = 1359.00;	data_table[32].edges[1] = 1323.00;	data_table[32].edges[2] = 41.00;	data_table[32].edges[3] = 0.00;	data_table[32].edges[4] = 0.00;	data_table[32].edges[5] = 1526.00;	data_table[32].edges[6] = 203.00;	data_table[32].edges[7] = 140.00;	data_table[32].edges[8] = 0.00;	data_table[32].edges[9] = 0.00;	data_table[32].edges[10] = 0.00;	data_table[32].edges[11] = 0.00;
				data_table[33].edges[0] = 1476.00;	data_table[33].edges[1] = 1436.00;	data_table[33].edges[2] = 57.00;	data_table[33].edges[3] = 0.00;	data_table[33].edges[4] = 0.00;	data_table[33].edges[5] = 1654.00;	data_table[33].edges[6] = 231.00;	data_table[33].edges[7] = 162.00;	data_table[33].edges[8] = 0.00;	data_table[33].edges[9] = 0.00;	data_table[33].edges[10] = 0.00;	data_table[33].edges[11] = 0.00;
				data_table[34].edges[0] = 1596.00;	data_table[34].edges[1] = 1150.00;	data_table[34].edges[2] = 69.00;	data_table[34].edges[3] = 0.00;	data_table[34].edges[4] = 0.00;	data_table[34].edges[5] = 1782.00;	data_table[34].edges[6] = 256.00;	data_table[34].edges[7] = 181.00;	data_table[34].edges[8] = 0.00;	data_table[34].edges[9] = 0.00;	data_table[34].edges[10] = 0.00;	data_table[34].edges[11] = 0.00;
				data_table[35].edges[0] = 1727.00;	data_table[35].edges[1] = 1675.00;	data_table[35].edges[2] = 89.00;	data_table[35].edges[3] = 11.00;	data_table[35].edges[4] = 0.00;	data_table[35].edges[5] = 1921.00;	data_table[35].edges[6] = 289.00;	data_table[35].edges[7] = 214.00;	data_table[35].edges[8] = 24.00;	data_table[35].edges[9] = 0.00;	data_table[35].edges[10] = 0.00;	data_table[35].edges[11] = 0.00;
				data_table[36].edges[0] = 1864.00;	data_table[36].edges[1] = 1804.00;	data_table[36].edges[2] = 110.00;	data_table[36].edges[3] = 14.00;	data_table[36].edges[4] = 0.00;	data_table[36].edges[5] = 2065.00;	data_table[36].edges[6] = 322.00;	data_table[36].edges[7] = 247.00;	data_table[36].edges[8] = 238.00;	data_table[36].edges[9] = 29.00;	data_table[36].edges[10] = 0.00;	data_table[36].edges[11] = 0.00;
				data_table[37].edges[0] = 2007.00;	data_table[37].edges[1] = 1940.00;	data_table[37].edges[2] = 133.00;	data_table[37].edges[3] = 20.00;	data_table[37].edges[4] = 0.00;	data_table[37].edges[5] = 2216.00;	data_table[37].edges[6] = 357.00;	data_table[37].edges[7] = 280.00;	data_table[37].edges[8] = 269.00;	data_table[37].edges[9] = 38.00;	data_table[37].edges[10] = 0.00;	data_table[37].edges[11] = 0.00;
				data_table[38].edges[0] = 2155.00;	data_table[38].edges[1] = 2080.00;	data_table[38].edges[2] = 157.00;	data_table[38].edges[3] = 26.00;	data_table[38].edges[4] = 0.00;	data_table[38].edges[5] = 2372.00;	data_table[38].edges[6] = 394.00;	data_table[38].edges[7] = 312.00;	data_table[38].edges[8] = 300.00;	data_table[38].edges[9] = 45.00;	data_table[38].edges[10] = 0.00;	data_table[38].edges[11] = 0.00;
				data_table[39].edges[0] = 2307.00;	data_table[39].edges[1] = 2222.00;	data_table[39].edges[2] = 180.00;	data_table[39].edges[3] = 29.00;	data_table[39].edges[4] = 0.00;	data_table[39].edges[5] = 2532.00;	data_table[39].edges[6] = 430.00;	data_table[39].edges[7] = 344.00;	data_table[39].edges[8] = 330.00;	data_table[39].edges[9] = 51.00;	data_table[39].edges[10] = 0.00;	data_table[39].edges[11] = 0.00;
				data_table[40].edges[0] = 2465.00;	data_table[40].edges[1] = 2371.00;	data_table[40].edges[2] = 205.00;	data_table[40].edges[3] = 34.00;	data_table[40].edges[4] = 0.00;	data_table[40].edges[5] = 2698.00;	data_table[40].edges[6] = 468.00;	data_table[40].edges[7] = 378.00;	data_table[40].edges[8] = 363.00;	data_table[40].edges[9] = 58.00;	data_table[40].edges[10] = 0.00;	data_table[40].edges[11] = 0.00;
				data_table[41].edges[0] = 2625.00;	data_table[41].edges[1] = 2520.00;	data_table[41].edges[2] = 227.00;	data_table[41].edges[3] = 35.00;	data_table[41].edges[4] = 0.00;	data_table[41].edges[5] = 2865.00;	data_table[41].edges[6] = 505.00;	data_table[41].edges[7] = 410.00;	data_table[41].edges[8] = 392.00;	data_table[41].edges[9] = 62.00;	data_table[41].edges[10] = 0.00;	data_table[41].edges[11] = 0.00;
				data_table[42].edges[0] = 2793.00;	data_table[42].edges[1] = 2677.00;	data_table[42].edges[2] = 253.00;	data_table[42].edges[3] = 39.00;	data_table[42].edges[4] = 0.00;	data_table[42].edges[5] = 3042.00;	data_table[42].edges[6] = 544.00;	data_table[42].edges[7] = 445.00;	data_table[42].edges[8] = 425.00;	data_table[42].edges[9] = 68.00;	data_table[42].edges[10] = 0.00;	data_table[42].edges[11] = 0.00;
				data_table[43].edges[0] = 2967.00;	data_table[43].edges[1] = 2838.00;	data_table[43].edges[2] = 279.00;	data_table[43].edges[3] = 43.00;	data_table[43].edges[4] = 0.00;	data_table[43].edges[5] = 3224.00;	data_table[43].edges[6] = 585.00;	data_table[43].edges[7] = 483.00;	data_table[43].edges[8] = 461.00;	data_table[43].edges[9] = 75.00;	data_table[43].edges[10] = 0.00;	data_table[43].edges[11] = 0.00;
				data_table[44].edges[0] = 3146.00;	data_table[44].edges[1] = 3004.00;	data_table[44].edges[2] = 307.00;	data_table[44].edges[3] = 48.00;	data_table[44].edges[4] = 0.00;	data_table[44].edges[5] = 3412.00;	data_table[44].edges[6] = 627.00;	data_table[44].edges[7] = 521.00;	data_table[44].edges[8] = 496.00;	data_table[44].edges[9] = 81.00;	data_table[44].edges[10] = 0.00;	data_table[44].edges[11] = 0.00;
				data_table[45].edges[0] = 3330.00;	data_table[45].edges[1] = 3173.00;	data_table[45].edges[2] = 335.00;	data_table[45].edges[3] = 51.00;	data_table[45].edges[4] = 0.00;	data_table[45].edges[5] = 3604.00;	data_table[45].edges[6] = 670.00;	data_table[45].edges[7] = 559.00;	data_table[45].edges[8] = 531.00;	data_table[45].edges[9] = 86.00;	data_table[45].edges[10] = 0.00;	data_table[45].edges[11] = 0.00;
				data_table[46].edges[0] = 3524.00;	data_table[46].edges[1] = 3351.00;	data_table[46].edges[2] = 367.00;	data_table[46].edges[3] = 0.00;	data_table[46].edges[4] = 0.00;	data_table[46].edges[5] = 3806.00;	data_table[46].edges[6] = 717.00;	data_table[46].edges[7] = 602.00;	data_table[46].edges[8] = 571.00;	data_table[46].edges[9] = 95.00;	data_table[46].edges[10] = 56.00;	data_table[46].edges[11] = 0.00;
				data_table[47].edges[0] = 3727.00;	data_table[47].edges[1] = 3538.00;	data_table[47].edges[2] = 404.00;	data_table[47].edges[3] = 0.00;	data_table[47].edges[4] = 0.00;	data_table[47].edges[5] = 4018.00;	data_table[47].edges[6] = 770.00;	data_table[47].edges[7] = 651.00;	data_table[47].edges[8] = 616.00;	data_table[47].edges[9] = 108.00;	data_table[47].edges[10] = 67.00;	data_table[47].edges[11] = 0.00;
				data_table[48].edges[0] = 3938.00;	data_table[48].edges[1] = 3730.00;	data_table[48].edges[2] = 443.00;	data_table[48].edges[3] = 0.00;	data_table[48].edges[4] = 0.00;	data_table[48].edges[5] = 4237.00;	data_table[48].edges[6] = 826.00;	data_table[48].edges[7] = 702.00;	data_table[48].edges[8] = 664.00;	data_table[48].edges[9] = 122.00;	data_table[48].edges[10] = 77.00;	data_table[48].edges[11] = 0.00;
				data_table[49].edges[0] = 4156.00;	data_table[49].edges[1] = 3929.00;	data_table[49].edges[2] = 485.00;	data_table[49].edges[3] = 24.00;	data_table[49].edges[4] = 0.00;	data_table[49].edges[5] = 4465.00;	data_table[49].edges[6] = 884.00;	data_table[49].edges[7] = 756.00;	data_table[49].edges[8] = 714.00;	data_table[49].edges[9] = 136.00;	data_table[49].edges[10] = 89.00;	data_table[49].edges[11] = 0.00;
				data_table[50].edges[0] = 4380.00;	data_table[50].edges[1] = 4132.00;	data_table[50].edges[2] = 528.00;	data_table[50].edges[3] = 31.00;	data_table[50].edges[4] = 0.00;	data_table[50].edges[5] = 944.00;	data_table[50].edges[6] = 812.00;	data_table[50].edges[7] = 766.00;	data_table[50].edges[8] = 152.00;	data_table[50].edges[9] = 98.00;	data_table[50].edges[10] = 0.00;	data_table[50].edges[11] = 0.00;
				data_table[51].edges[0] = 4612.00;	data_table[51].edges[1] = 4341.00;	data_table[51].edges[2] = 40.00;	data_table[51].edges[3] = 572.00;	data_table[51].edges[4] = 0.00;	data_table[51].edges[5] = 1006.00;	data_table[51].edges[6] = 870.00;	data_table[51].edges[7] = 819.00;	data_table[51].edges[8] = 168.00;	data_table[51].edges[9] = 110.00;	data_table[51].edges[10] = 0.00;	data_table[51].edges[11] = 0.00;
				data_table[52].edges[0] = 4852.00;	data_table[52].edges[1] = 4557.00;	data_table[52].edges[2] = 619.00;	data_table[52].edges[3] = 50.00;	data_table[52].edges[4] = 0.00;	data_table[52].edges[5] = 1072.00;	data_table[52].edges[6] = 930.00;	data_table[52].edges[7] = 875.00;	data_table[52].edges[8] = 186.00;	data_table[52].edges[9] = 123.00;	data_table[52].edges[10] = 0.00;	data_table[52].edges[11] = 0.00;
				data_table[53].edges[0] = 4782.00;	data_table[53].edges[1] = 672.00;	data_table[53].edges[2] = 63.00;	data_table[53].edges[3] = 0.00;	data_table[53].edges[4] = 0.00;	data_table[53].edges[5] = 1145.00;	data_table[53].edges[6] = 999.00;	data_table[53].edges[7] = 937.00;	data_table[53].edges[8] = 208.00;	data_table[53].edges[9] = 147.00;	data_table[53].edges[10] = 0.00;	data_table[53].edges[11] = 0.00;
				data_table[54].edges[0] = 740.00;	data_table[54].edges[1] = 726.00;	data_table[54].edges[2] = 76.00;	data_table[54].edges[3] = 0.00;	data_table[54].edges[4] = 0.00;	data_table[54].edges[5] = 1217.00;	data_table[54].edges[6] = 1065.00;	data_table[54].edges[7] = 998.00;	data_table[54].edges[8] = 231.00;	data_table[54].edges[9] = 162.00;	data_table[54].edges[10] = 0.00;	data_table[54].edges[11] = 0.00;
				data_table[55].edges[0] = 796.00;	data_table[55].edges[1] = 781.00;	data_table[55].edges[2] = 90.00;	data_table[55].edges[3] = 15.00;	data_table[55].edges[4] = 0.00;	data_table[55].edges[5] = 1293.00;	data_table[55].edges[6] = 1137.00;	data_table[55].edges[7] = 1062.00;	data_table[55].edges[8] = 253.00;	data_table[55].edges[9] = 180.00;	data_table[55].edges[10] = 0.00;	data_table[55].edges[11] = 0.00;
				data_table[56].edges[0] = 849.00;	data_table[56].edges[1] = 832.00;	data_table[56].edges[2] = 99.00;	data_table[56].edges[3] = 14.00;	data_table[56].edges[4] = 0.00;	data_table[56].edges[5] = 1361.00;	data_table[56].edges[6] = 1204.00;	data_table[56].edges[7] = 1123.00;	data_table[56].edges[8] = 270.00;	data_table[56].edges[9] = 191.00;	data_table[56].edges[10] = 0.00;	data_table[56].edges[11] = 0.00;
				data_table[57].edges[0] = 901.00;	data_table[57].edges[1] = 883.00;	data_table[57].edges[2] = 110.00;	data_table[57].edges[3] = 20.00;	data_table[57].edges[4] = 0.00;	data_table[57].edges[5] = 1435.00;	data_table[57].edges[6] = 1273.00;	data_table[57].edges[7] = 1185.00;	data_table[57].edges[8] = 290.00;	data_table[57].edges[9] = 207.00;	data_table[57].edges[10] = 0.00;	data_table[57].edges[11] = 0.00;
				data_table[58].edges[0] = 951.00;	data_table[58].edges[1] = 931.00;	data_table[58].edges[2] = 113.00;	data_table[58].edges[3] = 22.00;	data_table[58].edges[4] = 0.00;	data_table[58].edges[5] = 1511.00;	data_table[58].edges[6] = 1337.00;	data_table[58].edges[7] = 1242.00;	data_table[58].edges[8] = 305.00;	data_table[58].edges[9] = 218.00;	data_table[58].edges[10] = 0.00;	data_table[58].edges[11] = 0.00;
				data_table[59].edges[0] = 1000.00;	data_table[59].edges[1] = 978.00;	data_table[59].edges[2] = 118.00;	data_table[59].edges[3] = 21.00;	data_table[59].edges[4] = 0.00;	data_table[59].edges[5] = 1575.00;	data_table[59].edges[6] = 1403.00;	data_table[59].edges[7] = 1297.00;	data_table[59].edges[8] = 315.00;	data_table[59].edges[9] = 225.00;	data_table[59].edges[10] = 0.00;	data_table[59].edges[11] = 0.00;
				data_table[60].edges[0] = 1052.00;	data_table[60].edges[1] = 1027.00;	data_table[60].edges[2] = 120.00;	data_table[60].edges[3] = 24.00;	data_table[60].edges[4] = 0.00;	data_table[60].edges[5] = 1649.00;	data_table[60].edges[6] = 1471.00;	data_table[60].edges[7] = 1357.00;	data_table[60].edges[8] = 331.00;	data_table[60].edges[9] = 236.00;	data_table[60].edges[10] = 0.00;	data_table[60].edges[11] = 0.00;
				data_table[61].edges[0] = 1106.00;	data_table[61].edges[1] = 1080.00;	data_table[61].edges[2] = 129.00;	data_table[61].edges[3] = 21.00;	data_table[61].edges[4] = 0.00;	data_table[61].edges[5] = 1723.00;	data_table[61].edges[6] = 1541.00;	data_table[61].edges[7] = 1420.00;	data_table[61].edges[8] = 346.00;	data_table[61].edges[9] = 247.00;	data_table[61].edges[10] = 0.00;	data_table[61].edges[11] = 0.00;
				data_table[62].edges[0] = 1161.00;	data_table[62].edges[1] = 1131.00;	data_table[62].edges[2] = 133.00;	data_table[62].edges[3] = 22.00;	data_table[62].edges[4] = 0.00;	data_table[62].edges[5] = 1800.00;	data_table[62].edges[6] = 1614.00;	data_table[62].edges[7] = 1481.00;	data_table[62].edges[8] = 360.00;	data_table[62].edges[9] = 257.00;	data_table[62].edges[10] = 0.00;	data_table[62].edges[11] = 0.00;
				data_table[63].edges[0] = 1217.00;	data_table[63].edges[1] = 1185.00;	data_table[63].edges[2] = 140.00;	data_table[63].edges[3] = 20.00;	data_table[63].edges[4] = 0.00;	data_table[63].edges[5] = 1881.00;	data_table[63].edges[6] = 1688.00;	data_table[63].edges[7] = 1544.00;	data_table[63].edges[8] = 376.00;	data_table[63].edges[9] = 271.00;	data_table[63].edges[10] = 0.00;	data_table[63].edges[11] = 0.00;
				data_table[64].edges[0] = 1275.00;	data_table[64].edges[1] = 1241.00;	data_table[64].edges[2] = 147.00;	data_table[64].edges[3] = 25.00;	data_table[64].edges[4] = 0.00;	data_table[64].edges[5] = 1967.00;	data_table[64].edges[6] = 1768.00;	data_table[64].edges[7] = 1611.00;	data_table[64].edges[8] = 398.00;	data_table[64].edges[9] = 285.00;	data_table[64].edges[10] = 0.00;	data_table[64].edges[11] = 0.00;
				data_table[65].edges[0] = 1332.00;	data_table[65].edges[1] = 1295.00;	data_table[65].edges[2] = 154.00;	data_table[65].edges[3] = 26.00;	data_table[65].edges[4] = 0.00;	data_table[65].edges[5] = 2047.00;	data_table[65].edges[6] = 1842.00;	data_table[65].edges[7] = 1676.00;	data_table[65].edges[8] = 416.00;	data_table[65].edges[9] = 293.00;	data_table[65].edges[10] = 0.00;	data_table[65].edges[11] = 0.00;
				data_table[66].edges[0] = 1391.00;	data_table[66].edges[1] = 1351.00;	data_table[66].edges[2] = 161.00;	data_table[66].edges[3] = 20.00;	data_table[66].edges[4] = 0.00;	data_table[66].edges[5] = 2128.00;	data_table[66].edges[6] = 1923.00;	data_table[66].edges[7] = 1741.00;	data_table[66].edges[8] = 436.00;	data_table[66].edges[9] = 307.00;	data_table[66].edges[10] = 51.00;	data_table[66].edges[11] = 0.00;
				data_table[67].edges[0] = 1453.00;	data_table[67].edges[1] = 1409.00;	data_table[67].edges[2] = 168.00;	data_table[67].edges[3] = 29.00;	data_table[67].edges[4] = 0.00;	data_table[67].edges[5] = 2206.00;	data_table[67].edges[6] = 2006.00;	data_table[67].edges[7] = 1812.00;	data_table[67].edges[8] = 449.00;	data_table[67].edges[9] = 320.00;	data_table[67].edges[10] = 60.00;	data_table[67].edges[11] = 0.00;
				data_table[68].edges[0] = 1515.00;	data_table[68].edges[1] = 1468.00;	data_table[68].edges[2] = 180.00;	data_table[68].edges[3] = 32.00;	data_table[68].edges[4] = 0.00;	data_table[68].edges[5] = 2307.00;	data_table[68].edges[6] = 2090.00;	data_table[68].edges[7] = 1884.00;	data_table[68].edges[8] = 472.00;	data_table[68].edges[9] = 337.00;	data_table[68].edges[10] = 53.00;	data_table[68].edges[11] = 0.00;
				data_table[69].edges[0] = 1576.00;	data_table[69].edges[1] = 1528.00;	data_table[69].edges[2] = 185.00;	data_table[69].edges[3] = 24.00;	data_table[69].edges[4] = 0.00;	data_table[69].edges[5] = 2398.00;	data_table[69].edges[6] = 2173.00;	data_table[69].edges[7] = 1950.00;	data_table[69].edges[8] = 487.00;	data_table[69].edges[9] = 343.00;	data_table[69].edges[10] = 54.00;	data_table[69].edges[11] = 0.00;
				data_table[70].edges[0] = 1639.00;	data_table[70].edges[1] = 1588.00;	data_table[70].edges[2] = 195.00;	data_table[70].edges[3] = 28.00;	data_table[70].edges[4] = 0.00;	data_table[70].edges[5] = 2491.00;	data_table[70].edges[6] = 2263.00;	data_table[70].edges[7] = 2024.00;	data_table[70].edges[8] = 506.00;	data_table[70].edges[9] = 359.00;	data_table[70].edges[10] = 57.00;	data_table[70].edges[11] = 0.00;
				data_table[71].edges[0] = 1716.00;	data_table[71].edges[1] = 1662.00;	data_table[71].edges[2] = 31.00;	data_table[71].edges[3] = 0.00;	data_table[71].edges[4] = 0.00;	data_table[71].edges[5] = 2601.00;	data_table[71].edges[6] = 2365.00;	data_table[71].edges[7] = 2108.00;	data_table[71].edges[8] = 538.00;	data_table[71].edges[9] = 380.00;	data_table[71].edges[10] = 214.00;	data_table[71].edges[11] = 65.00;
				data_table[72].edges[0] = 1793.00;	data_table[72].edges[1] = 1735.00;	data_table[72].edges[2] = 36.00;	data_table[72].edges[3] = 0.00;	data_table[72].edges[4] = 0.00;	data_table[72].edges[5] = 2708.00;	data_table[72].edges[6] = 2469.00;	data_table[72].edges[7] = 2194.00;	data_table[72].edges[8] = 565.00;	data_table[72].edges[9] = 404.00;	data_table[72].edges[10] = 229.00;	data_table[72].edges[11] = 71.00;
				data_table[73].edges[0] = 1872.00;	data_table[73].edges[1] = 1809.00;	data_table[73].edges[2] = 36.00;	data_table[73].edges[3] = 0.00;	data_table[73].edges[4] = 0.00;	data_table[73].edges[5] = 2820.00;	data_table[73].edges[6] = 2575.00;	data_table[73].edges[7] = 2281.00;	data_table[73].edges[8] = 595.00;	data_table[73].edges[9] = 425.00;	data_table[73].edges[10] = 245.00;	data_table[73].edges[11] = 77.00;
				data_table[74].edges[0] = 1949.00;	data_table[74].edges[1] = 1883.00;	data_table[74].edges[2] = 35.00;	data_table[74].edges[3] = 0.00;	data_table[74].edges[4] = 0.00;	data_table[74].edges[5] = 2932.00;	data_table[74].edges[6] = 2682.00;	data_table[74].edges[7] = 2367.00;	data_table[74].edges[8] = 625.00;	data_table[74].edges[9] = 444.00;	data_table[74].edges[10] = 260.00;	data_table[74].edges[11] = 83.00;
				data_table[75].edges[0] = 2031.00;	data_table[75].edges[1] = 1960.00;	data_table[75].edges[2] = 45.00;	data_table[75].edges[3] = 0.00;	data_table[75].edges[4] = 0.00;	data_table[75].edges[5] = 2792.00;	data_table[75].edges[6] = 2457.00;	data_table[75].edges[7] = 654.00;	data_table[75].edges[8] = 468.00;	data_table[75].edges[9] = 273.00;	data_table[75].edges[10] = 46.00;	data_table[75].edges[11] = 84.00;
				data_table[76].edges[0] = 2116.00;	data_table[76].edges[1] = 2040.00;	data_table[76].edges[2] = 50.00;	data_table[76].edges[3] = 0.00;	data_table[76].edges[4] = 0.00;	data_table[76].edges[5] = 2909.00;	data_table[76].edges[6] = 2551.00;	data_table[76].edges[7] = 690.00;	data_table[76].edges[8] = 494.00;	data_table[76].edges[9] = 295.00;	data_table[76].edges[10] = 60.00;	data_table[76].edges[11] = 95.00;
				data_table[77].edges[0] = 2202.00;	data_table[77].edges[1] = 2122.00;	data_table[77].edges[2] = 52.00;	data_table[77].edges[3] = 0.00;	data_table[77].edges[4] = 0.00;	data_table[77].edges[5] = 3026.00;	data_table[77].edges[6] = 2645.00;	data_table[77].edges[7] = 722.00;	data_table[77].edges[8] = 519.00;	data_table[77].edges[9] = 313.00;	data_table[77].edges[10] = 102.00;	data_table[77].edges[11] = 71.00;
				data_table[78].edges[0] = 2291.00;	data_table[78].edges[1] = 2206.00;	data_table[78].edges[2] = 54.00;	data_table[78].edges[3] = 0.00;	data_table[78].edges[4] = 0.00;	data_table[78].edges[5] = 3148.00;	data_table[78].edges[6] = 2743.00;	data_table[78].edges[7] = 759.00;	data_table[78].edges[8] = 545.00;	data_table[78].edges[9] = 334.00;	data_table[78].edges[10] = 108.00;	data_table[78].edges[11] = 83.00;
				data_table[79].edges[0] = 2385.00;	data_table[79].edges[1] = 2295.00;	data_table[79].edges[2] = 58.00;	data_table[79].edges[3] = 0.00;	data_table[79].edges[4] = 0.00;	data_table[79].edges[5] = 3278.00;	data_table[79].edges[6] = 2847.00;	data_table[79].edges[7] = 800.00;	data_table[79].edges[8] = 571.00;	data_table[79].edges[9] = 360.00;	data_table[79].edges[10] = 120.00;	data_table[79].edges[11] = 98.00;
				data_table[80].edges[0] = 2485.00;	data_table[80].edges[1] = 2389.00;	data_table[80].edges[2] = 75.00;	data_table[80].edges[3] = 0.00;	data_table[80].edges[4] = 0.00;	data_table[80].edges[5] = 3416.00;	data_table[80].edges[6] = 2957.00;	data_table[80].edges[7] = 845.00;	data_table[80].edges[8] = 609.00;	data_table[80].edges[9] = 386.00;	data_table[80].edges[10] = 136.00;	data_table[80].edges[11] = 118.00;
				data_table[81].edges[0] = 2586.00;	data_table[81].edges[1] = 2484.00;	data_table[81].edges[2] = 86.00;	data_table[81].edges[3] = 19.00;	data_table[81].edges[4] = 0.00;	data_table[81].edges[5] = 3554.00;	data_table[81].edges[6] = 3066.00;	data_table[81].edges[7] = 894.00;	data_table[81].edges[8] = 644.00;	data_table[81].edges[9] = 413.00;	data_table[81].edges[10] = 147.00;	data_table[81].edges[11] = 138.00;
				data_table[82].edges[0] = 2688.00;	data_table[82].edges[1] = 2580.00;	data_table[82].edges[2] = 93.00;	data_table[82].edges[3] = 24.00;	data_table[82].edges[4] = 0.00;	data_table[82].edges[5] = 3696.00;	data_table[82].edges[6] = 3177.00;	data_table[82].edges[7] = 938.00;	data_table[82].edges[8] = 679.00;	data_table[82].edges[9] = 440.00;	data_table[82].edges[10] = 159.00;	data_table[82].edges[11] = 157.00;
				data_table[83].edges[0] = 3491.00;	data_table[83].edges[1] = 3332.00;	data_table[83].edges[2] = 0.00;	data_table[83].edges[3] = 0.00;	data_table[83].edges[4] = 0.00;	data_table[83].edges[5] = 4046.00;	data_table[83].edges[6] = 0.00;	data_table[83].edges[7] = 0.00;	data_table[83].edges[8] = 0.00;	data_table[83].edges[9] = 0.00;	data_table[83].edges[10] = 0.00;	data_table[83].edges[11] = 0.00;
				data_table[84].edges[0] = 0.00;	data_table[84].edges[1] = 0.00;	data_table[84].edges[2] = 0.00;	data_table[84].edges[3] = 0.00;	data_table[84].edges[4] = 0.00;	data_table[84].edges[5] = 0.00;	data_table[84].edges[6] = 0.00;	data_table[84].edges[7] = 0.00;	data_table[84].edges[8] = 0.00;	data_table[84].edges[9] = 0.00;	data_table[84].edges[10] = 0.00;	data_table[84].edges[11] = 0.00;
				data_table[85].edges[0] = 0.00;	data_table[85].edges[1] = 0.00;	data_table[85].edges[2] = 0.00;	data_table[85].edges[3] = 0.00;	data_table[85].edges[4] = 0.00;	data_table[85].edges[5] = 0.00;	data_table[85].edges[6] = 0.00;	data_table[85].edges[7] = 0.00;	data_table[85].edges[8] = 0.00;	data_table[85].edges[9] = 0.00;	data_table[85].edges[10] = 0.00;	data_table[85].edges[11] = 0.00;
				data_table[86].edges[0] = 0.00;	data_table[86].edges[1] = 0.00;	data_table[86].edges[2] = 0.00;	data_table[86].edges[3] = 0.00;	data_table[86].edges[4] = 0.00;	data_table[86].edges[5] = 0.00;	data_table[86].edges[6] = 0.00;	data_table[86].edges[7] = 0.00;	data_table[86].edges[8] = 0.00;	data_table[86].edges[9] = 0.00;	data_table[86].edges[10] = 0.00;	data_table[86].edges[11] = 0.00;
				data_table[87].edges[0] = 0.00;	data_table[87].edges[1] = 0.00;	data_table[87].edges[2] = 0.00;	data_table[87].edges[3] = 0.00;	data_table[87].edges[4] = 0.00;	data_table[87].edges[5] = 0.00;	data_table[87].edges[6] = 0.00;	data_table[87].edges[7] = 0.00;	data_table[87].edges[8] = 0.00;	data_table[87].edges[9] = 0.00;	data_table[87].edges[10] = 0.00;	data_table[87].edges[11] = 0.00;
				data_table[88].edges[0] = 0.00;	data_table[88].edges[1] = 0.00;	data_table[88].edges[2] = 0.00;	data_table[88].edges[3] = 0.00;	data_table[88].edges[4] = 0.00;	data_table[88].edges[5] = 0.00;	data_table[88].edges[6] = 0.00;	data_table[88].edges[7] = 0.00;	data_table[88].edges[8] = 0.00;	data_table[88].edges[9] = 0.00;	data_table[88].edges[10] = 0.00;	data_table[88].edges[11] = 0.00;
				data_table[89].edges[0] = 3491.00;	data_table[89].edges[1] = 3332.00;	data_table[89].edges[2] = 344.00;	data_table[89].edges[3] = 335.00;	data_table[89].edges[4] = 88.00;	data_table[89].edges[5] = 4046.00;	data_table[89].edges[6] = 1329.00;	data_table[89].edges[7] = 967.00;	data_table[89].edges[8] = 714.00;	data_table[89].edges[9] = 676.00;	data_table[89].edges[10] = 290.00;	data_table[89].edges[11] = 182.00;
				data_table[90].edges[0] = 0.00;	data_table[90].edges[1] = 0.00;	data_table[90].edges[2] = 0.00;	data_table[90].edges[3] = 0.00;	data_table[90].edges[4] = 0.00;	data_table[90].edges[5] = 0.00;	data_table[90].edges[6] = 0.00;	data_table[90].edges[7] = 0.00;	data_table[90].edges[8] = 0.00;	data_table[90].edges[9] = 0.00;	data_table[90].edges[10] = 0.00;	data_table[90].edges[11] = 0.00;
				data_table[91].edges[0] = 3728.00;	data_table[91].edges[1] = 3552.00;	data_table[91].edges[2] = 391.00;	data_table[91].edges[3] = 381.00;	data_table[91].edges[4] = 96.00;	data_table[91].edges[5] = 4303.00;	data_table[91].edges[6] = 1441.00;	data_table[91].edges[7] = 1045.00;	data_table[91].edges[8] = 780.00;	data_table[91].edges[9] = 738.00;	data_table[91].edges[10] = 324.00;	data_table[91].edges[11] = 195.00;
				data_table[92].edges[0] = 0.00;	data_table[92].edges[1] = 0.00;	data_table[92].edges[2] = 0.00;	data_table[92].edges[3] = 0.00;	data_table[92].edges[4] = 0.00;	data_table[92].edges[5] = 0.00;	data_table[92].edges[6] = 0.00;	data_table[92].edges[7] = 0.00;	data_table[92].edges[8] = 0.00;	data_table[92].edges[9] = 0.00;	data_table[92].edges[10] = 0.00;	data_table[92].edges[11] = 0.00;
				data_table[93].edges[0] = 0.00;	data_table[93].edges[1] = 0.00;	data_table[93].edges[2] = 0.00;	data_table[93].edges[3] = 0.00;	data_table[93].edges[4] = 0.00;	data_table[93].edges[5] = 0.00;	data_table[93].edges[6] = 0.00;	data_table[93].edges[7] = 0.00;	data_table[93].edges[8] = 0.00;	data_table[93].edges[9] = 0.00;	data_table[93].edges[10] = 0.00;	data_table[93].edges[11] = 0.00;
				data_table[94].edges[0] = 0.00;	data_table[94].edges[1] = 0.00;	data_table[94].edges[2] = 0.00;	data_table[94].edges[3] = 0.00;	data_table[94].edges[4] = 0.00;	data_table[94].edges[5] = 0.00;	data_table[94].edges[6] = 0.00;	data_table[94].edges[7] = 0.00;	data_table[94].edges[8] = 0.00;	data_table[94].edges[9] = 0.00;	data_table[94].edges[10] = 0.00;	data_table[94].edges[11] = 0.00;
				data_table[95].edges[0] = 0.00;	data_table[95].edges[1] = 0.00;	data_table[95].edges[2] = 0.00;	data_table[95].edges[3] = 0.00;	data_table[95].edges[4] = 0.00;	data_table[95].edges[5] = 0.00;	data_table[95].edges[6] = 0.00;	data_table[95].edges[7] = 0.00;	data_table[95].edges[8] = 0.00;	data_table[95].edges[9] = 0.00;	data_table[95].edges[10] = 0.00;	data_table[95].edges[11] = 0.00;
				data_table[96].edges[0] = 0.00;	data_table[96].edges[1] = 0.00;	data_table[96].edges[2] = 0.00;	data_table[96].edges[3] = 0.00;	data_table[96].edges[4] = 0.00;	data_table[96].edges[5] = 0.00;	data_table[96].edges[6] = 0.00;	data_table[96].edges[7] = 0.00;	data_table[96].edges[8] = 0.00;	data_table[96].edges[9] = 0.00;	data_table[96].edges[10] = 0.00;	data_table[96].edges[11] = 0.00;
				data_table[97].edges[0] = 0.00;	data_table[97].edges[1] = 0.00;	data_table[97].edges[2] = 0.00;	data_table[97].edges[3] = 0.00;	data_table[97].edges[4] = 0.00;	data_table[97].edges[5] = 0.00;	data_table[97].edges[6] = 0.00;	data_table[97].edges[7] = 0.00;	data_table[97].edges[8] = 0.00;	data_table[97].edges[9] = 0.00;	data_table[97].edges[10] = 0.00;	data_table[97].edges[11] = 0.00;
				data_table[98].edges[0] = 0.00;	data_table[98].edges[1] = 0.00;	data_table[98].edges[2] = 0.00;	data_table[98].edges[3] = 0.00;	data_table[98].edges[4] = 0.00;	data_table[98].edges[5] = 0.00;	data_table[98].edges[6] = 0.00;	data_table[98].edges[7] = 0.00;	data_table[98].edges[8] = 0.00;	data_table[98].edges[9] = 0.00;	data_table[98].edges[10] = 0.00;	data_table[98].edges[11] = 0.00;
				data_table[99].edges[0] = 0.00;	data_table[99].edges[1] = 0.00;	data_table[99].edges[2] = 0.00;	data_table[99].edges[3] = 0.00;	data_table[99].edges[4] = 0.00;	data_table[99].edges[5] = 0.00;	data_table[99].edges[6] = 0.00;	data_table[99].edges[7] = 0.00;	data_table[99].edges[8] = 0.00;	data_table[99].edges[9] = 0.00;	data_table[99].edges[10] = 0.00;	data_table[99].edges[11] = 0.00;
				data_table[100].edges[0] = 0.00;	data_table[100].edges[1] = 0.00;	data_table[100].edges[2] = 0.00;	data_table[100].edges[3] = 0.00;	data_table[100].edges[4] = 0.00;	data_table[100].edges[5] = 0.00;	data_table[100].edges[6] = 0.00;	data_table[100].edges[7] = 0.00;	data_table[100].edges[8] = 0.00;	data_table[100].edges[9] = 0.00;	data_table[100].edges[10] = 0.00;	data_table[100].edges[11] = 0.00;
				data_table[101].edges[0] = 0.00;	data_table[101].edges[1] = 0.00;	data_table[101].edges[2] = 0.00;	data_table[101].edges[3] = 0.00;	data_table[101].edges[4] = 0.00;	data_table[101].edges[5] = 0.00;	data_table[101].edges[6] = 0.00;	data_table[101].edges[7] = 0.00;	data_table[101].edges[8] = 0.00;	data_table[101].edges[9] = 0.00;	data_table[101].edges[10] = 0.00;	data_table[101].edges[11] = 0.00;
				data_table[102].edges[0] = 0.00;	data_table[102].edges[1] = 0.00;	data_table[102].edges[2] = 0.00;	data_table[102].edges[3] = 0.00;	data_table[102].edges[4] = 0.00;	data_table[102].edges[5] = 0.00;	data_table[102].edges[6] = 0.00;	data_table[102].edges[7] = 0.00;	data_table[102].edges[8] = 0.00;	data_table[102].edges[9] = 0.00;	data_table[102].edges[10] = 0.00;	data_table[102].edges[11] = 0.00;
			}

			// 1: Doyle and Turner parameterization - 4 Gaussians - [0, 4]
			void Load_feg_Doyle_neutral_0_4()
			{
				data_table[0].feg[0].cl[0] = 0.0000000000;		data_table[0].feg[0].cnl[0] = 0.0000000000;	data_table[0].feg[0].cl[1] = 0.0000000000;		data_table[0].feg[0].cnl[1] = 0.0000000000;	data_table[0].feg[0].cl[2] = 0.0000000000;		data_table[0].feg[0].cnl[2] = 0.0000000000;	data_table[0].feg[0].cl[3] = 0.0000000000;		data_table[0].feg[0].cnl[3] = 0.0000000000;	data_table[0].feg[0].cl[4] = 0.0000000000;		data_table[0].feg[0].cnl[4] = 0.0000000000;	data_table[0].feg[0].cl[5] = 0.0000000000;		data_table[0].feg[0].cnl[5] = 0.0000000000;
				data_table[1].feg[0].cl[0] = 0.0906000000;		data_table[1].feg[0].cnl[0] = 18.1834000000;	data_table[1].feg[0].cl[1] = 0.1814000000;		data_table[1].feg[0].cnl[1] = 6.2109000000;	data_table[1].feg[0].cl[2] = 0.1095000000;		data_table[1].feg[0].cnl[2] = 1.8026000000;	data_table[1].feg[0].cl[3] = 0.0362000000;		data_table[1].feg[0].cnl[3] = 0.2844000000;	data_table[1].feg[0].cl[4] = 0.0000000000;		data_table[1].feg[0].cnl[4] = 0.0000000000;	data_table[1].feg[0].cl[5] = 0.0000000000;		data_table[1].feg[0].cnl[5] = 0.0000000000;
				data_table[2].feg[0].cl[0] = 1.6108000000;		data_table[2].feg[0].cnl[0] = 107.6384000000;	data_table[2].feg[0].cl[1] = 1.2460000000;		data_table[2].feg[0].cnl[1] = 30.4795000000;	data_table[2].feg[0].cl[2] = 0.3257000000;		data_table[2].feg[0].cnl[2] = 4.5331000000;	data_table[2].feg[0].cl[3] = 0.0986000000;		data_table[2].feg[0].cnl[3] = 0.4951000000;	data_table[2].feg[0].cl[4] = 0.0000000000;		data_table[2].feg[0].cnl[4] = 0.0000000000;	data_table[2].feg[0].cl[5] = 0.0000000000;		data_table[2].feg[0].cnl[5] = 0.0000000000;
				data_table[3].feg[0].cl[0] = 1.2498000000;		data_table[3].feg[0].cnl[0] = 60.8042000000;	data_table[3].feg[0].cl[1] = 1.3335000000;		data_table[3].feg[0].cnl[1] = 18.5914000000;	data_table[3].feg[0].cl[2] = 0.3603000000;		data_table[3].feg[0].cnl[2] = 3.6534000000;	data_table[3].feg[0].cl[3] = 0.1055000000;		data_table[3].feg[0].cnl[3] = 0.4157000000;	data_table[3].feg[0].cl[4] = 0.0000000000;		data_table[3].feg[0].cnl[4] = 0.0000000000;	data_table[3].feg[0].cl[5] = 0.0000000000;		data_table[3].feg[0].cnl[5] = 0.0000000000;
				data_table[4].feg[0].cl[0] = 0.9446000000;		data_table[4].feg[0].cnl[0] = 46.4438000000;	data_table[4].feg[0].cl[1] = 1.3120000000;		data_table[4].feg[0].cnl[1] = 14.1778000000;	data_table[4].feg[0].cl[2] = 0.4188000000;		data_table[4].feg[0].cnl[2] = 3.2228000000;	data_table[4].feg[0].cl[3] = 0.1159000000;		data_table[4].feg[0].cnl[3] = 0.3767000000;	data_table[4].feg[0].cl[4] = 0.0000000000;		data_table[4].feg[0].cnl[4] = 0.0000000000;	data_table[4].feg[0].cl[5] = 0.0000000000;		data_table[4].feg[0].cnl[5] = 0.0000000000;
				data_table[5].feg[0].cl[0] = 0.7307000000;		data_table[5].feg[0].cnl[0] = 36.9951000000;	data_table[5].feg[0].cl[1] = 1.1951000000;		data_table[5].feg[0].cnl[1] = 11.2966000000;	data_table[5].feg[0].cl[2] = 0.4563000000;		data_table[5].feg[0].cnl[2] = 2.8139000000;	data_table[5].feg[0].cl[3] = 0.1247000000;		data_table[5].feg[0].cnl[3] = 0.3456000000;	data_table[5].feg[0].cl[4] = 0.0000000000;		data_table[5].feg[0].cnl[4] = 0.0000000000;	data_table[5].feg[0].cl[5] = 0.0000000000;		data_table[5].feg[0].cnl[5] = 0.0000000000;
				data_table[6].feg[0].cl[0] = 0.5717000000;		data_table[6].feg[0].cnl[0] = 28.8465000000;	data_table[6].feg[0].cl[1] = 1.0425000000;		data_table[6].feg[0].cnl[1] = 9.0542000000;	data_table[6].feg[0].cl[2] = 0.4647000000;		data_table[6].feg[0].cnl[2] = 2.4213000000;	data_table[6].feg[0].cl[3] = 0.1311000000;		data_table[6].feg[0].cnl[3] = 0.3167000000;	data_table[6].feg[0].cl[4] = 0.0000000000;		data_table[6].feg[0].cnl[4] = 0.0000000000;	data_table[6].feg[0].cl[5] = 0.0000000000;		data_table[6].feg[0].cnl[5] = 0.0000000000;
				data_table[7].feg[0].cl[0] = 0.4540000000;		data_table[7].feg[0].cnl[0] = 23.7803000000;	data_table[7].feg[0].cl[1] = 0.9173000000;		data_table[7].feg[0].cnl[1] = 7.6220000000;	data_table[7].feg[0].cl[2] = 0.4719000000;		data_table[7].feg[0].cnl[2] = 2.1440000000;	data_table[7].feg[0].cl[3] = 0.1384000000;		data_table[7].feg[0].cnl[3] = 0.2959000000;	data_table[7].feg[0].cl[4] = 0.0000000000;		data_table[7].feg[0].cnl[4] = 0.0000000000;	data_table[7].feg[0].cl[5] = 0.0000000000;		data_table[7].feg[0].cnl[5] = 0.0000000000;
				data_table[8].feg[0].cl[0] = 0.3686000000;		data_table[8].feg[0].cnl[0] = 20.2390000000;	data_table[8].feg[0].cl[1] = 0.8109000000;		data_table[8].feg[0].cnl[1] = 6.6093000000;	data_table[8].feg[0].cl[2] = 0.4751000000;		data_table[8].feg[0].cnl[2] = 1.9310000000;	data_table[8].feg[0].cl[3] = 0.1459000000;		data_table[8].feg[0].cnl[3] = 0.2793000000;	data_table[8].feg[0].cl[4] = 0.0000000000;		data_table[8].feg[0].cnl[4] = 0.0000000000;	data_table[8].feg[0].cl[5] = 0.0000000000;		data_table[8].feg[0].cnl[5] = 0.0000000000;
				data_table[9].feg[0].cl[0] = 0.3025000000;		data_table[9].feg[0].cnl[0] = 17.6390000000;	data_table[9].feg[0].cl[1] = 0.7202000000;		data_table[9].feg[0].cnl[1] = 5.8604000000;	data_table[9].feg[0].cl[2] = 0.4751000000;		data_table[9].feg[0].cnl[2] = 1.7623000000;	data_table[9].feg[0].cl[3] = 0.1534000000;		data_table[9].feg[0].cnl[3] = 0.2656000000;	data_table[9].feg[0].cl[4] = 0.0000000000;		data_table[9].feg[0].cnl[4] = 0.0000000000;	data_table[9].feg[0].cl[5] = 0.0000000000;		data_table[9].feg[0].cnl[5] = 0.0000000000;
				data_table[10].feg[0].cl[0] = 2.2406000000;	data_table[10].feg[0].cnl[0] = 108.0039000000;	data_table[10].feg[0].cl[1] = 1.3326000000;	data_table[10].feg[0].cnl[1] = 24.5047000000;	data_table[10].feg[0].cl[2] = 0.9070000000;	data_table[10].feg[0].cnl[2] = 3.3914000000;	data_table[10].feg[0].cl[3] = 0.2863000000;	data_table[10].feg[0].cnl[3] = 0.4346000000;	data_table[10].feg[0].cl[4] = 0.0000000000;	data_table[10].feg[0].cnl[4] = 0.0000000000;	data_table[10].feg[0].cl[5] = 0.0000000000;	data_table[10].feg[0].cnl[5] = 0.0000000000;
				data_table[11].feg[0].cl[0] = 2.2692000000;	data_table[11].feg[0].cnl[0] = 73.6704000000;	data_table[11].feg[0].cl[1] = 1.8025000000;	data_table[11].feg[0].cnl[1] = 20.1749000000;	data_table[11].feg[0].cl[2] = 0.8394000000;	data_table[11].feg[0].cnl[2] = 3.0131000000;	data_table[11].feg[0].cl[3] = 0.2892000000;	data_table[11].feg[0].cnl[3] = 0.4046000000;	data_table[11].feg[0].cl[4] = 0.0000000000;	data_table[11].feg[0].cnl[4] = 0.0000000000;	data_table[11].feg[0].cl[5] = 0.0000000000;	data_table[11].feg[0].cnl[5] = 0.0000000000;
				data_table[12].feg[0].cl[0] = 2.2756000000;	data_table[12].feg[0].cnl[0] = 72.3220000000;	data_table[12].feg[0].cl[1] = 2.4280000000;	data_table[12].feg[0].cnl[1] = 19.7729000000;	data_table[12].feg[0].cl[2] = 0.8578000000;	data_table[12].feg[0].cnl[2] = 3.0799000000;	data_table[12].feg[0].cl[3] = 0.3266000000;	data_table[12].feg[0].cnl[3] = 0.4076000000;	data_table[12].feg[0].cl[4] = 0.0000000000;	data_table[12].feg[0].cnl[4] = 0.0000000000;	data_table[12].feg[0].cl[5] = 0.0000000000;	data_table[12].feg[0].cnl[5] = 0.0000000000;
				data_table[13].feg[0].cl[0] = 2.1293000000;	data_table[13].feg[0].cnl[0] = 57.7748000000;	data_table[13].feg[0].cl[1] = 2.5333000000;	data_table[13].feg[0].cnl[1] = 16.4756000000;	data_table[13].feg[0].cl[2] = 0.8349000000;	data_table[13].feg[0].cnl[2] = 2.6796000000;	data_table[13].feg[0].cl[3] = 0.3216000000;	data_table[13].feg[0].cnl[3] = 0.3860000000;	data_table[13].feg[0].cl[4] = 0.0000000000;	data_table[13].feg[0].cnl[4] = 0.0000000000;	data_table[13].feg[0].cl[5] = 0.0000000000;	data_table[13].feg[0].cnl[5] = 0.0000000000;
				data_table[14].feg[0].cl[0] = 1.8882000000;	data_table[14].feg[0].cnl[0] = 44.8756000000;	data_table[14].feg[0].cl[1] = 2.4685000000;	data_table[14].feg[0].cnl[1] = 13.5383000000;	data_table[14].feg[0].cl[2] = 0.8046000000;	data_table[14].feg[0].cnl[2] = 2.6424000000;	data_table[14].feg[0].cl[3] = 0.3204000000;	data_table[14].feg[0].cnl[3] = 0.3603000000;	data_table[14].feg[0].cl[4] = 0.0000000000;	data_table[14].feg[0].cnl[4] = 0.0000000000;	data_table[14].feg[0].cl[5] = 0.0000000000;	data_table[14].feg[0].cnl[5] = 0.0000000000;
				data_table[15].feg[0].cl[0] = 1.6591000000;	data_table[15].feg[0].cnl[0] = 36.6500000000;	data_table[15].feg[0].cl[1] = 2.3863000000;	data_table[15].feg[0].cnl[1] = 11.4881000000;	data_table[15].feg[0].cl[2] = 0.7899000000;	data_table[15].feg[0].cnl[2] = 2.4686000000;	data_table[15].feg[0].cl[3] = 0.3208000000;	data_table[15].feg[0].cnl[3] = 0.3403000000;	data_table[15].feg[0].cl[4] = 0.0000000000;	data_table[15].feg[0].cnl[4] = 0.0000000000;	data_table[15].feg[0].cl[5] = 0.0000000000;	data_table[15].feg[0].cnl[5] = 0.0000000000;
				data_table[16].feg[0].cl[0] = 1.4524000000;	data_table[16].feg[0].cnl[0] = 30.9352000000;	data_table[16].feg[0].cl[1] = 2.2926000000;	data_table[16].feg[0].cnl[1] = 9.9798000000;	data_table[16].feg[0].cl[2] = 0.7874000000;	data_table[16].feg[0].cnl[2] = 2.3336000000;	data_table[16].feg[0].cl[3] = 0.3217000000;	data_table[16].feg[0].cnl[3] = 0.3228000000;	data_table[16].feg[0].cl[4] = 0.0000000000;	data_table[16].feg[0].cnl[4] = 0.0000000000;	data_table[16].feg[0].cl[5] = 0.0000000000;	data_table[16].feg[0].cnl[5] = 0.0000000000;
				data_table[17].feg[0].cl[0] = 1.2736000000;	data_table[17].feg[0].cnl[0] = 26.6823000000;	data_table[17].feg[0].cl[1] = 2.1894000000;	data_table[17].feg[0].cnl[1] = 8.8130000000;	data_table[17].feg[0].cl[2] = 0.7927000000;	data_table[17].feg[0].cnl[2] = 2.2186000000;	data_table[17].feg[0].cl[3] = 0.3225000000;	data_table[17].feg[0].cnl[3] = 0.3071000000;	data_table[17].feg[0].cl[4] = 0.0000000000;	data_table[17].feg[0].cnl[4] = 0.0000000000;	data_table[17].feg[0].cl[5] = 0.0000000000;	data_table[17].feg[0].cnl[5] = 0.0000000000;
				data_table[18].feg[0].cl[0] = 3.9507000000;	data_table[18].feg[0].cnl[0] = 137.0748000000;	data_table[18].feg[0].cl[1] = 2.5452000000;	data_table[18].feg[0].cnl[1] = 22.4017000000;	data_table[18].feg[0].cl[2] = 1.9795000000;	data_table[18].feg[0].cnl[2] = 4.5319000000;	data_table[18].feg[0].cl[3] = 0.4017000000;	data_table[18].feg[0].cnl[3] = 0.4340000000;	data_table[18].feg[0].cl[4] = 0.0000000000;	data_table[18].feg[0].cnl[4] = 0.0000000000;	data_table[18].feg[0].cl[5] = 0.0000000000;	data_table[18].feg[0].cnl[5] = 0.0000000000;
				data_table[19].feg[0].cl[0] = 4.4696000000;	data_table[19].feg[0].cnl[0] = 99.5228000000;	data_table[19].feg[0].cl[1] = 2.9703000000;	data_table[19].feg[0].cnl[1] = 22.6958000000;	data_table[19].feg[0].cl[2] = 1.9696000000;	data_table[19].feg[0].cnl[2] = 4.1954000000;	data_table[19].feg[0].cl[3] = 0.4818000000;	data_table[19].feg[0].cnl[3] = 0.4165000000;	data_table[19].feg[0].cl[4] = 0.0000000000;	data_table[19].feg[0].cnl[4] = 0.0000000000;	data_table[19].feg[0].cl[5] = 0.0000000000;	data_table[19].feg[0].cnl[5] = 0.0000000000;
				data_table[20].feg[0].cl[0] = 3.9659000000;	data_table[20].feg[0].cnl[0] = 38.9597000000;	data_table[20].feg[0].cl[1] = 2.9169000000;	data_table[20].feg[0].cnl[1] = 20.6061000000;	data_table[20].feg[0].cl[2] = 1.9254000000;	data_table[20].feg[0].cnl[2] = 3.8557000000;	data_table[20].feg[0].cl[3] = 0.4802000000;	data_table[20].feg[0].cnl[3] = 0.3988000000;	data_table[20].feg[0].cl[4] = 0.0000000000;	data_table[20].feg[0].cnl[4] = 0.0000000000;	data_table[20].feg[0].cl[5] = 0.0000000000;	data_table[20].feg[0].cnl[5] = 0.0000000000;
				data_table[21].feg[0].cl[0] = 3.5653000000;	data_table[21].feg[0].cnl[0] = 81.9821000000;	data_table[21].feg[0].cl[1] = 2.8181000000;	data_table[21].feg[0].cnl[1] = 19.0486000000;	data_table[21].feg[0].cl[2] = 1.8930000000;	data_table[21].feg[0].cnl[2] = 3.5904000000;	data_table[21].feg[0].cl[3] = 0.4825000000;	data_table[21].feg[0].cnl[3] = 0.3855000000;	data_table[21].feg[0].cl[4] = 0.0000000000;	data_table[21].feg[0].cnl[4] = 0.0000000000;	data_table[21].feg[0].cl[5] = 0.0000000000;	data_table[21].feg[0].cnl[5] = 0.0000000000;
				data_table[22].feg[0].cl[0] = 3.2449000000;	data_table[22].feg[0].cnl[0] = 76.3789000000;	data_table[22].feg[0].cl[1] = 2.6978000000;	data_table[22].feg[0].cnl[1] = 17.7262000000;	data_table[22].feg[0].cl[2] = 1.8597000000;	data_table[22].feg[0].cnl[2] = 3.3632000000;	data_table[22].feg[0].cl[3] = 0.4864000000;	data_table[22].feg[0].cnl[3] = 0.3743000000;	data_table[22].feg[0].cl[4] = 0.0000000000;	data_table[22].feg[0].cnl[4] = 0.0000000000;	data_table[22].feg[0].cl[5] = 0.0000000000;	data_table[22].feg[0].cnl[5] = 0.0000000000;
				data_table[23].feg[0].cl[0] = 2.3066000000;	data_table[23].feg[0].cnl[0] = 78.4051000000;	data_table[23].feg[0].cl[1] = 2.3339000000;	data_table[23].feg[0].cnl[1] = 15.7851000000;	data_table[23].feg[0].cl[2] = 1.8226000000;	data_table[23].feg[0].cnl[2] = 3.1566000000;	data_table[23].feg[0].cl[3] = 0.4901000000;	data_table[23].feg[0].cnl[3] = 0.3636000000;	data_table[23].feg[0].cl[4] = 0.0000000000;	data_table[23].feg[0].cnl[4] = 0.0000000000;	data_table[23].feg[0].cl[5] = 0.0000000000;	data_table[23].feg[0].cnl[5] = 0.0000000000;
				data_table[24].feg[0].cl[0] = 2.7467000000;	data_table[24].feg[0].cnl[0] = 67.7862000000;	data_table[24].feg[0].cl[1] = 2.4556000000;	data_table[24].feg[0].cnl[1] = 15.6743000000;	data_table[24].feg[0].cl[2] = 1.7923000000;	data_table[24].feg[0].cnl[2] = 2.9998000000;	data_table[24].feg[0].cl[3] = 0.4984000000;	data_table[24].feg[0].cnl[3] = 0.3569000000;	data_table[24].feg[0].cl[4] = 0.0000000000;	data_table[24].feg[0].cnl[4] = 0.0000000000;	data_table[24].feg[0].cl[5] = 0.0000000000;	data_table[24].feg[0].cnl[5] = 0.0000000000;
				data_table[25].feg[0].cl[0] = 2.5440000000;	data_table[25].feg[0].cnl[0] = 64.4244000000;	data_table[25].feg[0].cl[1] = 2.3434000000;	data_table[25].feg[0].cnl[1] = 14.5306000000;	data_table[25].feg[0].cl[2] = 1.7588000000;	data_table[25].feg[0].cnl[2] = 2.8539000000;	data_table[25].feg[0].cl[3] = 0.5062000000;	data_table[25].feg[0].cnl[3] = 0.3502000000;	data_table[25].feg[0].cl[4] = 0.0000000000;	data_table[25].feg[0].cnl[4] = 0.0000000000;	data_table[25].feg[0].cl[5] = 0.0000000000;	data_table[25].feg[0].cnl[5] = 0.0000000000;
				data_table[26].feg[0].cl[0] = 2.3668000000;	data_table[26].feg[0].cnl[0] = 61.4306000000;	data_table[26].feg[0].cl[1] = 2.2361000000;	data_table[26].feg[0].cnl[1] = 14.1793000000;	data_table[26].feg[0].cl[2] = 1.7243000000;	data_table[26].feg[0].cnl[2] = 2.7247000000;	data_table[26].feg[0].cl[3] = 0.5148000000;	data_table[26].feg[0].cnl[3] = 0.3442000000;	data_table[26].feg[0].cl[4] = 0.0000000000;	data_table[26].feg[0].cnl[4] = 0.0000000000;	data_table[26].feg[0].cl[5] = 0.0000000000;	data_table[26].feg[0].cnl[5] = 0.0000000000;
				data_table[27].feg[0].cl[0] = 2.2104000000;	data_table[27].feg[0].cnl[0] = 58.7267000000;	data_table[27].feg[0].cl[1] = 2.1342000000;	data_table[27].feg[0].cnl[1] = 13.5530000000;	data_table[27].feg[0].cl[2] = 1.6891000000;	data_table[27].feg[0].cnl[2] = 2.6094000000;	data_table[27].feg[0].cl[3] = 0.5238000000;	data_table[27].feg[0].cnl[3] = 0.3388000000;	data_table[27].feg[0].cl[4] = 0.0000000000;	data_table[27].feg[0].cnl[4] = 0.0000000000;	data_table[27].feg[0].cl[5] = 0.0000000000;	data_table[27].feg[0].cnl[5] = 0.0000000000;
				data_table[28].feg[0].cl[0] = 1.5792000000;	data_table[28].feg[0].cnl[0] = 62.9403000000;	data_table[28].feg[0].cl[1] = 1.8197000000;	data_table[28].feg[0].cnl[1] = 12.4527000000;	data_table[28].feg[0].cl[2] = 1.6576000000;	data_table[28].feg[0].cnl[2] = 2.5042000000;	data_table[28].feg[0].cl[3] = 0.5323000000;	data_table[28].feg[0].cnl[3] = 0.3331000000;	data_table[28].feg[0].cl[4] = 0.0000000000;	data_table[28].feg[0].cnl[4] = 0.0000000000;	data_table[28].feg[0].cl[5] = 0.0000000000;	data_table[28].feg[0].cnl[5] = 0.0000000000;
				data_table[29].feg[0].cl[0] = 1.9418000000;	data_table[29].feg[0].cnl[0] = 54.1621000000;	data_table[29].feg[0].cl[1] = 1.9501000000;	data_table[29].feg[0].cnl[1] = 12.5177000000;	data_table[29].feg[0].cl[2] = 1.6192000000;	data_table[29].feg[0].cnl[2] = 2.4164000000;	data_table[29].feg[0].cl[3] = 0.5434000000;	data_table[29].feg[0].cnl[3] = 0.3295000000;	data_table[29].feg[0].cl[4] = 0.0000000000;	data_table[29].feg[0].cnl[4] = 0.0000000000;	data_table[29].feg[0].cl[5] = 0.0000000000;	data_table[29].feg[0].cnl[5] = 0.0000000000;
				data_table[30].feg[0].cl[0] = 2.3205000000;	data_table[30].feg[0].cnl[0] = 65.6019000000;	data_table[30].feg[0].cl[1] = 2.4955000000;	data_table[30].feg[0].cnl[1] = 15.4577000000;	data_table[30].feg[0].cl[2] = 1.6879000000;	data_table[30].feg[0].cnl[2] = 2.5806000000;	data_table[30].feg[0].cl[3] = 0.5992000000;	data_table[30].feg[0].cnl[3] = 0.3510000000;	data_table[30].feg[0].cl[4] = 0.0000000000;	data_table[30].feg[0].cnl[4] = 0.0000000000;	data_table[30].feg[0].cl[5] = 0.0000000000;	data_table[30].feg[0].cnl[5] = 0.0000000000;
				data_table[31].feg[0].cl[0] = 2.4467000000;	data_table[31].feg[0].cnl[0] = 55.8930000000;	data_table[31].feg[0].cl[1] = 2.7015000000;	data_table[31].feg[0].cnl[1] = 14.3930000000;	data_table[31].feg[0].cl[2] = 1.6157000000;	data_table[31].feg[0].cnl[2] = 2.4461000000;	data_table[31].feg[0].cl[3] = 0.6009000000;	data_table[31].feg[0].cnl[3] = 0.3415000000;	data_table[31].feg[0].cl[4] = 0.0000000000;	data_table[31].feg[0].cnl[4] = 0.0000000000;	data_table[31].feg[0].cl[5] = 0.0000000000;	data_table[31].feg[0].cnl[5] = 0.0000000000;
				data_table[32].feg[0].cl[0] = 2.3989000000;	data_table[32].feg[0].cnl[0] = 45.7179000000;	data_table[32].feg[0].cl[1] = 2.7898000000;	data_table[32].feg[0].cnl[1] = 32.8166000000;	data_table[32].feg[0].cl[2] = 1.5288000000;	data_table[32].feg[0].cnl[2] = 2.2799000000;	data_table[32].feg[0].cl[3] = 0.5936000000;	data_table[32].feg[0].cnl[3] = 0.3277000000;	data_table[32].feg[0].cl[4] = 0.0000000000;	data_table[32].feg[0].cnl[4] = 0.0000000000;	data_table[32].feg[0].cl[5] = 0.0000000000;	data_table[32].feg[0].cnl[5] = 0.0000000000;
				data_table[33].feg[0].cl[0] = 2.2980000000;	data_table[33].feg[0].cnl[0] = 38.8296000000;	data_table[33].feg[0].cl[1] = 2.8541000000;	data_table[33].feg[0].cnl[1] = 11.5359000000;	data_table[33].feg[0].cl[2] = 1.4555000000;	data_table[33].feg[0].cnl[2] = 2.1463000000;	data_table[33].feg[0].cl[3] = 0.5895000000;	data_table[33].feg[0].cnl[3] = 0.3163000000;	data_table[33].feg[0].cl[4] = 0.0000000000;	data_table[33].feg[0].cnl[4] = 0.0000000000;	data_table[33].feg[0].cl[5] = 0.0000000000;	data_table[33].feg[0].cnl[5] = 0.0000000000;
				data_table[34].feg[0].cl[0] = 2.1659000000;	data_table[34].feg[0].cnl[0] = 33.8987000000;	data_table[34].feg[0].cl[1] = 2.9037000000;	data_table[34].feg[0].cnl[1] = 10.4996000000;	data_table[34].feg[0].cl[2] = 1.3951000000;	data_table[34].feg[0].cnl[2] = 2.0413000000;	data_table[34].feg[0].cl[3] = 0.5886000000;	data_table[34].feg[0].cnl[3] = 0.3070000000;	data_table[34].feg[0].cl[4] = 0.0000000000;	data_table[34].feg[0].cnl[4] = 0.0000000000;	data_table[34].feg[0].cl[5] = 0.0000000000;	data_table[34].feg[0].cnl[5] = 0.0000000000;
				data_table[35].feg[0].cl[0] = 2.0338000000;	data_table[35].feg[0].cnl[0] = 29.9992000000;	data_table[35].feg[0].cl[1] = 2.9271000000;	data_table[35].feg[0].cnl[1] = 9.5977000000;	data_table[35].feg[0].cl[2] = 1.3425000000;	data_table[35].feg[0].cnl[2] = 1.9520000000;	data_table[35].feg[0].cl[3] = 0.5888000000;	data_table[35].feg[0].cnl[3] = 0.2986000000;	data_table[35].feg[0].cl[4] = 0.0000000000;	data_table[35].feg[0].cnl[4] = 0.0000000000;	data_table[35].feg[0].cl[5] = 0.0000000000;	data_table[35].feg[0].cnl[5] = 0.0000000000;
				data_table[36].feg[0].cl[0] = 4.7760000000;	data_table[36].feg[0].cnl[0] = 140.7821000000;	data_table[36].feg[0].cl[1] = 3.8588000000;	data_table[36].feg[0].cnl[1] = 18.9910000000;	data_table[36].feg[0].cl[2] = 2.2339000000;	data_table[36].feg[0].cnl[2] = 3.7010000000;	data_table[36].feg[0].cl[3] = 0.8683000000;	data_table[36].feg[0].cnl[3] = 0.4194000000;	data_table[36].feg[0].cl[4] = 0.0000000000;	data_table[36].feg[0].cnl[4] = 0.0000000000;	data_table[36].feg[0].cl[5] = 0.0000000000;	data_table[36].feg[0].cnl[5] = 0.0000000000;
				data_table[37].feg[0].cl[0] = 5.8478000000;	data_table[37].feg[0].cnl[0] = 104.9721000000;	data_table[37].feg[0].cl[1] = 4.0026000000;	data_table[37].feg[0].cnl[1] = 19.3666000000;	data_table[37].feg[0].cl[2] = 2.3420000000;	data_table[37].feg[0].cnl[2] = 3.7368000000;	data_table[37].feg[0].cl[3] = 0.8795000000;	data_table[37].feg[0].cnl[3] = 0.4142000000;	data_table[37].feg[0].cl[4] = 0.0000000000;	data_table[37].feg[0].cnl[4] = 0.0000000000;	data_table[37].feg[0].cl[5] = 0.0000000000;	data_table[37].feg[0].cnl[5] = 0.0000000000;
				data_table[38].feg[0].cl[0] = 0.0000000000;	data_table[38].feg[0].cnl[0] = 0.0000000000;	data_table[38].feg[0].cl[1] = 0.0000000000;	data_table[38].feg[0].cnl[1] = 0.0000000000;	data_table[38].feg[0].cl[2] = 0.0000000000;	data_table[38].feg[0].cnl[2] = 0.0000000000;	data_table[38].feg[0].cl[3] = 0.0000000000;	data_table[38].feg[0].cnl[3] = 0.0000000000;	data_table[38].feg[0].cl[4] = 0.0000000000;	data_table[38].feg[0].cnl[4] = 0.0000000000;	data_table[38].feg[0].cl[5] = 0.0000000000;	data_table[38].feg[0].cnl[5] = 0.0000000000;
				data_table[39].feg[0].cl[0] = 0.0000000000;	data_table[39].feg[0].cnl[0] = 0.0000000000;	data_table[39].feg[0].cl[1] = 0.0000000000;	data_table[39].feg[0].cnl[1] = 0.0000000000;	data_table[39].feg[0].cl[2] = 0.0000000000;	data_table[39].feg[0].cnl[2] = 0.0000000000;	data_table[39].feg[0].cl[3] = 0.0000000000;	data_table[39].feg[0].cnl[3] = 0.0000000000;	data_table[39].feg[0].cl[4] = 0.0000000000;	data_table[39].feg[0].cnl[4] = 0.0000000000;	data_table[39].feg[0].cl[5] = 0.0000000000;	data_table[39].feg[0].cnl[5] = 0.0000000000;
				data_table[40].feg[0].cl[0] = 0.0000000000;	data_table[40].feg[0].cnl[0] = 0.0000000000;	data_table[40].feg[0].cl[1] = 0.0000000000;	data_table[40].feg[0].cnl[1] = 0.0000000000;	data_table[40].feg[0].cl[2] = 0.0000000000;	data_table[40].feg[0].cnl[2] = 0.0000000000;	data_table[40].feg[0].cl[3] = 0.0000000000;	data_table[40].feg[0].cnl[3] = 0.0000000000;	data_table[40].feg[0].cl[4] = 0.0000000000;	data_table[40].feg[0].cnl[4] = 0.0000000000;	data_table[40].feg[0].cl[5] = 0.0000000000;	data_table[40].feg[0].cnl[5] = 0.0000000000;
				data_table[41].feg[0].cl[0] = 3.1199000000;	data_table[41].feg[0].cnl[0] = 72.4642000000;	data_table[41].feg[0].cl[1] = 3.9061000000;	data_table[41].feg[0].cnl[1] = 14.6424000000;	data_table[41].feg[0].cl[2] = 2.3615000000;	data_table[41].feg[0].cnl[2] = 3.2370000000;	data_table[41].feg[0].cl[3] = 0.8504000000;	data_table[41].feg[0].cnl[3] = 0.3662000000;	data_table[41].feg[0].cl[4] = 0.0000000000;	data_table[41].feg[0].cnl[4] = 0.0000000000;	data_table[41].feg[0].cl[5] = 0.0000000000;	data_table[41].feg[0].cnl[5] = 0.0000000000;
				data_table[42].feg[0].cl[0] = 0.0000000000;	data_table[42].feg[0].cnl[0] = 0.0000000000;	data_table[42].feg[0].cl[1] = 0.0000000000;	data_table[42].feg[0].cnl[1] = 0.0000000000;	data_table[42].feg[0].cl[2] = 0.0000000000;	data_table[42].feg[0].cnl[2] = 0.0000000000;	data_table[42].feg[0].cl[3] = 0.0000000000;	data_table[42].feg[0].cnl[3] = 0.0000000000;	data_table[42].feg[0].cl[4] = 0.0000000000;	data_table[42].feg[0].cnl[4] = 0.0000000000;	data_table[42].feg[0].cl[5] = 0.0000000000;	data_table[42].feg[0].cnl[5] = 0.0000000000;
				data_table[43].feg[0].cl[0] = 0.0000000000;	data_table[43].feg[0].cnl[0] = 0.0000000000;	data_table[43].feg[0].cl[1] = 0.0000000000;	data_table[43].feg[0].cnl[1] = 0.0000000000;	data_table[43].feg[0].cl[2] = 0.0000000000;	data_table[43].feg[0].cnl[2] = 0.0000000000;	data_table[43].feg[0].cl[3] = 0.0000000000;	data_table[43].feg[0].cnl[3] = 0.0000000000;	data_table[43].feg[0].cl[4] = 0.0000000000;	data_table[43].feg[0].cnl[4] = 0.0000000000;	data_table[43].feg[0].cl[5] = 0.0000000000;	data_table[43].feg[0].cnl[5] = 0.0000000000;
				data_table[44].feg[0].cl[0] = 0.0000000000;	data_table[44].feg[0].cnl[0] = 0.0000000000;	data_table[44].feg[0].cl[1] = 0.0000000000;	data_table[44].feg[0].cnl[1] = 0.0000000000;	data_table[44].feg[0].cl[2] = 0.0000000000;	data_table[44].feg[0].cnl[2] = 0.0000000000;	data_table[44].feg[0].cl[3] = 0.0000000000;	data_table[44].feg[0].cnl[3] = 0.0000000000;	data_table[44].feg[0].cl[4] = 0.0000000000;	data_table[44].feg[0].cnl[4] = 0.0000000000;	data_table[44].feg[0].cl[5] = 0.0000000000;	data_table[44].feg[0].cnl[5] = 0.0000000000;
				data_table[45].feg[0].cl[0] = 0.0000000000;	data_table[45].feg[0].cnl[0] = 0.0000000000;	data_table[45].feg[0].cl[1] = 0.0000000000;	data_table[45].feg[0].cnl[1] = 0.0000000000;	data_table[45].feg[0].cl[2] = 0.0000000000;	data_table[45].feg[0].cnl[2] = 0.0000000000;	data_table[45].feg[0].cl[3] = 0.0000000000;	data_table[45].feg[0].cnl[3] = 0.0000000000;	data_table[45].feg[0].cl[4] = 0.0000000000;	data_table[45].feg[0].cnl[4] = 0.0000000000;	data_table[45].feg[0].cl[5] = 0.0000000000;	data_table[45].feg[0].cnl[5] = 0.0000000000;
				data_table[46].feg[0].cl[0] = 2.0355000000;	data_table[46].feg[0].cnl[0] = 61.4970000000;	data_table[46].feg[0].cl[1] = 3.2716000000;	data_table[46].feg[0].cnl[1] = 11.3237000000;	data_table[46].feg[0].cl[2] = 2.5105000000;	data_table[46].feg[0].cnl[2] = 2.8456000000;	data_table[46].feg[0].cl[3] = 0.8372000000;	data_table[46].feg[0].cnl[3] = 0.3271000000;	data_table[46].feg[0].cl[4] = 0.0000000000;	data_table[46].feg[0].cnl[4] = 0.0000000000;	data_table[46].feg[0].cl[5] = 0.0000000000;	data_table[46].feg[0].cnl[5] = 0.0000000000;
				data_table[47].feg[0].cl[0] = 2.5737000000;	data_table[47].feg[0].cnl[0] = 55.6752000000;	data_table[47].feg[0].cl[1] = 3.2536000000;	data_table[47].feg[0].cnl[1] = 11.8376000000;	data_table[47].feg[0].cl[2] = 2.5468000000;	data_table[47].feg[0].cnl[2] = 2.7842000000;	data_table[47].feg[0].cl[3] = 0.8379000000;	data_table[47].feg[0].cnl[3] = 0.3217000000;	data_table[47].feg[0].cl[4] = 0.0000000000;	data_table[47].feg[0].cnl[4] = 0.0000000000;	data_table[47].feg[0].cl[5] = 0.0000000000;	data_table[47].feg[0].cnl[5] = 0.0000000000;
				data_table[48].feg[0].cl[0] = 3.1528000000;	data_table[48].feg[0].cnl[0] = 66.6492000000;	data_table[48].feg[0].cl[1] = 3.5565000000;	data_table[48].feg[0].cnl[1] = 14.4494000000;	data_table[48].feg[0].cl[2] = 2.8180000000;	data_table[48].feg[0].cnl[2] = 2.9758000000;	data_table[48].feg[0].cl[3] = 0.8842000000;	data_table[48].feg[0].cnl[3] = 0.3345000000;	data_table[48].feg[0].cl[4] = 0.0000000000;	data_table[48].feg[0].cnl[4] = 0.0000000000;	data_table[48].feg[0].cl[5] = 0.0000000000;	data_table[48].feg[0].cnl[5] = 0.0000000000;
				data_table[49].feg[0].cl[0] = 3.4495000000;	data_table[49].feg[0].cnl[0] = 59.1042000000;	data_table[49].feg[0].cl[1] = 3.7349000000;	data_table[49].feg[0].cnl[1] = 14.1787000000;	data_table[49].feg[0].cl[2] = 2.7779000000;	data_table[49].feg[0].cnl[2] = 2.8548000000;	data_table[49].feg[0].cl[3] = 0.8786000000;	data_table[49].feg[0].cnl[3] = 0.3270000000;	data_table[49].feg[0].cl[4] = 0.0000000000;	data_table[49].feg[0].cnl[4] = 0.0000000000;	data_table[49].feg[0].cl[5] = 0.0000000000;	data_table[49].feg[0].cnl[5] = 0.0000000000;
				data_table[50].feg[0].cl[0] = 3.5644000000;	data_table[50].feg[0].cnl[0] = 50.4869000000;	data_table[50].feg[0].cl[1] = 3.8437000000;	data_table[50].feg[0].cnl[1] = 13.3156000000;	data_table[50].feg[0].cl[2] = 2.6366000000;	data_table[50].feg[0].cnl[2] = 2.6909000000;	data_table[50].feg[0].cl[3] = 0.8638000000;	data_table[50].feg[0].cnl[3] = 0.3161000000;	data_table[50].feg[0].cl[4] = 0.0000000000;	data_table[50].feg[0].cnl[4] = 0.0000000000;	data_table[50].feg[0].cl[5] = 0.0000000000;	data_table[50].feg[0].cnl[5] = 0.0000000000;
				data_table[51].feg[0].cl[0] = 0.0000000000;	data_table[51].feg[0].cnl[0] = 0.0000000000;	data_table[51].feg[0].cl[1] = 0.0000000000;	data_table[51].feg[0].cnl[1] = 0.0000000000;	data_table[51].feg[0].cl[2] = 0.0000000000;	data_table[51].feg[0].cnl[2] = 0.0000000000;	data_table[51].feg[0].cl[3] = 0.0000000000;	data_table[51].feg[0].cnl[3] = 0.0000000000;	data_table[51].feg[0].cl[4] = 0.0000000000;	data_table[51].feg[0].cnl[4] = 0.0000000000;	data_table[51].feg[0].cl[5] = 0.0000000000;	data_table[51].feg[0].cnl[5] = 0.0000000000;
				data_table[52].feg[0].cl[0] = 3.4728000000;	data_table[52].feg[0].cnl[0] = 39.4411000000;	data_table[52].feg[0].cl[1] = 4.0602000000;	data_table[52].feg[0].cnl[1] = 11.8161000000;	data_table[52].feg[0].cl[2] = 2.5215000000;	data_table[52].feg[0].cnl[2] = 2.4148000000;	data_table[52].feg[0].cl[3] = 0.8398000000;	data_table[52].feg[0].cnl[3] = 0.2976000000;	data_table[52].feg[0].cl[4] = 0.0000000000;	data_table[52].feg[0].cnl[4] = 0.0000000000;	data_table[52].feg[0].cl[5] = 0.0000000000;	data_table[52].feg[0].cnl[5] = 0.0000000000;
				data_table[53].feg[0].cl[0] = 3.3656000000;	data_table[53].feg[0].cnl[0] = 35.5094000000;	data_table[53].feg[0].cl[1] = 4.1468000000;	data_table[53].feg[0].cnl[1] = 11.1170000000;	data_table[53].feg[0].cl[2] = 2.4430000000;	data_table[53].feg[0].cnl[2] = 2.2940000000;	data_table[53].feg[0].cl[3] = 0.8293000000;	data_table[53].feg[0].cnl[3] = 0.2892000000;	data_table[53].feg[0].cl[4] = 0.0000000000;	data_table[53].feg[0].cnl[4] = 0.0000000000;	data_table[53].feg[0].cl[5] = 0.0000000000;	data_table[53].feg[0].cnl[5] = 0.0000000000;
				data_table[54].feg[0].cl[0] = 6.0620000000;	data_table[54].feg[0].cnl[0] = 155.8336000000;	data_table[54].feg[0].cl[1] = 5.9861000000;	data_table[54].feg[0].cnl[1] = 19.6951000000;	data_table[54].feg[0].cl[2] = 3.3033000000;	data_table[54].feg[0].cnl[2] = 3.3354000000;	data_table[54].feg[0].cl[3] = 1.0958000000;	data_table[54].feg[0].cnl[3] = 0.3793000000;	data_table[54].feg[0].cl[4] = 0.0000000000;	data_table[54].feg[0].cnl[4] = 0.0000000000;	data_table[54].feg[0].cl[5] = 0.0000000000;	data_table[54].feg[0].cnl[5] = 0.0000000000;
				data_table[55].feg[0].cl[0] = 7.8212000000;	data_table[55].feg[0].cnl[0] = 117.6575000000;	data_table[55].feg[0].cl[1] = 6.0040000000;	data_table[55].feg[0].cnl[1] = 18.7782000000;	data_table[55].feg[0].cl[2] = 3.2803000000;	data_table[55].feg[0].cnl[2] = 3.2634000000;	data_table[55].feg[0].cl[3] = 1.1030000000;	data_table[55].feg[0].cnl[3] = 0.3760000000;	data_table[55].feg[0].cl[4] = 0.0000000000;	data_table[55].feg[0].cnl[4] = 0.0000000000;	data_table[55].feg[0].cl[5] = 0.0000000000;	data_table[55].feg[0].cnl[5] = 0.0000000000;
				data_table[56].feg[0].cl[0] = 6.2661000000;	data_table[56].feg[0].cnl[0] = 100.2983000000;	data_table[56].feg[0].cl[1] = 4.8440000000;	data_table[56].feg[0].cnl[1] = 16.0662000000;	data_table[56].feg[0].cl[2] = 3.2023000000;	data_table[56].feg[0].cnl[2] = 2.9803000000;	data_table[56].feg[0].cl[3] = 1.2009000000;	data_table[56].feg[0].cnl[3] = 0.3674000000;	data_table[56].feg[0].cl[4] = 0.0000000000;	data_table[56].feg[0].cnl[4] = 0.0000000000;	data_table[56].feg[0].cl[5] = 0.0000000000;	data_table[56].feg[0].cnl[5] = 0.0000000000;
				data_table[57].feg[0].cl[0] = 0.0000000000;	data_table[57].feg[0].cnl[0] = 0.0000000000;	data_table[57].feg[0].cl[1] = 0.0000000000;	data_table[57].feg[0].cnl[1] = 0.0000000000;	data_table[57].feg[0].cl[2] = 0.0000000000;	data_table[57].feg[0].cnl[2] = 0.0000000000;	data_table[57].feg[0].cl[3] = 0.0000000000;	data_table[57].feg[0].cnl[3] = 0.0000000000;	data_table[57].feg[0].cl[4] = 0.0000000000;	data_table[57].feg[0].cnl[4] = 0.0000000000;	data_table[57].feg[0].cl[5] = 0.0000000000;	data_table[57].feg[0].cnl[5] = 0.0000000000;
				data_table[58].feg[0].cl[0] = 0.0000000000;	data_table[58].feg[0].cnl[0] = 0.0000000000;	data_table[58].feg[0].cl[1] = 0.0000000000;	data_table[58].feg[0].cnl[1] = 0.0000000000;	data_table[58].feg[0].cl[2] = 0.0000000000;	data_table[58].feg[0].cnl[2] = 0.0000000000;	data_table[58].feg[0].cl[3] = 0.0000000000;	data_table[58].feg[0].cnl[3] = 0.0000000000;	data_table[58].feg[0].cl[4] = 0.0000000000;	data_table[58].feg[0].cnl[4] = 0.0000000000;	data_table[58].feg[0].cl[5] = 0.0000000000;	data_table[58].feg[0].cnl[5] = 0.0000000000;
				data_table[59].feg[0].cl[0] = 0.0000000000;	data_table[59].feg[0].cnl[0] = 0.0000000000;	data_table[59].feg[0].cl[1] = 0.0000000000;	data_table[59].feg[0].cnl[1] = 0.0000000000;	data_table[59].feg[0].cl[2] = 0.0000000000;	data_table[59].feg[0].cnl[2] = 0.0000000000;	data_table[59].feg[0].cl[3] = 0.0000000000;	data_table[59].feg[0].cnl[3] = 0.0000000000;	data_table[59].feg[0].cl[4] = 0.0000000000;	data_table[59].feg[0].cnl[4] = 0.0000000000;	data_table[59].feg[0].cl[5] = 0.0000000000;	data_table[59].feg[0].cnl[5] = 0.0000000000;
				data_table[60].feg[0].cl[0] = 0.0000000000;	data_table[60].feg[0].cnl[0] = 0.0000000000;	data_table[60].feg[0].cl[1] = 0.0000000000;	data_table[60].feg[0].cnl[1] = 0.0000000000;	data_table[60].feg[0].cl[2] = 0.0000000000;	data_table[60].feg[0].cnl[2] = 0.0000000000;	data_table[60].feg[0].cl[3] = 0.0000000000;	data_table[60].feg[0].cnl[3] = 0.0000000000;	data_table[60].feg[0].cl[4] = 0.0000000000;	data_table[60].feg[0].cnl[4] = 0.0000000000;	data_table[60].feg[0].cl[5] = 0.0000000000;	data_table[60].feg[0].cnl[5] = 0.0000000000;
				data_table[61].feg[0].cl[0] = 0.0000000000;	data_table[61].feg[0].cnl[0] = 0.0000000000;	data_table[61].feg[0].cl[1] = 0.0000000000;	data_table[61].feg[0].cnl[1] = 0.0000000000;	data_table[61].feg[0].cl[2] = 0.0000000000;	data_table[61].feg[0].cnl[2] = 0.0000000000;	data_table[61].feg[0].cl[3] = 0.0000000000;	data_table[61].feg[0].cnl[3] = 0.0000000000;	data_table[61].feg[0].cl[4] = 0.0000000000;	data_table[61].feg[0].cnl[4] = 0.0000000000;	data_table[61].feg[0].cl[5] = 0.0000000000;	data_table[61].feg[0].cnl[5] = 0.0000000000;
				data_table[62].feg[0].cl[0] = 0.0000000000;	data_table[62].feg[0].cnl[0] = 0.0000000000;	data_table[62].feg[0].cl[1] = 0.0000000000;	data_table[62].feg[0].cnl[1] = 0.0000000000;	data_table[62].feg[0].cl[2] = 0.0000000000;	data_table[62].feg[0].cnl[2] = 0.0000000000;	data_table[62].feg[0].cl[3] = 0.0000000000;	data_table[62].feg[0].cnl[3] = 0.0000000000;	data_table[62].feg[0].cl[4] = 0.0000000000;	data_table[62].feg[0].cnl[4] = 0.0000000000;	data_table[62].feg[0].cl[5] = 0.0000000000;	data_table[62].feg[0].cnl[5] = 0.0000000000;
				data_table[63].feg[0].cl[0] = 0.0000000000;	data_table[63].feg[0].cnl[0] = 0.0000000000;	data_table[63].feg[0].cl[1] = 0.0000000000;	data_table[63].feg[0].cnl[1] = 0.0000000000;	data_table[63].feg[0].cl[2] = 0.0000000000;	data_table[63].feg[0].cnl[2] = 0.0000000000;	data_table[63].feg[0].cl[3] = 0.0000000000;	data_table[63].feg[0].cnl[3] = 0.0000000000;	data_table[63].feg[0].cl[4] = 0.0000000000;	data_table[63].feg[0].cnl[4] = 0.0000000000;	data_table[63].feg[0].cl[5] = 0.0000000000;	data_table[63].feg[0].cnl[5] = 0.0000000000;
				data_table[64].feg[0].cl[0] = 0.0000000000;	data_table[64].feg[0].cnl[0] = 0.0000000000;	data_table[64].feg[0].cl[1] = 0.0000000000;	data_table[64].feg[0].cnl[1] = 0.0000000000;	data_table[64].feg[0].cl[2] = 0.0000000000;	data_table[64].feg[0].cnl[2] = 0.0000000000;	data_table[64].feg[0].cl[3] = 0.0000000000;	data_table[64].feg[0].cnl[3] = 0.0000000000;	data_table[64].feg[0].cl[4] = 0.0000000000;	data_table[64].feg[0].cnl[4] = 0.0000000000;	data_table[64].feg[0].cl[5] = 0.0000000000;	data_table[64].feg[0].cnl[5] = 0.0000000000;
				data_table[65].feg[0].cl[0] = 0.0000000000;	data_table[65].feg[0].cnl[0] = 0.0000000000;	data_table[65].feg[0].cl[1] = 0.0000000000;	data_table[65].feg[0].cnl[1] = 0.0000000000;	data_table[65].feg[0].cl[2] = 0.0000000000;	data_table[65].feg[0].cnl[2] = 0.0000000000;	data_table[65].feg[0].cl[3] = 0.0000000000;	data_table[65].feg[0].cnl[3] = 0.0000000000;	data_table[65].feg[0].cl[4] = 0.0000000000;	data_table[65].feg[0].cnl[4] = 0.0000000000;	data_table[65].feg[0].cl[5] = 0.0000000000;	data_table[65].feg[0].cnl[5] = 0.0000000000;
				data_table[66].feg[0].cl[0] = 0.0000000000;	data_table[66].feg[0].cnl[0] = 0.0000000000;	data_table[66].feg[0].cl[1] = 0.0000000000;	data_table[66].feg[0].cnl[1] = 0.0000000000;	data_table[66].feg[0].cl[2] = 0.0000000000;	data_table[66].feg[0].cnl[2] = 0.0000000000;	data_table[66].feg[0].cl[3] = 0.0000000000;	data_table[66].feg[0].cnl[3] = 0.0000000000;	data_table[66].feg[0].cl[4] = 0.0000000000;	data_table[66].feg[0].cnl[4] = 0.0000000000;	data_table[66].feg[0].cl[5] = 0.0000000000;	data_table[66].feg[0].cnl[5] = 0.0000000000;
				data_table[67].feg[0].cl[0] = 0.0000000000;	data_table[67].feg[0].cnl[0] = 0.0000000000;	data_table[67].feg[0].cl[1] = 0.0000000000;	data_table[67].feg[0].cnl[1] = 0.0000000000;	data_table[67].feg[0].cl[2] = 0.0000000000;	data_table[67].feg[0].cnl[2] = 0.0000000000;	data_table[67].feg[0].cl[3] = 0.0000000000;	data_table[67].feg[0].cnl[3] = 0.0000000000;	data_table[67].feg[0].cl[4] = 0.0000000000;	data_table[67].feg[0].cnl[4] = 0.0000000000;	data_table[67].feg[0].cl[5] = 0.0000000000;	data_table[67].feg[0].cnl[5] = 0.0000000000;
				data_table[68].feg[0].cl[0] = 0.0000000000;	data_table[68].feg[0].cnl[0] = 0.0000000000;	data_table[68].feg[0].cl[1] = 0.0000000000;	data_table[68].feg[0].cnl[1] = 0.0000000000;	data_table[68].feg[0].cl[2] = 0.0000000000;	data_table[68].feg[0].cnl[2] = 0.0000000000;	data_table[68].feg[0].cl[3] = 0.0000000000;	data_table[68].feg[0].cnl[3] = 0.0000000000;	data_table[68].feg[0].cl[4] = 0.0000000000;	data_table[68].feg[0].cnl[4] = 0.0000000000;	data_table[68].feg[0].cl[5] = 0.0000000000;	data_table[68].feg[0].cnl[5] = 0.0000000000;
				data_table[69].feg[0].cl[0] = 0.0000000000;	data_table[69].feg[0].cnl[0] = 0.0000000000;	data_table[69].feg[0].cl[1] = 0.0000000000;	data_table[69].feg[0].cnl[1] = 0.0000000000;	data_table[69].feg[0].cl[2] = 0.0000000000;	data_table[69].feg[0].cnl[2] = 0.0000000000;	data_table[69].feg[0].cl[3] = 0.0000000000;	data_table[69].feg[0].cnl[3] = 0.0000000000;	data_table[69].feg[0].cl[4] = 0.0000000000;	data_table[69].feg[0].cnl[4] = 0.0000000000;	data_table[69].feg[0].cl[5] = 0.0000000000;	data_table[69].feg[0].cnl[5] = 0.0000000000;
				data_table[70].feg[0].cl[0] = 0.0000000000;	data_table[70].feg[0].cnl[0] = 0.0000000000;	data_table[70].feg[0].cl[1] = 0.0000000000;	data_table[70].feg[0].cnl[1] = 0.0000000000;	data_table[70].feg[0].cl[2] = 0.0000000000;	data_table[70].feg[0].cnl[2] = 0.0000000000;	data_table[70].feg[0].cl[3] = 0.0000000000;	data_table[70].feg[0].cnl[3] = 0.0000000000;	data_table[70].feg[0].cl[4] = 0.0000000000;	data_table[70].feg[0].cnl[4] = 0.0000000000;	data_table[70].feg[0].cl[5] = 0.0000000000;	data_table[70].feg[0].cnl[5] = 0.0000000000;
				data_table[71].feg[0].cl[0] = 0.0000000000;	data_table[71].feg[0].cnl[0] = 0.0000000000;	data_table[71].feg[0].cl[1] = 0.0000000000;	data_table[71].feg[0].cnl[1] = 0.0000000000;	data_table[71].feg[0].cl[2] = 0.0000000000;	data_table[71].feg[0].cnl[2] = 0.0000000000;	data_table[71].feg[0].cl[3] = 0.0000000000;	data_table[71].feg[0].cnl[3] = 0.0000000000;	data_table[71].feg[0].cl[4] = 0.0000000000;	data_table[71].feg[0].cnl[4] = 0.0000000000;	data_table[71].feg[0].cl[5] = 0.0000000000;	data_table[71].feg[0].cnl[5] = 0.0000000000;
				data_table[72].feg[0].cl[0] = 0.0000000000;	data_table[72].feg[0].cnl[0] = 0.0000000000;	data_table[72].feg[0].cl[1] = 0.0000000000;	data_table[72].feg[0].cnl[1] = 0.0000000000;	data_table[72].feg[0].cl[2] = 0.0000000000;	data_table[72].feg[0].cnl[2] = 0.0000000000;	data_table[72].feg[0].cl[3] = 0.0000000000;	data_table[72].feg[0].cnl[3] = 0.0000000000;	data_table[72].feg[0].cl[4] = 0.0000000000;	data_table[72].feg[0].cnl[4] = 0.0000000000;	data_table[72].feg[0].cl[5] = 0.0000000000;	data_table[72].feg[0].cnl[5] = 0.0000000000;
				data_table[73].feg[0].cl[0] = 0.0000000000;	data_table[73].feg[0].cnl[0] = 0.0000000000;	data_table[73].feg[0].cl[1] = 0.0000000000;	data_table[73].feg[0].cnl[1] = 0.0000000000;	data_table[73].feg[0].cl[2] = 0.0000000000;	data_table[73].feg[0].cnl[2] = 0.0000000000;	data_table[73].feg[0].cl[3] = 0.0000000000;	data_table[73].feg[0].cnl[3] = 0.0000000000;	data_table[73].feg[0].cl[4] = 0.0000000000;	data_table[73].feg[0].cnl[4] = 0.0000000000;	data_table[73].feg[0].cl[5] = 0.0000000000;	data_table[73].feg[0].cnl[5] = 0.0000000000;
				data_table[74].feg[0].cl[0] = 0.0000000000;	data_table[74].feg[0].cnl[0] = 0.0000000000;	data_table[74].feg[0].cl[1] = 0.0000000000;	data_table[74].feg[0].cnl[1] = 0.0000000000;	data_table[74].feg[0].cl[2] = 0.0000000000;	data_table[74].feg[0].cnl[2] = 0.0000000000;	data_table[74].feg[0].cl[3] = 0.0000000000;	data_table[74].feg[0].cnl[3] = 0.0000000000;	data_table[74].feg[0].cl[4] = 0.0000000000;	data_table[74].feg[0].cnl[4] = 0.0000000000;	data_table[74].feg[0].cl[5] = 0.0000000000;	data_table[74].feg[0].cnl[5] = 0.0000000000;
				data_table[75].feg[0].cl[0] = 0.0000000000;	data_table[75].feg[0].cnl[0] = 0.0000000000;	data_table[75].feg[0].cl[1] = 0.0000000000;	data_table[75].feg[0].cnl[1] = 0.0000000000;	data_table[75].feg[0].cl[2] = 0.0000000000;	data_table[75].feg[0].cnl[2] = 0.0000000000;	data_table[75].feg[0].cl[3] = 0.0000000000;	data_table[75].feg[0].cnl[3] = 0.0000000000;	data_table[75].feg[0].cl[4] = 0.0000000000;	data_table[75].feg[0].cnl[4] = 0.0000000000;	data_table[75].feg[0].cl[5] = 0.0000000000;	data_table[75].feg[0].cnl[5] = 0.0000000000;
				data_table[76].feg[0].cl[0] = 0.0000000000;	data_table[76].feg[0].cnl[0] = 0.0000000000;	data_table[76].feg[0].cl[1] = 0.0000000000;	data_table[76].feg[0].cnl[1] = 0.0000000000;	data_table[76].feg[0].cl[2] = 0.0000000000;	data_table[76].feg[0].cnl[2] = 0.0000000000;	data_table[76].feg[0].cl[3] = 0.0000000000;	data_table[76].feg[0].cnl[3] = 0.0000000000;	data_table[76].feg[0].cl[4] = 0.0000000000;	data_table[76].feg[0].cnl[4] = 0.0000000000;	data_table[76].feg[0].cl[5] = 0.0000000000;	data_table[76].feg[0].cnl[5] = 0.0000000000;
				data_table[77].feg[0].cl[0] = 0.0000000000;	data_table[77].feg[0].cnl[0] = 0.0000000000;	data_table[77].feg[0].cl[1] = 0.0000000000;	data_table[77].feg[0].cnl[1] = 0.0000000000;	data_table[77].feg[0].cl[2] = 0.0000000000;	data_table[77].feg[0].cnl[2] = 0.0000000000;	data_table[77].feg[0].cl[3] = 0.0000000000;	data_table[77].feg[0].cnl[3] = 0.0000000000;	data_table[77].feg[0].cl[4] = 0.0000000000;	data_table[77].feg[0].cnl[4] = 0.0000000000;	data_table[77].feg[0].cl[5] = 0.0000000000;	data_table[77].feg[0].cnl[5] = 0.0000000000;
				data_table[78].feg[0].cl[0] = 2.3880000000;	data_table[78].feg[0].cnl[0] = 42.8656000000;	data_table[78].feg[0].cl[1] = 4.2259000000;	data_table[78].feg[0].cnl[1] = 9.7430000000;	data_table[78].feg[0].cl[2] = 2.6886000000;	data_table[78].feg[0].cnl[2] = 2.2641000000;	data_table[78].feg[0].cl[3] = 1.2551000000;	data_table[78].feg[0].cnl[3] = 0.3067000000;	data_table[78].feg[0].cl[4] = 0.0000000000;	data_table[78].feg[0].cnl[4] = 0.0000000000;	data_table[78].feg[0].cl[5] = 0.0000000000;	data_table[78].feg[0].cnl[5] = 0.0000000000;
				data_table[79].feg[0].cl[0] = 2.6817000000;	data_table[79].feg[0].cnl[0] = 42.8217000000;	data_table[79].feg[0].cl[1] = 4.2414000000;	data_table[79].feg[0].cnl[1] = 9.8557000000;	data_table[79].feg[0].cl[2] = 2.7549000000;	data_table[79].feg[0].cnl[2] = 2.2951000000;	data_table[79].feg[0].cl[3] = 1.2706000000;	data_table[79].feg[0].cnl[3] = 0.3067000000;	data_table[79].feg[0].cl[4] = 0.0000000000;	data_table[79].feg[0].cnl[4] = 0.0000000000;	data_table[79].feg[0].cl[5] = 0.0000000000;	data_table[79].feg[0].cnl[5] = 0.0000000000;
				data_table[80].feg[0].cl[0] = 0.0000000000;	data_table[80].feg[0].cnl[0] = 0.0000000000;	data_table[80].feg[0].cl[1] = 0.0000000000;	data_table[80].feg[0].cnl[1] = 0.0000000000;	data_table[80].feg[0].cl[2] = 0.0000000000;	data_table[80].feg[0].cnl[2] = 0.0000000000;	data_table[80].feg[0].cl[3] = 0.0000000000;	data_table[80].feg[0].cnl[3] = 0.0000000000;	data_table[80].feg[0].cl[4] = 0.0000000000;	data_table[80].feg[0].cnl[4] = 0.0000000000;	data_table[80].feg[0].cl[5] = 0.0000000000;	data_table[80].feg[0].cnl[5] = 0.0000000000;
				data_table[81].feg[0].cl[0] = 3.5099000000;	data_table[81].feg[0].cnl[0] = 52.9141000000;	data_table[81].feg[0].cl[1] = 4.5523000000;	data_table[81].feg[0].cnl[1] = 11.8840000000;	data_table[81].feg[0].cl[2] = 3.1539000000;	data_table[81].feg[0].cnl[2] = 2.5713000000;	data_table[81].feg[0].cl[3] = 1.3591000000;	data_table[81].feg[0].cnl[3] = 0.3205000000;	data_table[81].feg[0].cl[4] = 0.0000000000;	data_table[81].feg[0].cnl[4] = 0.0000000000;	data_table[81].feg[0].cl[5] = 0.0000000000;	data_table[81].feg[0].cnl[5] = 0.0000000000;
				data_table[82].feg[0].cl[0] = 3.8412000000;	data_table[82].feg[0].cnl[0] = 50.2608000000;	data_table[82].feg[0].cl[1] = 4.6784000000;	data_table[82].feg[0].cnl[1] = 11.9988000000;	data_table[82].feg[0].cl[2] = 3.1924000000;	data_table[82].feg[0].cnl[2] = 2.5598000000;	data_table[82].feg[0].cl[3] = 1.3625000000;	data_table[82].feg[0].cnl[3] = 0.3177000000;	data_table[82].feg[0].cl[4] = 0.0000000000;	data_table[82].feg[0].cnl[4] = 0.0000000000;	data_table[82].feg[0].cl[5] = 0.0000000000;	data_table[82].feg[0].cnl[5] = 0.0000000000;
				data_table[83].feg[0].cl[0] = 0.0000000000;	data_table[83].feg[0].cnl[0] = 0.0000000000;	data_table[83].feg[0].cl[1] = 0.0000000000;	data_table[83].feg[0].cnl[1] = 0.0000000000;	data_table[83].feg[0].cl[2] = 0.0000000000;	data_table[83].feg[0].cnl[2] = 0.0000000000;	data_table[83].feg[0].cl[3] = 0.0000000000;	data_table[83].feg[0].cnl[3] = 0.0000000000;	data_table[83].feg[0].cl[4] = 0.0000000000;	data_table[83].feg[0].cnl[4] = 0.0000000000;	data_table[83].feg[0].cl[5] = 0.0000000000;	data_table[83].feg[0].cnl[5] = 0.0000000000;
				data_table[84].feg[0].cl[0] = 0.0000000000;	data_table[84].feg[0].cnl[0] = 0.0000000000;	data_table[84].feg[0].cl[1] = 0.0000000000;	data_table[84].feg[0].cnl[1] = 0.0000000000;	data_table[84].feg[0].cl[2] = 0.0000000000;	data_table[84].feg[0].cnl[2] = 0.0000000000;	data_table[84].feg[0].cl[3] = 0.0000000000;	data_table[84].feg[0].cnl[3] = 0.0000000000;	data_table[84].feg[0].cl[4] = 0.0000000000;	data_table[84].feg[0].cnl[4] = 0.0000000000;	data_table[84].feg[0].cl[5] = 0.0000000000;	data_table[84].feg[0].cnl[5] = 0.0000000000;
				data_table[85].feg[0].cl[0] = 4.0779000000;	data_table[85].feg[0].cnl[0] = 28.4058000000;	data_table[85].feg[0].cl[1] = 4.9778000000;	data_table[85].feg[0].cnl[1] = 11.0204000000;	data_table[85].feg[0].cl[2] = 3.0955000000;	data_table[85].feg[0].cnl[2] = 2.3549000000;	data_table[85].feg[0].cl[3] = 1.3259000000;	data_table[85].feg[0].cnl[3] = 0.2991000000;	data_table[85].feg[0].cl[4] = 0.0000000000;	data_table[85].feg[0].cnl[4] = 0.0000000000;	data_table[85].feg[0].cl[5] = 0.0000000000;	data_table[85].feg[0].cnl[5] = 0.0000000000;
				data_table[86].feg[0].cl[0] = 0.0000000000;	data_table[86].feg[0].cnl[0] = 0.0000000000;	data_table[86].feg[0].cl[1] = 0.0000000000;	data_table[86].feg[0].cnl[1] = 0.0000000000;	data_table[86].feg[0].cl[2] = 0.0000000000;	data_table[86].feg[0].cnl[2] = 0.0000000000;	data_table[86].feg[0].cl[3] = 0.0000000000;	data_table[86].feg[0].cnl[3] = 0.0000000000;	data_table[86].feg[0].cl[4] = 0.0000000000;	data_table[86].feg[0].cnl[4] = 0.0000000000;	data_table[86].feg[0].cl[5] = 0.0000000000;	data_table[86].feg[0].cnl[5] = 0.0000000000;
				data_table[87].feg[0].cl[0] = 0.0000000000;	data_table[87].feg[0].cnl[0] = 0.0000000000;	data_table[87].feg[0].cl[1] = 0.0000000000;	data_table[87].feg[0].cnl[1] = 0.0000000000;	data_table[87].feg[0].cl[2] = 0.0000000000;	data_table[87].feg[0].cnl[2] = 0.0000000000;	data_table[87].feg[0].cl[3] = 0.0000000000;	data_table[87].feg[0].cnl[3] = 0.0000000000;	data_table[87].feg[0].cl[4] = 0.0000000000;	data_table[87].feg[0].cnl[4] = 0.0000000000;	data_table[87].feg[0].cl[5] = 0.0000000000;	data_table[87].feg[0].cnl[5] = 0.0000000000;
				data_table[88].feg[0].cl[0] = 0.0000000000;	data_table[88].feg[0].cnl[0] = 0.0000000000;	data_table[88].feg[0].cl[1] = 0.0000000000;	data_table[88].feg[0].cnl[1] = 0.0000000000;	data_table[88].feg[0].cl[2] = 0.0000000000;	data_table[88].feg[0].cnl[2] = 0.0000000000;	data_table[88].feg[0].cl[3] = 0.0000000000;	data_table[88].feg[0].cnl[3] = 0.0000000000;	data_table[88].feg[0].cl[4] = 0.0000000000;	data_table[88].feg[0].cnl[4] = 0.0000000000;	data_table[88].feg[0].cl[5] = 0.0000000000;	data_table[88].feg[0].cnl[5] = 0.0000000000;
				data_table[89].feg[0].cl[0] = 0.0000000000;	data_table[89].feg[0].cnl[0] = 0.0000000000;	data_table[89].feg[0].cl[1] = 0.0000000000;	data_table[89].feg[0].cnl[1] = 0.0000000000;	data_table[89].feg[0].cl[2] = 0.0000000000;	data_table[89].feg[0].cnl[2] = 0.0000000000;	data_table[89].feg[0].cl[3] = 0.0000000000;	data_table[89].feg[0].cnl[3] = 0.0000000000;	data_table[89].feg[0].cl[4] = 0.0000000000;	data_table[89].feg[0].cnl[4] = 0.0000000000;	data_table[89].feg[0].cl[5] = 0.0000000000;	data_table[89].feg[0].cnl[5] = 0.0000000000;
				data_table[90].feg[0].cl[0] = 0.0000000000;	data_table[90].feg[0].cnl[0] = 0.0000000000;	data_table[90].feg[0].cl[1] = 0.0000000000;	data_table[90].feg[0].cnl[1] = 0.0000000000;	data_table[90].feg[0].cl[2] = 0.0000000000;	data_table[90].feg[0].cnl[2] = 0.0000000000;	data_table[90].feg[0].cl[3] = 0.0000000000;	data_table[90].feg[0].cnl[3] = 0.0000000000;	data_table[90].feg[0].cl[4] = 0.0000000000;	data_table[90].feg[0].cnl[4] = 0.0000000000;	data_table[90].feg[0].cl[5] = 0.0000000000;	data_table[90].feg[0].cnl[5] = 0.0000000000;
				data_table[91].feg[0].cl[0] = 6.7668000000;	data_table[91].feg[0].cnl[0] = 85.9510000000;	data_table[91].feg[0].cl[1] = 6.7287000000;	data_table[91].feg[0].cnl[1] = 15.6415000000;	data_table[91].feg[0].cl[2] = 4.0135000000;	data_table[91].feg[0].cnl[2] = 2.9364000000;	data_table[91].feg[0].cl[3] = 1.5607000000;	data_table[91].feg[0].cnl[3] = 0.3348000000;	data_table[91].feg[0].cl[4] = 0.0000000000;	data_table[91].feg[0].cnl[4] = 0.0000000000;	data_table[91].feg[0].cl[5] = 0.0000000000;	data_table[91].feg[0].cnl[5] = 0.0000000000;
				data_table[92].feg[0].cl[0] = 0.0000000000;	data_table[92].feg[0].cnl[0] = 0.0000000000;	data_table[92].feg[0].cl[1] = 0.0000000000;	data_table[92].feg[0].cnl[1] = 0.0000000000;	data_table[92].feg[0].cl[2] = 0.0000000000;	data_table[92].feg[0].cnl[2] = 0.0000000000;	data_table[92].feg[0].cl[3] = 0.0000000000;	data_table[92].feg[0].cnl[3] = 0.0000000000;	data_table[92].feg[0].cl[4] = 0.0000000000;	data_table[92].feg[0].cnl[4] = 0.0000000000;	data_table[92].feg[0].cl[5] = 0.0000000000;	data_table[92].feg[0].cnl[5] = 0.0000000000;
				data_table[93].feg[0].cl[0] = 0.0000000000;	data_table[93].feg[0].cnl[0] = 0.0000000000;	data_table[93].feg[0].cl[1] = 0.0000000000;	data_table[93].feg[0].cnl[1] = 0.0000000000;	data_table[93].feg[0].cl[2] = 0.0000000000;	data_table[93].feg[0].cnl[2] = 0.0000000000;	data_table[93].feg[0].cl[3] = 0.0000000000;	data_table[93].feg[0].cnl[3] = 0.0000000000;	data_table[93].feg[0].cl[4] = 0.0000000000;	data_table[93].feg[0].cnl[4] = 0.0000000000;	data_table[93].feg[0].cl[5] = 0.0000000000;	data_table[93].feg[0].cnl[5] = 0.0000000000;
				data_table[94].feg[0].cl[0] = 0.0000000000;	data_table[94].feg[0].cnl[0] = 0.0000000000;	data_table[94].feg[0].cl[1] = 0.0000000000;	data_table[94].feg[0].cnl[1] = 0.0000000000;	data_table[94].feg[0].cl[2] = 0.0000000000;	data_table[94].feg[0].cnl[2] = 0.0000000000;	data_table[94].feg[0].cl[3] = 0.0000000000;	data_table[94].feg[0].cnl[3] = 0.0000000000;	data_table[94].feg[0].cl[4] = 0.0000000000;	data_table[94].feg[0].cnl[4] = 0.0000000000;	data_table[94].feg[0].cl[5] = 0.0000000000;	data_table[94].feg[0].cnl[5] = 0.0000000000;
				data_table[95].feg[0].cl[0] = 0.0000000000;	data_table[95].feg[0].cnl[0] = 0.0000000000;	data_table[95].feg[0].cl[1] = 0.0000000000;	data_table[95].feg[0].cnl[1] = 0.0000000000;	data_table[95].feg[0].cl[2] = 0.0000000000;	data_table[95].feg[0].cnl[2] = 0.0000000000;	data_table[95].feg[0].cl[3] = 0.0000000000;	data_table[95].feg[0].cnl[3] = 0.0000000000;	data_table[95].feg[0].cl[4] = 0.0000000000;	data_table[95].feg[0].cnl[4] = 0.0000000000;	data_table[95].feg[0].cl[5] = 0.0000000000;	data_table[95].feg[0].cnl[5] = 0.0000000000;
				data_table[96].feg[0].cl[0] = 0.0000000000;	data_table[96].feg[0].cnl[0] = 0.0000000000;	data_table[96].feg[0].cl[1] = 0.0000000000;	data_table[96].feg[0].cnl[1] = 0.0000000000;	data_table[96].feg[0].cl[2] = 0.0000000000;	data_table[96].feg[0].cnl[2] = 0.0000000000;	data_table[96].feg[0].cl[3] = 0.0000000000;	data_table[96].feg[0].cnl[3] = 0.0000000000;	data_table[96].feg[0].cl[4] = 0.0000000000;	data_table[96].feg[0].cnl[4] = 0.0000000000;	data_table[96].feg[0].cl[5] = 0.0000000000;	data_table[96].feg[0].cnl[5] = 0.0000000000;
				data_table[97].feg[0].cl[0] = 0.0000000000;	data_table[97].feg[0].cnl[0] = 0.0000000000;	data_table[97].feg[0].cl[1] = 0.0000000000;	data_table[97].feg[0].cnl[1] = 0.0000000000;	data_table[97].feg[0].cl[2] = 0.0000000000;	data_table[97].feg[0].cnl[2] = 0.0000000000;	data_table[97].feg[0].cl[3] = 0.0000000000;	data_table[97].feg[0].cnl[3] = 0.0000000000;	data_table[97].feg[0].cl[4] = 0.0000000000;	data_table[97].feg[0].cnl[4] = 0.0000000000;	data_table[97].feg[0].cl[5] = 0.0000000000;	data_table[97].feg[0].cnl[5] = 0.0000000000;
				data_table[98].feg[0].cl[0] = 0.0000000000;	data_table[98].feg[0].cnl[0] = 0.0000000000;	data_table[98].feg[0].cl[1] = 0.0000000000;	data_table[98].feg[0].cnl[1] = 0.0000000000;	data_table[98].feg[0].cl[2] = 0.0000000000;	data_table[98].feg[0].cnl[2] = 0.0000000000;	data_table[98].feg[0].cl[3] = 0.0000000000;	data_table[98].feg[0].cnl[3] = 0.0000000000;	data_table[98].feg[0].cl[4] = 0.0000000000;	data_table[98].feg[0].cnl[4] = 0.0000000000;	data_table[98].feg[0].cl[5] = 0.0000000000;	data_table[98].feg[0].cnl[5] = 0.0000000000;
				data_table[99].feg[0].cl[0] = 0.0000000000;	data_table[99].feg[0].cnl[0] = 0.0000000000;	data_table[99].feg[0].cl[1] = 0.0000000000;	data_table[99].feg[0].cnl[1] = 0.0000000000;	data_table[99].feg[0].cl[2] = 0.0000000000;	data_table[99].feg[0].cnl[2] = 0.0000000000;	data_table[99].feg[0].cl[3] = 0.0000000000;	data_table[99].feg[0].cnl[3] = 0.0000000000;	data_table[99].feg[0].cl[4] = 0.0000000000;	data_table[99].feg[0].cnl[4] = 0.0000000000;	data_table[99].feg[0].cl[5] = 0.0000000000;	data_table[99].feg[0].cnl[5] = 0.0000000000;
				data_table[100].feg[0].cl[0] = 0.0000000000;	data_table[100].feg[0].cnl[0] = 0.0000000000;	data_table[100].feg[0].cl[1] = 0.0000000000;	data_table[100].feg[0].cnl[1] = 0.0000000000;	data_table[100].feg[0].cl[2] = 0.0000000000;	data_table[100].feg[0].cnl[2] = 0.0000000000;	data_table[100].feg[0].cl[3] = 0.0000000000;	data_table[100].feg[0].cnl[3] = 0.0000000000;	data_table[100].feg[0].cl[4] = 0.0000000000;	data_table[100].feg[0].cnl[4] = 0.0000000000;	data_table[100].feg[0].cl[5] = 0.0000000000;	data_table[100].feg[0].cnl[5] = 0.0000000000;
				data_table[101].feg[0].cl[0] = 0.0000000000;	data_table[101].feg[0].cnl[0] = 0.0000000000;	data_table[101].feg[0].cl[1] = 0.0000000000;	data_table[101].feg[0].cnl[1] = 0.0000000000;	data_table[101].feg[0].cl[2] = 0.0000000000;	data_table[101].feg[0].cnl[2] = 0.0000000000;	data_table[101].feg[0].cl[3] = 0.0000000000;	data_table[101].feg[0].cnl[3] = 0.0000000000;	data_table[101].feg[0].cl[4] = 0.0000000000;	data_table[101].feg[0].cnl[4] = 0.0000000000;	data_table[101].feg[0].cl[5] = 0.0000000000;	data_table[101].feg[0].cnl[5] = 0.0000000000;
				data_table[102].feg[0].cl[0] = 0.0000000000;	data_table[102].feg[0].cnl[0] = 0.0000000000;	data_table[102].feg[0].cl[1] = 0.0000000000;	data_table[102].feg[0].cnl[1] = 0.0000000000;	data_table[102].feg[0].cl[2] = 0.0000000000;	data_table[102].feg[0].cnl[2] = 0.0000000000;	data_table[102].feg[0].cl[3] = 0.0000000000;	data_table[102].feg[0].cnl[3] = 0.0000000000;	data_table[102].feg[0].cl[4] = 0.0000000000;	data_table[102].feg[0].cnl[4] = 0.0000000000;	data_table[102].feg[0].cl[5] = 0.0000000000;	data_table[102].feg[0].cnl[5] = 0.0000000000;
			}

			// 2: Peng et al. parameterization - 5 Gaussians - [0, 4]
			void Load_feg_Peng_neutral_0_4()
			{
				data_table[0].feg[0].cl[0] = 0.0349000000;		data_table[0].feg[0].cnl[0] = 0.5347000000;	data_table[0].feg[0].cl[1] = 0.1201000000;		data_table[0].feg[0].cnl[1] = 3.5867000000;	data_table[0].feg[0].cl[2] = 0.1970000000;		data_table[0].feg[0].cnl[2] = 12.3471000000;	data_table[0].feg[0].cl[3] = 0.0573000000;		data_table[0].feg[0].cnl[3] = 18.9525000000;	data_table[0].feg[0].cl[4] = 0.1195000000;		data_table[0].feg[0].cnl[4] = 38.6269000000;	data_table[0].feg[0].cl[5] = 0.0000000000;		data_table[0].feg[0].cnl[5] = 0.0000000000;
				data_table[1].feg[0].cl[0] = 0.0317000000;		data_table[1].feg[0].cnl[0] = 0.2507000000;	data_table[1].feg[0].cl[1] = 0.0838000000;		data_table[1].feg[0].cnl[1] = 1.4751000000;	data_table[1].feg[0].cl[2] = 0.1526000000;		data_table[1].feg[0].cnl[2] = 4.4938000000;	data_table[1].feg[0].cl[3] = 0.1334000000;		data_table[1].feg[0].cnl[3] = 12.6646000000;	data_table[1].feg[0].cl[4] = 0.0164000000;		data_table[1].feg[0].cnl[4] = 31.1653000000;	data_table[1].feg[0].cl[5] = 0.0000000000;		data_table[1].feg[0].cnl[5] = 0.0000000000;
				data_table[2].feg[0].cl[0] = 0.0750000000;		data_table[2].feg[0].cnl[0] = 0.3864000000;	data_table[2].feg[0].cl[1] = 0.2249000000;		data_table[2].feg[0].cnl[1] = 2.9383000000;	data_table[2].feg[0].cl[2] = 0.5548000000;		data_table[2].feg[0].cnl[2] = 15.3829000000;	data_table[2].feg[0].cl[3] = 1.4954000000;		data_table[2].feg[0].cnl[3] = 53.5545000000;	data_table[2].feg[0].cl[4] = 0.9354000000;		data_table[2].feg[0].cnl[4] = 138.7337000000;	data_table[2].feg[0].cl[5] = 0.0000000000;		data_table[2].feg[0].cnl[5] = 0.0000000000;
				data_table[3].feg[0].cl[0] = 0.0780000000;		data_table[3].feg[0].cnl[0] = 0.3131000000;	data_table[3].feg[0].cl[1] = 0.2210000000;		data_table[3].feg[0].cnl[1] = 2.2381000000;	data_table[3].feg[0].cl[2] = 0.6740000000;		data_table[3].feg[0].cnl[2] = 10.1517000000;	data_table[3].feg[0].cl[3] = 1.3867000000;		data_table[3].feg[0].cnl[3] = 30.9061000000;	data_table[3].feg[0].cl[4] = 0.6925000000;		data_table[3].feg[0].cnl[4] = 78.3273000000;	data_table[3].feg[0].cl[5] = 0.0000000000;		data_table[3].feg[0].cnl[5] = 0.0000000000;
				data_table[4].feg[0].cl[0] = 0.0909000000;		data_table[4].feg[0].cnl[0] = 0.2995000000;	data_table[4].feg[0].cl[1] = 0.2551000000;		data_table[4].feg[0].cnl[1] = 2.1155000000;	data_table[4].feg[0].cl[2] = 0.7738000000;		data_table[4].feg[0].cnl[2] = 8.3816000000;	data_table[4].feg[0].cl[3] = 1.2136000000;		data_table[4].feg[0].cnl[3] = 24.1292000000;	data_table[4].feg[0].cl[4] = 0.4606000000;		data_table[4].feg[0].cnl[4] = 63.1314000000;	data_table[4].feg[0].cl[5] = 0.0000000000;		data_table[4].feg[0].cnl[5] = 0.0000000000;
				data_table[5].feg[0].cl[0] = 0.0893000000;		data_table[5].feg[0].cnl[0] = 0.2465000000;	data_table[5].feg[0].cl[1] = 0.2563000000;		data_table[5].feg[0].cnl[1] = 1.7100000000;	data_table[5].feg[0].cl[2] = 0.7570000000;		data_table[5].feg[0].cnl[2] = 6.4094000000;	data_table[5].feg[0].cl[3] = 1.0487000000;		data_table[5].feg[0].cnl[3] = 18.6113000000;	data_table[5].feg[0].cl[4] = 0.3575000000;		data_table[5].feg[0].cnl[4] = 50.2523000000;	data_table[5].feg[0].cl[5] = 0.0000000000;		data_table[5].feg[0].cnl[5] = 0.0000000000;
				data_table[6].feg[0].cl[0] = 0.1022000000;		data_table[6].feg[0].cnl[0] = 0.2451000000;	data_table[6].feg[0].cl[1] = 0.3219000000;		data_table[6].feg[0].cnl[1] = 1.7481000000;	data_table[6].feg[0].cl[2] = 0.7982000000;		data_table[6].feg[0].cnl[2] = 6.1925000000;	data_table[6].feg[0].cl[3] = 0.8197000000;		data_table[6].feg[0].cnl[3] = 17.3894000000;	data_table[6].feg[0].cl[4] = 0.1715000000;		data_table[6].feg[0].cnl[4] = 48.1431000000;	data_table[6].feg[0].cl[5] = 0.0000000000;		data_table[6].feg[0].cnl[5] = 0.0000000000;
				data_table[7].feg[0].cl[0] = 0.0974000000;		data_table[7].feg[0].cnl[0] = 0.2067000000;	data_table[7].feg[0].cl[1] = 0.2921000000;		data_table[7].feg[0].cnl[1] = 1.3815000000;	data_table[7].feg[0].cl[2] = 0.6910000000;		data_table[7].feg[0].cnl[2] = 4.6943000000;	data_table[7].feg[0].cl[3] = 0.6990000000;		data_table[7].feg[0].cnl[3] = 12.7105000000;	data_table[7].feg[0].cl[4] = 0.2039000000;		data_table[7].feg[0].cnl[4] = 32.4726000000;	data_table[7].feg[0].cl[5] = 0.0000000000;		data_table[7].feg[0].cnl[5] = 0.0000000000;
				data_table[8].feg[0].cl[0] = 0.1083000000;		data_table[8].feg[0].cnl[0] = 0.2057000000;	data_table[8].feg[0].cl[1] = 0.3175000000;		data_table[8].feg[0].cnl[1] = 1.3439000000;	data_table[8].feg[0].cl[2] = 0.6487000000;		data_table[8].feg[0].cnl[2] = 4.2788000000;	data_table[8].feg[0].cl[3] = 0.5846000000;		data_table[8].feg[0].cnl[3] = 11.3932000000;	data_table[8].feg[0].cl[4] = 0.1421000000;		data_table[8].feg[0].cnl[4] = 28.7881000000;	data_table[8].feg[0].cl[5] = 0.0000000000;		data_table[8].feg[0].cnl[5] = 0.0000000000;
				data_table[9].feg[0].cl[0] = 0.1269000000;		data_table[9].feg[0].cnl[0] = 0.2200000000;	data_table[9].feg[0].cl[1] = 0.3535000000;		data_table[9].feg[0].cnl[1] = 1.3779000000;	data_table[9].feg[0].cl[2] = 0.5582000000;		data_table[9].feg[0].cnl[2] = 4.0203000000;	data_table[9].feg[0].cl[3] = 0.4674000000;		data_table[9].feg[0].cnl[3] = 9.4934000000;	data_table[9].feg[0].cl[4] = 0.1460000000;		data_table[9].feg[0].cnl[4] = 23.1278000000;	data_table[9].feg[0].cl[5] = 0.0000000000;		data_table[9].feg[0].cnl[5] = 0.0000000000;
				data_table[10].feg[0].cl[0] = 0.2142000000;	data_table[10].feg[0].cnl[0] = 0.3334000000;	data_table[10].feg[0].cl[1] = 0.6853000000;	data_table[10].feg[0].cnl[1] = 2.3446000000;	data_table[10].feg[0].cl[2] = 0.7692000000;	data_table[10].feg[0].cnl[2] = 10.0830000000;	data_table[10].feg[0].cl[3] = 1.6589000000;	data_table[10].feg[0].cnl[3] = 48.3037000000;	data_table[10].feg[0].cl[4] = 1.4482000000;	data_table[10].feg[0].cnl[4] = 138.2700000000;	data_table[10].feg[0].cl[5] = 0.0000000000;	data_table[10].feg[0].cnl[5] = 0.0000000000;
				data_table[11].feg[0].cl[0] = 0.2314000000;	data_table[11].feg[0].cnl[0] = 0.3278000000;	data_table[11].feg[0].cl[1] = 0.6866000000;	data_table[11].feg[0].cnl[1] = 2.2720000000;	data_table[11].feg[0].cl[2] = 0.9677000000;	data_table[11].feg[0].cnl[2] = 10.9241000000;	data_table[11].feg[0].cl[3] = 2.1882000000;	data_table[11].feg[0].cnl[3] = 39.2898000000;	data_table[11].feg[0].cl[4] = 1.1339000000;	data_table[11].feg[0].cnl[4] = 101.9748000000;	data_table[11].feg[0].cl[5] = 0.0000000000;	data_table[11].feg[0].cnl[5] = 0.0000000000;
				data_table[12].feg[0].cl[0] = 0.2390000000;	data_table[12].feg[0].cnl[0] = 0.3138000000;	data_table[12].feg[0].cl[1] = 0.6573000000;	data_table[12].feg[0].cnl[1] = 2.1063000000;	data_table[12].feg[0].cl[2] = 1.2011000000;	data_table[12].feg[0].cnl[2] = 10.4163000000;	data_table[12].feg[0].cl[3] = 2.5586000000;	data_table[12].feg[0].cnl[3] = 34.4552000000;	data_table[12].feg[0].cl[4] = 1.2312000000;	data_table[12].feg[0].cnl[4] = 98.5344000000;	data_table[12].feg[0].cl[5] = 0.0000000000;	data_table[12].feg[0].cnl[5] = 0.0000000000;
				data_table[13].feg[0].cl[0] = 0.2519000000;	data_table[13].feg[0].cnl[0] = 0.3075000000;	data_table[13].feg[0].cl[1] = 0.6372000000;	data_table[13].feg[0].cnl[1] = 2.0174000000;	data_table[13].feg[0].cl[2] = 1.3795000000;	data_table[13].feg[0].cnl[2] = 9.6746000000;	data_table[13].feg[0].cl[3] = 2.5082000000;	data_table[13].feg[0].cnl[3] = 29.3744000000;	data_table[13].feg[0].cl[4] = 1.0500000000;	data_table[13].feg[0].cnl[4] = 80.4732000000;	data_table[13].feg[0].cl[5] = 0.0000000000;	data_table[13].feg[0].cnl[5] = 0.0000000000;
				data_table[14].feg[0].cl[0] = 0.2548000000;	data_table[14].feg[0].cnl[0] = 0.2908000000;	data_table[14].feg[0].cl[1] = 0.6106000000;	data_table[14].feg[0].cnl[1] = 1.8740000000;	data_table[14].feg[0].cl[2] = 1.4541000000;	data_table[14].feg[0].cnl[2] = 8.5176000000;	data_table[14].feg[0].cl[3] = 2.3204000000;	data_table[14].feg[0].cnl[3] = 24.3434000000;	data_table[14].feg[0].cl[4] = 0.8477000000;	data_table[14].feg[0].cnl[4] = 63.2996000000;	data_table[14].feg[0].cl[5] = 0.0000000000;	data_table[14].feg[0].cnl[5] = 0.0000000000;
				data_table[15].feg[0].cl[0] = 0.2497000000;	data_table[15].feg[0].cnl[0] = 0.2681000000;	data_table[15].feg[0].cl[1] = 0.5628000000;	data_table[15].feg[0].cnl[1] = 1.6711000000;	data_table[15].feg[0].cl[2] = 1.3899000000;	data_table[15].feg[0].cnl[2] = 7.0267000000;	data_table[15].feg[0].cl[3] = 2.1865000000;	data_table[15].feg[0].cnl[3] = 19.5377000000;	data_table[15].feg[0].cl[4] = 0.7715000000;	data_table[15].feg[0].cnl[4] = 50.3888000000;	data_table[15].feg[0].cl[5] = 0.0000000000;	data_table[15].feg[0].cnl[5] = 0.0000000000;
				data_table[16].feg[0].cl[0] = 0.2443000000;	data_table[16].feg[0].cnl[0] = 0.2468000000;	data_table[16].feg[0].cl[1] = 0.5397000000;	data_table[16].feg[0].cnl[1] = 1.5242000000;	data_table[16].feg[0].cl[2] = 1.3919000000;	data_table[16].feg[0].cnl[2] = 6.1537000000;	data_table[16].feg[0].cl[3] = 2.0197000000;	data_table[16].feg[0].cnl[3] = 16.6687000000;	data_table[16].feg[0].cl[4] = 0.6621000000;	data_table[16].feg[0].cnl[4] = 42.3086000000;	data_table[16].feg[0].cl[5] = 0.0000000000;	data_table[16].feg[0].cnl[5] = 0.0000000000;
				data_table[17].feg[0].cl[0] = 0.2385000000;	data_table[17].feg[0].cnl[0] = 0.2289000000;	data_table[17].feg[0].cl[1] = 0.5017000000;	data_table[17].feg[0].cnl[1] = 1.3694000000;	data_table[17].feg[0].cl[2] = 1.3428000000;	data_table[17].feg[0].cnl[2] = 5.2561000000;	data_table[17].feg[0].cl[3] = 1.8899000000;	data_table[17].feg[0].cnl[3] = 14.0928000000;	data_table[17].feg[0].cl[4] = 0.6079000000;	data_table[17].feg[0].cnl[4] = 35.5361000000;	data_table[17].feg[0].cl[5] = 0.0000000000;	data_table[17].feg[0].cnl[5] = 0.0000000000;
				data_table[18].feg[0].cl[0] = 0.4115000000;	data_table[18].feg[0].cnl[0] = 0.3703000000;	data_table[18].feg[0].cl[1] = 1.4031000000;	data_table[18].feg[0].cnl[1] = 3.3874000000;	data_table[18].feg[0].cl[2] = 2.2784000000;	data_table[18].feg[0].cnl[2] = 13.1029000000;	data_table[18].feg[0].cl[3] = 2.6742000000;	data_table[18].feg[0].cnl[3] = 68.9592000000;	data_table[18].feg[0].cl[4] = 2.2162000000;	data_table[18].feg[0].cnl[4] = 194.4329000000;	data_table[18].feg[0].cl[5] = 0.0000000000;	data_table[18].feg[0].cnl[5] = 0.0000000000;
				data_table[19].feg[0].cl[0] = 0.4054000000;	data_table[19].feg[0].cnl[0] = 0.3499000000;	data_table[19].feg[0].cl[1] = 1.3880000000;	data_table[19].feg[0].cnl[1] = 3.0991000000;	data_table[19].feg[0].cl[2] = 2.1602000000;	data_table[19].feg[0].cnl[2] = 11.9608000000;	data_table[19].feg[0].cl[3] = 3.7532000000;	data_table[19].feg[0].cnl[3] = 53.9353000000;	data_table[19].feg[0].cl[4] = 2.2063000000;	data_table[19].feg[0].cnl[4] = 142.3892000000;	data_table[19].feg[0].cl[5] = 0.0000000000;	data_table[19].feg[0].cnl[5] = 0.0000000000;
				data_table[20].feg[0].cl[0] = 0.3787000000;	data_table[20].feg[0].cnl[0] = 0.3133000000;	data_table[20].feg[0].cl[1] = 1.2181000000;	data_table[20].feg[0].cnl[1] = 2.5856000000;	data_table[20].feg[0].cl[2] = 2.0594000000;	data_table[20].feg[0].cnl[2] = 9.5813000000;	data_table[20].feg[0].cl[3] = 3.2618000000;	data_table[20].feg[0].cnl[3] = 41.7688000000;	data_table[20].feg[0].cl[4] = 2.3870000000;	data_table[20].feg[0].cnl[4] = 116.7282000000;	data_table[20].feg[0].cl[5] = 0.0000000000;	data_table[20].feg[0].cnl[5] = 0.0000000000;
				data_table[21].feg[0].cl[0] = 0.3825000000;	data_table[21].feg[0].cnl[0] = 0.3040000000;	data_table[21].feg[0].cl[1] = 1.2598000000;	data_table[21].feg[0].cnl[1] = 2.4863000000;	data_table[21].feg[0].cl[2] = 2.0008000000;	data_table[21].feg[0].cnl[2] = 9.2783000000;	data_table[21].feg[0].cl[3] = 3.0617000000;	data_table[21].feg[0].cnl[3] = 39.0751000000;	data_table[21].feg[0].cl[4] = 2.0694000000;	data_table[21].feg[0].cnl[4] = 109.4583000000;	data_table[21].feg[0].cl[5] = 0.0000000000;	data_table[21].feg[0].cnl[5] = 0.0000000000;
				data_table[22].feg[0].cl[0] = 0.3876000000;	data_table[22].feg[0].cnl[0] = 0.2967000000;	data_table[22].feg[0].cl[1] = 1.2750000000;	data_table[22].feg[0].cnl[1] = 2.3780000000;	data_table[22].feg[0].cl[2] = 1.9109000000;	data_table[22].feg[0].cnl[2] = 8.7981000000;	data_table[22].feg[0].cl[3] = 2.8314000000;	data_table[22].feg[0].cnl[3] = 35.9528000000;	data_table[22].feg[0].cl[4] = 1.8979000000;	data_table[22].feg[0].cnl[4] = 101.7201000000;	data_table[22].feg[0].cl[5] = 0.0000000000;	data_table[22].feg[0].cnl[5] = 0.0000000000;
				data_table[23].feg[0].cl[0] = 0.4046000000;	data_table[23].feg[0].cnl[0] = 0.2986000000;	data_table[23].feg[0].cl[1] = 1.3696000000;	data_table[23].feg[0].cnl[1] = 2.3958000000;	data_table[23].feg[0].cl[2] = 1.8941000000;	data_table[23].feg[0].cnl[2] = 9.1406000000;	data_table[23].feg[0].cl[3] = 2.0800000000;	data_table[23].feg[0].cnl[3] = 37.4701000000;	data_table[23].feg[0].cl[4] = 1.2196000000;	data_table[23].feg[0].cnl[4] = 113.7121000000;	data_table[23].feg[0].cl[5] = 0.0000000000;	data_table[23].feg[0].cnl[5] = 0.0000000000;
				data_table[24].feg[0].cl[0] = 0.3796000000;	data_table[24].feg[0].cnl[0] = 0.2699000000;	data_table[24].feg[0].cl[1] = 1.2094000000;	data_table[24].feg[0].cnl[1] = 2.0455000000;	data_table[24].feg[0].cl[2] = 1.7815000000;	data_table[24].feg[0].cnl[2] = 7.4726000000;	data_table[24].feg[0].cl[3] = 2.5420000000;	data_table[24].feg[0].cnl[3] = 31.0604000000;	data_table[24].feg[0].cl[4] = 1.5937000000;	data_table[24].feg[0].cnl[4] = 91.5622000000;	data_table[24].feg[0].cl[5] = 0.0000000000;	data_table[24].feg[0].cnl[5] = 0.0000000000;
				data_table[25].feg[0].cl[0] = 0.3946000000;	data_table[25].feg[0].cnl[0] = 0.2717000000;	data_table[25].feg[0].cl[1] = 1.2725000000;	data_table[25].feg[0].cnl[1] = 2.0443000000;	data_table[25].feg[0].cl[2] = 1.7031000000;	data_table[25].feg[0].cnl[2] = 7.6007000000;	data_table[25].feg[0].cl[3] = 2.3140000000;	data_table[25].feg[0].cnl[3] = 29.9714000000;	data_table[25].feg[0].cl[4] = 1.4795000000;	data_table[25].feg[0].cnl[4] = 86.2265000000;	data_table[25].feg[0].cl[5] = 0.0000000000;	data_table[25].feg[0].cnl[5] = 0.0000000000;
				data_table[26].feg[0].cl[0] = 0.4118000000;	data_table[26].feg[0].cnl[0] = 0.2742000000;	data_table[26].feg[0].cl[1] = 1.3161000000;	data_table[26].feg[0].cnl[1] = 2.0372000000;	data_table[26].feg[0].cl[2] = 1.6493000000;	data_table[26].feg[0].cnl[2] = 7.7205000000;	data_table[26].feg[0].cl[3] = 2.1930000000;	data_table[26].feg[0].cnl[3] = 29.9680000000;	data_table[26].feg[0].cl[4] = 1.2830000000;	data_table[26].feg[0].cnl[4] = 84.9383000000;	data_table[26].feg[0].cl[5] = 0.0000000000;	data_table[26].feg[0].cnl[5] = 0.0000000000;
				data_table[27].feg[0].cl[0] = 0.3860000000;	data_table[27].feg[0].cnl[0] = 0.2478000000;	data_table[27].feg[0].cl[1] = 1.1765000000;	data_table[27].feg[0].cnl[1] = 1.7660000000;	data_table[27].feg[0].cl[2] = 1.5451000000;	data_table[27].feg[0].cnl[2] = 6.3107000000;	data_table[27].feg[0].cl[3] = 2.0730000000;	data_table[27].feg[0].cnl[3] = 25.2204000000;	data_table[27].feg[0].cl[4] = 1.3814000000;	data_table[27].feg[0].cnl[4] = 74.3146000000;	data_table[27].feg[0].cl[5] = 0.0000000000;	data_table[27].feg[0].cnl[5] = 0.0000000000;
				data_table[28].feg[0].cl[0] = 0.4314000000;	data_table[28].feg[0].cnl[0] = 0.2694000000;	data_table[28].feg[0].cl[1] = 1.3208000000;	data_table[28].feg[0].cnl[1] = 1.9223000000;	data_table[28].feg[0].cl[2] = 1.5236000000;	data_table[28].feg[0].cnl[2] = 7.3474000000;	data_table[28].feg[0].cl[3] = 1.4671000000;	data_table[28].feg[0].cnl[3] = 28.9892000000;	data_table[28].feg[0].cl[4] = 0.8562000000;	data_table[28].feg[0].cnl[4] = 90.6246000000;	data_table[28].feg[0].cl[5] = 0.0000000000;	data_table[28].feg[0].cnl[5] = 0.0000000000;
				data_table[29].feg[0].cl[0] = 0.4288000000;	data_table[29].feg[0].cnl[0] = 0.2593000000;	data_table[29].feg[0].cl[1] = 1.2646000000;	data_table[29].feg[0].cnl[1] = 1.7998000000;	data_table[29].feg[0].cl[2] = 1.4472000000;	data_table[29].feg[0].cnl[2] = 6.7500000000;	data_table[29].feg[0].cl[3] = 1.8294000000;	data_table[29].feg[0].cnl[3] = 25.5860000000;	data_table[29].feg[0].cl[4] = 1.0934000000;	data_table[29].feg[0].cnl[4] = 73.5284000000;	data_table[29].feg[0].cl[5] = 0.0000000000;	data_table[29].feg[0].cnl[5] = 0.0000000000;
				data_table[30].feg[0].cl[0] = 0.4818000000;	data_table[30].feg[0].cnl[0] = 0.2825000000;	data_table[30].feg[0].cl[1] = 1.4032000000;	data_table[30].feg[0].cnl[1] = 1.9785000000;	data_table[30].feg[0].cl[2] = 1.6561000000;	data_table[30].feg[0].cnl[2] = 8.7546000000;	data_table[30].feg[0].cl[3] = 2.4605000000;	data_table[30].feg[0].cnl[3] = 32.5238000000;	data_table[30].feg[0].cl[4] = 1.1054000000;	data_table[30].feg[0].cnl[4] = 98.5523000000;	data_table[30].feg[0].cl[5] = 0.0000000000;	data_table[30].feg[0].cnl[5] = 0.0000000000;
				data_table[31].feg[0].cl[0] = 0.4655000000;	data_table[31].feg[0].cnl[0] = 0.2647000000;	data_table[31].feg[0].cl[1] = 1.3014000000;	data_table[31].feg[0].cnl[1] = 1.7926000000;	data_table[31].feg[0].cl[2] = 1.6088000000;	data_table[31].feg[0].cnl[2] = 7.6071000000;	data_table[31].feg[0].cl[3] = 2.6998000000;	data_table[31].feg[0].cnl[3] = 26.5541000000;	data_table[31].feg[0].cl[4] = 1.3003000000;	data_table[31].feg[0].cnl[4] = 77.5238000000;	data_table[31].feg[0].cl[5] = 0.0000000000;	data_table[31].feg[0].cnl[5] = 0.0000000000;
				data_table[32].feg[0].cl[0] = 0.4517000000;	data_table[32].feg[0].cnl[0] = 0.2493000000;	data_table[32].feg[0].cl[1] = 1.2229000000;	data_table[32].feg[0].cnl[1] = 1.6436000000;	data_table[32].feg[0].cl[2] = 1.5852000000;	data_table[32].feg[0].cnl[2] = 6.8154000000;	data_table[32].feg[0].cl[3] = 2.7958000000;	data_table[32].feg[0].cnl[3] = 22.3681000000;	data_table[32].feg[0].cl[4] = 1.2638000000;	data_table[32].feg[0].cnl[4] = 62.0390000000;	data_table[32].feg[0].cl[5] = 0.0000000000;	data_table[32].feg[0].cnl[5] = 0.0000000000;
				data_table[33].feg[0].cl[0] = 0.4477000000;	data_table[33].feg[0].cnl[0] = 0.2405000000;	data_table[33].feg[0].cl[1] = 1.1678000000;	data_table[33].feg[0].cnl[1] = 1.5442000000;	data_table[33].feg[0].cl[2] = 1.5843000000;	data_table[33].feg[0].cnl[2] = 6.3231000000;	data_table[33].feg[0].cl[3] = 2.8087000000;	data_table[33].feg[0].cnl[3] = 19.4610000000;	data_table[33].feg[0].cl[4] = 1.1956000000;	data_table[33].feg[0].cnl[4] = 52.0233000000;	data_table[33].feg[0].cl[5] = 0.0000000000;	data_table[33].feg[0].cnl[5] = 0.0000000000;
				data_table[34].feg[0].cl[0] = 0.4798000000;	data_table[34].feg[0].cnl[0] = 0.2504000000;	data_table[34].feg[0].cl[1] = 1.1948000000;	data_table[34].feg[0].cnl[1] = 1.5963000000;	data_table[34].feg[0].cl[2] = 1.8695000000;	data_table[34].feg[0].cnl[2] = 6.9653000000;	data_table[34].feg[0].cl[3] = 2.6953000000;	data_table[34].feg[0].cnl[3] = 19.8492000000;	data_table[34].feg[0].cl[4] = 0.8203000000;	data_table[34].feg[0].cnl[4] = 50.3233000000;	data_table[34].feg[0].cl[5] = 0.0000000000;	data_table[34].feg[0].cnl[5] = 0.0000000000;
				data_table[35].feg[0].cl[0] = 0.4546000000;	data_table[35].feg[0].cnl[0] = 0.2309000000;	data_table[35].feg[0].cl[1] = 1.0993000000;	data_table[35].feg[0].cnl[1] = 1.4279000000;	data_table[35].feg[0].cl[2] = 1.7696000000;	data_table[35].feg[0].cnl[2] = 5.9449000000;	data_table[35].feg[0].cl[3] = 2.7068000000;	data_table[35].feg[0].cnl[3] = 16.6752000000;	data_table[35].feg[0].cl[4] = 0.8672000000;	data_table[35].feg[0].cnl[4] = 42.2243000000;	data_table[35].feg[0].cl[5] = 0.0000000000;	data_table[35].feg[0].cnl[5] = 0.0000000000;
				data_table[36].feg[0].cl[0] = 1.0160000000;	data_table[36].feg[0].cnl[0] = 0.4853000000;	data_table[36].feg[0].cl[1] = 2.8528000000;	data_table[36].feg[0].cnl[1] = 5.0925000000;	data_table[36].feg[0].cl[2] = 3.5466000000;	data_table[36].feg[0].cnl[2] = 25.7851000000;	data_table[36].feg[0].cl[3] = -7.7804000000;	data_table[36].feg[0].cnl[3] = 130.4515000000;	data_table[36].feg[0].cl[4] = 12.1148000000;	data_table[36].feg[0].cnl[4] = 138.6775000000;	data_table[36].feg[0].cl[5] = 0.0000000000;	data_table[36].feg[0].cnl[5] = 0.0000000000;
				data_table[37].feg[0].cl[0] = 0.6703000000;	data_table[37].feg[0].cnl[0] = 0.3190000000;	data_table[37].feg[0].cl[1] = 1.4926000000;	data_table[37].feg[0].cnl[1] = 2.2287000000;	data_table[37].feg[0].cl[2] = 3.3368000000;	data_table[37].feg[0].cnl[2] = 10.3504000000;	data_table[37].feg[0].cl[3] = 4.4600000000;	data_table[37].feg[0].cnl[3] = 52.3291000000;	data_table[37].feg[0].cl[4] = 3.1501000000;	data_table[37].feg[0].cnl[4] = 151.2216000000;	data_table[37].feg[0].cl[5] = 0.0000000000;	data_table[37].feg[0].cnl[5] = 0.0000000000;
				data_table[38].feg[0].cl[0] = 0.6894000000;	data_table[38].feg[0].cnl[0] = 0.3189000000;	data_table[38].feg[0].cl[1] = 1.5474000000;	data_table[38].feg[0].cnl[1] = 2.2904000000;	data_table[38].feg[0].cl[2] = 3.2450000000;	data_table[38].feg[0].cnl[2] = 10.0062000000;	data_table[38].feg[0].cl[3] = 4.2126000000;	data_table[38].feg[0].cnl[3] = 44.0771000000;	data_table[38].feg[0].cl[4] = 2.9764000000;	data_table[38].feg[0].cnl[4] = 125.0120000000;	data_table[38].feg[0].cl[5] = 0.0000000000;	data_table[38].feg[0].cnl[5] = 0.0000000000;
				data_table[39].feg[0].cl[0] = 0.6719000000;	data_table[39].feg[0].cnl[0] = 0.3036000000;	data_table[39].feg[0].cl[1] = 1.4684000000;	data_table[39].feg[0].cnl[1] = 2.1249000000;	data_table[39].feg[0].cl[2] = 3.1668000000;	data_table[39].feg[0].cnl[2] = 8.9236000000;	data_table[39].feg[0].cl[3] = 3.9557000000;	data_table[39].feg[0].cnl[3] = 36.8458000000;	data_table[39].feg[0].cl[4] = 2.8920000000;	data_table[39].feg[0].cnl[4] = 108.2049000000;	data_table[39].feg[0].cl[5] = 0.0000000000;	data_table[39].feg[0].cnl[5] = 0.0000000000;
				data_table[40].feg[0].cl[0] = 0.6123000000;	data_table[40].feg[0].cnl[0] = 0.2709000000;	data_table[40].feg[0].cl[1] = 1.2677000000;	data_table[40].feg[0].cnl[1] = 1.7683000000;	data_table[40].feg[0].cl[2] = 3.0348000000;	data_table[40].feg[0].cnl[2] = 7.2489000000;	data_table[40].feg[0].cl[3] = 3.3841000000;	data_table[40].feg[0].cnl[3] = 27.9465000000;	data_table[40].feg[0].cl[4] = 2.3683000000;	data_table[40].feg[0].cnl[4] = 98.5624000000;	data_table[40].feg[0].cl[5] = 0.0000000000;	data_table[40].feg[0].cnl[5] = 0.0000000000;
				data_table[41].feg[0].cl[0] = 0.6773000000;	data_table[41].feg[0].cnl[0] = 0.2920000000;	data_table[41].feg[0].cl[1] = 1.4798000000;	data_table[41].feg[0].cnl[1] = 2.0606000000;	data_table[41].feg[0].cl[2] = 3.1788000000;	data_table[41].feg[0].cnl[2] = 8.1129000000;	data_table[41].feg[0].cl[3] = 3.0824000000;	data_table[41].feg[0].cnl[3] = 30.5336000000;	data_table[41].feg[0].cl[4] = 1.8384000000;	data_table[41].feg[0].cnl[4] = 100.0658000000;	data_table[41].feg[0].cl[5] = 0.0000000000;	data_table[41].feg[0].cnl[5] = 0.0000000000;
				data_table[42].feg[0].cl[0] = 0.7082000000;	data_table[42].feg[0].cnl[0] = 0.2976000000;	data_table[42].feg[0].cl[1] = 1.6392000000;	data_table[42].feg[0].cnl[1] = 2.2106000000;	data_table[42].feg[0].cl[2] = 3.1993000000;	data_table[42].feg[0].cnl[2] = 8.5246000000;	data_table[42].feg[0].cl[3] = 3.4327000000;	data_table[42].feg[0].cnl[3] = 33.1456000000;	data_table[42].feg[0].cl[4] = 1.8711000000;	data_table[42].feg[0].cnl[4] = 96.6377000000;	data_table[42].feg[0].cl[5] = 0.0000000000;	data_table[42].feg[0].cnl[5] = 0.0000000000;
				data_table[43].feg[0].cl[0] = 0.6735000000;	data_table[43].feg[0].cnl[0] = 0.2773000000;	data_table[43].feg[0].cl[1] = 1.4934000000;	data_table[43].feg[0].cnl[1] = 1.9716000000;	data_table[43].feg[0].cl[2] = 3.0966000000;	data_table[43].feg[0].cnl[2] = 7.3249000000;	data_table[43].feg[0].cl[3] = 2.7254000000;	data_table[43].feg[0].cnl[3] = 26.6891000000;	data_table[43].feg[0].cl[4] = 1.5597000000;	data_table[43].feg[0].cnl[4] = 90.5581000000;	data_table[43].feg[0].cl[5] = 0.0000000000;	data_table[43].feg[0].cnl[5] = 0.0000000000;
				data_table[44].feg[0].cl[0] = 0.6413000000;	data_table[44].feg[0].cnl[0] = 0.2580000000;	data_table[44].feg[0].cl[1] = 1.3690000000;	data_table[44].feg[0].cnl[1] = 1.7721000000;	data_table[44].feg[0].cl[2] = 2.9854000000;	data_table[44].feg[0].cnl[2] = 6.3854000000;	data_table[44].feg[0].cl[3] = 2.6952000000;	data_table[44].feg[0].cnl[3] = 23.2549000000;	data_table[44].feg[0].cl[4] = 1.5433000000;	data_table[44].feg[0].cnl[4] = 85.1517000000;	data_table[44].feg[0].cl[5] = 0.0000000000;	data_table[44].feg[0].cnl[5] = 0.0000000000;
				data_table[45].feg[0].cl[0] = 0.5904000000;	data_table[45].feg[0].cnl[0] = 0.2324000000;	data_table[45].feg[0].cl[1] = 1.1775000000;	data_table[45].feg[0].cnl[1] = 1.5019000000;	data_table[45].feg[0].cl[2] = 2.6519000000;	data_table[45].feg[0].cnl[2] = 5.1591000000;	data_table[45].feg[0].cl[3] = 2.2875000000;	data_table[45].feg[0].cnl[3] = 15.5428000000;	data_table[45].feg[0].cl[4] = 0.8689000000;	data_table[45].feg[0].cnl[4] = 46.8213000000;	data_table[45].feg[0].cl[5] = 0.0000000000;	data_table[45].feg[0].cnl[5] = 0.0000000000;
				data_table[46].feg[0].cl[0] = 0.6377000000;	data_table[46].feg[0].cnl[0] = 0.2466000000;	data_table[46].feg[0].cl[1] = 1.3790000000;	data_table[46].feg[0].cnl[1] = 1.6974000000;	data_table[46].feg[0].cl[2] = 2.8294000000;	data_table[46].feg[0].cnl[2] = 5.7656000000;	data_table[46].feg[0].cl[3] = 2.3631000000;	data_table[46].feg[0].cnl[3] = 20.0943000000;	data_table[46].feg[0].cl[4] = 1.4553000000;	data_table[46].feg[0].cnl[4] = 76.7372000000;	data_table[46].feg[0].cl[5] = 0.0000000000;	data_table[46].feg[0].cnl[5] = 0.0000000000;
				data_table[47].feg[0].cl[0] = 0.6364000000;	data_table[47].feg[0].cnl[0] = 0.2407000000;	data_table[47].feg[0].cl[1] = 1.4247000000;	data_table[47].feg[0].cnl[1] = 1.6823000000;	data_table[47].feg[0].cl[2] = 2.7802000000;	data_table[47].feg[0].cnl[2] = 5.6588000000;	data_table[47].feg[0].cl[3] = 2.5973000000;	data_table[47].feg[0].cnl[3] = 20.7219000000;	data_table[47].feg[0].cl[4] = 1.7886000000;	data_table[47].feg[0].cnl[4] = 69.1109000000;	data_table[47].feg[0].cl[5] = 0.0000000000;	data_table[47].feg[0].cnl[5] = 0.0000000000;
				data_table[48].feg[0].cl[0] = 0.6768000000;	data_table[48].feg[0].cnl[0] = 0.2522000000;	data_table[48].feg[0].cl[1] = 1.6589000000;	data_table[48].feg[0].cnl[1] = 1.8545000000;	data_table[48].feg[0].cl[2] = 2.7740000000;	data_table[48].feg[0].cnl[2] = 6.2936000000;	data_table[48].feg[0].cl[3] = 3.1835000000;	data_table[48].feg[0].cnl[3] = 25.1457000000;	data_table[48].feg[0].cl[4] = 2.1326000000;	data_table[48].feg[0].cnl[4] = 84.5448000000;	data_table[48].feg[0].cl[5] = 0.0000000000;	data_table[48].feg[0].cnl[5] = 0.0000000000;
				data_table[49].feg[0].cl[0] = 0.7224000000;	data_table[49].feg[0].cnl[0] = 0.2651000000;	data_table[49].feg[0].cl[1] = 1.9610000000;	data_table[49].feg[0].cnl[1] = 2.0604000000;	data_table[49].feg[0].cl[2] = 2.7161000000;	data_table[49].feg[0].cnl[2] = 7.3011000000;	data_table[49].feg[0].cl[3] = 3.5603000000;	data_table[49].feg[0].cnl[3] = 27.5493000000;	data_table[49].feg[0].cl[4] = 1.8972000000;	data_table[49].feg[0].cnl[4] = 81.3349000000;	data_table[49].feg[0].cl[5] = 0.0000000000;	data_table[49].feg[0].cnl[5] = 0.0000000000;
				data_table[50].feg[0].cl[0] = 0.7106000000;	data_table[50].feg[0].cnl[0] = 0.2562000000;	data_table[50].feg[0].cl[1] = 1.9247000000;	data_table[50].feg[0].cnl[1] = 1.9646000000;	data_table[50].feg[0].cl[2] = 2.6149000000;	data_table[50].feg[0].cnl[2] = 6.8852000000;	data_table[50].feg[0].cl[3] = 3.8322000000;	data_table[50].feg[0].cnl[3] = 24.7648000000;	data_table[50].feg[0].cl[4] = 1.8899000000;	data_table[50].feg[0].cnl[4] = 68.9168000000;	data_table[50].feg[0].cl[5] = 0.0000000000;	data_table[50].feg[0].cnl[5] = 0.0000000000;
				data_table[51].feg[0].cl[0] = 0.6947000000;	data_table[51].feg[0].cnl[0] = 0.2459000000;	data_table[51].feg[0].cl[1] = 1.8690000000;	data_table[51].feg[0].cnl[1] = 1.8542000000;	data_table[51].feg[0].cl[2] = 2.5356000000;	data_table[51].feg[0].cnl[2] = 6.4411000000;	data_table[51].feg[0].cl[3] = 4.0013000000;	data_table[51].feg[0].cnl[3] = 22.1730000000;	data_table[51].feg[0].cl[4] = 1.8955000000;	data_table[51].feg[0].cnl[4] = 59.2206000000;	data_table[51].feg[0].cl[5] = 0.0000000000;	data_table[51].feg[0].cnl[5] = 0.0000000000;
				data_table[52].feg[0].cl[0] = 0.7047000000;	data_table[52].feg[0].cnl[0] = 0.2455000000;	data_table[52].feg[0].cl[1] = 1.9484000000;	data_table[52].feg[0].cnl[1] = 1.8638000000;	data_table[52].feg[0].cl[2] = 2.5940000000;	data_table[52].feg[0].cnl[2] = 6.7639000000;	data_table[52].feg[0].cl[3] = 4.1526000000;	data_table[52].feg[0].cnl[3] = 21.8007000000;	data_table[52].feg[0].cl[4] = 1.5057000000;	data_table[52].feg[0].cnl[4] = 56.4395000000;	data_table[52].feg[0].cl[5] = 0.0000000000;	data_table[52].feg[0].cnl[5] = 0.0000000000;
				data_table[53].feg[0].cl[0] = 0.6737000000;	data_table[53].feg[0].cnl[0] = 0.2305000000;	data_table[53].feg[0].cl[1] = 1.7908000000;	data_table[53].feg[0].cnl[1] = 1.6890000000;	data_table[53].feg[0].cl[2] = 2.4129000000;	data_table[53].feg[0].cnl[2] = 5.8218000000;	data_table[53].feg[0].cl[3] = 4.2100000000;	data_table[53].feg[0].cnl[3] = 18.3928000000;	data_table[53].feg[0].cl[4] = 1.7058000000;	data_table[53].feg[0].cnl[4] = 47.2496000000;	data_table[53].feg[0].cl[5] = 0.0000000000;	data_table[53].feg[0].cnl[5] = 0.0000000000;
				data_table[54].feg[0].cl[0] = 1.2704000000;	data_table[54].feg[0].cnl[0] = 0.4356000000;	data_table[54].feg[0].cl[1] = 3.8018000000;	data_table[54].feg[0].cnl[1] = 4.2058000000;	data_table[54].feg[0].cl[2] = 5.6618000000;	data_table[54].feg[0].cnl[2] = 23.4342000000;	data_table[54].feg[0].cl[3] = 0.9205000000;	data_table[54].feg[0].cnl[3] = 136.7783000000;	data_table[54].feg[0].cl[4] = 4.8105000000;	data_table[54].feg[0].cnl[4] = 171.7561000000;	data_table[54].feg[0].cl[5] = 0.0000000000;	data_table[54].feg[0].cnl[5] = 0.0000000000;
				data_table[55].feg[0].cl[0] = 0.9049000000;	data_table[55].feg[0].cnl[0] = 0.3066000000;	data_table[55].feg[0].cl[1] = 2.6076000000;	data_table[55].feg[0].cnl[1] = 2.4363000000;	data_table[55].feg[0].cl[2] = 4.8498000000;	data_table[55].feg[0].cnl[2] = 12.1821000000;	data_table[55].feg[0].cl[3] = 5.1603000000;	data_table[55].feg[0].cnl[3] = 54.6135000000;	data_table[55].feg[0].cl[4] = 4.7388000000;	data_table[55].feg[0].cnl[4] = 161.9978000000;	data_table[55].feg[0].cl[5] = 0.0000000000;	data_table[55].feg[0].cnl[5] = 0.0000000000;
				data_table[56].feg[0].cl[0] = 0.8405000000;	data_table[56].feg[0].cnl[0] = 0.2791000000;	data_table[56].feg[0].cl[1] = 2.3863000000;	data_table[56].feg[0].cnl[1] = 2.1410000000;	data_table[56].feg[0].cl[2] = 4.6139000000;	data_table[56].feg[0].cnl[2] = 10.3400000000;	data_table[56].feg[0].cl[3] = 5.1514000000;	data_table[56].feg[0].cnl[3] = 41.9148000000;	data_table[56].feg[0].cl[4] = 4.7949000000;	data_table[56].feg[0].cnl[4] = 132.0204000000;	data_table[56].feg[0].cl[5] = 0.0000000000;	data_table[56].feg[0].cnl[5] = 0.0000000000;
				data_table[57].feg[0].cl[0] = 0.8551000000;	data_table[57].feg[0].cnl[0] = 0.2805000000;	data_table[57].feg[0].cl[1] = 2.3915000000;	data_table[57].feg[0].cnl[1] = 2.1200000000;	data_table[57].feg[0].cl[2] = 4.5772000000;	data_table[57].feg[0].cnl[2] = 10.1808000000;	data_table[57].feg[0].cl[3] = 5.0278000000;	data_table[57].feg[0].cnl[3] = 42.0633000000;	data_table[57].feg[0].cl[4] = 4.5118000000;	data_table[57].feg[0].cnl[4] = 130.9893000000;	data_table[57].feg[0].cl[5] = 0.0000000000;	data_table[57].feg[0].cnl[5] = 0.0000000000;
				data_table[58].feg[0].cl[0] = 0.9096000000;	data_table[58].feg[0].cnl[0] = 0.2939000000;	data_table[58].feg[0].cl[1] = 2.5313000000;	data_table[58].feg[0].cnl[1] = 2.2471000000;	data_table[58].feg[0].cl[2] = 4.5266000000;	data_table[58].feg[0].cnl[2] = 10.8266000000;	data_table[58].feg[0].cl[3] = 4.6376000000;	data_table[58].feg[0].cnl[3] = 48.8842000000;	data_table[58].feg[0].cl[4] = 4.3690000000;	data_table[58].feg[0].cnl[4] = 147.6020000000;	data_table[58].feg[0].cl[5] = 0.0000000000;	data_table[58].feg[0].cnl[5] = 0.0000000000;
				data_table[59].feg[0].cl[0] = 0.8807000000;	data_table[59].feg[0].cnl[0] = 0.2802000000;	data_table[59].feg[0].cl[1] = 2.4183000000;	data_table[59].feg[0].cnl[1] = 2.0836000000;	data_table[59].feg[0].cl[2] = 4.4448000000;	data_table[59].feg[0].cnl[2] = 10.0357000000;	data_table[59].feg[0].cl[3] = 4.6858000000;	data_table[59].feg[0].cnl[3] = 47.4506000000;	data_table[59].feg[0].cl[4] = 4.1725000000;	data_table[59].feg[0].cnl[4] = 146.9976000000;	data_table[59].feg[0].cl[5] = 0.0000000000;	data_table[59].feg[0].cnl[5] = 0.0000000000;
				data_table[60].feg[0].cl[0] = 0.9471000000;	data_table[60].feg[0].cnl[0] = 0.2977000000;	data_table[60].feg[0].cl[1] = 2.5463000000;	data_table[60].feg[0].cnl[1] = 2.2276000000;	data_table[60].feg[0].cl[2] = 4.3523000000;	data_table[60].feg[0].cnl[2] = 10.5762000000;	data_table[60].feg[0].cl[3] = 4.4789000000;	data_table[60].feg[0].cnl[3] = 49.3619000000;	data_table[60].feg[0].cl[4] = 3.9080000000;	data_table[60].feg[0].cnl[4] = 145.3580000000;	data_table[60].feg[0].cl[5] = 0.0000000000;	data_table[60].feg[0].cnl[5] = 0.0000000000;
				data_table[61].feg[0].cl[0] = 0.9699000000;	data_table[61].feg[0].cnl[0] = 0.3003000000;	data_table[61].feg[0].cl[1] = 2.5837000000;	data_table[61].feg[0].cnl[1] = 2.2447000000;	data_table[61].feg[0].cl[2] = 4.2778000000;	data_table[61].feg[0].cnl[2] = 10.6487000000;	data_table[61].feg[0].cl[3] = 4.4575000000;	data_table[61].feg[0].cnl[3] = 50.7994000000;	data_table[61].feg[0].cl[4] = 3.5985000000;	data_table[61].feg[0].cnl[4] = 146.4179000000;	data_table[61].feg[0].cl[5] = 0.0000000000;	data_table[61].feg[0].cnl[5] = 0.0000000000;
				data_table[62].feg[0].cl[0] = 0.8694000000;	data_table[62].feg[0].cnl[0] = 0.2653000000;	data_table[62].feg[0].cl[1] = 2.2413000000;	data_table[62].feg[0].cnl[1] = 1.8590000000;	data_table[62].feg[0].cl[2] = 3.9196000000;	data_table[62].feg[0].cnl[2] = 8.3998000000;	data_table[62].feg[0].cl[3] = 3.9694000000;	data_table[62].feg[0].cnl[3] = 36.7397000000;	data_table[62].feg[0].cl[4] = 4.5498000000;	data_table[62].feg[0].cnl[4] = 125.7089000000;	data_table[62].feg[0].cl[5] = 0.0000000000;	data_table[62].feg[0].cnl[5] = 0.0000000000;
				data_table[63].feg[0].cl[0] = 0.9673000000;	data_table[63].feg[0].cnl[0] = 0.2909000000;	data_table[63].feg[0].cl[1] = 2.4702000000;	data_table[63].feg[0].cnl[1] = 2.1014000000;	data_table[63].feg[0].cl[2] = 4.1148000000;	data_table[63].feg[0].cnl[2] = 9.7067000000;	data_table[63].feg[0].cl[3] = 4.4972000000;	data_table[63].feg[0].cnl[3] = 43.4270000000;	data_table[63].feg[0].cl[4] = 3.2099000000;	data_table[63].feg[0].cnl[4] = 125.9474000000;	data_table[63].feg[0].cl[5] = 0.0000000000;	data_table[63].feg[0].cnl[5] = 0.0000000000;
				data_table[64].feg[0].cl[0] = 0.9325000000;	data_table[64].feg[0].cnl[0] = 0.2761000000;	data_table[64].feg[0].cl[1] = 2.3673000000;	data_table[64].feg[0].cnl[1] = 1.9511000000;	data_table[64].feg[0].cl[2] = 3.8791000000;	data_table[64].feg[0].cnl[2] = 8.9296000000;	data_table[64].feg[0].cl[3] = 3.9674000000;	data_table[64].feg[0].cnl[3] = 41.5937000000;	data_table[64].feg[0].cl[4] = 3.7996000000;	data_table[64].feg[0].cnl[4] = 131.0122000000;	data_table[64].feg[0].cl[5] = 0.0000000000;	data_table[64].feg[0].cnl[5] = 0.0000000000;
				data_table[65].feg[0].cl[0] = 0.9505000000;	data_table[65].feg[0].cnl[0] = 0.2773000000;	data_table[65].feg[0].cl[1] = 2.3705000000;	data_table[65].feg[0].cnl[1] = 1.9469000000;	data_table[65].feg[0].cl[2] = 3.8218000000;	data_table[65].feg[0].cnl[2] = 8.8862000000;	data_table[65].feg[0].cl[3] = 4.0471000000;	data_table[65].feg[0].cnl[3] = 43.0938000000;	data_table[65].feg[0].cl[4] = 3.4451000000;	data_table[65].feg[0].cnl[4] = 133.1396000000;	data_table[65].feg[0].cl[5] = 0.0000000000;	data_table[65].feg[0].cnl[5] = 0.0000000000;
				data_table[66].feg[0].cl[0] = 0.9248000000;	data_table[66].feg[0].cnl[0] = 0.2660000000;	data_table[66].feg[0].cl[1] = 2.2428000000;	data_table[66].feg[0].cnl[1] = 1.8183000000;	data_table[66].feg[0].cl[2] = 3.6182000000;	data_table[66].feg[0].cnl[2] = 7.9655000000;	data_table[66].feg[0].cl[3] = 3.7910000000;	data_table[66].feg[0].cnl[3] = 33.1129000000;	data_table[66].feg[0].cl[4] = 3.7912000000;	data_table[66].feg[0].cnl[4] = 101.8139000000;	data_table[66].feg[0].cl[5] = 0.0000000000;	data_table[66].feg[0].cnl[5] = 0.0000000000;
				data_table[67].feg[0].cl[0] = 1.0373000000;	data_table[67].feg[0].cnl[0] = 0.2944000000;	data_table[67].feg[0].cl[1] = 2.4824000000;	data_table[67].feg[0].cnl[1] = 2.0797000000;	data_table[67].feg[0].cl[2] = 3.6558000000;	data_table[67].feg[0].cnl[2] = 9.4156000000;	data_table[67].feg[0].cl[3] = 3.8925000000;	data_table[67].feg[0].cnl[3] = 45.8056000000;	data_table[67].feg[0].cl[4] = 3.0056000000;	data_table[67].feg[0].cnl[4] = 132.7720000000;	data_table[67].feg[0].cl[5] = 0.0000000000;	data_table[67].feg[0].cnl[5] = 0.0000000000;
				data_table[68].feg[0].cl[0] = 1.0075000000;	data_table[68].feg[0].cnl[0] = 0.2816000000;	data_table[68].feg[0].cl[1] = 2.3787000000;	data_table[68].feg[0].cnl[1] = 1.9486000000;	data_table[68].feg[0].cl[2] = 3.5440000000;	data_table[68].feg[0].cnl[2] = 8.7162000000;	data_table[68].feg[0].cl[3] = 3.6932000000;	data_table[68].feg[0].cnl[3] = 41.8420000000;	data_table[68].feg[0].cl[4] = 3.1759000000;	data_table[68].feg[0].cnl[4] = 125.0320000000;	data_table[68].feg[0].cl[5] = 0.0000000000;	data_table[68].feg[0].cnl[5] = 0.0000000000;
				data_table[69].feg[0].cl[0] = 1.0347000000;	data_table[69].feg[0].cnl[0] = 0.2855000000;	data_table[69].feg[0].cl[1] = 2.3911000000;	data_table[69].feg[0].cnl[1] = 1.9679000000;	data_table[69].feg[0].cl[2] = 3.4619000000;	data_table[69].feg[0].cnl[2] = 8.7619000000;	data_table[69].feg[0].cl[3] = 3.6556000000;	data_table[69].feg[0].cnl[3] = 42.3304000000;	data_table[69].feg[0].cl[4] = 3.0052000000;	data_table[69].feg[0].cnl[4] = 125.6499000000;	data_table[69].feg[0].cl[5] = 0.0000000000;	data_table[69].feg[0].cnl[5] = 0.0000000000;
				data_table[70].feg[0].cl[0] = 0.9927000000;	data_table[70].feg[0].cnl[0] = 0.2701000000;	data_table[70].feg[0].cl[1] = 2.2436000000;	data_table[70].feg[0].cnl[1] = 1.8073000000;	data_table[70].feg[0].cl[2] = 3.3554000000;	data_table[70].feg[0].cnl[2] = 7.8112000000;	data_table[70].feg[0].cl[3] = 3.7813000000;	data_table[70].feg[0].cnl[3] = 34.4849000000;	data_table[70].feg[0].cl[4] = 3.0994000000;	data_table[70].feg[0].cnl[4] = 103.3526000000;	data_table[70].feg[0].cl[5] = 0.0000000000;	data_table[70].feg[0].cnl[5] = 0.0000000000;
				data_table[71].feg[0].cl[0] = 1.0295000000;	data_table[71].feg[0].cnl[0] = 0.2761000000;	data_table[71].feg[0].cl[1] = 2.2911000000;	data_table[71].feg[0].cnl[1] = 1.8625000000;	data_table[71].feg[0].cl[2] = 3.4110000000;	data_table[71].feg[0].cnl[2] = 8.0961000000;	data_table[71].feg[0].cl[3] = 3.9497000000;	data_table[71].feg[0].cnl[3] = 34.2712000000;	data_table[71].feg[0].cl[4] = 2.4925000000;	data_table[71].feg[0].cnl[4] = 98.5295000000;	data_table[71].feg[0].cl[5] = 0.0000000000;	data_table[71].feg[0].cnl[5] = 0.0000000000;
				data_table[72].feg[0].cl[0] = 1.0190000000;	data_table[72].feg[0].cnl[0] = 0.2694000000;	data_table[72].feg[0].cl[1] = 2.2291000000;	data_table[72].feg[0].cnl[1] = 1.7962000000;	data_table[72].feg[0].cl[2] = 3.4097000000;	data_table[72].feg[0].cnl[2] = 7.6944000000;	data_table[72].feg[0].cl[3] = 3.9252000000;	data_table[72].feg[0].cnl[3] = 31.0942000000;	data_table[72].feg[0].cl[4] = 2.2679000000;	data_table[72].feg[0].cnl[4] = 91.1089000000;	data_table[72].feg[0].cl[5] = 0.0000000000;	data_table[72].feg[0].cnl[5] = 0.0000000000;
				data_table[73].feg[0].cl[0] = 0.9853000000;	data_table[73].feg[0].cnl[0] = 0.2569000000;	data_table[73].feg[0].cl[1] = 2.1167000000;	data_table[73].feg[0].cnl[1] = 1.6745000000;	data_table[73].feg[0].cl[2] = 3.3570000000;	data_table[73].feg[0].cnl[2] = 7.0098000000;	data_table[73].feg[0].cl[3] = 3.7981000000;	data_table[73].feg[0].cnl[3] = 26.9234000000;	data_table[73].feg[0].cl[4] = 2.2798000000;	data_table[73].feg[0].cnl[4] = 81.3910000000;	data_table[73].feg[0].cl[5] = 0.0000000000;	data_table[73].feg[0].cnl[5] = 0.0000000000;
				data_table[74].feg[0].cl[0] = 0.9914000000;	data_table[74].feg[0].cnl[0] = 0.2548000000;	data_table[74].feg[0].cl[1] = 2.0858000000;	data_table[74].feg[0].cnl[1] = 1.6518000000;	data_table[74].feg[0].cl[2] = 3.4531000000;	data_table[74].feg[0].cnl[2] = 6.8845000000;	data_table[74].feg[0].cl[3] = 3.8812000000;	data_table[74].feg[0].cnl[3] = 26.7234000000;	data_table[74].feg[0].cl[4] = 1.8526000000;	data_table[74].feg[0].cnl[4] = 81.7215000000;	data_table[74].feg[0].cl[5] = 0.0000000000;	data_table[74].feg[0].cnl[5] = 0.0000000000;
				data_table[75].feg[0].cl[0] = 0.9813000000;	data_table[75].feg[0].cnl[0] = 0.2487000000;	data_table[75].feg[0].cl[1] = 2.0322000000;	data_table[75].feg[0].cnl[1] = 1.5973000000;	data_table[75].feg[0].cl[2] = 3.3665000000;	data_table[75].feg[0].cnl[2] = 6.4737000000;	data_table[75].feg[0].cl[3] = 3.6235000000;	data_table[75].feg[0].cnl[3] = 23.2817000000;	data_table[75].feg[0].cl[4] = 1.9741000000;	data_table[75].feg[0].cnl[4] = 70.9254000000;	data_table[75].feg[0].cl[5] = 0.0000000000;	data_table[75].feg[0].cnl[5] = 0.0000000000;
				data_table[76].feg[0].cl[0] = 1.0194000000;	data_table[76].feg[0].cnl[0] = 0.2554000000;	data_table[76].feg[0].cl[1] = 2.0645000000;	data_table[76].feg[0].cnl[1] = 1.6475000000;	data_table[76].feg[0].cl[2] = 3.4425000000;	data_table[76].feg[0].cnl[2] = 6.5966000000;	data_table[76].feg[0].cl[3] = 3.4914000000;	data_table[76].feg[0].cnl[3] = 23.2269000000;	data_table[76].feg[0].cl[4] = 1.6976000000;	data_table[76].feg[0].cnl[4] = 70.0272000000;	data_table[76].feg[0].cl[5] = 0.0000000000;	data_table[76].feg[0].cnl[5] = 0.0000000000;
				data_table[77].feg[0].cl[0] = 0.9148000000;	data_table[77].feg[0].cnl[0] = 0.2263000000;	data_table[77].feg[0].cl[1] = 1.8096000000;	data_table[77].feg[0].cnl[1] = 1.3813000000;	data_table[77].feg[0].cl[2] = 3.2134000000;	data_table[77].feg[0].cnl[2] = 5.3243000000;	data_table[77].feg[0].cl[3] = 3.2953000000;	data_table[77].feg[0].cnl[3] = 17.5987000000;	data_table[77].feg[0].cl[4] = 1.5754000000;	data_table[77].feg[0].cnl[4] = 60.0171000000;	data_table[77].feg[0].cl[5] = 0.0000000000;	data_table[77].feg[0].cnl[5] = 0.0000000000;
				data_table[78].feg[0].cl[0] = 0.9674000000;	data_table[78].feg[0].cnl[0] = 0.2358000000;	data_table[78].feg[0].cl[1] = 1.8916000000;	data_table[78].feg[0].cnl[1] = 1.4712000000;	data_table[78].feg[0].cl[2] = 3.3993000000;	data_table[78].feg[0].cnl[2] = 5.6758000000;	data_table[78].feg[0].cl[3] = 3.0524000000;	data_table[78].feg[0].cnl[3] = 18.7119000000;	data_table[78].feg[0].cl[4] = 1.2607000000;	data_table[78].feg[0].cnl[4] = 61.5286000000;	data_table[78].feg[0].cl[5] = 0.0000000000;	data_table[78].feg[0].cnl[5] = 0.0000000000;
				data_table[79].feg[0].cl[0] = 1.0033000000;	data_table[79].feg[0].cnl[0] = 0.2413000000;	data_table[79].feg[0].cl[1] = 1.9469000000;	data_table[79].feg[0].cnl[1] = 1.5298000000;	data_table[79].feg[0].cl[2] = 3.4396000000;	data_table[79].feg[0].cnl[2] = 5.8009000000;	data_table[79].feg[0].cl[3] = 3.1548000000;	data_table[79].feg[0].cnl[3] = 19.4520000000;	data_table[79].feg[0].cl[4] = 1.4180000000;	data_table[79].feg[0].cnl[4] = 60.5753000000;	data_table[79].feg[0].cl[5] = 0.0000000000;	data_table[79].feg[0].cnl[5] = 0.0000000000;
				data_table[80].feg[0].cl[0] = 1.0689000000;	data_table[80].feg[0].cnl[0] = 0.2540000000;	data_table[80].feg[0].cl[1] = 2.1038000000;	data_table[80].feg[0].cnl[1] = 1.6715000000;	data_table[80].feg[0].cl[2] = 3.6039000000;	data_table[80].feg[0].cnl[2] = 6.3509000000;	data_table[80].feg[0].cl[3] = 3.4927000000;	data_table[80].feg[0].cnl[3] = 23.1531000000;	data_table[80].feg[0].cl[4] = 1.8283000000;	data_table[80].feg[0].cnl[4] = 78.7099000000;	data_table[80].feg[0].cl[5] = 0.0000000000;	data_table[80].feg[0].cnl[5] = 0.0000000000;
				data_table[81].feg[0].cl[0] = 1.0891000000;	data_table[81].feg[0].cnl[0] = 0.2552000000;	data_table[81].feg[0].cl[1] = 2.1867000000;	data_table[81].feg[0].cnl[1] = 1.7174000000;	data_table[81].feg[0].cl[2] = 3.6160000000;	data_table[81].feg[0].cnl[2] = 6.5131000000;	data_table[81].feg[0].cl[3] = 3.8031000000;	data_table[81].feg[0].cnl[3] = 23.9170000000;	data_table[81].feg[0].cl[4] = 1.8994000000;	data_table[81].feg[0].cnl[4] = 74.7039000000;	data_table[81].feg[0].cl[5] = 0.0000000000;	data_table[81].feg[0].cnl[5] = 0.0000000000;
				data_table[82].feg[0].cl[0] = 1.1007000000;	data_table[82].feg[0].cnl[0] = 0.2546000000;	data_table[82].feg[0].cl[1] = 2.2306000000;	data_table[82].feg[0].cnl[1] = 1.7351000000;	data_table[82].feg[0].cl[2] = 3.5689000000;	data_table[82].feg[0].cnl[2] = 6.4948000000;	data_table[82].feg[0].cl[3] = 4.1549000000;	data_table[82].feg[0].cnl[3] = 23.6464000000;	data_table[82].feg[0].cl[4] = 2.0382000000;	data_table[82].feg[0].cnl[4] = 70.3780000000;	data_table[82].feg[0].cl[5] = 0.0000000000;	data_table[82].feg[0].cnl[5] = 0.0000000000;
				data_table[83].feg[0].cl[0] = 1.1568000000;	data_table[83].feg[0].cnl[0] = 0.2648000000;	data_table[83].feg[0].cl[1] = 2.4353000000;	data_table[83].feg[0].cnl[1] = 1.8786000000;	data_table[83].feg[0].cl[2] = 3.6459000000;	data_table[83].feg[0].cnl[2] = 7.1749000000;	data_table[83].feg[0].cl[3] = 4.4064000000;	data_table[83].feg[0].cnl[3] = 25.1766000000;	data_table[83].feg[0].cl[4] = 1.7179000000;	data_table[83].feg[0].cnl[4] = 69.2821000000;	data_table[83].feg[0].cl[5] = 0.0000000000;	data_table[83].feg[0].cnl[5] = 0.0000000000;
				data_table[84].feg[0].cl[0] = 1.0909000000;	data_table[84].feg[0].cnl[0] = 0.2466000000;	data_table[84].feg[0].cl[1] = 2.1976000000;	data_table[84].feg[0].cnl[1] = 1.6707000000;	data_table[84].feg[0].cl[2] = 3.3831000000;	data_table[84].feg[0].cnl[2] = 6.0197000000;	data_table[84].feg[0].cl[3] = 4.6700000000;	data_table[84].feg[0].cnl[3] = 20.7657000000;	data_table[84].feg[0].cl[4] = 2.1277000000;	data_table[84].feg[0].cnl[4] = 57.2663000000;	data_table[84].feg[0].cl[5] = 0.0000000000;	data_table[84].feg[0].cnl[5] = 0.0000000000;
				data_table[85].feg[0].cl[0] = 1.0756000000;	data_table[85].feg[0].cnl[0] = 0.2402000000;	data_table[85].feg[0].cl[1] = 2.1630000000;	data_table[85].feg[0].cnl[1] = 1.6169000000;	data_table[85].feg[0].cl[2] = 3.3178000000;	data_table[85].feg[0].cnl[2] = 5.7644000000;	data_table[85].feg[0].cl[3] = 4.8852000000;	data_table[85].feg[0].cnl[3] = 19.4568000000;	data_table[85].feg[0].cl[4] = 2.0489000000;	data_table[85].feg[0].cnl[4] = 52.5009000000;	data_table[85].feg[0].cl[5] = 0.0000000000;	data_table[85].feg[0].cnl[5] = 0.0000000000;
				data_table[86].feg[0].cl[0] = 1.4282000000;	data_table[86].feg[0].cnl[0] = 0.3183000000;	data_table[86].feg[0].cl[1] = 3.5081000000;	data_table[86].feg[0].cnl[1] = 2.6889000000;	data_table[86].feg[0].cl[2] = 5.6767000000;	data_table[86].feg[0].cnl[2] = 13.4816000000;	data_table[86].feg[0].cl[3] = 4.1964000000;	data_table[86].feg[0].cnl[3] = 54.3866000000;	data_table[86].feg[0].cl[4] = 3.8946000000;	data_table[86].feg[0].cnl[4] = 200.8321000000;	data_table[86].feg[0].cl[5] = 0.0000000000;	data_table[86].feg[0].cnl[5] = 0.0000000000;
				data_table[87].feg[0].cl[0] = 1.3127000000;	data_table[87].feg[0].cnl[0] = 0.2887000000;	data_table[87].feg[0].cl[1] = 3.1243000000;	data_table[87].feg[0].cnl[1] = 2.2897000000;	data_table[87].feg[0].cl[2] = 5.2988000000;	data_table[87].feg[0].cnl[2] = 10.8276000000;	data_table[87].feg[0].cl[3] = 5.3891000000;	data_table[87].feg[0].cnl[3] = 43.5389000000;	data_table[87].feg[0].cl[4] = 5.4133000000;	data_table[87].feg[0].cnl[4] = 145.6109000000;	data_table[87].feg[0].cl[5] = 0.0000000000;	data_table[87].feg[0].cnl[5] = 0.0000000000;
				data_table[88].feg[0].cl[0] = 1.3128000000;	data_table[88].feg[0].cnl[0] = 0.2861000000;	data_table[88].feg[0].cl[1] = 3.1021000000;	data_table[88].feg[0].cnl[1] = 2.2509000000;	data_table[88].feg[0].cl[2] = 5.3385000000;	data_table[88].feg[0].cnl[2] = 10.5287000000;	data_table[88].feg[0].cl[3] = 5.9611000000;	data_table[88].feg[0].cnl[3] = 41.7796000000;	data_table[88].feg[0].cl[4] = 4.7562000000;	data_table[88].feg[0].cnl[4] = 128.2973000000;	data_table[88].feg[0].cl[5] = 0.0000000000;	data_table[88].feg[0].cnl[5] = 0.0000000000;
				data_table[89].feg[0].cl[0] = 1.2553000000;	data_table[89].feg[0].cnl[0] = 0.2701000000;	data_table[89].feg[0].cl[1] = 2.9178000000;	data_table[89].feg[0].cnl[1] = 2.0636000000;	data_table[89].feg[0].cl[2] = 5.0862000000;	data_table[89].feg[0].cnl[2] = 9.3051000000;	data_table[89].feg[0].cl[3] = 6.1206000000;	data_table[89].feg[0].cnl[3] = 34.5977000000;	data_table[89].feg[0].cl[4] = 4.7122000000;	data_table[89].feg[0].cnl[4] = 107.9200000000;	data_table[89].feg[0].cl[5] = 0.0000000000;	data_table[89].feg[0].cnl[5] = 0.0000000000;
				data_table[90].feg[0].cl[0] = 1.3218000000;	data_table[90].feg[0].cnl[0] = 0.2827000000;	data_table[90].feg[0].cl[1] = 3.1444000000;	data_table[90].feg[0].cnl[1] = 2.2250000000;	data_table[90].feg[0].cl[2] = 5.4371000000;	data_table[90].feg[0].cnl[2] = 10.2454000000;	data_table[90].feg[0].cl[3] = 5.6444000000;	data_table[90].feg[0].cnl[3] = 41.1162000000;	data_table[90].feg[0].cl[4] = 4.0107000000;	data_table[90].feg[0].cnl[4] = 124.4449000000;	data_table[90].feg[0].cl[5] = 0.0000000000;	data_table[90].feg[0].cnl[5] = 0.0000000000;
				data_table[91].feg[0].cl[0] = 1.3382000000;	data_table[91].feg[0].cnl[0] = 0.2838000000;	data_table[91].feg[0].cl[1] = 3.2043000000;	data_table[91].feg[0].cnl[1] = 2.2452000000;	data_table[91].feg[0].cl[2] = 5.4558000000;	data_table[91].feg[0].cnl[2] = 10.2519000000;	data_table[91].feg[0].cl[3] = 5.4839000000;	data_table[91].feg[0].cnl[3] = 41.7251000000;	data_table[91].feg[0].cl[4] = 3.6342000000;	data_table[91].feg[0].cnl[4] = 124.9023000000;	data_table[91].feg[0].cl[5] = 0.0000000000;	data_table[91].feg[0].cnl[5] = 0.0000000000;
				data_table[92].feg[0].cl[0] = 1.5193000000;	data_table[92].feg[0].cnl[0] = 0.3213000000;	data_table[92].feg[0].cl[1] = 4.0053000000;	data_table[92].feg[0].cnl[1] = 2.8206000000;	data_table[92].feg[0].cl[2] = 6.5327000000;	data_table[92].feg[0].cnl[2] = 14.8878000000;	data_table[92].feg[0].cl[3] = -0.1402000000;	data_table[92].feg[0].cnl[3] = 68.9103000000;	data_table[92].feg[0].cl[4] = 6.7489000000;	data_table[92].feg[0].cnl[4] = 81.7257000000;	data_table[92].feg[0].cl[5] = 0.0000000000;	data_table[92].feg[0].cnl[5] = 0.0000000000;
				data_table[93].feg[0].cl[0] = 1.3517000000;	data_table[93].feg[0].cnl[0] = 0.2813000000;	data_table[93].feg[0].cl[1] = 3.2937000000;	data_table[93].feg[0].cnl[1] = 2.2418000000;	data_table[93].feg[0].cl[2] = 5.3213000000;	data_table[93].feg[0].cnl[2] = 9.9952000000;	data_table[93].feg[0].cl[3] = 4.6466000000;	data_table[93].feg[0].cnl[3] = 42.7939000000;	data_table[93].feg[0].cl[4] = 3.5714000000;	data_table[93].feg[0].cnl[4] = 132.1739000000;	data_table[93].feg[0].cl[5] = 0.0000000000;	data_table[93].feg[0].cnl[5] = 0.0000000000;
				data_table[94].feg[0].cl[0] = 1.2135000000;	data_table[94].feg[0].cnl[0] = 0.2483000000;	data_table[94].feg[0].cl[1] = 2.7962000000;	data_table[94].feg[0].cnl[1] = 1.8437000000;	data_table[94].feg[0].cl[2] = 4.7545000000;	data_table[94].feg[0].cnl[2] = 7.5421000000;	data_table[94].feg[0].cl[3] = 4.5731000000;	data_table[94].feg[0].cnl[3] = 29.3841000000;	data_table[94].feg[0].cl[4] = 4.4786000000;	data_table[94].feg[0].cnl[4] = 112.4579000000;	data_table[94].feg[0].cl[5] = 0.0000000000;	data_table[94].feg[0].cnl[5] = 0.0000000000;
				data_table[95].feg[0].cl[0] = 1.2937000000;	data_table[95].feg[0].cnl[0] = 0.2638000000;	data_table[95].feg[0].cl[1] = 3.1100000000;	data_table[95].feg[0].cnl[1] = 2.0341000000;	data_table[95].feg[0].cl[2] = 5.0393000000;	data_table[95].feg[0].cnl[2] = 8.7101000000;	data_table[95].feg[0].cl[3] = 4.7546000000;	data_table[95].feg[0].cnl[3] = 35.2992000000;	data_table[95].feg[0].cl[4] = 3.5031000000;	data_table[95].feg[0].cnl[4] = 109.4972000000;	data_table[95].feg[0].cl[5] = 0.0000000000;	data_table[95].feg[0].cnl[5] = 0.0000000000;
				data_table[96].feg[0].cl[0] = 1.2915000000;	data_table[96].feg[0].cnl[0] = 0.2611000000;	data_table[96].feg[0].cl[1] = 3.1023000000;	data_table[96].feg[0].cnl[1] = 2.0023000000;	data_table[96].feg[0].cl[2] = 4.9309000000;	data_table[96].feg[0].cnl[2] = 8.4377000000;	data_table[96].feg[0].cl[3] = 4.6009000000;	data_table[96].feg[0].cnl[3] = 34.1559000000;	data_table[96].feg[0].cl[4] = 3.4661000000;	data_table[96].feg[0].cnl[4] = 105.8911000000;	data_table[96].feg[0].cl[5] = 0.0000000000;	data_table[96].feg[0].cnl[5] = 0.0000000000;
				data_table[97].feg[0].cl[0] = 1.2089000000;	data_table[97].feg[0].cnl[0] = 0.2421000000;	data_table[97].feg[0].cl[1] = 2.7391000000;	data_table[97].feg[0].cnl[1] = 1.7487000000;	data_table[97].feg[0].cl[2] = 4.3482000000;	data_table[97].feg[0].cnl[2] = 6.7262000000;	data_table[97].feg[0].cl[3] = 4.0047000000;	data_table[97].feg[0].cnl[3] = 23.2153000000;	data_table[97].feg[0].cl[4] = 4.6497000000;	data_table[97].feg[0].cnl[4] = 80.3108000000;	data_table[97].feg[0].cl[5] = 0.0000000000;	data_table[97].feg[0].cnl[5] = 0.0000000000;
				data_table[98].feg[0].cl[0] = 0.0000000000;	data_table[98].feg[0].cnl[0] = 0.0000000000;	data_table[98].feg[0].cl[1] = 0.0000000000;	data_table[98].feg[0].cnl[1] = 0.0000000000;	data_table[98].feg[0].cl[2] = 0.0000000000;	data_table[98].feg[0].cnl[2] = 0.0000000000;	data_table[98].feg[0].cl[3] = 0.0000000000;	data_table[98].feg[0].cnl[3] = 0.0000000000;	data_table[98].feg[0].cl[4] = 0.0000000000;	data_table[98].feg[0].cnl[4] = 0.0000000000;	data_table[98].feg[0].cl[5] = 0.0000000000;	data_table[98].feg[0].cnl[5] = 0.0000000000;
				data_table[99].feg[0].cl[0] = 0.0000000000;	data_table[99].feg[0].cnl[0] = 0.0000000000;	data_table[99].feg[0].cl[1] = 0.0000000000;	data_table[99].feg[0].cnl[1] = 0.0000000000;	data_table[99].feg[0].cl[2] = 0.0000000000;	data_table[99].feg[0].cnl[2] = 0.0000000000;	data_table[99].feg[0].cl[3] = 0.0000000000;	data_table[99].feg[0].cnl[3] = 0.0000000000;	data_table[99].feg[0].cl[4] = 0.0000000000;	data_table[99].feg[0].cnl[4] = 0.0000000000;	data_table[99].feg[0].cl[5] = 0.0000000000;	data_table[99].feg[0].cnl[5] = 0.0000000000;
				data_table[100].feg[0].cl[0] = 0.0000000000;	data_table[100].feg[0].cnl[0] = 0.0000000000;	data_table[100].feg[0].cl[1] = 0.0000000000;	data_table[100].feg[0].cnl[1] = 0.0000000000;	data_table[100].feg[0].cl[2] = 0.0000000000;	data_table[100].feg[0].cnl[2] = 0.0000000000;	data_table[100].feg[0].cl[3] = 0.0000000000;	data_table[100].feg[0].cnl[3] = 0.0000000000;	data_table[100].feg[0].cl[4] = 0.0000000000;	data_table[100].feg[0].cnl[4] = 0.0000000000;	data_table[100].feg[0].cl[5] = 0.0000000000;	data_table[100].feg[0].cnl[5] = 0.0000000000;
				data_table[101].feg[0].cl[0] = 0.0000000000;	data_table[101].feg[0].cnl[0] = 0.0000000000;	data_table[101].feg[0].cl[1] = 0.0000000000;	data_table[101].feg[0].cnl[1] = 0.0000000000;	data_table[101].feg[0].cl[2] = 0.0000000000;	data_table[101].feg[0].cnl[2] = 0.0000000000;	data_table[101].feg[0].cl[3] = 0.0000000000;	data_table[101].feg[0].cnl[3] = 0.0000000000;	data_table[101].feg[0].cl[4] = 0.0000000000;	data_table[101].feg[0].cnl[4] = 0.0000000000;	data_table[101].feg[0].cl[5] = 0.0000000000;	data_table[101].feg[0].cnl[5] = 0.0000000000;
				data_table[102].feg[0].cl[0] = 0.0000000000;	data_table[102].feg[0].cnl[0] = 0.0000000000;	data_table[102].feg[0].cl[1] = 0.0000000000;	data_table[102].feg[0].cnl[1] = 0.0000000000;	data_table[102].feg[0].cl[2] = 0.0000000000;	data_table[102].feg[0].cnl[2] = 0.0000000000;	data_table[102].feg[0].cl[3] = 0.0000000000;	data_table[102].feg[0].cnl[3] = 0.0000000000;	data_table[102].feg[0].cl[4] = 0.0000000000;	data_table[102].feg[0].cnl[4] = 0.0000000000;	data_table[102].feg[0].cl[5] = 0.0000000000;	data_table[102].feg[0].cnl[5] = 0.0000000000;
			}

			// 3: Peng et al. parameterization - 5 Gaussians - [0, 12]
			void Load_feg_Peng_neutral_0_12()
			{
				data_table[0].feg[0].cl[0] = 0.0088000000;		data_table[0].feg[0].cnl[0] = 0.1152000000;	data_table[0].feg[0].cl[1] = 0.0449000000;		data_table[0].feg[0].cnl[1] = 1.0867000000;	data_table[0].feg[0].cl[2] = 0.1481000000;		data_table[0].feg[0].cnl[2] = 4.9755000000;	data_table[0].feg[0].cl[3] = 0.2356000000;		data_table[0].feg[0].cnl[3] = 16.5591000000;	data_table[0].feg[0].cl[4] = 0.0914000000;		data_table[0].feg[0].cnl[4] = 43.2743000000;	data_table[0].feg[0].cl[5] = 0.0000000000;		data_table[0].feg[0].cnl[5] = 0.0000000000;
				data_table[1].feg[0].cl[0] = 0.0084000000;		data_table[1].feg[0].cnl[0] = 0.0596000000;	data_table[1].feg[0].cl[1] = 0.0443000000;		data_table[1].feg[0].cnl[1] = 0.5360000000;	data_table[1].feg[0].cl[2] = 0.1314000000;		data_table[1].feg[0].cnl[2] = 2.4274000000;	data_table[1].feg[0].cl[3] = 0.1671000000;		data_table[1].feg[0].cnl[3] = 7.7852000000;	data_table[1].feg[0].cl[4] = 0.0666000000;		data_table[1].feg[0].cnl[4] = 20.3126000000;	data_table[1].feg[0].cl[5] = 0.0000000000;		data_table[1].feg[0].cnl[5] = 0.0000000000;
				data_table[2].feg[0].cl[0] = 0.0478000000;		data_table[2].feg[0].cnl[0] = 0.2258000000;	data_table[2].feg[0].cl[1] = 0.2048000000;		data_table[2].feg[0].cnl[1] = 2.1032000000;	data_table[2].feg[0].cl[2] = 0.5253000000;		data_table[2].feg[0].cnl[2] = 12.9349000000;	data_table[2].feg[0].cl[3] = 1.5225000000;		data_table[2].feg[0].cnl[3] = 50.7501000000;	data_table[2].feg[0].cl[4] = 0.9853000000;		data_table[2].feg[0].cnl[4] = 136.6280000000;	data_table[2].feg[0].cl[5] = 0.0000000000;		data_table[2].feg[0].cnl[5] = 0.0000000000;
				data_table[3].feg[0].cl[0] = 0.0423000000;		data_table[3].feg[0].cnl[0] = 0.1445000000;	data_table[3].feg[0].cl[1] = 0.1874000000;		data_table[3].feg[0].cnl[1] = 1.4180000000;	data_table[3].feg[0].cl[2] = 0.6019000000;		data_table[3].feg[0].cnl[2] = 8.1165000000;	data_table[3].feg[0].cl[3] = 1.4311000000;		data_table[3].feg[0].cnl[3] = 27.9705000000;	data_table[3].feg[0].cl[4] = 0.7891000000;		data_table[3].feg[0].cnl[4] = 74.8684000000;	data_table[3].feg[0].cl[5] = 0.0000000000;		data_table[3].feg[0].cnl[5] = 0.0000000000;
				data_table[4].feg[0].cl[0] = 0.0436000000;		data_table[4].feg[0].cnl[0] = 0.1207000000;	data_table[4].feg[0].cl[1] = 0.1898000000;		data_table[4].feg[0].cnl[1] = 1.1595000000;	data_table[4].feg[0].cl[2] = 0.6788000000;		data_table[4].feg[0].cnl[2] = 6.2474000000;	data_table[4].feg[0].cl[3] = 1.3273000000;		data_table[4].feg[0].cnl[3] = 21.0460000000;	data_table[4].feg[0].cl[4] = 0.5544000000;		data_table[4].feg[0].cnl[4] = 59.3619000000;	data_table[4].feg[0].cl[5] = 0.0000000000;		data_table[4].feg[0].cnl[5] = 0.0000000000;
				data_table[5].feg[0].cl[0] = 0.0489000000;		data_table[5].feg[0].cnl[0] = 0.1140000000;	data_table[5].feg[0].cl[1] = 0.2091000000;		data_table[5].feg[0].cnl[1] = 1.0825000000;	data_table[5].feg[0].cl[2] = 0.7537000000;		data_table[5].feg[0].cnl[2] = 5.4281000000;	data_table[5].feg[0].cl[3] = 1.1420000000;		data_table[5].feg[0].cnl[3] = 17.8811000000;	data_table[5].feg[0].cl[4] = 0.3555000000;		data_table[5].feg[0].cnl[4] = 51.1341000000;	data_table[5].feg[0].cl[5] = 0.0000000000;		data_table[5].feg[0].cnl[5] = 0.0000000000;
				data_table[6].feg[0].cl[0] = 0.0267000000;		data_table[6].feg[0].cnl[0] = 0.0541000000;	data_table[6].feg[0].cl[1] = 0.1328000000;		data_table[6].feg[0].cnl[1] = 0.5165000000;	data_table[6].feg[0].cl[2] = 0.5301000000;		data_table[6].feg[0].cnl[2] = 2.8207000000;	data_table[6].feg[0].cl[3] = 1.1020000000;		data_table[6].feg[0].cnl[3] = 10.6297000000;	data_table[6].feg[0].cl[4] = 0.4215000000;		data_table[6].feg[0].cnl[4] = 34.3764000000;	data_table[6].feg[0].cl[5] = 0.0000000000;		data_table[6].feg[0].cnl[5] = 0.0000000000;
				data_table[7].feg[0].cl[0] = 0.0365000000;		data_table[7].feg[0].cnl[0] = 0.0652000000;	data_table[7].feg[0].cl[1] = 0.1729000000;		data_table[7].feg[0].cnl[1] = 0.6184000000;	data_table[7].feg[0].cl[2] = 0.5805000000;		data_table[7].feg[0].cnl[2] = 2.9449000000;	data_table[7].feg[0].cl[3] = 0.8814000000;		data_table[7].feg[0].cnl[3] = 9.6298000000;	data_table[7].feg[0].cl[4] = 0.3121000000;		data_table[7].feg[0].cnl[4] = 28.2194000000;	data_table[7].feg[0].cl[5] = 0.0000000000;		data_table[7].feg[0].cnl[5] = 0.0000000000;
				data_table[8].feg[0].cl[0] = 0.0382000000;		data_table[8].feg[0].cnl[0] = 0.0613000000;	data_table[8].feg[0].cl[1] = 0.1822000000;		data_table[8].feg[0].cnl[1] = 0.5753000000;	data_table[8].feg[0].cl[2] = 0.5972000000;		data_table[8].feg[0].cnl[2] = 2.6858000000;	data_table[8].feg[0].cl[3] = 0.7707000000;		data_table[8].feg[0].cnl[3] = 8.8214000000;	data_table[8].feg[0].cl[4] = 0.2130000000;		data_table[8].feg[0].cnl[4] = 25.6668000000;	data_table[8].feg[0].cl[5] = 0.0000000000;		data_table[8].feg[0].cnl[5] = 0.0000000000;
				data_table[9].feg[0].cl[0] = 0.0380000000;		data_table[9].feg[0].cnl[0] = 0.0554000000;	data_table[9].feg[0].cl[1] = 0.1785000000;		data_table[9].feg[0].cnl[1] = 0.5087000000;	data_table[9].feg[0].cl[2] = 0.5494000000;		data_table[9].feg[0].cnl[2] = 2.2639000000;	data_table[9].feg[0].cl[3] = 0.6942000000;		data_table[9].feg[0].cnl[3] = 7.3316000000;	data_table[9].feg[0].cl[4] = 0.1918000000;		data_table[9].feg[0].cnl[4] = 21.6912000000;	data_table[9].feg[0].cl[5] = 0.0000000000;		data_table[9].feg[0].cnl[5] = 0.0000000000;
				data_table[10].feg[0].cl[0] = 0.1260000000;	data_table[10].feg[0].cnl[0] = 0.1684000000;	data_table[10].feg[0].cl[1] = 0.6442000000;	data_table[10].feg[0].cnl[1] = 1.7150000000;	data_table[10].feg[0].cl[2] = 0.8893000000;	data_table[10].feg[0].cnl[2] = 8.8386000000;	data_table[10].feg[0].cl[3] = 1.8197000000;	data_table[10].feg[0].cnl[3] = 50.8265000000;	data_table[10].feg[0].cl[4] = 1.2988000000;	data_table[10].feg[0].cnl[4] = 147.2073000000;	data_table[10].feg[0].cl[5] = 0.0000000000;	data_table[10].feg[0].cnl[5] = 0.0000000000;
				data_table[11].feg[0].cl[0] = 0.1130000000;	data_table[11].feg[0].cnl[0] = 0.1356000000;	data_table[11].feg[0].cl[1] = 0.5575000000;	data_table[11].feg[0].cnl[1] = 1.3579000000;	data_table[11].feg[0].cl[2] = 0.9046000000;	data_table[11].feg[0].cnl[2] = 6.9255000000;	data_table[11].feg[0].cl[3] = 2.1580000000;	data_table[11].feg[0].cnl[3] = 32.3165000000;	data_table[11].feg[0].cl[4] = 1.4735000000;	data_table[11].feg[0].cnl[4] = 92.1138000000;	data_table[11].feg[0].cl[5] = 0.0000000000;	data_table[11].feg[0].cnl[5] = 0.0000000000;
				data_table[12].feg[0].cl[0] = 0.1165000000;	data_table[12].feg[0].cnl[0] = 0.1295000000;	data_table[12].feg[0].cl[1] = 0.5504000000;	data_table[12].feg[0].cnl[1] = 1.2619000000;	data_table[12].feg[0].cl[2] = 1.0179000000;	data_table[12].feg[0].cnl[2] = 6.8242000000;	data_table[12].feg[0].cl[3] = 2.6295000000;	data_table[12].feg[0].cnl[3] = 28.4577000000;	data_table[12].feg[0].cl[4] = 1.5711000000;	data_table[12].feg[0].cnl[4] = 88.4750000000;	data_table[12].feg[0].cl[5] = 0.0000000000;	data_table[12].feg[0].cnl[5] = 0.0000000000;
				data_table[13].feg[0].cl[0] = 0.0567000000;	data_table[13].feg[0].cnl[0] = 0.0582000000;	data_table[13].feg[0].cl[1] = 0.3365000000;	data_table[13].feg[0].cnl[1] = 0.6155000000;	data_table[13].feg[0].cl[2] = 0.8104000000;	data_table[13].feg[0].cnl[2] = 3.2522000000;	data_table[13].feg[0].cl[3] = 2.4960000000;	data_table[13].feg[0].cnl[3] = 16.7929000000;	data_table[13].feg[0].cl[4] = 2.1186000000;	data_table[13].feg[0].cnl[4] = 57.6767000000;	data_table[13].feg[0].cl[5] = 0.0000000000;	data_table[13].feg[0].cnl[5] = 0.0000000000;
				data_table[14].feg[0].cl[0] = 0.1005000000;	data_table[14].feg[0].cnl[0] = 0.0977000000;	data_table[14].feg[0].cl[1] = 0.4615000000;	data_table[14].feg[0].cnl[1] = 0.9084000000;	data_table[14].feg[0].cl[2] = 1.0663000000;	data_table[14].feg[0].cnl[2] = 4.9654000000;	data_table[14].feg[0].cl[3] = 2.5854000000;	data_table[14].feg[0].cnl[3] = 18.5471000000;	data_table[14].feg[0].cl[4] = 1.2725000000;	data_table[14].feg[0].cnl[4] = 54.3648000000;	data_table[14].feg[0].cl[5] = 0.0000000000;	data_table[14].feg[0].cnl[5] = 0.0000000000;
				data_table[15].feg[0].cl[0] = 0.0915000000;	data_table[15].feg[0].cnl[0] = 0.0838000000;	data_table[15].feg[0].cl[1] = 0.4312000000;	data_table[15].feg[0].cnl[1] = 0.7788000000;	data_table[15].feg[0].cl[2] = 1.0847000000;	data_table[15].feg[0].cnl[2] = 4.3462000000;	data_table[15].feg[0].cl[3] = 2.4671000000;	data_table[15].feg[0].cnl[3] = 15.5846000000;	data_table[15].feg[0].cl[4] = 1.0852000000;	data_table[15].feg[0].cnl[4] = 44.6365000000;	data_table[15].feg[0].cl[5] = 0.0000000000;	data_table[15].feg[0].cnl[5] = 0.0000000000;
				data_table[16].feg[0].cl[0] = 0.0799000000;	data_table[16].feg[0].cnl[0] = 0.0694000000;	data_table[16].feg[0].cl[1] = 0.3891000000;	data_table[16].feg[0].cnl[1] = 0.6443000000;	data_table[16].feg[0].cl[2] = 1.0037000000;	data_table[16].feg[0].cnl[2] = 3.5351000000;	data_table[16].feg[0].cl[3] = 2.3332000000;	data_table[16].feg[0].cnl[3] = 12.5058000000;	data_table[16].feg[0].cl[4] = 1.0507000000;	data_table[16].feg[0].cnl[4] = 35.8633000000;	data_table[16].feg[0].cl[5] = 0.0000000000;	data_table[16].feg[0].cnl[5] = 0.0000000000;
				data_table[17].feg[0].cl[0] = 0.1044000000;	data_table[17].feg[0].cnl[0] = 0.0853000000;	data_table[17].feg[0].cl[1] = 0.4551000000;	data_table[17].feg[0].cnl[1] = 0.7701000000;	data_table[17].feg[0].cl[2] = 1.4232000000;	data_table[17].feg[0].cnl[2] = 4.4684000000;	data_table[17].feg[0].cl[3] = 2.1533000000;	data_table[17].feg[0].cnl[3] = 14.5864000000;	data_table[17].feg[0].cl[4] = 0.4459000000;	data_table[17].feg[0].cnl[4] = 41.2474000000;	data_table[17].feg[0].cl[5] = 0.0000000000;	data_table[17].feg[0].cnl[5] = 0.0000000000;
				data_table[18].feg[0].cl[0] = 0.2149000000;	data_table[18].feg[0].cnl[0] = 0.1660000000;	data_table[18].feg[0].cl[1] = 0.8703000000;	data_table[18].feg[0].cnl[1] = 1.6906000000;	data_table[18].feg[0].cl[2] = 2.4999000000;	data_table[18].feg[0].cnl[2] = 8.7447000000;	data_table[18].feg[0].cl[3] = 2.3591000000;	data_table[18].feg[0].cnl[3] = 46.7825000000;	data_table[18].feg[0].cl[4] = 3.0318000000;	data_table[18].feg[0].cnl[4] = 165.6923000000;	data_table[18].feg[0].cl[5] = 0.0000000000;	data_table[18].feg[0].cnl[5] = 0.0000000000;
				data_table[19].feg[0].cl[0] = 0.2355000000;	data_table[19].feg[0].cnl[0] = 0.1742000000;	data_table[19].feg[0].cl[1] = 0.9916000000;	data_table[19].feg[0].cnl[1] = 1.8329000000;	data_table[19].feg[0].cl[2] = 2.3959000000;	data_table[19].feg[0].cnl[2] = 8.8407000000;	data_table[19].feg[0].cl[3] = 3.7252000000;	data_table[19].feg[0].cnl[3] = 47.4583000000;	data_table[19].feg[0].cl[4] = 2.5647000000;	data_table[19].feg[0].cnl[4] = 134.9613000000;	data_table[19].feg[0].cl[5] = 0.0000000000;	data_table[19].feg[0].cnl[5] = 0.0000000000;
				data_table[20].feg[0].cl[0] = 0.4636000000;	data_table[20].feg[0].cnl[0] = 0.3682000000;	data_table[20].feg[0].cl[1] = 2.0802000000;	data_table[20].feg[0].cnl[1] = 4.0312000000;	data_table[20].feg[0].cl[2] = 2.9003000000;	data_table[20].feg[0].cnl[2] = 22.6493000000;	data_table[20].feg[0].cl[3] = 1.4193000000;	data_table[20].feg[0].cnl[3] = 71.8200000000;	data_table[20].feg[0].cl[4] = 2.4323000000;	data_table[20].feg[0].cnl[4] = 103.3691000000;	data_table[20].feg[0].cl[5] = 0.0000000000;	data_table[20].feg[0].cnl[5] = 0.0000000000;
				data_table[21].feg[0].cl[0] = 0.2123000000;	data_table[21].feg[0].cnl[0] = 0.1399000000;	data_table[21].feg[0].cl[1] = 0.8960000000;	data_table[21].feg[0].cnl[1] = 1.4568000000;	data_table[21].feg[0].cl[2] = 2.1765000000;	data_table[21].feg[0].cnl[2] = 6.7534000000;	data_table[21].feg[0].cl[3] = 3.0436000000;	data_table[21].feg[0].cnl[3] = 33.1168000000;	data_table[21].feg[0].cl[4] = 2.4439000000;	data_table[21].feg[0].cnl[4] = 101.8238000000;	data_table[21].feg[0].cl[5] = 0.0000000000;	data_table[21].feg[0].cnl[5] = 0.0000000000;
				data_table[22].feg[0].cl[0] = 0.2369000000;	data_table[22].feg[0].cnl[0] = 0.1505000000;	data_table[22].feg[0].cl[1] = 1.0774000000;	data_table[22].feg[0].cnl[1] = 1.6392000000;	data_table[22].feg[0].cl[2] = 2.1894000000;	data_table[22].feg[0].cnl[2] = 7.5691000000;	data_table[22].feg[0].cl[3] = 3.0825000000;	data_table[22].feg[0].cnl[3] = 36.8741000000;	data_table[22].feg[0].cl[4] = 1.7190000000;	data_table[22].feg[0].cnl[4] = 107.8517000000;	data_table[22].feg[0].cl[5] = 0.0000000000;	data_table[22].feg[0].cnl[5] = 0.0000000000;
				data_table[23].feg[0].cl[0] = 0.1970000000;	data_table[23].feg[0].cnl[0] = 0.1197000000;	data_table[23].feg[0].cl[1] = 0.8228000000;	data_table[23].feg[0].cnl[1] = 1.1985000000;	data_table[23].feg[0].cl[2] = 2.0200000000;	data_table[23].feg[0].cnl[2] = 5.4097000000;	data_table[23].feg[0].cl[3] = 2.1717000000;	data_table[23].feg[0].cnl[3] = 25.2361000000;	data_table[23].feg[0].cl[4] = 1.7516000000;	data_table[23].feg[0].cnl[4] = 94.4290000000;	data_table[23].feg[0].cl[5] = 0.0000000000;	data_table[23].feg[0].cnl[5] = 0.0000000000;
				data_table[24].feg[0].cl[0] = 0.1943000000;	data_table[24].feg[0].cnl[0] = 0.1135000000;	data_table[24].feg[0].cl[1] = 0.8190000000;	data_table[24].feg[0].cnl[1] = 1.1313000000;	data_table[24].feg[0].cl[2] = 1.9296000000;	data_table[24].feg[0].cnl[2] = 5.0341000000;	data_table[24].feg[0].cl[3] = 2.4968000000;	data_table[24].feg[0].cnl[3] = 24.1798000000;	data_table[24].feg[0].cl[4] = 2.0625000000;	data_table[24].feg[0].cnl[4] = 80.5598000000;	data_table[24].feg[0].cl[5] = 0.0000000000;	data_table[24].feg[0].cnl[5] = 0.0000000000;
				data_table[25].feg[0].cl[0] = 0.1929000000;	data_table[25].feg[0].cnl[0] = 0.1087000000;	data_table[25].feg[0].cl[1] = 0.8239000000;	data_table[25].feg[0].cnl[1] = 1.0806000000;	data_table[25].feg[0].cl[2] = 1.8689000000;	data_table[25].feg[0].cnl[2] = 4.7637000000;	data_table[25].feg[0].cl[3] = 2.3694000000;	data_table[25].feg[0].cnl[3] = 22.8500000000;	data_table[25].feg[0].cl[4] = 1.9060000000;	data_table[25].feg[0].cnl[4] = 76.7309000000;	data_table[25].feg[0].cl[5] = 0.0000000000;	data_table[25].feg[0].cnl[5] = 0.0000000000;
				data_table[26].feg[0].cl[0] = 0.2186000000;	data_table[26].feg[0].cnl[0] = 0.1182000000;	data_table[26].feg[0].cl[1] = 0.9861000000;	data_table[26].feg[0].cnl[1] = 1.2300000000;	data_table[26].feg[0].cl[2] = 1.8540000000;	data_table[26].feg[0].cnl[2] = 5.4177000000;	data_table[26].feg[0].cl[3] = 2.3258000000;	data_table[26].feg[0].cnl[3] = 25.7602000000;	data_table[26].feg[0].cl[4] = 1.4685000000;	data_table[26].feg[0].cnl[4] = 80.8542000000;	data_table[26].feg[0].cl[5] = 0.0000000000;	data_table[26].feg[0].cnl[5] = 0.0000000000;
				data_table[27].feg[0].cl[0] = 0.2313000000;	data_table[27].feg[0].cnl[0] = 0.1210000000;	data_table[27].feg[0].cl[1] = 1.0657000000;	data_table[27].feg[0].cnl[1] = 1.2691000000;	data_table[27].feg[0].cl[2] = 1.8229000000;	data_table[27].feg[0].cnl[2] = 5.6870000000;	data_table[27].feg[0].cl[3] = 2.2609000000;	data_table[27].feg[0].cnl[3] = 27.0917000000;	data_table[27].feg[0].cl[4] = 1.1883000000;	data_table[27].feg[0].cnl[4] = 83.0285000000;	data_table[27].feg[0].cl[5] = 0.0000000000;	data_table[27].feg[0].cnl[5] = 0.0000000000;
				data_table[28].feg[0].cl[0] = 0.3501000000;	data_table[28].feg[0].cnl[0] = 0.1867000000;	data_table[28].feg[0].cl[1] = 1.6558000000;	data_table[28].feg[0].cnl[1] = 1.9917000000;	data_table[28].feg[0].cl[2] = 1.9582000000;	data_table[28].feg[0].cnl[2] = 11.3396000000;	data_table[28].feg[0].cl[3] = 0.2134000000;	data_table[28].feg[0].cnl[3] = 53.2619000000;	data_table[28].feg[0].cl[4] = 1.4109000000;	data_table[28].feg[0].cnl[4] = 63.2520000000;	data_table[28].feg[0].cl[5] = 0.0000000000;	data_table[28].feg[0].cnl[5] = 0.0000000000;
				data_table[29].feg[0].cl[0] = 0.1780000000;	data_table[29].feg[0].cnl[0] = 0.0876000000;	data_table[29].feg[0].cl[1] = 0.8096000000;	data_table[29].feg[0].cnl[1] = 0.8650000000;	data_table[29].feg[0].cl[2] = 1.6744000000;	data_table[29].feg[0].cnl[2] = 3.8612000000;	data_table[29].feg[0].cl[3] = 1.9499000000;	data_table[29].feg[0].cnl[3] = 18.8726000000;	data_table[29].feg[0].cl[4] = 1.4495000000;	data_table[29].feg[0].cnl[4] = 64.7016000000;	data_table[29].feg[0].cl[5] = 0.0000000000;	data_table[29].feg[0].cnl[5] = 0.0000000000;
				data_table[30].feg[0].cl[0] = 0.2135000000;	data_table[30].feg[0].cnl[0] = 0.1020000000;	data_table[30].feg[0].cl[1] = 0.9768000000;	data_table[30].feg[0].cnl[1] = 1.0219000000;	data_table[30].feg[0].cl[2] = 1.6669000000;	data_table[30].feg[0].cnl[2] = 4.6275000000;	data_table[30].feg[0].cl[3] = 2.5662000000;	data_table[30].feg[0].cnl[3] = 22.8742000000;	data_table[30].feg[0].cl[4] = 1.6790000000;	data_table[30].feg[0].cnl[4] = 80.1535000000;	data_table[30].feg[0].cl[5] = 0.0000000000;	data_table[30].feg[0].cnl[5] = 0.0000000000;
				data_table[31].feg[0].cl[0] = 0.2135000000;	data_table[31].feg[0].cnl[0] = 0.0989000000;	data_table[31].feg[0].cl[1] = 0.9761000000;	data_table[31].feg[0].cnl[1] = 0.9845000000;	data_table[31].feg[0].cl[2] = 1.6555000000;	data_table[31].feg[0].cnl[2] = 4.5527000000;	data_table[31].feg[0].cl[3] = 2.8938000000;	data_table[31].feg[0].cnl[3] = 21.5563000000;	data_table[31].feg[0].cl[4] = 1.6356000000;	data_table[31].feg[0].cnl[4] = 70.3903000000;	data_table[31].feg[0].cl[5] = 0.0000000000;	data_table[31].feg[0].cnl[5] = 0.0000000000;
				data_table[32].feg[0].cl[0] = 0.2059000000;	data_table[32].feg[0].cnl[0] = 0.0926000000;	data_table[32].feg[0].cl[1] = 0.9518000000;	data_table[32].feg[0].cnl[1] = 0.9182000000;	data_table[32].feg[0].cl[2] = 1.6372000000;	data_table[32].feg[0].cnl[2] = 4.3291000000;	data_table[32].feg[0].cl[3] = 3.0490000000;	data_table[32].feg[0].cnl[3] = 19.2996000000;	data_table[32].feg[0].cl[4] = 1.4756000000;	data_table[32].feg[0].cnl[4] = 58.9329000000;	data_table[32].feg[0].cl[5] = 0.0000000000;	data_table[32].feg[0].cnl[5] = 0.0000000000;
				data_table[33].feg[0].cl[0] = 0.1574000000;	data_table[33].feg[0].cnl[0] = 0.0686000000;	data_table[33].feg[0].cl[1] = 0.7614000000;	data_table[33].feg[0].cnl[1] = 0.6808000000;	data_table[33].feg[0].cl[2] = 1.4834000000;	data_table[33].feg[0].cnl[2] = 3.1163000000;	data_table[33].feg[0].cl[3] = 3.0016000000;	data_table[33].feg[0].cnl[3] = 14.3458000000;	data_table[33].feg[0].cl[4] = 1.7978000000;	data_table[33].feg[0].cnl[4] = 44.0455000000;	data_table[33].feg[0].cl[5] = 0.0000000000;	data_table[33].feg[0].cnl[5] = 0.0000000000;
				data_table[34].feg[0].cl[0] = 0.1899000000;	data_table[34].feg[0].cnl[0] = 0.0810000000;	data_table[34].feg[0].cl[1] = 0.8983000000;	data_table[34].feg[0].cnl[1] = 0.7957000000;	data_table[34].feg[0].cl[2] = 1.6358000000;	data_table[34].feg[0].cnl[2] = 3.9054000000;	data_table[34].feg[0].cl[3] = 3.1845000000;	data_table[34].feg[0].cnl[3] = 15.7701000000;	data_table[34].feg[0].cl[4] = 1.1518000000;	data_table[34].feg[0].cnl[4] = 45.6124000000;	data_table[34].feg[0].cl[5] = 0.0000000000;	data_table[34].feg[0].cnl[5] = 0.0000000000;
				data_table[35].feg[0].cl[0] = 0.1742000000;	data_table[35].feg[0].cnl[0] = 0.0723000000;	data_table[35].feg[0].cl[1] = 0.8447000000;	data_table[35].feg[0].cnl[1] = 0.7123000000;	data_table[35].feg[0].cl[2] = 1.5944000000;	data_table[35].feg[0].cnl[2] = 3.5192000000;	data_table[35].feg[0].cl[3] = 3.1507000000;	data_table[35].feg[0].cnl[3] = 13.7724000000;	data_table[35].feg[0].cl[4] = 1.1338000000;	data_table[35].feg[0].cnl[4] = 39.1148000000;	data_table[35].feg[0].cl[5] = 0.0000000000;	data_table[35].feg[0].cnl[5] = 0.0000000000;
				data_table[36].feg[0].cl[0] = 0.3781000000;	data_table[36].feg[0].cnl[0] = 0.1557000000;	data_table[36].feg[0].cl[1] = 1.4904000000;	data_table[36].feg[0].cnl[1] = 1.5347000000;	data_table[36].feg[0].cl[2] = 3.5753000000;	data_table[36].feg[0].cnl[2] = 9.9947000000;	data_table[36].feg[0].cl[3] = 3.0031000000;	data_table[36].feg[0].cnl[3] = 51.4251000000;	data_table[36].feg[0].cl[4] = 3.3272000000;	data_table[36].feg[0].cnl[4] = 185.9828000000;	data_table[36].feg[0].cl[5] = 0.0000000000;	data_table[36].feg[0].cnl[5] = 0.0000000000;
				data_table[37].feg[0].cl[0] = 0.3723000000;	data_table[37].feg[0].cnl[0] = 0.1480000000;	data_table[37].feg[0].cl[1] = 1.4598000000;	data_table[37].feg[0].cnl[1] = 1.4643000000;	data_table[37].feg[0].cl[2] = 3.5124000000;	data_table[37].feg[0].cnl[2] = 9.2320000000;	data_table[37].feg[0].cl[3] = 4.4612000000;	data_table[37].feg[0].cnl[3] = 49.8807000000;	data_table[37].feg[0].cl[4] = 3.3031000000;	data_table[37].feg[0].cnl[4] = 148.0937000000;	data_table[37].feg[0].cl[5] = 0.0000000000;	data_table[37].feg[0].cnl[5] = 0.0000000000;
				data_table[38].feg[0].cl[0] = 0.3234000000;	data_table[38].feg[0].cnl[0] = 0.1244000000;	data_table[38].feg[0].cl[1] = 1.2737000000;	data_table[38].feg[0].cnl[1] = 1.1948000000;	data_table[38].feg[0].cl[2] = 3.2115000000;	data_table[38].feg[0].cnl[2] = 7.2756000000;	data_table[38].feg[0].cl[3] = 4.0563000000;	data_table[38].feg[0].cnl[3] = 34.1430000000;	data_table[38].feg[0].cl[4] = 3.7962000000;	data_table[38].feg[0].cnl[4] = 111.2079000000;	data_table[38].feg[0].cl[5] = 0.0000000000;	data_table[38].feg[0].cnl[5] = 0.0000000000;
				data_table[39].feg[0].cl[0] = 0.2997000000;	data_table[39].feg[0].cnl[0] = 0.1121000000;	data_table[39].feg[0].cl[1] = 1.1879000000;	data_table[39].feg[0].cnl[1] = 1.0638000000;	data_table[39].feg[0].cl[2] = 3.1075000000;	data_table[39].feg[0].cnl[2] = 6.3891000000;	data_table[39].feg[0].cl[3] = 3.9740000000;	data_table[39].feg[0].cnl[3] = 28.7081000000;	data_table[39].feg[0].cl[4] = 3.5769000000;	data_table[39].feg[0].cnl[4] = 97.4289000000;	data_table[39].feg[0].cl[5] = 0.0000000000;	data_table[39].feg[0].cnl[5] = 0.0000000000;
				data_table[40].feg[0].cl[0] = 0.1680000000;	data_table[40].feg[0].cnl[0] = 0.0597000000;	data_table[40].feg[0].cl[1] = 0.9370000000;	data_table[40].feg[0].cnl[1] = 0.6524000000;	data_table[40].feg[0].cl[2] = 2.7300000000;	data_table[40].feg[0].cnl[2] = 4.4317000000;	data_table[40].feg[0].cl[3] = 3.8150000000;	data_table[40].feg[0].cnl[3] = 19.5540000000;	data_table[40].feg[0].cl[4] = 3.0053000000;	data_table[40].feg[0].cnl[4] = 85.5011000000;	data_table[40].feg[0].cl[5] = 0.0000000000;	data_table[40].feg[0].cnl[5] = 0.0000000000;
				data_table[41].feg[0].cl[0] = 0.3069000000;	data_table[41].feg[0].cnl[0] = 0.1101000000;	data_table[41].feg[0].cl[1] = 1.1714000000;	data_table[41].feg[0].cnl[1] = 1.0222000000;	data_table[41].feg[0].cl[2] = 3.2293000000;	data_table[41].feg[0].cnl[2] = 5.9613000000;	data_table[41].feg[0].cl[3] = 3.4254000000;	data_table[41].feg[0].cnl[3] = 25.1965000000;	data_table[41].feg[0].cl[4] = 2.1224000000;	data_table[41].feg[0].cnl[4] = 93.5831000000;	data_table[41].feg[0].cl[5] = 0.0000000000;	data_table[41].feg[0].cnl[5] = 0.0000000000;
				data_table[42].feg[0].cl[0] = 0.2928000000;	data_table[42].feg[0].cnl[0] = 0.1020000000;	data_table[42].feg[0].cl[1] = 1.1267000000;	data_table[42].feg[0].cnl[1] = 0.9481000000;	data_table[42].feg[0].cl[2] = 3.1675000000;	data_table[42].feg[0].cnl[2] = 5.4713000000;	data_table[42].feg[0].cl[3] = 3.6619000000;	data_table[42].feg[0].cnl[3] = 23.8153000000;	data_table[42].feg[0].cl[4] = 2.5942000000;	data_table[42].feg[0].cnl[4] = 82.8991000000;	data_table[42].feg[0].cl[5] = 0.0000000000;	data_table[42].feg[0].cnl[5] = 0.0000000000;
				data_table[43].feg[0].cl[0] = 0.2604000000;	data_table[43].feg[0].cnl[0] = 0.0887000000;	data_table[43].feg[0].cl[1] = 1.0442000000;	data_table[43].feg[0].cnl[1] = 0.8240000000;	data_table[43].feg[0].cl[2] = 3.0761000000;	data_table[43].feg[0].cnl[2] = 4.8278000000;	data_table[43].feg[0].cl[3] = 3.2175000000;	data_table[43].feg[0].cnl[3] = 19.8977000000;	data_table[43].feg[0].cl[4] = 1.9448000000;	data_table[43].feg[0].cnl[4] = 80.4566000000;	data_table[43].feg[0].cl[5] = 0.0000000000;	data_table[43].feg[0].cnl[5] = 0.0000000000;
				data_table[44].feg[0].cl[0] = 0.2713000000;	data_table[44].feg[0].cnl[0] = 0.0907000000;	data_table[44].feg[0].cl[1] = 1.0556000000;	data_table[44].feg[0].cnl[1] = 0.8324000000;	data_table[44].feg[0].cl[2] = 3.1416000000;	data_table[44].feg[0].cnl[2] = 4.7702000000;	data_table[44].feg[0].cl[3] = 3.0451000000;	data_table[44].feg[0].cnl[3] = 19.7862000000;	data_table[44].feg[0].cl[4] = 1.7179000000;	data_table[44].feg[0].cnl[4] = 80.2540000000;	data_table[44].feg[0].cl[5] = 0.0000000000;	data_table[44].feg[0].cnl[5] = 0.0000000000;
				data_table[45].feg[0].cl[0] = 0.2003000000;	data_table[45].feg[0].cnl[0] = 0.0659000000;	data_table[45].feg[0].cl[1] = 0.8779000000;	data_table[45].feg[0].cnl[1] = 0.6111000000;	data_table[45].feg[0].cl[2] = 2.6135000000;	data_table[45].feg[0].cnl[2] = 3.5563000000;	data_table[45].feg[0].cl[3] = 2.8594000000;	data_table[45].feg[0].cnl[3] = 12.7638000000;	data_table[45].feg[0].cl[4] = 1.0258000000;	data_table[45].feg[0].cnl[4] = 44.4283000000;	data_table[45].feg[0].cl[5] = 0.0000000000;	data_table[45].feg[0].cnl[5] = 0.0000000000;
				data_table[46].feg[0].cl[0] = 0.2739000000;	data_table[46].feg[0].cnl[0] = 0.0881000000;	data_table[46].feg[0].cl[1] = 1.0503000000;	data_table[46].feg[0].cnl[1] = 0.8028000000;	data_table[46].feg[0].cl[2] = 3.1564000000;	data_table[46].feg[0].cnl[2] = 4.4451000000;	data_table[46].feg[0].cl[3] = 2.7543000000;	data_table[46].feg[0].cnl[3] = 18.7011000000;	data_table[46].feg[0].cl[4] = 1.4328000000;	data_table[46].feg[0].cnl[4] = 79.2633000000;	data_table[46].feg[0].cl[5] = 0.0000000000;	data_table[46].feg[0].cnl[5] = 0.0000000000;
				data_table[47].feg[0].cl[0] = 0.3072000000;	data_table[47].feg[0].cnl[0] = 0.0966000000;	data_table[47].feg[0].cl[1] = 1.1303000000;	data_table[47].feg[0].cnl[1] = 0.8856000000;	data_table[47].feg[0].cl[2] = 3.2046000000;	data_table[47].feg[0].cnl[2] = 4.6273000000;	data_table[47].feg[0].cl[3] = 2.9329000000;	data_table[47].feg[0].cnl[3] = 20.6789000000;	data_table[47].feg[0].cl[4] = 1.6560000000;	data_table[47].feg[0].cnl[4] = 73.4723000000;	data_table[47].feg[0].cl[5] = 0.0000000000;	data_table[47].feg[0].cnl[5] = 0.0000000000;
				data_table[48].feg[0].cl[0] = 0.3564000000;	data_table[48].feg[0].cnl[0] = 0.1091000000;	data_table[48].feg[0].cl[1] = 1.3011000000;	data_table[48].feg[0].cnl[1] = 1.0452000000;	data_table[48].feg[0].cl[2] = 3.2424000000;	data_table[48].feg[0].cnl[2] = 5.0900000000;	data_table[48].feg[0].cl[3] = 3.4839000000;	data_table[48].feg[0].cnl[3] = 24.6578000000;	data_table[48].feg[0].cl[4] = 2.0459000000;	data_table[48].feg[0].cnl[4] = 88.0513000000;	data_table[48].feg[0].cl[5] = 0.0000000000;	data_table[48].feg[0].cnl[5] = 0.0000000000;
				data_table[49].feg[0].cl[0] = 0.2966000000;	data_table[49].feg[0].cnl[0] = 0.0896000000;	data_table[49].feg[0].cl[1] = 1.1157000000;	data_table[49].feg[0].cnl[1] = 0.8268000000;	data_table[49].feg[0].cl[2] = 3.0973000000;	data_table[49].feg[0].cnl[2] = 4.2242000000;	data_table[49].feg[0].cl[3] = 3.8156000000;	data_table[49].feg[0].cnl[3] = 20.6900000000;	data_table[49].feg[0].cl[4] = 2.5281000000;	data_table[49].feg[0].cnl[4] = 71.3399000000;	data_table[49].feg[0].cl[5] = 0.0000000000;	data_table[49].feg[0].cnl[5] = 0.0000000000;
				data_table[50].feg[0].cl[0] = 0.2725000000;	data_table[50].feg[0].cnl[0] = 0.0809000000;	data_table[50].feg[0].cl[1] = 1.0651000000;	data_table[50].feg[0].cnl[1] = 0.7488000000;	data_table[50].feg[0].cl[2] = 2.9940000000;	data_table[50].feg[0].cnl[2] = 3.8710000000;	data_table[50].feg[0].cl[3] = 4.0697000000;	data_table[50].feg[0].cnl[3] = 18.8800000000;	data_table[50].feg[0].cl[4] = 2.5682000000;	data_table[50].feg[0].cnl[4] = 60.6499000000;	data_table[50].feg[0].cl[5] = 0.0000000000;	data_table[50].feg[0].cnl[5] = 0.0000000000;
				data_table[51].feg[0].cl[0] = 0.2422000000;	data_table[51].feg[0].cnl[0] = 0.0708000000;	data_table[51].feg[0].cl[1] = 0.9692000000;	data_table[51].feg[0].cnl[1] = 0.6472000000;	data_table[51].feg[0].cl[2] = 2.8114000000;	data_table[51].feg[0].cnl[2] = 3.3609000000;	data_table[51].feg[0].cl[3] = 4.1509000000;	data_table[51].feg[0].cnl[3] = 16.0752000000;	data_table[51].feg[0].cl[4] = 2.8161000000;	data_table[51].feg[0].cnl[4] = 50.1724000000;	data_table[51].feg[0].cl[5] = 0.0000000000;	data_table[51].feg[0].cnl[5] = 0.0000000000;
				data_table[52].feg[0].cl[0] = 0.2617000000;	data_table[52].feg[0].cnl[0] = 0.0749000000;	data_table[52].feg[0].cl[1] = 1.0325000000;	data_table[52].feg[0].cnl[1] = 0.6914000000;	data_table[52].feg[0].cl[2] = 2.8097000000;	data_table[52].feg[0].cnl[2] = 3.4634000000;	data_table[52].feg[0].cl[3] = 4.4809000000;	data_table[52].feg[0].cnl[3] = 16.3603000000;	data_table[52].feg[0].cl[4] = 2.3190000000;	data_table[52].feg[0].cnl[4] = 48.2522000000;	data_table[52].feg[0].cl[5] = 0.0000000000;	data_table[52].feg[0].cnl[5] = 0.0000000000;
				data_table[53].feg[0].cl[0] = 0.2334000000;	data_table[53].feg[0].cnl[0] = 0.0655000000;	data_table[53].feg[0].cl[1] = 0.9496000000;	data_table[53].feg[0].cnl[1] = 0.6050000000;	data_table[53].feg[0].cl[2] = 2.6381000000;	data_table[53].feg[0].cnl[2] = 3.0389000000;	data_table[53].feg[0].cl[3] = 4.4680000000;	data_table[53].feg[0].cnl[3] = 14.0809000000;	data_table[53].feg[0].cl[4] = 2.5020000000;	data_table[53].feg[0].cnl[4] = 41.0005000000;	data_table[53].feg[0].cl[5] = 0.0000000000;	data_table[53].feg[0].cnl[5] = 0.0000000000;
				data_table[54].feg[0].cl[0] = 0.5713000000;	data_table[54].feg[0].cnl[0] = 0.1626000000;	data_table[54].feg[0].cl[1] = 2.4866000000;	data_table[54].feg[0].cnl[1] = 1.8213000000;	data_table[54].feg[0].cl[2] = 4.9795000000;	data_table[54].feg[0].cnl[2] = 11.1049000000;	data_table[54].feg[0].cl[3] = 4.0198000000;	data_table[54].feg[0].cnl[3] = 49.0568000000;	data_table[54].feg[0].cl[4] = 4.4403000000;	data_table[54].feg[0].cnl[4] = 202.9987000000;	data_table[54].feg[0].cl[5] = 0.0000000000;	data_table[54].feg[0].cnl[5] = 0.0000000000;
				data_table[55].feg[0].cl[0] = 0.5229000000;	data_table[55].feg[0].cnl[0] = 0.1434000000;	data_table[55].feg[0].cl[1] = 2.2874000000;	data_table[55].feg[0].cnl[1] = 1.6019000000;	data_table[55].feg[0].cl[2] = 4.7243000000;	data_table[55].feg[0].cnl[2] = 9.4511000000;	data_table[55].feg[0].cl[3] = 5.0807000000;	data_table[55].feg[0].cnl[3] = 42.7685000000;	data_table[55].feg[0].cl[4] = 5.6389000000;	data_table[55].feg[0].cnl[4] = 148.4969000000;	data_table[55].feg[0].cl[5] = 0.0000000000;	data_table[55].feg[0].cnl[5] = 0.0000000000;
				data_table[56].feg[0].cl[0] = 0.5461000000;	data_table[56].feg[0].cnl[0] = 0.1479000000;	data_table[56].feg[0].cl[1] = 2.3856000000;	data_table[56].feg[0].cnl[1] = 1.6552000000;	data_table[56].feg[0].cl[2] = 5.0653000000;	data_table[56].feg[0].cnl[2] = 10.0059000000;	data_table[56].feg[0].cl[3] = 5.7601000000;	data_table[56].feg[0].cnl[3] = 47.3245000000;	data_table[56].feg[0].cl[4] = 4.0463000000;	data_table[56].feg[0].cnl[4] = 145.8464000000;	data_table[56].feg[0].cl[5] = 0.0000000000;	data_table[56].feg[0].cnl[5] = 0.0000000000;
				data_table[57].feg[0].cl[0] = 0.2227000000;	data_table[57].feg[0].cnl[0] = 0.0571000000;	data_table[57].feg[0].cl[1] = 1.0760000000;	data_table[57].feg[0].cnl[1] = 0.5946000000;	data_table[57].feg[0].cl[2] = 2.9482000000;	data_table[57].feg[0].cnl[2] = 3.2022000000;	data_table[57].feg[0].cl[3] = 5.8496000000;	data_table[57].feg[0].cnl[3] = 16.4253000000;	data_table[57].feg[0].cl[4] = 7.1834000000;	data_table[57].feg[0].cnl[4] = 95.7030000000;	data_table[57].feg[0].cl[5] = 0.0000000000;	data_table[57].feg[0].cnl[5] = 0.0000000000;
				data_table[58].feg[0].cl[0] = 0.5237000000;	data_table[58].feg[0].cnl[0] = 0.1360000000;	data_table[58].feg[0].cl[1] = 2.2913000000;	data_table[58].feg[0].cnl[1] = 1.5068000000;	data_table[58].feg[0].cl[2] = 4.6161000000;	data_table[58].feg[0].cnl[2] = 8.8213000000;	data_table[58].feg[0].cl[3] = 4.7233000000;	data_table[58].feg[0].cnl[3] = 41.9536000000;	data_table[58].feg[0].cl[4] = 4.8173000000;	data_table[58].feg[0].cnl[4] = 141.2424000000;	data_table[58].feg[0].cl[5] = 0.0000000000;	data_table[58].feg[0].cnl[5] = 0.0000000000;
				data_table[59].feg[0].cl[0] = 0.5368000000;	data_table[59].feg[0].cnl[0] = 0.1378000000;	data_table[59].feg[0].cl[1] = 2.3301000000;	data_table[59].feg[0].cnl[1] = 1.5140000000;	data_table[59].feg[0].cl[2] = 4.6058000000;	data_table[59].feg[0].cnl[2] = 8.8719000000;	data_table[59].feg[0].cl[3] = 4.6621000000;	data_table[59].feg[0].cnl[3] = 43.5967000000;	data_table[59].feg[0].cl[4] = 4.4622000000;	data_table[59].feg[0].cnl[4] = 141.8065000000;	data_table[59].feg[0].cl[5] = 0.0000000000;	data_table[59].feg[0].cnl[5] = 0.0000000000;
				data_table[60].feg[0].cl[0] = 0.5232000000;	data_table[60].feg[0].cnl[0] = 0.1317000000;	data_table[60].feg[0].cl[1] = 2.2627000000;	data_table[60].feg[0].cnl[1] = 1.4336000000;	data_table[60].feg[0].cl[2] = 4.4552000000;	data_table[60].feg[0].cnl[2] = 8.3087000000;	data_table[60].feg[0].cl[3] = 4.4787000000;	data_table[60].feg[0].cnl[3] = 40.6010000000;	data_table[60].feg[0].cl[4] = 4.5073000000;	data_table[60].feg[0].cnl[4] = 135.9196000000;	data_table[60].feg[0].cl[5] = 0.0000000000;	data_table[60].feg[0].cnl[5] = 0.0000000000;
				data_table[61].feg[0].cl[0] = 0.5162000000;	data_table[61].feg[0].cnl[0] = 0.1279000000;	data_table[61].feg[0].cl[1] = 2.2302000000;	data_table[61].feg[0].cnl[1] = 1.3811000000;	data_table[61].feg[0].cl[2] = 4.3449000000;	data_table[61].feg[0].cnl[2] = 7.9629000000;	data_table[61].feg[0].cl[3] = 4.3598000000;	data_table[61].feg[0].cnl[3] = 39.1213000000;	data_table[61].feg[0].cl[4] = 4.4292000000;	data_table[61].feg[0].cnl[4] = 132.7846000000;	data_table[61].feg[0].cl[5] = 0.0000000000;	data_table[61].feg[0].cnl[5] = 0.0000000000;
				data_table[62].feg[0].cl[0] = 0.5272000000;	data_table[62].feg[0].cnl[0] = 0.1285000000;	data_table[62].feg[0].cl[1] = 2.2844000000;	data_table[62].feg[0].cnl[1] = 1.3943000000;	data_table[62].feg[0].cl[2] = 4.3361000000;	data_table[62].feg[0].cnl[2] = 8.1081000000;	data_table[62].feg[0].cl[3] = 4.3178000000;	data_table[62].feg[0].cnl[3] = 40.9631000000;	data_table[62].feg[0].cl[4] = 4.0908000000;	data_table[62].feg[0].cnl[4] = 134.1233000000;	data_table[62].feg[0].cl[5] = 0.0000000000;	data_table[62].feg[0].cnl[5] = 0.0000000000;
				data_table[63].feg[0].cl[0] = 0.9664000000;	data_table[63].feg[0].cnl[0] = 0.2641000000;	data_table[63].feg[0].cl[1] = 3.4052000000;	data_table[63].feg[0].cnl[1] = 2.6586000000;	data_table[63].feg[0].cl[2] = 5.0803000000;	data_table[63].feg[0].cnl[2] = 16.2213000000;	data_table[63].feg[0].cl[3] = 1.4991000000;	data_table[63].feg[0].cnl[3] = 80.2060000000;	data_table[63].feg[0].cl[4] = 4.2528000000;	data_table[63].feg[0].cnl[4] = 92.5359000000;	data_table[63].feg[0].cl[5] = 0.0000000000;	data_table[63].feg[0].cnl[5] = 0.0000000000;
				data_table[64].feg[0].cl[0] = 0.5110000000;	data_table[64].feg[0].cnl[0] = 0.1210000000;	data_table[64].feg[0].cl[1] = 2.1570000000;	data_table[64].feg[0].cnl[1] = 1.2704000000;	data_table[64].feg[0].cl[2] = 4.0308000000;	data_table[64].feg[0].cnl[2] = 7.1368000000;	data_table[64].feg[0].cl[3] = 3.9936000000;	data_table[64].feg[0].cnl[3] = 35.0354000000;	data_table[64].feg[0].cl[4] = 4.2466000000;	data_table[64].feg[0].cnl[4] = 123.5062000000;	data_table[64].feg[0].cl[5] = 0.0000000000;	data_table[64].feg[0].cnl[5] = 0.0000000000;
				data_table[65].feg[0].cl[0] = 0.4974000000;	data_table[65].feg[0].cnl[0] = 0.1157000000;	data_table[65].feg[0].cl[1] = 2.1097000000;	data_table[65].feg[0].cnl[1] = 1.2108000000;	data_table[65].feg[0].cl[2] = 3.8906000000;	data_table[65].feg[0].cnl[2] = 6.7377000000;	data_table[65].feg[0].cl[3] = 3.8100000000;	data_table[65].feg[0].cnl[3] = 32.4150000000;	data_table[65].feg[0].cl[4] = 4.3084000000;	data_table[65].feg[0].cnl[4] = 116.9225000000;	data_table[65].feg[0].cl[5] = 0.0000000000;	data_table[65].feg[0].cnl[5] = 0.0000000000;
				data_table[66].feg[0].cl[0] = 0.4679000000;	data_table[66].feg[0].cnl[0] = 0.1069000000;	data_table[66].feg[0].cl[1] = 1.9693000000;	data_table[66].feg[0].cnl[1] = 1.0994000000;	data_table[66].feg[0].cl[2] = 3.7191000000;	data_table[66].feg[0].cnl[2] = 5.9769000000;	data_table[66].feg[0].cl[3] = 3.9632000000;	data_table[66].feg[0].cnl[3] = 27.1491000000;	data_table[66].feg[0].cl[4] = 4.2432000000;	data_table[66].feg[0].cnl[4] = 96.3119000000;	data_table[66].feg[0].cl[5] = 0.0000000000;	data_table[66].feg[0].cnl[5] = 0.0000000000;
				data_table[67].feg[0].cl[0] = 0.5034000000;	data_table[67].feg[0].cnl[0] = 0.1141000000;	data_table[67].feg[0].cl[1] = 2.1088000000;	data_table[67].feg[0].cnl[1] = 1.1769000000;	data_table[67].feg[0].cl[2] = 3.8232000000;	data_table[67].feg[0].cnl[2] = 6.6087000000;	data_table[67].feg[0].cl[3] = 3.7299000000;	data_table[67].feg[0].cnl[3] = 33.4332000000;	data_table[67].feg[0].cl[4] = 3.8963000000;	data_table[67].feg[0].cnl[4] = 116.4913000000;	data_table[67].feg[0].cl[5] = 0.0000000000;	data_table[67].feg[0].cnl[5] = 0.0000000000;
				data_table[68].feg[0].cl[0] = 0.4839000000;	data_table[68].feg[0].cnl[0] = 0.1081000000;	data_table[68].feg[0].cl[1] = 2.0262000000;	data_table[68].feg[0].cnl[1] = 1.1012000000;	data_table[68].feg[0].cl[2] = 3.6851000000;	data_table[68].feg[0].cnl[2] = 6.1114000000;	data_table[68].feg[0].cl[3] = 3.5874000000;	data_table[68].feg[0].cnl[3] = 30.3728000000;	data_table[68].feg[0].cl[4] = 4.0037000000;	data_table[68].feg[0].cnl[4] = 110.5988000000;	data_table[68].feg[0].cl[5] = 0.0000000000;	data_table[68].feg[0].cnl[5] = 0.0000000000;
				data_table[69].feg[0].cl[0] = 0.5221000000;	data_table[69].feg[0].cnl[0] = 0.1148000000;	data_table[69].feg[0].cl[1] = 2.1695000000;	data_table[69].feg[0].cnl[1] = 1.1860000000;	data_table[69].feg[0].cl[2] = 3.7567000000;	data_table[69].feg[0].cnl[2] = 6.7520000000;	data_table[69].feg[0].cl[3] = 3.6685000000;	data_table[69].feg[0].cnl[3] = 35.6807000000;	data_table[69].feg[0].cl[4] = 3.4274000000;	data_table[69].feg[0].cnl[4] = 118.0692000000;	data_table[69].feg[0].cl[5] = 0.0000000000;	data_table[69].feg[0].cnl[5] = 0.0000000000;
				data_table[70].feg[0].cl[0] = 0.4680000000;	data_table[70].feg[0].cnl[0] = 0.1015000000;	data_table[70].feg[0].cl[1] = 1.9466000000;	data_table[70].feg[0].cnl[1] = 1.0195000000;	data_table[70].feg[0].cl[2] = 3.5428000000;	data_table[70].feg[0].cnl[2] = 5.6058000000;	data_table[70].feg[0].cl[3] = 3.8490000000;	data_table[70].feg[0].cnl[3] = 27.4899000000;	data_table[70].feg[0].cl[4] = 3.6594000000;	data_table[70].feg[0].cnl[4] = 95.2846000000;	data_table[70].feg[0].cl[5] = 0.0000000000;	data_table[70].feg[0].cnl[5] = 0.0000000000;
				data_table[71].feg[0].cl[0] = 0.4048000000;	data_table[71].feg[0].cnl[0] = 0.0868000000;	data_table[71].feg[0].cl[1] = 1.7370000000;	data_table[71].feg[0].cnl[1] = 0.8585000000;	data_table[71].feg[0].cl[2] = 3.3399000000;	data_table[71].feg[0].cnl[2] = 4.6378000000;	data_table[71].feg[0].cl[3] = 3.9448000000;	data_table[71].feg[0].cnl[3] = 21.6900000000;	data_table[71].feg[0].cl[4] = 3.7293000000;	data_table[71].feg[0].cnl[4] = 80.2408000000;	data_table[71].feg[0].cl[5] = 0.0000000000;	data_table[71].feg[0].cnl[5] = 0.0000000000;
				data_table[72].feg[0].cl[0] = 0.3835000000;	data_table[72].feg[0].cnl[0] = 0.0810000000;	data_table[72].feg[0].cl[1] = 1.6747000000;	data_table[72].feg[0].cnl[1] = 0.8020000000;	data_table[72].feg[0].cl[2] = 3.2986000000;	data_table[72].feg[0].cnl[2] = 4.3545000000;	data_table[72].feg[0].cl[3] = 4.0462000000;	data_table[72].feg[0].cnl[3] = 19.9644000000;	data_table[72].feg[0].cl[4] = 3.4303000000;	data_table[72].feg[0].cnl[4] = 73.6337000000;	data_table[72].feg[0].cl[5] = 0.0000000000;	data_table[72].feg[0].cnl[5] = 0.0000000000;
				data_table[73].feg[0].cl[0] = 0.3661000000;	data_table[73].feg[0].cnl[0] = 0.0761000000;	data_table[73].feg[0].cl[1] = 1.6191000000;	data_table[73].feg[0].cnl[1] = 0.7543000000;	data_table[73].feg[0].cl[2] = 3.2455000000;	data_table[73].feg[0].cnl[2] = 4.0952000000;	data_table[73].feg[0].cl[3] = 4.0856000000;	data_table[73].feg[0].cnl[3] = 18.2886000000;	data_table[73].feg[0].cl[4] = 3.2064000000;	data_table[73].feg[0].cnl[4] = 68.0967000000;	data_table[73].feg[0].cl[5] = 0.0000000000;	data_table[73].feg[0].cnl[5] = 0.0000000000;
				data_table[74].feg[0].cl[0] = 0.3933000000;	data_table[74].feg[0].cnl[0] = 0.0806000000;	data_table[74].feg[0].cl[1] = 1.6973000000;	data_table[74].feg[0].cnl[1] = 0.7972000000;	data_table[74].feg[0].cl[2] = 3.4202000000;	data_table[74].feg[0].cnl[2] = 4.4237000000;	data_table[74].feg[0].cl[3] = 4.1274000000;	data_table[74].feg[0].cnl[3] = 19.5692000000;	data_table[74].feg[0].cl[4] = 2.6158000000;	data_table[74].feg[0].cnl[4] = 68.7477000000;	data_table[74].feg[0].cl[5] = 0.0000000000;	data_table[74].feg[0].cnl[5] = 0.0000000000;
				data_table[75].feg[0].cl[0] = 0.3854000000;	data_table[75].feg[0].cnl[0] = 0.0787000000;	data_table[75].feg[0].cl[1] = 1.6555000000;	data_table[75].feg[0].cnl[1] = 0.7638000000;	data_table[75].feg[0].cl[2] = 3.4129000000;	data_table[75].feg[0].cnl[2] = 4.2441000000;	data_table[75].feg[0].cl[3] = 4.1111000000;	data_table[75].feg[0].cnl[3] = 18.3700000000;	data_table[75].feg[0].cl[4] = 2.4106000000;	data_table[75].feg[0].cnl[4] = 65.1071000000;	data_table[75].feg[0].cl[5] = 0.0000000000;	data_table[75].feg[0].cnl[5] = 0.0000000000;
				data_table[76].feg[0].cl[0] = 0.3510000000;	data_table[76].feg[0].cnl[0] = 0.0706000000;	data_table[76].feg[0].cl[1] = 1.5620000000;	data_table[76].feg[0].cnl[1] = 0.6904000000;	data_table[76].feg[0].cl[2] = 3.2946000000;	data_table[76].feg[0].cnl[2] = 3.8266000000;	data_table[76].feg[0].cl[3] = 4.0615000000;	data_table[76].feg[0].cnl[3] = 16.0812000000;	data_table[76].feg[0].cl[4] = 2.4382000000;	data_table[76].feg[0].cnl[4] = 58.7638000000;	data_table[76].feg[0].cl[5] = 0.0000000000;	data_table[76].feg[0].cnl[5] = 0.0000000000;
				data_table[77].feg[0].cl[0] = 0.3083000000;	data_table[77].feg[0].cnl[0] = 0.0609000000;	data_table[77].feg[0].cl[1] = 1.4158000000;	data_table[77].feg[0].cnl[1] = 0.5993000000;	data_table[77].feg[0].cl[2] = 2.9662000000;	data_table[77].feg[0].cnl[2] = 3.1921000000;	data_table[77].feg[0].cl[3] = 3.9349000000;	data_table[77].feg[0].cnl[3] = 12.5285000000;	data_table[77].feg[0].cl[4] = 2.1709000000;	data_table[77].feg[0].cnl[4] = 49.7675000000;	data_table[77].feg[0].cl[5] = 0.0000000000;	data_table[77].feg[0].cnl[5] = 0.0000000000;
				data_table[78].feg[0].cl[0] = 0.3055000000;	data_table[78].feg[0].cnl[0] = 0.0596000000;	data_table[78].feg[0].cl[1] = 1.3945000000;	data_table[78].feg[0].cnl[1] = 0.5827000000;	data_table[78].feg[0].cl[2] = 2.9617000000;	data_table[78].feg[0].cnl[2] = 3.1035000000;	data_table[78].feg[0].cl[3] = 3.8990000000;	data_table[78].feg[0].cnl[3] = 11.9693000000;	data_table[78].feg[0].cl[4] = 2.0026000000;	data_table[78].feg[0].cnl[4] = 47.9106000000;	data_table[78].feg[0].cl[5] = 0.0000000000;	data_table[78].feg[0].cnl[5] = 0.0000000000;
				data_table[79].feg[0].cl[0] = 0.3593000000;	data_table[79].feg[0].cnl[0] = 0.0694000000;	data_table[79].feg[0].cl[1] = 1.5736000000;	data_table[79].feg[0].cnl[1] = 0.6758000000;	data_table[79].feg[0].cl[2] = 3.5237000000;	data_table[79].feg[0].cnl[2] = 3.8457000000;	data_table[79].feg[0].cl[3] = 3.8109000000;	data_table[79].feg[0].cnl[3] = 15.6203000000;	data_table[79].feg[0].cl[4] = 1.6953000000;	data_table[79].feg[0].cnl[4] = 56.6614000000;	data_table[79].feg[0].cl[5] = 0.0000000000;	data_table[79].feg[0].cnl[5] = 0.0000000000;
				data_table[80].feg[0].cl[0] = 0.3511000000;	data_table[80].feg[0].cnl[0] = 0.0672000000;	data_table[80].feg[0].cl[1] = 1.5489000000;	data_table[80].feg[0].cnl[1] = 0.6522000000;	data_table[80].feg[0].cl[2] = 3.5676000000;	data_table[80].feg[0].cnl[2] = 3.7420000000;	data_table[80].feg[0].cl[3] = 4.0900000000;	data_table[80].feg[0].cnl[3] = 15.9791000000;	data_table[80].feg[0].cl[4] = 2.5251000000;	data_table[80].feg[0].cnl[4] = 65.1354000000;	data_table[80].feg[0].cl[5] = 0.0000000000;	data_table[80].feg[0].cnl[5] = 0.0000000000;
				data_table[81].feg[0].cl[0] = 0.3540000000;	data_table[81].feg[0].cnl[0] = 0.0668000000;	data_table[81].feg[0].cl[1] = 1.5453000000;	data_table[81].feg[0].cnl[1] = 0.6465000000;	data_table[81].feg[0].cl[2] = 3.5975000000;	data_table[81].feg[0].cnl[2] = 3.6968000000;	data_table[81].feg[0].cl[3] = 4.3152000000;	data_table[81].feg[0].cnl[3] = 16.2056000000;	data_table[81].feg[0].cl[4] = 2.7743000000;	data_table[81].feg[0].cnl[4] = 61.4909000000;	data_table[81].feg[0].cl[5] = 0.0000000000;	data_table[81].feg[0].cnl[5] = 0.0000000000;
				data_table[82].feg[0].cl[0] = 0.3530000000;	data_table[82].feg[0].cnl[0] = 0.0661000000;	data_table[82].feg[0].cl[1] = 1.5258000000;	data_table[82].feg[0].cnl[1] = 0.6324000000;	data_table[82].feg[0].cl[2] = 3.5815000000;	data_table[82].feg[0].cnl[2] = 3.5906000000;	data_table[82].feg[0].cl[3] = 4.5532000000;	data_table[82].feg[0].cnl[3] = 15.9962000000;	data_table[82].feg[0].cl[4] = 3.0714000000;	data_table[82].feg[0].cnl[4] = 57.5760000000;	data_table[82].feg[0].cl[5] = 0.0000000000;	data_table[82].feg[0].cnl[5] = 0.0000000000;
				data_table[83].feg[0].cl[0] = 0.3673000000;	data_table[83].feg[0].cnl[0] = 0.0678000000;	data_table[83].feg[0].cl[1] = 1.5772000000;	data_table[83].feg[0].cnl[1] = 0.6527000000;	data_table[83].feg[0].cl[2] = 3.7079000000;	data_table[83].feg[0].cnl[2] = 3.7396000000;	data_table[83].feg[0].cl[3] = 4.8582000000;	data_table[83].feg[0].cnl[3] = 17.0668000000;	data_table[83].feg[0].cl[4] = 2.8440000000;	data_table[83].feg[0].cnl[4] = 55.9789000000;	data_table[83].feg[0].cl[5] = 0.0000000000;	data_table[83].feg[0].cnl[5] = 0.0000000000;
				data_table[84].feg[0].cl[0] = 0.3547000000;	data_table[84].feg[0].cnl[0] = 0.0649000000;	data_table[84].feg[0].cl[1] = 1.5206000000;	data_table[84].feg[0].cnl[1] = 0.6188000000;	data_table[84].feg[0].cl[2] = 3.5621000000;	data_table[84].feg[0].cnl[2] = 3.4696000000;	data_table[84].feg[0].cl[3] = 5.0184000000;	data_table[84].feg[0].cnl[3] = 15.6090000000;	data_table[84].feg[0].cl[4] = 3.0075000000;	data_table[84].feg[0].cnl[4] = 49.4818000000;	data_table[84].feg[0].cl[5] = 0.0000000000;	data_table[84].feg[0].cnl[5] = 0.0000000000;
				data_table[85].feg[0].cl[0] = 0.4586000000;	data_table[85].feg[0].cnl[0] = 0.0831000000;	data_table[85].feg[0].cl[1] = 1.7781000000;	data_table[85].feg[0].cnl[1] = 0.7840000000;	data_table[85].feg[0].cl[2] = 3.9877000000;	data_table[85].feg[0].cnl[2] = 4.3599000000;	data_table[85].feg[0].cl[3] = 5.7273000000;	data_table[85].feg[0].cnl[3] = 20.0128000000;	data_table[85].feg[0].cl[4] = 1.5460000000;	data_table[85].feg[0].cnl[4] = 62.1535000000;	data_table[85].feg[0].cl[5] = 0.0000000000;	data_table[85].feg[0].cnl[5] = 0.0000000000;
				data_table[86].feg[0].cl[0] = 0.8282000000;	data_table[86].feg[0].cnl[0] = 0.1515000000;	data_table[86].feg[0].cl[1] = 2.9941000000;	data_table[86].feg[0].cnl[1] = 1.6163000000;	data_table[86].feg[0].cl[2] = 5.6597000000;	data_table[86].feg[0].cnl[2] = 9.7752000000;	data_table[86].feg[0].cl[3] = 4.9292000000;	data_table[86].feg[0].cnl[3] = 42.8480000000;	data_table[86].feg[0].cl[4] = 4.2889000000;	data_table[86].feg[0].cnl[4] = 190.7366000000;	data_table[86].feg[0].cl[5] = 0.0000000000;	data_table[86].feg[0].cnl[5] = 0.0000000000;
				data_table[87].feg[0].cl[0] = 1.4129000000;	data_table[87].feg[0].cnl[0] = 0.2921000000;	data_table[87].feg[0].cl[1] = 4.4269000000;	data_table[87].feg[0].cnl[1] = 3.1381000000;	data_table[87].feg[0].cl[2] = 7.0460000000;	data_table[87].feg[0].cnl[2] = 19.6767000000;	data_table[87].feg[0].cl[3] = -1.0573000000;	data_table[87].feg[0].cnl[3] = 102.0436000000;	data_table[87].feg[0].cl[4] = 8.6430000000;	data_table[87].feg[0].cnl[4] = 113.9798000000;	data_table[87].feg[0].cl[5] = 0.0000000000;	data_table[87].feg[0].cnl[5] = 0.0000000000;
				data_table[88].feg[0].cl[0] = 0.7169000000;	data_table[88].feg[0].cnl[0] = 0.1263000000;	data_table[88].feg[0].cl[1] = 2.5710000000;	data_table[88].feg[0].cnl[1] = 1.2900000000;	data_table[88].feg[0].cl[2] = 5.1791000000;	data_table[88].feg[0].cnl[2] = 7.3686000000;	data_table[88].feg[0].cl[3] = 6.3484000000;	data_table[88].feg[0].cnl[3] = 32.4490000000;	data_table[88].feg[0].cl[4] = 5.6474000000;	data_table[88].feg[0].cnl[4] = 118.0558000000;	data_table[88].feg[0].cl[5] = 0.0000000000;	data_table[88].feg[0].cnl[5] = 0.0000000000;
				data_table[89].feg[0].cl[0] = 0.6958000000;	data_table[89].feg[0].cnl[0] = 0.1211000000;	data_table[89].feg[0].cl[1] = 2.4936000000;	data_table[89].feg[0].cnl[1] = 1.2247000000;	data_table[89].feg[0].cl[2] = 5.1269000000;	data_table[89].feg[0].cnl[2] = 6.9398000000;	data_table[89].feg[0].cl[3] = 6.6988000000;	data_table[89].feg[0].cnl[3] = 30.0991000000;	data_table[89].feg[0].cl[4] = 5.0799000000;	data_table[89].feg[0].cnl[4] = 105.1960000000;	data_table[89].feg[0].cl[5] = 0.0000000000;	data_table[89].feg[0].cnl[5] = 0.0000000000;
				data_table[90].feg[0].cl[0] = 1.2502000000;	data_table[90].feg[0].cnl[0] = 0.2415000000;	data_table[90].feg[0].cl[1] = 4.2284000000;	data_table[90].feg[0].cnl[1] = 2.6442000000;	data_table[90].feg[0].cl[2] = 7.0489000000;	data_table[90].feg[0].cnl[2] = 16.3313000000;	data_table[90].feg[0].cl[3] = 1.1390000000;	data_table[90].feg[0].cnl[3] = 73.5757000000;	data_table[90].feg[0].cl[4] = 5.8222000000;	data_table[90].feg[0].cnl[4] = 91.9401000000;	data_table[90].feg[0].cl[5] = 0.0000000000;	data_table[90].feg[0].cnl[5] = 0.0000000000;
				data_table[91].feg[0].cl[0] = 0.6410000000;	data_table[91].feg[0].cnl[0] = 0.1097000000;	data_table[91].feg[0].cl[1] = 2.2643000000;	data_table[91].feg[0].cnl[1] = 1.0644000000;	data_table[91].feg[0].cl[2] = 4.8713000000;	data_table[91].feg[0].cnl[2] = 5.7907000000;	data_table[91].feg[0].cl[3] = 5.9287000000;	data_table[91].feg[0].cnl[3] = 25.0261000000;	data_table[91].feg[0].cl[4] = 5.3935000000;	data_table[91].feg[0].cnl[4] = 101.3899000000;	data_table[91].feg[0].cl[5] = 0.0000000000;	data_table[91].feg[0].cnl[5] = 0.0000000000;
				data_table[92].feg[0].cl[0] = 0.6938000000;	data_table[92].feg[0].cnl[0] = 0.1171000000;	data_table[92].feg[0].cl[1] = 2.4652000000;	data_table[92].feg[0].cnl[1] = 1.1757000000;	data_table[92].feg[0].cl[2] = 5.1227000000;	data_table[92].feg[0].cnl[2] = 6.4053000000;	data_table[92].feg[0].cl[3] = 5.5965000000;	data_table[92].feg[0].cnl[3] = 27.5217000000;	data_table[92].feg[0].cl[4] = 4.8543000000;	data_table[92].feg[0].cnl[4] = 103.0482000000;	data_table[92].feg[0].cl[5] = 0.0000000000;	data_table[92].feg[0].cnl[5] = 0.0000000000;
				data_table[93].feg[0].cl[0] = 0.6902000000;	data_table[93].feg[0].cnl[0] = 0.1153000000;	data_table[93].feg[0].cl[1] = 2.4509000000;	data_table[93].feg[0].cnl[1] = 1.1545000000;	data_table[93].feg[0].cl[2] = 5.1284000000;	data_table[93].feg[0].cnl[2] = 6.2291000000;	data_table[93].feg[0].cl[3] = 5.0339000000;	data_table[93].feg[0].cnl[3] = 27.0741000000;	data_table[93].feg[0].cl[4] = 4.8575000000;	data_table[93].feg[0].cnl[4] = 111.3150000000;	data_table[93].feg[0].cl[5] = 0.0000000000;	data_table[93].feg[0].cnl[5] = 0.0000000000;
				data_table[94].feg[0].cl[0] = 0.7577000000;	data_table[94].feg[0].cnl[0] = 0.1257000000;	data_table[94].feg[0].cl[1] = 2.7264000000;	data_table[94].feg[0].cnl[1] = 1.3044000000;	data_table[94].feg[0].cl[2] = 5.4184000000;	data_table[94].feg[0].cnl[2] = 7.1035000000;	data_table[94].feg[0].cl[3] = 4.8198000000;	data_table[94].feg[0].cnl[3] = 32.4649000000;	data_table[94].feg[0].cl[4] = 4.1013000000;	data_table[94].feg[0].cnl[4] = 118.8647000000;	data_table[94].feg[0].cl[5] = 0.0000000000;	data_table[94].feg[0].cnl[5] = 0.0000000000;
				data_table[95].feg[0].cl[0] = 0.7567000000;	data_table[95].feg[0].cnl[0] = 0.1239000000;	data_table[95].feg[0].cl[1] = 2.7565000000;	data_table[95].feg[0].cnl[1] = 1.2979000000;	data_table[95].feg[0].cl[2] = 5.4364000000;	data_table[95].feg[0].cnl[2] = 7.0798000000;	data_table[95].feg[0].cl[3] = 5.1918000000;	data_table[95].feg[0].cnl[3] = 32.7871000000;	data_table[95].feg[0].cl[4] = 3.5643000000;	data_table[95].feg[0].cnl[4] = 110.1512000000;	data_table[95].feg[0].cl[5] = 0.0000000000;	data_table[95].feg[0].cnl[5] = 0.0000000000;
				data_table[96].feg[0].cl[0] = 0.7492000000;	data_table[96].feg[0].cnl[0] = 0.1217000000;	data_table[96].feg[0].cl[1] = 2.7267000000;	data_table[96].feg[0].cnl[1] = 1.2651000000;	data_table[96].feg[0].cl[2] = 5.3521000000;	data_table[96].feg[0].cnl[2] = 6.8101000000;	data_table[96].feg[0].cl[3] = 5.0369000000;	data_table[96].feg[0].cnl[3] = 31.6088000000;	data_table[96].feg[0].cl[4] = 3.5321000000;	data_table[96].feg[0].cnl[4] = 106.4853000000;	data_table[96].feg[0].cl[5] = 0.0000000000;	data_table[96].feg[0].cnl[5] = 0.0000000000;
				data_table[97].feg[0].cl[0] = 0.8100000000;	data_table[97].feg[0].cnl[0] = 0.1310000000;	data_table[97].feg[0].cl[1] = 3.0001000000;	data_table[97].feg[0].cnl[1] = 1.4038000000;	data_table[97].feg[0].cl[2] = 5.4635000000;	data_table[97].feg[0].cnl[2] = 7.6057000000;	data_table[97].feg[0].cl[3] = 4.1756000000;	data_table[97].feg[0].cnl[3] = 34.0186000000;	data_table[97].feg[0].cl[4] = 3.5066000000;	data_table[97].feg[0].cnl[4] = 90.5226000000;	data_table[97].feg[0].cl[5] = 0.0000000000;	data_table[97].feg[0].cnl[5] = 0.0000000000;
				data_table[98].feg[0].cl[0] = 0.0000000000;	data_table[98].feg[0].cnl[0] = 0.0000000000;	data_table[98].feg[0].cl[1] = 0.0000000000;	data_table[98].feg[0].cnl[1] = 0.0000000000;	data_table[98].feg[0].cl[2] = 0.0000000000;	data_table[98].feg[0].cnl[2] = 0.0000000000;	data_table[98].feg[0].cl[3] = 0.0000000000;	data_table[98].feg[0].cnl[3] = 0.0000000000;	data_table[98].feg[0].cl[4] = 0.0000000000;	data_table[98].feg[0].cnl[4] = 0.0000000000;	data_table[98].feg[0].cl[5] = 0.0000000000;	data_table[98].feg[0].cnl[5] = 0.0000000000;
				data_table[99].feg[0].cl[0] = 0.0000000000;	data_table[99].feg[0].cnl[0] = 0.0000000000;	data_table[99].feg[0].cl[1] = 0.0000000000;	data_table[99].feg[0].cnl[1] = 0.0000000000;	data_table[99].feg[0].cl[2] = 0.0000000000;	data_table[99].feg[0].cnl[2] = 0.0000000000;	data_table[99].feg[0].cl[3] = 0.0000000000;	data_table[99].feg[0].cnl[3] = 0.0000000000;	data_table[99].feg[0].cl[4] = 0.0000000000;	data_table[99].feg[0].cnl[4] = 0.0000000000;	data_table[99].feg[0].cl[5] = 0.0000000000;	data_table[99].feg[0].cnl[5] = 0.0000000000;
				data_table[100].feg[0].cl[0] = 0.0000000000;	data_table[100].feg[0].cnl[0] = 0.0000000000;	data_table[100].feg[0].cl[1] = 0.0000000000;	data_table[100].feg[0].cnl[1] = 0.0000000000;	data_table[100].feg[0].cl[2] = 0.0000000000;	data_table[100].feg[0].cnl[2] = 0.0000000000;	data_table[100].feg[0].cl[3] = 0.0000000000;	data_table[100].feg[0].cnl[3] = 0.0000000000;	data_table[100].feg[0].cl[4] = 0.0000000000;	data_table[100].feg[0].cnl[4] = 0.0000000000;	data_table[100].feg[0].cl[5] = 0.0000000000;	data_table[100].feg[0].cnl[5] = 0.0000000000;
				data_table[101].feg[0].cl[0] = 0.0000000000;	data_table[101].feg[0].cnl[0] = 0.0000000000;	data_table[101].feg[0].cl[1] = 0.0000000000;	data_table[101].feg[0].cnl[1] = 0.0000000000;	data_table[101].feg[0].cl[2] = 0.0000000000;	data_table[101].feg[0].cnl[2] = 0.0000000000;	data_table[101].feg[0].cl[3] = 0.0000000000;	data_table[101].feg[0].cnl[3] = 0.0000000000;	data_table[101].feg[0].cl[4] = 0.0000000000;	data_table[101].feg[0].cnl[4] = 0.0000000000;	data_table[101].feg[0].cl[5] = 0.0000000000;	data_table[101].feg[0].cnl[5] = 0.0000000000;
				data_table[102].feg[0].cl[0] = 0.0000000000;	data_table[102].feg[0].cnl[0] = 0.0000000000;	data_table[102].feg[0].cl[1] = 0.0000000000;	data_table[102].feg[0].cnl[1] = 0.0000000000;	data_table[102].feg[0].cl[2] = 0.0000000000;	data_table[102].feg[0].cnl[2] = 0.0000000000;	data_table[102].feg[0].cl[3] = 0.0000000000;	data_table[102].feg[0].cnl[3] = 0.0000000000;	data_table[102].feg[0].cl[4] = 0.0000000000;	data_table[102].feg[0].cnl[4] = 0.0000000000;	data_table[102].feg[0].cl[5] = 0.0000000000;	data_table[102].feg[0].cnl[5] = 0.0000000000;
			}

			// 4: Kirkland parameterization - 3 Yukawa + 3 Gaussians - [0, 12]
			void Load_feg_Kirkland_neutral_0_12()
			{
				data_table[0].feg[0].cl[0] = 0.0042029832;		data_table[0].feg[0].cnl[0] = 0.2253508880;	data_table[0].feg[0].cl[1] = 0.0627762505;		data_table[0].feg[0].cnl[1] = 0.2253669500;	data_table[0].feg[0].cl[2] = 0.0300907347;		data_table[0].feg[0].cnl[2] = 0.2253317560;	data_table[0].feg[0].cl[3] = 0.0677756695;		data_table[0].feg[0].cnl[3] = 4.3885400100;	data_table[0].feg[0].cl[4] = 0.0035660924;		data_table[0].feg[0].cnl[4] = 0.4038848230;	data_table[0].feg[0].cl[5] = 0.0276135815;		data_table[0].feg[0].cnl[5] = 1.4449016600;
				data_table[1].feg[0].cl[0] = 0.0000187544;		data_table[1].feg[0].cnl[0] = 0.2124279970;	data_table[1].feg[0].cl[1] = 0.0004105958;		data_table[1].feg[0].cnl[1] = 0.3322122790;	data_table[1].feg[0].cl[2] = 0.1963000590;		data_table[1].feg[0].cnl[2] = 0.5173251520;	data_table[1].feg[0].cl[3] = 0.0083601574;		data_table[1].feg[0].cnl[3] = 0.3666682390;	data_table[1].feg[0].cl[4] = 0.0295102022;		data_table[1].feg[0].cnl[4] = 1.3717182700;	data_table[1].feg[0].cl[5] = 0.0000004659;		data_table[1].feg[0].cnl[5] = 37576.8025000000;
				data_table[2].feg[0].cl[0] = 0.0745843816;		data_table[2].feg[0].cnl[0] = 0.8811514240;	data_table[2].feg[0].cl[1] = 0.0715382250;		data_table[2].feg[0].cnl[1] = 0.0459142904;	data_table[2].feg[0].cl[2] = 0.1453152290;		data_table[2].feg[0].cnl[2] = 0.8813017140;	data_table[2].feg[0].cl[3] = 1.1212576900;		data_table[2].feg[0].cnl[3] = 18.8483665000;	data_table[2].feg[0].cl[4] = 0.0025173653;		data_table[2].feg[0].cnl[4] = 0.1591899950;	data_table[2].feg[0].cl[5] = 0.3584349710;		data_table[2].feg[0].cnl[5] = 6.1237100000;
				data_table[3].feg[0].cl[0] = 0.0611642897;		data_table[3].feg[0].cnl[0] = 0.0990182132;	data_table[3].feg[0].cl[1] = 0.1257550340;		data_table[3].feg[0].cnl[1] = 0.0990272412;	data_table[3].feg[0].cl[2] = 0.2008315480;		data_table[3].feg[0].cnl[2] = 1.8739250900;	data_table[3].feg[0].cl[3] = 0.7872428760;		data_table[3].feg[0].cnl[3] = 9.3279492900;	data_table[3].feg[0].cl[4] = 0.0015884785;		data_table[3].feg[0].cnl[4] = 0.0891900236;	data_table[3].feg[0].cl[5] = 0.2739620310;		data_table[3].feg[0].cnl[5] = 3.2068765800;
				data_table[4].feg[0].cl[0] = 0.1257160660;		data_table[4].feg[0].cnl[0] = 0.1482588300;	data_table[4].feg[0].cl[1] = 0.1733144520;		data_table[4].feg[0].cnl[1] = 0.1482572160;	data_table[4].feg[0].cl[2] = 0.1847748110;		data_table[4].feg[0].cnl[2] = 3.3422731100;	data_table[4].feg[0].cl[3] = 0.1952502210;		data_table[4].feg[0].cnl[3] = 1.9733946300;	data_table[4].feg[0].cl[4] = 0.5296420750;		data_table[4].feg[0].cnl[4] = 5.7003555300;	data_table[4].feg[0].cl[5] = 0.0010823050;		data_table[4].feg[0].cnl[5] = 0.0564857237;
				data_table[5].feg[0].cl[0] = 0.2120807670;		data_table[5].feg[0].cnl[0] = 0.2086054170;	data_table[5].feg[0].cl[1] = 0.1998118650;		data_table[5].feg[0].cnl[1] = 0.2086101860;	data_table[5].feg[0].cl[2] = 0.1682543850;		data_table[5].feg[0].cnl[2] = 5.5787077300;	data_table[5].feg[0].cl[3] = 0.1420483600;		data_table[5].feg[0].cnl[3] = 1.3331188700;	data_table[5].feg[0].cl[4] = 0.3638306720;		data_table[5].feg[0].cnl[4] = 3.8080026300;	data_table[5].feg[0].cl[5] = 0.0008350120;		data_table[5].feg[0].cnl[5] = 0.0403982620;
				data_table[6].feg[0].cl[0] = 0.5330155540;		data_table[6].feg[0].cnl[0] = 0.2909525150;	data_table[6].feg[0].cl[1] = 0.0529008883;		data_table[6].feg[0].cnl[1] = 10.3547896000;	data_table[6].feg[0].cl[2] = 0.0924159648;		data_table[6].feg[0].cnl[2] = 10.3540028000;	data_table[6].feg[0].cl[3] = 0.2617991010;		data_table[6].feg[0].cnl[3] = 2.7625272300;	data_table[6].feg[0].cl[4] = 0.0008802621;		data_table[6].feg[0].cnl[4] = 0.0347681236;	data_table[6].feg[0].cl[5] = 0.1101665550;		data_table[6].feg[0].cnl[5] = 0.9934217360;
				data_table[7].feg[0].cl[0] = 0.3399692040;		data_table[7].feg[0].cnl[0] = 0.3815702800;	data_table[7].feg[0].cl[1] = 0.3075701720;		data_table[7].feg[0].cnl[1] = 0.3815714360;	data_table[7].feg[0].cl[2] = 0.1303690720;		data_table[7].feg[0].cnl[2] = 19.1919745000;	data_table[7].feg[0].cl[3] = 0.0883326058;		data_table[7].feg[0].cnl[3] = 0.7606355250;	data_table[7].feg[0].cl[4] = 0.1965867000;		data_table[7].feg[0].cnl[4] = 2.0740109400;	data_table[7].feg[0].cl[5] = 0.0009962200;		data_table[7].feg[0].cnl[5] = 0.0303266869;
				data_table[8].feg[0].cl[0] = 0.2305605930;		data_table[8].feg[0].cnl[0] = 0.4807542130;	data_table[8].feg[0].cl[1] = 0.5268896480;		data_table[8].feg[0].cnl[1] = 0.4807638950;	data_table[8].feg[0].cl[2] = 0.1243467550;		data_table[8].feg[0].cnl[2] = 39.5306720000;	data_table[8].feg[0].cl[3] = 0.0012461689;		data_table[8].feg[0].cnl[3] = 0.0262181803;	data_table[8].feg[0].cl[4] = 0.0720452555;		data_table[8].feg[0].cnl[4] = 0.5924955930;	data_table[8].feg[0].cl[5] = 0.1530757770;		data_table[8].feg[0].cnl[5] = 1.5912767100;
				data_table[9].feg[0].cl[0] = 0.4083717710;		data_table[9].feg[0].cnl[0] = 0.5882286270;	data_table[9].feg[0].cl[1] = 0.4544188580;		data_table[9].feg[0].cnl[1] = 0.5882886550;	data_table[9].feg[0].cl[2] = 0.1445649230;		data_table[9].feg[0].cnl[2] = 121.2460130000;	data_table[9].feg[0].cl[3] = 0.0591531395;		data_table[9].feg[0].cnl[3] = 0.4639635400;	data_table[9].feg[0].cl[4] = 0.1240037180;		data_table[9].feg[0].cnl[4] = 1.2341302500;	data_table[9].feg[0].cl[5] = 0.0016498604;		data_table[9].feg[0].cnl[5] = 0.0205869217;
				data_table[10].feg[0].cl[0] = 0.1364716620;	data_table[10].feg[0].cnl[0] = 0.0499965301;	data_table[10].feg[0].cl[1] = 0.7706778650;	data_table[10].feg[0].cnl[1] = 0.8818996640;	data_table[10].feg[0].cl[2] = 0.1568620140;	data_table[10].feg[0].cnl[2] = 16.1768579000;	data_table[10].feg[0].cl[3] = 0.9968215130;	data_table[10].feg[0].cnl[3] = 20.0132610000;	data_table[10].feg[0].cl[4] = 0.0380304670;	data_table[10].feg[0].cnl[4] = 0.2605162540;	data_table[10].feg[0].cl[5] = 0.1276850890;	data_table[10].feg[0].cnl[5] = 0.6995593290;
				data_table[11].feg[0].cl[0] = 0.3043841210;	data_table[11].feg[0].cnl[0] = 0.0842014377;	data_table[11].feg[0].cl[1] = 0.7562705630;	data_table[11].feg[0].cnl[1] = 1.6406559800;	data_table[11].feg[0].cl[2] = 0.1011648090;	data_table[11].feg[0].cnl[2] = 29.7142975000;	data_table[11].feg[0].cl[3] = 0.0345203403;	data_table[11].feg[0].cnl[3] = 0.2165960940;	data_table[11].feg[0].cl[4] = 0.9717513270;	data_table[11].feg[0].cnl[4] = 12.1236852000;	data_table[11].feg[0].cl[5] = 0.1205930120;	data_table[11].feg[0].cnl[5] = 0.5608658380;
				data_table[12].feg[0].cl[0] = 0.7774194240;	data_table[12].feg[0].cnl[0] = 2.7105822700;	data_table[12].feg[0].cl[1] = 0.0578312036;	data_table[12].feg[0].cnl[1] = 71.7532098000;	data_table[12].feg[0].cl[2] = 0.4263864990;	data_table[12].feg[0].cnl[2] = 0.0913331555;	data_table[12].feg[0].cl[3] = 0.1134072200;	data_table[12].feg[0].cnl[3] = 0.4488674510;	data_table[12].feg[0].cl[4] = 0.7901140350;	data_table[12].feg[0].cnl[4] = 8.6636671800;	data_table[12].feg[0].cl[5] = 0.0323293496;	data_table[12].feg[0].cnl[5] = 0.1785034630;
				data_table[13].feg[0].cl[0] = 1.0654389200;	data_table[13].feg[0].cnl[0] = 1.0411845500;	data_table[13].feg[0].cl[1] = 0.1201436910;	data_table[13].feg[0].cnl[1] = 68.7113368000;	data_table[13].feg[0].cl[2] = 0.1809152630;	data_table[13].feg[0].cnl[2] = 0.0887533926;	data_table[13].feg[0].cl[3] = 1.1206562000;	data_table[13].feg[0].cnl[3] = 3.7006261900;	data_table[13].feg[0].cl[4] = 0.0305452816;	data_table[13].feg[0].cnl[4] = 0.2140978970;	data_table[13].feg[0].cl[5] = 1.5996350200;	data_table[13].feg[0].cnl[5] = 9.9909663800;
				data_table[14].feg[0].cl[0] = 1.0528444700;	data_table[14].feg[0].cnl[0] = 1.3196259000;	data_table[14].feg[0].cl[1] = 0.2994402840;	data_table[14].feg[0].cnl[1] = 0.1284605200;	data_table[14].feg[0].cl[2] = 0.1174607480;	data_table[14].feg[0].cnl[2] = 102.1901630000;	data_table[14].feg[0].cl[3] = 0.9606434520;	data_table[14].feg[0].cnl[3] = 2.8747755500;	data_table[14].feg[0].cl[4] = 0.0263555748;	data_table[14].feg[0].cnl[4] = 0.1820768440;	data_table[14].feg[0].cl[5] = 1.3805933000;	data_table[14].feg[0].cnl[5] = 7.4916552600;
				data_table[15].feg[0].cl[0] = 1.0164691600;	data_table[15].feg[0].cnl[0] = 1.6918196500;	data_table[15].feg[0].cl[1] = 0.4417667480;	data_table[15].feg[0].cnl[1] = 0.1741802880;	data_table[15].feg[0].cl[2] = 0.1215038630;	data_table[15].feg[0].cnl[2] = 167.0110910000;	data_table[15].feg[0].cl[3] = 0.8279666700;	data_table[15].feg[0].cnl[3] = 2.3034281000;	data_table[15].feg[0].cl[4] = 0.0233022533;	data_table[15].feg[0].cnl[4] = 0.1569541500;	data_table[15].feg[0].cl[5] = 1.1830284600;	data_table[15].feg[0].cnl[5] = 5.8578289100;
				data_table[16].feg[0].cl[0] = 0.9442211160;	data_table[16].feg[0].cnl[0] = 0.2400523740;	data_table[16].feg[0].cl[1] = 0.4373220490;	data_table[16].feg[0].cnl[1] = 9.3051043900;	data_table[16].feg[0].cl[2] = 0.2545479260;	data_table[16].feg[0].cnl[2] = 9.3048634600;	data_table[16].feg[0].cl[3] = 0.0547763323;	data_table[16].feg[0].cnl[3] = 0.1686556880;	data_table[16].feg[0].cl[4] = 0.8000874880;	data_table[16].feg[0].cnl[4] = 2.9784977400;	data_table[16].feg[0].cl[5] = 0.0107488641;	data_table[16].feg[0].cnl[5] = 0.0684240646;
				data_table[17].feg[0].cl[0] = 1.0698328800;	data_table[17].feg[0].cnl[0] = 0.2877910220;	data_table[17].feg[0].cl[1] = 0.4246317860;	data_table[17].feg[0].cnl[1] = 12.4156957000;	data_table[17].feg[0].cl[2] = 0.2438979490;	data_table[17].feg[0].cnl[2] = 12.4158868000;	data_table[17].feg[0].cl[3] = 0.0479446296;	data_table[17].feg[0].cnl[3] = 0.1369797960;	data_table[17].feg[0].cl[4] = 0.7649589520;	data_table[17].feg[0].cnl[4] = 2.4394072900;	data_table[17].feg[0].cl[5] = 0.0082312843;	data_table[17].feg[0].cnl[5] = 0.0527258749;
				data_table[18].feg[0].cl[0] = 0.6927178650;	data_table[18].feg[0].cnl[0] = 7.1084999000;	data_table[18].feg[0].cl[1] = 0.9651610850;	data_table[18].feg[0].cnl[1] = 0.3575329010;	data_table[18].feg[0].cl[2] = 0.1484665880;	data_table[18].feg[0].cnl[2] = 0.0393763275;	data_table[18].feg[0].cl[3] = 0.0264645027;	data_table[18].feg[0].cnl[3] = 0.1035913210;	data_table[18].feg[0].cl[4] = 1.8088376800;	data_table[18].feg[0].cnl[4] = 32.2845199000;	data_table[18].feg[0].cl[5] = 0.5439000180;	data_table[18].feg[0].cnl[5] = 1.6779137400;
				data_table[19].feg[0].cl[0] = 0.3669028710;	data_table[19].feg[0].cnl[0] = 0.0614274129;	data_table[19].feg[0].cl[1] = 0.8663789990;	data_table[19].feg[0].cnl[1] = 0.5708817270;	data_table[19].feg[0].cl[2] = 0.6672033000;	data_table[19].feg[0].cnl[2] = 7.8296563900;	data_table[19].feg[0].cl[3] = 0.4877436360;	data_table[19].feg[0].cnl[3] = 1.3253131800;	data_table[19].feg[0].cl[4] = 1.8240631400;	data_table[19].feg[0].cnl[4] = 21.0056032000;	data_table[19].feg[0].cl[5] = 0.0220248453;	data_table[19].feg[0].cnl[5] = 0.0911853450;
				data_table[20].feg[0].cl[0] = 0.3788717770;	data_table[20].feg[0].cnl[0] = 0.0698910162;	data_table[20].feg[0].cl[1] = 0.9000225050;	data_table[20].feg[0].cnl[1] = 0.5210615410;	data_table[20].feg[0].cl[2] = 0.7152889140;	data_table[20].feg[0].cnl[2] = 7.8770792000;	data_table[20].feg[0].cl[3] = 0.0188640973;	data_table[20].feg[0].cnl[3] = 0.0817512708;	data_table[20].feg[0].cl[4] = 0.4079459490;	data_table[20].feg[0].cnl[4] = 1.1114138800;	data_table[20].feg[0].cl[5] = 1.6178654000;	data_table[20].feg[0].cnl[5] = 18.0840759000;
				data_table[21].feg[0].cl[0] = 0.3623832670;	data_table[21].feg[0].cnl[0] = 0.0754707114;	data_table[21].feg[0].cl[1] = 0.9842329660;	data_table[21].feg[0].cnl[1] = 0.4977573090;	data_table[21].feg[0].cl[2] = 0.7417156420;	data_table[21].feg[0].cnl[2] = 8.1765939100;	data_table[21].feg[0].cl[3] = 0.3625552690;	data_table[21].feg[0].cnl[3] = 0.9555249060;	data_table[21].feg[0].cl[4] = 1.4915939000;	data_table[21].feg[0].cnl[4] = 16.2221677000;	data_table[21].feg[0].cl[5] = 0.0161659509;	data_table[21].feg[0].cnl[5] = 0.0733140839;
				data_table[22].feg[0].cl[0] = 0.3529613780;	data_table[22].feg[0].cnl[0] = 0.0819204103;	data_table[22].feg[0].cl[1] = 0.7467910140;	data_table[22].feg[0].cnl[1] = 8.8118951100;	data_table[22].feg[0].cl[2] = 1.0836406800;	data_table[22].feg[0].cnl[2] = 0.5106460750;	data_table[22].feg[0].cl[3] = 1.3901361000;	data_table[22].feg[0].cnl[3] = 14.8901841000;	data_table[22].feg[0].cl[4] = 0.3312733560;	data_table[22].feg[0].cnl[4] = 0.8385430790;	data_table[22].feg[0].cl[5] = 0.0140422612;	data_table[22].feg[0].cnl[5] = 0.0657432678;
				data_table[23].feg[0].cl[0] = 1.3434837900;	data_table[23].feg[0].cnl[0] = 1.2581435300;	data_table[23].feg[0].cl[1] = 0.5070403280;	data_table[23].feg[0].cnl[1] = 11.5042811000;	data_table[23].feg[0].cl[2] = 0.4263589550;	data_table[23].feg[0].cnl[2] = 0.0853660389;	data_table[23].feg[0].cl[3] = 0.0117241826;	data_table[23].feg[0].cnl[3] = 0.0600177061;	data_table[23].feg[0].cl[4] = 0.5119665160;	data_table[23].feg[0].cnl[4] = 1.5377245100;	data_table[23].feg[0].cl[5] = 0.3382858280;	data_table[23].feg[0].cnl[5] = 0.6624183190;
				data_table[24].feg[0].cl[0] = 0.3266976130;	data_table[24].feg[0].cnl[0] = 0.0888813083;	data_table[24].feg[0].cl[1] = 0.7172970000;	data_table[24].feg[0].cnl[1] = 11.1300198000;	data_table[24].feg[0].cl[2] = 1.3321246400;	data_table[24].feg[0].cnl[2] = 0.5821411040;	data_table[24].feg[0].cl[3] = 0.2808017020;	data_table[24].feg[0].cnl[3] = 0.6715831450;	data_table[24].feg[0].cl[4] = 1.1549924100;	data_table[24].feg[0].cnl[4] = 12.6825395000;	data_table[24].feg[0].cl[5] = 0.0111984488;	data_table[24].feg[0].cnl[5] = 0.0532334467;
				data_table[25].feg[0].cl[0] = 0.3134548470;	data_table[25].feg[0].cnl[0] = 0.0899325756;	data_table[25].feg[0].cl[1] = 0.6892900160;	data_table[25].feg[0].cnl[1] = 13.0366038000;	data_table[25].feg[0].cl[2] = 1.4714153100;	data_table[25].feg[0].cnl[2] = 0.6333452910;	data_table[25].feg[0].cl[3] = 1.0329868800;	data_table[25].feg[0].cnl[3] = 11.6783425000;	data_table[25].feg[0].cl[4] = 0.2582802850;	data_table[25].feg[0].cnl[4] = 0.6091164460;	data_table[25].feg[0].cl[5] = 0.0103460690;	data_table[25].feg[0].cnl[5] = 0.0481610627;
				data_table[26].feg[0].cl[0] = 0.3158782780;	data_table[26].feg[0].cnl[0] = 0.0946683246;	data_table[26].feg[0].cl[1] = 1.6013900500;	data_table[26].feg[0].cnl[1] = 0.6994364490;	data_table[26].feg[0].cl[2] = 0.6563943380;	data_table[26].feg[0].cnl[2] = 15.6954403000;	data_table[26].feg[0].cl[3] = 0.9367466240;	data_table[26].feg[0].cnl[3] = 10.9392410000;	data_table[26].feg[0].cl[4] = 0.0097756265;	data_table[26].feg[0].cnl[4] = 0.0437446816;	data_table[26].feg[0].cl[5] = 0.2383785780;	data_table[26].feg[0].cnl[5] = 0.5562864830;
				data_table[27].feg[0].cl[0] = 1.7225463000;	data_table[27].feg[0].cnl[0] = 0.7766069080;	data_table[27].feg[0].cl[1] = 0.3295430440;	data_table[27].feg[0].cnl[1] = 0.1022623600;	data_table[27].feg[0].cl[2] = 0.6230072000;	data_table[27].feg[0].cnl[2] = 19.4156207000;	data_table[27].feg[0].cl[3] = 0.0094349651;	data_table[27].feg[0].cnl[3] = 0.0398684596;	data_table[27].feg[0].cl[4] = 0.8540635150;	data_table[27].feg[0].cnl[4] = 10.4078166000;	data_table[27].feg[0].cl[5] = 0.2210735150;	data_table[27].feg[0].cnl[5] = 0.5108693300;
				data_table[28].feg[0].cl[0] = 0.3587745310;	data_table[28].feg[0].cnl[0] = 0.1061534630;	data_table[28].feg[0].cl[1] = 1.7618134800;	data_table[28].feg[0].cnl[1] = 1.0164099500;	data_table[28].feg[0].cl[2] = 0.6369050530;	data_table[28].feg[0].cnl[2] = 15.3659093000;	data_table[28].feg[0].cl[3] = 0.0074493067;	data_table[28].feg[0].cnl[3] = 0.0385345989;	data_table[28].feg[0].cl[4] = 0.1890023470;	data_table[28].feg[0].cnl[4] = 0.3984277900;	data_table[28].feg[0].cl[5] = 0.2296195890;	data_table[28].feg[0].cnl[5] = 0.9014198430;
				data_table[29].feg[0].cl[0] = 0.5708939730;	data_table[29].feg[0].cnl[0] = 0.1265346140;	data_table[29].feg[0].cl[1] = 1.9890885600;	data_table[29].feg[0].cnl[1] = 2.1778196500;	data_table[29].feg[0].cl[2] = 0.3060605850;	data_table[29].feg[0].cnl[2] = 37.8619003000;	data_table[29].feg[0].cl[3] = 0.2356002230;	data_table[29].feg[0].cnl[3] = 0.3670190410;	data_table[29].feg[0].cl[4] = 0.3970611020;	data_table[29].feg[0].cnl[4] = 0.8664195960;	data_table[29].feg[0].cl[5] = 0.0068565723;	data_table[29].feg[0].cnl[5] = 0.0335778823;
				data_table[30].feg[0].cl[0] = 0.6255284640;	data_table[30].feg[0].cnl[0] = 0.1100056500;	data_table[30].feg[0].cl[1] = 2.0530290100;	data_table[30].feg[0].cnl[1] = 2.4109578600;	data_table[30].feg[0].cl[2] = 0.2896081200;	data_table[30].feg[0].cnl[2] = 47.8685736000;	data_table[30].feg[0].cl[3] = 0.2079105940;	data_table[30].feg[0].cnl[3] = 0.3278072240;	data_table[30].feg[0].cl[4] = 0.3450796170;	data_table[30].feg[0].cnl[4] = 0.7431390610;	data_table[30].feg[0].cl[5] = 0.0065563430;	data_table[30].feg[0].cnl[5] = 0.0309411369;
				data_table[31].feg[0].cl[0] = 0.5909526900;	data_table[31].feg[0].cnl[0] = 0.1183759760;	data_table[31].feg[0].cl[1] = 0.5399806600;	data_table[31].feg[0].cnl[1] = 71.8937433000;	data_table[31].feg[0].cl[2] = 2.0062618800;	data_table[31].feg[0].cnl[2] = 1.3930488900;	data_table[31].feg[0].cl[3] = 0.7497050410;	data_table[31].feg[0].cnl[3] = 6.8994335000;	data_table[31].feg[0].cl[4] = 0.1835813470;	data_table[31].feg[0].cnl[4] = 0.3646672320;	data_table[31].feg[0].cl[5] = 0.0095219074;	data_table[31].feg[0].cnl[5] = 0.0269888650;
				data_table[32].feg[0].cl[0] = 0.7778752180;	data_table[32].feg[0].cnl[0] = 0.1507331570;	data_table[32].feg[0].cl[1] = 0.5938481500;	data_table[32].feg[0].cnl[1] = 142.8822090000;	data_table[32].feg[0].cl[2] = 1.9591875100;	data_table[32].feg[0].cnl[2] = 1.7475033900;	data_table[32].feg[0].cl[3] = 0.1798802260;	data_table[32].feg[0].cnl[3] = 0.3318008520;	data_table[32].feg[0].cl[4] = 0.8632672220;	data_table[32].feg[0].cnl[4] = 5.8549027400;	data_table[32].feg[0].cl[5] = 0.0095905343;	data_table[32].feg[0].cnl[5] = 0.0233777569;
				data_table[33].feg[0].cl[0] = 0.9583906810;	data_table[33].feg[0].cnl[0] = 0.1837755570;	data_table[33].feg[0].cl[1] = 0.6038513420;	data_table[33].feg[0].cnl[1] = 196.8192240000;	data_table[33].feg[0].cl[2] = 1.9082893100;	data_table[33].feg[0].cnl[2] = 2.1508205300;	data_table[33].feg[0].cl[3] = 0.1738859560;	data_table[33].feg[0].cnl[3] = 0.3000060240;	data_table[33].feg[0].cl[4] = 0.9352651450;	data_table[33].feg[0].cnl[4] = 4.9247121500;	data_table[33].feg[0].cl[5] = 0.0086225466;	data_table[33].feg[0].cnl[5] = 0.0212308108;
				data_table[34].feg[0].cl[0] = 1.1413617000;	data_table[34].feg[0].cnl[0] = 0.2187087100;	data_table[34].feg[0].cl[1] = 0.5181187370;	data_table[34].feg[0].cnl[1] = 193.9166820000;	data_table[34].feg[0].cl[2] = 1.8573197500;	data_table[34].feg[0].cnl[2] = 2.6575539600;	data_table[34].feg[0].cl[3] = 0.1682173990;	data_table[34].feg[0].cnl[3] = 0.2717199180;	data_table[34].feg[0].cl[4] = 0.9757056060;	data_table[34].feg[0].cnl[4] = 4.1948250000;	data_table[34].feg[0].cl[5] = 0.0072418787;	data_table[34].feg[0].cnl[5] = 0.0199325718;
				data_table[35].feg[0].cl[0] = 0.3243869700;	data_table[35].feg[0].cnl[0] = 63.1317973000;	data_table[35].feg[0].cl[1] = 1.3173216300;	data_table[35].feg[0].cnl[1] = 0.2547060360;	data_table[35].feg[0].cl[2] = 1.7991261400;	data_table[35].feg[0].cnl[2] = 3.2366839400;	data_table[35].feg[0].cl[3] = 0.0042996143;	data_table[35].feg[0].cnl[3] = 0.0198965610;	data_table[35].feg[0].cl[4] = 1.0042943300;	data_table[35].feg[0].cnl[4] = 3.6109451300;	data_table[35].feg[0].cl[5] = 0.1621881970;	data_table[35].feg[0].cnl[5] = 0.2455836720;
				data_table[36].feg[0].cl[0] = 0.2904453510;	data_table[36].feg[0].cnl[0] = 0.0368420227;	data_table[36].feg[0].cl[1] = 2.4420132900;	data_table[36].feg[0].cnl[1] = 1.1601333200;	data_table[36].feg[0].cl[2] = 0.7694354490;	data_table[36].feg[0].cnl[2] = 16.9591472000;	data_table[36].feg[0].cl[3] = 1.5868700000;	data_table[36].feg[0].cnl[3] = 2.5308257400;	data_table[36].feg[0].cl[4] = 0.0028161759;	data_table[36].feg[0].cnl[4] = 0.0188577417;	data_table[36].feg[0].cl[5] = 0.1286638300;	data_table[36].feg[0].cnl[5] = 0.2107539690;
				data_table[37].feg[0].cl[0] = 0.0137373086;	data_table[37].feg[0].cnl[0] = 0.0187469061;	data_table[37].feg[0].cl[1] = 1.9754867200;	data_table[37].feg[0].cnl[1] = 6.3607923000;	data_table[37].feg[0].cl[2] = 1.5926102900;	data_table[37].feg[0].cnl[2] = 0.2219924820;	data_table[37].feg[0].cl[3] = 0.1732638820;	data_table[37].feg[0].cnl[3] = 0.2016249580;	data_table[37].feg[0].cl[4] = 4.6628037800;	data_table[37].feg[0].cnl[4] = 25.3027803000;	data_table[37].feg[0].cl[5] = 0.0016126506;	data_table[37].feg[0].cnl[5] = 0.0153610568;
				data_table[38].feg[0].cl[0] = 0.6753027470;	data_table[38].feg[0].cnl[0] = 0.0654331847;	data_table[38].feg[0].cl[1] = 0.4702867200;	data_table[38].feg[0].cnl[1] = 106.1087090000;	data_table[38].feg[0].cl[2] = 2.6349767700;	data_table[38].feg[0].cnl[2] = 2.0664354000;	data_table[38].feg[0].cl[3] = 0.1096217460;	data_table[38].feg[0].cnl[3] = 0.1931319250;	data_table[38].feg[0].cl[4] = 0.9603487730;	data_table[38].feg[0].cnl[4] = 1.6331093800;	data_table[38].feg[0].cl[5] = 0.0052892156;	data_table[38].feg[0].cnl[5] = 0.0166083821;
				data_table[39].feg[0].cl[0] = 2.6436550500;	data_table[39].feg[0].cnl[0] = 2.2020269900;	data_table[39].feg[0].cl[1] = 0.5542251470;	data_table[39].feg[0].cnl[1] = 178.2601070000;	data_table[39].feg[0].cl[2] = 0.7613766250;	data_table[39].feg[0].cnl[2] = 0.0767218745;	data_table[39].feg[0].cl[3] = 0.0060294689;	data_table[39].feg[0].cnl[3] = 0.0155143296;	data_table[39].feg[0].cl[4] = 0.0991630530;	data_table[39].feg[0].cnl[4] = 0.1761759950;	data_table[39].feg[0].cl[5] = 0.9567820200;	data_table[39].feg[0].cnl[5] = 1.5433068200;
				data_table[40].feg[0].cl[0] = 0.6595328750;	data_table[40].feg[0].cnl[0] = 0.0866145490;	data_table[40].feg[0].cl[1] = 1.8454585400;	data_table[40].feg[0].cnl[1] = 5.9477439800;	data_table[40].feg[0].cl[2] = 1.2558440500;	data_table[40].feg[0].cnl[2] = 0.6408514750;	data_table[40].feg[0].cl[3] = 0.1222534220;	data_table[40].feg[0].cnl[3] = 0.1666460500;	data_table[40].feg[0].cl[4] = 0.7066383280;	data_table[40].feg[0].cnl[4] = 1.6285326800;	data_table[40].feg[0].cl[5] = 0.0026238159;	data_table[40].feg[0].cnl[5] = 0.0082625786;
				data_table[41].feg[0].cl[0] = 0.6101601200;	data_table[41].feg[0].cnl[0] = 0.0911628054;	data_table[41].feg[0].cl[1] = 1.2654400000;	data_table[41].feg[0].cnl[1] = 0.5067760250;	data_table[41].feg[0].cl[2] = 1.9742876200;	data_table[41].feg[0].cnl[2] = 5.8959038100;	data_table[41].feg[0].cl[3] = 0.6480289620;	data_table[41].feg[0].cnl[3] = 1.4663410800;	data_table[41].feg[0].cl[4] = 0.0026038082;	data_table[41].feg[0].cnl[4] = 0.0078433631;	data_table[41].feg[0].cl[5] = 0.1138874930;	data_table[41].feg[0].cnl[5] = 0.1551143400;
				data_table[42].feg[0].cl[0] = 0.8551891830;	data_table[42].feg[0].cnl[0] = 0.1029621510;	data_table[42].feg[0].cl[1] = 1.6621964100;	data_table[42].feg[0].cnl[1] = 7.6490700000;	data_table[42].feg[0].cl[2] = 1.4557547500;	data_table[42].feg[0].cnl[2] = 1.0163998700;	data_table[42].feg[0].cl[3] = 0.1054456640;	data_table[42].feg[0].cnl[3] = 0.1423033380;	data_table[42].feg[0].cl[4] = 0.7716571120;	data_table[42].feg[0].cnl[4] = 1.3465934900;	data_table[42].feg[0].cl[5] = 0.0022099264;	data_table[42].feg[0].cnl[5] = 0.0079035898;
				data_table[43].feg[0].cl[0] = 0.4708470930;	data_table[43].feg[0].cnl[0] = 0.0933029874;	data_table[43].feg[0].cl[1] = 1.5818078100;	data_table[43].feg[0].cnl[1] = 0.4528313470;	data_table[43].feg[0].cl[2] = 2.0241981800;	data_table[43].feg[0].cnl[2] = 7.1148902300;	data_table[43].feg[0].cl[3] = 0.0019703626;	data_table[43].feg[0].cnl[3] = 0.0075618160;	data_table[43].feg[0].cl[4] = 0.6269126390;	data_table[43].feg[0].cnl[4] = 1.2539985800;	data_table[43].feg[0].cl[5] = 0.1026413200;	data_table[43].feg[0].cnl[5] = 0.1337860870;
				data_table[44].feg[0].cl[0] = 0.4200515530;	data_table[44].feg[0].cnl[0] = 0.0938882628;	data_table[44].feg[0].cl[1] = 1.7626650700;	data_table[44].feg[0].cnl[1] = 0.4644416870;	data_table[44].feg[0].cl[2] = 2.0273564100;	data_table[44].feg[0].cnl[2] = 8.1934604600;	data_table[44].feg[0].cl[3] = 0.0014548718;	data_table[44].feg[0].cnl[3] = 0.0078270452;	data_table[44].feg[0].cl[4] = 0.6228096000;	data_table[44].feg[0].cnl[4] = 1.1719415300;	data_table[44].feg[0].cl[5] = 0.0991529915;	data_table[44].feg[0].cnl[5] = 0.1245328390;
				data_table[45].feg[0].cl[0] = 2.1047515500;	data_table[45].feg[0].cnl[0] = 8.6860647000;	data_table[45].feg[0].cl[1] = 2.0388448700;	data_table[45].feg[0].cnl[1] = 0.3789244490;	data_table[45].feg[0].cl[2] = 0.1820672640;	data_table[45].feg[0].cnl[2] = 0.1429216340;	data_table[45].feg[0].cl[3] = 0.0952040948;	data_table[45].feg[0].cnl[3] = 0.1171259000;	data_table[45].feg[0].cl[4] = 0.5914452480;	data_table[45].feg[0].cnl[4] = 1.0784380800;	data_table[45].feg[0].cl[5] = 0.0011332868;	data_table[45].feg[0].cnl[5] = 0.0078025209;
				data_table[46].feg[0].cl[0] = 2.0798139000;	data_table[46].feg[0].cnl[0] = 9.9254029700;	data_table[46].feg[0].cl[1] = 0.4431707260;	data_table[46].feg[0].cnl[1] = 0.1049201040;	data_table[46].feg[0].cl[2] = 1.9651521500;	data_table[46].feg[0].cnl[2] = 0.6401038390;	data_table[46].feg[0].cl[3] = 0.5961305910;	data_table[46].feg[0].cnl[3] = 0.8895947900;	data_table[46].feg[0].cl[4] = 0.4780163330;	data_table[46].feg[0].cnl[4] = 1.9850940700;	data_table[46].feg[0].cl[5] = 0.0946458470;	data_table[46].feg[0].cnl[5] = 0.1127444640;
				data_table[47].feg[0].cl[0] = 1.6365754900;	data_table[47].feg[0].cnl[0] = 12.4540381000;	data_table[47].feg[0].cl[1] = 2.1792798900;	data_table[47].feg[0].cnl[1] = 1.4513466000;	data_table[47].feg[0].cl[2] = 0.7713006900;	data_table[47].feg[0].cnl[2] = 0.1266957570;	data_table[47].feg[0].cl[3] = 0.6641938800;	data_table[47].feg[0].cnl[3] = 0.7776592020;	data_table[47].feg[0].cl[4] = 0.7645632850;	data_table[47].feg[0].cnl[4] = 1.6607521000;	data_table[47].feg[0].cl[5] = 0.0861126689;	data_table[47].feg[0].cnl[5] = 0.1057283570;
				data_table[48].feg[0].cl[0] = 2.2482063200;	data_table[48].feg[0].cnl[0] = 1.5191350700;	data_table[48].feg[0].cl[1] = 1.6470686400;	data_table[48].feg[0].cnl[1] = 13.0113424000;	data_table[48].feg[0].cl[2] = 0.7886792650;	data_table[48].feg[0].cnl[2] = 0.1061281840;	data_table[48].feg[0].cl[3] = 0.0812579069;	data_table[48].feg[0].cnl[3] = 0.0994045620;	data_table[48].feg[0].cl[4] = 0.6682803460;	data_table[48].feg[0].cnl[4] = 1.4974206300;	data_table[48].feg[0].cl[5] = 0.6384674750;	data_table[48].feg[0].cnl[5] = 0.7184226350;
				data_table[49].feg[0].cl[0] = 2.1664462000;	data_table[49].feg[0].cnl[0] = 11.3174909000;	data_table[49].feg[0].cl[1] = 0.6886910210;	data_table[49].feg[0].cnl[1] = 0.1101312850;	data_table[49].feg[0].cl[2] = 1.9243175100;	data_table[49].feg[0].cnl[2] = 0.6744648530;	data_table[49].feg[0].cl[3] = 0.5653598880;	data_table[49].feg[0].cnl[3] = 0.7335646100;	data_table[49].feg[0].cl[4] = 0.9186838610;	data_table[49].feg[0].cnl[4] = 10.2310312000;	data_table[49].feg[0].cl[5] = 0.0780542213;	data_table[49].feg[0].cnl[5] = 0.0931104308;
				data_table[50].feg[0].cl[0] = 1.7366211400;	data_table[50].feg[0].cnl[0] = 0.8843347190;	data_table[50].feg[0].cl[1] = 0.9998713800;	data_table[50].feg[0].cnl[1] = 0.1384621210;	data_table[50].feg[0].cl[2] = 2.1397240900;	data_table[50].feg[0].cnl[2] = 11.9666432000;	data_table[50].feg[0].cl[3] = 0.5605665260;	data_table[50].feg[0].cnl[3] = 0.6726728800;	data_table[50].feg[0].cl[4] = 0.9937727470;	data_table[50].feg[0].cnl[4] = 8.7233041100;	data_table[50].feg[0].cl[5] = 0.0737374982;	data_table[50].feg[0].cnl[5] = 0.0878577715;
				data_table[51].feg[0].cl[0] = 2.0938388200;	data_table[51].feg[0].cnl[0] = 12.6856869000;	data_table[51].feg[0].cl[1] = 1.5694051900;	data_table[51].feg[0].cnl[1] = 1.2123653700;	data_table[51].feg[0].cl[2] = 1.3094199300;	data_table[51].feg[0].cnl[2] = 0.1666332920;	data_table[51].feg[0].cl[3] = 0.0698067804;	data_table[51].feg[0].cnl[3] = 0.0830817576;	data_table[51].feg[0].cl[4] = 1.0496953700;	data_table[51].feg[0].cnl[4] = 7.4314785700;	data_table[51].feg[0].cl[5] = 0.5555943540;	data_table[51].feg[0].cnl[5] = 0.6174876760;
				data_table[52].feg[0].cl[0] = 1.6018692500;	data_table[52].feg[0].cnl[0] = 0.1950315380;	data_table[52].feg[0].cl[1] = 1.9851026400;	data_table[52].feg[0].cnl[1] = 13.6976183000;	data_table[52].feg[0].cl[2] = 1.4822620000;	data_table[52].feg[0].cnl[2] = 1.8030479500;	data_table[52].feg[0].cl[3] = 0.5538071990;	data_table[52].feg[0].cnl[3] = 0.5679123400;	data_table[52].feg[0].cl[4] = 1.1172872200;	data_table[52].feg[0].cnl[4] = 6.4087987800;	data_table[52].feg[0].cl[5] = 0.0660720847;	data_table[52].feg[0].cnl[5] = 0.0786615429;
				data_table[53].feg[0].cl[0] = 1.6001548700;	data_table[53].feg[0].cnl[0] = 2.9291335400;	data_table[53].feg[0].cl[1] = 1.7164458100;	data_table[53].feg[0].cnl[1] = 15.5882990000;	data_table[53].feg[0].cl[2] = 1.8496835100;	data_table[53].feg[0].cnl[2] = 0.2225259830;	data_table[53].feg[0].cl[3] = 0.0623813648;	data_table[53].feg[0].cnl[3] = 0.0745581223;	data_table[53].feg[0].cl[4] = 1.2138755500;	data_table[53].feg[0].cnl[4] = 5.5601327100;	data_table[53].feg[0].cl[5] = 0.5540519460;	data_table[53].feg[0].cnl[5] = 0.5219945210;
				data_table[54].feg[0].cl[0] = 2.9523685400;	data_table[54].feg[0].cnl[0] = 6.0146195200;	data_table[54].feg[0].cl[1] = 0.4281057210;	data_table[54].feg[0].cnl[1] = 46.4151246000;	data_table[54].feg[0].cl[2] = 1.8959923300;	data_table[54].feg[0].cnl[2] = 0.1801097560;	data_table[54].feg[0].cl[3] = 0.0548012938;	data_table[54].feg[0].cnl[3] = 0.0712799633;	data_table[54].feg[0].cl[4] = 4.7083860000;	data_table[54].feg[0].cnl[4] = 45.6702799000;	data_table[54].feg[0].cl[5] = 0.5903567190;	data_table[54].feg[0].cnl[5] = 0.4702363100;
				data_table[55].feg[0].cl[0] = 3.1943424300;	data_table[55].feg[0].cnl[0] = 9.2735224100;	data_table[55].feg[0].cl[1] = 1.9828958600;	data_table[55].feg[0].cnl[1] = 0.2287416320;	data_table[55].feg[0].cl[2] = 0.1551210520;	data_table[55].feg[0].cnl[2] = 0.0382000231;	data_table[55].feg[0].cl[3] = 0.0673222354;	data_table[55].feg[0].cnl[3] = 0.0730961745;	data_table[55].feg[0].cl[4] = 4.4847421100;	data_table[55].feg[0].cnl[4] = 29.5703565000;	data_table[55].feg[0].cl[5] = 0.5426744140;	data_table[55].feg[0].cnl[5] = 0.4086470150;
				data_table[56].feg[0].cl[0] = 2.0503642500;	data_table[56].feg[0].cnl[0] = 0.2203484170;	data_table[56].feg[0].cl[1] = 0.1421143110;	data_table[56].feg[0].cnl[1] = 0.0396438056;	data_table[56].feg[0].cl[2] = 3.2353815100;	data_table[56].feg[0].cnl[2] = 9.5697916900;	data_table[56].feg[0].cl[3] = 0.0634683429;	data_table[56].feg[0].cnl[3] = 0.0692443091;	data_table[56].feg[0].cl[4] = 3.9796058600;	data_table[56].feg[0].cnl[4] = 25.3178406000;	data_table[56].feg[0].cl[5] = 0.5201167110;	data_table[56].feg[0].cnl[5] = 0.3836140980;
				data_table[57].feg[0].cl[0] = 3.2299075900;	data_table[57].feg[0].cnl[0] = 9.9466013500;	data_table[57].feg[0].cl[1] = 0.1576183070;	data_table[57].feg[0].cnl[1] = 0.0415378676;	data_table[57].feg[0].cl[2] = 2.1347783800;	data_table[57].feg[0].cnl[2] = 0.2404805720;	data_table[57].feg[0].cl[3] = 0.5019076090;	data_table[57].feg[0].cnl[3] = 0.3662520190;	data_table[57].feg[0].cl[4] = 3.8088901000;	data_table[57].feg[0].cnl[4] = 24.3275968000;	data_table[57].feg[0].cl[5] = 0.0596625028;	data_table[57].feg[0].cnl[5] = 0.0659653503;
				data_table[58].feg[0].cl[0] = 0.1581893240;	data_table[58].feg[0].cnl[0] = 0.0391309056;	data_table[58].feg[0].cl[1] = 3.1814199500;	data_table[58].feg[0].cnl[1] = 10.4139545000;	data_table[58].feg[0].cl[2] = 2.2762214000;	data_table[58].feg[0].cnl[2] = 0.2816717570;	data_table[58].feg[0].cl[3] = 3.9770547200;	data_table[58].feg[0].cnl[3] = 26.1872978000;	data_table[58].feg[0].cl[4] = 0.0558448277;	data_table[58].feg[0].cnl[4] = 0.0630921695;	data_table[58].feg[0].cl[5] = 0.4852079540;	data_table[58].feg[0].cnl[5] = 0.3542343690;
				data_table[59].feg[0].cl[0] = 0.1813794170;	data_table[59].feg[0].cnl[0] = 0.0437324793;	data_table[59].feg[0].cl[1] = 3.1761639600;	data_table[59].feg[0].cnl[1] = 10.7842572000;	data_table[59].feg[0].cl[2] = 2.3522151900;	data_table[59].feg[0].cnl[2] = 0.3055718330;	data_table[59].feg[0].cl[3] = 3.8312576300;	data_table[59].feg[0].cnl[3] = 25.4745408000;	data_table[59].feg[0].cl[4] = 0.0525889976;	data_table[59].feg[0].cnl[4] = 0.0602676073;	data_table[59].feg[0].cl[5] = 0.4700907420;	data_table[59].feg[0].cnl[5] = 0.3390170030;
				data_table[60].feg[0].cl[0] = 0.1929868110;	data_table[60].feg[0].cnl[0] = 0.0437785970;	data_table[60].feg[0].cl[1] = 2.4375602300;	data_table[60].feg[0].cnl[1] = 0.3293369960;	data_table[60].feg[0].cl[2] = 3.1724850400;	data_table[60].feg[0].cnl[2] = 11.1259996000;	data_table[60].feg[0].cl[3] = 3.5810541400;	data_table[60].feg[0].cnl[3] = 24.6709586000;	data_table[60].feg[0].cl[4] = 0.4565293940;	data_table[60].feg[0].cnl[4] = 0.3249902820;	data_table[60].feg[0].cl[5] = 0.0494812177;	data_table[60].feg[0].cnl[5] = 0.0576553100;
				data_table[61].feg[0].cl[0] = 0.2120025950;	data_table[61].feg[0].cnl[0] = 0.0457703608;	data_table[61].feg[0].cl[1] = 3.1689175400;	data_table[61].feg[0].cnl[1] = 11.4536599000;	data_table[61].feg[0].cl[2] = 2.5150349400;	data_table[61].feg[0].cnl[2] = 0.3555610540;	data_table[61].feg[0].cl[3] = 0.4440808450;	data_table[61].feg[0].cnl[3] = 0.3119533630;	data_table[61].feg[0].cl[4] = 3.3674210100;	data_table[61].feg[0].cnl[4] = 24.0291435000;	data_table[61].feg[0].cl[5] = 0.0465652543;	data_table[61].feg[0].cnl[5] = 0.0552266819;
				data_table[62].feg[0].cl[0] = 2.5935500200;	data_table[62].feg[0].cnl[0] = 0.3824526120;	data_table[62].feg[0].cl[1] = 3.1655752200;	data_table[62].feg[0].cnl[1] = 11.7675155000;	data_table[62].feg[0].cl[2] = 0.2294026520;	data_table[62].feg[0].cnl[2] = 0.0476642249;	data_table[62].feg[0].cl[3] = 0.4322577800;	data_table[62].feg[0].cnl[3] = 0.2997198330;	data_table[62].feg[0].cl[4] = 3.1726192000;	data_table[62].feg[0].cnl[4] = 23.4462738000;	data_table[62].feg[0].cl[5] = 0.0437958317;	data_table[62].feg[0].cnl[5] = 0.0529440680;
				data_table[63].feg[0].cl[0] = 3.1914493900;	data_table[63].feg[0].cnl[0] = 12.0224655000;	data_table[63].feg[0].cl[1] = 2.5576643100;	data_table[63].feg[0].cnl[1] = 0.4083388760;	data_table[63].feg[0].cl[2] = 0.3326819340;	data_table[63].feg[0].cnl[2] = 0.0585819814;	data_table[63].feg[0].cl[3] = 0.0414243130;	data_table[63].feg[0].cnl[3] = 0.0506771477;	data_table[63].feg[0].cl[4] = 2.6103672800;	data_table[63].feg[0].cnl[4] = 19.9344244000;	data_table[63].feg[0].cl[5] = 0.4205268630;	data_table[63].feg[0].cnl[5] = 0.2856862400;
				data_table[64].feg[0].cl[0] = 0.2594074620;	data_table[64].feg[0].cnl[0] = 0.0504689354;	data_table[64].feg[0].cl[1] = 3.1617785500;	data_table[64].feg[0].cnl[1] = 12.3140183000;	data_table[64].feg[0].cl[2] = 2.7509575100;	data_table[64].feg[0].cnl[2] = 0.4383376260;	data_table[64].feg[0].cl[3] = 2.7924768600;	data_table[64].feg[0].cnl[3] = 22.3797309000;	data_table[64].feg[0].cl[4] = 0.0385931001;	data_table[64].feg[0].cnl[4] = 0.0487920992;	data_table[64].feg[0].cl[5] = 0.4108817080;	data_table[64].feg[0].cnl[5] = 0.2776228920;
				data_table[65].feg[0].cl[0] = 3.1605539600;	data_table[65].feg[0].cnl[0] = 12.5470414000;	data_table[65].feg[0].cl[1] = 2.8275170900;	data_table[65].feg[0].cnl[1] = 0.4678990940;	data_table[65].feg[0].cl[2] = 0.2751402550;	data_table[65].feg[0].cnl[2] = 0.0523226982;	data_table[65].feg[0].cl[3] = 0.4009671600;	data_table[65].feg[0].cnl[3] = 0.2676148840;	data_table[65].feg[0].cl[4] = 2.6311083400;	data_table[65].feg[0].cnl[4] = 21.9498166000;	data_table[65].feg[0].cl[5] = 0.0361333817;	data_table[65].feg[0].cnl[5] = 0.0468871497;
				data_table[66].feg[0].cl[0] = 0.2886424670;	data_table[66].feg[0].cnl[0] = 0.0540507687;	data_table[66].feg[0].cl[1] = 2.9056729600;	data_table[66].feg[0].cnl[1] = 0.4975810770;	data_table[66].feg[0].cl[2] = 3.1596015900;	data_table[66].feg[0].cnl[2] = 12.7599505000;	data_table[66].feg[0].cl[3] = 0.3912802590;	data_table[66].feg[0].cnl[3] = 0.2581518310;	data_table[66].feg[0].cl[4] = 2.4859603800;	data_table[66].feg[0].cnl[4] = 21.5400972000;	data_table[66].feg[0].cl[5] = 0.0337664478;	data_table[66].feg[0].cnl[5] = 0.0450664323;
				data_table[67].feg[0].cl[0] = 3.1557321300;	data_table[67].feg[0].cnl[0] = 12.9729009000;	data_table[67].feg[0].cl[1] = 0.3115195600;	data_table[67].feg[0].cnl[1] = 0.0581399387;	data_table[67].feg[0].cl[2] = 2.9772240600;	data_table[67].feg[0].cnl[2] = 0.5312133940;	data_table[67].feg[0].cl[3] = 0.3815638540;	data_table[67].feg[0].cnl[3] = 0.2491957760;	data_table[67].feg[0].cl[4] = 2.4024753200;	data_table[67].feg[0].cnl[4] = 21.3627616000;	data_table[67].feg[0].cl[5] = 0.0315224214;	data_table[67].feg[0].cnl[5] = 0.0433253257;
				data_table[68].feg[0].cl[0] = 3.1559197000;	data_table[68].feg[0].cnl[0] = 13.1232407000;	data_table[68].feg[0].cl[1] = 0.3225447100;	data_table[68].feg[0].cnl[1] = 0.0597223323;	data_table[68].feg[0].cl[2] = 3.0556905300;	data_table[68].feg[0].cnl[2] = 0.5618767730;	data_table[68].feg[0].cl[3] = 0.0292845100;	data_table[68].feg[0].cnl[3] = 0.0416534255;	data_table[68].feg[0].cl[4] = 0.3724872050;	data_table[68].feg[0].cnl[4] = 0.2408219670;	data_table[68].feg[0].cl[5] = 2.2783369500;	data_table[68].feg[0].cnl[5] = 21.0034185000;
				data_table[69].feg[0].cl[0] = 3.1079470400;	data_table[69].feg[0].cnl[0] = 0.6063478470;	data_table[69].feg[0].cl[1] = 3.1409122100;	data_table[69].feg[0].cnl[1] = 13.3705269000;	data_table[69].feg[0].cl[2] = 0.3756604540;	data_table[69].feg[0].cnl[2] = 0.0729814740;	data_table[69].feg[0].cl[3] = 0.3619010970;	data_table[69].feg[0].cnl[3] = 0.2326520510;	data_table[69].feg[0].cl[4] = 2.4540908200;	data_table[69].feg[0].cnl[4] = 21.2695209000;	data_table[69].feg[0].cl[5] = 0.0272383990;	data_table[69].feg[0].cnl[5] = 0.0399969597;
				data_table[70].feg[0].cl[0] = 3.1144686300;	data_table[70].feg[0].cnl[0] = 13.8968881000;	data_table[70].feg[0].cl[1] = 0.5396343530;	data_table[70].feg[0].cnl[1] = 0.0891708508;	data_table[70].feg[0].cl[2] = 3.0646091500;	data_table[70].feg[0].cnl[2] = 0.6799195630;	data_table[70].feg[0].cl[3] = 0.0258563745;	data_table[70].feg[0].cnl[3] = 0.0382808522;	data_table[70].feg[0].cl[4] = 2.1398355600;	data_table[70].feg[0].cnl[4] = 18.0078788000;	data_table[70].feg[0].cl[5] = 0.3477882310;	data_table[70].feg[0].cnl[5] = 0.2227065910;
				data_table[71].feg[0].cl[0] = 3.0116689900;	data_table[71].feg[0].cnl[0] = 0.7104018890;	data_table[71].feg[0].cl[1] = 3.1628478800;	data_table[71].feg[0].cnl[1] = 13.8262192000;	data_table[71].feg[0].cl[2] = 0.6334217710;	data_table[71].feg[0].cnl[2] = 0.0948486572;	data_table[71].feg[0].cl[3] = 0.3414171980;	data_table[71].feg[0].cnl[3] = 0.2141296780;	data_table[71].feg[0].cl[4] = 1.5356601300;	data_table[71].feg[0].cnl[4] = 15.5298698000;	data_table[71].feg[0].cl[5] = 0.0240723773;	data_table[71].feg[0].cnl[5] = 0.0367833690;
				data_table[72].feg[0].cl[0] = 3.2023682100;	data_table[72].feg[0].cnl[0] = 13.8446369000;	data_table[72].feg[0].cl[1] = 0.8300984130;	data_table[72].feg[0].cnl[1] = 0.1183815810;	data_table[72].feg[0].cl[2] = 2.8655229700;	data_table[72].feg[0].cnl[2] = 0.7663691180;	data_table[72].feg[0].cl[3] = 0.0224813887;	data_table[72].feg[0].cnl[3] = 0.0352934622;	data_table[72].feg[0].cl[4] = 1.4016526300;	data_table[72].feg[0].cnl[4] = 14.6148877000;	data_table[72].feg[0].cl[5] = 0.3337405960;	data_table[72].feg[0].cnl[5] = 0.2057044860;
				data_table[73].feg[0].cl[0] = 0.9249068550;	data_table[73].feg[0].cnl[0] = 0.1286633770;	data_table[73].feg[0].cl[1] = 2.7555455700;	data_table[73].feg[0].cnl[1] = 0.7658264790;	data_table[73].feg[0].cl[2] = 3.3044006000;	data_table[73].feg[0].cnl[2] = 13.4471170000;	data_table[73].feg[0].cl[3] = 0.3299738620;	data_table[73].feg[0].cnl[3] = 0.1982188950;	data_table[73].feg[0].cl[4] = 1.0991644400;	data_table[73].feg[0].cnl[4] = 13.5087534000;	data_table[73].feg[0].cl[5] = 0.0206498883;	data_table[73].feg[0].cnl[5] = 0.0338918459;
				data_table[74].feg[0].cl[0] = 1.9695210500;	data_table[74].feg[0].cnl[0] = 49.8830620000;	data_table[74].feg[0].cl[1] = 1.2172661900;	data_table[74].feg[0].cnl[1] = 0.1332438090;	data_table[74].feg[0].cl[2] = 4.1039168500;	data_table[74].feg[0].cnl[2] = 1.8439691600;	data_table[74].feg[0].cl[3] = 0.0290791978;	data_table[74].feg[0].cnl[3] = 0.0284192813;	data_table[74].feg[0].cl[4] = 0.2306966690;	data_table[74].feg[0].cnl[4] = 0.1909687840;	data_table[74].feg[0].cl[5] = 0.6088402990;	data_table[74].feg[0].cnl[5] = 1.3709035600;
				data_table[75].feg[0].cl[0] = 2.0638586700;	data_table[75].feg[0].cnl[0] = 40.5671697000;	data_table[75].feg[0].cl[1] = 1.2960340600;	data_table[75].feg[0].cnl[1] = 0.1465590470;	data_table[75].feg[0].cl[2] = 3.9692067300;	data_table[75].feg[0].cnl[2] = 1.8256159600;	data_table[75].feg[0].cl[3] = 0.0269835487;	data_table[75].feg[0].cnl[3] = 0.0284172045;	data_table[75].feg[0].cl[4] = 0.2310839990;	data_table[75].feg[0].cnl[4] = 0.1797651840;	data_table[75].feg[0].cl[5] = 0.6304667740;	data_table[75].feg[0].cnl[5] = 1.3891154300;
				data_table[76].feg[0].cl[0] = 2.2152272600;	data_table[76].feg[0].cnl[0] = 32.4464090000;	data_table[76].feg[0].cl[1] = 1.3757315500;	data_table[76].feg[0].cnl[1] = 0.1609200480;	data_table[76].feg[0].cl[2] = 3.7824440500;	data_table[76].feg[0].cnl[2] = 1.7875655300;	data_table[76].feg[0].cl[3] = 0.0244643240;	data_table[76].feg[0].cnl[3] = 0.0282909938;	data_table[76].feg[0].cl[4] = 0.2369320160;	data_table[76].feg[0].cnl[4] = 0.1706923680;	data_table[76].feg[0].cl[5] = 0.6484714120;	data_table[76].feg[0].cnl[5] = 1.3792839000;
				data_table[77].feg[0].cl[0] = 0.9846979400;	data_table[77].feg[0].cnl[0] = 0.1609108390;	data_table[77].feg[0].cl[1] = 2.7398707900;	data_table[77].feg[0].cnl[1] = 0.7189716670;	data_table[77].feg[0].cl[2] = 3.6169671500;	data_table[77].feg[0].cnl[2] = 12.9281016000;	data_table[77].feg[0].cl[3] = 0.3028856020;	data_table[77].feg[0].cnl[3] = 0.1701348540;	data_table[77].feg[0].cl[4] = 0.2783707260;	data_table[77].feg[0].cnl[4] = 1.4986270300;	data_table[77].feg[0].cl[5] = 0.0152124129;	data_table[77].feg[0].cnl[5] = 0.0283510822;
				data_table[78].feg[0].cl[0] = 0.9612633980;	data_table[78].feg[0].cnl[0] = 0.1709322770;	data_table[78].feg[0].cl[1] = 3.6958103000;	data_table[78].feg[0].cnl[1] = 12.9335319000;	data_table[78].feg[0].cl[2] = 2.7756749100;	data_table[78].feg[0].cnl[2] = 0.6899970700;	data_table[78].feg[0].cl[3] = 0.2954141760;	data_table[78].feg[0].cnl[3] = 0.1635255100;	data_table[78].feg[0].cl[4] = 0.3114757430;	data_table[78].feg[0].cnl[4] = 1.3920090100;	data_table[78].feg[0].cl[5] = 0.0143237267;	data_table[78].feg[0].cnl[5] = 0.0271265337;
				data_table[79].feg[0].cl[0] = 1.2920049100;	data_table[79].feg[0].cnl[0] = 0.1834328650;	data_table[79].feg[0].cl[1] = 2.7516147800;	data_table[79].feg[0].cnl[1] = 0.9423683710;	data_table[79].feg[0].cl[2] = 3.4938794900;	data_table[79].feg[0].cnl[2] = 14.6235654000;	data_table[79].feg[0].cl[3] = 0.2773046360;	data_table[79].feg[0].cnl[3] = 0.1551101440;	data_table[79].feg[0].cl[4] = 0.4302328100;	data_table[79].feg[0].cnl[4] = 1.2887167000;	data_table[79].feg[0].cl[5] = 0.0148294351;	data_table[79].feg[0].cnl[5] = 0.0261903834;
				data_table[80].feg[0].cl[0] = 3.7596473000;	data_table[80].feg[0].cnl[0] = 13.5041513000;	data_table[80].feg[0].cl[1] = 3.2119590400;	data_table[80].feg[0].cnl[1] = 0.6663309930;	data_table[80].feg[0].cl[2] = 0.6477678250;	data_table[80].feg[0].cnl[2] = 0.0922518234;	data_table[80].feg[0].cl[3] = 0.2761232740;	data_table[80].feg[0].cnl[3] = 0.1503128970;	data_table[80].feg[0].cl[4] = 0.3188388100;	data_table[80].feg[0].cnl[4] = 1.1256558800;	data_table[80].feg[0].cl[5] = 0.0131668419;	data_table[80].feg[0].cnl[5] = 0.0248879842;
				data_table[81].feg[0].cl[0] = 1.0079597500;	data_table[81].feg[0].cnl[0] = 0.1172684270;	data_table[81].feg[0].cl[1] = 3.0979615300;	data_table[81].feg[0].cnl[1] = 0.8804532350;	data_table[81].feg[0].cl[2] = 3.6129686400;	data_table[81].feg[0].cnl[2] = 14.7325812000;	data_table[81].feg[0].cl[3] = 0.2624014760;	data_table[81].feg[0].cnl[3] = 0.1434910140;	data_table[81].feg[0].cl[4] = 0.4056219950;	data_table[81].feg[0].cnl[4] = 1.0410350600;	data_table[81].feg[0].cl[5] = 0.0131812509;	data_table[81].feg[0].cnl[5] = 0.0239575415;
				data_table[82].feg[0].cl[0] = 1.5982687500;	data_table[82].feg[0].cnl[0] = 0.1568974710;	data_table[82].feg[0].cl[1] = 4.3823392500;	data_table[82].feg[0].cnl[1] = 2.4709469200;	data_table[82].feg[0].cl[2] = 2.0607471900;	data_table[82].feg[0].cnl[2] = 57.2438972000;	data_table[82].feg[0].cl[3] = 0.1944260230;	data_table[82].feg[0].cnl[3] = 0.1329791090;	data_table[82].feg[0].cl[4] = 0.8227049780;	data_table[82].feg[0].cnl[4] = 0.9565325280;	data_table[82].feg[0].cl[5] = 0.0233226953;	data_table[82].feg[0].cnl[5] = 0.0223038435;
				data_table[83].feg[0].cl[0] = 1.7146322300;	data_table[83].feg[0].cnl[0] = 97.9262841000;	data_table[83].feg[0].cl[1] = 2.1411596000;	data_table[83].feg[0].cnl[1] = 0.2101937170;	data_table[83].feg[0].cl[2] = 4.3751241300;	data_table[83].feg[0].cnl[2] = 3.6694881200;	data_table[83].feg[0].cl[3] = 0.0216216680;	data_table[83].feg[0].cnl[3] = 0.0198456144;	data_table[83].feg[0].cl[4] = 0.1978438370;	data_table[83].feg[0].cnl[4] = 0.1337588070;	data_table[83].feg[0].cl[5] = 0.6520479200;	data_table[83].feg[0].cnl[5] = 0.7804321040;
				data_table[84].feg[0].cl[0] = 1.4804779400;	data_table[84].feg[0].cnl[0] = 125.9439190000;	data_table[84].feg[0].cl[1] = 2.0917463000;	data_table[84].feg[0].cnl[1] = 0.1838030080;	data_table[84].feg[0].cl[2] = 4.7524603300;	data_table[84].feg[0].cnl[2] = 4.1989059600;	data_table[84].feg[0].cl[3] = 0.0185643958;	data_table[84].feg[0].cnl[3] = 0.0181383503;	data_table[84].feg[0].cl[4] = 0.2058593750;	data_table[84].feg[0].cnl[4] = 0.1330354040;	data_table[84].feg[0].cl[5] = 0.7135409480;	data_table[84].feg[0].cnl[5] = 0.7030319380;
				data_table[85].feg[0].cl[0] = 0.6300222950;	data_table[85].feg[0].cnl[0] = 0.1409097620;	data_table[85].feg[0].cl[1] = 3.8096288100;	data_table[85].feg[0].cnl[1] = 30.8515540000;	data_table[85].feg[0].cl[2] = 3.8975606700;	data_table[85].feg[0].cnl[2] = 0.6515597630;	data_table[85].feg[0].cl[3] = 0.2407551000;	data_table[85].feg[0].cnl[3] = 0.1088996720;	data_table[85].feg[0].cl[4] = 2.6286857700;	data_table[85].feg[0].cnl[4] = 6.4238326100;	data_table[85].feg[0].cl[5] = 0.0314285931;	data_table[85].feg[0].cnl[5] = 0.0242346699;
				data_table[86].feg[0].cl[0] = 5.2328813500;	data_table[86].feg[0].cnl[0] = 8.6059953600;	data_table[86].feg[0].cl[1] = 2.4860420500;	data_table[86].feg[0].cnl[1] = 0.3045439820;	data_table[86].feg[0].cl[2] = 0.3234313540;	data_table[86].feg[0].cnl[2] = 0.0387759096;	data_table[86].feg[0].cl[3] = 0.2554035960;	data_table[86].feg[0].cnl[3] = 0.1287177240;	data_table[86].feg[0].cl[4] = 0.5536072280;	data_table[86].feg[0].cnl[4] = 0.5369774520;	data_table[86].feg[0].cl[5] = 0.0057527889;	data_table[86].feg[0].cnl[5] = 0.0129417790;
				data_table[87].feg[0].cl[0] = 1.4419268500;	data_table[87].feg[0].cnl[0] = 0.1187408730;	data_table[87].feg[0].cl[1] = 3.5529172500;	data_table[87].feg[0].cnl[1] = 1.0173975000;	data_table[87].feg[0].cl[2] = 3.9125958600;	data_table[87].feg[0].cnl[2] = 63.1814783000;	data_table[87].feg[0].cl[3] = 0.2161735190;	data_table[87].feg[0].cnl[3] = 0.0955806441;	data_table[87].feg[0].cl[4] = 3.9419160500;	data_table[87].feg[0].cnl[4] = 35.0602732000;	data_table[87].feg[0].cl[5] = 0.0460422605;	data_table[87].feg[0].cnl[5] = 0.0220850385;
				data_table[88].feg[0].cl[0] = 1.4586412700;	data_table[88].feg[0].cnl[0] = 0.1077604940;	data_table[88].feg[0].cl[1] = 4.1894540500;	data_table[88].feg[0].cnl[1] = 88.9090649000;	data_table[88].feg[0].cl[2] = 3.6586618200;	data_table[88].feg[0].cnl[2] = 1.0508893100;	data_table[88].feg[0].cl[3] = 0.2084792290;	data_table[88].feg[0].cnl[3] = 0.0909335557;	data_table[88].feg[0].cl[4] = 3.1652811700;	data_table[88].feg[0].cnl[4] = 31.3297788000;	data_table[88].feg[0].cl[5] = 0.0523892556;	data_table[88].feg[0].cnl[5] = 0.0208807697;
				data_table[89].feg[0].cl[0] = 1.1901406400;	data_table[89].feg[0].cnl[0] = 0.0773468729;	data_table[89].feg[0].cl[1] = 2.5538060700;	data_table[89].feg[0].cnl[1] = 0.6596936810;	data_table[89].feg[0].cl[2] = 4.6811018100;	data_table[89].feg[0].cnl[2] = 12.8013896000;	data_table[89].feg[0].cl[3] = 0.2261213030;	data_table[89].feg[0].cnl[3] = 0.1086321940;	data_table[89].feg[0].cl[4] = 0.3582505450;	data_table[89].feg[0].cnl[4] = 0.4567656640;	data_table[89].feg[0].cl[5] = 0.0078226395;	data_table[89].feg[0].cnl[5] = 0.0162623474;
				data_table[90].feg[0].cl[0] = 4.6853750400;	data_table[90].feg[0].cnl[0] = 14.4503632000;	data_table[90].feg[0].cl[1] = 2.9841370800;	data_table[90].feg[0].cnl[1] = 0.5564385920;	data_table[90].feg[0].cl[2] = 0.8919880610;	data_table[90].feg[0].cnl[2] = 0.0669512914;	data_table[90].feg[0].cl[3] = 0.2248253840;	data_table[90].feg[0].cnl[3] = 0.1032353960;	data_table[90].feg[0].cl[4] = 0.3044448460;	data_table[90].feg[0].cnl[4] = 0.4272556470;	data_table[90].feg[0].cl[5] = 0.0094816271;	data_table[90].feg[0].cnl[5] = 0.0177730611;
				data_table[91].feg[0].cl[0] = 4.6334360600;	data_table[91].feg[0].cnl[0] = 16.3377267000;	data_table[91].feg[0].cl[1] = 3.1815705600;	data_table[91].feg[0].cnl[1] = 0.5695178680;	data_table[91].feg[0].cl[2] = 0.8764550750;	data_table[91].feg[0].cnl[2] = 0.0688860012;	data_table[91].feg[0].cl[3] = 0.2216854770;	data_table[91].feg[0].cnl[3] = 0.0984254550;	data_table[91].feg[0].cl[4] = 0.2729171000;	data_table[91].feg[0].cnl[4] = 0.4094709170;	data_table[91].feg[0].cl[5] = 0.0111737298;	data_table[91].feg[0].cnl[5] = 0.0186215410;
				data_table[92].feg[0].cl[0] = 4.5677388800;	data_table[92].feg[0].cnl[0] = 19.0992795000;	data_table[92].feg[0].cl[1] = 3.4032517900;	data_table[92].feg[0].cnl[1] = 0.5900996340;	data_table[92].feg[0].cl[2] = 0.8618419230;	data_table[92].feg[0].cnl[2] = 0.0703204851;	data_table[92].feg[0].cl[3] = 0.2197288700;	data_table[92].feg[0].cnl[3] = 0.0936334280;	data_table[92].feg[0].cl[4] = 0.2381769030;	data_table[92].feg[0].cnl[4] = 0.3935548820;	data_table[92].feg[0].cl[5] = 0.0138306499;	data_table[92].feg[0].cnl[5] = 0.0194437286;
				data_table[93].feg[0].cl[0] = 5.4567112300;	data_table[93].feg[0].cnl[0] = 10.1892720000;	data_table[93].feg[0].cl[1] = 0.1116879060;	data_table[93].feg[0].cnl[1] = 0.0398131313;	data_table[93].feg[0].cl[2] = 3.3026034300;	data_table[93].feg[0].cnl[2] = 0.3146222120;	data_table[93].feg[0].cl[3] = 0.1845683190;	data_table[93].feg[0].cnl[3] = 0.1042208600;	data_table[93].feg[0].cl[4] = 0.4936442630;	data_table[93].feg[0].cnl[4] = 0.4630805400;	data_table[93].feg[0].cl[5] = 3.5748474300;	data_table[93].feg[0].cnl[5] = 21.9369542000;
				data_table[94].feg[0].cl[0] = 5.3832199900;	data_table[94].feg[0].cnl[0] = 10.7289857000;	data_table[94].feg[0].cl[1] = 0.1233432360;	data_table[94].feg[0].cnl[1] = 0.0415137806;	data_table[94].feg[0].cl[2] = 3.4646909000;	data_table[94].feg[0].cnl[2] = 0.3393262080;	data_table[94].feg[0].cl[3] = 0.1754371320;	data_table[94].feg[0].cnl[3] = 0.0998932346;	data_table[94].feg[0].cl[4] = 3.3980007300;	data_table[94].feg[0].cnl[4] = 21.1601535000;	data_table[94].feg[0].cl[5] = 0.4694595190;	data_table[94].feg[0].cnl[5] = 0.4519969700;
				data_table[95].feg[0].cl[0] = 5.3840237700;	data_table[95].feg[0].cnl[0] = 11.1211419000;	data_table[95].feg[0].cl[1] = 3.4986126400;	data_table[95].feg[0].cnl[1] = 0.3567502100;	data_table[95].feg[0].cl[2] = 0.1880395470;	data_table[95].feg[0].cnl[2] = 0.0539853583;	data_table[95].feg[0].cl[3] = 0.1691431370;	data_table[95].feg[0].cnl[3] = 0.0960082633;	data_table[95].feg[0].cl[4] = 3.1959501600;	data_table[95].feg[0].cnl[4] = 18.0694389000;	data_table[95].feg[0].cl[5] = 0.4643930590;	data_table[95].feg[0].cnl[5] = 0.4363181970;
				data_table[96].feg[0].cl[0] = 3.6609068800;	data_table[96].feg[0].cnl[0] = 0.3844209060;	data_table[96].feg[0].cl[1] = 0.2030546780;	data_table[96].feg[0].cnl[1] = 0.0548547131;	data_table[96].feg[0].cl[2] = 5.3069751500;	data_table[96].feg[0].cnl[2] = 11.7150262000;	data_table[96].feg[0].cl[3] = 0.1609340460;	data_table[96].feg[0].cnl[3] = 0.0921020329;	data_table[96].feg[0].cl[4] = 3.0480840100;	data_table[96].feg[0].cnl[4] = 17.3525367000;	data_table[96].feg[0].cl[5] = 0.4436102950;	data_table[96].feg[0].cnl[5] = 0.4271323590;
				data_table[97].feg[0].cl[0] = 3.9415039000;	data_table[97].feg[0].cnl[0] = 0.4182467220;	data_table[97].feg[0].cl[1] = 5.1691534500;	data_table[97].feg[0].cnl[1] = 12.5201788000;	data_table[97].feg[0].cl[2] = 0.1619410740;	data_table[97].feg[0].cnl[2] = 0.0481540117;	data_table[97].feg[0].cl[3] = 0.4152995610;	data_table[97].feg[0].cnl[3] = 0.4249138560;	data_table[97].feg[0].cl[4] = 2.9176132500;	data_table[97].feg[0].cnl[4] = 19.0899693000;	data_table[97].feg[0].cl[5] = 0.1514749270;	data_table[97].feg[0].cnl[5] = 0.0881568925;
				data_table[98].feg[0].cl[0] = 4.0978062300;	data_table[98].feg[0].cnl[0] = 0.4460211450;	data_table[98].feg[0].cl[1] = 5.1007939300;	data_table[98].feg[0].cnl[1] = 13.1768613000;	data_table[98].feg[0].cl[2] = 0.1746172890;	data_table[98].feg[0].cnl[2] = 0.0502742829;	data_table[98].feg[0].cl[3] = 2.7677465800;	data_table[98].feg[0].cnl[3] = 18.4815393000;	data_table[98].feg[0].cl[4] = 0.1444966390;	data_table[98].feg[0].cnl[4] = 0.0846232592;	data_table[98].feg[0].cl[5] = 0.4027721090;	data_table[98].feg[0].cnl[5] = 0.4176401000;
				data_table[99].feg[0].cl[0] = 4.2493482000;	data_table[99].feg[0].cnl[0] = 0.4752639330;	data_table[99].feg[0].cl[1] = 5.0355659400;	data_table[99].feg[0].cnl[1] = 13.8570834000;	data_table[99].feg[0].cl[2] = 0.1889206130;	data_table[99].feg[0].cnl[2] = 0.0526975158;	data_table[99].feg[0].cl[3] = 0.3943560580;	data_table[99].feg[0].cnl[3] = 0.4111937510;	data_table[99].feg[0].cl[4] = 2.6121310000;	data_table[99].feg[0].cnl[4] = 17.8537905000;	data_table[99].feg[0].cl[5] = 0.1380019270;	data_table[99].feg[0].cnl[5] = 0.0812774434;
				data_table[100].feg[0].cl[0] = 0.2009429310;	data_table[100].feg[0].cnl[0] = 0.0548366518;	data_table[100].feg[0].cl[1] = 4.4011986900;	data_table[100].feg[0].cnl[1] = 0.5042484340;	data_table[100].feg[0].cl[2] = 4.9725010200;	data_table[100].feg[0].cnl[2] = 14.5721366000;	data_table[100].feg[0].cl[3] = 2.4753059900;	data_table[100].feg[0].cnl[3] = 17.2978308000;	data_table[100].feg[0].cl[4] = 0.3868831970;	data_table[100].feg[0].cnl[4] = 0.4050438980;	data_table[100].feg[0].cl[5] = 0.1319360950;	data_table[100].feg[0].cnl[5] = 0.0780821071;
				data_table[101].feg[0].cl[0] = 0.2160528990;	data_table[101].feg[0].cnl[0] = 0.0583584058;	data_table[101].feg[0].cl[1] = 4.9110679900;	data_table[101].feg[0].cnl[1] = 15.3264212000;	data_table[101].feg[0].cl[2] = 4.5486287000;	data_table[101].feg[0].cnl[2] = 0.5344347600;	data_table[101].feg[0].cl[3] = 2.3611424900;	data_table[101].feg[0].cnl[3] = 16.8164803000;	data_table[101].feg[0].cl[4] = 0.1262772920;	data_table[101].feg[0].cnl[4] = 0.0750304633;	data_table[101].feg[0].cl[5] = 0.3813645010;	data_table[101].feg[0].cnl[5] = 0.3993058520;
				data_table[102].feg[0].cl[0] = 4.8673801400;	data_table[102].feg[0].cnl[0] = 16.0320520000;	data_table[102].feg[0].cl[1] = 0.3199744010;	data_table[102].feg[0].cnl[1] = 0.0670871138;	data_table[102].feg[0].cl[2] = 4.5887242500;	data_table[102].feg[0].cnl[2] = 0.5770393730;	data_table[102].feg[0].cl[3] = 0.1214824480;	data_table[102].feg[0].cnl[3] = 0.0722275899;	data_table[102].feg[0].cl[4] = 2.3163987200;	data_table[102].feg[0].cnl[4] = 14.1279737000;	data_table[102].feg[0].cl[5] = 0.3792581370;	data_table[102].feg[0].cnl[5] = 0.3899734840;
			}

			// 5: Weickenmeier and H.Kohl - a*(1-exp(-bg^2)/g^2 - [0, 12]
			void Load_feg_Weickenmeier_neutral_0_12()
			{
				data_table[0].feg[0].cl[0] = 0.0000000000;		data_table[0].feg[0].cnl[0] = 0.0000000000;	data_table[0].feg[0].cl[1] = 0.0000000000;		data_table[0].feg[0].cnl[1] = 0.0000000000;	data_table[0].feg[0].cl[2] = 0.0000000000;		data_table[0].feg[0].cnl[2] = 0.0000000000;	data_table[0].feg[0].cl[3] = 0.0000000000;		data_table[0].feg[0].cnl[3] = 0.0000000000;	data_table[0].feg[0].cl[4] = 0.0000000000;		data_table[0].feg[0].cnl[4] = 0.0000000000;	data_table[0].feg[0].cl[5] = 0.0000000000;		data_table[0].feg[0].cnl[5] = 0.0000000000;
				data_table[1].feg[0].cl[0] = 0.5000000000;		data_table[1].feg[0].cnl[0] = 2.5420000000;	data_table[1].feg[0].cl[1] = 0.5000000000;		data_table[1].feg[0].cnl[1] = 8.7430000000;	data_table[1].feg[0].cl[2] = 0.5000000000;		data_table[1].feg[0].cnl[2] = 12.6900000000;	data_table[1].feg[0].cl[3] = 0.5000000000;		data_table[1].feg[0].cnl[3] = 0.4371000000;	data_table[1].feg[0].cl[4] = 0.5000000000;		data_table[1].feg[0].cnl[4] = 5.2940000000;	data_table[1].feg[0].cl[5] = 0.5000000000;		data_table[1].feg[0].cnl[5] = 28.2500000000;
				data_table[2].feg[0].cl[0] = 0.5000000000;		data_table[2].feg[0].cnl[0] = 0.6845000000;	data_table[2].feg[0].cl[1] = 0.5000000000;		data_table[2].feg[0].cnl[1] = 3.0650000000;	data_table[2].feg[0].cl[2] = 0.5000000000;		data_table[2].feg[0].cnl[2] = 6.2400000000;	data_table[2].feg[0].cl[3] = 0.5000000000;		data_table[2].feg[0].cnl[3] = 126.2000000000;	data_table[2].feg[0].cl[4] = 0.5000000000;		data_table[2].feg[0].cnl[4] = 131.2000000000;	data_table[2].feg[0].cl[5] = 0.5000000000;		data_table[2].feg[0].cnl[5] = 131.8000000000;
				data_table[3].feg[0].cl[0] = 0.3000000000;		data_table[3].feg[0].cnl[0] = 0.5400000000;	data_table[3].feg[0].cl[1] = 0.3000000000;		data_table[3].feg[0].cnl[1] = 3.3880000000;	data_table[3].feg[0].cl[2] = 0.3000000000;		data_table[3].feg[0].cnl[2] = 55.6200000000;	data_table[3].feg[0].cl[3] = 0.3000000000;		data_table[3].feg[0].cnl[3] = 50.7800000000;	data_table[3].feg[0].cl[4] = 0.3000000000;		data_table[3].feg[0].cnl[4] = 67.0100000000;	data_table[3].feg[0].cl[5] = 0.3000000000;		data_table[3].feg[0].cnl[5] = 96.3700000000;
				data_table[4].feg[0].cl[0] = 0.5000000000;		data_table[4].feg[0].cnl[0] = 0.3314000000;	data_table[4].feg[0].cl[1] = 0.5000000000;		data_table[4].feg[0].cnl[1] = 2.9750000000;	data_table[4].feg[0].cl[2] = 0.5000000000;		data_table[4].feg[0].cnl[2] = 34.0100000000;	data_table[4].feg[0].cl[3] = 0.5000000000;		data_table[4].feg[0].cnl[3] = 35.9800000000;	data_table[4].feg[0].cl[4] = 0.5000000000;		data_table[4].feg[0].cnl[4] = 36.6800000000;	data_table[4].feg[0].cl[5] = 0.5000000000;		data_table[4].feg[0].cnl[5] = 60.8100000000;
				data_table[5].feg[0].cl[0] = 0.5000000000;		data_table[5].feg[0].cnl[0] = 0.2946000000;	data_table[5].feg[0].cl[1] = 0.5000000000;		data_table[5].feg[0].cnl[1] = 3.9340000000;	data_table[5].feg[0].cl[2] = 0.5000000000;		data_table[5].feg[0].cnl[2] = 24.9800000000;	data_table[5].feg[0].cl[3] = 0.5000000000;		data_table[5].feg[0].cnl[3] = 25.2800000000;	data_table[5].feg[0].cl[4] = 0.5000000000;		data_table[5].feg[0].cnl[4] = 25.4700000000;	data_table[5].feg[0].cl[5] = 0.5000000000;		data_table[5].feg[0].cnl[5] = 46.7000000000;
				data_table[6].feg[0].cl[0] = 0.5000000000;		data_table[6].feg[0].cnl[0] = 0.2393000000;	data_table[6].feg[0].cl[1] = 0.5000000000;		data_table[6].feg[0].cnl[1] = 4.9350000000;	data_table[6].feg[0].cl[2] = 0.5000000000;		data_table[6].feg[0].cnl[2] = 18.1200000000;	data_table[6].feg[0].cl[3] = 0.5000000000;		data_table[6].feg[0].cnl[3] = 15.7000000000;	data_table[6].feg[0].cl[4] = 0.5000000000;		data_table[6].feg[0].cnl[4] = 15.8200000000;	data_table[6].feg[0].cl[5] = 0.5000000000;		data_table[6].feg[0].cnl[5] = 40.2400000000;
				data_table[7].feg[0].cl[0] = 0.5000000000;		data_table[7].feg[0].cnl[0] = 6.3760000000;	data_table[7].feg[0].cl[1] = 0.5000000000;		data_table[7].feg[0].cnl[1] = 8.0370000000;	data_table[7].feg[0].cl[2] = 0.5000000000;		data_table[7].feg[0].cnl[2] = 27.2100000000;	data_table[7].feg[0].cl[3] = 0.5000000000;		data_table[7].feg[0].cnl[3] = 0.1116000000;	data_table[7].feg[0].cl[4] = 0.5000000000;		data_table[7].feg[0].cnl[4] = 0.3869000000;	data_table[7].feg[0].cl[5] = 0.5000000000;		data_table[7].feg[0].cnl[5] = 10.9000000000;
				data_table[8].feg[0].cl[0] = 0.5000000000;		data_table[8].feg[0].cnl[0] = 0.2180000000;	data_table[8].feg[0].cl[1] = 0.5000000000;		data_table[8].feg[0].cnl[1] = 6.7700000000;	data_table[8].feg[0].cl[2] = 0.5000000000;		data_table[8].feg[0].cnl[2] = 7.0510000000;	data_table[8].feg[0].cl[3] = 0.5000000000;		data_table[8].feg[0].cnl[3] = 6.6750000000;	data_table[8].feg[0].cl[4] = 0.5000000000;		data_table[8].feg[0].cnl[4] = 12.3800000000;	data_table[8].feg[0].cl[5] = 0.5000000000;		data_table[8].feg[0].cnl[5] = 28.0800000000;
				data_table[9].feg[0].cl[0] = 0.5000000000;		data_table[9].feg[0].cnl[0] = 0.2006000000;	data_table[9].feg[0].cl[1] = 0.5000000000;		data_table[9].feg[0].cnl[1] = 5.4980000000;	data_table[9].feg[0].cl[2] = 0.5000000000;		data_table[9].feg[0].cnl[2] = 6.2810000000;	data_table[9].feg[0].cl[3] = 0.5000000000;		data_table[9].feg[0].cnl[3] = 7.1920000000;	data_table[9].feg[0].cl[4] = 0.5000000000;		data_table[9].feg[0].cnl[4] = 7.5480000000;	data_table[9].feg[0].cl[5] = 0.5000000000;		data_table[9].feg[0].cnl[5] = 23.2600000000;
				data_table[10].feg[0].cl[0] = 0.5000000000;	data_table[10].feg[0].cnl[0] = 0.2190000000;	data_table[10].feg[0].cl[1] = 0.5000000000;	data_table[10].feg[0].cnl[1] = 5.3000000000;	data_table[10].feg[0].cl[2] = 0.5000000000;	data_table[10].feg[0].cnl[2] = 5.3190000000;	data_table[10].feg[0].cl[3] = 0.5000000000;	data_table[10].feg[0].cnl[3] = 5.2830000000;	data_table[10].feg[0].cl[4] = 0.5000000000;	data_table[10].feg[0].cnl[4] = 5.2850000000;	data_table[10].feg[0].cl[5] = 0.5000000000;	data_table[10].feg[0].cnl[5] = 128.2000000000;
				data_table[11].feg[0].cl[0] = 0.5000000000;	data_table[11].feg[0].cnl[0] = 1.9760000000;	data_table[11].feg[0].cl[1] = 0.5000000000;	data_table[11].feg[0].cnl[1] = 2.8090000000;	data_table[11].feg[0].cl[2] = 0.5000000000;	data_table[11].feg[0].cnl[2] = 16.3900000000;	data_table[11].feg[0].cl[3] = 0.5000000000;	data_table[11].feg[0].cnl[3] = 0.0549000000;	data_table[11].feg[0].cl[4] = 0.5000000000;	data_table[11].feg[0].cnl[4] = 2.0610000000;	data_table[11].feg[0].cl[5] = 0.5000000000;	data_table[11].feg[0].cnl[5] = 121.7000000000;
				data_table[12].feg[0].cl[0] = 0.4000000000;	data_table[12].feg[0].cnl[0] = 2.2970000000;	data_table[12].feg[0].cl[1] = 0.4000000000;	data_table[12].feg[0].cnl[1] = 2.3580000000;	data_table[12].feg[0].cl[2] = 0.4000000000;	data_table[12].feg[0].cnl[2] = 24.9900000000;	data_table[12].feg[0].cl[3] = 0.4000000000;	data_table[12].feg[0].cnl[3] = 0.0746000000;	data_table[12].feg[0].cl[4] = 0.4000000000;	data_table[12].feg[0].cnl[4] = 0.5595000000;	data_table[12].feg[0].cl[5] = 0.4000000000;	data_table[12].feg[0].cnl[5] = 128.5000000000;
				data_table[13].feg[0].cl[0] = 0.5000000000;	data_table[13].feg[0].cnl[0] = 1.7370000000;	data_table[13].feg[0].cl[1] = 0.5000000000;	data_table[13].feg[0].cnl[1] = 3.0430000000;	data_table[13].feg[0].cl[2] = 0.5000000000;	data_table[13].feg[0].cnl[2] = 30.5700000000;	data_table[13].feg[0].cl[3] = 0.5000000000;	data_table[13].feg[0].cnl[3] = 0.0507000000;	data_table[13].feg[0].cl[4] = 0.5000000000;	data_table[13].feg[0].cnl[4] = 0.9918000000;	data_table[13].feg[0].cl[5] = 0.5000000000;	data_table[13].feg[0].cnl[5] = 86.1800000000;
				data_table[14].feg[0].cl[0] = 0.5000000000;	data_table[14].feg[0].cnl[0] = 0.1795000000;	data_table[14].feg[0].cl[1] = 0.5000000000;	data_table[14].feg[0].cnl[1] = 2.6320000000;	data_table[14].feg[0].cl[2] = 0.5000000000;	data_table[14].feg[0].cnl[2] = 2.6760000000;	data_table[14].feg[0].cl[3] = 0.5000000000;	data_table[14].feg[0].cnl[3] = 34.5700000000;	data_table[14].feg[0].cl[4] = 0.5000000000;	data_table[14].feg[0].cnl[4] = 36.7800000000;	data_table[14].feg[0].cl[5] = 0.5000000000;	data_table[14].feg[0].cnl[5] = 54.0600000000;
				data_table[15].feg[0].cl[0] = 0.5000000000;	data_table[15].feg[0].cnl[0] = 1.0060000000;	data_table[15].feg[0].cl[1] = 0.5000000000;	data_table[15].feg[0].cnl[1] = 4.9040000000;	data_table[15].feg[0].cl[2] = 0.5000000000;	data_table[15].feg[0].cnl[2] = 31.3500000000;	data_table[15].feg[0].cl[3] = 0.5000000000;	data_table[15].feg[0].cnl[3] = 0.0370000000;	data_table[15].feg[0].cl[4] = 0.5000000000;	data_table[15].feg[0].cnl[4] = 0.9870000000;	data_table[15].feg[0].cl[5] = 0.5000000000;	data_table[15].feg[0].cnl[5] = 44.9400000000;
				data_table[16].feg[0].cl[0] = 0.5000000000;	data_table[16].feg[0].cnl[0] = 0.1846000000;	data_table[16].feg[0].cl[1] = 0.5000000000;	data_table[16].feg[0].cnl[1] = 1.4800000000;	data_table[16].feg[0].cl[2] = 0.5000000000;	data_table[16].feg[0].cnl[2] = 5.2100000000;	data_table[16].feg[0].cl[3] = 0.5000000000;	data_table[16].feg[0].cnl[3] = 24.7900000000;	data_table[16].feg[0].cl[4] = 0.5000000000;	data_table[16].feg[0].cnl[4] = 32.0600000000;	data_table[16].feg[0].cl[5] = 0.5000000000;	data_table[16].feg[0].cnl[5] = 39.1000000000;
				data_table[17].feg[0].cl[0] = 0.5000000000;	data_table[17].feg[0].cnl[0] = 0.2006000000;	data_table[17].feg[0].cl[1] = 0.5000000000;	data_table[17].feg[0].cnl[1] = 6.5330000000;	data_table[17].feg[0].cl[2] = 0.5000000000;	data_table[17].feg[0].cnl[2] = 22.7200000000;	data_table[17].feg[0].cl[3] = 0.5000000000;	data_table[17].feg[0].cnl[3] = 1.2000000000;	data_table[17].feg[0].cl[4] = 0.5000000000;	data_table[17].feg[0].cnl[4] = 1.2740000000;	data_table[17].feg[0].cl[5] = 0.5000000000;	data_table[17].feg[0].cnl[5] = 36.2600000000;
				data_table[18].feg[0].cl[0] = 0.2000000000;	data_table[18].feg[0].cnl[0] = 0.4442000000;	data_table[18].feg[0].cl[1] = 0.2000000000;	data_table[18].feg[0].cnl[1] = 3.3670000000;	data_table[18].feg[0].cl[2] = 0.2000000000;	data_table[18].feg[0].cnl[2] = 19.6300000000;	data_table[18].feg[0].cl[3] = 0.2000000000;	data_table[18].feg[0].cnl[3] = 0.0182000000;	data_table[18].feg[0].cl[4] = 0.2000000000;	data_table[18].feg[0].cnl[4] = 23.5100000000;	data_table[18].feg[0].cl[5] = 0.2000000000;	data_table[18].feg[0].cnl[5] = 212.9000000000;
				data_table[19].feg[0].cl[0] = 0.3000000000;	data_table[19].feg[0].cnl[0] = 0.1827000000;	data_table[19].feg[0].cl[1] = 0.3000000000;	data_table[19].feg[0].cnl[1] = 2.0660000000;	data_table[19].feg[0].cl[2] = 0.3000000000;	data_table[19].feg[0].cnl[2] = 16.9900000000;	data_table[19].feg[0].cl[3] = 0.3000000000;	data_table[19].feg[0].cnl[3] = 11.5800000000;	data_table[19].feg[0].cl[4] = 0.3000000000;	data_table[19].feg[0].cnl[4] = 13.9800000000;	data_table[19].feg[0].cl[5] = 0.3000000000;	data_table[19].feg[0].cnl[5] = 186.1000000000;
				data_table[20].feg[0].cl[0] = 0.5000000000;	data_table[20].feg[0].cnl[0] = 0.1425000000;	data_table[20].feg[0].cl[1] = 0.5000000000;	data_table[20].feg[0].cnl[1] = 1.4660000000;	data_table[20].feg[0].cl[2] = 0.5000000000;	data_table[20].feg[0].cnl[2] = 15.4700000000;	data_table[20].feg[0].cl[3] = 0.5000000000;	data_table[20].feg[0].cnl[3] = 4.2430000000;	data_table[20].feg[0].cl[4] = 0.5000000000;	data_table[20].feg[0].cnl[4] = 9.8040000000;	data_table[20].feg[0].cl[5] = 0.5000000000;	data_table[20].feg[0].cnl[5] = 121.5000000000;
				data_table[21].feg[0].cl[0] = 0.5000000000;	data_table[21].feg[0].cnl[0] = 0.1278000000;	data_table[21].feg[0].cl[1] = 0.5000000000;	data_table[21].feg[0].cnl[1] = 1.4560000000;	data_table[21].feg[0].cl[2] = 0.5000000000;	data_table[21].feg[0].cnl[2] = 12.1000000000;	data_table[21].feg[0].cl[3] = 0.5000000000;	data_table[21].feg[0].cnl[3] = 4.6170000000;	data_table[21].feg[0].cl[4] = 0.5000000000;	data_table[21].feg[0].cnl[4] = 11.9700000000;	data_table[21].feg[0].cl[5] = 0.5000000000;	data_table[21].feg[0].cnl[5] = 105.0000000000;
				data_table[22].feg[0].cl[0] = 0.5000000000;	data_table[22].feg[0].cnl[0] = 0.1313000000;	data_table[22].feg[0].cl[1] = 0.5000000000;	data_table[22].feg[0].cnl[1] = 1.3990000000;	data_table[22].feg[0].cl[2] = 0.5000000000;	data_table[22].feg[0].cnl[2] = 8.0080000000;	data_table[22].feg[0].cl[3] = 0.5000000000;	data_table[22].feg[0].cnl[3] = 7.9810000000;	data_table[22].feg[0].cl[4] = 0.5000000000;	data_table[22].feg[0].cnl[4] = 13.4100000000;	data_table[22].feg[0].cl[5] = 0.5000000000;	data_table[22].feg[0].cnl[5] = 95.3100000000;
				data_table[23].feg[0].cl[0] = 0.5000000000;	data_table[23].feg[0].cnl[0] = 0.1231000000;	data_table[23].feg[0].cl[1] = 0.5000000000;	data_table[23].feg[0].cnl[1] = 2.3840000000;	data_table[23].feg[0].cl[2] = 0.5000000000;	data_table[23].feg[0].cnl[2] = 9.9210000000;	data_table[23].feg[0].cl[3] = 0.5000000000;	data_table[23].feg[0].cnl[3] = 1.6480000000;	data_table[23].feg[0].cl[4] = 0.5000000000;	data_table[23].feg[0].cnl[4] = 11.0000000000;	data_table[23].feg[0].cl[5] = 0.5000000000;	data_table[23].feg[0].cnl[5] = 68.4600000000;
				data_table[24].feg[0].cl[0] = 0.5000000000;	data_table[24].feg[0].cnl[0] = 0.4817000000;	data_table[24].feg[0].cl[1] = 0.5000000000;	data_table[24].feg[0].cnl[1] = 3.7830000000;	data_table[24].feg[0].cl[2] = 0.5000000000;	data_table[24].feg[0].cnl[2] = 8.4730000000;	data_table[24].feg[0].cl[3] = 0.5000000000;	data_table[24].feg[0].cnl[3] = 0.0469000000;	data_table[24].feg[0].cl[4] = 0.5000000000;	data_table[24].feg[0].cnl[4] = 8.7450000000;	data_table[24].feg[0].cl[5] = 0.5000000000;	data_table[24].feg[0].cnl[5] = 77.4400000000;
				data_table[25].feg[0].cl[0] = 0.5000000000;	data_table[25].feg[0].cnl[0] = 0.4470000000;	data_table[25].feg[0].cl[1] = 0.5000000000;	data_table[25].feg[0].cnl[1] = 6.8940000000;	data_table[25].feg[0].cl[2] = 0.5000000000;	data_table[25].feg[0].cnl[2] = 6.9030000000;	data_table[25].feg[0].cl[3] = 0.5000000000;	data_table[25].feg[0].cnl[3] = 0.0569000000;	data_table[25].feg[0].cl[4] = 0.5000000000;	data_table[25].feg[0].cnl[4] = 3.0260000000;	data_table[25].feg[0].cl[5] = 0.5000000000;	data_table[25].feg[0].cnl[5] = 70.8700000000;
				data_table[26].feg[0].cl[0] = 0.5000000000;	data_table[26].feg[0].cnl[0] = 0.1071000000;	data_table[26].feg[0].cl[1] = 0.5000000000;	data_table[26].feg[0].cnl[1] = 3.6360000000;	data_table[26].feg[0].cl[2] = 0.5000000000;	data_table[26].feg[0].cnl[2] = 7.5580000000;	data_table[26].feg[0].cl[3] = 0.5000000000;	data_table[26].feg[0].cnl[3] = 1.2800000000;	data_table[26].feg[0].cl[4] = 0.5000000000;	data_table[26].feg[0].cnl[4] = 5.1400000000;	data_table[26].feg[0].cl[5] = 0.5000000000;	data_table[26].feg[0].cnl[5] = 67.1600000000;
				data_table[27].feg[0].cl[0] = 0.5000000000;	data_table[27].feg[0].cnl[0] = 0.1107000000;	data_table[27].feg[0].cl[1] = 0.5000000000;	data_table[27].feg[0].cnl[1] = 1.6190000000;	data_table[27].feg[0].cl[2] = 0.5000000000;	data_table[27].feg[0].cnl[2] = 6.0030000000;	data_table[27].feg[0].cl[3] = 0.5000000000;	data_table[27].feg[0].cnl[3] = 5.9750000000;	data_table[27].feg[0].cl[4] = 0.5000000000;	data_table[27].feg[0].cnl[4] = 6.0600000000;	data_table[27].feg[0].cl[5] = 0.5000000000;	data_table[27].feg[0].cnl[5] = 59.4100000000;
				data_table[28].feg[0].cl[0] = 0.5000000000;	data_table[28].feg[0].cnl[0] = 0.1129000000;	data_table[28].feg[0].cl[1] = 0.5000000000;	data_table[28].feg[0].cnl[1] = 1.8910000000;	data_table[28].feg[0].cl[2] = 0.5000000000;	data_table[28].feg[0].cnl[2] = 5.0850000000;	data_table[28].feg[0].cl[3] = 0.5000000000;	data_table[28].feg[0].cnl[3] = 5.0730000000;	data_table[28].feg[0].cl[4] = 0.5000000000;	data_table[28].feg[0].cnl[4] = 5.0990000000;	data_table[28].feg[0].cl[5] = 0.5000000000;	data_table[28].feg[0].cnl[5] = 46.3900000000;
				data_table[29].feg[0].cl[0] = 0.5000000000;	data_table[29].feg[0].cnl[0] = 0.1021000000;	data_table[29].feg[0].cl[1] = 0.5000000000;	data_table[29].feg[0].cnl[1] = 1.7340000000;	data_table[29].feg[0].cl[2] = 0.5000000000;	data_table[29].feg[0].cnl[2] = 4.7830000000;	data_table[29].feg[0].cl[3] = 0.5000000000;	data_table[29].feg[0].cnl[3] = 4.8070000000;	data_table[29].feg[0].cl[4] = 0.5000000000;	data_table[29].feg[0].cnl[4] = 5.6450000000;	data_table[29].feg[0].cl[5] = 0.5000000000;	data_table[29].feg[0].cnl[5] = 51.2200000000;
				data_table[30].feg[0].cl[0] = 0.5000000000;	data_table[30].feg[0].cnl[0] = 0.1064000000;	data_table[30].feg[0].cl[1] = 0.5000000000;	data_table[30].feg[0].cnl[1] = 1.5370000000;	data_table[30].feg[0].cl[2] = 0.5000000000;	data_table[30].feg[0].cnl[2] = 5.1380000000;	data_table[30].feg[0].cl[3] = 0.5000000000;	data_table[30].feg[0].cnl[3] = 4.7430000000;	data_table[30].feg[0].cl[4] = 0.5000000000;	data_table[30].feg[0].cnl[4] = 5.0000000000;	data_table[30].feg[0].cl[5] = 0.5000000000;	data_table[30].feg[0].cnl[5] = 61.4300000000;
				data_table[31].feg[0].cl[0] = 0.5000000000;	data_table[31].feg[0].cnl[0] = 0.0958000000;	data_table[31].feg[0].cl[1] = 0.5000000000;	data_table[31].feg[0].cnl[1] = 1.6770000000;	data_table[31].feg[0].cl[2] = 0.5000000000;	data_table[31].feg[0].cnl[2] = 4.7030000000;	data_table[31].feg[0].cl[3] = 0.5000000000;	data_table[31].feg[0].cnl[3] = 2.9120000000;	data_table[31].feg[0].cl[4] = 0.5000000000;	data_table[31].feg[0].cnl[4] = 7.8700000000;	data_table[31].feg[0].cl[5] = 0.5000000000;	data_table[31].feg[0].cnl[5] = 64.9400000000;
				data_table[32].feg[0].cl[0] = 0.5000000000;	data_table[32].feg[0].cnl[0] = 0.0943000000;	data_table[32].feg[0].cl[1] = 0.5000000000;	data_table[32].feg[0].cnl[1] = 2.2140000000;	data_table[32].feg[0].cl[2] = 0.5000000000;	data_table[32].feg[0].cnl[2] = 3.9510000000;	data_table[32].feg[0].cl[3] = 0.5000000000;	data_table[32].feg[0].cnl[3] = 1.5210000000;	data_table[32].feg[0].cl[4] = 0.5000000000;	data_table[32].feg[0].cnl[4] = 15.8100000000;	data_table[32].feg[0].cl[5] = 0.5000000000;	data_table[32].feg[0].cnl[5] = 52.4100000000;
				data_table[33].feg[0].cl[0] = 0.5000000000;	data_table[33].feg[0].cnl[0] = 0.0925000000;	data_table[33].feg[0].cl[1] = 0.5000000000;	data_table[33].feg[0].cnl[1] = 1.6020000000;	data_table[33].feg[0].cl[2] = 0.5000000000;	data_table[33].feg[0].cnl[2] = 3.0490000000;	data_table[33].feg[0].cl[3] = 0.5000000000;	data_table[33].feg[0].cnl[3] = 3.1850000000;	data_table[33].feg[0].cl[4] = 0.5000000000;	data_table[33].feg[0].cnl[4] = 18.9400000000;	data_table[33].feg[0].cl[5] = 0.5000000000;	data_table[33].feg[0].cnl[5] = 47.6300000000;
				data_table[34].feg[0].cl[0] = 0.5000000000;	data_table[34].feg[0].cnl[0] = 0.0925000000;	data_table[34].feg[0].cl[1] = 0.5000000000;	data_table[34].feg[0].cnl[1] = 1.7730000000;	data_table[34].feg[0].cl[2] = 0.5000000000;	data_table[34].feg[0].cnl[2] = 3.4810000000;	data_table[34].feg[0].cl[3] = 0.5000000000;	data_table[34].feg[0].cnl[3] = 1.8840000000;	data_table[34].feg[0].cl[4] = 0.5000000000;	data_table[34].feg[0].cnl[4] = 22.6900000000;	data_table[34].feg[0].cl[5] = 0.5000000000;	data_table[34].feg[0].cnl[5] = 40.6900000000;
				data_table[35].feg[0].cl[0] = 0.5000000000;	data_table[35].feg[0].cnl[0] = 0.4932000000;	data_table[35].feg[0].cl[1] = 0.5000000000;	data_table[35].feg[0].cnl[1] = 2.0830000000;	data_table[35].feg[0].cl[2] = 0.5000000000;	data_table[35].feg[0].cnl[2] = 11.4100000000;	data_table[35].feg[0].cl[3] = 0.5000000000;	data_table[35].feg[0].cnl[3] = 0.0333000000;	data_table[35].feg[0].cl[4] = 0.5000000000;	data_table[35].feg[0].cnl[4] = 2.0970000000;	data_table[35].feg[0].cl[5] = 0.5000000000;	data_table[35].feg[0].cnl[5] = 42.3800000000;
				data_table[36].feg[0].cl[0] = 0.2000000000;	data_table[36].feg[0].cnl[0] = 0.1580000000;	data_table[36].feg[0].cl[1] = 0.2000000000;	data_table[36].feg[0].cnl[1] = 1.7150000000;	data_table[36].feg[0].cl[2] = 0.2000000000;	data_table[36].feg[0].cnl[2] = 9.3920000000;	data_table[36].feg[0].cl[3] = 0.2000000000;	data_table[36].feg[0].cnl[3] = 1.6750000000;	data_table[36].feg[0].cl[4] = 0.2000000000;	data_table[36].feg[0].cnl[4] = 23.5900000000;	data_table[36].feg[0].cl[5] = 0.2000000000;	data_table[36].feg[0].cnl[5] = 152.5000000000;
				data_table[37].feg[0].cl[0] = 0.3000000000;	data_table[37].feg[0].cnl[0] = 0.3605000000;	data_table[37].feg[0].cl[1] = 0.3000000000;	data_table[37].feg[0].cnl[1] = 2.1280000000;	data_table[37].feg[0].cl[2] = 0.3000000000;	data_table[37].feg[0].cnl[2] = 12.4600000000;	data_table[37].feg[0].cl[3] = 0.3000000000;	data_table[37].feg[0].cnl[3] = 0.0153000000;	data_table[37].feg[0].cl[4] = 0.3000000000;	data_table[37].feg[0].cnl[4] = 2.1080000000;	data_table[37].feg[0].cl[5] = 0.3000000000;	data_table[37].feg[0].cnl[5] = 133.2000000000;
				data_table[38].feg[0].cl[0] = 0.5000000000;	data_table[38].feg[0].cnl[0] = 0.0900000000;	data_table[38].feg[0].cl[1] = 0.5000000000;	data_table[38].feg[0].cnl[1] = 1.4140000000;	data_table[38].feg[0].cl[2] = 0.5000000000;	data_table[38].feg[0].cnl[2] = 2.0530000000;	data_table[38].feg[0].cl[3] = 0.5000000000;	data_table[38].feg[0].cnl[3] = 10.2600000000;	data_table[38].feg[0].cl[4] = 0.5000000000;	data_table[38].feg[0].cnl[4] = 10.7500000000;	data_table[38].feg[0].cl[5] = 0.5000000000;	data_table[38].feg[0].cnl[5] = 90.6400000000;
				data_table[39].feg[0].cl[0] = 0.5000000000;	data_table[39].feg[0].cnl[0] = 0.1009000000;	data_table[39].feg[0].cl[1] = 0.5000000000;	data_table[39].feg[0].cnl[1] = 1.1540000000;	data_table[39].feg[0].cl[2] = 0.5000000000;	data_table[39].feg[0].cnl[2] = 2.3470000000;	data_table[39].feg[0].cl[3] = 0.5000000000;	data_table[39].feg[0].cnl[3] = 10.5800000000;	data_table[39].feg[0].cl[4] = 0.5000000000;	data_table[39].feg[0].cnl[4] = 10.9500000000;	data_table[39].feg[0].cl[5] = 0.5000000000;	data_table[39].feg[0].cnl[5] = 82.8200000000;
				data_table[40].feg[0].cl[0] = 0.5000000000;	data_table[40].feg[0].cnl[0] = 0.0924000000;	data_table[40].feg[0].cl[1] = 0.5000000000;	data_table[40].feg[0].cnl[1] = 1.1700000000;	data_table[40].feg[0].cl[2] = 0.5000000000;	data_table[40].feg[0].cnl[2] = 5.9400000000;	data_table[40].feg[0].cl[3] = 0.5000000000;	data_table[40].feg[0].cnl[3] = 1.3060000000;	data_table[40].feg[0].cl[4] = 0.5000000000;	data_table[40].feg[0].cnl[4] = 13.4300000000;	data_table[40].feg[0].cl[5] = 0.5000000000;	data_table[40].feg[0].cnl[5] = 66.3700000000;
				data_table[41].feg[0].cl[0] = 0.5000000000;	data_table[41].feg[0].cnl[0] = 0.4354000000;	data_table[41].feg[0].cl[1] = 0.5000000000;	data_table[41].feg[0].cnl[1] = 1.2480000000;	data_table[41].feg[0].cl[2] = 0.5000000000;	data_table[41].feg[0].cnl[2] = 7.4540000000;	data_table[41].feg[0].cl[3] = 0.5000000000;	data_table[41].feg[0].cnl[3] = 0.0354000000;	data_table[41].feg[0].cl[4] = 0.5000000000;	data_table[41].feg[0].cnl[4] = 9.9140000000;	data_table[41].feg[0].cl[5] = 0.5000000000;	data_table[41].feg[0].cnl[5] = 61.7200000000;
				data_table[42].feg[0].cl[0] = 0.5000000000;	data_table[42].feg[0].cnl[0] = 0.4594000000;	data_table[42].feg[0].cl[1] = 0.5000000000;	data_table[42].feg[0].cnl[1] = 1.1820000000;	data_table[42].feg[0].cl[2] = 0.5000000000;	data_table[42].feg[0].cnl[2] = 8.3170000000;	data_table[42].feg[0].cl[3] = 0.5000000000;	data_table[42].feg[0].cnl[3] = 0.0323000000;	data_table[42].feg[0].cl[4] = 0.5000000000;	data_table[42].feg[0].cnl[4] = 8.3230000000;	data_table[42].feg[0].cl[5] = 0.5000000000;	data_table[42].feg[0].cnl[5] = 64.9800000000;
				data_table[43].feg[0].cl[0] = 0.4000000000;	data_table[43].feg[0].cnl[0] = 0.0860000000;	data_table[43].feg[0].cl[1] = 0.4000000000;	data_table[43].feg[0].cnl[1] = 1.3960000000;	data_table[43].feg[0].cl[2] = 0.4000000000;	data_table[43].feg[0].cnl[2] = 11.7000000000;	data_table[43].feg[0].cl[3] = 0.4000000000;	data_table[43].feg[0].cnl[3] = 1.3960000000;	data_table[43].feg[0].cl[4] = 0.4000000000;	data_table[43].feg[0].cnl[4] = 3.4520000000;	data_table[43].feg[0].cl[5] = 0.4000000000;	data_table[43].feg[0].cnl[5] = 55.5600000000;
				data_table[44].feg[0].cl[0] = 0.5000000000;	data_table[44].feg[0].cnl[0] = 0.0921000000;	data_table[44].feg[0].cl[1] = 0.5000000000;	data_table[44].feg[0].cnl[1] = 1.1130000000;	data_table[44].feg[0].cl[2] = 0.5000000000;	data_table[44].feg[0].cnl[2] = 7.6580000000;	data_table[44].feg[0].cl[3] = 0.5000000000;	data_table[44].feg[0].cnl[3] = 1.1260000000;	data_table[44].feg[0].cl[4] = 0.5000000000;	data_table[44].feg[0].cnl[4] = 8.3250000000;	data_table[44].feg[0].cl[5] = 0.5000000000;	data_table[44].feg[0].cnl[5] = 48.3800000000;
				data_table[45].feg[0].cl[0] = 0.5000000000;	data_table[45].feg[0].cnl[0] = 0.0901000000;	data_table[45].feg[0].cl[1] = 0.5000000000;	data_table[45].feg[0].cnl[1] = 1.1250000000;	data_table[45].feg[0].cl[2] = 0.5000000000;	data_table[45].feg[0].cnl[2] = 9.6980000000;	data_table[45].feg[0].cl[3] = 0.5000000000;	data_table[45].feg[0].cnl[3] = 1.0850000000;	data_table[45].feg[0].cl[4] = 0.5000000000;	data_table[45].feg[0].cnl[4] = 5.7090000000;	data_table[45].feg[0].cl[5] = 0.5000000000;	data_table[45].feg[0].cnl[5] = 33.4900000000;
				data_table[46].feg[0].cl[0] = 0.5000000000;	data_table[46].feg[0].cnl[0] = 0.0894000000;	data_table[46].feg[0].cl[1] = 0.5000000000;	data_table[46].feg[0].cnl[1] = 3.1910000000;	data_table[46].feg[0].cl[2] = 0.5000000000;	data_table[46].feg[0].cnl[2] = 9.1000000000;	data_table[46].feg[0].cl[3] = 0.5000000000;	data_table[46].feg[0].cnl[3] = 0.8090000000;	data_table[46].feg[0].cl[4] = 0.5000000000;	data_table[46].feg[0].cnl[4] = 0.8144000000;	data_table[46].feg[0].cl[5] = 0.5000000000;	data_table[46].feg[0].cnl[5] = 41.3400000000;
				data_table[47].feg[0].cl[0] = 0.3000000000;	data_table[47].feg[0].cnl[0] = 0.2885000000;	data_table[47].feg[0].cl[1] = 0.3000000000;	data_table[47].feg[0].cnl[1] = 1.6130000000;	data_table[47].feg[0].cl[2] = 0.3000000000;	data_table[47].feg[0].cnl[2] = 8.9970000000;	data_table[47].feg[0].cl[3] = 0.3000000000;	data_table[47].feg[0].cnl[3] = 0.0171000000;	data_table[47].feg[0].cl[4] = 0.3000000000;	data_table[47].feg[0].cnl[4] = 9.4670000000;	data_table[47].feg[0].cl[5] = 0.3000000000;	data_table[47].feg[0].cnl[5] = 58.1300000000;
				data_table[48].feg[0].cl[0] = 0.4000000000;	data_table[48].feg[0].cnl[0] = 0.0895000000;	data_table[48].feg[0].cl[1] = 0.4000000000;	data_table[48].feg[0].cnl[1] = 1.2330000000;	data_table[48].feg[0].cl[2] = 0.4000000000;	data_table[48].feg[0].cnl[2] = 8.2310000000;	data_table[48].feg[0].cl[3] = 0.4000000000;	data_table[48].feg[0].cnl[3] = 1.2240000000;	data_table[48].feg[0].cl[4] = 0.4000000000;	data_table[48].feg[0].cnl[4] = 7.0620000000;	data_table[48].feg[0].cl[5] = 0.4000000000;	data_table[48].feg[0].cnl[5] = 59.7000000000;
				data_table[49].feg[0].cl[0] = 0.6000000000;	data_table[49].feg[0].cnl[0] = 0.0712000000;	data_table[49].feg[0].cl[1] = 0.6000000000;	data_table[49].feg[0].cnl[1] = 0.8553000000;	data_table[49].feg[0].cl[2] = 0.6000000000;	data_table[49].feg[0].cnl[2] = 6.4010000000;	data_table[49].feg[0].cl[3] = 0.6000000000;	data_table[49].feg[0].cnl[3] = 1.3360000000;	data_table[49].feg[0].cl[4] = 0.6000000000;	data_table[49].feg[0].cnl[4] = 6.3820000000;	data_table[49].feg[0].cl[5] = 0.6000000000;	data_table[49].feg[0].cnl[5] = 50.9200000000;
				data_table[50].feg[0].cl[0] = 0.6000000000;	data_table[50].feg[0].cnl[0] = 0.3575000000;	data_table[50].feg[0].cl[1] = 0.6000000000;	data_table[50].feg[0].cnl[1] = 1.3250000000;	data_table[50].feg[0].cl[2] = 0.6000000000;	data_table[50].feg[0].cnl[2] = 6.5170000000;	data_table[50].feg[0].cl[3] = 0.6000000000;	data_table[50].feg[0].cnl[3] = 0.0355000000;	data_table[50].feg[0].cl[4] = 0.6000000000;	data_table[50].feg[0].cnl[4] = 6.5190000000;	data_table[50].feg[0].cl[5] = 0.6000000000;	data_table[50].feg[0].cnl[5] = 50.8100000000;
				data_table[51].feg[0].cl[0] = 0.6000000000;	data_table[51].feg[0].cnl[0] = 0.5009000000;	data_table[51].feg[0].cl[1] = 0.6000000000;	data_table[51].feg[0].cnl[1] = 3.9530000000;	data_table[51].feg[0].cl[2] = 0.6000000000;	data_table[51].feg[0].cnl[2] = 7.6280000000;	data_table[51].feg[0].cl[3] = 0.6000000000;	data_table[51].feg[0].cnl[3] = 0.0301000000;	data_table[51].feg[0].cl[4] = 0.6000000000;	data_table[51].feg[0].cnl[4] = 0.5074000000;	data_table[51].feg[0].cl[5] = 0.6000000000;	data_table[51].feg[0].cnl[5] = 49.6300000000;
				data_table[52].feg[0].cl[0] = 0.4000000000;	data_table[52].feg[0].cnl[0] = 0.0843000000;	data_table[52].feg[0].cl[1] = 0.4000000000;	data_table[52].feg[0].cnl[1] = 1.1300000000;	data_table[52].feg[0].cl[2] = 0.4000000000;	data_table[52].feg[0].cnl[2] = 8.8620000000;	data_table[52].feg[0].cl[3] = 0.4000000000;	data_table[52].feg[0].cnl[3] = 1.1300000000;	data_table[52].feg[0].cl[4] = 0.4000000000;	data_table[52].feg[0].cnl[4] = 9.1320000000;	data_table[52].feg[0].cl[5] = 0.4000000000;	data_table[52].feg[0].cnl[5] = 56.0200000000;
				data_table[53].feg[0].cl[0] = 0.4000000000;	data_table[53].feg[0].cnl[0] = 0.2780000000;	data_table[53].feg[0].cl[1] = 0.4000000000;	data_table[53].feg[0].cnl[1] = 1.6210000000;	data_table[53].feg[0].cl[2] = 0.4000000000;	data_table[53].feg[0].cnl[2] = 11.4500000000;	data_table[53].feg[0].cl[3] = 0.4000000000;	data_table[53].feg[0].cnl[3] = 0.0203000000;	data_table[53].feg[0].cl[4] = 0.4000000000;	data_table[53].feg[0].cnl[4] = 3.2750000000;	data_table[53].feg[0].cl[5] = 0.4000000000;	data_table[53].feg[0].cnl[5] = 51.4400000000;
				data_table[54].feg[0].cl[0] = 0.1000000000;	data_table[54].feg[0].cnl[0] = 0.1204000000;	data_table[54].feg[0].cl[1] = 0.1000000000;	data_table[54].feg[0].cnl[1] = 1.5370000000;	data_table[54].feg[0].cl[2] = 0.1000000000;	data_table[54].feg[0].cnl[2] = 9.8160000000;	data_table[54].feg[0].cl[3] = 0.1000000000;	data_table[54].feg[0].cnl[3] = 41.2200000000;	data_table[54].feg[0].cl[4] = 0.1000000000;	data_table[54].feg[0].cnl[4] = 42.6200000000;	data_table[54].feg[0].cl[5] = 0.1000000000;	data_table[54].feg[0].cnl[5] = 224.3000000000;
				data_table[55].feg[0].cl[0] = 0.1000000000;	data_table[55].feg[0].cnl[0] = 0.1223000000;	data_table[55].feg[0].cl[1] = 0.1000000000;	data_table[55].feg[0].cnl[1] = 1.4490000000;	data_table[55].feg[0].cl[2] = 0.1000000000;	data_table[55].feg[0].cnl[2] = 9.5020000000;	data_table[55].feg[0].cl[3] = 0.1000000000;	data_table[55].feg[0].cnl[3] = 49.4100000000;	data_table[55].feg[0].cl[4] = 0.1000000000;	data_table[55].feg[0].cnl[4] = 74.9500000000;	data_table[55].feg[0].cl[5] = 0.1000000000;	data_table[55].feg[0].cnl[5] = 217.0000000000;
				data_table[56].feg[0].cl[0] = 0.3000000000;	data_table[56].feg[0].cnl[0] = 0.0893000000;	data_table[56].feg[0].cl[1] = 0.3000000000;	data_table[56].feg[0].cnl[1] = 1.2620000000;	data_table[56].feg[0].cl[2] = 0.3000000000;	data_table[56].feg[0].cnl[2] = 8.0970000000;	data_table[56].feg[0].cl[3] = 0.3000000000;	data_table[56].feg[0].cnl[3] = 1.2030000000;	data_table[56].feg[0].cl[4] = 0.3000000000;	data_table[56].feg[0].cnl[4] = 17.6600000000;	data_table[56].feg[0].cl[5] = 0.3000000000;	data_table[56].feg[0].cnl[5] = 116.6000000000;
				data_table[57].feg[0].cl[0] = 0.3000000000;	data_table[57].feg[0].cnl[0] = 0.0850000000;	data_table[57].feg[0].cl[1] = 0.3000000000;	data_table[57].feg[0].cnl[1] = 1.2830000000;	data_table[57].feg[0].cl[2] = 0.3000000000;	data_table[57].feg[0].cnl[2] = 11.2200000000;	data_table[57].feg[0].cl[3] = 0.3000000000;	data_table[57].feg[0].cnl[3] = 1.3270000000;	data_table[57].feg[0].cl[4] = 0.3000000000;	data_table[57].feg[0].cnl[4] = 4.6100000000;	data_table[57].feg[0].cl[5] = 0.3000000000;	data_table[57].feg[0].cnl[5] = 112.2000000000;
				data_table[58].feg[0].cl[0] = 0.2000000000;	data_table[58].feg[0].cnl[0] = 0.0981000000;	data_table[58].feg[0].cl[1] = 0.2000000000;	data_table[58].feg[0].cnl[1] = 1.5260000000;	data_table[58].feg[0].cl[2] = 0.2000000000;	data_table[58].feg[0].cnl[2] = 8.5900000000;	data_table[58].feg[0].cl[3] = 0.2000000000;	data_table[58].feg[0].cnl[3] = 1.2390000000;	data_table[58].feg[0].cl[4] = 0.2000000000;	data_table[58].feg[0].cnl[4] = 22.4900000000;	data_table[58].feg[0].cl[5] = 0.2000000000;	data_table[58].feg[0].cnl[5] = 140.0000000000;
				data_table[59].feg[0].cl[0] = 0.2000000000;	data_table[59].feg[0].cnl[0] = 0.0941000000;	data_table[59].feg[0].cl[1] = 0.2000000000;	data_table[59].feg[0].cnl[1] = 1.2660000000;	data_table[59].feg[0].cl[2] = 0.2000000000;	data_table[59].feg[0].cnl[2] = 5.9880000000;	data_table[59].feg[0].cl[3] = 0.2000000000;	data_table[59].feg[0].cnl[3] = 17.7900000000;	data_table[59].feg[0].cl[4] = 0.2000000000;	data_table[59].feg[0].cnl[4] = 18.1400000000;	data_table[59].feg[0].cl[5] = 0.2000000000;	data_table[59].feg[0].cnl[5] = 132.6000000000;
				data_table[60].feg[0].cl[0] = 0.2000000000;	data_table[60].feg[0].cnl[0] = 0.0945000000;	data_table[60].feg[0].cl[1] = 0.2000000000;	data_table[60].feg[0].cnl[1] = 1.2510000000;	data_table[60].feg[0].cl[2] = 0.2000000000;	data_table[60].feg[0].cnl[2] = 5.9120000000;	data_table[60].feg[0].cl[3] = 0.2000000000;	data_table[60].feg[0].cnl[3] = 16.2900000000;	data_table[60].feg[0].cl[4] = 0.2000000000;	data_table[60].feg[0].cnl[4] = 16.7300000000;	data_table[60].feg[0].cl[5] = 0.2000000000;	data_table[60].feg[0].cnl[5] = 127.9000000000;
				data_table[61].feg[0].cl[0] = 0.2000000000;	data_table[61].feg[0].cnl[0] = 0.0906000000;	data_table[61].feg[0].cl[1] = 0.2000000000;	data_table[61].feg[0].cnl[1] = 1.5930000000;	data_table[61].feg[0].cl[2] = 0.2000000000;	data_table[61].feg[0].cnl[2] = 10.6400000000;	data_table[61].feg[0].cl[3] = 0.2000000000;	data_table[61].feg[0].cnl[3] = 1.7890000000;	data_table[61].feg[0].cl[4] = 0.2000000000;	data_table[61].feg[0].cnl[4] = 2.2210000000;	data_table[61].feg[0].cl[5] = 0.2000000000;	data_table[61].feg[0].cnl[5] = 124.6000000000;
				data_table[62].feg[0].cl[0] = 0.1000000000;	data_table[62].feg[0].cnl[0] = 0.1049000000;	data_table[62].feg[0].cl[1] = 0.1000000000;	data_table[62].feg[0].cnl[1] = 1.5440000000;	data_table[62].feg[0].cl[2] = 0.1000000000;	data_table[62].feg[0].cnl[2] = 8.6520000000;	data_table[62].feg[0].cl[3] = 0.1000000000;	data_table[62].feg[0].cnl[3] = 7.0930000000;	data_table[62].feg[0].cl[4] = 0.1000000000;	data_table[62].feg[0].cnl[4] = 53.3700000000;	data_table[62].feg[0].cl[5] = 0.1000000000;	data_table[62].feg[0].cnl[5] = 183.7000000000;
				data_table[63].feg[0].cl[0] = 0.2000000000;	data_table[63].feg[0].cnl[0] = 0.0934000000;	data_table[63].feg[0].cl[1] = 0.2000000000;	data_table[63].feg[0].cnl[1] = 1.3870000000;	data_table[63].feg[0].cl[2] = 0.2000000000;	data_table[63].feg[0].cnl[2] = 7.3590000000;	data_table[63].feg[0].cl[3] = 0.2000000000;	data_table[63].feg[0].cnl[3] = 1.5510000000;	data_table[63].feg[0].cl[4] = 0.2000000000;	data_table[63].feg[0].cnl[4] = 20.8200000000;	data_table[63].feg[0].cl[5] = 0.2000000000;	data_table[63].feg[0].cnl[5] = 111.0000000000;
				data_table[64].feg[0].cl[0] = 0.1000000000;	data_table[64].feg[0].cnl[0] = 0.1019000000;	data_table[64].feg[0].cl[1] = 0.1000000000;	data_table[64].feg[0].cnl[1] = 1.5240000000;	data_table[64].feg[0].cl[2] = 0.1000000000;	data_table[64].feg[0].cnl[2] = 7.1690000000;	data_table[64].feg[0].cl[3] = 0.1000000000;	data_table[64].feg[0].cnl[3] = 20.8600000000;	data_table[64].feg[0].cl[4] = 0.1000000000;	data_table[64].feg[0].cnl[4] = 49.2900000000;	data_table[64].feg[0].cl[5] = 0.1000000000;	data_table[64].feg[0].cnl[5] = 166.1000000000;
				data_table[65].feg[0].cl[0] = 0.2000000000;	data_table[65].feg[0].cnl[0] = 0.0840000000;	data_table[65].feg[0].cl[1] = 0.2000000000;	data_table[65].feg[0].cnl[1] = 1.4090000000;	data_table[65].feg[0].cl[2] = 0.2000000000;	data_table[65].feg[0].cnl[2] = 7.1400000000;	data_table[65].feg[0].cl[3] = 0.2000000000;	data_table[65].feg[0].cnl[3] = 1.3480000000;	data_table[65].feg[0].cl[4] = 0.2000000000;	data_table[65].feg[0].cnl[4] = 11.4200000000;	data_table[65].feg[0].cl[5] = 0.2000000000;	data_table[65].feg[0].cnl[5] = 108.0000000000;
				data_table[66].feg[0].cl[0] = 0.1000000000;	data_table[66].feg[0].cnl[0] = 0.0944000000;	data_table[66].feg[0].cl[1] = 0.1000000000;	data_table[66].feg[0].cnl[1] = 1.6180000000;	data_table[66].feg[0].cl[2] = 0.1000000000;	data_table[66].feg[0].cnl[2] = 6.2710000000;	data_table[66].feg[0].cl[3] = 0.1000000000;	data_table[66].feg[0].cnl[3] = 40.3500000000;	data_table[66].feg[0].cl[4] = 0.1000000000;	data_table[66].feg[0].cnl[4] = 42.8300000000;	data_table[66].feg[0].cl[5] = 0.1000000000;	data_table[66].feg[0].cnl[5] = 130.6000000000;
				data_table[67].feg[0].cl[0] = 0.2000000000;	data_table[67].feg[0].cnl[0] = 0.0821000000;	data_table[67].feg[0].cl[1] = 0.2000000000;	data_table[67].feg[0].cnl[1] = 1.2510000000;	data_table[67].feg[0].cl[2] = 0.2000000000;	data_table[67].feg[0].cnl[2] = 4.8120000000;	data_table[67].feg[0].cl[3] = 0.2000000000;	data_table[67].feg[0].cnl[3] = 10.8400000000;	data_table[67].feg[0].cl[4] = 0.2000000000;	data_table[67].feg[0].cnl[4] = 10.9000000000;	data_table[67].feg[0].cl[5] = 0.2000000000;	data_table[67].feg[0].cnl[5] = 100.1000000000;
				data_table[68].feg[0].cl[0] = 0.1000000000;	data_table[68].feg[0].cnl[0] = 0.0966000000;	data_table[68].feg[0].cl[1] = 0.1000000000;	data_table[68].feg[0].cnl[1] = 1.6020000000;	data_table[68].feg[0].cl[2] = 0.1000000000;	data_table[68].feg[0].cnl[2] = 5.6750000000;	data_table[68].feg[0].cl[3] = 0.1000000000;	data_table[68].feg[0].cnl[3] = 30.5900000000;	data_table[68].feg[0].cl[4] = 0.1000000000;	data_table[68].feg[0].cnl[4] = 31.1300000000;	data_table[68].feg[0].cl[5] = 0.1000000000;	data_table[68].feg[0].cnl[5] = 138.7000000000;
				data_table[69].feg[0].cl[0] = 0.1000000000;	data_table[69].feg[0].cnl[0] = 0.0949000000;	data_table[69].feg[0].cl[1] = 0.1000000000;	data_table[69].feg[0].cnl[1] = 1.6020000000;	data_table[69].feg[0].cl[2] = 0.1000000000;	data_table[69].feg[0].cnl[2] = 5.4390000000;	data_table[69].feg[0].cl[3] = 0.1000000000;	data_table[69].feg[0].cnl[3] = 28.3100000000;	data_table[69].feg[0].cl[4] = 0.1000000000;	data_table[69].feg[0].cnl[4] = 29.2800000000;	data_table[69].feg[0].cl[5] = 0.1000000000;	data_table[69].feg[0].cnl[5] = 138.1000000000;
				data_table[70].feg[0].cl[0] = 0.1000000000;	data_table[70].feg[0].cnl[0] = 0.0966000000;	data_table[70].feg[0].cl[1] = 0.1000000000;	data_table[70].feg[0].cnl[1] = 1.5680000000;	data_table[70].feg[0].cl[2] = 0.1000000000;	data_table[70].feg[0].cnl[2] = 5.3220000000;	data_table[70].feg[0].cl[3] = 0.1000000000;	data_table[70].feg[0].cnl[3] = 34.1800000000;	data_table[70].feg[0].cl[4] = 0.1000000000;	data_table[70].feg[0].cnl[4] = 35.2500000000;	data_table[70].feg[0].cl[5] = 0.1000000000;	data_table[70].feg[0].cnl[5] = 121.4000000000;
				data_table[71].feg[0].cl[0] = 0.1000000000;	data_table[71].feg[0].cnl[0] = 0.0929000000;	data_table[71].feg[0].cl[1] = 0.1000000000;	data_table[71].feg[0].cnl[1] = 1.5550000000;	data_table[71].feg[0].cl[2] = 0.1000000000;	data_table[71].feg[0].cnl[2] = 5.2510000000;	data_table[71].feg[0].cl[3] = 0.1000000000;	data_table[71].feg[0].cnl[3] = 37.5200000000;	data_table[71].feg[0].cl[4] = 0.1000000000;	data_table[71].feg[0].cnl[4] = 38.8800000000;	data_table[71].feg[0].cl[5] = 0.1000000000;	data_table[71].feg[0].cnl[5] = 105.2000000000;
				data_table[72].feg[0].cl[0] = 0.4000000000;	data_table[72].feg[0].cnl[0] = 0.0630000000;	data_table[72].feg[0].cl[1] = 0.4000000000;	data_table[72].feg[0].cnl[1] = 0.8195000000;	data_table[72].feg[0].cl[2] = 0.4000000000;	data_table[72].feg[0].cnl[2] = 2.8910000000;	data_table[72].feg[0].cl[3] = 0.4000000000;	data_table[72].feg[0].cnl[3] = 5.5430000000;	data_table[72].feg[0].cl[4] = 0.4000000000;	data_table[72].feg[0].cnl[4] = 5.9810000000;	data_table[72].feg[0].cl[5] = 0.4000000000;	data_table[72].feg[0].cnl[5] = 54.4200000000;
				data_table[73].feg[0].cl[0] = 0.2000000000;	data_table[73].feg[0].cnl[0] = 0.0790000000;	data_table[73].feg[0].cl[1] = 0.2000000000;	data_table[73].feg[0].cnl[1] = 1.3710000000;	data_table[73].feg[0].cl[2] = 0.2000000000;	data_table[73].feg[0].cnl[2] = 8.2340000000;	data_table[73].feg[0].cl[3] = 0.2000000000;	data_table[73].feg[0].cnl[3] = 1.3830000000;	data_table[73].feg[0].cl[4] = 0.2000000000;	data_table[73].feg[0].cnl[4] = 1.3920000000;	data_table[73].feg[0].cl[5] = 0.2000000000;	data_table[73].feg[0].cnl[5] = 77.1200000000;
				data_table[74].feg[0].cl[0] = 0.5000000000;	data_table[74].feg[0].cnl[0] = 0.0527000000;	data_table[74].feg[0].cl[1] = 0.5000000000;	data_table[74].feg[0].cnl[1] = 0.9072000000;	data_table[74].feg[0].cl[2] = 0.5000000000;	data_table[74].feg[0].cnl[2] = 4.4380000000;	data_table[74].feg[0].cl[3] = 0.5000000000;	data_table[74].feg[0].cnl[3] = 0.9459000000;	data_table[74].feg[0].cl[4] = 0.5000000000;	data_table[74].feg[0].cnl[4] = 4.3750000000;	data_table[74].feg[0].cl[5] = 0.5000000000;	data_table[74].feg[0].cnl[5] = 43.9800000000;
				data_table[75].feg[0].cl[0] = 0.4000000000;	data_table[75].feg[0].cnl[0] = 0.2270000000;	data_table[75].feg[0].cl[1] = 0.4000000000;	data_table[75].feg[0].cnl[1] = 1.5700000000;	data_table[75].feg[0].cl[2] = 0.4000000000;	data_table[75].feg[0].cnl[2] = 6.3450000000;	data_table[75].feg[0].cl[3] = 0.4000000000;	data_table[75].feg[0].cnl[3] = 0.0156000000;	data_table[75].feg[0].cl[4] = 0.4000000000;	data_table[75].feg[0].cnl[4] = 1.6180000000;	data_table[75].feg[0].cl[5] = 0.4000000000;	data_table[75].feg[0].cnl[5] = 46.1600000000;
				data_table[76].feg[0].cl[0] = 0.5000000000;	data_table[76].feg[0].cnl[0] = 0.0506000000;	data_table[76].feg[0].cl[1] = 0.5000000000;	data_table[76].feg[0].cnl[1] = 0.8677000000;	data_table[76].feg[0].cl[2] = 0.5000000000;	data_table[76].feg[0].cnl[2] = 5.0930000000;	data_table[76].feg[0].cl[3] = 0.5000000000;	data_table[76].feg[0].cnl[3] = 0.8812000000;	data_table[76].feg[0].cl[4] = 0.5000000000;	data_table[76].feg[0].cnl[4] = 3.5690000000;	data_table[76].feg[0].cl[5] = 0.5000000000;	data_table[76].feg[0].cnl[5] = 39.7700000000;
				data_table[77].feg[0].cl[0] = 0.5000000000;	data_table[77].feg[0].cnl[0] = 0.0525000000;	data_table[77].feg[0].cl[1] = 0.5000000000;	data_table[77].feg[0].cnl[1] = 0.8377000000;	data_table[77].feg[0].cl[2] = 0.5000000000;	data_table[77].feg[0].cnl[2] = 3.9590000000;	data_table[77].feg[0].cl[3] = 0.5000000000;	data_table[77].feg[0].cnl[3] = 0.8152000000;	data_table[77].feg[0].cl[4] = 0.5000000000;	data_table[77].feg[0].cnl[4] = 6.4420000000;	data_table[77].feg[0].cl[5] = 0.5000000000;	data_table[77].feg[0].cnl[5] = 34.2100000000;
				data_table[78].feg[0].cl[0] = 0.4000000000;	data_table[78].feg[0].cnl[0] = 0.5493000000;	data_table[78].feg[0].cl[1] = 0.4000000000;	data_table[78].feg[0].cnl[1] = 1.7280000000;	data_table[78].feg[0].cl[2] = 0.4000000000;	data_table[78].feg[0].cnl[2] = 6.7200000000;	data_table[78].feg[0].cl[3] = 0.4000000000;	data_table[78].feg[0].cnl[3] = 0.0264000000;	data_table[78].feg[0].cl[4] = 0.4000000000;	data_table[78].feg[0].cnl[4] = 0.0725000000;	data_table[78].feg[0].cl[5] = 0.4000000000;	data_table[78].feg[0].cnl[5] = 35.4600000000;
				data_table[79].feg[0].cl[0] = 0.4000000000;	data_table[79].feg[0].cnl[0] = 0.2194000000;	data_table[79].feg[0].cl[1] = 0.4000000000;	data_table[79].feg[0].cnl[1] = 1.4160000000;	data_table[79].feg[0].cl[2] = 0.4000000000;	data_table[79].feg[0].cnl[2] = 6.6820000000;	data_table[79].feg[0].cl[3] = 0.4000000000;	data_table[79].feg[0].cnl[3] = 0.0147000000;	data_table[79].feg[0].cl[4] = 0.4000000000;	data_table[79].feg[0].cnl[4] = 1.5760000000;	data_table[79].feg[0].cl[5] = 0.4000000000;	data_table[79].feg[0].cnl[5] = 37.1600000000;
				data_table[80].feg[0].cl[0] = 0.4000000000;	data_table[80].feg[0].cnl[0] = 0.2246000000;	data_table[80].feg[0].cl[1] = 0.4000000000;	data_table[80].feg[0].cnl[1] = 1.1280000000;	data_table[80].feg[0].cl[2] = 0.4000000000;	data_table[80].feg[0].cnl[2] = 4.3030000000;	data_table[80].feg[0].cl[3] = 0.4000000000;	data_table[80].feg[0].cnl[3] = 0.0149000000;	data_table[80].feg[0].cl[4] = 0.4000000000;	data_table[80].feg[0].cnl[4] = 7.1560000000;	data_table[80].feg[0].cl[5] = 0.4000000000;	data_table[80].feg[0].cnl[5] = 43.0900000000;
				data_table[81].feg[0].cl[0] = 0.3000000000;	data_table[81].feg[0].cnl[0] = 0.0643000000;	data_table[81].feg[0].cl[1] = 0.3000000000;	data_table[81].feg[0].cnl[1] = 1.1940000000;	data_table[81].feg[0].cl[2] = 0.3000000000;	data_table[81].feg[0].cnl[2] = 7.3930000000;	data_table[81].feg[0].cl[3] = 0.3000000000;	data_table[81].feg[0].cnl[3] = 1.1420000000;	data_table[81].feg[0].cl[4] = 0.3000000000;	data_table[81].feg[0].cnl[4] = 1.2890000000;	data_table[81].feg[0].cl[5] = 0.3000000000;	data_table[81].feg[0].cnl[5] = 51.1300000000;
				data_table[82].feg[0].cl[0] = 0.4000000000;	data_table[82].feg[0].cnl[0] = 0.0538000000;	data_table[82].feg[0].cl[1] = 0.4000000000;	data_table[82].feg[0].cnl[1] = 0.8672000000;	data_table[82].feg[0].cl[2] = 0.4000000000;	data_table[82].feg[0].cnl[2] = 1.8750000000;	data_table[82].feg[0].cl[3] = 0.4000000000;	data_table[82].feg[0].cnl[3] = 7.6480000000;	data_table[82].feg[0].cl[4] = 0.4000000000;	data_table[82].feg[0].cnl[4] = 7.8680000000;	data_table[82].feg[0].cl[5] = 0.4000000000;	data_table[82].feg[0].cnl[5] = 45.6400000000;
				data_table[83].feg[0].cl[0] = 0.4000000000;	data_table[83].feg[0].cnl[0] = 0.5011000000;	data_table[83].feg[0].cl[1] = 0.4000000000;	data_table[83].feg[0].cnl[1] = 1.6380000000;	data_table[83].feg[0].cl[2] = 0.4000000000;	data_table[83].feg[0].cnl[2] = 6.7860000000;	data_table[83].feg[0].cl[3] = 0.4000000000;	data_table[83].feg[0].cnl[3] = 0.0219000000;	data_table[83].feg[0].cl[4] = 0.4000000000;	data_table[83].feg[0].cnl[4] = 0.0860000000;	data_table[83].feg[0].cl[5] = 0.4000000000;	data_table[83].feg[0].cnl[5] = 46.7300000000;
				data_table[84].feg[0].cl[0] = 0.4000000000;	data_table[84].feg[0].cnl[0] = 0.2232000000;	data_table[84].feg[0].cl[1] = 0.4000000000;	data_table[84].feg[0].cnl[1] = 1.1080000000;	data_table[84].feg[0].cl[2] = 0.4000000000;	data_table[84].feg[0].cnl[2] = 3.5910000000;	data_table[84].feg[0].cl[3] = 0.4000000000;	data_table[84].feg[0].cnl[3] = 0.0101000000;	data_table[84].feg[0].cl[4] = 0.4000000000;	data_table[84].feg[0].cnl[4] = 11.6400000000;	data_table[84].feg[0].cl[5] = 0.4000000000;	data_table[84].feg[0].cnl[5] = 45.0700000000;
				data_table[85].feg[0].cl[0] = 0.4000000000;	data_table[85].feg[0].cnl[0] = 0.2115000000;	data_table[85].feg[0].cl[1] = 0.4000000000;	data_table[85].feg[0].cnl[1] = 1.1400000000;	data_table[85].feg[0].cl[2] = 0.4000000000;	data_table[85].feg[0].cnl[2] = 3.4150000000;	data_table[85].feg[0].cl[3] = 0.4000000000;	data_table[85].feg[0].cnl[3] = 0.0119000000;	data_table[85].feg[0].cl[4] = 0.4000000000;	data_table[85].feg[0].cnl[4] = 13.4100000000;	data_table[85].feg[0].cl[5] = 0.4000000000;	data_table[85].feg[0].cnl[5] = 43.1100000000;
				data_table[86].feg[0].cl[0] = 0.1000000000;	data_table[86].feg[0].cnl[0] = 0.0944000000;	data_table[86].feg[0].cl[1] = 0.1000000000;	data_table[86].feg[0].cnl[1] = 1.0260000000;	data_table[86].feg[0].cl[2] = 0.1000000000;	data_table[86].feg[0].cnl[2] = 6.2550000000;	data_table[86].feg[0].cl[3] = 0.1000000000;	data_table[86].feg[0].cnl[3] = 32.5100000000;	data_table[86].feg[0].cl[4] = 0.1000000000;	data_table[86].feg[0].cnl[4] = 36.2900000000;	data_table[86].feg[0].cl[5] = 0.1000000000;	data_table[86].feg[0].cnl[5] = 149.1000000000;
				data_table[87].feg[0].cl[0] = 0.2000000000;	data_table[87].feg[0].cnl[0] = 0.0730000000;	data_table[87].feg[0].cl[1] = 0.2000000000;	data_table[87].feg[0].cnl[1] = 1.0180000000;	data_table[87].feg[0].cl[2] = 0.2000000000;	data_table[87].feg[0].cnl[2] = 5.8960000000;	data_table[87].feg[0].cl[3] = 0.2000000000;	data_table[87].feg[0].cnl[3] = 1.0310000000;	data_table[87].feg[0].cl[4] = 0.2000000000;	data_table[87].feg[0].cnl[4] = 20.3700000000;	data_table[87].feg[0].cl[5] = 0.2000000000;	data_table[87].feg[0].cnl[5] = 115.3000000000;
				data_table[88].feg[0].cl[0] = 0.2000000000;	data_table[88].feg[0].cnl[0] = 0.0752000000;	data_table[88].feg[0].cl[1] = 0.2000000000;	data_table[88].feg[0].cnl[1] = 0.9494000000;	data_table[88].feg[0].cl[2] = 0.2000000000;	data_table[88].feg[0].cnl[2] = 3.7250000000;	data_table[88].feg[0].cl[3] = 0.2000000000;	data_table[88].feg[0].cnl[3] = 17.5800000000;	data_table[88].feg[0].cl[4] = 0.2000000000;	data_table[88].feg[0].cnl[4] = 19.7500000000;	data_table[88].feg[0].cl[5] = 0.2000000000;	data_table[88].feg[0].cnl[5] = 109.1000000000;
				data_table[89].feg[0].cl[0] = 0.3000000000;	data_table[89].feg[0].cnl[0] = 0.0639000000;	data_table[89].feg[0].cl[1] = 0.3000000000;	data_table[89].feg[0].cnl[1] = 0.9019000000;	data_table[89].feg[0].cl[2] = 0.3000000000;	data_table[89].feg[0].cnl[2] = 4.6570000000;	data_table[89].feg[0].cl[3] = 0.3000000000;	data_table[89].feg[0].cnl[3] = 0.9025000000;	data_table[89].feg[0].cl[4] = 0.3000000000;	data_table[89].feg[0].cnl[4] = 15.7100000000;	data_table[89].feg[0].cl[5] = 0.3000000000;	data_table[89].feg[0].cnl[5] = 83.7000000000;
				data_table[90].feg[0].cl[0] = 0.2000000000;	data_table[90].feg[0].cnl[0] = 0.0756000000;	data_table[90].feg[0].cl[1] = 0.2000000000;	data_table[90].feg[0].cnl[1] = 0.8492000000;	data_table[90].feg[0].cl[2] = 0.2000000000;	data_table[90].feg[0].cnl[2] = 4.0100000000;	data_table[90].feg[0].cl[3] = 0.2000000000;	data_table[90].feg[0].cnl[3] = 16.9500000000;	data_table[90].feg[0].cl[4] = 0.2000000000;	data_table[90].feg[0].cnl[4] = 17.7900000000;	data_table[90].feg[0].cl[5] = 0.2000000000;	data_table[90].feg[0].cnl[5] = 100.2000000000;
				data_table[91].feg[0].cl[0] = 0.2000000000;	data_table[91].feg[0].cnl[0] = 0.0714000000;	data_table[91].feg[0].cl[1] = 0.2000000000;	data_table[91].feg[0].cnl[1] = 1.1490000000;	data_table[91].feg[0].cl[2] = 0.2000000000;	data_table[91].feg[0].cnl[2] = 9.2120000000;	data_table[91].feg[0].cl[3] = 0.2000000000;	data_table[91].feg[0].cnl[3] = 0.9592000000;	data_table[91].feg[0].cl[4] = 0.2000000000;	data_table[91].feg[0].cnl[4] = 1.2030000000;	data_table[91].feg[0].cl[5] = 0.2000000000;	data_table[91].feg[0].cnl[5] = 104.3000000000;
				data_table[92].feg[0].cl[0] = 0.2000000000;	data_table[92].feg[0].cnl[0] = 0.0692000000;	data_table[92].feg[0].cl[1] = 0.2000000000;	data_table[92].feg[0].cnl[1] = 0.9810000000;	data_table[92].feg[0].cl[2] = 0.2000000000;	data_table[92].feg[0].cnl[2] = 5.9540000000;	data_table[92].feg[0].cl[3] = 0.2000000000;	data_table[92].feg[0].cnl[3] = 0.9909000000;	data_table[92].feg[0].cl[4] = 0.2000000000;	data_table[92].feg[0].cnl[4] = 22.0600000000;	data_table[92].feg[0].cl[5] = 0.2000000000;	data_table[92].feg[0].cnl[5] = 90.9800000000;
				data_table[93].feg[0].cl[0] = 0.2000000000;	data_table[93].feg[0].cnl[0] = 0.0714000000;	data_table[93].feg[0].cl[1] = 0.2000000000;	data_table[93].feg[0].cnl[1] = 0.9577000000;	data_table[93].feg[0].cl[2] = 0.2000000000;	data_table[93].feg[0].cnl[2] = 6.1320000000;	data_table[93].feg[0].cl[3] = 0.2000000000;	data_table[93].feg[0].cnl[3] = 0.9744000000;	data_table[93].feg[0].cl[4] = 0.2000000000;	data_table[93].feg[0].cnl[4] = 15.6700000000;	data_table[93].feg[0].cl[5] = 0.2000000000;	data_table[93].feg[0].cnl[5] = 89.8700000000;
				data_table[94].feg[0].cl[0] = 0.2000000000;	data_table[94].feg[0].cnl[0] = 0.0730000000;	data_table[94].feg[0].cl[1] = 0.2000000000;	data_table[94].feg[0].cnl[1] = 0.9327000000;	data_table[94].feg[0].cl[2] = 0.2000000000;	data_table[94].feg[0].cnl[2] = 6.3480000000;	data_table[94].feg[0].cl[3] = 0.2000000000;	data_table[94].feg[0].cnl[3] = 0.9103000000;	data_table[94].feg[0].cl[4] = 0.2000000000;	data_table[94].feg[0].cnl[4] = 13.2600000000;	data_table[94].feg[0].cl[5] = 0.2000000000;	data_table[94].feg[0].cnl[5] = 86.8600000000;
				data_table[95].feg[0].cl[0] = 0.3000000000;	data_table[95].feg[0].cnl[0] = 0.0578000000;	data_table[95].feg[0].cl[1] = 0.3000000000;	data_table[95].feg[0].cnl[1] = 0.7227000000;	data_table[95].feg[0].cl[2] = 0.3000000000;	data_table[95].feg[0].cnl[2] = 3.0110000000;	data_table[95].feg[0].cl[3] = 0.3000000000;	data_table[95].feg[0].cnl[3] = 9.2190000000;	data_table[95].feg[0].cl[4] = 0.3000000000;	data_table[95].feg[0].cnl[4] = 9.5340000000;	data_table[95].feg[0].cl[5] = 0.3000000000;	data_table[95].feg[0].cnl[5] = 65.8700000000;
				data_table[96].feg[0].cl[0] = 0.2000000000;	data_table[96].feg[0].cnl[0] = 0.0709000000;	data_table[96].feg[0].cl[1] = 0.2000000000;	data_table[96].feg[0].cnl[1] = 0.7759000000;	data_table[96].feg[0].cl[2] = 0.2000000000;	data_table[96].feg[0].cnl[2] = 6.1430000000;	data_table[96].feg[0].cl[3] = 0.2000000000;	data_table[96].feg[0].cnl[3] = 1.7900000000;	data_table[96].feg[0].cl[4] = 0.2000000000;	data_table[96].feg[0].cnl[4] = 15.1200000000;	data_table[96].feg[0].cl[5] = 0.2000000000;	data_table[96].feg[0].cnl[5] = 83.5700000000;
				data_table[97].feg[0].cl[0] = 0.3000000000;	data_table[97].feg[0].cnl[0] = 0.0616000000;	data_table[97].feg[0].cl[1] = 0.3000000000;	data_table[97].feg[0].cnl[1] = 0.8136000000;	data_table[97].feg[0].cl[2] = 0.3000000000;	data_table[97].feg[0].cnl[2] = 6.5620000000;	data_table[97].feg[0].cl[3] = 0.3000000000;	data_table[97].feg[0].cnl[3] = 0.8381000000;	data_table[97].feg[0].cl[4] = 0.3000000000;	data_table[97].feg[0].cnl[4] = 4.1890000000;	data_table[97].feg[0].cl[5] = 0.3000000000;	data_table[97].feg[0].cnl[5] = 61.4100000000;
				data_table[98].feg[0].cl[0] = 0.0000000000;	data_table[98].feg[0].cnl[0] = 0.0000000000;	data_table[98].feg[0].cl[1] = 0.0000000000;	data_table[98].feg[0].cnl[1] = 0.0000000000;	data_table[98].feg[0].cl[2] = 0.0000000000;	data_table[98].feg[0].cnl[2] = 0.0000000000;	data_table[98].feg[0].cl[3] = 0.0000000000;	data_table[98].feg[0].cnl[3] = 0.0000000000;	data_table[98].feg[0].cl[4] = 0.0000000000;	data_table[98].feg[0].cnl[4] = 0.0000000000;	data_table[98].feg[0].cl[5] = 0.0000000000;	data_table[98].feg[0].cnl[5] = 0.0000000000;
				data_table[99].feg[0].cl[0] = 0.0000000000;	data_table[99].feg[0].cnl[0] = 0.0000000000;	data_table[99].feg[0].cl[1] = 0.0000000000;	data_table[99].feg[0].cnl[1] = 0.0000000000;	data_table[99].feg[0].cl[2] = 0.0000000000;	data_table[99].feg[0].cnl[2] = 0.0000000000;	data_table[99].feg[0].cl[3] = 0.0000000000;	data_table[99].feg[0].cnl[3] = 0.0000000000;	data_table[99].feg[0].cl[4] = 0.0000000000;	data_table[99].feg[0].cnl[4] = 0.0000000000;	data_table[99].feg[0].cl[5] = 0.0000000000;	data_table[99].feg[0].cnl[5] = 0.0000000000;
				data_table[100].feg[0].cl[0] = 0.0000000000;	data_table[100].feg[0].cnl[0] = 0.0000000000;	data_table[100].feg[0].cl[1] = 0.0000000000;	data_table[100].feg[0].cnl[1] = 0.0000000000;	data_table[100].feg[0].cl[2] = 0.0000000000;	data_table[100].feg[0].cnl[2] = 0.0000000000;	data_table[100].feg[0].cl[3] = 0.0000000000;	data_table[100].feg[0].cnl[3] = 0.0000000000;	data_table[100].feg[0].cl[4] = 0.0000000000;	data_table[100].feg[0].cnl[4] = 0.0000000000;	data_table[100].feg[0].cl[5] = 0.0000000000;	data_table[100].feg[0].cnl[5] = 0.0000000000;
				data_table[101].feg[0].cl[0] = 0.0000000000;	data_table[101].feg[0].cnl[0] = 0.0000000000;	data_table[101].feg[0].cl[1] = 0.0000000000;	data_table[101].feg[0].cnl[1] = 0.0000000000;	data_table[101].feg[0].cl[2] = 0.0000000000;	data_table[101].feg[0].cnl[2] = 0.0000000000;	data_table[101].feg[0].cl[3] = 0.0000000000;	data_table[101].feg[0].cnl[3] = 0.0000000000;	data_table[101].feg[0].cl[4] = 0.0000000000;	data_table[101].feg[0].cnl[4] = 0.0000000000;	data_table[101].feg[0].cl[5] = 0.0000000000;	data_table[101].feg[0].cnl[5] = 0.0000000000;
				data_table[102].feg[0].cl[0] = 0.0000000000;	data_table[102].feg[0].cnl[0] = 0.0000000000;	data_table[102].feg[0].cl[1] = 0.0000000000;	data_table[102].feg[0].cnl[1] = 0.0000000000;	data_table[102].feg[0].cl[2] = 0.0000000000;	data_table[102].feg[0].cnl[2] = 0.0000000000;	data_table[102].feg[0].cl[3] = 0.0000000000;	data_table[102].feg[0].cnl[3] = 0.0000000000;	data_table[102].feg[0].cl[4] = 0.0000000000;	data_table[102].feg[0].cnl[4] = 0.0000000000;	data_table[102].feg[0].cl[5] = 0.0000000000;	data_table[102].feg[0].cnl[5] = 0.0000000000;
			}

			// 6: Lobato parameterization - Hydrogen functions - [0, 12]
			void Load_feg_Lobato_neutral_0_12()
			{
				data_table[0].feg[0].cl[0] = 5.291772107783e-02;	data_table[0].feg[0].cnl[0] = 2.763770717440e+00;	data_table[0].feg[0].cl[1] = 5.291772107783e-02;	data_table[0].feg[0].cnl[1] = 2.763770717440e+00;	data_table[0].feg[0].cl[2] = 5.291772107783e-02;	data_table[0].feg[0].cnl[2] = 2.763770717440e+00;	data_table[0].feg[0].cl[3] = 5.291772107783e-02;	data_table[0].feg[0].cnl[3] = 2.763770717440e+00;	data_table[0].feg[0].cl[4] = 5.291772107783e-02;	data_table[0].feg[0].cnl[4] = 2.763770717440e+00;	data_table[0].feg[0].cl[5] = 0.000000000000e+00;	data_table[0].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[1].feg[0].cl[0] = -6.598207946998e-04;	data_table[1].feg[0].cnl[0] = 2.332824832085e+01;	data_table[1].feg[0].cl[1] = 7.117210312090e-04;	data_table[1].feg[0].cnl[1] = 1.571138854405e+01;	data_table[1].feg[0].cl[2] = 8.652912786985e-02;	data_table[1].feg[0].cnl[2] = 1.536482140133e+00;	data_table[1].feg[0].cl[3] = 1.085231255103e-01;	data_table[1].feg[0].cnl[3] = 1.049132750140e+00;	data_table[1].feg[0].cl[4] = 1.389584638334e-02;	data_table[1].feg[0].cnl[4] = 4.384227214906e-01;	data_table[1].feg[0].cl[5] = 0.000000000000e+00;	data_table[1].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[2].feg[0].cl[0] = -1.807352999788e+00;	data_table[2].feg[0].cnl[0] = 1.196169927876e+01;	data_table[2].feg[0].cl[1] = 5.170205409177e+00;	data_table[2].feg[0].cnl[1] = 9.967399064528e+00;	data_table[2].feg[0].cl[2] = -1.805368014034e+00;	data_table[2].feg[0].cnl[2] = 6.431382442020e+00;	data_table[2].feg[0].cl[3] = 7.638651418607e-02;	data_table[2].feg[0].cnl[3] = 4.870256721308e-01;	data_table[2].feg[0].cl[4] = 9.179090459244e-03;	data_table[2].feg[0].cnl[4] = 2.112236635855e-01;	data_table[2].feg[0].cl[5] = 0.000000000000e+00;	data_table[2].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[3].feg[0].cl[0] = -1.650011428450e+00;	data_table[3].feg[0].cnl[0] = 6.318482962363e+00;	data_table[3].feg[0].cl[1] = 3.478474844054e+00;	data_table[3].feg[0].cnl[1] = 5.717983721704e+00;	data_table[3].feg[0].cl[2] = -3.497450366516e-01;	data_table[3].feg[0].cnl[2] = 1.983981533223e+00;	data_table[3].feg[0].cl[3] = 4.327223808732e-02;	data_table[3].feg[0].cnl[3] = 2.542383113232e-01;	data_table[3].feg[0].cl[4] = 4.959382960071e-03;	data_table[3].feg[0].cnl[4] = 1.185871494524e-01;	data_table[3].feg[0].cl[5] = 0.000000000000e+00;	data_table[3].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[4].feg[0].cl[0] = 5.434637970245e-01;	data_table[4].feg[0].cnl[0] = 4.758725154049e+00;	data_table[4].feg[0].cl[1] = 9.943375413498e-01;	data_table[4].feg[0].cnl[1] = 3.094110637011e+00;	data_table[4].feg[0].cl[2] = -1.720216827177e-01;	data_table[4].feg[0].cnl[2] = 9.666945857142e-01;	data_table[4].feg[0].cl[3] = 2.787953732690e-02;	data_table[4].feg[0].cnl[3] = 1.615194613399e-01;	data_table[4].feg[0].cl[4] = 3.840807016439e-03;	data_table[4].feg[0].cnl[4] = 7.928114138632e-02;	data_table[4].feg[0].cl[5] = 0.000000000000e+00;	data_table[4].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[5].feg[0].cl[0] = 6.829736893745e-01;	data_table[5].feg[0].cnl[0] = 3.491981604158e+00;	data_table[5].feg[0].cl[1] = 6.632431043459e-01;	data_table[5].feg[0].cnl[1] = 1.909779371044e+00;	data_table[5].feg[0].cl[2] = -1.143538015536e-01;	data_table[5].feg[0].cnl[2] = 5.828594428935e-01;	data_table[5].feg[0].cl[3] = 1.931977587551e-02;	data_table[5].feg[0].cnl[3] = 1.120906362608e-01;	data_table[5].feg[0].cl[4] = 3.167231957656e-03;	data_table[5].feg[0].cnl[4] = 5.719855269582e-02;	data_table[5].feg[0].cl[5] = 0.000000000000e+00;	data_table[5].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[6].feg[0].cl[0] = 5.790202573656e-01;	data_table[6].feg[0].cnl[0] = 2.648609977381e+00;	data_table[6].feg[0].cl[1] = 5.889614595690e-01;	data_table[6].feg[0].cnl[1] = 1.399462066106e+00;	data_table[6].feg[0].cl[2] = -7.683160617318e-02;	data_table[6].feg[0].cnl[2] = 3.732092216804e-01;	data_table[6].feg[0].cl[3] = 1.429711777913e-02;	data_table[6].feg[0].cnl[3] = 8.530120762025e-02;	data_table[6].feg[0].cl[4] = 3.052771459369e-03;	data_table[6].feg[0].cnl[4] = 4.428086466707e-02;	data_table[6].feg[0].cl[5] = 0.000000000000e+00;	data_table[6].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[7].feg[0].cl[0] = 5.261125701136e-01;	data_table[7].feg[0].cnl[0] = 2.079265351330e+00;	data_table[7].feg[0].cl[1] = 5.109028911223e-01;	data_table[7].feg[0].cnl[1] = 1.048129520181e+00;	data_table[7].feg[0].cl[2] = -5.690179380942e-02;	data_table[7].feg[0].cnl[2] = 2.577224704813e-01;	data_table[7].feg[0].cl[3] = 1.109300661291e-02;	data_table[7].feg[0].cnl[3] = 6.852979260850e-02;	data_table[7].feg[0].cl[4] = 2.993325960599e-03;	data_table[7].feg[0].cnl[4] = 3.549862030724e-02;	data_table[7].feg[0].cl[5] = 0.000000000000e+00;	data_table[7].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[8].feg[0].cl[0] = 4.788645038502e-01;	data_table[8].feg[0].cnl[0] = 1.689889754866e+00;	data_table[8].feg[0].cl[1] = 4.552695091260e-01;	data_table[8].feg[0].cnl[1] = 8.163825921430e-01;	data_table[8].feg[0].cl[2] = -4.417062832513e-02;	data_table[8].feg[0].cnl[2] = 1.857630136333e-01;	data_table[8].feg[0].cl[3] = 9.091563254928e-03;	data_table[8].feg[0].cnl[3] = 5.773344563066e-02;	data_table[8].feg[0].cl[4] = 2.945052093955e-03;	data_table[8].feg[0].cnl[4] = 2.919381111144e-02;	data_table[8].feg[0].cl[5] = 0.000000000000e+00;	data_table[8].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[9].feg[0].cl[0] = 4.360411450621e-01;	data_table[9].feg[0].cnl[0] = 1.409227350817e+00;	data_table[9].feg[0].cl[1] = 4.138599450963e-01;	data_table[9].feg[0].cnl[1] = 6.580470753006e-01;	data_table[9].feg[0].cl[2] = -3.647792823568e-02;	data_table[9].feg[0].cnl[2] = 1.318704770671e-01;	data_table[9].feg[0].cl[3] = 9.385476084002e-03;	data_table[9].feg[0].cnl[3] = 5.548472514575e-02;	data_table[9].feg[0].cl[4] = 3.141361993254e-03;	data_table[9].feg[0].cnl[4] = 2.483849804973e-02;	data_table[9].feg[0].cl[5] = 0.000000000000e+00;	data_table[9].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[10].feg[0].cl[0] = 4.461007444237e+00;	data_table[10].feg[0].cnl[0] = 9.707259027226e+00;	data_table[10].feg[0].cl[1] = -2.619311289092e+00;	data_table[10].feg[0].cnl[1] = 7.197666158078e+00;	data_table[10].feg[0].cl[2] = 5.958033705019e-01;	data_table[10].feg[0].cnl[2] = 5.674273314527e-01;	data_table[10].feg[0].cl[3] = -5.609872877390e-02;	data_table[10].feg[0].cnl[3] = 1.935729235170e-01;	data_table[10].feg[0].cl[4] = 4.549203126519e-03;	data_table[10].feg[0].cnl[4] = 2.306465741282e-02;	data_table[10].feg[0].cl[5] = 0.000000000000e+00;	data_table[10].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[11].feg[0].cl[0] = 5.056965905799e+00;	data_table[11].feg[0].cnl[0] = 5.897923680253e+00;	data_table[11].feg[0].cl[1] = -2.860536132251e+00;	data_table[11].feg[0].cnl[1] = 4.241253213256e+00;	data_table[11].feg[0].cl[2] = 5.146688476179e-01;	data_table[11].feg[0].cnl[2] = 3.744706145656e-01;	data_table[11].feg[0].cl[3] = -1.142624494895e-01;	data_table[11].feg[0].cnl[3] = 1.873815955152e-01;	data_table[11].feg[0].cl[4] = 3.913828324258e-03;	data_table[11].feg[0].cnl[4] = 1.944751249106e-02;	data_table[11].feg[0].cl[5] = 0.000000000000e+00;	data_table[11].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[12].feg[0].cl[0] = 2.904786099285e+00;	data_table[12].feg[0].cnl[0] = 6.508577332507e+00;	data_table[12].feg[0].cl[1] = -3.032205903196e-01;	data_table[12].feg[0].cnl[1] = 1.333702951222e+00;	data_table[12].feg[0].cl[2] = 5.341791098085e-01;	data_table[12].feg[0].cnl[2] = 2.712983831862e-01;	data_table[12].feg[0].cl[3] = -1.961212742394e-01;	data_table[12].feg[0].cnl[3] = 1.706568042018e-01;	data_table[12].feg[0].cl[4] = 3.426655465978e-03;	data_table[12].feg[0].cnl[4] = 1.664690826183e-02;	data_table[12].feg[0].cl[5] = 0.000000000000e+00;	data_table[12].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[13].feg[0].cl[0] = 2.921358442389e+00;	data_table[13].feg[0].cnl[0] = 4.945980711274e+00;	data_table[13].feg[0].cl[1] = -3.214280237771e-01;	data_table[13].feg[0].cnl[1] = 8.870990766578e-01;	data_table[13].feg[0].cl[2] = 8.474022627799e-01;	data_table[13].feg[0].cnl[2] = 1.992810441512e-01;	data_table[13].feg[0].cl[3] = -5.449947429152e-01;	data_table[13].feg[0].cnl[3] = 1.627076453793e-01;	data_table[13].feg[0].cl[4] = 3.012061523181e-03;	data_table[13].feg[0].cnl[4] = 1.439769516961e-02;	data_table[13].feg[0].cl[5] = 0.000000000000e+00;	data_table[13].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[14].feg[0].cl[0] = 2.837426146223e+00;	data_table[14].feg[0].cnl[0] = 3.707392522481e+00;	data_table[14].feg[0].cl[1] = -3.796278147439e-01;	data_table[14].feg[0].cnl[1] = 6.767395866138e-01;	data_table[14].feg[0].cl[2] = 6.553422797629e-01;	data_table[14].feg[0].cnl[2] = 1.729022763348e-01;	data_table[14].feg[0].cl[3] = -3.685319057733e-01;	data_table[14].feg[0].cnl[3] = 1.329388422237e-01;	data_table[14].feg[0].cl[4] = 2.691294531031e-03;	data_table[14].feg[0].cnl[4] = 1.260043764273e-02;	data_table[14].feg[0].cl[5] = 0.000000000000e+00;	data_table[14].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[15].feg[0].cl[0] = 2.707818764508e+00;	data_table[15].feg[0].cnl[0] = 2.906651593454e+00;	data_table[15].feg[0].cl[1] = -4.166315461511e-01;	data_table[15].feg[0].cnl[1] = 5.058076208103e-01;	data_table[15].feg[0].cl[2] = 5.146184680373e-01;	data_table[15].feg[0].cnl[2] = 1.574007035130e-01;	data_table[15].feg[0].cl[3] = -2.234290941523e-01;	data_table[15].feg[0].cnl[3] = 1.082783542400e-01;	data_table[15].feg[0].cl[4] = 2.423407758202e-03;	data_table[15].feg[0].cnl[4] = 1.112442674484e-02;	data_table[15].feg[0].cl[5] = 0.000000000000e+00;	data_table[15].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[16].feg[0].cl[0] = 2.566021353326e+00;	data_table[16].feg[0].cnl[0] = 2.353232106435e+00;	data_table[16].feg[0].cl[1] = -4.785471011730e-01;	data_table[16].feg[0].cnl[1] = 3.695843551991e-01;	data_table[16].feg[0].cl[2] = 4.653976190005e-01;	data_table[16].feg[0].cnl[2] = 1.534423865540e-01;	data_table[16].feg[0].cl[3] = -1.235710412480e-01;	data_table[16].feg[0].cnl[3] = 8.681952609634e-02;	data_table[16].feg[0].cl[4] = 2.199170094442e-03;	data_table[16].feg[0].cnl[4] = 9.899285961702e-03;	data_table[16].feg[0].cl[5] = 0.000000000000e+00;	data_table[16].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[17].feg[0].cl[0] = 2.425980796349e+00;	data_table[17].feg[0].cnl[0] = 1.952990057609e+00;	data_table[17].feg[0].cl[1] = -7.026177947527e-01;	data_table[17].feg[0].cnl[1] = 2.603030874337e-01;	data_table[17].feg[0].cl[2] = 6.444439386431e-01;	data_table[17].feg[0].cnl[2] = 1.584358024881e-01;	data_table[17].feg[0].cl[3] = -7.891602139406e-02;	data_table[17].feg[0].cnl[3] = 7.085503793437e-02;	data_table[17].feg[0].cl[4] = 2.009081154588e-03;	data_table[17].feg[0].cnl[4] = 8.870405143324e-03;	data_table[17].feg[0].cl[5] = 0.000000000000e+00;	data_table[17].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[18].feg[0].cl[0] = 2.485912993737e+00;	data_table[18].feg[0].cnl[0] = 2.450976850735e+01;	data_table[18].feg[0].cl[1] = 1.914786560063e+00;	data_table[18].feg[0].cnl[1] = 2.105731275399e+00;	data_table[18].feg[0].cl[2] = 4.523257468324e+00;	data_table[18].feg[0].cnl[2] = 5.545543558684e-02;	data_table[18].feg[0].cl[3] = -4.454146252312e+00;	data_table[18].feg[0].cnl[3] = 5.497591772299e-02;	data_table[18].feg[0].cl[4] = 2.189230188528e-03;	data_table[18].feg[0].cnl[4] = 8.338800175046e-03;	data_table[18].feg[0].cl[5] = 0.000000000000e+00;	data_table[18].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[19].feg[0].cl[0] = 3.438132731610e+00;	data_table[19].feg[0].cnl[0] = 1.458451252935e+01;	data_table[19].feg[0].cl[1] = 1.453864257573e+00;	data_table[19].feg[0].cnl[1] = 1.658111628125e+00;	data_table[19].feg[0].cl[2] = 2.231576982870e+00;	data_table[19].feg[0].cnl[2] = 4.764165928423e-02;	data_table[19].feg[0].cl[3] = -2.172417098642e+00;	data_table[19].feg[0].cnl[3] = 4.690683153977e-02;	data_table[19].feg[0].cl[4] = 2.093126590441e-03;	data_table[19].feg[0].cnl[4] = 7.619449067265e-03;	data_table[19].feg[0].cl[5] = 0.000000000000e+00;	data_table[19].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[20].feg[0].cl[0] = 3.171483304275e+00;	data_table[20].feg[0].cnl[0] = 1.248855957617e+01;	data_table[20].feg[0].cl[1] = 1.416944704369e+00;	data_table[20].feg[0].cnl[1] = 1.460547000906e+00;	data_table[20].feg[0].cl[2] = 1.019963297752e+00;	data_table[20].feg[0].cnl[2] = 4.046780130696e-02;	data_table[20].feg[0].cl[3] = -9.705370495354e-01;	data_table[20].feg[0].cnl[3] = 3.926957138514e-02;	data_table[20].feg[0].cl[4] = 2.095743136919e-03;	data_table[20].feg[0].cnl[4] = 7.062374073734e-03;	data_table[20].feg[0].cl[5] = 0.000000000000e+00;	data_table[20].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[21].feg[0].cl[0] = 2.912700023579e+00;	data_table[21].feg[0].cnl[0] = 1.122040500589e+01;	data_table[21].feg[0].cl[1] = 1.411312593605e+00;	data_table[21].feg[0].cnl[1] = 1.315929465210e+00;	data_table[21].feg[0].cl[2] = 4.435853205108e+00;	data_table[21].feg[0].cnl[2] = 3.411607351590e-02;	data_table[21].feg[0].cl[3] = -4.393975859431e+00;	data_table[21].feg[0].cnl[3] = 3.391222953370e-02;	data_table[21].feg[0].cl[4] = 2.110037131771e-03;	data_table[21].feg[0].cnl[4] = 6.578039176085e-03;	data_table[21].feg[0].cl[5] = 0.000000000000e+00;	data_table[21].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[22].feg[0].cl[0] = 2.725200616254e+00;	data_table[22].feg[0].cnl[0] = 9.922562133847e+00;	data_table[22].feg[0].cl[1] = 1.360497319257e+00;	data_table[22].feg[0].cnl[1] = 1.165272804360e+00;	data_table[22].feg[0].cl[2] = 4.318014125581e-01;	data_table[22].feg[0].cnl[2] = 3.003251626117e-02;	data_table[22].feg[0].cl[3] = -3.962680689313e-01;	data_table[22].feg[0].cnl[3] = 2.836667234559e-02;	data_table[22].feg[0].cl[4] = 2.168720874813e-03;	data_table[22].feg[0].cnl[4] = 6.171445378065e-03;	data_table[22].feg[0].cl[5] = 0.000000000000e+00;	data_table[22].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[23].feg[0].cl[0] = 1.890134056605e+00;	data_table[23].feg[0].cnl[0] = 1.038543467671e+01;	data_table[23].feg[0].cl[1] = 1.558721367614e+00;	data_table[23].feg[0].cnl[1] = 1.140072547401e+00;	data_table[23].feg[0].cl[2] = 1.325725835245e+00;	data_table[23].feg[0].cnl[2] = 2.533206852666e-02;	data_table[23].feg[0].cl[3] = -1.295383889297e+00;	data_table[23].feg[0].cnl[3] = 2.492114616709e-02;	data_table[23].feg[0].cl[4] = 2.302629798472e-03;	data_table[23].feg[0].cnl[4] = 5.846747202574e-03;	data_table[23].feg[0].cl[5] = 0.000000000000e+00;	data_table[23].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[24].feg[0].cl[0] = 2.389454308099e+00;	data_table[24].feg[0].cnl[0] = 8.679554632026e+00;	data_table[24].feg[0].cl[1] = 1.318293216199e+00;	data_table[24].feg[0].cnl[1] = 9.506916940910e-01;	data_table[24].feg[0].cl[2] = 3.364468602187e-01;	data_table[24].feg[0].cnl[2] = 2.225901004211e-02;	data_table[24].feg[0].cl[3] = -3.109220810130e-01;	data_table[24].feg[0].cnl[3] = 2.095729439190e-02;	data_table[24].feg[0].cl[4] = 2.527696486204e-03;	data_table[24].feg[0].cnl[4] = 5.588549400473e-03;	data_table[24].feg[0].cl[5] = 0.000000000000e+00;	data_table[24].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[25].feg[0].cl[0] = 2.281632151450e+00;	data_table[25].feg[0].cnl[0] = 8.069378430777e+00;	data_table[25].feg[0].cl[1] = 1.274104495231e+00;	data_table[25].feg[0].cnl[1] = 8.545129353173e-01;	data_table[25].feg[0].cl[2] = 1.651504241428e-01;	data_table[25].feg[0].cnl[2] = 1.962009336643e-02;	data_table[25].feg[0].cl[3] = -1.439119035401e-01;	data_table[25].feg[0].cnl[3] = 1.745913585876e-02;	data_table[25].feg[0].cl[4] = 2.924832674094e-03;	data_table[25].feg[0].cnl[4] = 5.409434596089e-03;	data_table[25].feg[0].cl[5] = 0.000000000000e+00;	data_table[25].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[26].feg[0].cl[0] = 2.146415979962e+00;	data_table[26].feg[0].cnl[0] = 7.625958741392e+00;	data_table[26].feg[0].cl[1] = 1.257062759559e+00;	data_table[26].feg[0].cnl[1] = 7.874141626337e-01;	data_table[26].feg[0].cl[2] = 1.526263997700e-01;	data_table[26].feg[0].cnl[2] = 1.701950713295e-02;	data_table[26].feg[0].cl[3] = -1.348394567868e-01;	data_table[26].feg[0].cnl[3] = 1.508741980835e-02;	data_table[26].feg[0].cl[4] = 3.584317503198e-03;	data_table[26].feg[0].cnl[4] = 5.299051717315e-03;	data_table[26].feg[0].cl[5] = 0.000000000000e+00;	data_table[26].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[27].feg[0].cl[0] = 2.034284901837e+00;	data_table[27].feg[0].cnl[0] = 7.184258945837e+00;	data_table[27].feg[0].cl[1] = 1.224448397173e+00;	data_table[27].feg[0].cnl[1] = 7.169997923193e-01;	data_table[27].feg[0].cl[2] = 1.856549262548e-01;	data_table[27].feg[0].cnl[2] = 1.346454955189e-02;	data_table[27].feg[0].cl[3] = -1.756998563211e-01;	data_table[27].feg[0].cnl[3] = 1.209738462579e-02;	data_table[27].feg[0].cl[4] = 8.261631316624e-03;	data_table[27].feg[0].cnl[4] = 5.797487482153e-03;	data_table[27].feg[0].cl[5] = 0.000000000000e+00;	data_table[27].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[28].feg[0].cl[0] = 1.468530847248e+00;	data_table[28].feg[0].cnl[0] = 7.133832365511e+00;	data_table[28].feg[0].cl[1] = 1.303592647163e+00;	data_table[28].feg[0].cnl[1] = 6.885880148197e-01;	data_table[28].feg[0].cl[2] = 6.363376449150e-02;	data_table[28].feg[0].cnl[2] = 1.340220935295e-02;	data_table[28].feg[0].cl[3] = -5.626371565390e-02;	data_table[28].feg[0].cnl[3] = 9.901498478554e-03;	data_table[28].feg[0].cl[4] = 8.856456911193e-03;	data_table[28].feg[0].cnl[4] = 5.495274889916e-03;	data_table[28].feg[0].cl[5] = 0.000000000000e+00;	data_table[28].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[29].feg[0].cl[0] = 2.141061171975e+00;	data_table[29].feg[0].cnl[0] = 5.280402318068e+00;	data_table[29].feg[0].cl[1] = 2.502164361704e+00;	data_table[29].feg[0].cnl[1] = 2.564948614672e-01;	data_table[29].feg[0].cl[2] = -3.319211861394e+00;	data_table[29].feg[0].cnl[2] = 1.503836612572e-01;	data_table[29].feg[0].cl[3] = 1.708852225253e+00;	data_table[29].feg[0].cnl[3] = 1.166905416861e-01;	data_table[29].feg[0].cl[4] = 3.841024618203e-04;	data_table[29].feg[0].cnl[4] = 2.770069931941e-03;	data_table[29].feg[0].cl[5] = 0.000000000000e+00;	data_table[29].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[30].feg[0].cl[0] = 2.735134293380e+00;	data_table[30].feg[0].cnl[0] = 5.920088010090e+00;	data_table[30].feg[0].cl[1] = 2.455670364197e+00;	data_table[30].feg[0].cnl[1] = 2.240734508902e-01;	data_table[30].feg[0].cl[2] = -3.871722837838e+00;	data_table[30].feg[0].cnl[2] = 1.334711195521e-01;	data_table[30].feg[0].cl[3] = 2.230718492671e+00;	data_table[30].feg[0].cnl[3] = 1.092475681770e-01;	data_table[30].feg[0].cl[4] = 3.496875909339e-04;	data_table[30].feg[0].cnl[4] = 2.579606101869e-03;	data_table[30].feg[0].cl[5] = 0.000000000000e+00;	data_table[30].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[31].feg[0].cl[0] = 2.998375774095e+00;	data_table[31].feg[0].cnl[0] = 5.096437839056e+00;	data_table[31].feg[0].cl[1] = 3.205420885137e+00;	data_table[31].feg[0].cnl[1] = 1.763710635250e-01;	data_table[31].feg[0].cl[2] = -4.058119990716e+00;	data_table[31].feg[0].cnl[2] = 1.263537079937e-01;	data_table[31].feg[0].cl[3] = 1.543218140035e+00;	data_table[31].feg[0].cnl[3] = 9.473256565380e-02;	data_table[31].feg[0].cl[4] = 3.051914119292e-04;	data_table[31].feg[0].cnl[4] = 2.388763189692e-03;	data_table[31].feg[0].cl[5] = 0.000000000000e+00;	data_table[31].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[32].feg[0].cl[0] = 3.080056421378e+00;	data_table[32].feg[0].cnl[0] = 4.256370135669e+00;	data_table[32].feg[0].cl[1] = 3.134598461469e+00;	data_table[32].feg[0].cnl[1] = 1.477140523971e-01;	data_table[32].feg[0].cl[2] = -4.033699983384e+00;	data_table[32].feg[0].cnl[2] = 1.095072556715e-01;	data_table[32].feg[0].cl[3] = 1.485788494491e+00;	data_table[32].feg[0].cnl[3] = 8.285091395989e-02;	data_table[32].feg[0].cl[4] = 2.566060266361e-04;	data_table[32].feg[0].cnl[4] = 2.200491757677e-03;	data_table[32].feg[0].cl[5] = 0.000000000000e+00;	data_table[32].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[33].feg[0].cl[0] = 3.102847302382e+00;	data_table[33].feg[0].cnl[0] = 3.598564726720e+00;	data_table[33].feg[0].cl[1] = 2.731174524981e+00;	data_table[33].feg[0].cnl[1] = 1.304319816954e-01;	data_table[33].feg[0].cl[2] = -3.969954988844e+00;	data_table[33].feg[0].cnl[2] = 9.562394636736e-02;	data_table[33].feg[0].cl[3] = 1.744746317329e+00;	data_table[33].feg[0].cnl[3] = 7.634068756810e-02;	data_table[33].feg[0].cl[4] = 2.368441575458e-04;	data_table[33].feg[0].cnl[4] = 2.061887907813e-03;	data_table[33].feg[0].cl[5] = 0.000000000000e+00;	data_table[33].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[34].feg[0].cl[0] = 3.093497258129e+00;	data_table[34].feg[0].cnl[0] = 3.099619218801e+00;	data_table[34].feg[0].cl[1] = 2.774044983635e+00;	data_table[34].feg[0].cnl[1] = 1.134061824866e-01;	data_table[34].feg[0].cl[2] = -3.886354979745e+00;	data_table[34].feg[0].cnl[2] = 8.663832349327e-02;	data_table[34].feg[0].cl[3] = 1.551650463756e+00;	data_table[34].feg[0].cnl[3] = 6.854036643005e-02;	data_table[34].feg[0].cl[4] = 2.122742305052e-04;	data_table[34].feg[0].cnl[4] = 1.924868113136e-03;	data_table[34].feg[0].cl[5] = 0.000000000000e+00;	data_table[34].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[35].feg[0].cl[0] = 3.062654111432e+00;	data_table[35].feg[0].cnl[0] = 2.714305136714e+00;	data_table[35].feg[0].cl[1] = 2.515541674380e+00;	data_table[35].feg[0].cnl[1] = 1.010486891990e-01;	data_table[35].feg[0].cl[2] = -3.794064947527e+00;	data_table[35].feg[0].cnl[2] = 7.709842535612e-02;	data_table[35].feg[0].cl[3] = 1.664827543148e+00;	data_table[35].feg[0].cnl[3] = 6.275726417422e-02;	data_table[35].feg[0].cl[4] = 1.916185682213e-04;	data_table[35].feg[0].cnl[4] = 1.801289587459e-03;	data_table[35].feg[0].cl[5] = 0.000000000000e+00;	data_table[35].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[36].feg[0].cl[0] = 2.949983117627e+00;	data_table[36].feg[0].cnl[0] = 2.364523241062e+01;	data_table[36].feg[0].cl[1] = 2.368651437463e+00;	data_table[36].feg[0].cnl[1] = 2.701688212118e+00;	data_table[36].feg[0].cl[2] = 5.335572927259e-01;	data_table[36].feg[0].cnl[2] = 2.793313835774e-01;	data_table[36].feg[0].cl[3] = 9.176712247131e-03;	data_table[36].feg[0].cnl[3] = 1.677304569876e-02;	data_table[36].feg[0].cl[4] = 1.314399374960e-04;	data_table[36].feg[0].cnl[4] = 1.574925169829e-03;	data_table[36].feg[0].cl[5] = 0.000000000000e+00;	data_table[36].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[37].feg[0].cl[0] = 4.241235455816e+00;	data_table[37].feg[0].cnl[0] = 1.463137021307e+01;	data_table[37].feg[0].cl[1] = 1.763722032677e+00;	data_table[37].feg[0].cnl[1] = 2.305659127808e+00;	data_table[37].feg[0].cl[2] = 5.114496412019e-01;	data_table[37].feg[0].cnl[2] = 2.599925904330e-01;	data_table[37].feg[0].cl[3] = 7.991466200910e-03;	data_table[37].feg[0].cnl[3] = 1.466884011485e-02;	data_table[37].feg[0].cl[4] = 1.014041039043e-04;	data_table[37].feg[0].cnl[4] = 1.425652862963e-03;	data_table[37].feg[0].cl[5] = 0.000000000000e+00;	data_table[37].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[38].feg[0].cl[0] = 4.079636669365e+00;	data_table[38].feg[0].cnl[0] = 1.218118885705e+01;	data_table[38].feg[0].cl[1] = 1.777184479237e+00;	data_table[38].feg[0].cnl[1] = 2.102565053886e+00;	data_table[38].feg[0].cl[2] = 4.543656690947e-01;	data_table[38].feg[0].cnl[2] = 2.319005324732e-01;	data_table[38].feg[0].cl[3] = 6.743834665502e-03;	data_table[38].feg[0].cnl[3] = 1.251526301879e-02;	data_table[38].feg[0].cl[4] = 6.934763743650e-05;	data_table[38].feg[0].cnl[4] = 1.253327825451e-03;	data_table[38].feg[0].cl[5] = 0.000000000000e+00;	data_table[38].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[39].feg[0].cl[0] = 3.789155411093e+00;	data_table[39].feg[0].cnl[0] = 1.062914522329e+01;	data_table[39].feg[0].cl[1] = 1.881078927948e+00;	data_table[39].feg[0].cnl[1] = 2.002795731783e+00;	data_table[39].feg[0].cl[2] = 4.066830560072e-01;	data_table[39].feg[0].cnl[2] = 2.068618651311e-01;	data_table[39].feg[0].cl[3] = 5.548798977761e-03;	data_table[39].feg[0].cnl[3] = 1.038524420132e-02;	data_table[39].feg[0].cl[4] = 3.380597303298e-05;	data_table[39].feg[0].cnl[4] = 1.011819041116e-03;	data_table[39].feg[0].cl[5] = 0.000000000000e+00;	data_table[39].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[40].feg[0].cl[0] = 2.382790675111e+00;	data_table[40].feg[0].cnl[0] = 1.111594850662e+01;	data_table[40].feg[0].cl[1] = 2.608977555943e+00;	data_table[40].feg[0].cnl[1] = 2.161507307192e+00;	data_table[40].feg[0].cl[2] = 3.591552704924e-01;	data_table[40].feg[0].cnl[2] = 1.829839341140e-01;	data_table[40].feg[0].cl[3] = 4.569470093002e-03;	data_table[40].feg[0].cnl[3] = 8.620279012938e-03;	data_table[40].feg[0].cl[4] = 7.028362363305e-06;	data_table[40].feg[0].cnl[4] = 6.452970292696e-04;	data_table[40].feg[0].cl[5] = 0.000000000000e+00;	data_table[40].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[41].feg[0].cl[0] = 2.099137834016e+00;	data_table[41].feg[0].cnl[0] = 1.048296346698e+01;	data_table[41].feg[0].cl[1] = 2.711084186853e+00;	data_table[41].feg[0].cnl[1] = 2.031770881549e+00;	data_table[41].feg[0].cl[2] = 3.282749545510e-01;	data_table[41].feg[0].cnl[2] = 1.689467957968e-01;	data_table[41].feg[0].cl[3] = 4.481651577360e-03;	data_table[41].feg[0].cnl[3] = 8.659102495325e-03;	data_table[41].feg[0].cl[4] = 2.137300256622e-05;	data_table[41].feg[0].cnl[4] = 8.337627704780e-04;	data_table[41].feg[0].cl[5] = 0.000000000000e+00;	data_table[41].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[42].feg[0].cl[0] = 2.798815288684e+00;	data_table[42].feg[0].cnl[0] = 8.903325832353e+00;	data_table[42].feg[0].cl[1] = 2.315495406054e+00;	data_table[42].feg[0].cnl[1] = 1.711082703938e+00;	data_table[42].feg[0].cl[2] = 2.936521854080e-01;	data_table[42].feg[0].cnl[2] = 1.531304123154e-01;	data_table[42].feg[0].cl[3] = 4.021506040290e-03;	data_table[42].feg[0].cnl[3] = 7.881537158588e-03;	data_table[42].feg[0].cl[4] = 1.561381502828e-05;	data_table[42].feg[0].cnl[4] = 7.401731426112e-04;	data_table[42].feg[0].cl[5] = 0.000000000000e+00;	data_table[42].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[43].feg[0].cl[0] = 1.656901947905e+00;	data_table[43].feg[0].cnl[0] = 1.006749257872e+01;	data_table[43].feg[0].cl[1] = 2.852635279267e+00;	data_table[43].feg[0].cnl[1] = 1.753516952634e+00;	data_table[43].feg[0].cl[2] = 2.630882319889e-01;	data_table[43].feg[0].cnl[2] = 1.375439434014e-01;	data_table[43].feg[0].cl[3] = 3.370416530862e-03;	data_table[43].feg[0].cnl[3] = 6.742520105449e-03;	data_table[43].feg[0].cl[4] = 4.124308787469e-06;	data_table[43].feg[0].cnl[4] = 4.966391629978e-04;	data_table[43].feg[0].cl[5] = 0.000000000000e+00;	data_table[43].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[44].feg[0].cl[0] = 1.512798019316e+00;	data_table[44].feg[0].cnl[0] = 9.861891560811e+00;	data_table[44].feg[0].cl[1] = 2.866258747437e+00;	data_table[44].feg[0].cnl[1] = 1.620481252322e+00;	data_table[44].feg[0].cl[2] = 2.360230128815e-01;	data_table[44].feg[0].cnl[2] = 1.251051707293e-01;	data_table[44].feg[0].cl[3] = 3.070220349223e-03;	data_table[44].feg[0].cnl[3] = 6.149147255418e-03;	data_table[44].feg[0].cl[4] = 1.681261436276e-11;	data_table[44].feg[0].cnl[4] = 9.894226825917e-06;	data_table[44].feg[0].cl[5] = 0.000000000000e+00;	data_table[44].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[45].feg[0].cl[0] = 3.600390925658e+00;	data_table[45].feg[0].cnl[0] = 1.777766900138e+00;	data_table[45].feg[0].cl[1] = 8.296669152435e-01;	data_table[45].feg[0].cnl[1] = 4.950259367891e-02;	data_table[45].feg[0].cl[2] = -1.327398314213e+00;	data_table[45].feg[0].cnl[2] = 3.411306721001e-02;	data_table[45].feg[0].cl[3] = 6.888815505153e-01;	data_table[45].feg[0].cnl[3] = 2.815437131522e-02;	data_table[45].feg[0].cl[4] = 5.892279905207e-05;	data_table[45].feg[0].cnl[4] = 9.472589692928e-04;	data_table[45].feg[0].cl[5] = 0.000000000000e+00;	data_table[45].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[46].feg[0].cl[0] = 1.291211451471e+00;	data_table[46].feg[0].cnl[0] = 9.842479063423e+00;	data_table[46].feg[0].cl[1] = 2.845585820493e+00;	data_table[46].feg[0].cnl[1] = 1.392796371595e+00;	data_table[46].feg[0].cl[2] = 1.958632497752e-01;	data_table[46].feg[0].cnl[2] = 1.076507878835e-01;	data_table[46].feg[0].cl[3] = 2.939478261842e-03;	data_table[46].feg[0].cnl[3] = 5.811139417875e-03;	data_table[46].feg[0].cl[4] = 1.258664135485e-13;	data_table[46].feg[0].cnl[4] = 1.865576630908e-06;	data_table[46].feg[0].cl[5] = 0.000000000000e+00;	data_table[46].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[47].feg[0].cl[0] = 1.923829541399e+00;	data_table[47].feg[0].cnl[0] = 7.658787390823e+00;	data_table[47].feg[0].cl[1] = 2.504683119626e+00;	data_table[47].feg[0].cnl[1] = 1.237028936471e+00;	data_table[47].feg[0].cl[2] = 1.848673451713e-01;	data_table[47].feg[0].cnl[2] = 1.023468840505e-01;	data_table[47].feg[0].cl[3] = 2.919993742015e-03;	data_table[47].feg[0].cnl[3] = 5.691700041708e-03;	data_table[47].feg[0].cl[4] = 6.602194436147e-11;	data_table[47].feg[0].cnl[4] = 1.429796481985e-05;	data_table[47].feg[0].cl[5] = 0.000000000000e+00;	data_table[47].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[48].feg[0].cl[0] = 2.767653130082e+00;	data_table[48].feg[0].cnl[0] = 8.127770566981e+00;	data_table[48].feg[0].cl[1] = 2.266504950572e+00;	data_table[48].feg[0].cnl[1] = 1.112551099242e+00;	data_table[48].feg[0].cl[2] = 1.749609922749e-01;	data_table[48].feg[0].cnl[2] = 9.754388085710e-02;	data_table[48].feg[0].cl[3] = 2.880927069060e-03;	data_table[48].feg[0].cnl[3] = 5.544501860424e-03;	data_table[48].feg[0].cl[4] = 2.245176184313e-14;	data_table[48].feg[0].cnl[4] = 1.000000000000e-06;	data_table[48].feg[0].cl[5] = 0.000000000000e+00;	data_table[48].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[49].feg[0].cl[0] = 3.255461945971e+00;	data_table[49].feg[0].cnl[0] = 6.949165861385e+00;	data_table[49].feg[0].cl[1] = 2.006854690705e+00;	data_table[49].feg[0].cnl[1] = 9.956523106562e-01;	data_table[49].feg[0].cl[2] = 1.639332619692e-01;	data_table[49].feg[0].cnl[2] = 9.190385905760e-02;	data_table[49].feg[0].cl[3] = 2.750101189335e-03;	data_table[49].feg[0].cnl[3] = 5.299968837729e-03;	data_table[49].feg[0].cl[4] = 1.588084660702e-10;	data_table[49].feg[0].cnl[4] = 1.800187289546e-05;	data_table[49].feg[0].cl[5] = 0.000000000000e+00;	data_table[49].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[50].feg[0].cl[0] = 3.616597940931e+00;	data_table[50].feg[0].cnl[0] = 5.732236167846e+00;	data_table[50].feg[0].cl[1] = 1.719412666509e+00;	data_table[50].feg[0].cnl[1] = 8.799645330194e-01;	data_table[50].feg[0].cl[2] = 1.562477411339e-01;	data_table[50].feg[0].cnl[2] = 8.829755113788e-02;	data_table[50].feg[0].cl[3] = 2.741651415321e-03;	data_table[50].feg[0].cnl[3] = 5.192244716633e-03;	data_table[50].feg[0].cl[4] = 2.643849878895e-14;	data_table[50].feg[0].cnl[4] = 1.000056740754e-06;	data_table[50].feg[0].cl[5] = 0.000000000000e+00;	data_table[50].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[51].feg[0].cl[0] = 3.948500628807e+00;	data_table[51].feg[0].cnl[0] = 4.672763082298e+00;	data_table[51].feg[0].cl[1] = 1.400040089177e+00;	data_table[51].feg[0].cnl[1] = 7.513923148953e-01;	data_table[51].feg[0].cl[2] = 1.439115437707e-01;	data_table[51].feg[0].cnl[2] = 8.231324144488e-02;	data_table[51].feg[0].cl[3] = 2.547738238993e-03;	data_table[51].feg[0].cnl[3] = 4.884507367831e-03;	data_table[51].feg[0].cl[4] = 6.561935737562e-12;	data_table[51].feg[0].cnl[4] = 5.998278122244e-06;	data_table[51].feg[0].cl[5] = 0.000000000000e+00;	data_table[51].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[52].feg[0].cl[0] = 4.136569496020e+00;	data_table[52].feg[0].cnl[0] = 4.013637853826e+00;	data_table[52].feg[0].cl[1] = 1.185132281895e+00;	data_table[52].feg[0].cnl[1] = 6.607679863545e-01;	data_table[52].feg[0].cl[2] = 1.334595783690e-01;	data_table[52].feg[0].cnl[2] = 7.684417819778e-02;	data_table[52].feg[0].cl[3] = 2.338643716269e-03;	data_table[52].feg[0].cnl[3] = 4.558903821153e-03;	data_table[52].feg[0].cl[4] = 3.202510781627e-14;	data_table[52].feg[0].cnl[4] = 1.001555370470e-06;	data_table[52].feg[0].cl[5] = 0.000000000000e+00;	data_table[52].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[53].feg[0].cl[0] = 4.266218604732e+00;	data_table[53].feg[0].cnl[0] = 3.504697596599e+00;	data_table[53].feg[0].cl[1] = 1.005029691758e+00;	data_table[53].feg[0].cnl[1] = 5.851407565166e-01;	data_table[53].feg[0].cl[2] = 1.246194375216e-01;	data_table[53].feg[0].cnl[2] = 7.198187788903e-02;	data_table[53].feg[0].cl[3] = 2.132265991790e-03;	data_table[53].feg[0].cnl[3] = 4.234572723947e-03;	data_table[53].feg[0].cl[4] = 3.511838631788e-14;	data_table[53].feg[0].cnl[4] = 1.000000000000e-06;	data_table[53].feg[0].cl[5] = 0.000000000000e+00;	data_table[53].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[54].feg[0].cl[0] = 5.190940434882e+00;	data_table[54].feg[0].cnl[0] = 1.664349355966e+01;	data_table[54].feg[0].cl[1] = 2.841745148751e+00;	data_table[54].feg[0].cnl[1] = 1.191829182197e+00;	data_table[54].feg[0].cl[2] = 1.635663767305e-01;	data_table[54].feg[0].cnl[2] = 7.848498222869e-02;	data_table[54].feg[0].cl[3] = 1.748039528416e-03;	data_table[54].feg[0].cnl[3] = 3.603430066218e-03;	data_table[54].feg[0].cl[4] = 1.090291951475e-10;	data_table[54].feg[0].cnl[4] = 1.341476216088e-05;	data_table[54].feg[0].cl[5] = 0.000000000000e+00;	data_table[54].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[55].feg[0].cl[0] = 6.570171258806e+00;	data_table[55].feg[0].cnl[0] = 1.385610929468e+01;	data_table[55].feg[0].cl[1] = 2.365684989403e+00;	data_table[55].feg[0].cnl[1] = 9.860953649390e-01;	data_table[55].feg[0].cl[2] = 1.442866019710e-01;	data_table[55].feg[0].cnl[2] = 7.251030506732e-02;	data_table[55].feg[0].cl[3] = 1.857149819325e-03;	data_table[55].feg[0].cnl[3] = 3.728859863919e-03;	data_table[55].feg[0].cl[4] = 4.733024042487e-12;	data_table[55].feg[0].cnl[4] = 4.747189535309e-06;	data_table[55].feg[0].cl[5] = 0.000000000000e+00;	data_table[55].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[56].feg[0].cl[0] = 6.558837092513e+00;	data_table[56].feg[0].cnl[0] = 1.138154017744e+01;	data_table[56].feg[0].cl[1] = 2.197192669037e+00;	data_table[56].feg[0].cnl[1] = 9.157361219983e-01;	data_table[56].feg[0].cl[2] = 1.358248988816e-01;	data_table[56].feg[0].cnl[2] = 6.802346634381e-02;	data_table[56].feg[0].cl[3] = 1.645339585659e-03;	data_table[56].feg[0].cnl[3] = 3.395993729765e-03;	data_table[56].feg[0].cl[4] = 9.068661198337e-12;	data_table[56].feg[0].cnl[4] = 5.679861398627e-06;	data_table[56].feg[0].cl[5] = 0.000000000000e+00;	data_table[56].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[57].feg[0].cl[0] = 6.323929807231e+00;	data_table[57].feg[0].cnl[0] = 1.116505832155e+01;	data_table[57].feg[0].cl[1] = 2.226853584534e+00;	data_table[57].feg[0].cnl[1] = 8.875878162711e-01;	data_table[57].feg[0].cl[2] = 1.287455969130e-01;	data_table[57].feg[0].cnl[2] = 6.424775143993e-02;	data_table[57].feg[0].cl[3] = 1.471011323566e-03;	data_table[57].feg[0].cnl[3] = 3.107132960347e-03;	data_table[57].feg[0].cl[4] = 5.205316129758e-14;	data_table[57].feg[0].cnl[4] = 1.000000000000e-06;	data_table[57].feg[0].cl[5] = 0.000000000000e+00;	data_table[57].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[58].feg[0].cl[0] = 5.996921786878e+00;	data_table[58].feg[0].cnl[0] = 1.266699829864e+01;	data_table[58].feg[0].cl[1] = 2.357565442387e+00;	data_table[58].feg[0].cnl[1] = 8.585252858745e-01;	data_table[58].feg[0].cl[2] = 1.154074752833e-01;	data_table[58].feg[0].cnl[2] = 5.796091591359e-02;	data_table[58].feg[0].cl[3] = 1.105295435698e-03;	data_table[58].feg[0].cnl[3] = 2.525105409170e-03;	data_table[58].feg[0].cl[4] = 1.869462204486e-11;	data_table[58].feg[0].cnl[4] = 6.626891083751e-06;	data_table[58].feg[0].cl[5] = 0.000000000000e+00;	data_table[58].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[59].feg[0].cl[0] = 5.072240962201e+00;	data_table[59].feg[0].cnl[0] = 1.416698682791e+01;	data_table[59].feg[0].cl[1] = 1.970039126936e+00;	data_table[59].feg[0].cnl[1] = 1.990597670731e+00;	data_table[59].feg[0].cl[2] = 1.132649547534e+00;	data_table[59].feg[0].cnl[2] = 4.849790433014e-01;	data_table[59].feg[0].cl[3] = 6.885041069236e-02;	data_table[59].feg[0].cnl[3] = 3.898110046297e-02;	data_table[59].feg[0].cl[4] = 2.199526364998e-04;	data_table[59].feg[0].cnl[4] = 7.464571005975e-04;	data_table[59].feg[0].cl[5] = 0.000000000000e+00;	data_table[59].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[60].feg[0].cl[0] = 4.955669650183e+00;	data_table[60].feg[0].cnl[0] = 1.400045016898e+01;	data_table[60].feg[0].cl[1] = 1.904178583217e+00;	data_table[60].feg[0].cnl[1] = 1.927650290410e+00;	data_table[60].feg[0].cl[2] = 1.162107656701e+00;	data_table[60].feg[0].cnl[2] = 4.772782477738e-01;	data_table[60].feg[0].cl[3] = 6.632980109417e-02;	data_table[60].feg[0].cnl[3] = 3.755055357828e-02;	data_table[60].feg[0].cl[4] = 2.143088041065e-04;	data_table[60].feg[0].cnl[4] = 7.221933906814e-04;	data_table[60].feg[0].cl[5] = 0.000000000000e+00;	data_table[60].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[61].feg[0].cl[0] = 4.851344526046e+00;	data_table[61].feg[0].cnl[0] = 1.364320494775e+01;	data_table[61].feg[0].cl[1] = 1.830268901176e+00;	data_table[61].feg[0].cnl[1] = 1.832864734912e+00;	data_table[61].feg[0].cl[2] = 1.171803387526e+00;	data_table[61].feg[0].cnl[2] = 4.653877183727e-01;	data_table[61].feg[0].cl[3] = 6.387443746649e-02;	data_table[61].feg[0].cnl[3] = 3.619213775415e-02;	data_table[61].feg[0].cl[4] = 2.087477855624e-04;	data_table[61].feg[0].cnl[4] = 6.990908378726e-04;	data_table[61].feg[0].cl[5] = 0.000000000000e+00;	data_table[61].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[62].feg[0].cl[0] = 4.738602796902e+00;	data_table[62].feg[0].cnl[0] = 1.334647424565e+01;	data_table[62].feg[0].cl[1] = 1.803521016929e+00;	data_table[62].feg[0].cnl[1] = 1.734762804901e+00;	data_table[62].feg[0].cl[2] = 1.150595613348e+00;	data_table[62].feg[0].cnl[2] = 4.449128398926e-01;	data_table[62].feg[0].cl[3] = 6.057779012313e-02;	data_table[62].feg[0].cnl[3] = 3.459771403961e-02;	data_table[62].feg[0].cl[4] = 2.027826977644e-04;	data_table[62].feg[0].cnl[4] = 6.769165200263e-04;	data_table[62].feg[0].cl[5] = 0.000000000000e+00;	data_table[62].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[63].feg[0].cl[0] = 4.716238753702e+00;	data_table[63].feg[0].cnl[0] = 1.117212894733e+01;	data_table[63].feg[0].cl[1] = 1.658631832949e+00;	data_table[63].feg[0].cnl[1] = 1.844944358335e+00;	data_table[63].feg[0].cl[2] = 1.205221478349e+00;	data_table[63].feg[0].cnl[2] = 4.362658793004e-01;	data_table[63].feg[0].cl[3] = 5.771141533917e-02;	data_table[63].feg[0].cnl[3] = 3.309986499844e-02;	data_table[63].feg[0].cl[4] = 1.965196604787e-04;	data_table[63].feg[0].cnl[4] = 6.556770071422e-04;	data_table[63].feg[0].cl[5] = 0.000000000000e+00;	data_table[63].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[64].feg[0].cl[0] = 4.508588666409e+00;	data_table[64].feg[0].cnl[0] = 1.292402663919e+01;	data_table[64].feg[0].cl[1] = 1.713369231765e+00;	data_table[64].feg[0].cnl[1] = 1.650227685814e+00;	data_table[64].feg[0].cl[2] = 1.177241604867e+00;	data_table[64].feg[0].cnl[2] = 4.211621058265e-01;	data_table[64].feg[0].cl[3] = 5.560903747152e-02;	data_table[64].feg[0].cnl[3] = 3.197344248625e-02;	data_table[64].feg[0].cl[4] = 1.914594874646e-04;	data_table[64].feg[0].cnl[4] = 6.356801777645e-04;	data_table[64].feg[0].cl[5] = 0.000000000000e+00;	data_table[64].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[65].feg[0].cl[0] = 4.396949521013e+00;	data_table[65].feg[0].cnl[0] = 1.269203417363e+01;	data_table[65].feg[0].cl[1] = 1.650197466569e+00;	data_table[65].feg[0].cnl[1] = 1.636296904251e+00;	data_table[65].feg[0].cl[2] = 1.208637277878e+00;	data_table[65].feg[0].cnl[2] = 4.115274729460e-01;	data_table[65].feg[0].cl[3] = 5.253066619258e-02;	data_table[65].feg[0].cnl[3] = 3.043042600796e-02;	data_table[65].feg[0].cl[4] = 1.850683483962e-04;	data_table[65].feg[0].cnl[4] = 6.161780604303e-04;	data_table[65].feg[0].cl[5] = 0.000000000000e+00;	data_table[65].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[66].feg[0].cl[0] = 4.241694847677e+00;	data_table[66].feg[0].cnl[0] = 1.261911118722e+01;	data_table[66].feg[0].cl[1] = 1.680760792093e+00;	data_table[66].feg[0].cnl[1] = 1.634480642481e+00;	data_table[66].feg[0].cl[2] = 1.195055981189e+00;	data_table[66].feg[0].cnl[2] = 3.923435868025e-01;	data_table[66].feg[0].cl[3] = 4.930966653806e-02;	data_table[66].feg[0].cnl[3] = 2.892483736659e-02;	data_table[66].feg[0].cl[4] = 1.787125028877e-04;	data_table[66].feg[0].cnl[4] = 5.975418791675e-04;	data_table[66].feg[0].cl[5] = 0.000000000000e+00;	data_table[66].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[67].feg[0].cl[0] = 4.087683802848e+00;	data_table[67].feg[0].cnl[0] = 1.249267693555e+01;	data_table[67].feg[0].cl[1] = 1.636977049502e+00;	data_table[67].feg[0].cnl[1] = 1.653287467948e+00;	data_table[67].feg[0].cl[2] = 1.235515529434e+00;	data_table[67].feg[0].cnl[2] = 3.882007271675e-01;	data_table[67].feg[0].cl[3] = 4.814728135789e-02;	data_table[67].feg[0].cnl[3] = 2.821992151070e-02;	data_table[67].feg[0].cl[4] = 1.763368580427e-04;	data_table[67].feg[0].cnl[4] = 5.804414167892e-04;	data_table[67].feg[0].cl[5] = 0.000000000000e+00;	data_table[67].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[68].feg[0].cl[0] = 4.074299774907e+00;	data_table[68].feg[0].cnl[0] = 1.189682973116e+01;	data_table[68].feg[0].cl[1] = 1.549035563336e+00;	data_table[68].feg[0].cnl[1] = 1.518122257319e+00;	data_table[68].feg[0].cl[2] = 1.207860242806e+00;	data_table[68].feg[0].cnl[2] = 3.731462227561e-01;	data_table[68].feg[0].cl[3] = 4.663015064765e-02;	data_table[68].feg[0].cnl[3] = 2.747847138939e-02;	data_table[68].feg[0].cl[4] = 1.742683035876e-04;	data_table[68].feg[0].cnl[4] = 5.641109067039e-04;	data_table[68].feg[0].cl[5] = 0.000000000000e+00;	data_table[68].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[69].feg[0].cl[0] = 3.994491783546e+00;	data_table[69].feg[0].cnl[0] = 1.104890097615e+01;	data_table[69].feg[0].cl[1] = 1.396503969404e+00;	data_table[69].feg[0].cnl[1] = 1.419209658980e+00;	data_table[69].feg[0].cl[2] = 1.236194180484e+00;	data_table[69].feg[0].cnl[2] = 3.708063809543e-01;	data_table[69].feg[0].cl[3] = 4.613765016486e-02;	data_table[69].feg[0].cnl[3] = 2.701644128199e-02;	data_table[69].feg[0].cl[4] = 1.724164015558e-04;	data_table[69].feg[0].cnl[4] = 5.485135633366e-04;	data_table[69].feg[0].cl[5] = 0.000000000000e+00;	data_table[69].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[70].feg[0].cl[0] = 4.042533681724e+00;	data_table[70].feg[0].cnl[0] = 9.270460467663e+00;	data_table[70].feg[0].cl[1] = 1.268316698125e+00;	data_table[70].feg[0].cnl[1] = 1.601326270981e+00;	data_table[70].feg[0].cl[2] = 1.290771160405e+00;	data_table[70].feg[0].cnl[2] = 3.634424507096e-01;	data_table[70].feg[0].cl[3] = 4.470815108007e-02;	data_table[70].feg[0].cnl[3] = 2.632521943092e-02;	data_table[70].feg[0].cl[4] = 1.703086670707e-04;	data_table[70].feg[0].cnl[4] = 5.334875293597e-04;	data_table[70].feg[0].cl[5] = 0.000000000000e+00;	data_table[70].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[71].feg[0].cl[0] = 3.984682678791e+00;	data_table[71].feg[0].cnl[0] = 8.030076981353e+00;	data_table[71].feg[0].cl[1] = 1.258524413902e+00;	data_table[71].feg[0].cnl[1] = 1.584727144410e+00;	data_table[71].feg[0].cl[2] = 1.236485955715e+00;	data_table[71].feg[0].cnl[2] = 3.434580335946e-01;	data_table[71].feg[0].cl[3] = 4.263984879087e-02;	data_table[71].feg[0].cnl[3] = 2.537580469647e-02;	data_table[71].feg[0].cl[4] = 1.671028017813e-04;	data_table[71].feg[0].cnl[4] = 5.188343291963e-04;	data_table[71].feg[0].cl[5] = 0.000000000000e+00;	data_table[71].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[72].feg[0].cl[0] = 4.817901900524e+00;	data_table[72].feg[0].cnl[0] = 5.426400271572e+00;	data_table[72].feg[0].cl[1] = 1.519717440754e+00;	data_table[72].feg[0].cnl[1] = 3.624940040879e-01;	data_table[72].feg[0].cl[2] = 7.473079246650e-02;	data_table[72].feg[0].cnl[2] = 1.269501673532e-02;	data_table[72].feg[0].cl[3] = -4.358782637620e-02;	data_table[72].feg[0].cnl[3] = 9.835303968311e-03;	data_table[72].feg[0].cl[4] = 2.376925993528e-04;	data_table[72].feg[0].cnl[4] = 5.241057252213e-04;	data_table[72].feg[0].cl[5] = 0.000000000000e+00;	data_table[72].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[73].feg[0].cl[0] = 4.774437195700e+00;	data_table[73].feg[0].cnl[0] = 4.830902604138e+00;	data_table[73].feg[0].cl[1] = 1.435902529044e+00;	data_table[73].feg[0].cnl[1] = 3.415682858732e-01;	data_table[73].feg[0].cl[2] = 6.967779522030e-02;	data_table[73].feg[0].cnl[2] = 1.206224833212e-02;	data_table[73].feg[0].cl[3] = -4.025316655835e-02;	data_table[73].feg[0].cnl[3] = 9.263301726683e-03;	data_table[73].feg[0].cl[4] = 2.356469479354e-04;	data_table[73].feg[0].cnl[4] = 5.110242119502e-04;	data_table[73].feg[0].cl[5] = 0.000000000000e+00;	data_table[73].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[74].feg[0].cl[0] = 4.766686518798e+00;	data_table[74].feg[0].cnl[0] = 4.306677349144e+00;	data_table[74].feg[0].cl[1] = 1.338427589796e+00;	data_table[74].feg[0].cnl[1] = 3.164304083801e-01;	data_table[74].feg[0].cl[2] = 4.365696565125e-02;	data_table[74].feg[0].cnl[2] = 1.139424638784e-02;	data_table[74].feg[0].cl[3] = -1.753260675432e-02;	data_table[74].feg[0].cnl[3] = 6.997060564266e-03;	data_table[74].feg[0].cl[4] = 2.615319473821e-04;	data_table[74].feg[0].cnl[4] = 5.051470884569e-04;	data_table[74].feg[0].cl[5] = 0.000000000000e+00;	data_table[74].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[75].feg[0].cl[0] = 4.676000312409e+00;	data_table[75].feg[0].cnl[0] = 3.865557182748e+00;	data_table[75].feg[0].cl[1] = 1.273344866365e+00;	data_table[75].feg[0].cnl[1] = 3.028993048000e-01;	data_table[75].feg[0].cl[2] = 5.409355590252e-02;	data_table[75].feg[0].cnl[2] = 1.108430187698e-02;	data_table[75].feg[0].cl[3] = -2.767964025732e-02;	data_table[75].feg[0].cnl[3] = 7.882286252226e-03;	data_table[75].feg[0].cl[4] = 2.409050891036e-04;	data_table[75].feg[0].cnl[4] = 4.879082519332e-04;	data_table[75].feg[0].cl[5] = 0.000000000000e+00;	data_table[75].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[76].feg[0].cl[0] = 4.617623226181e+00;	data_table[76].feg[0].cnl[0] = 3.412442820968e+00;	data_table[76].feg[0].cl[1] = 1.176976436205e+00;	data_table[76].feg[0].cnl[1] = 2.808615391884e-01;	data_table[76].feg[0].cl[2] = 4.221910652529e-02;	data_table[76].feg[0].cnl[2] = 1.041307154535e-02;	data_table[76].feg[0].cl[3] = -1.808427255381e-02;	data_table[76].feg[0].cnl[3] = 6.509692519465e-03;	data_table[76].feg[0].cl[4] = 2.655037530051e-04;	data_table[76].feg[0].cnl[4] = 4.814893695047e-04;	data_table[76].feg[0].cl[5] = 0.000000000000e+00;	data_table[76].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[77].feg[0].cl[0] = 4.297575590751e+00;	data_table[77].feg[0].cnl[0] = 2.806366885349e+00;	data_table[77].feg[0].cl[1] = 1.082376077985e+00;	data_table[77].feg[0].cnl[1] = 2.631812747251e-01;	data_table[77].feg[0].cl[2] = 3.997378475454e-02;	data_table[77].feg[0].cnl[2] = 1.007519901883e-02;	data_table[77].feg[0].cl[3] = -1.669384357940e-02;	data_table[77].feg[0].cnl[3] = 6.149755341311e-03;	data_table[77].feg[0].cl[4] = 2.683899648674e-04;	data_table[77].feg[0].cnl[4] = 4.706255960743e-04;	data_table[77].feg[0].cl[5] = 0.000000000000e+00;	data_table[77].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[78].feg[0].cl[0] = 4.319253070395e+00;	data_table[78].feg[0].cnl[0] = 2.430804640247e+00;	data_table[78].feg[0].cl[1] = 9.361407976022e-01;	data_table[78].feg[0].cnl[1] = 2.326073755862e-01;	data_table[78].feg[0].cl[2] = 7.588030908582e-02;	data_table[78].feg[0].cnl[2] = 8.257084212608e-03;	data_table[78].feg[0].cl[3] = -5.503761562648e-02;	data_table[78].feg[0].cnl[3] = 6.877684796876e-03;	data_table[78].feg[0].cl[4] = 2.634385321135e-04;	data_table[78].feg[0].cnl[4] = 4.587717445328e-04;	data_table[78].feg[0].cl[5] = 0.000000000000e+00;	data_table[78].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[79].feg[0].cl[0] = 1.685762695137e+00;	data_table[79].feg[0].cnl[0] = 5.699443864462e+00;	data_table[79].feg[0].cl[1] = 3.048868823223e+00;	data_table[79].feg[0].cnl[1] = 1.538051399135e+00;	data_table[79].feg[0].cl[2] = 7.029763988130e-01;	data_table[79].feg[0].cnl[2] = 1.934650027769e-01;	data_table[79].feg[0].cl[3] = 2.326700329799e-02;	data_table[79].feg[0].cnl[3] = 1.608026793360e-02;	data_table[79].feg[0].cl[4] = 1.250795337370e-04;	data_table[79].feg[0].cnl[4] = 4.167058863466e-04;	data_table[79].feg[0].cl[5] = 0.000000000000e+00;	data_table[79].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[80].feg[0].cl[0] = 2.828978370248e+00;	data_table[80].feg[0].cnl[0] = 8.513830179204e+00;	data_table[80].feg[0].cl[1] = 2.881925073356e+00;	data_table[80].feg[0].cnl[1] = 1.332948750181e+00;	data_table[80].feg[0].cl[2] = 6.298422694932e-01;	data_table[80].feg[0].cnl[2] = 1.772467356348e-01;	data_table[80].feg[0].cl[3] = 2.113600361384e-02;	data_table[80].feg[0].cnl[3] = 1.493800717596e-02;	data_table[80].feg[0].cl[4] = 1.182832853609e-04;	data_table[80].feg[0].cnl[4] = 4.054040056495e-04;	data_table[80].feg[0].cl[5] = 0.000000000000e+00;	data_table[80].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[81].feg[0].cl[0] = 3.410340859295e+00;	data_table[81].feg[0].cnl[0] = 6.738247028041e+00;	data_table[81].feg[0].cl[1] = 2.487288720979e+00;	data_table[81].feg[0].cnl[1] = 1.185221394897e+00;	data_table[81].feg[0].cl[2] = 6.024932571690e-01;	data_table[81].feg[0].cnl[2] = 1.694306049981e-01;	data_table[81].feg[0].cl[3] = 1.976567414282e-02;	data_table[81].feg[0].cnl[3] = 1.404902485888e-02;	data_table[81].feg[0].cl[4] = 1.114884158005e-04;	data_table[81].feg[0].cnl[4] = 3.944501643171e-04;	data_table[81].feg[0].cl[5] = 0.000000000000e+00;	data_table[81].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[82].feg[0].cl[0] = 3.888919287457e+00;	data_table[82].feg[0].cnl[0] = 5.184823019984e+00;	data_table[82].feg[0].cl[1] = 2.010985450052e+00;	data_table[82].feg[0].cnl[1] = 1.042316727779e+00;	data_table[82].feg[0].cl[2] = 5.910876080376e-01;	data_table[82].feg[0].cnl[2] = 1.651423478571e-01;	data_table[82].feg[0].cl[3] = 1.890167442523e-02;	data_table[82].feg[0].cnl[3] = 1.339208996724e-02;	data_table[82].feg[0].cl[4] = 1.059800257370e-04;	data_table[82].feg[0].cnl[4] = 3.841135706125e-04;	data_table[82].feg[0].cl[5] = 0.000000000000e+00;	data_table[82].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[83].feg[0].cl[0] = 3.895470428731e+00;	data_table[83].feg[0].cnl[0] = 3.771404146309e+00;	data_table[83].feg[0].cl[1] = 1.622709537429e+00;	data_table[83].feg[0].cnl[1] = 9.810595863021e-01;	data_table[83].feg[0].cl[2] = 5.947573595513e-01;	data_table[83].feg[0].cnl[2] = 1.612894773037e-01;	data_table[83].feg[0].cl[3] = 1.746595684824e-02;	data_table[83].feg[0].cnl[3] = 1.240544459433e-02;	data_table[83].feg[0].cl[4] = 9.671744024565e-05;	data_table[83].feg[0].cnl[4] = 3.729452600179e-04;	data_table[83].feg[0].cl[5] = 0.000000000000e+00;	data_table[83].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[84].feg[0].cl[0] = 4.772690328171e+00;	data_table[84].feg[0].cnl[0] = 4.062706726401e+00;	data_table[84].feg[0].cl[1] = 1.384048085551e+00;	data_table[84].feg[0].cnl[1] = 8.149198933100e-01;	data_table[84].feg[0].cl[2] = 5.504642853228e-01;	data_table[84].feg[0].cnl[2] = 1.520919680403e-01;	data_table[84].feg[0].cl[3] = 1.620711203298e-02;	data_table[84].feg[0].cnl[3] = 1.160703684042e-02;	data_table[84].feg[0].cl[4] = 9.018892176023e-05;	data_table[84].feg[0].cnl[4] = 3.626421650928e-04;	data_table[84].feg[0].cl[5] = 0.000000000000e+00;	data_table[84].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[85].feg[0].cl[0] = 5.121588284998e+00;	data_table[85].feg[0].cnl[0] = 3.564768992576e+00;	data_table[85].feg[0].cl[1] = 1.065311970337e+00;	data_table[85].feg[0].cnl[1] = 6.945489117785e-01;	data_table[85].feg[0].cl[2] = 5.333451761011e-01;	data_table[85].feg[0].cnl[2] = 1.468296828139e-01;	data_table[85].feg[0].cl[3] = 1.517123066434e-02;	data_table[85].feg[0].cnl[3] = 1.088525725670e-02;	data_table[85].feg[0].cl[4] = 8.333789972382e-05;	data_table[85].feg[0].cnl[4] = 3.523966050259e-04;	data_table[85].feg[0].cl[5] = 0.000000000000e+00;	data_table[85].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[86].feg[0].cl[0] = 4.326300506863e+00;	data_table[86].feg[0].cnl[0] = 1.598464225924e+01;	data_table[86].feg[0].cl[1] = 3.927685983348e+00;	data_table[86].feg[0].cnl[1] = 1.785016789169e+00;	data_table[86].feg[0].cl[2] = 6.817081940053e-01;	data_table[86].feg[0].cnl[2] = 1.631874713912e-01;	data_table[86].feg[0].cl[3] = 1.572388032955e-02;	data_table[86].feg[0].cnl[3] = 1.088933530078e-02;	data_table[86].feg[0].cl[4] = 8.143545562484e-05;	data_table[86].feg[0].cnl[4] = 3.443767774590e-04;	data_table[86].feg[0].cl[5] = 0.000000000000e+00;	data_table[86].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[87].feg[0].cl[0] = 6.206606028573e+00;	data_table[87].feg[0].cnl[0] = 1.312206218351e+01;	data_table[87].feg[0].cl[1] = 3.158572113291e+00;	data_table[87].feg[0].cnl[1] = 1.359507039642e+00;	data_table[87].feg[0].cl[2] = 5.763788069594e-01;	data_table[87].feg[0].cnl[2] = 1.425271037182e-01;	data_table[87].feg[0].cl[3] = 1.287606664376e-02;	data_table[87].feg[0].cnl[3] = 9.318402543442e-03;	data_table[87].feg[0].cl[4] = 6.698453367208e-05;	data_table[87].feg[0].cnl[4] = 3.307005301731e-04;	data_table[87].feg[0].cl[5] = 0.000000000000e+00;	data_table[87].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[88].feg[0].cl[0] = 6.904795469039e+00;	data_table[88].feg[0].cnl[0] = 1.093977295526e+01;	data_table[88].feg[0].cl[1] = 2.791640980424e+00;	data_table[88].feg[0].cnl[1] = 1.205162050086e+00;	data_table[88].feg[0].cl[2] = 5.394488841112e-01;	data_table[88].feg[0].cnl[2] = 1.340293069461e-01;	data_table[88].feg[0].cl[3] = 1.155875654381e-02;	data_table[88].feg[0].cnl[3] = 8.424990860991e-03;	data_table[88].feg[0].cl[4] = 5.590988205797e-05;	data_table[88].feg[0].cnl[4] = 3.176743009512e-04;	data_table[88].feg[0].cl[5] = 0.000000000000e+00;	data_table[88].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[89].feg[0].cl[0] = 7.108608985958e+00;	data_table[89].feg[0].cnl[0] = 8.936472275737e+00;	data_table[89].feg[0].cl[1] = 2.468518228417e+00;	data_table[89].feg[0].cnl[1] = 1.074932025413e+00;	data_table[89].feg[0].cl[2] = 5.068648952366e-01;	data_table[89].feg[0].cnl[2] = 1.264470423469e-01;	data_table[89].feg[0].cl[3] = 1.046185831411e-02;	data_table[89].feg[0].cnl[3] = 7.668062648681e-03;	data_table[89].feg[0].cl[4] = 4.603207245220e-05;	data_table[89].feg[0].cnl[4] = 3.041243360628e-04;	data_table[89].feg[0].cl[5] = 0.000000000000e+00;	data_table[89].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[90].feg[0].cl[0] = 6.265118991091e+00;	data_table[90].feg[0].cnl[0] = 1.013987934922e+01;	data_table[90].feg[0].cl[1] = 2.997883072507e+00;	data_table[90].feg[0].cnl[1] = 1.174043143373e+00;	data_table[90].feg[0].cl[2] = 4.871287920782e-01;	data_table[90].feg[0].cnl[2] = 1.193071834458e-01;	data_table[90].feg[0].cl[3] = 8.844055245587e-03;	data_table[90].feg[0].cnl[3] = 6.475339055316e-03;	data_table[90].feg[0].cl[4] = 2.508908042739e-05;	data_table[90].feg[0].cnl[4] = 2.735135302295e-04;	data_table[90].feg[0].cl[5] = 0.000000000000e+00;	data_table[90].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[91].feg[0].cl[0] = 6.002970274769e+00;	data_table[91].feg[0].cnl[0] = 9.825086413018e+00;	data_table[91].feg[0].cl[1] = 3.068445619847e+00;	data_table[91].feg[0].cnl[1] = 1.135512631855e+00;	data_table[91].feg[0].cl[2] = 4.555780347632e-01;	data_table[91].feg[0].cnl[2] = 1.113666249326e-01;	data_table[91].feg[0].cl[3] = 7.503868907966e-03;	data_table[91].feg[0].cnl[3] = 5.398534942030e-03;	data_table[91].feg[0].cl[4] = 2.201729737597e-06;	data_table[91].feg[0].cnl[4] = 1.622502547230e-04;	data_table[91].feg[0].cl[5] = 0.000000000000e+00;	data_table[91].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[92].feg[0].cl[0] = 5.776246099937e+00;	data_table[92].feg[0].cnl[0] = 9.627666105822e+00;	data_table[92].feg[0].cl[1] = 3.137669947153e+00;	data_table[92].feg[0].cnl[1] = 1.105431535799e+00;	data_table[92].feg[0].cl[2] = 4.311200585167e-01;	data_table[92].feg[0].cnl[2] = 1.055575226249e-01;	data_table[92].feg[0].cl[3] = 6.963860955603e-03;	data_table[92].feg[0].cnl[3] = 5.046086894603e-03;	data_table[92].feg[0].cl[4] = 3.342636320371e-08;	data_table[92].feg[0].cnl[4] = 4.945724792708e-05;	data_table[92].feg[0].cl[5] = 0.000000000000e+00;	data_table[92].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[93].feg[0].cl[0] = 5.114441086159e+00;	data_table[93].feg[0].cnl[0] = 1.110961902164e+01;	data_table[93].feg[0].cl[1] = 3.505295669439e+00;	data_table[93].feg[0].cnl[1] = 1.141789662453e+00;	data_table[93].feg[0].cl[2] = 4.160774022955e-01;	data_table[93].feg[0].cnl[2] = 1.015949971146e-01;	data_table[93].feg[0].cl[3] = 6.685840916280e-03;	data_table[93].feg[0].cnl[3] = 4.868898995445e-03;	data_table[93].feg[0].cl[4] = 1.190788405432e-09;	data_table[93].feg[0].cnl[4] = 1.725005671512e-05;	data_table[93].feg[0].cl[5] = 0.000000000000e+00;	data_table[93].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[94].feg[0].cl[0] = 4.928379432669e+00;	data_table[94].feg[0].cnl[0] = 1.094873051522e+01;	data_table[94].feg[0].cl[1] = 3.532165584477e+00;	data_table[94].feg[0].cnl[1] = 1.099697230044e+00;	data_table[94].feg[0].cl[2] = 3.936063503053e-01;	data_table[94].feg[0].cnl[2] = 9.662556321866e-02;	data_table[94].feg[0].cl[3] = 6.348631807030e-03;	data_table[94].feg[0].cnl[3] = 4.671056132769e-03;	data_table[94].feg[0].cl[4] = 7.282240817667e-10;	data_table[94].feg[0].cnl[4] = 1.446137126888e-05;	data_table[94].feg[0].cl[5] = 0.000000000000e+00;	data_table[94].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[95].feg[0].cl[0] = 5.194007987831e+00;	data_table[95].feg[0].cnl[0] = 9.045839152597e+00;	data_table[95].feg[0].cl[1] = 3.236455236583e+00;	data_table[95].feg[0].cnl[1] = 9.895966606816e-01;	data_table[95].feg[0].cl[2] = 3.626720076528e-01;	data_table[95].feg[0].cnl[2] = 9.035552827339e-02;	data_table[95].feg[0].cl[3] = 5.864767558415e-03;	data_table[95].feg[0].cnl[3] = 4.403009455041e-03;	data_table[95].feg[0].cl[4] = 4.186081279040e-10;	data_table[95].feg[0].cnl[4] = 1.182689042828e-05;	data_table[95].feg[0].cl[5] = 0.000000000000e+00;	data_table[95].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[96].feg[0].cl[0] = 5.062674964788e+00;	data_table[96].feg[0].cnl[0] = 8.942161813774e+00;	data_table[96].feg[0].cl[1] = 3.251994776161e+00;	data_table[96].feg[0].cnl[1] = 9.519372069789e-01;	data_table[96].feg[0].cl[2] = 3.423070976390e-01;	data_table[96].feg[0].cnl[2] = 8.581899834632e-02;	data_table[96].feg[0].cl[3] = 5.523161420640e-03;	data_table[96].feg[0].cnl[3] = 4.199464398228e-03;	data_table[96].feg[0].cl[4] = 2.320587774719e-13;	data_table[96].feg[0].cnl[4] = 1.003320766129e-06;	data_table[96].feg[0].cl[5] = 0.000000000000e+00;	data_table[96].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[97].feg[0].cl[0] = 4.459033865939e+00;	data_table[97].feg[0].cnl[0] = 1.035332827659e+01;	data_table[97].feg[0].cl[1] = 3.540957137520e+00;	data_table[97].feg[0].cnl[1] = 9.740623966592e-01;	data_table[97].feg[0].cl[2] = 3.335691264893e-01;	data_table[97].feg[0].cnl[2] = 8.347403612471e-02;	data_table[97].feg[0].cl[3] = 5.439870086607e-03;	data_table[97].feg[0].cnl[3] = 4.121177065070e-03;	data_table[97].feg[0].cl[4] = 2.399504795464e-13;	data_table[97].feg[0].cnl[4] = 1.000000000000e-06;	data_table[97].feg[0].cl[5] = 0.000000000000e+00;	data_table[97].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[98].feg[0].cl[0] = 4.360442820372e+00;	data_table[98].feg[0].cnl[0] = 1.003183148831e+01;	data_table[98].feg[0].cl[1] = 3.511473670570e+00;	data_table[98].feg[0].cnl[1] = 9.187495133185e-01;	data_table[98].feg[0].cl[2] = 3.027297662978e-01;	data_table[98].feg[0].cnl[2] = 7.561230629461e-02;	data_table[98].feg[0].cl[3] = 4.556929533227e-03;	data_table[98].feg[0].cnl[3] = 2.324924961379e-03;	data_table[98].feg[0].cl[4] = -7.031867827314e-04;	data_table[98].feg[0].cnl[4] = 9.468127404760e-04;	data_table[98].feg[0].cl[5] = 0.000000000000e+00;	data_table[98].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[99].feg[0].cl[0] = 4.223354382825e+00;	data_table[99].feg[0].cnl[0] = 9.822572452935e+00;	data_table[99].feg[0].cl[1] = 3.497909486848e+00;	data_table[99].feg[0].cnl[1] = 8.851991604703e-01;	data_table[99].feg[0].cl[2] = 2.890667681101e-01;	data_table[99].feg[0].cnl[2] = 7.262699608308e-02;	data_table[99].feg[0].cl[3] = 1.086385321320e-01;	data_table[99].feg[0].cnl[3] = 1.616074855132e-03;	data_table[99].feg[0].cl[4] = -1.049691699837e-01;	data_table[99].feg[0].cnl[4] = 1.590155088388e-03;	data_table[99].feg[0].cl[5] = 0.000000000000e+00;	data_table[99].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[100].feg[0].cl[0] = 4.105093096117e+00;	data_table[100].feg[0].cnl[0] = 9.604366132224e+00;	data_table[100].feg[0].cl[1] = 3.478902121090e+00;	data_table[100].feg[0].cnl[1] = 8.506197530668e-01;	data_table[100].feg[0].cl[2] = 2.739290417276e-01;	data_table[100].feg[0].cnl[2] = 6.947173969610e-02;	data_table[100].feg[0].cl[3] = 5.465078212511e-01;	data_table[100].feg[0].cnl[3] = 1.567207693675e-03;	data_table[100].feg[0].cl[4] = -5.429320802213e-01;	data_table[100].feg[0].cnl[4] = 1.562369978280e-03;	data_table[100].feg[0].cl[5] = 0.000000000000e+00;	data_table[100].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[101].feg[0].cl[0] = 3.978174422541e+00;	data_table[101].feg[0].cnl[0] = 9.322844061694e+00;	data_table[101].feg[0].cl[1] = 3.455845623349e+00;	data_table[101].feg[0].cnl[1] = 8.190918007908e-01;	data_table[101].feg[0].cl[2] = 2.609732707382e-01;	data_table[101].feg[0].cnl[2] = 6.674448990551e-02;	data_table[101].feg[0].cl[3] = 1.448317747385e-02;	data_table[101].feg[0].cnl[3] = 1.633126678336e-03;	data_table[101].feg[0].cl[4] = -1.097649403824e-02;	data_table[101].feg[0].cnl[4] = 1.433092520468e-03;	data_table[101].feg[0].cl[5] = 0.000000000000e+00;	data_table[101].feg[0].cnl[5] = 0.000000000000e+00;
				data_table[102].feg[0].cl[0] = 4.472319512080e+00;	data_table[102].feg[0].cnl[0] = 8.221771154610e+00;	data_table[102].feg[0].cl[1] = 3.204128540727e+00;	data_table[102].feg[0].cnl[1] = 7.452706498861e-01;	data_table[102].feg[0].cl[2] = 2.387265503153e-01;	data_table[102].feg[0].cnl[2] = 6.242119838316e-02;	data_table[102].feg[0].cl[3] = 2.697110402440e-02;	data_table[102].feg[0].cnl[3] = 1.514952131673e-03;	data_table[102].feg[0].cl[4] = -2.364570714437e-02;	data_table[102].feg[0].cnl[4] = 1.423557559439e-03;	data_table[102].feg[0].cl[5] = 0.000000000000e+00;	data_table[102].feg[0].cnl[5] = 0.000000000000e+00;				
			}

			// 7: Peng et al. parameterization for Ions- 5 Gaussians - [0, 4]
			void Load_feg_Peng_ion_0_4()
			{
				data_table[0].feg[1].cl[0]= 0.1400000000;	data_table[0].feg[1].cnl[0]= 0.9840000000;	data_table[0].feg[1].cl[1]= 0.6490000000;	data_table[0].feg[1].cnl[1]= 8.6700000000;	data_table[0].feg[1].cl[2]= 1.3700000000;	data_table[0].feg[1].cnl[2]= 38.9000000000;	data_table[0].feg[1].cl[3]= 0.3370000000;	data_table[0].feg[1].cnl[3]= 111.0000000000;	data_table[0].feg[1].cl[4]= 0.7870000000;	data_table[0].feg[1].cnl[4]= 166.0000000000;
				data_table[2].feg[2].cl[0]= 0.0046000000;	data_table[2].feg[2].cnl[0]= 0.0358000000;	data_table[2].feg[2].cl[1]= 0.0165000000;	data_table[2].feg[2].cnl[1]= 0.2390000000;	data_table[2].feg[2].cl[2]= 0.0435000000;	data_table[2].feg[2].cnl[2]= 0.8790000000;	data_table[2].feg[2].cl[3]= 0.0649000000;	data_table[2].feg[2].cnl[3]= 2.6400000000;	data_table[2].feg[2].cl[4]= 0.0270000000;	data_table[2].feg[2].cnl[4]= 7.0900000000;
				data_table[3].feg[4].cl[0]= 0.0034000000;	data_table[3].feg[4].cnl[0]= 0.0267000000;	data_table[3].feg[4].cl[1]= 0.0103000000;	data_table[3].feg[4].cnl[1]= 0.1620000000;	data_table[3].feg[4].cl[2]= 0.0233000000;	data_table[3].feg[4].cnl[2]= 0.5310000000;	data_table[3].feg[4].cl[3]= 0.0325000000;	data_table[3].feg[4].cnl[3]= 1.4800000000;	data_table[3].feg[4].cl[4]= 0.0120000000;	data_table[3].feg[4].cnl[4]= 3.8800000000;
				data_table[7].feg[1].cl[0]= 0.2050000000;	data_table[7].feg[1].cnl[0]= 0.3970000000;	data_table[7].feg[1].cl[1]= 0.6280000000;	data_table[7].feg[1].cnl[1]= 2.6400000000;	data_table[7].feg[1].cl[2]= 1.1700000000;	data_table[7].feg[1].cnl[2]= 8.8000000000;	data_table[7].feg[1].cl[3]= 1.0300000000;	data_table[7].feg[1].cnl[3]= 27.1000000000;	data_table[7].feg[1].cl[4]= 0.2900000000;	data_table[7].feg[1].cnl[4]= 91.8000000000;
				data_table[7].feg[3].cl[0]= 0.0421000000;	data_table[7].feg[3].cnl[0]= 0.0609000000;	data_table[7].feg[3].cl[1]= 0.2100000000;	data_table[7].feg[3].cnl[1]= 0.5590000000;	data_table[7].feg[3].cl[2]= 0.8520000000;	data_table[7].feg[3].cnl[2]= 2.9600000000;	data_table[7].feg[3].cl[3]= 1.8200000000;	data_table[7].feg[3].cnl[3]= 11.5000000000;	data_table[7].feg[3].cl[4]= 1.1700000000;	data_table[7].feg[3].cnl[4]= 37.7000000000;
				data_table[8].feg[1].cl[0]= 0.1340000000;	data_table[8].feg[1].cnl[0]= 0.2280000000;	data_table[8].feg[1].cl[1]= 0.3910000000;	data_table[8].feg[1].cnl[1]= 1.4700000000;	data_table[8].feg[1].cl[2]= 0.8140000000;	data_table[8].feg[1].cnl[2]= 4.6800000000;	data_table[8].feg[1].cl[3]= 0.9280000000;	data_table[8].feg[1].cnl[3]= 13.2000000000;	data_table[8].feg[1].cl[4]= 0.3470000000;	data_table[8].feg[1].cnl[4]= 36.0000000000;
				data_table[10].feg[2].cl[0]= 0.0256000000;	data_table[10].feg[2].cnl[0]= 0.0397000000;	data_table[10].feg[2].cl[1]= 0.0919000000;	data_table[10].feg[2].cnl[1]= 0.2870000000;	data_table[10].feg[2].cl[2]= 0.2970000000;	data_table[10].feg[2].cnl[2]= 1.1800000000;	data_table[10].feg[2].cl[3]= 0.5140000000;	data_table[10].feg[2].cnl[3]= 3.7500000000;	data_table[10].feg[2].cl[4]= 0.1990000000;	data_table[10].feg[2].cnl[4]= 10.8000000000;
				data_table[11].feg[4].cl[0]= 0.0210000000;	data_table[11].feg[4].cnl[0]= 0.0331000000;	data_table[11].feg[4].cl[1]= 0.0672000000;	data_table[11].feg[4].cnl[1]= 0.2220000000;	data_table[11].feg[4].cl[2]= 0.1980000000;	data_table[11].feg[4].cnl[2]= 0.8380000000;	data_table[11].feg[4].cl[3]= 0.3680000000;	data_table[11].feg[4].cnl[3]= 2.4800000000;	data_table[11].feg[4].cl[4]= 0.1740000000;	data_table[11].feg[4].cnl[4]= 6.7500000000;
				data_table[12].feg[6].cl[0]= 0.0192000000;	data_table[12].feg[6].cnl[0]= 0.0306000000;	data_table[12].feg[6].cl[1]= 0.0579000000;	data_table[12].feg[6].cnl[1]= 0.1980000000;	data_table[12].feg[6].cl[2]= 0.1630000000;	data_table[12].feg[6].cnl[2]= 0.7130000000;	data_table[12].feg[6].cl[3]= 0.2840000000;	data_table[12].feg[6].cnl[3]= 2.0400000000;	data_table[12].feg[6].cl[4]= 0.1140000000;	data_table[12].feg[6].cnl[4]= 5.2500000000;
				data_table[13].feg[8].cl[0]= 0.1920000000;	data_table[13].feg[8].cnl[0]= 0.3590000000;	data_table[13].feg[8].cl[1]= 0.2890000000;	data_table[13].feg[8].cnl[1]= 1.9600000000;	data_table[13].feg[8].cl[2]= 0.1000000000;	data_table[13].feg[8].cnl[2]= 9.3400000000;	data_table[13].feg[8].cl[3]=-0.0728000000;	data_table[13].feg[8].cnl[3]= 11.1000000000;	data_table[13].feg[8].cl[4]= 0.0012000000;	data_table[13].feg[8].cnl[4]= 13.4000000000;
				data_table[16].feg[1].cl[0]= 0.2650000000;	data_table[16].feg[1].cnl[0]= 0.2520000000;	data_table[16].feg[1].cl[1]= 0.5960000000;	data_table[16].feg[1].cnl[1]= 1.5600000000;	data_table[16].feg[1].cl[2]= 1.6000000000;	data_table[16].feg[1].cnl[2]= 6.2100000000;	data_table[16].feg[1].cl[3]= 2.6900000000;	data_table[16].feg[1].cnl[3]= 17.8000000000;	data_table[16].feg[1].cl[4]= 1.2300000000;	data_table[16].feg[1].cnl[4]= 47.8000000000;
				data_table[18].feg[2].cl[0]= 0.1990000000;	data_table[18].feg[2].cnl[0]= 0.1920000000;	data_table[18].feg[2].cl[1]= 0.3960000000;	data_table[18].feg[2].cnl[1]= 1.1000000000;	data_table[18].feg[2].cl[2]= 0.9280000000;	data_table[18].feg[2].cnl[2]= 3.9100000000;	data_table[18].feg[2].cl[3]= 1.4500000000;	data_table[18].feg[2].cnl[3]= 9.7500000000;	data_table[18].feg[2].cl[4]= 0.4500000000;	data_table[18].feg[2].cnl[4]= 23.4000000000;
				data_table[19].feg[4].cl[0]= 0.1640000000;	data_table[19].feg[4].cnl[0]= 0.1570000000;	data_table[19].feg[4].cl[1]= 0.3270000000;	data_table[19].feg[4].cnl[1]= 0.8940000000;	data_table[19].feg[4].cl[2]= 0.7430000000;	data_table[19].feg[4].cnl[2]= 3.1500000000;	data_table[19].feg[4].cl[3]= 1.1600000000;	data_table[19].feg[4].cnl[3]= 7.6700000000;	data_table[19].feg[4].cl[4]= 0.3070000000;	data_table[19].feg[4].cnl[4]= 17.7000000000;
				data_table[20].feg[6].cl[0]= 0.1630000000;	data_table[20].feg[6].cnl[0]= 0.1570000000;	data_table[20].feg[6].cl[1]= 0.3070000000;	data_table[20].feg[6].cnl[1]= 0.8990000000;	data_table[20].feg[6].cl[2]= 0.7160000000;	data_table[20].feg[6].cnl[2]= 3.0600000000;	data_table[20].feg[6].cl[3]= 0.8800000000;	data_table[20].feg[6].cnl[3]= 7.0500000000;	data_table[20].feg[6].cl[4]= 0.1390000000;	data_table[20].feg[6].cnl[4]= 16.1000000000;
				data_table[21].feg[4].cl[0]= 0.3990000000;	data_table[21].feg[4].cnl[0]= 0.3760000000;	data_table[21].feg[4].cl[1]= 1.0400000000;	data_table[21].feg[4].cnl[1]= 2.7400000000;	data_table[21].feg[4].cl[2]= 1.2100000000;	data_table[21].feg[4].cnl[2]= 8.1000000000;	data_table[21].feg[4].cl[3]=-0.0797000000;	data_table[21].feg[4].cnl[3]= 14.2000000000;	data_table[21].feg[4].cl[4]= 0.3520000000;	data_table[21].feg[4].cnl[4]= 23.2000000000;
				data_table[21].feg[6].cl[0]= 0.3640000000;	data_table[21].feg[6].cnl[0]= 0.3640000000;	data_table[21].feg[6].cl[1]= 0.9190000000;	data_table[21].feg[6].cnl[1]= 2.6700000000;	data_table[21].feg[6].cl[2]= 1.3500000000;	data_table[21].feg[6].cnl[2]= 8.1800000000;	data_table[21].feg[6].cl[3]=-0.9330000000;	data_table[21].feg[6].cnl[3]= 11.8000000000;	data_table[21].feg[6].cl[4]= 0.5890000000;	data_table[21].feg[6].cnl[4]= 14.9000000000;
				data_table[21].feg[8].cl[0]= 0.1160000000;	data_table[21].feg[8].cnl[0]= 0.1080000000;	data_table[21].feg[8].cl[1]= 0.2560000000;	data_table[21].feg[8].cnl[1]= 0.6550000000;	data_table[21].feg[8].cl[2]= 0.5650000000;	data_table[21].feg[8].cnl[2]= 2.3800000000;	data_table[21].feg[8].cl[3]= 0.7720000000;	data_table[21].feg[8].cnl[3]= 5.5100000000;	data_table[21].feg[8].cl[4]= 0.1320000000;	data_table[21].feg[8].cnl[4]= 12.3000000000;
				data_table[22].feg[4].cl[0]= 0.3170000000;	data_table[22].feg[4].cnl[0]= 0.2690000000;	data_table[22].feg[4].cl[1]= 0.9390000000;	data_table[22].feg[4].cnl[1]= 2.0900000000;	data_table[22].feg[4].cl[2]= 1.4900000000;	data_table[22].feg[4].cnl[2]= 7.2200000000;	data_table[22].feg[4].cl[3]=-1.3100000000;	data_table[22].feg[4].cnl[3]= 15.2000000000;	data_table[22].feg[4].cl[4]= 1.4700000000;	data_table[22].feg[4].cnl[4]= 17.6000000000;
				data_table[22].feg[6].cl[0]= 0.3410000000;	data_table[22].feg[6].cnl[0]= 0.3210000000;	data_table[22].feg[6].cl[1]= 0.8050000000;	data_table[22].feg[6].cnl[1]= 2.2300000000;	data_table[22].feg[6].cl[2]= 0.9420000000;	data_table[22].feg[6].cnl[2]= 5.9900000000;	data_table[22].feg[6].cl[3]= 0.0783000000;	data_table[22].feg[6].cnl[3]= 13.4000000000;	data_table[22].feg[6].cl[4]= 0.1560000000;	data_table[22].feg[6].cnl[4]= 16.9000000000;
				data_table[22].feg[10].cl[0]= 0.0367000000;	data_table[22].feg[10].cnl[0]= 0.0330000000;	data_table[22].feg[10].cl[1]= 0.1240000000;	data_table[22].feg[10].cnl[1]= 0.2220000000;	data_table[22].feg[10].cl[2]= 0.2440000000;	data_table[22].feg[10].cnl[2]= 0.8240000000;	data_table[22].feg[10].cl[3]= 0.7230000000;	data_table[22].feg[10].cnl[3]= 2.8000000000;	data_table[22].feg[10].cl[4]= 0.4350000000;	data_table[22].feg[10].cnl[4]= 6.7000000000;
				data_table[23].feg[4].cl[0]= 0.2370000000;	data_table[23].feg[4].cnl[0]= 0.1770000000;	data_table[23].feg[4].cl[1]= 0.6340000000;	data_table[23].feg[4].cnl[1]= 1.3500000000;	data_table[23].feg[4].cl[2]= 1.2300000000;	data_table[23].feg[4].cnl[2]= 4.3000000000;	data_table[23].feg[4].cl[3]= 0.7130000000;	data_table[23].feg[4].cnl[3]= 12.2000000000;	data_table[23].feg[4].cl[4]= 0.0859000000;	data_table[23].feg[4].cnl[4]= 39.0000000000;
				data_table[23].feg[6].cl[0]= 0.3930000000;	data_table[23].feg[6].cnl[0]= 0.3590000000;	data_table[23].feg[6].cl[1]= 1.0500000000;	data_table[23].feg[6].cnl[1]= 2.5700000000;	data_table[23].feg[6].cl[2]= 1.6200000000;	data_table[23].feg[6].cnl[2]= 8.6800000000;	data_table[23].feg[6].cl[3]=-1.1500000000;	data_table[23].feg[6].cnl[3]= 11.0000000000;	data_table[23].feg[6].cl[4]= 0.4070000000;	data_table[23].feg[6].cnl[4]= 15.8000000000;
				data_table[23].feg[8].cl[0]= 0.1320000000;	data_table[23].feg[8].cnl[0]= 0.1090000000;	data_table[23].feg[8].cl[1]= 0.2920000000;	data_table[23].feg[8].cnl[1]= 0.6950000000;	data_table[23].feg[8].cl[2]= 0.7030000000;	data_table[23].feg[8].cnl[2]= 2.3900000000;	data_table[23].feg[8].cl[3]= 0.6920000000;	data_table[23].feg[8].cnl[3]= 5.6500000000;	data_table[23].feg[8].cl[4]= 0.0959000000;	data_table[23].feg[8].cnl[4]= 14.7000000000;
				data_table[24].feg[4].cl[0]= 0.0576000000;	data_table[24].feg[4].cnl[0]= 0.0398000000;	data_table[24].feg[4].cl[1]= 0.2100000000;	data_table[24].feg[4].cnl[1]= 0.2840000000;	data_table[24].feg[4].cl[2]= 0.6040000000;	data_table[24].feg[4].cnl[2]= 1.2900000000;	data_table[24].feg[4].cl[3]= 1.3200000000;	data_table[24].feg[4].cnl[3]= 4.2300000000;	data_table[24].feg[4].cl[4]= 0.6590000000;	data_table[24].feg[4].cnl[4]= 14.5000000000;
				data_table[24].feg[6].cl[0]= 0.1160000000;	data_table[24].feg[6].cnl[0]= 0.0117000000;	data_table[24].feg[6].cl[1]= 0.5230000000;	data_table[24].feg[6].cnl[1]= 0.8760000000;	data_table[24].feg[6].cl[2]= 0.8810000000;	data_table[24].feg[6].cnl[2]= 3.0600000000;	data_table[24].feg[6].cl[3]= 0.5890000000;	data_table[24].feg[6].cnl[3]= 6.4400000000;	data_table[24].feg[6].cl[4]= 0.2140000000;	data_table[24].feg[6].cnl[4]= 14.3000000000;
				data_table[24].feg[8].cl[0]= 0.3810000000;	data_table[24].feg[8].cnl[0]= 0.3540000000;	data_table[24].feg[8].cl[1]= 1.8300000000;	data_table[24].feg[8].cnl[1]= 2.7200000000;	data_table[24].feg[8].cl[2]=-1.3300000000;	data_table[24].feg[8].cnl[2]= 3.4700000000;	data_table[24].feg[8].cl[3]= 0.9950000000;	data_table[24].feg[8].cnl[3]= 5.4700000000;	data_table[24].feg[8].cl[4]= 0.0618000000;	data_table[24].feg[8].cnl[4]= 16.1000000000;
				data_table[25].feg[4].cl[0]= 0.3070000000;	data_table[25].feg[4].cnl[0]= 0.2300000000;	data_table[25].feg[4].cl[1]= 0.8380000000;	data_table[25].feg[4].cnl[1]= 1.6200000000;	data_table[25].feg[4].cl[2]= 1.1100000000;	data_table[25].feg[4].cnl[2]= 4.8700000000;	data_table[25].feg[4].cl[3]= 0.2800000000;	data_table[25].feg[4].cnl[3]= 10.7000000000;	data_table[25].feg[4].cl[4]= 0.2770000000;	data_table[25].feg[4].cnl[4]= 19.2000000000;
				data_table[25].feg[6].cl[0]= 0.1980000000;	data_table[25].feg[6].cnl[0]= 0.1540000000;	data_table[25].feg[6].cl[1]= 0.3870000000;	data_table[25].feg[6].cnl[1]= 0.8930000000;	data_table[25].feg[6].cl[2]= 0.8890000000;	data_table[25].feg[6].cnl[2]= 2.6200000000;	data_table[25].feg[6].cl[3]= 0.7090000000;	data_table[25].feg[6].cnl[3]= 6.6500000000;	data_table[25].feg[6].cl[4]= 0.1170000000;	data_table[25].feg[6].cnl[4]= 18.0000000000;
				data_table[26].feg[4].cl[0]= 0.2130000000;	data_table[26].feg[4].cnl[0]= 0.1480000000;	data_table[26].feg[4].cl[1]= 0.4880000000;	data_table[26].feg[4].cnl[1]= 0.9390000000;	data_table[26].feg[4].cl[2]= 0.9980000000;	data_table[26].feg[4].cnl[2]= 2.7800000000;	data_table[26].feg[4].cl[3]= 0.8280000000;	data_table[26].feg[4].cnl[3]= 7.3100000000;	data_table[26].feg[4].cl[4]= 0.2300000000;	data_table[26].feg[4].cnl[4]= 20.7000000000;
				data_table[26].feg[6].cl[0]= 0.3310000000;	data_table[26].feg[6].cnl[0]= 0.2670000000;	data_table[26].feg[6].cl[1]= 0.4870000000;	data_table[26].feg[6].cnl[1]= 1.4100000000;	data_table[26].feg[6].cl[2]= 0.7290000000;	data_table[26].feg[6].cnl[2]= 2.8900000000;	data_table[26].feg[6].cl[3]= 0.6080000000;	data_table[26].feg[6].cnl[3]= 6.4500000000;	data_table[26].feg[6].cl[4]= 0.1310000000;	data_table[26].feg[6].cnl[4]= 15.8000000000;
				data_table[27].feg[4].cl[0]= 0.3380000000;	data_table[27].feg[4].cnl[0]= 0.2370000000;	data_table[27].feg[4].cl[1]= 0.9820000000;	data_table[27].feg[4].cnl[1]= 1.6700000000;	data_table[27].feg[4].cl[2]= 1.3200000000;	data_table[27].feg[4].cnl[2]= 5.7300000000;	data_table[27].feg[4].cl[3]=-3.5600000000;	data_table[27].feg[4].cnl[3]= 11.4000000000;	data_table[27].feg[4].cl[4]= 3.6200000000;	data_table[27].feg[4].cnl[4]= 12.1000000000;
				data_table[27].feg[6].cl[0]= 0.3470000000;	data_table[27].feg[6].cnl[0]= 0.2600000000;	data_table[27].feg[6].cl[1]= 0.8770000000;	data_table[27].feg[6].cnl[1]= 1.7100000000;	data_table[27].feg[6].cl[2]= 0.7900000000;	data_table[27].feg[6].cnl[2]= 4.7500000000;	data_table[27].feg[6].cl[3]= 0.0538000000;	data_table[27].feg[6].cnl[3]= 7.5100000000;	data_table[27].feg[6].cl[4]= 0.1920000000;	data_table[27].feg[6].cnl[4]= 13.0000000000;
				data_table[28].feg[2].cl[0]= 0.3120000000;	data_table[28].feg[2].cnl[0]= 0.2010000000;	data_table[28].feg[2].cl[1]= 0.8120000000;	data_table[28].feg[2].cnl[1]= 1.3100000000;	data_table[28].feg[2].cl[2]= 1.1100000000;	data_table[28].feg[2].cnl[2]= 3.8000000000;	data_table[28].feg[2].cl[3]= 0.7940000000;	data_table[28].feg[2].cnl[3]= 10.5000000000;	data_table[28].feg[2].cl[4]= 0.2570000000;	data_table[28].feg[2].cnl[4]= 28.2000000000;
				data_table[28].feg[4].cl[0]= 0.2240000000;	data_table[28].feg[4].cnl[0]= 0.1450000000;	data_table[28].feg[4].cl[1]= 0.5440000000;	data_table[28].feg[4].cnl[1]= 0.9330000000;	data_table[28].feg[4].cl[2]= 0.9700000000;	data_table[28].feg[4].cnl[2]= 2.6900000000;	data_table[28].feg[4].cl[3]= 0.7270000000;	data_table[28].feg[4].cnl[3]= 7.1100000000;	data_table[28].feg[4].cl[4]= 0.1820000000;	data_table[28].feg[4].cnl[4]= 19.4000000000;
				data_table[29].feg[4].cl[0]= 0.2520000000;	data_table[29].feg[4].cnl[0]= 0.1610000000;	data_table[29].feg[4].cl[1]= 0.6000000000;	data_table[29].feg[4].cnl[1]= 1.0100000000;	data_table[29].feg[4].cl[2]= 0.9170000000;	data_table[29].feg[4].cnl[2]= 2.7600000000;	data_table[29].feg[4].cl[3]= 0.6630000000;	data_table[29].feg[4].cnl[3]= 7.0800000000;	data_table[29].feg[4].cl[4]= 0.1610000000;	data_table[29].feg[4].cnl[4]= 19.0000000000;
				data_table[30].feg[6].cl[0]= 0.3910000000;	data_table[30].feg[6].cnl[0]= 0.2640000000;	data_table[30].feg[6].cl[1]= 0.9470000000;	data_table[30].feg[6].cnl[1]= 1.6500000000;	data_table[30].feg[6].cl[2]= 0.6900000000;	data_table[30].feg[6].cnl[2]= 4.8200000000;	data_table[30].feg[6].cl[3]= 0.0709000000;	data_table[30].feg[6].cnl[3]= 10.7000000000;	data_table[30].feg[6].cl[4]= 0.0653000000;	data_table[30].feg[6].cnl[4]= 15.2000000000;
				data_table[31].feg[8].cl[0]= 0.3460000000;	data_table[31].feg[8].cnl[0]= 0.2320000000;	data_table[31].feg[8].cl[1]= 0.8300000000;	data_table[31].feg[8].cnl[1]= 1.4500000000;	data_table[31].feg[8].cl[2]= 0.5990000000;	data_table[31].feg[8].cnl[2]= 4.0800000000;	data_table[31].feg[8].cl[3]= 0.0949000000;	data_table[31].feg[8].cnl[3]= 13.2000000000;	data_table[31].feg[8].cl[4]=-0.0217000000;	data_table[31].feg[8].cnl[4]= 29.5000000000;
				data_table[34].feg[1].cl[0]= 0.1250000000;	data_table[34].feg[1].cnl[0]= 0.0530000000;	data_table[34].feg[1].cl[1]= 0.5630000000;	data_table[34].feg[1].cnl[1]= 0.4690000000;	data_table[34].feg[1].cl[2]= 1.4300000000;	data_table[34].feg[1].cnl[2]= 2.1500000000;	data_table[34].feg[1].cl[3]= 3.5200000000;	data_table[34].feg[1].cnl[3]= 11.1000000000;	data_table[34].feg[1].cl[4]= 3.2200000000;	data_table[34].feg[1].cnl[4]= 38.9000000000;
				data_table[36].feg[2].cl[0]= 0.3680000000;	data_table[36].feg[2].cnl[0]= 0.1870000000;	data_table[36].feg[2].cl[1]= 0.8840000000;	data_table[36].feg[2].cnl[1]= 1.1200000000;	data_table[36].feg[2].cl[2]= 1.1400000000;	data_table[36].feg[2].cnl[2]= 3.9800000000;	data_table[36].feg[2].cl[3]= 2.2600000000;	data_table[36].feg[2].cnl[3]= 10.9000000000;	data_table[36].feg[2].cl[4]= 0.8810000000;	data_table[36].feg[2].cnl[4]= 26.6000000000;
				data_table[37].feg[4].cl[0]= 0.3460000000;	data_table[37].feg[4].cnl[0]= 0.1760000000;	data_table[37].feg[4].cl[1]= 0.8040000000;	data_table[37].feg[4].cnl[1]= 1.0400000000;	data_table[37].feg[4].cl[2]= 0.9880000000;	data_table[37].feg[4].cnl[2]= 3.5900000000;	data_table[37].feg[4].cl[3]= 1.8900000000;	data_table[37].feg[4].cnl[3]= 9.3200000000;	data_table[37].feg[4].cl[4]= 0.6090000000;	data_table[37].feg[4].cnl[4]= 21.4000000000;
				data_table[38].feg[6].cl[0]= 0.4650000000;	data_table[38].feg[6].cnl[0]= 0.2400000000;	data_table[38].feg[6].cl[1]= 0.9230000000;	data_table[38].feg[6].cnl[1]= 1.4300000000;	data_table[38].feg[6].cl[2]= 2.4100000000;	data_table[38].feg[6].cnl[2]= 6.4500000000;	data_table[38].feg[6].cl[3]=-2.3100000000;	data_table[38].feg[6].cnl[3]= 9.9700000000;	data_table[38].feg[6].cl[4]= 2.4800000000;	data_table[38].feg[6].cnl[4]= 12.2000000000;
				data_table[39].feg[8].cl[0]= 0.2340000000;	data_table[39].feg[8].cnl[0]= 0.1130000000;	data_table[39].feg[8].cl[1]= 0.6420000000;	data_table[39].feg[8].cnl[1]= 0.7360000000;	data_table[39].feg[8].cl[2]= 0.7470000000;	data_table[39].feg[8].cnl[2]= 2.5400000000;	data_table[39].feg[8].cl[3]= 1.4700000000;	data_table[39].feg[8].cnl[3]= 6.7200000000;	data_table[39].feg[8].cl[4]= 0.3770000000;	data_table[39].feg[8].cnl[4]= 14.7000000000;
				data_table[40].feg[6].cl[0]= 0.3770000000;	data_table[40].feg[6].cnl[0]= 0.1840000000;	data_table[40].feg[6].cl[1]= 0.7490000000;	data_table[40].feg[6].cnl[1]= 1.0200000000;	data_table[40].feg[6].cl[2]= 1.2900000000;	data_table[40].feg[6].cnl[2]= 3.8000000000;	data_table[40].feg[6].cl[3]= 1.6100000000;	data_table[40].feg[6].cnl[3]= 9.4400000000;	data_table[40].feg[6].cl[4]= 0.4810000000;	data_table[40].feg[6].cnl[4]= 25.7000000000;
				data_table[40].feg[10].cl[0]= 0.0828000000;	data_table[40].feg[10].cnl[0]= 0.0369000000;	data_table[40].feg[10].cl[1]= 0.2710000000;	data_table[40].feg[10].cnl[1]= 0.2610000000;	data_table[40].feg[10].cl[2]= 0.6540000000;	data_table[40].feg[10].cnl[2]= 0.9570000000;	data_table[40].feg[10].cl[3]= 1.2400000000;	data_table[40].feg[10].cnl[3]= 3.9400000000;	data_table[40].feg[10].cl[4]= 0.8290000000;	data_table[40].feg[10].cnl[4]= 9.4400000000;
				data_table[41].feg[6].cl[0]= 0.4010000000;	data_table[41].feg[6].cnl[0]= 0.1910000000;	data_table[41].feg[6].cl[1]= 0.7560000000;	data_table[41].feg[6].cnl[1]= 1.0600000000;	data_table[41].feg[6].cl[2]= 1.3800000000;	data_table[41].feg[6].cnl[2]= 3.8400000000;	data_table[41].feg[6].cl[3]= 1.5800000000;	data_table[41].feg[6].cnl[3]= 9.3800000000;	data_table[41].feg[6].cl[4]= 0.4970000000;	data_table[41].feg[6].cnl[4]= 24.6000000000;
				data_table[41].feg[10].cl[0]= 0.4790000000;	data_table[41].feg[10].cnl[0]= 0.2410000000;	data_table[41].feg[10].cl[1]= 0.8460000000;	data_table[41].feg[10].cnl[1]= 1.4600000000;	data_table[41].feg[10].cl[2]= 15.6000000000;	data_table[41].feg[10].cnl[2]= 6.7900000000;	data_table[41].feg[10].cl[3]=-15.2000000000;	data_table[41].feg[10].cnl[3]= 7.1300000000;	data_table[41].feg[10].cl[4]= 1.6000000000;	data_table[41].feg[10].cnl[4]= 10.4000000000;
				data_table[41].feg[12].cl[0]= 0.2030000000;	data_table[41].feg[12].cnl[0]= 0.0971000000;	data_table[41].feg[12].cl[1]= 0.5670000000;	data_table[41].feg[12].cnl[1]= 0.6470000000;	data_table[41].feg[12].cl[2]= 0.6460000000;	data_table[41].feg[12].cnl[2]= 2.2800000000;	data_table[41].feg[12].cl[3]= 1.1600000000;	data_table[41].feg[12].cnl[3]= 5.6100000000;	data_table[41].feg[12].cl[4]= 0.1710000000;	data_table[41].feg[12].cnl[4]= 12.4000000000;
				data_table[43].feg[6].cl[0]= 0.4280000000;	data_table[43].feg[6].cnl[0]= 0.1910000000;	data_table[43].feg[6].cl[1]= 0.7730000000;	data_table[43].feg[6].cnl[1]= 1.0900000000;	data_table[43].feg[6].cl[2]= 1.5500000000;	data_table[43].feg[6].cnl[2]= 3.8200000000;	data_table[43].feg[6].cl[3]= 1.4600000000;	data_table[43].feg[6].cnl[3]= 9.0800000000;	data_table[43].feg[6].cl[4]= 0.4860000000;	data_table[43].feg[6].cnl[4]= 21.7000000000;
				data_table[43].feg[8].cl[0]= 0.2820000000;	data_table[43].feg[8].cnl[0]= 0.1250000000;	data_table[43].feg[8].cl[1]= 0.6530000000;	data_table[43].feg[8].cnl[1]= 0.7530000000;	data_table[43].feg[8].cl[2]= 1.1400000000;	data_table[43].feg[8].cnl[2]= 2.8500000000;	data_table[43].feg[8].cl[3]= 1.5300000000;	data_table[43].feg[8].cnl[3]= 7.0100000000;	data_table[43].feg[8].cl[4]= 0.4180000000;	data_table[43].feg[8].cnl[4]= 17.5000000000;
				data_table[44].feg[6].cl[0]= 0.3520000000;	data_table[44].feg[6].cnl[0]= 0.1510000000;	data_table[44].feg[6].cl[1]= 0.7230000000;	data_table[44].feg[6].cnl[1]= 0.8780000000;	data_table[44].feg[6].cl[2]= 1.5000000000;	data_table[44].feg[6].cnl[2]= 3.2800000000;	data_table[44].feg[6].cl[3]= 1.6300000000;	data_table[44].feg[6].cnl[3]= 8.1600000000;	data_table[44].feg[6].cl[4]= 0.4990000000;	data_table[44].feg[6].cnl[4]= 20.7000000000;
				data_table[44].feg[8].cl[0]= 0.3970000000;	data_table[44].feg[8].cnl[0]= 0.1770000000;	data_table[44].feg[8].cl[1]= 0.7250000000;	data_table[44].feg[8].cnl[1]= 1.0100000000;	data_table[44].feg[8].cl[2]= 1.5100000000;	data_table[44].feg[8].cnl[2]= 3.6200000000;	data_table[44].feg[8].cl[3]= 1.1900000000;	data_table[44].feg[8].cnl[3]= 8.5600000000;	data_table[44].feg[8].cl[4]= 0.2510000000;	data_table[44].feg[8].cnl[4]= 18.9000000000;
				data_table[45].feg[4].cl[0]= 0.9350000000;	data_table[45].feg[4].cnl[0]= 0.3930000000;	data_table[45].feg[4].cl[1]= 3.1100000000;	data_table[45].feg[4].cnl[1]= 4.0600000000;	data_table[45].feg[4].cl[2]= 24.6000000000;	data_table[45].feg[4].cnl[2]= 43.1000000000;	data_table[45].feg[4].cl[3]=-43.6000000000;	data_table[45].feg[4].cnl[3]= 54.0000000000;	data_table[45].feg[4].cl[4]= 21.1000000000;	data_table[45].feg[4].cnl[4]= 69.8000000000;
				data_table[45].feg[8].cl[0]= 0.3480000000;	data_table[45].feg[8].cnl[0]= 0.1510000000;	data_table[45].feg[8].cl[1]= 0.6400000000;	data_table[45].feg[8].cnl[1]= 0.8320000000;	data_table[45].feg[8].cl[2]= 1.2200000000;	data_table[45].feg[8].cnl[2]= 2.8500000000;	data_table[45].feg[8].cl[3]= 1.4500000000;	data_table[45].feg[8].cnl[3]= 6.5900000000;	data_table[45].feg[8].cl[4]= 0.4270000000;	data_table[45].feg[8].cnl[4]= 15.6000000000;
				data_table[46].feg[2].cl[0]= 0.5030000000;	data_table[46].feg[2].cnl[0]= 0.1990000000;	data_table[46].feg[2].cl[1]= 0.9400000000;	data_table[46].feg[2].cnl[1]= 1.1900000000;	data_table[46].feg[2].cl[2]= 2.1700000000;	data_table[46].feg[2].cnl[2]= 4.0500000000;	data_table[46].feg[2].cl[3]= 1.9900000000;	data_table[46].feg[2].cnl[3]= 11.3000000000;	data_table[46].feg[2].cl[4]= 0.7260000000;	data_table[46].feg[2].cnl[4]= 32.4000000000;
				data_table[46].feg[4].cl[0]= 0.4310000000;	data_table[46].feg[4].cnl[0]= 0.1750000000;	data_table[46].feg[4].cl[1]= 0.7560000000;	data_table[46].feg[4].cnl[1]= 0.9790000000;	data_table[46].feg[4].cl[2]= 1.7200000000;	data_table[46].feg[4].cnl[2]= 3.3000000000;	data_table[46].feg[4].cl[3]= 1.7800000000;	data_table[46].feg[4].cnl[3]= 8.2400000000;	data_table[46].feg[4].cl[4]= 0.5260000000;	data_table[46].feg[4].cnl[4]= 21.4000000000;
				data_table[47].feg[4].cl[0]= 0.4250000000;	data_table[47].feg[4].cnl[0]= 0.1680000000;	data_table[47].feg[4].cl[1]= 0.7450000000;	data_table[47].feg[4].cnl[1]= 0.9440000000;	data_table[47].feg[4].cl[2]= 1.7300000000;	data_table[47].feg[4].cnl[2]= 3.1400000000;	data_table[47].feg[4].cl[3]= 1.7400000000;	data_table[47].feg[4].cnl[3]= 7.8400000000;	data_table[47].feg[4].cl[4]= 0.4870000000;	data_table[47].feg[4].cnl[4]= 20.4000000000;
				data_table[48].feg[6].cl[0]= 0.4170000000;	data_table[48].feg[6].cnl[0]= 0.1640000000;	data_table[48].feg[6].cl[1]= 0.7550000000;	data_table[48].feg[6].cnl[1]= 0.9600000000;	data_table[48].feg[6].cl[2]= 1.5900000000;	data_table[48].feg[6].cnl[2]= 3.0800000000;	data_table[48].feg[6].cl[3]= 1.3600000000;	data_table[48].feg[6].cnl[3]= 7.0300000000;	data_table[48].feg[6].cl[4]= 0.4510000000;	data_table[48].feg[6].cnl[4]= 16.1000000000;
				data_table[49].feg[4].cl[0]= 0.7970000000;	data_table[49].feg[4].cnl[0]= 0.3170000000;	data_table[49].feg[4].cl[1]= 2.1300000000;	data_table[49].feg[4].cnl[1]= 2.5100000000;	data_table[49].feg[4].cl[2]= 2.1500000000;	data_table[49].feg[4].cnl[2]= 9.0400000000;	data_table[49].feg[4].cl[3]=-1.6400000000;	data_table[49].feg[4].cnl[3]= 24.2000000000;	data_table[49].feg[4].cl[4]= 2.7200000000;	data_table[49].feg[4].cnl[4]= 26.4000000000;
				data_table[49].feg[8].cl[0]= 0.2610000000;	data_table[49].feg[8].cnl[0]= 0.0957000000;	data_table[49].feg[8].cl[1]= 0.6420000000;	data_table[49].feg[8].cnl[1]= 0.6250000000;	data_table[49].feg[8].cl[2]= 1.5300000000;	data_table[49].feg[8].cnl[2]= 2.5100000000;	data_table[49].feg[8].cl[3]= 1.3600000000;	data_table[49].feg[8].cnl[3]= 6.3100000000;	data_table[49].feg[8].cl[4]= 0.1770000000;	data_table[49].feg[8].cnl[4]= 15.9000000000;
				data_table[50].feg[6].cl[0]= 0.5520000000;	data_table[50].feg[6].cnl[0]= 0.2120000000;	data_table[50].feg[6].cl[1]= 1.1400000000;	data_table[50].feg[6].cnl[1]= 1.4200000000;	data_table[50].feg[6].cl[2]= 1.8700000000;	data_table[50].feg[6].cnl[2]= 4.2100000000;	data_table[50].feg[6].cl[3]= 1.3600000000;	data_table[50].feg[6].cnl[3]= 12.5000000000;	data_table[50].feg[6].cl[4]= 0.4140000000;	data_table[50].feg[6].cnl[4]= 29.0000000000;
				data_table[50].feg[10].cl[0]= 0.3770000000;	data_table[50].feg[10].cnl[0]= 0.1510000000;	data_table[50].feg[10].cl[1]= 0.5880000000;	data_table[50].feg[10].cnl[1]= 0.8120000000;	data_table[50].feg[10].cl[2]= 1.2200000000;	data_table[50].feg[10].cnl[2]= 2.4000000000;	data_table[50].feg[10].cl[3]= 1.1800000000;	data_table[50].feg[10].cnl[3]= 5.2700000000;	data_table[50].feg[10].cl[4]= 0.2440000000;	data_table[50].feg[10].cnl[4]= 11.9000000000;
				data_table[52].feg[1].cl[0]= 0.9010000000;	data_table[52].feg[1].cnl[0]= 0.3120000000;	data_table[52].feg[1].cl[1]= 2.8000000000;	data_table[52].feg[1].cnl[1]= 2.5900000000;	data_table[52].feg[1].cl[2]= 5.6100000000;	data_table[52].feg[1].cnl[2]= 14.1000000000;	data_table[52].feg[1].cl[3]=-8.6900000000;	data_table[52].feg[1].cnl[3]= 34.4000000000;	data_table[52].feg[1].cl[4]= 12.6000000000;	data_table[52].feg[1].cnl[4]= 39.5000000000;
				data_table[54].feg[2].cl[0]= 0.5870000000;	data_table[54].feg[2].cnl[0]= 0.2000000000;	data_table[54].feg[2].cl[1]= 1.4000000000;	data_table[54].feg[2].cnl[1]= 1.3800000000;	data_table[54].feg[2].cl[2]= 1.8700000000;	data_table[54].feg[2].cnl[2]= 4.1200000000;	data_table[54].feg[2].cl[3]= 3.4800000000;	data_table[54].feg[2].cnl[3]= 13.0000000000;	data_table[54].feg[2].cl[4]= 1.6700000000;	data_table[54].feg[2].cnl[4]= 31.8000000000;
				data_table[55].feg[4].cl[0]= 0.7330000000;	data_table[55].feg[4].cnl[0]= 0.2580000000;	data_table[55].feg[4].cl[1]= 2.0500000000;	data_table[55].feg[4].cnl[1]= 1.9600000000;	data_table[55].feg[4].cl[2]= 23.0000000000;	data_table[55].feg[4].cnl[2]= 11.8000000000;	data_table[55].feg[4].cl[3]=-152.0000000000;	data_table[55].feg[4].cnl[3]= 14.4000000000;	data_table[55].feg[4].cl[4]= 134.0000000000;	data_table[55].feg[4].cnl[4]= 14.9000000000;
				data_table[56].feg[6].cl[0]= 0.4930000000;	data_table[56].feg[6].cnl[0]= 0.1670000000;	data_table[56].feg[6].cl[1]= 1.1000000000;	data_table[56].feg[6].cnl[1]= 1.1100000000;	data_table[56].feg[6].cl[2]= 1.5000000000;	data_table[56].feg[6].cnl[2]= 3.1100000000;	data_table[56].feg[6].cl[3]= 2.7000000000;	data_table[56].feg[6].cnl[3]= 9.6100000000;	data_table[56].feg[6].cl[4]= 1.0800000000;	data_table[56].feg[6].cnl[4]= 21.2000000000;
				data_table[57].feg[6].cl[0]= 0.5600000000;	data_table[57].feg[6].cnl[0]= 0.1900000000;	data_table[57].feg[6].cl[1]= 1.3500000000;	data_table[57].feg[6].cnl[1]= 1.3000000000;	data_table[57].feg[6].cl[2]= 1.5900000000;	data_table[57].feg[6].cnl[2]= 3.9300000000;	data_table[57].feg[6].cl[3]= 2.6300000000;	data_table[57].feg[6].cnl[3]= 10.7000000000;	data_table[57].feg[6].cl[4]= 0.7060000000;	data_table[57].feg[6].cnl[4]= 23.8000000000;
				data_table[57].feg[8].cl[0]= 0.4830000000;	data_table[57].feg[8].cnl[0]= 0.1650000000;	data_table[57].feg[8].cl[1]= 1.0900000000;	data_table[57].feg[8].cnl[1]= 1.1000000000;	data_table[57].feg[8].cl[2]= 1.3400000000;	data_table[57].feg[8].cnl[2]= 3.0200000000;	data_table[57].feg[8].cl[3]= 2.4500000000;	data_table[57].feg[8].cnl[3]= 8.8500000000;	data_table[57].feg[8].cl[4]= 0.7970000000;	data_table[57].feg[8].cnl[4]= 18.8000000000;
				data_table[58].feg[6].cl[0]= 0.6630000000;	data_table[58].feg[6].cnl[0]= 0.2260000000;	data_table[58].feg[6].cl[1]= 1.7300000000;	data_table[58].feg[6].cnl[1]= 1.6100000000;	data_table[58].feg[6].cl[2]= 2.3500000000;	data_table[58].feg[6].cnl[2]= 6.3300000000;	data_table[58].feg[6].cl[3]= 0.3510000000;	data_table[58].feg[6].cnl[3]= 11.0000000000;	data_table[58].feg[6].cl[4]= 1.5900000000;	data_table[58].feg[6].cnl[4]= 16.9000000000;
				data_table[58].feg[8].cl[0]= 0.5210000000;	data_table[58].feg[8].cnl[0]= 0.1770000000;	data_table[58].feg[8].cl[1]= 1.1900000000;	data_table[58].feg[8].cnl[1]= 1.1700000000;	data_table[58].feg[8].cl[2]= 1.3300000000;	data_table[58].feg[8].cnl[2]= 3.2800000000;	data_table[58].feg[8].cl[3]= 2.3600000000;	data_table[58].feg[8].cnl[3]= 8.9400000000;	data_table[58].feg[8].cl[4]= 0.6900000000;	data_table[58].feg[8].cnl[4]= 19.3000000000;
				data_table[59].feg[6].cl[0]= 0.5010000000;	data_table[59].feg[6].cnl[0]= 0.1620000000;	data_table[59].feg[6].cl[1]= 1.1800000000;	data_table[59].feg[6].cnl[1]= 1.0800000000;	data_table[59].feg[6].cl[2]= 1.4500000000;	data_table[59].feg[6].cnl[2]= 3.0600000000;	data_table[59].feg[6].cl[3]= 2.5300000000;	data_table[59].feg[6].cnl[3]= 8.8000000000;	data_table[59].feg[6].cl[4]= 0.9200000000;	data_table[59].feg[6].cnl[4]= 19.6000000000;
				data_table[60].feg[6].cl[0]= 0.4960000000;	data_table[60].feg[6].cnl[0]= 0.1560000000;	data_table[60].feg[6].cl[1]= 1.2000000000;	data_table[60].feg[6].cnl[1]= 1.0500000000;	data_table[60].feg[6].cl[2]= 1.4700000000;	data_table[60].feg[6].cnl[2]= 3.0700000000;	data_table[60].feg[6].cl[3]= 2.4300000000;	data_table[60].feg[6].cnl[3]= 8.5600000000;	data_table[60].feg[6].cl[4]= 0.9430000000;	data_table[60].feg[6].cnl[4]= 19.2000000000;
				data_table[61].feg[6].cl[0]= 0.5180000000;	data_table[61].feg[6].cnl[0]= 0.1630000000;	data_table[61].feg[6].cl[1]= 1.2400000000;	data_table[61].feg[6].cnl[1]= 1.0800000000;	data_table[61].feg[6].cl[2]= 1.4300000000;	data_table[61].feg[6].cnl[2]= 3.1100000000;	data_table[61].feg[6].cl[3]= 2.4000000000;	data_table[61].feg[6].cnl[3]= 8.5200000000;	data_table[61].feg[6].cl[4]= 0.7810000000;	data_table[61].feg[6].cnl[4]= 19.1000000000;
				data_table[62].feg[4].cl[0]= 0.6130000000;	data_table[62].feg[4].cnl[0]= 0.1900000000;	data_table[62].feg[4].cl[1]= 1.5300000000;	data_table[62].feg[4].cnl[1]= 1.2700000000;	data_table[62].feg[4].cl[2]= 1.8400000000;	data_table[62].feg[4].cnl[2]= 4.1800000000;	data_table[62].feg[4].cl[3]= 2.4600000000;	data_table[62].feg[4].cnl[3]= 10.7000000000;	data_table[62].feg[4].cl[4]= 0.7140000000;	data_table[62].feg[4].cnl[4]= 26.2000000000;
				data_table[62].feg[6].cl[0]= 0.4960000000;	data_table[62].feg[6].cnl[0]= 0.1520000000;	data_table[62].feg[6].cl[1]= 1.2100000000;	data_table[62].feg[6].cnl[1]= 1.0100000000;	data_table[62].feg[6].cl[2]= 1.4500000000;	data_table[62].feg[6].cnl[2]= 2.9500000000;	data_table[62].feg[6].cl[3]= 2.3600000000;	data_table[62].feg[6].cnl[3]= 8.1800000000;	data_table[62].feg[6].cl[4]= 0.7740000000;	data_table[62].feg[6].cnl[4]= 18.5000000000;
				data_table[63].feg[6].cl[0]= 0.4900000000;	data_table[63].feg[6].cnl[0]= 0.1480000000;	data_table[63].feg[6].cl[1]= 1.1900000000;	data_table[63].feg[6].cnl[1]= 0.9740000000;	data_table[63].feg[6].cl[2]= 1.4200000000;	data_table[63].feg[6].cnl[2]= 2.8100000000;	data_table[63].feg[6].cl[3]= 2.3000000000;	data_table[63].feg[6].cnl[3]= 7.7800000000;	data_table[63].feg[6].cl[4]= 0.7950000000;	data_table[63].feg[6].cnl[4]= 17.7000000000;
				data_table[64].feg[6].cl[0]= 0.5030000000;	data_table[64].feg[6].cnl[0]= 0.1500000000;	data_table[64].feg[6].cl[1]= 1.2200000000;	data_table[64].feg[6].cnl[1]= 0.9820000000;	data_table[64].feg[6].cl[2]= 1.4200000000;	data_table[64].feg[6].cnl[2]= 2.8600000000;	data_table[64].feg[6].cl[3]= 2.2400000000;	data_table[64].feg[6].cnl[3]= 7.7700000000;	data_table[64].feg[6].cl[4]= 0.7100000000;	data_table[64].feg[6].cnl[4]= 17.7000000000;
				data_table[65].feg[6].cl[0]= 0.5030000000;	data_table[65].feg[6].cnl[0]= 0.1480000000;	data_table[65].feg[6].cl[1]= 1.2400000000;	data_table[65].feg[6].cnl[1]= 0.9700000000;	data_table[65].feg[6].cl[2]= 1.4400000000;	data_table[65].feg[6].cnl[2]= 2.8800000000;	data_table[65].feg[6].cl[3]= 2.1700000000;	data_table[65].feg[6].cnl[3]= 7.7300000000;	data_table[65].feg[6].cl[4]= 0.6430000000;	data_table[65].feg[6].cnl[4]= 17.6000000000;
				data_table[66].feg[6].cl[0]= 0.4560000000;	data_table[66].feg[6].cnl[0]= 0.1290000000;	data_table[66].feg[6].cl[1]= 1.1700000000;	data_table[66].feg[6].cnl[1]= 0.8690000000;	data_table[66].feg[6].cl[2]= 1.4300000000;	data_table[66].feg[6].cnl[2]= 2.6100000000;	data_table[66].feg[6].cl[3]= 2.1500000000;	data_table[66].feg[6].cnl[3]= 7.2400000000;	data_table[66].feg[6].cl[4]= 0.6920000000;	data_table[66].feg[6].cnl[4]= 16.7000000000;
				data_table[67].feg[6].cl[0]= 0.5220000000;	data_table[67].feg[6].cnl[0]= 0.1500000000;	data_table[67].feg[6].cl[1]= 1.2800000000;	data_table[67].feg[6].cnl[1]= 0.9640000000;	data_table[67].feg[6].cl[2]= 1.4600000000;	data_table[67].feg[6].cnl[2]= 2.9300000000;	data_table[67].feg[6].cl[3]= 2.0500000000;	data_table[67].feg[6].cnl[3]= 7.7200000000;	data_table[67].feg[6].cl[4]= 0.5080000000;	data_table[67].feg[6].cnl[4]= 17.8000000000;
				data_table[68].feg[6].cl[0]= 0.4750000000;	data_table[68].feg[6].cnl[0]= 0.1320000000;	data_table[68].feg[6].cl[1]= 1.2000000000;	data_table[68].feg[6].cnl[1]= 0.8640000000;	data_table[68].feg[6].cl[2]= 1.4200000000;	data_table[68].feg[6].cnl[2]= 2.6000000000;	data_table[68].feg[6].cl[3]= 2.0500000000;	data_table[68].feg[6].cnl[3]= 7.0900000000;	data_table[68].feg[6].cl[4]= 0.5840000000;	data_table[68].feg[6].cnl[4]= 16.6000000000;
				data_table[69].feg[4].cl[0]= 0.5080000000;	data_table[69].feg[4].cnl[0]= 0.1360000000;	data_table[69].feg[4].cl[1]= 1.3700000000;	data_table[69].feg[4].cnl[1]= 0.9220000000;	data_table[69].feg[4].cl[2]= 1.7600000000;	data_table[69].feg[4].cnl[2]= 3.1200000000;	data_table[69].feg[4].cl[3]= 2.2300000000;	data_table[69].feg[4].cnl[3]= 8.7200000000;	data_table[69].feg[4].cl[4]= 0.5840000000;	data_table[69].feg[4].cnl[4]= 23.7000000000;
				data_table[69].feg[6].cl[0]= 0.4980000000;	data_table[69].feg[6].cnl[0]= 0.1380000000;	data_table[69].feg[6].cl[1]= 1.2200000000;	data_table[69].feg[6].cnl[1]= 0.8810000000;	data_table[69].feg[6].cl[2]= 1.3900000000;	data_table[69].feg[6].cnl[2]= 2.6300000000;	data_table[69].feg[6].cl[3]= 1.9700000000;	data_table[69].feg[6].cnl[3]= 6.9900000000;	data_table[69].feg[6].cl[4]= 0.5590000000;	data_table[69].feg[6].cnl[4]= 16.3000000000;
				data_table[70].feg[6].cl[0]= 0.4830000000;	data_table[70].feg[6].cnl[0]= 0.1310000000;	data_table[70].feg[6].cl[1]= 1.2100000000;	data_table[70].feg[6].cnl[1]= 0.8450000000;	data_table[70].feg[6].cl[2]= 1.4100000000;	data_table[70].feg[6].cnl[2]= 2.5700000000;	data_table[70].feg[6].cl[3]= 1.9400000000;	data_table[70].feg[6].cnl[3]= 6.8800000000;	data_table[70].feg[6].cl[4]= 0.5220000000;	data_table[70].feg[6].cnl[4]= 16.2000000000;
				data_table[71].feg[8].cl[0]= 0.5220000000;	data_table[71].feg[8].cnl[0]= 0.1450000000;	data_table[71].feg[8].cl[1]= 1.2200000000;	data_table[71].feg[8].cnl[1]= 0.8960000000;	data_table[71].feg[8].cl[2]= 1.3700000000;	data_table[71].feg[8].cnl[2]= 2.7400000000;	data_table[71].feg[8].cl[3]= 1.6800000000;	data_table[71].feg[8].cnl[3]= 6.9100000000;	data_table[71].feg[8].cl[4]= 0.3120000000;	data_table[71].feg[8].cnl[4]= 16.1000000000;
				data_table[72].feg[10].cl[0]= 0.5690000000;	data_table[72].feg[10].cnl[0]= 0.1610000000;	data_table[72].feg[10].cl[1]= 1.2600000000;	data_table[72].feg[10].cnl[1]= 0.9720000000;	data_table[72].feg[10].cl[2]= 0.9790000000;	data_table[72].feg[10].cnl[2]= 2.7600000000;	data_table[72].feg[10].cl[3]= 1.2900000000;	data_table[72].feg[10].cnl[3]= 5.4000000000;	data_table[72].feg[10].cl[4]= 0.5510000000;	data_table[72].feg[10].cnl[4]= 10.9000000000;
				data_table[73].feg[12].cl[0]= 0.1810000000;	data_table[73].feg[12].cnl[0]= 0.0118000000;	data_table[73].feg[12].cl[1]= 0.8730000000;	data_table[73].feg[12].cnl[1]= 0.4420000000;	data_table[73].feg[12].cl[2]= 1.1800000000;	data_table[73].feg[12].cnl[2]= 1.5200000000;	data_table[73].feg[12].cl[3]= 1.4800000000;	data_table[73].feg[12].cnl[3]= 4.3500000000;	data_table[73].feg[12].cl[4]= 0.5620000000;	data_table[73].feg[12].cnl[4]= 9.4200000000;
				data_table[75].feg[8].cl[0]= 0.5860000000;	data_table[75].feg[8].cnl[0]= 0.1550000000;	data_table[75].feg[8].cl[1]= 1.3100000000;	data_table[75].feg[8].cnl[1]= 0.9380000000;	data_table[75].feg[8].cl[2]= 1.6300000000;	data_table[75].feg[8].cnl[2]= 3.1900000000;	data_table[75].feg[8].cl[3]= 1.7100000000;	data_table[75].feg[8].cnl[3]= 7.8400000000;	data_table[75].feg[8].cl[4]= 0.5400000000;	data_table[75].feg[8].cnl[4]= 19.3000000000;
				data_table[76].feg[6].cl[0]= 0.6920000000;	data_table[76].feg[6].cnl[0]= 0.1820000000;	data_table[76].feg[6].cl[1]= 1.3700000000;	data_table[76].feg[6].cnl[1]= 1.0400000000;	data_table[76].feg[6].cl[2]= 1.8000000000;	data_table[76].feg[6].cnl[2]= 3.4700000000;	data_table[76].feg[6].cl[3]= 1.9700000000;	data_table[76].feg[6].cnl[3]= 8.5100000000;	data_table[76].feg[6].cl[4]= 0.8040000000;	data_table[76].feg[6].cnl[4]= 21.2000000000;
				data_table[76].feg[8].cl[0]= 0.6530000000;	data_table[76].feg[8].cnl[0]= 0.1740000000;	data_table[76].feg[8].cl[1]= 1.2900000000;	data_table[76].feg[8].cnl[1]= 0.9920000000;	data_table[76].feg[8].cl[2]= 1.5000000000;	data_table[76].feg[8].cnl[2]= 3.1400000000;	data_table[76].feg[8].cl[3]= 1.7400000000;	data_table[76].feg[8].cnl[3]= 7.2200000000;	data_table[76].feg[8].cl[4]= 0.6830000000;	data_table[76].feg[8].cnl[4]= 17.2000000000;
				data_table[77].feg[4].cl[0]= 0.8720000000;	data_table[77].feg[4].cnl[0]= 0.2230000000;	data_table[77].feg[4].cl[1]= 1.6800000000;	data_table[77].feg[4].cnl[1]= 1.3500000000;	data_table[77].feg[4].cl[2]= 2.6300000000;	data_table[77].feg[4].cnl[2]= 4.9900000000;	data_table[77].feg[4].cl[3]= 1.9300000000;	data_table[77].feg[4].cnl[3]= 13.6000000000;	data_table[77].feg[4].cl[4]= 0.4750000000;	data_table[77].feg[4].cnl[4]= 33.0000000000;
				data_table[77].feg[8].cl[0]= 0.5500000000;	data_table[77].feg[8].cnl[0]= 0.1420000000;	data_table[77].feg[8].cl[1]= 1.2100000000;	data_table[77].feg[8].cnl[1]= 0.8330000000;	data_table[77].feg[8].cl[2]= 1.6200000000;	data_table[77].feg[8].cnl[2]= 2.8100000000;	data_table[77].feg[8].cl[3]= 1.9500000000;	data_table[77].feg[8].cnl[3]= 7.2100000000;	data_table[77].feg[8].cl[4]= 0.6100000000;	data_table[77].feg[8].cnl[4]= 17.7000000000;
				data_table[78].feg[2].cl[0]= 0.8110000000;	data_table[78].feg[2].cnl[0]= 0.2010000000;	data_table[78].feg[2].cl[1]= 1.5700000000;	data_table[78].feg[2].cnl[1]= 1.1800000000;	data_table[78].feg[2].cl[2]= 2.6300000000;	data_table[78].feg[2].cnl[2]= 4.2500000000;	data_table[78].feg[2].cl[3]= 2.6800000000;	data_table[78].feg[2].cnl[3]= 12.1000000000;	data_table[78].feg[2].cl[4]= 0.9980000000;	data_table[78].feg[2].cnl[4]= 34.4000000000;
				data_table[78].feg[6].cl[0]= 0.7220000000;	data_table[78].feg[6].cnl[0]= 0.1840000000;	data_table[78].feg[6].cl[1]= 1.3900000000;	data_table[78].feg[6].cnl[1]= 1.0600000000;	data_table[78].feg[6].cl[2]= 1.9400000000;	data_table[78].feg[6].cnl[2]= 3.5800000000;	data_table[78].feg[6].cl[3]= 1.9400000000;	data_table[78].feg[6].cnl[3]= 8.5600000000;	data_table[78].feg[6].cl[4]= 0.6990000000;	data_table[78].feg[6].cnl[4]= 20.4000000000;
				data_table[79].feg[2].cl[0]= 0.7960000000;	data_table[79].feg[2].cnl[0]= 0.1940000000;	data_table[79].feg[2].cl[1]= 1.5600000000;	data_table[79].feg[2].cnl[1]= 1.1400000000;	data_table[79].feg[2].cl[2]= 2.7200000000;	data_table[79].feg[2].cnl[2]= 4.2100000000;	data_table[79].feg[2].cl[3]= 2.7600000000;	data_table[79].feg[2].cnl[3]= 12.4000000000;	data_table[79].feg[2].cl[4]= 1.1800000000;	data_table[79].feg[2].cnl[4]= 36.2000000000;
				data_table[79].feg[4].cl[0]= 0.7730000000;	data_table[79].feg[4].cnl[0]= 0.1910000000;	data_table[79].feg[4].cl[1]= 1.4900000000;	data_table[79].feg[4].cnl[1]= 1.1200000000;	data_table[79].feg[4].cl[2]= 2.4500000000;	data_table[79].feg[4].cnl[2]= 4.0000000000;	data_table[79].feg[4].cl[3]= 2.2300000000;	data_table[79].feg[4].cnl[3]= 10.8000000000;	data_table[79].feg[4].cl[4]= 0.5700000000;	data_table[79].feg[4].cnl[4]= 27.6000000000;
				data_table[80].feg[2].cl[0]= 0.8200000000;	data_table[80].feg[2].cnl[0]= 0.1970000000;	data_table[80].feg[2].cl[1]= 1.5700000000;	data_table[80].feg[2].cnl[1]= 1.1600000000;	data_table[80].feg[2].cl[2]= 2.7800000000;	data_table[80].feg[2].cnl[2]= 4.2300000000;	data_table[80].feg[2].cl[3]= 2.8200000000;	data_table[80].feg[2].cnl[3]= 12.7000000000;	data_table[80].feg[2].cl[4]= 1.3100000000;	data_table[80].feg[2].cnl[4]= 35.7000000000;
				data_table[80].feg[6].cl[0]= 0.8360000000;	data_table[80].feg[6].cnl[0]= 0.2080000000;	data_table[80].feg[6].cl[1]= 1.4300000000;	data_table[80].feg[6].cnl[1]= 1.2000000000;	data_table[80].feg[6].cl[2]= 0.3940000000;	data_table[80].feg[6].cnl[2]= 2.5700000000;	data_table[80].feg[6].cl[3]= 2.5100000000;	data_table[80].feg[6].cnl[3]= 4.8600000000;	data_table[80].feg[6].cl[4]= 1.5000000000;	data_table[80].feg[6].cnl[4]= 13.5000000000;
				data_table[81].feg[4].cl[0]= 0.7550000000;	data_table[81].feg[4].cnl[0]= 0.1810000000;	data_table[81].feg[4].cl[1]= 1.4400000000;	data_table[81].feg[4].cnl[1]= 1.0500000000;	data_table[81].feg[4].cl[2]= 2.4800000000;	data_table[81].feg[4].cnl[2]= 3.7500000000;	data_table[81].feg[4].cl[3]= 2.4500000000;	data_table[81].feg[4].cnl[3]= 10.6000000000;	data_table[81].feg[4].cl[4]= 1.0300000000;	data_table[81].feg[4].cnl[4]= 27.9000000000;
				data_table[81].feg[8].cl[0]= 0.5830000000;	data_table[81].feg[8].cnl[0]= 0.1440000000;	data_table[81].feg[8].cl[1]= 1.1400000000;	data_table[81].feg[8].cnl[1]= 0.7960000000;	data_table[81].feg[8].cl[2]= 1.6000000000;	data_table[81].feg[8].cnl[2]= 2.5800000000;	data_table[81].feg[8].cl[3]= 2.0600000000;	data_table[81].feg[8].cnl[3]= 6.2200000000;	data_table[81].feg[8].cl[4]= 0.6620000000;	data_table[81].feg[8].cnl[4]= 14.8000000000;
				data_table[82].feg[6].cl[0]= 0.7080000000;	data_table[82].feg[6].cnl[0]= 0.1700000000;	data_table[82].feg[6].cl[1]= 1.3500000000;	data_table[82].feg[6].cnl[1]= 0.9810000000;	data_table[82].feg[6].cl[2]= 2.2800000000;	data_table[82].feg[6].cnl[2]= 3.4400000000;	data_table[82].feg[6].cl[3]= 2.1800000000;	data_table[82].feg[6].cnl[3]= 9.4100000000;	data_table[82].feg[6].cl[4]= 0.7970000000;	data_table[82].feg[6].cnl[4]= 23.7000000000;
				data_table[82].feg[10].cl[0]= 0.6540000000;	data_table[82].feg[10].cnl[0]= 0.1620000000;	data_table[82].feg[10].cl[1]= 1.1800000000;	data_table[82].feg[10].cnl[1]= 0.9050000000;	data_table[82].feg[10].cl[2]= 1.2500000000;	data_table[82].feg[10].cnl[2]= 2.6800000000;	data_table[82].feg[10].cl[3]= 1.6600000000;	data_table[82].feg[10].cnl[3]= 5.1400000000;	data_table[82].feg[10].cl[4]= 0.7780000000;	data_table[82].feg[10].cnl[4]= 11.2000000000;
				data_table[87].feg[4].cl[0]= 0.9110000000;	data_table[87].feg[4].cnl[0]= 0.2040000000;	data_table[87].feg[4].cl[1]= 1.6500000000;	data_table[87].feg[4].cnl[1]= 1.2600000000;	data_table[87].feg[4].cl[2]= 2.5300000000;	data_table[87].feg[4].cnl[2]= 4.0300000000;	data_table[87].feg[4].cl[3]= 3.6200000000;	data_table[87].feg[4].cnl[3]= 12.6000000000;	data_table[87].feg[4].cl[4]= 1.5800000000;	data_table[87].feg[4].cnl[4]= 30.0000000000;
				data_table[88].feg[6].cl[0]= 0.9150000000;	data_table[88].feg[6].cnl[0]= 0.2050000000;	data_table[88].feg[6].cl[1]= 1.6400000000;	data_table[88].feg[6].cnl[1]= 1.2800000000;	data_table[88].feg[6].cl[2]= 2.2600000000;	data_table[88].feg[6].cnl[2]= 3.9200000000;	data_table[88].feg[6].cl[3]= 3.1800000000;	data_table[88].feg[6].cnl[3]= 11.3000000000;	data_table[88].feg[6].cl[4]= 1.2500000000;	data_table[88].feg[6].cnl[4]= 25.1000000000;
				data_table[91].feg[6].cl[0]= 1.1400000000;	data_table[91].feg[6].cnl[0]= 0.2500000000;	data_table[91].feg[6].cl[1]= 2.4800000000;	data_table[91].feg[6].cnl[1]= 1.8400000000;	data_table[91].feg[6].cl[2]= 3.6100000000;	data_table[91].feg[6].cnl[2]= 7.3900000000;	data_table[91].feg[6].cl[3]= 1.1300000000;	data_table[91].feg[6].cnl[3]= 18.0000000000;	data_table[91].feg[6].cl[4]= 0.9000000000;	data_table[91].feg[6].cnl[4]= 22.7000000000;
				data_table[91].feg[8].cl[0]= 1.0900000000;	data_table[91].feg[8].cnl[0]= 0.2430000000;	data_table[91].feg[8].cl[1]= 2.3200000000;	data_table[91].feg[8].cnl[1]= 1.7500000000;	data_table[91].feg[8].cl[2]= 12.0000000000;	data_table[91].feg[8].cnl[2]= 7.7900000000;	data_table[91].feg[8].cl[3]=-9.1100000000;	data_table[91].feg[8].cnl[3]= 8.3100000000;	data_table[91].feg[8].cl[4]= 2.1500000000;	data_table[91].feg[8].cnl[4]= 16.5000000000;
				data_table[91].feg[12].cl[0]= 0.6870000000;	data_table[91].feg[12].cnl[0]= 0.1540000000;	data_table[91].feg[12].cl[1]= 1.1400000000;	data_table[91].feg[12].cnl[1]= 0.8610000000;	data_table[91].feg[12].cl[2]= 1.8300000000;	data_table[91].feg[12].cnl[2]= 2.5800000000;	data_table[91].feg[12].cl[3]= 2.5300000000;	data_table[91].feg[12].cnl[3]= 7.7000000000;	data_table[91].feg[12].cl[4]= 0.9570000000;	data_table[91].feg[12].cnl[4]= 15.9000000000;
			}

			// 1: Doyle and Turner parameterization - 4 Gaussians - [0, 4]
			// 2: Peng et al. parameterization - 5 Gaussians - [0, 4]
			// 3: Peng et al. parameterization - 5 Gaussians - [0, 12]
			template <class TPP_Coef>
			void set_Doyle_Peng_neutral_coef(const int &Z, TPP_Coef &feg, TPP_Coef &fxg, TPP_Coef &Pr, TPP_Coef &Vr, TPP_Coef &VR)		
			{
				int iZ = Z-1;

				for(auto j = 0; j < 5; j++)
				{
					auto cl = data_table[iZ].feg[0].cl[j];
					auto cnl = (isZero(cl))?0.0:data_table[iZ].feg[0].cnl[j]/4.0;

					if(nonZero(cl, cnl))
					{
						feg.cl[j] = cl;
						feg.cnl[j] = cnl;

						fxg.cl[j] = c_2Pi2a0*cl;
						fxg.cnl[j] = cnl;

						Pr.cl[j] = 0.5*c_2Pi2a0*cl*pow(c_Pi/cnl, 3.5);
						Pr.cnl[j] = c_Pi2/cnl;

						Vr.cl[j] = c_Potf*cl*pow(c_Pi/cnl, 1.5);
						Vr.cnl[j] = c_Pi2/cnl;

						VR.cl[j] = c_Potf*c_Pi*cl/cnl;
						VR.cnl[j] = c_Pi2/cnl;
					}
					else
					{
						feg.cl[j] = 0.0;
						feg.cnl[j] = 1.0;

						fxg.cl[j] = 0.0;
						fxg.cnl[j] = 1.0;

						Pr.cl[j] = 0.0;
						Pr.cnl[j] = 1.0;

						Vr.cl[j] = 0.0;
						Vr.cnl[j] = 1.0;

						VR.cl[j] = 0.0;
						VR.cnl[j] = 1.0;
					}
				}
			}

			// 4: Kirkland parameterization - 3 Yukawa + 3 Gaussians - [0, 12]			
			template <class TPP_Coef>
			void set_Kirkland_neutral_coef(const int &Z, TPP_Coef &feg, TPP_Coef &fxg, TPP_Coef &Pr, TPP_Coef &Vr, TPP_Coef &VR)		
			{
				int iZ = Z-1;

				for(auto j = 0; j < 3; j++)
				{
					auto cl = data_table[iZ].feg[0].cl[j];
					auto cnl = (isZero(cl))?0.0:data_table[iZ].feg[0].cnl[j];

					if(nonZero(cl, cnl))
					{
						feg.cl[j] = cl;
						feg.cnl[j] = cnl;

						fxg.cl[j] = c_2Pi2a0*cl;
						fxg.cnl[j] = cnl;

						Pr.cl[j] = c_2Pi2a0*c_Pi*cl*cnl;
						Pr.cnl[j] = c_2Pi*sqrt(cnl);

						Vr.cl[j] = c_Potf*c_Pi*cl;
						Vr.cnl[j] = c_2Pi*sqrt(cnl);

						VR.cl[j] = c_Potf*c_2Pi*cl;
						VR.cnl[j] = c_2Pi*sqrt(cnl);
					}
				}

				for(auto j = 3; j < 6; j++)
				{
					auto cl = data_table[iZ].feg[0].cl[j];
					auto cnl = (isZero(cl))?0.0:data_table[iZ].feg[0].cnl[j];

					if(nonZero(cl, cnl))
					{
						feg.cl[j] = cl;
						feg.cnl[j] = cnl;

						fxg.cl[j] = c_2Pi2a0*cl;
						fxg.cnl[j] = cnl;

						Pr.cl[j] = 0.5*c_2Pi2a0*cl*pow(c_Pi/cnl, 3.5);
						Pr.cnl[j] = c_Pi2/cnl;

						Vr.cl[j] = c_Potf*cl*pow(c_Pi/cnl, 1.5);
						Vr.cnl[j] = c_Pi2/cnl;

						VR.cl[j] = c_Potf*c_Pi*cl/cnl;
						VR.cnl[j] = c_Pi2/cnl;
					}
				}
			}
			
			// 5: Weickenmeier and H.Kohl - a*(1-exp(-bg^2)/g^2 - [0, 12]
			template <class TPP_Coef>
			void set_Weickenmeier_neutral_coef(const int &Z, TPP_Coef &feg, TPP_Coef &fxg, TPP_Coef &Pr, TPP_Coef &Vr, TPP_Coef &VR)		
			{
				int iZ = Z-1;

				for(auto j = 0; j < 6; j++)
				{
					// 1.0/c_2Pi2a0 = 4.0*0.0239336609991378
					auto cl = Z/(c_2Pi2a0*3.0*(1.0+data_table[iZ].feg[0].cl[j]));
					cl *= (j>= 3)?data_table[iZ].feg[0].cl[j]:1;
					auto cnl = (isZero(cl))?0.0:data_table[iZ].feg[0].cnl[j]/4.0;
					cl = (isZero(cnl))?0.0:cl;

					if(nonZero(cl, cnl))
					{
						feg.cl[j] = cl;
						feg.cnl[j] = cnl;

						fxg.cl[j] = c_2Pi2a0*cl;
						fxg.cnl[j] = cnl;

						Pr.cl[j] = c_2Pi2a0*cl*pow(c_Pi/cnl, 1.5);
						Pr.cnl[j] = c_Pi2/cnl;

						Vr.cl[j] = c_Potf*c_Pi*cl;
						Vr.cnl[j] = c_Pi/sqrt(cnl);

						// there is not analytic expression for the projected potential
						VR.cl[j] = Vr.cl[j];
						VR.cnl[j] = Vr.cnl[j];
					}
				}
			}
			
			// 6: Lobato parameterization - Hydrogen functions - [0, 12]
			template <class TPP_Coef>
			void set_Lobato_neutral_coef(const int &Z, TPP_Coef &feg, TPP_Coef &fxg, TPP_Coef &Pr, TPP_Coef &Vr, TPP_Coef &VR)		
			{
				int iZ = Z-1;

				for(auto j = 0; j < 6; j++)
				{
					auto cl = data_table[iZ].feg[0].cl[j];
					auto cnl = (isZero(cl))?0.0:data_table[iZ].feg[0].cnl[j];
	
					if(nonZero(cl, cnl))
					{
						feg.cl[j] = cl;
						feg.cnl[j] = cnl;

						fxg.cl[j] = c_2Pi2a0*cl/cnl;
						fxg.cnl[j] = cnl;

						Pr.cl[j] = c_2Pi2a0*c_Pi2*cl/pow(cnl, 2.5);
						Pr.cnl[j] = c_2Pi/sqrt(cnl);

						Vr.cl[j] = c_Potf*c_Pi2*cl/pow(cnl, 1.5);
						Vr.cnl[j] = c_2Pi/sqrt(cnl);

						VR.cl[j] = 2.0*c_Potf*c_Pi2*cl/pow(cnl, 1.5);
						VR.cnl[j] = c_2Pi/sqrt(cnl);
					}
					else
					{
						feg.cl[j] = 0.0;
						feg.cnl[j] = 1.0;

						fxg.cl[j] = 0.0;
						fxg.cnl[j] = 1.0;

						Pr.cl[j] = 0.0;
						Pr.cnl[j] = 1.0;

						Vr.cl[j] = 0.0;
						Vr.cnl[j] = 1.0;

						VR.cl[j] = 0.0;
						VR.cnl[j] = 1.0;
					}
				}
			}

			// Peng et al. parameterization for ions - 5 Gaussians - [0, 4]
			template <class TPP_Coef>
			void set_Peng_ions_coef(const int &Z, const int &charge, TPP_Coef &feg, TPP_Coef &fxg, TPP_Coef &Pr, TPP_Coef &Vr, TPP_Coef &VR)		
			{
				using T = Value_type<TPP_Coef>;

				int iZ = Z-1;
				int icharge = (charge>= 0)?(2*charge):(-2*charge-1);

				for(auto j = 0; j < 5; j++)
				{
					auto cl = data_table[iZ].feg[icharge].cl[j];
					auto cnl = (isZero(cl))?0.0:data_table[iZ].feg[icharge].cnl[j]/4.0;

					if(nonZero(cl, cnl))
					{
						feg.cl[j] = cl;
						feg.cnl[j] = cnl;

						fxg.cl[j] = c_2Pi2a0*cl;
						fxg.cnl[j] = cnl;

						Pr.cl[j] = 0.5*c_2Pi2a0*cl*pow(c_Pi/cnl, 3.5);
						Pr.cnl[j] = c_Pi2/cnl;

						Vr.cl[j] = c_Potf*cl*pow(c_Pi/cnl, 1.5);
						Vr.cnl[j] = c_Pi2/cnl;

						VR.cl[j] = c_Potf*c_Pi*cl/cnl;
						VR.cnl[j] = c_Pi2/cnl;
					}
				}

				// 1/g2
				if(nonZero(feg.cl[0]))
				{
					int j = 5;
					T r0 = 2.0;
					auto cl = charge/c_2Pi2a0;
					auto cnl = pow(c_2Pi*r0, -2);

					feg.cl[j] = cl;
					feg.cnl[j] = cnl;

					fxg.cl[j] = c_2Pi2a0*cl;
					fxg.cnl[j] = cnl;

					Pr.cl[j] = c_2Pi2a0*c_Pi*cl*cnl;
					Pr.cnl[j] = c_2Pi*sqrt(cnl);

					Vr.cl[j] = c_Potf*c_Pi*cl;
					Vr.cnl[j] = c_2Pi*sqrt(cnl);

					VR.cl[j] = c_Potf*c_2Pi*cl;
					VR.cnl[j] = c_2Pi*sqrt(cnl);
				}
			}

			// get Coefficients
			template <class TAtom_Type>
			void get_atom_coef(int Z, double Vrl, int nR, double R_min, TAtom_Type &atom_type)		
			{
				using T = Value_type<TAtom_Type>;

				for(auto i = 0; i < c_feg.size(); i++)
				{
					c_feg[i].fill(0);
					c_fxg[i].fill(0);
					c_Pr[i].fill(0);
					c_Vr[i].fill(0);
					c_VR[i].fill(0);
				}

				if(static_cast<int>(potential_type) <= 3)
				{
					// 1: Doyle and Turner parameterization - 4 Gaussians - [0, 4]
					// 2: Peng et al. parameterization - 5 Gaussians - [0, 4]
					// 3: Peng et al. parameterization - 5 Gaussians - [0, 12]
					set_Doyle_Peng_neutral_coef(Z, c_feg[0], c_fxg[0], c_Pr[0], c_Vr[0], c_VR[0]);
				}
				else if(potential_type == ePT_Kirkland_0_12)
				{
					// 4: Kirkland parameterization - 3 Yukawa + 3 Gaussians - [0, 12]
					set_Kirkland_neutral_coef(Z, c_feg[0], c_fxg[0], c_Pr[0], c_Vr[0], c_VR[0]);
				}
				else if(potential_type == ePT_Weickenmeier_0_12)
				{
					// 5: Weickenmeier and H.Kohl - a*(1-exp(-bg^2)/g^2 - [0, 12]
					set_Weickenmeier_neutral_coef(Z, c_feg[0], c_fxg[0], c_Pr[0], c_Vr[0], c_VR[0]);
				}
				else if(potential_type == ePT_Lobato_0_12)
				{
					// 6: Lobato parameterization - Hydrogen functions - [0, 12]
					set_Lobato_neutral_coef(Z, c_feg[0], c_fxg[0], c_Pr[0], c_Vr[0], c_VR[0]);
				}

				auto from_idx_to_charge = [](const int &i)->int{return ((i%2 == 0)?1:-1)*((i+1)/2); };

				// Peng et al. parameterizationfor ions - 5 Gaussians - [0, 4]
				int ncoef = 1;
				for(auto i = 1; i < c_feg.size(); i++)
				{
					auto charge = from_idx_to_charge(i); 
					set_Peng_ions_coef(Z, charge, c_feg[i], c_fxg[i], c_Pr[i], c_Vr[i], c_VR[i]);
					ncoef += (nonZero(c_feg[i].cl[0]))?1:0;
				}

				// copy all coefficients
				atom_type.coef.resize(ncoef);
				int icoef = 0;
				for(auto i = 0; i < c_feg.size(); i++)
				{
					if((i == 0) || nonZero(c_feg[i].cl[0]))
					{
						auto &coef = atom_type.coef[icoef++];
						coef.charge = from_idx_to_charge(i); 

						coef.feg.assign(c_feg[i]);
						coef.fxg.assign(c_fxg[i]);
						coef.Pr.assign(c_Pr[i]);
						coef.Vr.assign(c_Vr[i]);
						coef.VR.assign(c_VR[i]);

						coef.R_min = R_min;
						c_atom_cal.Set_Atom_Type(potential_type, coef.charge, &atom_type);
						coef.R_max = (coef.charge == 0)?c_atom_cal.AtomicRadius_Cutoff(3, Vrl):atom_type.coef[0].R_max;
						coef.R_max = ::fmax(coef.R_max, T(2)*R_min);

						if(isZero(coef.R_max))
						{
							coef.R_max = 1.75*atom_type.ra_e;
						}

						//coef.R_tap = 0.85*coef.R_max;
						coef.R_tap = coef.R_min + 0.85*(coef.R_max - coef.R_min);
						coef.tap_cf = c_i2Pi/(coef.R2_max()-coef.R2_tap());

						// R and R2
						coef.R.resize(nR);
						coef.R2.resize(nR);
						T dlnr = log(coef.R_max/coef.R_min)/T(nR-1);
						for(auto iR = 0; iR < nR; iR++)
						{
							coef.R[iR] = coef.R_min*exp(iR*dlnr);
							coef.R2[iR] = pow(coef.R[iR], 2);
						}

						Vector<T, e_host> VR(nR);
						Vector<T, e_host> dVR(nR);

						c_atom_cal.Set_Atom_Type(potential_type, coef.charge, &atom_type);
						c_atom_cal.VR_dVR(nR, coef.R.data(), VR.data(), dVR.data());

						auto x_tap = coef.R2_tap();
						auto tap_cf = coef.tap_cf;
						coef.ciVR.resize(nR);
						for(auto iR = 0; iR < nR; iR++)
						{
							auto x = coef.R2[iR];
							auto y = VR[iR];
							auto dy = 0.5*dVR[iR]/coef.R[iR];
							host_device_detail::apply_tapering(x_tap, tap_cf, x, y, dy);

							coef.ciVR.c0[iR] = y;
							coef.ciVR.c1[iR] = dy;
						}

						for(auto iR = 0; iR < nR-1; iR++)
						{
							T idR = 1.0/(coef.R2[iR+1]-coef.R2[iR]);
							T V = coef.ciVR.c0[iR]; 
							T Vn = coef.ciVR.c0[iR+1];
							T dV = coef.ciVR.c1[iR]; 
							T dVn = coef.ciVR.c1[iR+1];
							T m = (Vn-V)*idR; 
							T n = dV+dVn;
							coef.ciVR.c2[iR] = (3.0*m-n-dV)*idR;
							coef.ciVR.c3[iR] = (n-2.0*m)*idR*idR;
						}
					}
				}
			}

			ePotential_Type potential_type;
			Vector<Data_Table, e_host> data_table;

			Vector<PP_Coef<double, e_host>, e_host> c_feg;
			Vector<PP_Coef<double, e_host>, e_host> c_fxg;
			Vector<PP_Coef<double, e_host>, e_host> c_Pr;
			Vector<PP_Coef<double, e_host>, e_host> c_Vr;
			Vector<PP_Coef<double, e_host>, e_host> c_VR;

			Atom_Cal<double> c_atom_cal;
			Atom_Type<double, e_host> c_atom_type;
	};

} // namespace mt
#endif