import { Role } from 'aws-cdk-lib/aws-iam';
import { Location } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
/**
* The props for LF-S3-Location Construct.
*/
export interface LakeFormationS3LocationProps {
    /**
     * S3 location to be registered with Lakeformation
     */
    readonly s3Location: Location;
    /**
     * KMS key used to encrypt the S3 Location
     * @default - No encryption is used
     */
    readonly kmsKeyId: string;
    /**
     * Account ID owning the S3 location
     * @default - Current account is used
     */
    readonly accountId?: string;
}
/**
 * This CDK construct aims to register an S3 Location for Lakeformation with Read and Write access.
 * If the location is in a different account, cross account access should be granted via the [S3CrossAccount]{@link S3CrossAccount} construct.
 * If the S3 location is encrypted with KMS, the key must be explicitly passed to the construct because CDK cannot retrieve bucket encryption key from imported buckets.
 * Imported buckets are generally used in cross account setup like data mesh.
 *
 * This construct instantiate 2 objects:
 * * An IAM role with read/write permissions to the S3 location and encrypt/decrypt access to the KMS key used to encypt the bucket
 * * A CfnResource is based on an IAM role with 2 policy statement folowing the least privilege AWS best practices:
 *   * Statement 1 for S3 permissions
 *   * Statement 2 for KMS permissions if the bucket is encrypted
 *
 * The CDK construct instantiate the CfnResource in order to register the S3 location with Lakeformation using the IAM role defined above.
 *
 * Usage example:
 * ```typescript
 * import * as cdk from 'aws-cdk-lib';
 * import { LakeformationS3Location } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new cdk.App();
 * const stack = new cdk.Stack(exampleApp, 'LakeformationS3LocationStack');
 *
 * const myKey = new Key(stack, 'MyKey')
 * const myBucket = new Bucket(stack, 'MyBucket', {
 *   encryptionKey: myKey,
 * })
 *
 * new LakeFormationS3Location(stack, 'MyLakeformationS3Location', {
 *   s3Location: {
 *     bucketName: myBucket.bucketName,
 *     objectKey: 'my-prefix',
 *   },
 *   kmsKeyId: myBucket.encryptionKey.keyId,
 * });
 * ```
 */
export declare class LakeFormationS3Location extends Construct {
    readonly dataAccessRole: Role;
    constructor(scope: Construct, id: string, props: LakeFormationS3LocationProps);
}
