"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrManagedEndpointProvider = void 0;
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const constructs_1 = require("constructs");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
/**
 * A custom resource provider for CRUD operations on Amazon EMR on EKS Managed Endpoints.
 * @private
 */
class EmrManagedEndpointProvider extends constructs_1.Construct {
    /**
     * Constructs a new instance of the ManageEndpointProvider. The provider can then be used to create Amazon EMR on EKS Managed Endpoint custom resources
     * @param { Construct} scope the Scope of the CDK Construct
     * @param id the ID of the CDK Construct
     * @private
     */
    constructor(scope, id, props) {
        super(scope, id);
        //The policy allowing the managed endpoint custom resource to create call the APIs for managed endpoint
        const lambdaPolicy = [
            new aws_iam_1.PolicyStatement({
                resources: [props.assetBucket.bucketArn],
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
            }),
            new aws_iam_1.PolicyStatement({
                resources: ['*'],
                actions: ['emr-containers:DescribeManagedEndpoint'],
            }),
            new aws_iam_1.PolicyStatement({
                resources: ['*'],
                actions: [
                    'emr-containers:DeleteManagedEndpoint'
                ],
            }),
            new aws_iam_1.PolicyStatement({
                resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:emr-containers:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/virtualclusters/*`],
                actions: ['emr-containers:CreateManagedEndpoint'],
                conditions: { StringEquals: { 'aws:ResourceTag/for-use-with': 'cdk-analytics-reference-architecture' } },
            }),
            new aws_iam_1.PolicyStatement({
                resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:emr-containers:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/virtualclusters/*`],
                actions: ['emr-containers:TagResource'],
                conditions: { StringEquals: { 'aws:ResourceTag/for-use-with': 'cdk-analytics-reference-architecture' } },
            }),
            new aws_iam_1.PolicyStatement({
                resources: ['*'],
                actions: [
                    'ec2:CreateSecurityGroup',
                    'ec2:DeleteSecurityGroup',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:AuthorizeSecurityGroupEgress',
                    'ec2:AuthorizeSecurityGroupIngress',
                    'ec2:RevokeSecurityGroupEgress',
                    'ec2:RevokeSecurityGroupIngress',
                ],
            }),
        ];
        // AWS Lambda function supporting the create, update, delete operations on Amazon EMR on EKS managed endpoints
        const onEvent = new pre_bundled_function_1.PreBundledFunction(this, 'OnEvent', {
            codePath: 'emr-eks-platform/resources/lambdas/managed-endpoint',
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            handler: 'lambda.on_event',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaPolicy,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(120),
        });
        // AWS Lambda supporting the status check on asynchronous create, update and delete operations
        const isComplete = new pre_bundled_function_1.PreBundledFunction(this, 'IsComplete', {
            codePath: 'emr-eks-platform/resources/lambdas/managed-endpoint',
            handler: 'lambda.is_complete',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaPolicy,
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(120),
        });
        this.provider = new custom_resources_1.Provider(this, 'CustomResourceProvider', {
            onEventHandler: onEvent,
            isCompleteHandler: isComplete,
            totalTimeout: aws_cdk_lib_1.Duration.minutes(30),
            queryInterval: aws_cdk_lib_1.Duration.seconds(20),
            providerFunctionName: 'managedEndpointProviderFn',
        });
    }
}
exports.EmrManagedEndpointProvider = EmrManagedEndpointProvider;
//# sourceMappingURL=data:application/json;base64,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